/*

sshconfig.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Processing configuration data in SSH (both client and server).
  
*/

#include "ssh2includes.h"
#include "sshconfig.h"
#include "sshuser.h"

#include "sshuserfile.h"

#include "sshkeyfile.h"
#include "sshuserfiles.h"
#include "sshcipherlist.h"
#include "sshsnlist.h"
#include "sshappcommon.h"
#include "sshglob.h"
#include "sshdsprintf.h"
#include "sshhostkey.h"

#include "sshbuffer.h"



#include "sshmsgs.h"
#include "sshtcp.h"
#include "sshdlex.h"
#include "sshadt_strmap.h"

#define SSH_DEBUG_MODULE "SshConfig"

/* Free the "vars" and "vals" arrays */

void ssh_free_varsvals(int n, char **vars, char **vals)
{
  int i;

  for (i = 0; i < n; i++)
    {
      ssh_xfree(vars[i]);
      ssh_xfree(vals[i]);
    }
  ssh_xfree(vars);
  ssh_xfree(vals);
}

/* For SSH_ADT_DESTROY. */
static void destr_xfree(void *obj, void *context)
{
  ssh_xfree(obj);
}

typedef void *(*SshParameterAllocator)(char *token, void *context);

/* Parses a given comma-separated list to tokens, which are stored in
   a SshADTContainer. The list is allocated and returned by this
   function. On error, returns TRUE. */
Boolean ssh_config_parse_list_check_validity(char *string,
                                             SshADTContainer container,
                                             SshParameterAllocator allocator,
                                             void *alloc_context,
                                             SshParameterValidityProc function,
                                             void *context)
{
  char *rest;
  char *current;

  SSH_PRECOND(string != NULL);

  rest = string;

  while (strlen(rest) != 0 &&
         (current = ssh_app_param_list_get_next(rest)) != NULL)
    {
      char *temp;
      int i, j;

      rest += strlen(current);
      if (*rest == ',')
        rest++;

      /* strip whitespaces and non-printable characters */
      temp = ssh_xcalloc(strlen(current) + 1, sizeof(char));

      for (i = 0, j = 0; i < strlen(current) ; i ++)
        if(isascii(current[i]) && isprint(current[i]) && !isspace(current[i]))
          {
            temp[j] = current[i];
            j++;
          }

      temp[j] = '\0';
      ssh_xfree(current);

      /* strip escapes preceding list separator. */
      SSH_DEBUG(8, ("current: %s", temp));      
      current = ssh_xstrdup(temp);
      for (i = 0, j = 0; i < strlen(temp) ; i++,j++)
        {
          if (temp[i] == ',' && ssh_glob_isescaped(&temp[i], temp))
            current[--j] = temp[i];
          else
            current[j] = temp[i];
        }
      current[j] = '\0';

      SSH_DEBUG(8, ("stripped: %s", current));
      ssh_xfree(temp);

      /* If validity function is given, invoke it to check the current
         parameter.*/
      if (function)
        if ((*function)(current, context))
          {
            ssh_xfree(current);
            ssh_adt_clear(container);
            return TRUE;
          }

      if (!allocator)
        SSH_VERIFY(ssh_adt_insert(container, (void *)current) !=
                   SSH_ADT_INVALID);
      else
        SSH_VERIFY(ssh_adt_insert(container,
                                  (*allocator)(current, alloc_context)) !=
                   SSH_ADT_INVALID);
    }
  return FALSE;
}

/* List is not valid if NULL is returned. */
SshADTContainer
ssh_config_parse_list_check_validity_alloc(char *string,
                                           SshParameterAllocator allocator,
                                           void *alloc_context,
                                           SshParameterValidityProc function,
                                           void *val_context)
{
  SshADTContainer container = ssh_adt_create_generic(SSH_ADT_LIST,
                                                     SSH_ADT_DESTROY,
                                                     destr_xfree,
                                                     SSH_ADT_ARGS_END);
  SSH_VERIFY(container);
  
  if (ssh_config_parse_list_check_validity(string, container,
                                           allocator, alloc_context,
                                           function, val_context))
    ssh_adt_clear(container);

  if (ssh_adt_num_objects(container) < 1)
    {
      ssh_adt_destroy(container);
      container = NULL;
    }
  return container;
}

SshADTContainer ssh_config_parse_list_alloc(char *string,
                                            SshParameterAllocator allocator,
                                            void *alloc_context)
{
  return ssh_config_parse_list_check_validity_alloc
    (string, allocator, alloc_context, NULL, NULL);
}

void ssh_config_parse_list(char *string,
                           SshParameterAllocator allocator,
                           void *alloc_context,
                           SshADTContainer container)
{
  (void)ssh_config_parse_list_check_validity(string, container,
                                             allocator, alloc_context,
                                             NULL, NULL);
}

/* Allocates and initializes a config structure */
SshConfig ssh_config_init(Boolean client)
{
  SshConfig config;



  config = ssh_xcalloc(1, sizeof(*config));
  config->client = client;

  config->host_keys_ctx = ssh_host_key_list_init(config);
  
  config->random_seed_file = ssh_xstrdup(SSH_RANDSEED_FILE);
  config->pgp_public_key_file = ssh_xstrdup(SSH_PGP_PUBLIC_KEY_FILE);
  config->pgp_secret_key_file = ssh_xstrdup(SSH_PGP_SECRET_KEY_FILE);

  config->forward_agent = TRUE;
  config->forward_x11 = TRUE;
  config->x11_forwarding = TRUE;
  config->allow_tcp_forwarding = TRUE;
  config->allow_agent_forwarding = TRUE;
  config->password_authentication = -1;
  config->rhosts_authentication = TRUE;
  config->rhosts_pubkey_authentication = TRUE;
  config->pubkey_authentication = -1;
  config->force_ptty_allocation = FALSE;
  config->verbose_mode = FALSE;
  config->compression = FALSE;
  config->compression_level = -1;

  config->allowed_authentications = ssh_adt_create_generic(SSH_ADT_LIST,
                                                           SSH_ADT_DESTROY,
                                                           destr_xfree,
                                                           SSH_ADT_ARGS_END);
  config->required_authentications = ssh_adt_create_generic(SSH_ADT_LIST,
                                                            SSH_ADT_DESTROY,
                                                            destr_xfree,
                                                            SSH_ADT_ARGS_END);
  SSH_VERIFY(config->allowed_authentications);
  SSH_VERIFY(config->required_authentications);
  
  config->user_known_hosts = TRUE;
  config->port = ssh_xstrdup(SSH_DEFAULT_PORT);


  config->ciphers = NULL;












  config->macs = NULL;
  config->user_conf_dir = ssh_xstrdup(SSH_USER_CONFIG_DIRECTORY);
  config->identity_file = ssh_xstrdup(SSH_IDENTIFICATION_FILE);
  config->authorization_file = ssh_xstrdup(SSH_AUTHORIZATION_FILE);






  
  config->password_prompt = ssh_xstrdup("%U's password: ");
  config->password_guesses = 3;
  config->number_of_password_prompts = 3;

  config->remote_env = NULL;
  config->settable_env_vars = NULL;
  
  config->rekey_interval_bytes = 0L;
  config->rekey_interval_seconds = 3600L;

#ifndef SSH2_MAX_CONNECTIONS
  config->max_connections = 0;
#else /* ! SSH2_MAX_CONNECTIONS */
  config->max_connections = SSH2_MAX_CONNECTIONS;
#endif /* ! SSH2_MAX_CONNECTIONS */

  config->host_to_connect = NULL;
  config->login_as_user = NULL;
  config->local_forwards = NULL;
  config->remote_forwards = NULL;

  config->allowed_hosts = NULL;
  config->denied_hosts = NULL;
  config->allowed_shosts = NULL;
  config->denied_shosts = NULL;

  config->allowed_users = NULL;
  config->denied_users = NULL;
  config->allowed_groups = NULL;
  config->denied_groups = NULL;

  config->require_reverse_mapping = FALSE;
  config->try_reverse_mapping = TRUE;

  config->log_facility = SSH_LOGFACILITY_AUTH;
  config->sftp_server_log_facility = -1;
  config->debug_log_file_name = NULL;
  
  config->fall_back_to_rsh = TRUE;
  config->use_rsh = TRUE;
  config->batch_mode = FALSE;
  config->authentication_notify = FALSE;
  config->strict_host_key_checking = SSH_STRICT_HOSTKEY_CHECKING_ASK;
  config->escape_char = ssh_xstrdup("~");
  config->go_background = FALSE;
  config->one_shot_forwarding = FALSE;
  config->dont_read_stdin = FALSE;
  config->gateway_ports = FALSE;

  config->ignore_rhosts = FALSE;
  config->ignore_root_rhosts = -1;
  config->permit_root_login = SSH_ROOTLOGIN_TRUE;
  config->permit_empty_passwords = TRUE;
  config->try_empty_password = FALSE;
  config->strict_modes = TRUE;
  config->quiet_mode = FALSE;
  config->fascist_logging = FALSE;
  config->print_motd = TRUE;
  config->check_mail = TRUE;
  config->auth_success_msg = TRUE;
  config->keep_alive = TRUE;
  config->no_delay = FALSE;
  config->inetd_mode = FALSE;
  config->hostbased_force_client_hostname_dns_match = FALSE;
  config->listen_address = ssh_xstrdup(SSH_IPADDR_ANY);
  config->login_grace_time = 600;
  config->chroot_users = NULL;
  config->chroot_groups = NULL;
  config->allowed_tcp_forwarding_users = NULL;
  config->denied_tcp_forwarding_users = NULL;
  config->denied_tcp_forwarding_groups = NULL;
  config->allowed_tcp_forwarding_groups = NULL;
  config->socks_server = NULL;

  config->subsystems = ssh_adt_create_strmap();

  /* This is overridden in default config file.  However it is not
     nice to start new service as default, if user has old config
     file template. */
  config->broadcasts_allowed_per_second = 0;

#ifdef SSH1_COMPATIBILITY
  config->ssh1_path = ssh_xstrdup(client ? SSH1_PATH : SSHD1_PATH);
  config->ssh1compatibility = TRUE;
#else /* SSH1_COMPATIBILITY */
  config->ssh1_path = NULL;
  config->ssh1compatibility = FALSE;
#endif /* SSH1_COMPATIBILITY */
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
  config->ssh1_emulation_internal = TRUE;
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
  config->sshd1_config_file_path = NULL;
  config->ssh_agent_compat = SSH_AGENT_COMPAT_NONE;

  config->ssh_pam_client_path = ssh_xstrdup(SSH_PAM_CLIENT_PATH);
  config->signer_path = ssh_xstrdup(SSH_SIGNER_PATH);

  config->default_domain = NULL;
  
#ifdef SSH_SERVER_WITH_SECURID
  config->securid_guesses = 3;
#endif /* SSH_SERVER_WITH_SECURID */





  config->banner_message_file_path = ssh_xstrdup(SSH_BANNER_MSG_FILE);
  config->banner_msg = NULL;

  config->idle_timeout = SSH_SERVER_DEFAULT_IDLE_TIMEOUT;











































  return config;
}

/* This should be called after initializing the config-struct
   (ie. after command-line variables have been parsed. This checks,
   that some required members are initialized properly.)*/
void ssh_config_init_finalize(SshConfig config)
{
  /* Common. */
  /* "hostbased"-authentication method is not enabled by default. */
  if (ssh_adt_num_objects(config->allowed_authentications) == 0)
    {
      ssh_config_parse_list((char *)
#ifdef KERBEROS
                            SSH_AUTH_KERBEROS_TGT ","
                            SSH_AUTH_KERBEROS ","
#endif /* KERBEROS */
                            SSH_AUTH_PUBKEY "," SSH_AUTH_PASSWD,
                            NULL, NULL, config->allowed_authentications);
    }
  
  if (config->ciphers == NULL)
    {
      char *hlp;

      hlp = ssh_cipher_get_supported_native();
      config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp);
      ssh_xfree(hlp);
      hlp = config->ciphers;
      config->ciphers = ssh_cipher_list_exclude(hlp, "none");
      ssh_xfree(hlp);
    }






















  /* Client. */
  if (config->client)
    {
      /* Nothing here yet! */
    }
  /* Server. */

  else
    {
      SshUserFile f;
      uid_t uid = 999;


      uid = getuid();


      /* If IgnoreRootRhosts isn't defined at this stage, assign it to
         the same as IgnoreRhosts. */
      if (config->ignore_root_rhosts == -1)
        config->ignore_root_rhosts = config->ignore_rhosts;

      if ((f = ssh_userfile_open(uid, config->banner_message_file_path,
                                 O_RDONLY, 0755)) != NULL)
        {
          SshBuffer buf;
          char linebuf[1024];

          buf = ssh_xbuffer_allocate();

          while (ssh_userfile_gets(linebuf, sizeof(linebuf) - 1, f) != NULL)
            ssh_xbuffer_append(buf, linebuf, strlen(linebuf));

          if (ssh_buffer_len(buf) > 0)
            {
              size_t len = ssh_buffer_len(buf) + 1;

              if (len > SSH_MAX_PAYLOAD_LENGTH - 1)
                len = SSH_MAX_PAYLOAD_LENGTH - 1;
              
              config->banner_msg = ssh_xcalloc(len,
                                               sizeof(char));
              memcpy(config->banner_msg, ssh_buffer_ptr(buf),
                     len - 1);
            }

          ssh_buffer_free(buf);

          ssh_userfile_close(f);
        }




















    }

}

void ssh_free_forward(SshForward fwd)
{
  SSH_PRECOND(fwd);
  ssh_xfree(fwd->local_addr);
  ssh_xfree(fwd->port);
  ssh_xfree(fwd->connect_to_host);
  ssh_xfree(fwd->connect_to_port);
  ssh_xfree(fwd->protocol);
  ssh_xfree(fwd);
}





























/* Frees client configuration data. */
void ssh_config_free(SshConfig config)
{
  SshForward ptr = NULL;
  char *str = NULL;
  SshADTHandle h;
  
  /* free all allocated memory */
  ssh_xfree(config->random_seed_file);
  ssh_xfree(config->pgp_public_key_file);
  ssh_xfree(config->pgp_secret_key_file);

  ssh_xfree(config->port);
  ssh_xfree(config->ciphers);
  ssh_xfree(config->macs);
  ssh_xfree(config->user_conf_dir);





  ssh_xfree(config->identity_file);
  ssh_xfree(config->authorization_file);
  ssh_xfree(config->escape_char);
  ssh_xfree(config->listen_address);
  ssh_xfree(config->password_prompt);

  ssh_host_key_list_free(config->host_keys_ctx);
  
  ssh_xfree(config->host_to_connect);
  ssh_xfree(config->login_as_user);

  for (ptr = config->local_forwards; ptr;)
    {
      SshForward to_be_deleted = ptr;
      ptr = ptr->next;
      ssh_free_forward(to_be_deleted);
    }

  for (ptr = config->remote_forwards; ptr;)
    {
      SshForward to_be_deleted = ptr;
      ptr = ptr->next;
      ssh_free_forward(to_be_deleted);
    }

  ssh_adt_destroy(config->allowed_hosts);
  ssh_adt_destroy(config->denied_hosts);
  
  ssh_adt_destroy(config->allowed_shosts);
  ssh_adt_destroy(config->denied_shosts);  

  ssh_adt_destroy(config->allowed_users);
  ssh_adt_destroy(config->denied_users);
  ssh_adt_destroy(config->allowed_groups);
  ssh_adt_destroy(config->denied_groups);

  ssh_adt_destroy(config->allowed_authentications);
  ssh_adt_destroy(config->required_authentications);

  ssh_adt_destroy(config->chroot_users);
  ssh_adt_destroy(config->chroot_groups);

  ssh_adt_destroy(config->allowed_tcp_forwarding_users);
  ssh_adt_destroy(config->denied_tcp_forwarding_users);
  ssh_adt_destroy(config->denied_tcp_forwarding_groups);
  ssh_adt_destroy(config->allowed_tcp_forwarding_groups);

  ssh_adt_destroy(config->remote_env);
  ssh_adt_destroy(config->settable_env_vars);

  ssh_xfree(config->debug_log_file_name);  
  ssh_xfree(config->socks_server);

  for (h = ssh_adt_enumerate_start(config->subsystems);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(config->subsystems, h))
    {
      str = ssh_adt_map_lookup(config->subsystems, h);
      ssh_xfree(str);
    }
  ssh_adt_destroy(config->subsystems);
  
  ssh_xfree(config->ssh1_path);
  ssh_xfree(config->sshd1_config_file_path);
  
  ssh_xfree(config->ssh_pam_client_path);

  ssh_xfree(config->signer_path);
  ssh_xfree(config->default_domain);

  ssh_xfree(config->banner_message_file_path);
  ssh_xfree(config->banner_msg);






















  memset(config, 0, sizeof(*config));
  ssh_xfree(config);
}


/* Returns default configuration information for the server. */

SshConfig ssh_server_create_config()
{
  return ssh_config_init(FALSE);
}

/* Returns default configuration information for the client. */

SshConfig ssh_client_create_config()
{
  return ssh_config_init(TRUE);
}


struct LogFacility
{
  SshLogFacility facility;
  char *fac_name;
} logfacilities[] =
{
  {SSH_LOGFACILITY_AUTH, "AUTH"},
  {SSH_LOGFACILITY_SECURITY, "SECURITY"},
  {SSH_LOGFACILITY_DAEMON, "DAEMON"},
  {SSH_LOGFACILITY_USER, "USER"},
  {SSH_LOGFACILITY_MAIL, "MAIL"},
  {SSH_LOGFACILITY_LOCAL0, "LOCAL0"},
  {SSH_LOGFACILITY_LOCAL1, "LOCAL1"},
  {SSH_LOGFACILITY_LOCAL2, "LOCAL2"},
  {SSH_LOGFACILITY_LOCAL3, "LOCAL3"},
  {SSH_LOGFACILITY_LOCAL4, "LOCAL4"},
  {SSH_LOGFACILITY_LOCAL5, "LOCAL5"},
  {SSH_LOGFACILITY_LOCAL6, "LOCAL6"},
  {SSH_LOGFACILITY_LOCAL7, "LOCAL7"},
  {-1, NULL}
};

Boolean auth_param_validity(const char *param, void *context)
{
  if (strcasecmp(param, SSH_AUTH_PUBKEY) == 0)
    return FALSE;

  if (strcasecmp(param, SSH_AUTH_PASSWD) == 0)
    return FALSE;

#if defined (SSH_SERVER_WITH_SECURID) || defined(SSH_CLIENT_WITH_SECURID)
  if (strcasecmp(param, SSH_AUTH_SECURID) == 0)
    return FALSE;
#endif /* SSH_SERVER_WITH_SECURID || SSH_CLIENT_WITH_SECURID */


#if defined (DAEMON_WITH_PAM) || defined(CLIENT_WITH_PAM)
  if (strcasecmp(param, SSH_AUTH_PAM) == 0)
    return FALSE;
#endif /* DAEMON_WITH_PAM || CLIENT_WITH_PAM */

  if (strcasecmp(param, SSH_AUTH_HOSTBASED) == 0)
    return FALSE;

#ifdef KERBEROS
  if (strcasecmp(param, SSH_AUTH_KERBEROS) == 0)
    return FALSE;

  if (strcasecmp(param, SSH_AUTH_KERBEROS_TGT) == 0)
    return FALSE;
#endif /* KERBEROS */




  return TRUE;
}

SshPatternHolder ssh_config_pattern_alloc(char *token,
                                          SshMetaConfig metaconfig)
{
  SshPatternHolder holder;
  
  SSH_PRECOND(token);
  SSH_PRECOND(metaconfig);

  holder = ssh_xcalloc(1, sizeof(*holder));

  holder->pattern = token;
  holder->regex_syntax = metaconfig->regex_syntax;

  return holder;
}

void ssh_config_pattern_destructor(SshPatternHolder holder, void *context)
{
  SSH_PRECOND(holder);

  ssh_xfree(holder->pattern);
  ssh_xfree(holder);
}

void config_pattern_destructor(void *obj, void *context)
{
  SshPatternHolder holder = (SshPatternHolder) obj;
  ssh_config_pattern_destructor(holder, context);
}

void *config_pattern_alloc(char *token,
                           void *context)
{
  SshMetaConfig metaconfig = (SshMetaConfig) context;
  return (void *)ssh_config_pattern_alloc(token, metaconfig);
}

char *ssh_config_remove_whitespace(char *str)
{
  char *new_str;
  int i = 0, j = 0, len = 0;

  len = strlen(str);
  
  new_str = ssh_xcalloc(len + 1, sizeof(char));
  for (i = 0; i < len ; i++)
    {
      if (!isspace(str[i]))
        {
          new_str[j] = str[i];
          j++;
        }
    }
  return new_str;
}

#define PARSE_LIST(list)                                                 \
  do {                                                                   \
    SshADTContainer container = (list);                                  \
                                                                         \
    if (!container)                                                      \
      {                                                                  \
        container = ssh_adt_create_generic(SSH_ADT_LIST,                 \
                                           SSH_ADT_DESTROY,              \
                                           config_pattern_destructor,    \
                                           SSH_ADT_ARGS_END);            \
        SSH_VERIFY(container != NULL);                                   \
        (list) = container;                                              \
      }                                                                  \
                                                                         \
    ssh_config_parse_list((val), config_pattern_alloc,                   \
                          metaconfig, container);                        \
    return FALSE;                                                        \
                                                                         \
  } while(0)

/* Set the variable corresponding to `var' to `val' in config */

Boolean ssh_config_set_parameter(SshConfig config, SshMetaConfig metaconfig,
                                 char *var, char *val)
{
  Boolean bool_val;
  unsigned int i;
  int num;
  SshForward current_forward;
  SshRegexSyntax rex_syntax;
  int config_version_major;
  int config_version_minor;
  char *new_val;












  
  if (!metaconfig)
    {
      config_version_major = 1;
      config_version_minor = 0;
      rex_syntax = SSH_REGEX_SYNTAX_ZSH_FILEGLOB;
    }
  else
    {
      config_version_major = metaconfig->version_major;
      config_version_minor = metaconfig->version_minor;
      rex_syntax = metaconfig->regex_syntax;
    }
  
  switch (val[0])
    {
    case 'y':  /* for "yes" */
    case 'Y':
    case 't':  /* for "true" */
    case 'T':
    case 'k':  /* for kylla [finnish] :-) */
    case 'K':

      bool_val = TRUE;
      break;

    default:
      bool_val = FALSE;
    }

  num = atoi(val);

  /* These configuration parameters are common for both client and
     server */

  if (strcmp(var, "allowedauthentications") == 0)
    {
      if (config->password_authentication != -1 &&
          (!config->client))
        ssh_warning("Defining AllowedAuthentications. Parameter "
                    "PasswordAuthentication (already defined) will be "
                    "ignored.");

      if (config->pubkey_authentication != -1 &&
          (!config->client))
        ssh_warning("Defining AllowedAuthentications. Parameter "
                    "PubkeyAuthentication (already defined) will be "
                    "ignored.");

      ssh_adt_clear(config->allowed_authentications);
      if (ssh_config_parse_list_check_validity(val,
                                               config->allowed_authentications,
                                               NULL, NULL,
                                               auth_param_validity,
                                               NULL))
        {
          ssh_warning("Parsing of value for AllowedAuthentications failed.");
          return TRUE;
        }
      else
        {
          return FALSE;
        }      
    }

  if (strcmp(var, "passwordauthentication") == 0)
    {
      if (ssh_adt_num_objects(config->allowed_authentications) > 0)
        {
          if (!config->client || config->verbose_mode)
            ssh_warning("AllowedAuthentications is already defined, ignoring "
                        "PasswordAuthentication keyword.");
          return TRUE;
        }

      config->password_authentication = bool_val;
      ssh_warning("PasswordAuthentication configuration keyword is "
                  "deprecated. Use AllowedAuthentications.");
      return FALSE;
    }

  if (strcmp(var, "pubkeyauthentication") == 0 ||
      strcmp(var, "rsaauthentication") == 0)
    {
      if (ssh_adt_num_objects(config->allowed_authentications) > 0)
        {
          if (!config->client || config->verbose_mode)
            ssh_warning("AllowedAuthentications is already defined, ignoring "
                        "PubkeyAuthentication keyword.");
          return TRUE;
        }

      config->pubkey_authentication = bool_val;
      ssh_warning("PubkeyAuthentication configuration keyword is "
                  "deprecated. Use AllowedAuthentications.");
      return FALSE;
    }

  if (strcmp(var, "port") == 0)
    {
      if (num >= 1 && num < 65536)
        {
          ssh_xfree(config->port);
          config->port = ssh_xstrdup(val);
        }
      else
        {
          ssh_warning("Ignoring illegal port number %s", val);
          return TRUE;
        }
      return FALSE;
    }

  if (strcmp(var, "ciphers") == 0)
    {
      SSH_DEBUG(5, ("Got config cipherlist \"%s\"", val));
      ssh_xfree(config->ciphers);
      new_val = ssh_config_remove_whitespace(val);
      
      if (strcasecmp(new_val, "any") == 0)
        {
          char *hlp1, *hlp2;

          hlp1 = ssh_cipher_get_supported_native();
          config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp1);
          hlp2 = ssh_cipher_list_exclude(config->ciphers, "none");
          ssh_xfree(config->ciphers);
          ssh_dsprintf(&config->ciphers, "%s,%s", hlp2, hlp1);
          ssh_xfree(hlp1);
          ssh_xfree(hlp2);
          hlp1 = ssh_cipher_list_canonicalize(config->ciphers);
          ssh_xfree(config->ciphers);
          config->ciphers = hlp1;
        }
      else if (strcasecmp(new_val, "anycipher") == 0)
        {
          char *hlp1, *hlp2;

          hlp2 = ssh_cipher_get_supported_native();
          hlp1 = ssh_cipher_list_exclude(hlp2, "none");
          ssh_xfree(hlp2);
          config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp1);
          hlp2 = ssh_cipher_list_exclude(config->ciphers, "none");
          ssh_xfree(config->ciphers);
          ssh_dsprintf(&config->ciphers, "%s,%s", hlp2, hlp1);
          ssh_xfree(hlp1);
          ssh_xfree(hlp2);
          hlp1 = ssh_cipher_list_canonicalize(config->ciphers);
          ssh_xfree(config->ciphers);
          config->ciphers = hlp1;
        }
      else if (strcasecmp(new_val, "anystd") == 0)
        {
          char *hlp = ssh_cipher_get_supported_native();
          config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp);
          ssh_xfree(hlp);
        }
      else if (strcasecmp(new_val, "anystdcipher") == 0)
        {
          char *hlp = ssh_cipher_get_supported_native();
          config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp);
          ssh_xfree(hlp);
          hlp = config->ciphers;
          config->ciphers = ssh_cipher_list_exclude(hlp, "none");
          ssh_xfree(hlp);
        }
      else
        {
          config->ciphers = ssh_cipher_list_canonicalize(new_val);
        }
      ssh_xfree(new_val);
      SSH_DEBUG(5, ("Final cipherlist \"%s\"", config->ciphers));
      return FALSE;
    }

  if (strcmp(var, "macs") == 0)
    {
      SSH_DEBUG(5, ("Got config maclist \"%s\"", val));
      ssh_xfree(config->macs);
      new_val = ssh_config_remove_whitespace(val);

      if (strcasecmp(new_val, "any") == 0)
        {
          char *hlp1, *hlp2;

          hlp1 = ssh_mac_get_supported();
          config->macs = ssh_snlist_intersection(SSH_STD_MACS, hlp1);
          hlp2 = ssh_cipher_list_exclude(config->macs, "none");
          ssh_xfree(config->macs);
          ssh_dsprintf(&config->macs, "%s,%s", hlp2, hlp1);
          ssh_xfree(hlp1);
          ssh_xfree(hlp2);
          hlp1 = ssh_hash_list_canonicalize(config->macs);
          ssh_xfree(config->macs);
          config->macs = hlp1;
        }
      else if (strcasecmp(new_val, "anymac") == 0)
        {
          char *hlp1, *hlp2;

          hlp2 = ssh_mac_get_supported();
          hlp1 = ssh_cipher_list_exclude(hlp2, "none");
          ssh_xfree(hlp2);
          config->macs = ssh_snlist_intersection(SSH_STD_MACS, hlp1);
          hlp2 = ssh_cipher_list_exclude(config->macs, "none");
          ssh_xfree(config->macs);
          ssh_dsprintf(&config->macs, "%s,%s", hlp2, hlp1);
          ssh_xfree(hlp1);
          ssh_xfree(hlp2);
          hlp1 = ssh_hash_list_canonicalize(config->macs);
          ssh_xfree(config->macs);
          config->macs = hlp1;
        }
      else if (strcasecmp(new_val, "anystd") == 0)
        {
          char *hlp = ssh_mac_get_supported();
          config->macs = ssh_snlist_intersection(hlp, SSH_STD_MACS);
          ssh_xfree(hlp);
        }
      else if (strcasecmp(new_val, "anystdmac") == 0)
        {
          char *hlp = ssh_mac_get_supported();
          config->macs = ssh_snlist_intersection(hlp, SSH_STD_MACS);
          ssh_xfree(hlp);
          hlp = config->macs;
          config->macs = ssh_cipher_list_exclude(hlp, "none");
          ssh_xfree(hlp);
        }
      else
        {
          config->macs = ssh_hash_list_canonicalize(new_val);
        }
      ssh_xfree(new_val);
      SSH_DEBUG(5, ("Final maclist \"%s\"", config->macs));
      return FALSE;
    }

  if (strcmp(var, "userconfigdirectory") == 0)
    {
      ssh_xfree(config->user_conf_dir);
      config->user_conf_dir = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "identityfile") == 0)
    {
      ssh_xfree(config->identity_file);
      config->identity_file = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "authorizationfile") == 0)
    {
      ssh_xfree(config->authorization_file);
      config->authorization_file = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "randomseedfile") == 0)
    {
      ssh_xfree(config->random_seed_file);
      config->random_seed_file = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "pgppublickeyfile") == 0)
    {
      ssh_xfree(config->pgp_public_key_file);
      config->pgp_public_key_file = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "pgpsecretkeyfile") == 0)
    {
      ssh_xfree(config->pgp_secret_key_file);
      config->pgp_secret_key_file = ssh_xstrdup(val);
      return FALSE;
    }

  if (strcmp(var, "forcepttyallocation") == 0)
    {
      config->force_ptty_allocation = bool_val;
      return FALSE;
    }

  if (strcmp(var, "verbosemode") == 0)
    {
      config->verbose_mode = bool_val;
      if (bool_val)
        ssh_debug_set_level_string("2");
      return FALSE;
    }

  if (strcmp(var, "quietmode") == 0)
    {
      config->quiet_mode = bool_val;
      return FALSE;
    }

  if (strcmp(var, "fascistlogging") == 0)
    {
      config->fascist_logging = bool_val;
      return FALSE;
    }

  if (strcmp(var, "keepalive") == 0)
    {
      config->keep_alive = bool_val;
      return FALSE;
    }

  if (strcmp(var, "nodelay") == 0)
    {
      config->no_delay = bool_val;
      return FALSE;
    }

  if (strcmp(var, "ssh1compatibility") == 0)
    {
      config->ssh1compatibility = bool_val;
      return FALSE;
    }

  /* XXX not yet implemented */
  if (strcmp(var, "rekeyintervalbytes") == 0)
    {
      config->rekey_interval_bytes = strtoul(val, NULL, 0);
      return FALSE;
    }

  if (strcmp(var, "rekeyintervalseconds") == 0)
    {
      config->rekey_interval_seconds = strtoul(val, NULL, 0);
      return FALSE;
    }








































































































  /* Client uses socks for connections.
     Server uses socks for LDAP CRL queries. */
  if (strcmp(var, "socksserver") == 0)
    {
      ssh_xfree(config->socks_server);
      config->socks_server = ssh_xstrdup(val);
      return FALSE;
    }

  /* for client only */

  if (config->client == TRUE)
    {
      if (strcmp(var, "host") == 0)
        {
          ssh_xfree(config->host_to_connect);
          config->host_to_connect = ssh_xstrdup(val);
          return FALSE;
        }
      if (strcmp(var, "user") == 0)
        {
          ssh_xfree(config->login_as_user);
          config->login_as_user = ssh_xstrdup(val);
          return FALSE;
        }
      if (strcmp(var, "compression") == 0)
        {
          config->compression = bool_val;
          return FALSE;
        }
      if (strcmp(var, "compressionlevel") == 0)
        {
          if (num > 9 || num < 1)
            {
              ssh_warning("Invalid value for CompressionLevel.");
              return TRUE;
            }

          config->compression = num;
          return FALSE;
        }

      if (strcmp(var, "fallbacktorsh") == 0)
        {
          config->fall_back_to_rsh = bool_val;
          return FALSE;
        }

      if (strcmp(var, "usersh") == 0)
        {
          config->use_rsh = bool_val;
          return FALSE;
        }

      if (strcmp(var, "batchmode") == 0)
        {
          config->batch_mode = bool_val;
          return FALSE;
        }

      if (strcmp(var, "authenticationnotify") == 0)
        {
          config->authentication_notify = bool_val;
          return FALSE;
        }

      if (strcmp(var, "stricthostkeychecking") == 0)
        {
          if (strcasecmp(val, "ask") == 0)
            config->strict_host_key_checking = SSH_STRICT_HOSTKEY_CHECKING_ASK;
          else
            config->strict_host_key_checking = bool_val;

          return FALSE;
        }

      if (strcmp(var, "escapechar") == 0)
        {
          ssh_xfree(config->escape_char);
          if (strcmp(val, "none") == 0)
            config->escape_char = ssh_xstrdup("");
          else
            config->escape_char = ssh_xstrdup(val);

          return FALSE;
        }

      if (strcmp(var, "passwordprompt") == 0)
        {
          ssh_xfree(config->password_prompt);
          config->password_prompt = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "gobackground") == 0)
        {
          if (strcasecmp(val, "oneshot") == 0)
            {
              config->go_background = TRUE;
              config->one_shot_forwarding = TRUE;
              return FALSE;
            }
          config->go_background = bool_val;
          return FALSE;
        }

      if (strcmp(var, "dontreadstdin") == 0)
        {
          config->dont_read_stdin = bool_val;
          return FALSE;
        }

      if (strcmp(var, "ssh1path") == 0)
        {
          ssh_xfree(config->ssh1_path);
          config->ssh1_path = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "ssh1internalemulation") == 0)
        {
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
          config->ssh1_emulation_internal = bool_val;
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
          return FALSE;
        }

      if (strcmp(var, "ssh1maskpasswordlength") == 0)
        {
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
          config->ssh1_no_ignore_packets_in_password_auth =
            bool_val ? FALSE : TRUE;
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
          return FALSE;
        }
      if (strcmp(var, "ssh1agentcompatibility") == 0)
        {
          if (strcasecmp(val, "none") == 0)
            {
              config->ssh_agent_compat = SSH_AGENT_COMPAT_NONE;
              return FALSE;
            }
          else if (strcasecmp(val, "traditional") == 0)
            {
              config->ssh_agent_compat = SSH_AGENT_COMPAT_TRADITIONAL;
              return FALSE;
            }
          else if (strcasecmp(val, "ssh2") == 0)
            {
              config->ssh_agent_compat = SSH_AGENT_COMPAT_SSH2;
              return FALSE;
            }
          else
            {
              ssh_warning("Bad Ssh1AgentCompatibility definition \"%s\"",
                          val);
              return TRUE;
            }
        }

      if (strcmp(var, "localforward") == 0)
        {
          if(ssh_parse_forward(&(config->local_forwards), val))
            {
              ssh_warning("Bad LocalForward definition \"%s\"", val);
              return TRUE;
            }
          return FALSE;
        }

      if (strcmp(var, "remoteforward") == 0)
        {
          if(ssh_parse_forward(&(config->remote_forwards), val))
            {
              ssh_warning("Bad RemoteForward definition \"%s\"", val);
              return TRUE;
            }
          return FALSE;
        }

      if (strcmp(var, "clearallforwardings") == 0)
        {
          if (bool_val == TRUE)
            {
              while (config->remote_forwards != NULL)
                {
                  current_forward = config->remote_forwards;
                  config->remote_forwards = config->remote_forwards->next;
                  ssh_xfree(current_forward);
                }

              while (config->local_forwards != NULL)
                {
                  current_forward = config->local_forwards;
                  config->local_forwards = config->local_forwards->next;
                  ssh_xfree(current_forward);
                }
            }

          return FALSE;

        }


      if (strcmp(var, "forwardx11") == 0)
        {
          config->forward_x11 = bool_val;
          return FALSE;
        }

      if (strcmp(var, "forwardagent") == 0)
        {
          config->forward_agent = bool_val;
          return FALSE;
        }

      if (strcmp(var, "sshsignerpath") == 0)
        {
          ssh_xfree(config->signer_path);
          config->signer_path = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "gatewayports") == 0)
        {
          config->gateway_ports = bool_val;
          return FALSE;
        }

      if (strcmp(var, "tryemptypassword") == 0)
        {
          config->try_empty_password = bool_val;
          return FALSE;
        }

      if (strcmp(var, "defaultdomain") == 0)
        {
          ssh_xfree(config->default_domain);
          config->default_domain = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "authenticationsuccessmsg") == 0)
        {
          config->auth_success_msg = bool_val;
          return FALSE;
        }

      if (strcmp(var, "numberofpasswordprompts") == 0)
        {
          config->number_of_password_prompts = num;
          return FALSE;
        }













































      if (strcmp(var, "setremoteenv") == 0)
        {
          if (!config->remote_env)
            {
              config->remote_env = ssh_adt_create_generic(SSH_ADT_LIST,
                                                          SSH_ADT_DESTROY,
                                                          destr_xfree,
                                                          SSH_ADT_ARGS_END);
              SSH_VERIFY(config->remote_env != NULL);
            }
          
          SSH_VERIFY(ssh_adt_insert(config->remote_env, ssh_xstrdup(val)) !=
                     SSH_ADT_INVALID);
          return FALSE;
        }

      if (strcmp(var, "debuglogfile") == 0)
        {
          ssh_xfree(config->debug_log_file_name);
          config->debug_log_file_name = ssh_xstrdup(val);
          return FALSE;
        }
    }
  else
    {
      /* These parameters are only for the server */
      if (strcmp(var, "allowhosts") == 0)
        PARSE_LIST(config->allowed_hosts);

      if (strcmp(var, "denyhosts") == 0)
        PARSE_LIST(config->denied_hosts);

      if (strcmp(var, "allowusers") == 0)
        PARSE_LIST(config->allowed_users);

      if (strcmp(var, "denyusers") == 0)
        PARSE_LIST(config->denied_users);

      if (strcmp(var, "allowgroups") == 0)
        PARSE_LIST(config->allowed_groups);

      if (strcmp(var, "denygroups") == 0)
        PARSE_LIST(config->denied_groups);

      if (strcmp(var, "allowtcpforwardingforusers") == 0)
        PARSE_LIST(config->allowed_tcp_forwarding_users);

      if (strcmp(var, "denytcpforwardingforusers") == 0)
        PARSE_LIST(config->denied_tcp_forwarding_users);
      
      if (strcmp(var, "allowtcpforwardingforgroups") == 0)
        PARSE_LIST(config->allowed_tcp_forwarding_groups);

      if (strcmp(var, "denytcpforwardingforgroups") == 0)
        PARSE_LIST(config->denied_tcp_forwarding_groups);
      
      if (strcmp(var, "allowshosts") == 0)
        PARSE_LIST(config->allowed_shosts);

      if (strcmp(var, "denyshosts") == 0)
        PARSE_LIST(config->denied_shosts);

      if (strcmp(var, "settableenvironmentvars") == 0)
        PARSE_LIST(config->settable_env_vars);

      if (strcmp(var, "requiredauthentications") == 0)
        {
          ssh_adt_clear(config->required_authentications);
          if (ssh_config_parse_list_check_validity
              (val, config->required_authentications,
               NULL, NULL, auth_param_validity, NULL))
            {
              ssh_warning("Parsing of value for "
                          "RequiredAuthentications failed.");
              return TRUE;
            }
          else
            {
              return FALSE;
            }
        }

      if (strcmp(var, "requirereversemapping") == 0)
        {
          config->require_reverse_mapping = bool_val;
          if (!bool_val)
            {
              if (strcmp(val, "disable") == 0)
                config->try_reverse_mapping = FALSE;
            }
          return FALSE;
        }

      if (strcmp(var, "userknownhosts") == 0)
        {
          config->user_known_hosts = bool_val;
          return FALSE;
        }

      if (strcmp(var, "syslogfacility") == 0)
        {
          for (i = 0; logfacilities[i].fac_name != NULL; i++)
            {
              if (strcasecmp(logfacilities[i].fac_name, val) == 0)
                {
                  config->log_facility = logfacilities[i].facility;
                  return FALSE;
                }
            }
          ssh_warning("Unknown SyslogFacility \"%s\".", val);
          return TRUE;
        }

      if (strcmp(var, "sftpsyslogfacility") == 0)
        {
          for (i = 0; logfacilities[i].fac_name != NULL; i++)
            {
              if (strcasecmp(logfacilities[i].fac_name, val) == 0)
                {
                  config->sftp_server_log_facility = logfacilities[i].facility;
                  return FALSE;
                }
            }
          ssh_warning("Unknown SftpSyslogFacility \"%s\".", val);
          return TRUE;
        }

      if (strcmp(var, "ignorerhosts") == 0)
        {
          config->ignore_rhosts = bool_val;
          return FALSE;
        }

      if (strcmp(var, "ignorerootrhosts") == 0)
        {
          config->ignore_root_rhosts = bool_val;
          return FALSE;
        }

      if (strcmp(var, "permitrootlogin") == 0)
        {
          if (strcmp(val, "nopwd") == 0)
            config->permit_root_login = SSH_ROOTLOGIN_NOPWD;
          else
            config->permit_root_login = bool_val;

          return FALSE;
        }

      if (strcmp(var, "chrootusers") == 0)
        PARSE_LIST(config->chroot_users);      

      if (strcmp(var, "chrootgroups") == 0)
        PARSE_LIST(config->chroot_groups);      

      if (strcmp(var, "permitemptypasswords") == 0)
        {
          config->permit_empty_passwords = bool_val;
          return FALSE;
        }

      if (strcmp(var, "strictmodes") == 0)
        {
          config->strict_modes = bool_val;
          return FALSE;
        }

      if (strcmp(var, "printmotd") == 0)
        {
          config->print_motd = bool_val;
          return FALSE;
        }

      if (strcmp(var, "checkmail") == 0)
        {
          config->check_mail = bool_val;
          return FALSE;
        }

      if (strcmp(var, "listenaddress") == 0)
        {
          if (!strcasecmp(val, "any"))
            {
              ssh_xfree(config->listen_address);
              config->listen_address = ssh_xstrdup(SSH_IPADDR_ANY);
            }
          else
            {    
              /* XXX some checks here */
              ssh_xfree(config->listen_address);
              config->listen_address = ssh_xstrdup(val);
            }
          
          return FALSE;
        }

      if (strcmp(var, "allowx11forwarding") == 0 ||
          strcmp(var, "x11forwarding") == 0 ||
          strcmp(var, "forwardx11") == 0)
        {
          config->x11_forwarding = bool_val;
          return FALSE;
        }

      if (strcmp(var, "allowtcpforwarding") == 0)
        {
          config->allow_tcp_forwarding = bool_val;
          return FALSE;
        }

      if (strcmp(var, "allowagentforwarding") == 0 ||
          strcmp(var, "forwardagent") == 0)
        {
          config->allow_agent_forwarding = bool_val;
          return FALSE;
        }

      if (strcmp(var, "hostkeyfile") == 0)
        {
          ssh_host_key_add_private_key(val, config->host_keys_ctx);
          return FALSE;
        }

      if (strcmp(var, "publichostkeyfile") == 0)
        {
          ssh_host_key_add_public_key(val, config->host_keys_ctx);
          return FALSE;
        }






























































































































































      
      if (strcmp(var, "logingracetime") == 0)
        {
          if (num < 0)
            {
              ssh_warning("Ignoring illegal login grace time %d",
                          num);
              return TRUE;
            }
          config->login_grace_time = num;
          return FALSE;
        }

      if (strcmp(var, "maxbroadcastspersecond") == 0)
        {
          if (num < 0)
            {
              ssh_warning("Ignoring illegal broadcasts per second value %d",
                          num);
              return TRUE;
            }
          config->broadcasts_allowed_per_second = num;
          return FALSE;
        }

      if (strcmp(var, "passwordguesses") == 0)
        {
          config->password_guesses = num;
          return FALSE;
        }

#ifdef SSH_SERVER_WITH_SECURID
      if (strcmp(var, "securidguesses") == 0)
        {
          config->securid_guesses = num;
          return FALSE;
        }
#endif /* SSH_SERVER_WITH_SECURID */

      if (strcmp(var, "maxconnections") == 0)
        {
#ifndef SSH2_MAX_CONNECTIONS
          config->max_connections = num;
#else /* ! SSH2_MAX_CONNECTIONS */
#if SSH2_MAX_CONNECTIONS == 0
          config->max_connections = num;
#else /* SSH2_MAX_CONNECTIONS == 0 */
          if ((num > SSH2_MAX_CONNECTIONS) || (num <= 0))
            {
              if (num > 0)
                ssh_warning("Maximum of %d connections requested while "
                            "only %d allowed.",
                            config->max_connections, SSH2_MAX_CONNECTIONS);
              else
                ssh_warning("Unlimited connections requested while only "
                            "%d allowed.",
                            SSH2_MAX_CONNECTIONS);
              config->max_connections = SSH2_MAX_CONNECTIONS;
            }
          else
            {
              config->max_connections = num;
            }
#endif /* SSH2_MAX_CONNECTIONS == 0 */
#endif /* ! SSH2_MAX_CONNECTIONS */















          return FALSE;
        }

      if (strcmp(var, "sshd1path") == 0)
        {
          ssh_xfree(config->ssh1_path);
          config->ssh1_path = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "sshd1configfile") == 0)
        {
          ssh_xfree(config->sshd1_config_file_path);
          config->sshd1_config_file_path = ssh_xstrdup(val);
          return FALSE;
        }
      
      if (strcmp(var, "sshpamclientpath") == 0)
        {
          ssh_xfree(config->ssh_pam_client_path);
          config->ssh_pam_client_path = ssh_xstrdup(val);
          return FALSE;
        }

      if (strcmp(var, "bannermessagefile") == 0)
        {
          ssh_xfree(config->banner_message_file_path);
          config->banner_message_file_path = ssh_xstrdup(val);
          return FALSE;
        }

      /* Parse subsystem definitions */
      if (strncmp(var, SUBSYSTEM_PREFIX, SUBSYSTEM_PREFIX_LEN) == 0)
        {
          SshADTContainer c = config->subsystems;
          SshADTHandle h;
          char *subsys_key = &var[SUBSYSTEM_PREFIX_LEN], *path = NULL;
          
          if (strlen(val) < 1)
            {
              if ((h = ssh_adt_get_handle_to_equal(c, subsys_key))
                  != SSH_ADT_INVALID)
                {
                  path = ssh_adt_map_lookup(c, h);
                  SSH_DEBUG(5, ("Removing subsystem `%s' (value: %s).",
                                subsys_key, path));
                  ssh_xfree(path);
                  ssh_adt_delete(c, h);
                }
              else
                {
                  SSH_DEBUG(5, ("Subsystem `%s' not defined, so not "
                                "removing.", subsys_key));
                }              
              return FALSE;
            }

          if ((h = ssh_adt_get_handle_to_equal(c, subsys_key))
              != SSH_ADT_INVALID)
            {
              path = ssh_adt_map_lookup(c, h);
              SSH_DEBUG(5, ("Replacing subsystem `%s' (old: %s, new: %s).",
                            subsys_key, path, val));
              ssh_xfree(path);
              ssh_adt_map_attach(c, h, ssh_xstrdup(val));
            }
          else
            {
              SSH_DEBUG(5, ("Adding subsystem `%s' (path: %s).",
                            subsys_key, val));
              h = ssh_adt_strmap_add(c, subsys_key, ssh_xstrdup(val));
              SSH_VERIFY(h != SSH_ADT_INVALID);
            }

          return FALSE;
        }

      if (strcmp(var, "idletimeout") == 0)
        {
          num = ssh_config_parse_idle_timeout(val);

          if (num == -1)
            {
              ssh_warning("Illegal IdleTimeout value '%s'.", val);
              return TRUE;
            }
          
          config->idle_timeout = num;
          return FALSE;
        }

      if (strcmp(var, "hostbasedauthforceclienthostnamednsmatch") == 0)
        {
          config->hostbased_force_client_hostname_dns_match = bool_val;
          return FALSE;
        }
      






















































    }

  ssh_warning("Unrecognized configuration parameter %s", var);
  return TRUE;
}

/* Checks whether linebuf is a heading. */
char *ssh2_config_line_heading_separator(char *linebuf)
{
  char *hlp;

  hlp = &linebuf[strlen(linebuf) - 1];
  
  if (*hlp == ':')
    return hlp;
  else
    return NULL;
}

void ssh2_config_remove_quotes(char *str)
{
  char *hlp1, *hlp2;
  int in_quotes = 0;
  int quoted = 0;

  hlp1 = hlp2 = str;

  while (*hlp1)
    {
      switch (*hlp1)
        {
        case '"':
          if (quoted)
            {
              *hlp2 = *hlp1;
              hlp2++;
            }
          in_quotes = !in_quotes;
          break;


        case '\\':
          if ((!in_quotes) || quoted)
            {
              *hlp2 = *hlp1;
              hlp2++;
              quoted = 0;
            }
          else
            {
              quoted = 1;
            }
          break;


        default:
          *hlp2 = *hlp1;
          hlp2++;
          quoted = 0;
        }
      hlp1++;
    }
  *hlp2 = '\0';
}

/* Parse a configuration/authorization file into an array of variable
   name <-> value pairs. Return the number of variables (if no variables
   could be read, returns 0) or -1 if file couldn't be opened. Pointers
   to tables of pointers to null-terminated strings are placed at
   *variables and *values. */
int ssh2_parse_config(SshUser user, const char *instance, const char *path,
                      char ***variables, char ***values,
                      SshMetaConfig metaconfig,
                      Boolean remove_quotes)
{
  int i = 0;

  SshUserFile f;
  char **vars, **vals, *var = NULL, *val = NULL, *hlp;
  char linebuf[1024];
  size_t n;
  int j, k;
  int line, ch;
  Boolean matching;
  Boolean doing_metaconfig = TRUE;
  Boolean metaconf_initialized = FALSE;
  int version_major = 1, version_minor = 0;
  /* The default, if version is old, this is set ZSH_FILEGLOB later. */
  SshRegexSyntax rex_syntax = SSH_REGEX_SYNTAX_EGREP;
  const char *metaconf_rex[] = { "^#.*VERSION~s+",
                                 "^#[#~s]+REGEX-SYNTAX+~s+$",
                                 "[~d.]+",
                                 "~w+",
                                 NULL };
  int num_metaconf_rex = 0;
#define VERSION_TOKEN 0
#define REGEX_SYNTAX_TOKEN 1
#define DIGIT_TOKEN 2
#define WORD_TOKEN 3  
  SshDLexer lexer;

  SSH_PRECOND(variables);
  SSH_PRECOND(values);
  
  while (metaconf_rex[num_metaconf_rex] != NULL)
    num_metaconf_rex++;

  lexer = ssh_dlex_create(ssh_app_get_global_regex_context(),
                          metaconf_rex,
                          num_metaconf_rex,
                          SSH_REGEX_SYNTAX_SSH,
                          SSH_DLEX_FIRST_MATCH);

  ssh_userfile_init(ssh_user_name(user), ssh_user_uid(user),
                    ssh_user_gid(user), NULL, NULL);
  
  if ((f = ssh_userfile_open
       (

        user == NULL ? getuid() :

        ssh_user_uid(user), path, O_RDONLY, 0755)) ==
      NULL)
    {
      SSH_TRACE(0, ("Unable to open %s", path));
      ssh_dlex_destroy(lexer);
      ssh_userfile_uninit();
      return -1;
    }

  line = 0;
  i = 0;
  n = 16;
  matching = TRUE;
  vars = ssh_xcalloc(n, sizeof(char *));
  vals = ssh_xcalloc(n, sizeof(char *));

  while (ssh_userfile_gets(linebuf, sizeof(linebuf) - 1, f) != NULL)
    {
      line++;
      
      /* skip the starting white spaces and comment lines */
      for (j = 0;; j++)
        {
          ch = linebuf[j];
          if (ch == '\0' || ch == '#')
            {
              if (ch == '#' && doing_metaconfig)
                {
                  char *ptr = linebuf, *end = &linebuf[strlen(linebuf)];
                  int previous_token = -1;
                  int len = 0, token = 0;
                  Boolean lex_ret = FALSE;
                  
                  while (ptr < end &&
                         (lex_ret = ssh_dlex_next(lexer, ptr, end - ptr,
                                             &len, &token)) == TRUE)
                    {                      
                      switch (token)
                        {
                        case VERSION_TOKEN:
                          if (line != 1)
                            {
                              ssh_warning("%s: Configuration version number "
                                          "at wrong line (%d) (must be "
                                          "first)",
                                          path, line);
                              doing_metaconfig = FALSE;
                              goto skip;
                            }
                          previous_token = VERSION_TOKEN;
                          break;
                        case REGEX_SYNTAX_TOKEN:
                          previous_token = REGEX_SYNTAX_TOKEN;
                          break;
                        case DIGIT_TOKEN:
                        case WORD_TOKEN:
                          if (previous_token == -1)
                            goto metaconfig_error;
                          
                          if (previous_token == VERSION_TOKEN &&
                              token == DIGIT_TOKEN)
                            {
                              char *temp = ssh_xstrdup(ptr), *temp2 = NULL;
                              temp[len] = '\0';
                              if ((temp2 = strchr(temp, '.')) != NULL)
                                {
                                  *temp2 = '\0';
                                  temp2++;
                                  version_major = atoi(temp);
                                  if (strlen(temp2))
                                    version_minor = atoi(temp2);
                                  else
                                    version_minor = 0;
                                }
                              else
                                {
                                  version_major = atoi(temp);
                                  version_minor = 0;
                                }

                              ssh_xfree(temp);

                              if (version_major < 1 || version_minor < 0)
                                {
                                  SSH_TRACE(3, ("%s: Invalid version number "
                                                "(%d.%d).", path,
                                                version_major, version_minor));
                                  goto metaconfig_error;
                                }
                              metaconf_initialized = TRUE;
                              SSH_TRACE(3, ("Configuration file `%s' version "
                                            "%d.%d.", path, version_major,
                                            version_minor));
                            }
                          else if (previous_token == REGEX_SYNTAX_TOKEN &&
                                   token == WORD_TOKEN)
                            {
                              if (!strncasecmp(ptr, "ssh", len))
                                rex_syntax = SSH_REGEX_SYNTAX_SSH;
                              else if (!strncasecmp(ptr, "egrep", len))
                                rex_syntax = SSH_REGEX_SYNTAX_EGREP;
                              else if (!strncasecmp(ptr, "zsh_fileglob",
                                                    len) ||
                                       !strncasecmp(ptr, "traditional", len))
                                rex_syntax = SSH_REGEX_SYNTAX_ZSH_FILEGLOB;
                              else
                                goto metaconfig_error;

                              SSH_TRACE(3,
                                        ("Metaconfig specifies regex style "
                                         "'%s'.",
                                         rex_syntax == SSH_REGEX_SYNTAX_SSH ?
                                         "SSH" :
                                         rex_syntax == SSH_REGEX_SYNTAX_EGREP ?
                                         "EGREP" :
                                         "ZSH_FILEGLOB"));
                            }
                          else
                            {
                              goto metaconfig_error;
                            }
                          break;
                        default:
                        metaconfig_error:
                          ssh_warning("%s: Invalid metaconfig line (%d: %s)",
                                      path, line, linebuf);
                          doing_metaconfig = FALSE;
                          goto skip;
                        }
                      ptr += len;
                    }
                  if (lex_ret == FALSE)
                    {
                      /* This error is ignored, as we reach this with
                         newlines etc. which we don't have a pattern
                         for. */
                      goto skip;
                    }
                }
              else
                {
                  goto skip;
                }
            }
          if (!isspace(ch))
            break;
        }

      /* Remove trailing whitespace. */
      for (k = strlen(linebuf) - 1; k >= 0; k--)
        {
          if (!isspace(linebuf[k]))
            {
              linebuf[k + 1] = '\0';
              break;
            }
        }

      if (k <= 0)
        goto skip;
      
      if (version_major == 1 && version_minor == 0 && !metaconf_initialized)
        {
          /* Here we set defaults for metaconfig parameters, if
             conf version is too old (or version is missing) . */
          SSH_TRACE(3, ("Configuration file `%s' is old-style. (1.0)", path));
          doing_metaconfig = FALSE;
          rex_syntax = SSH_REGEX_SYNTAX_ZSH_FILEGLOB;
          metaconf_initialized = TRUE;
        }

      if (doing_metaconfig)
        {
          doing_metaconfig = FALSE;
          metaconf_initialized = TRUE;
        }
      
      /* see if this is a heading or not.. */
      if ((hlp = ssh2_config_line_heading_separator(&linebuf[j])) != NULL)
        {
          SshRegexMatcher matcher = NULL;
          SSH_DEBUG(3, ("Found heading '%s'.", &linebuf[j]));
          *hlp = '\0';

          matcher = ssh_regex_create(ssh_app_get_global_regex_context(),
                                     &linebuf[j],
                                     rex_syntax);

          if (!matcher)
            {
              ssh_warning("%s: %d: Illegal regex pattern '%s'.", path, line,
                          &linebuf[j]);
              matching = FALSE;
            }
          else
            {
              matching = ssh_regex_match_cstr(matcher, instance);
              if (matching)
                {
                  int m_index = 0;
                  size_t m_len = 0;
                  SSH_DEBUG(3, ("'%s' matches (at least partially) with "
                                "instance '%s'", &linebuf[j], instance));
                  
                  (void) ssh_regex_access_submatch(matcher, 0, &m_index,
                                                   &m_len);
                  if (m_index != 0 || m_len != strlen(instance))
                    {
                      /* Did not match whole instance. */
                      SSH_DEBUG(3, ("'%s' didn't match instance '%s' "
                                    "completely", &linebuf[j], instance));
                      matching = FALSE;
                    }
                }
              ssh_regex_free(matcher);
            }
          continue;
        }

      /* ok, it must be a variable definition */
      if (!matching)
        goto skip;

      if (ssh_config_parse_line(linebuf, &var, &val))
        {
          ssh_warning("%s: %d: parsing line failed.", path, line);
          goto skip;
        }
      
      vars[i] = var;
      vals[i] = val;
      
      if (remove_quotes)
        ssh2_config_remove_quotes(vals[i]);

      i++;

      /* get more space if needed */
      if (i >= n)
        {
          n = 2 * n;
          vars = ssh_xrealloc(vars, n * sizeof(char *));
          vals = ssh_xrealloc(vals, n * sizeof(char *));
        }
    skip:
      ;
    }

  ssh_userfile_close(f);
  *variables = vars;
  *values = vals;
  ssh_dlex_destroy(lexer);
  if (metaconfig)
    {
      metaconfig->version_major = version_major;
      metaconfig->version_minor = version_minor;
      metaconfig->regex_syntax = rex_syntax;
    }
  
  ssh_userfile_uninit();

  return i;
}

  
/* Parse a line of input */
Boolean ssh_config_parse_line(char *line, char **var, char **val)
{
  SshRegexMatcher r;
  char *rex_str = "^~s*([~S-=]+){~s*=~s*|~s+}(.*)$";
  char *n_var = NULL, *n_val = NULL;
  int i = 0;
  
  r = ssh_regex_create(ssh_app_get_global_regex_context(),
                       rex_str, SSH_REGEX_SYNTAX_SSH);

  SSH_VERIFY(r != NULL);

  if (!ssh_regex_match_cstr(r, line))
    {
      SSH_DEBUG(4, ("Regex didn't match."));
      ssh_regex_free(r);
      return TRUE;
    }

  n_var = ssh_regex_get_submatch(r, 1);
  n_val = ssh_regex_get_submatch(r, 2);

  SSH_DEBUG(6, ("n_var = `%s', n_val = `%s'", n_var, n_val));
  
  SSH_ASSERT(n_var != NULL);
  SSH_ASSERT(n_val != NULL);

  for (i = 0; i < strlen(n_var); i++)
    if (isalnum(n_var[i]))
      n_var[i] = tolower(n_var[i]);
  
  *var = ssh_xstrdup(n_var);
  *val = ssh_xstrdup(n_val);

  ssh_regex_free(r);

  return FALSE;  
}

/* Parse line of input and set the parameters in the SshConfig struct.
   Returns TRUE on failure, FALSE otherwise. */
Boolean ssh_config_parse_line_and_set_params(SshConfig config, char *line)
{
  char *var = NULL, *val = NULL;
  SshMetaConfigStruct metaconfig;
  Boolean ret = TRUE;
  
  metaconfig.version_major = 1;
  metaconfig.version_minor = 1;
  metaconfig.regex_syntax = SSH_REGEX_SYNTAX_EGREP;
  
  if (ssh_config_parse_line(line, &var, &val))
    return TRUE;
  
  ret = ssh_config_set_parameter(config, &metaconfig, var, val);

  ssh_xfree(var);
  ssh_xfree(val);
  
  return ret;
}

/* Reads config data from the given file.  Returns FALSE if an error
   occurs (displays error messages with ssh_warning.) */


Boolean ssh_config_read_file(SshUser user, SshConfig config,
                             char *instance, const char *filename)
{
  SshUser user_data;
  SshMetaConfigStruct metaconfig;
  char **vars, **vals;
  int i, n;

  if (filename == NULL || strlen(filename) == 0)
    return FALSE;

  if (user == NULL)
    user_data = ssh_user_initialize(NULL, FALSE);
  else
    user_data = user;

  /* try to read in the file */
  instance = (instance ? instance : "");

  memset(&metaconfig, 0, sizeof(metaconfig));
  
  n = ssh2_parse_config(user_data, instance, filename, &vars, &vals,
                        &metaconfig, TRUE);

  if (n < 0)
    {
      if (user_data != user)
        ssh_user_free(user_data, FALSE);
      return FALSE;
    }

  /* ok, now fill in the fields */

  for (i = 0; i < n; i++)
    ssh_config_set_parameter(config, &metaconfig, vars[i], vals[i]);

  ssh_free_varsvals(n, vars, vals);

  if (user_data != user)
    ssh_user_free(user_data, FALSE);

  return TRUE;
}


/* Parse forwarding definitions. Format is
   [protocol/][localhost:]port:remotehost:remoteport */
Boolean ssh_parse_forward(SshForward *fws, char *s)
{
  SshForward fw;
  char *lp, *lh, *rh, *rp, *pr;
  size_t l;
  SshRegexContext rc = ssh_app_get_global_regex_context();
  SshRegexMatcher r;

  l = strlen(s);
  r = ssh_regex_create(
             rc,
             "^([^:/]+)/\\[([^\\]]+)\\]:([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
             SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 1 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 2 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/\\[([^\\]]+)\\]:([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 3 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 4 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 5 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 6 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^\\[([^\\]]+)\\]:([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 7 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 8 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^\\[([^\\]]+)\\]:([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 9 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 10 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 11 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 12 in forward parser.");
    }
  SSH_DEBUG(5, ("forward parsing failed"));
  return TRUE;

 done:
  SSH_DEBUG(5, ("fw: lh=\"%s\" lp=\"%s\" rh=\"%s\" rp=\"%s\" pr=\"%s\"",
                lh, lp, rh, rp, pr));
  fw = ssh_xcalloc(1, sizeof (*fw));
  fw->local_addr = lh;
  fw->port = lp;
  fw->connect_to_host = rh;
  fw->connect_to_port = rp;
  fw->protocol = pr;
  fw->next = *fws;
  *fws = fw;
  return FALSE;
}

int ssh_config_parse_idle_timeout(const char *value)
{
  char ch, *str;
  int num;
  
  str = ssh_xstrdup(value);
          
  ch = str[strlen(str) - 1];

  if (!isdigit(ch))
    {
      str[strlen(str) - 1] = '\0';
      ch = tolower(ch);
    }
          
  num = atoi(str);
          
  if (num < 0)
    {
      num = -1;
      goto error;
    }

  if (!isdigit(ch))
    {
      switch (ch)
        {
        case 'w': /* Weeks. */
          num *= 7 * 24 * 60 * 60;
          break;
        case 'd': /* Days. */
          num *= 24 * 60 * 60;
          break;
        case 'h': /* Hours. */
          num *= 60 * 60;
          break;
        case 'm': /* Minutes. */
          num *= 60;
          break;
        case 's': /* Seconds. */
          /* Nothing. */
          break;
        default:
          num = -1;
        }
    }

 error:
  ssh_xfree(str);
  return num;
}
