/*
  sshd-check-conf.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Description: Program to check configuration of sshd2.

  Created: Sun May 27 12:52:20 2001

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#include "ssh2includes.h"
#include "sshconfig.h"
#include "sshgetopt.h"
#include "sshuserfiles.h"
#include "sshdsprintf.h"
#include "ssheloop.h"
#include "sshregex.h"
#include "sshinet.h"
#include "sshtcp.h"
#include "sshappcommon.h"
#include "auths-common.h"
#include "sigchld.h"
#include "ssholdfsm.h"
#include "sshreadline.h"
#include "sshtty.h"
#include "sshfdstream.h"

#define SSH_DEBUG_MODULE "SshdCheckConf"

/* For SSH_ADT_DESTROY. */
static void destr_xfree(void *obj, void *context)
{
  ssh_xfree(obj);
}

void check_conf(const char *user, const char *remote_name,
                const char *remote_ip, SshConfig config)
{
  const char *ip = NULL;
  SshUser uc = ssh_user_initialize(user, FALSE);

  if (uc == NULL)
    {
      Boolean nan = FALSE;
      char *cp;

      SSH_TRACE(2, ("User couldn't be initialized with name, assuming "
                    "given username is a UID.."));

      for (cp = (char *)user; *cp; cp++)
        if (!isdigit(*cp))
          nan = TRUE;

      if (nan)
        {
          SSH_TRACE(2, ("Given string is neither a UID or a username."));
        }
      else
        {
          uc = ssh_user_initialize_with_uid(atoi(user), FALSE);
          if (uc)
            user = ssh_user_name(uc);
          else
            SSH_TRACE(2, ("User still couldn't be initialized, user probably "
                          "doesn't exist."));
        }
    }

  SSH_PRECOND(user);
  SSH_PRECOND(remote_name);
  SSH_PRECOND(remote_ip);
  SSH_PRECOND(config);

  if (ssh_inet_is_valid_ip_address(remote_ip))
    ip = remote_ip;
  else
    ip = "UNKNOWN";

  ssh_informational("\r\nVerifying %s@%s[%s]...\r\n", user, remote_name, ip);

  if (ssh_server_auth_check_host_generic(remote_name, ip,
                                         config->denied_hosts,
                                         config->allowed_hosts,
                                         config->require_reverse_mapping))
    ssh_informational("  Logins from %s[%s] denied.\r\n", remote_name,
                      ip);
  else
    ssh_informational("  Logins from %s[%s] allowed.\r\n", remote_name,
                      ip);

  if (ssh_server_auth_check_host_generic(remote_name, ip,
                                         config->denied_shosts,
                                         config->allowed_shosts,
                                         FALSE))
    ssh_informational("  Hostbased cannot be used from %s[%s].\r\n",
                      remote_name, ip);
  else
    ssh_informational("  Hostbased can be used from %s[%s].\r\n", remote_name,
                      ip);

  if (ssh_server_auth_check_user_generic(uc, user,
                                         "is allowed to login",
                                         "is denied login",
                                         remote_name, ip,
                                         config->allowed_users,
                                         config->denied_users,
                                         config->allowed_groups,
                                         config->denied_groups))
    ssh_informational("  Login by user %s denied.\r\n", user);
  else
    ssh_informational("  Login by user %s allowed.\r\n", user);

  if (ssh_server_auth_check_user_generic(uc, user,
                                         "won't be chrooted",
                                         "will be chrooted",
                                         remote_name, ip,
                                         NULL,
                                         config->chroot_users,
                                         NULL,
                                         config->chroot_groups))
    ssh_informational("  User %s will be chrooted.\r\n", user);
  else
    ssh_informational("  User %s will not be chrooted.\r\n", user);

  if (ssh_server_auth_check_user_generic(uc, user,
                                         "is allowed TCP forwarding",
                                         "is denied TCP forwarding",
                                         remote_name, ip,
                                         config->allowed_tcp_forwarding_users,
                                         config->denied_tcp_forwarding_users,
                                         config->allowed_tcp_forwarding_groups,
                                         config->denied_tcp_forwarding_groups))
    ssh_informational("  TCP forwarding by user %s denied.\r\n", user);
  else
    ssh_informational("  TCP forwarding by user %s allowed.\r\n", user);

}

/**************************************************************/
/* The state machine */

/* Structures. */
typedef struct CheckGDataRec
{
  SshOldFSM fsm;
  SshConfig config;
  char *current;
} CheckGDataStruct, *CheckGData;

typedef struct StepTDataRec
{
  SshADTContainer list;
  SshADTHandle h;
} StepTDataStruct, *StepTData;

typedef struct ReadlineTDataRec
{
  SshOperationHandle op_handle;
  char *prompt;
  char *line;
  SshStream stdio_stream;
  SshReadLineCtx rl_ctx;
} ReadlineTDataStruct, *ReadlineTData;

typedef struct CheckTDataRec
{
  SshOperationHandle op_handle;
  char *user;
  char *host_or_ip;
  Boolean ip_lookup;
  char *lookup_result;
  SshOldFSMThread parent;
} CheckTDataStruct, *CheckTData;

/* Destructors. */
void check_gdata_destructor(void *cgdata)
{
  CheckGData gdata = (CheckGData)cgdata;

  if (gdata->config)
    ssh_config_free(gdata->config);

  ssh_xfree(gdata->current);
}

void check_thread_destructor(void *ctdata)
{
  CheckTData tdata = (CheckTData) ctdata;

  if (tdata->op_handle)
    {
      ssh_operation_abort(tdata->op_handle);
      tdata->op_handle = NULL;
    }
  ssh_xfree(tdata->user);
  ssh_xfree(tdata->host_or_ip);
  ssh_xfree(tdata->lookup_result);
}

void step_thread_destructor(void *ctdata)
{
  StepTData tdata = (StepTData) ctdata;
  if (tdata->list)
    ssh_adt_destroy(tdata->list);
}

void readline_thread_destructor(void *ctdata)
{
  ReadlineTData tdata = (ReadlineTData) ctdata;

  if (tdata->op_handle)
    ssh_operation_abort(tdata->op_handle);
  ssh_xfree(tdata->line);
  ssh_stream_destroy(tdata->stdio_stream);
  ssh_readline_eloop_unitialize(tdata->rl_ctx);
  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);
}

SSH_OLDFSM_STEP(check_readline);
SSH_OLDFSM_STEP(check_readline_process);
SSH_OLDFSM_STEP(check_step_list);
SSH_OLDFSM_STEP(check_begin);
SSH_OLDFSM_STEP(check_lookup);
SSH_OLDFSM_STEP(check_complete);

SshOldFSMStateMapItemStruct check_states[] =
{
  { "check_step_list", "Step forward in list.", check_step_list },

  { "check_readline", "Read a line.", check_readline },
  { "check_readline_process", "Process read line.", check_readline_process },

  { "check_begin", "Initialize check.", check_begin },
  { "check_lookup", "Start lookup of item.", check_lookup },
  { "check_complete", "Perform check.", check_complete }
};

void rl_callback(const char *line, void *context)
{
  SshOldFSMThread thread = (SshOldFSMThread)context;
  SSH_OLDFSM_TDATA(ReadlineTData);

  ssh_xfree(tdata->line);

  if (line == NULL)
    tdata->line = NULL;
  else
    tdata->line = ssh_xstrdup(line);

  SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_OLDFSM_STEP(check_readline)
{
  SSH_OLDFSM_TDATA(ReadlineTData);
  SSH_OLDFSM_SET_NEXT("check_readline_process");
  SSH_OLDFSM_ASYNC_CALL(ssh_readline_eloop(tdata->prompt, "", tdata->rl_ctx,
                                        rl_callback, thread));
}

SSH_OLDFSM_STEP(check_readline_process)
{
  char *cp = NULL, *p = NULL;
  SshOldFSMThread ct;
  CheckTData child_tdata = NULL;
  SSH_OLDFSM_DATA(CheckGData, ReadlineTData);
  SSH_OLDFSM_SET_NEXT("check_readline");

  SSH_TRACE(2, ("Got line `%s'.", tdata->line ? tdata->line : "NULL"));

  if (tdata->line == NULL)
    {
      ssh_oldfsm_destroy(gdata->fsm);
      return SSH_OLDFSM_FINISH;
    }

  cp = tdata->line;

  /* Why is it that stripping whitespace always looks (and is) a
     terrible kludge? Here we go... */
  for (cp = tdata->line; *cp != '\0' && isspace(*cp); cp++)
    ;

  if (strlen(cp) <= 0)
    return SSH_OLDFSM_CONTINUE;

  for (p = cp + strlen(cp) - 1; p > cp && isspace(*p); p--)
    ;

  *(p + 1) = '\0';
  /* You may look again. */

  if (strlen(cp) <= 0)
    return SSH_OLDFSM_CONTINUE;

  SSH_TRACE(4, ("stripped line: `%s'", cp));

  if (strcmp(cp, "quit") == 0)
    {
      ssh_oldfsm_destroy(gdata->fsm);
      return SSH_OLDFSM_FINISH;
    }
  else
    {
      ssh_xfree(gdata->current);
      gdata->current = ssh_xstrdup(cp);

      ct = SSH_OLDFSM_FORK(sizeof(*child_tdata),
                        "check_begin",
                        NULL,
                        check_thread_destructor);

      child_tdata = ssh_oldfsm_get_tdata(ct);
      memset(child_tdata, 0, sizeof(*child_tdata));

      child_tdata->parent = thread;

      return SSH_OLDFSM_SUSPENDED;
    }

  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(check_step_list)
{
  SshOldFSMThread ct;
  CheckTData child_tdata = NULL;
  SSH_OLDFSM_DATA(CheckGData, StepTData);

  if (tdata->h == NULL)
    tdata->h = ssh_adt_enumerate_start(tdata->list);
  else
    tdata->h = ssh_adt_enumerate_next(tdata->list, tdata->h);

  if (tdata->h == SSH_ADT_INVALID)
    {
      SSH_TRACE(1, ("List at end."));
      ssh_oldfsm_destroy(gdata->fsm);
      return SSH_OLDFSM_FINISH;
    }

  ssh_xfree(gdata->current);
  gdata->current = ssh_xstrdup(ssh_adt_get(tdata->list, tdata->h));

  ct = SSH_OLDFSM_FORK(sizeof(*child_tdata),
                    "check_begin",
                    NULL,
                    check_thread_destructor);

  child_tdata = ssh_oldfsm_get_tdata(ct);
  memset(child_tdata, 0, sizeof(*child_tdata));

  child_tdata->parent = thread;

  return SSH_OLDFSM_SUSPENDED;
}

SSH_OLDFSM_STEP(check_begin)
{
  char *current = NULL, *user = NULL, *host = NULL;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
  SshRegexMatcher rex;
  SSH_OLDFSM_DATA(CheckGData, CheckTData);

  SSH_OLDFSM_SET_NEXT("check_lookup");

  current = gdata->current;

  rex = ssh_regex_create(rex_ctx,
                         "^{([-@]+)@}?(.*)$",
                         SSH_REGEX_SYNTAX_SSH);
  if (!rex)
    ssh_fatal("Unable to create regex for commandline parameter parsing.");

  if (ssh_regex_match_cstr(rex, current))
    {
      user = ssh_regex_get_submatch(rex, 1);
      host = ssh_regex_get_submatch(rex, 2);
    }
  else
    {
      ssh_warning("Regex didn't match `%s'", current);
      ssh_regex_free(rex);
      ssh_oldfsm_continue(tdata->parent);
      return SSH_OLDFSM_FINISH;
    }

  tdata->user = ssh_xstrdup(user ? user : "UNKNOWN");
  tdata->host_or_ip =  ssh_xstrdup(host ? host : "UNKNOWN");

  ssh_regex_free(rex);

  return SSH_OLDFSM_CONTINUE;
}

void addr_lookup_cb(SshIpError error, const char *result,
                    void *context)
{
  SshOldFSMThread thread = (void *)context;
  SSH_OLDFSM_TDATA(CheckTData);

  tdata->op_handle = NULL;

  SSH_DEBUG(4, ("Received error code %d", error));

  if (error != SSH_IP_OK)
    {
      tdata->lookup_result = NULL;
      ssh_warning("DNS lookup failed for %s.", tdata->host_or_ip);
    }
  else
    {
      tdata->lookup_result = ssh_xstrdup(result);
      SSH_TRACE(3, ("Lookup result: %s", result));
    }

  SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_OLDFSM_STEP(check_lookup)
{
  SSH_OLDFSM_TDATA(CheckTData);

  SSH_OLDFSM_SET_NEXT("check_complete");

  if (ssh_inet_is_valid_ip_address(tdata->host_or_ip))
    {
      tdata->ip_lookup = FALSE;
      /* Host name is a valid IP-address, so we need hostname. */
      SSH_OLDFSM_ASYNC_CALL(tdata->op_handle =
                         ssh_tcp_get_host_by_addr(tdata->host_or_ip,
                                                  addr_lookup_cb,
                                                  (void *)thread));
    }
  else
    {
      tdata->ip_lookup = TRUE;
      /* Host name is not a valid IP-address, so we are assuming that
         we were given valid host name, and we try to resolve host's
         IP-address(es). */
      SSH_OLDFSM_ASYNC_CALL(tdata->op_handle =
                         ssh_tcp_get_host_addrs_by_name(tdata->host_or_ip,
                                                        addr_lookup_cb,
                                                        (void *)thread));
    }
}

SSH_OLDFSM_STEP(check_complete)
{
  char *cp = NULL;
  SSH_OLDFSM_DATA(CheckGData, CheckTData);

  if (tdata->ip_lookup)
    {
      char *p = NULL;

      cp = tdata->lookup_result ? tdata->lookup_result : "UNKNOWN";

      while (cp != NULL && *cp != '\0')
        {
          p = strchr(cp, ',');
          if (p)
            {
              *p = '\0';
              p++;
            }
          check_conf(tdata->user, tdata->host_or_ip, cp, gdata->config);
          cp = p;
        }
    }
  else
    {
      cp = tdata->lookup_result ? tdata->lookup_result : "UNKNOWN";

      check_conf(tdata->user, cp, tdata->host_or_ip, gdata->config);
    }

  ssh_oldfsm_continue(tdata->parent);
  return SSH_OLDFSM_FINISH;
}

/**************************************************************/

void sigint_handler(int signal, void *context)
{
  SSH_ASSERT(signal == SIGINT);
  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);
  exit(0);
}

void fatal_callback(const char *msg, void *context)
{
  fprintf(stderr, "\r\nFATAL: %s\r\n", msg);
  ssh_leave_raw_mode(-1);
  ssh_leave_non_blocking(-1);
  exit(255);
}

void warning_callback(const char *msg, void *context)
{
  fprintf(stderr, "\r\nWarning: %s\r\n", msg);
}

void debug_callback(const char *msg, void *context)
{
  fprintf(stderr, "debug: %s\r\n", msg);
}

void sshd_check_conf_usage(const char *prog_name)
{
  ssh_informational("Usage: %s [-d debug_level] [-v] [-V] [-h] "
                    "[-f config_file] [...]\n", prog_name);
}

int main(int argc, char **argv)
{
  SshConfig config;
  char *config_file = NULL;
  SshUser user;
  struct stat st;
  Boolean readline_mode = TRUE;
  char *av0;
  SshOldFSM fsm;
  CheckGData gdata = NULL;
  SshOldFSMThread cthread = NULL;

  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];

  user = ssh_user_initialize(NULL, FALSE);

  if (!user)
    ssh_fatal("Couldn't initialize current user.");

  ssh_event_loop_initialize();

  ssh_debug_register_callbacks(fatal_callback, warning_callback,
                               debug_callback, NULL);

 config = ssh_server_create_config();

  /* Register SIGCHLD signal handler. (so that SshUserFile SIGCHLDs
     are catched.) */
  ssh_sigchld_initialize();

  while (1)
    {
      int option;

      option = ssh_getopt(argc, argv, "Vvd:f:h", NULL);

      if (option == -1)
        break;

      switch (option)
        {
        case 'V':
          ssh2_version(av0);
          return 0;
        case 'v':
          ssh_debug_set_global_level(2);
          break;
        case 'd':
          if (!ssh_optval)
            ssh_fatal("Bad -d parameter.");
          ssh_debug_set_level_string(ssh_optarg);
          break;
        case 'f':
          if (!ssh_optval)
            ssh_fatal("Illegal -f parameter.");

          if (stat(ssh_optarg, &st) < 0)
            ssh_fatal("Alternate config file \"%s\" does not exist.",
                      ssh_optarg);

          config_file = ssh_xstrdup(ssh_optarg);
          break;
        case 'h':
          sshd_check_conf_usage(av0);
          return 0;
        default:
          if (ssh_optmissarg)
            {
              ssh_fatal("Option -%c needs an argument\n", ssh_optopt);
            }
          else
            {
              ssh_fatal("Unknown option -%c\n", ssh_optopt);
            }
          ssh_config_free(config);
          sshd_check_conf_usage(av0);
          return 1;
        }
    }

  /* If we have command line arguments, process them instead of going
     to interactive mode. */
  if (ssh_optind < argc)
    {
      readline_mode = FALSE;
    }

  if (config_file == NULL)
    {
      char *conf_dir = NULL;

      if (ssh_user_uid(user) == 0 )
        {
          conf_dir = ssh_xstrdup(SSH_SERVER_DIR);
        }
      else
        {
          if ((conf_dir = ssh_userdir(user, config, FALSE)) == NULL)
            ssh_fatal("No ssh2 user directory");
        }

      ssh_dsprintf(&config_file, "%s/%s", conf_dir, SSH_SERVER_CONFIG_FILE);
      ssh_xfree(conf_dir);
    }

  SSH_TRACE(1, ("Reading configuration file `%s'.", config_file));

  if (!ssh_config_read_file(user, config, NULL, config_file))
    ssh_fatal("Failed to read config file \"%s\"", config_file);

  fsm = ssh_oldfsm_allocate(sizeof(CheckGDataStruct),
                         check_states,
                         SSH_OLDFSM_NUM_STATES(check_states),
                         check_gdata_destructor);

  gdata = ssh_oldfsm_get_gdata_fsm(fsm);
  memset(gdata, 0, sizeof(gdata));
  gdata->fsm = fsm;
  gdata->config = config;

  if (!readline_mode)
    {
      /* Start FSM, and walk through all the rest of the command
         line. */
      int i = 0;
      StepTData ctdata = NULL;

      SshADTContainer list = ssh_adt_create_generic(SSH_ADT_LIST,
                                                    SSH_ADT_DESTROY,
                                                    destr_xfree,
                                                    SSH_ADT_ARGS_END);

      for (i = ssh_optind; i < argc ; i++)
        SSH_VERIFY(ssh_adt_insert(list, ssh_xstrdup(argv[i])) !=
                   SSH_ADT_INVALID);

      cthread = ssh_oldfsm_spawn(gdata->fsm,
                              sizeof(*ctdata),
                              "check_step_list",
                              NULL,
                              step_thread_destructor);
      ctdata = ssh_oldfsm_get_tdata(cthread);
      memset(ctdata, 0, sizeof(*ctdata));

      ctdata->list = list;
    }
  else
    {
      ReadlineTData ctdata = NULL;

      /* Start FSM, and goto readline-mode. */
      cthread = ssh_oldfsm_spawn(gdata->fsm,
                              sizeof(*ctdata),
                              "check_readline",
                              NULL,
                              readline_thread_destructor);
      ctdata = ssh_oldfsm_get_tdata(cthread);
      memset(ctdata, 0, sizeof(*ctdata));

      ssh_enter_non_blocking(-1);
      ssh_enter_raw_mode(-1, TRUE);
      /* XXX */
      ssh_register_signal(SIGINT, sigint_handler, NULL);
      ctdata->stdio_stream = ssh_stream_fd_stdio();
      ctdata->rl_ctx = ssh_readline_eloop_initialize(ctdata->stdio_stream);
      ssh_dsprintf(&ctdata->prompt, "\r\n%s> ", av0);

      if (!ctdata->rl_ctx)
        {
          ssh_stream_destroy(ctdata->stdio_stream);
          ssh_leave_non_blocking(-1);
          ssh_leave_raw_mode(-1);
          ssh_fatal("Couldn't initialize ReadLine.");
        }
    }

  ssh_event_loop_run();

  ssh_app_free_global_regex_context();
  ssh_event_loop_uninitialize();
  return 0;
}
