/*
  sshfc_overwrite.h

  Author: Tuomas Pelkonen <tpelkone@ssh.com>

  Copyright (C) 1999-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions for handling file overwrite
 */

#ifndef SSHFILECOPYOVERWRITE_H
#define SSHFILECOPYOVERWRITE_H

/* these definitions are here because they are not defined on windows */
#define        SSH_UNIX_S_IFMT   0170000  /* type of file */
#define        SSH_UNIX_S_IFIFO  0010000  /* named pipe (fifo) */
#define        SSH_UNIX_S_IFCHR  0020000  /* character special */
#define        SSH_UNIX_S_IFDIR  0040000  /* directory */
#define        SSH_UNIX_S_IFBLK  0060000  /* block special */
#define        SSH_UNIX_S_IFREG  0100000  /* regular */
#define        SSH_UNIX_S_IFLNK  0120000  /* symbolic link */
#define        SSH_UNIX_S_IFSOCK 0140000  /* socket */
#define        SSH_UNIX_S_IFWHT  0160000  /* whiteout */

typedef struct SshFileCopyOverwriteFileListRec *SshFileCopyOverwriteFileList;
typedef struct SshFileCopyOverwriteFileRec *SshFileCopyOverwriteFile;
typedef struct SshFileCopyOverwriteDirRec *SshFileCopyOverwriteDir;

/* Values for overwrite_file function. */
typedef enum
{
  SSH_FC_OVERWRITE_NO,
  SSH_FC_OVERWRITE_YES,
  SSH_FC_OVERWRITE_YES_TO_ALL,
  SSH_FC_OVERWRITE_NO_TO_ALL,
  SSH_FC_OVERWRITE_CANCEL
} SshFileCopyOverwrite;


/* Callback, which is called when file to be transferred already exists */
typedef void (*SshFileCopyOverwriteCallback)(SshFileCopyOverwriteFileList list,
                                             SshFileCopyOverwriteFile file,
                                             void *context);

/* Callback, which is called when all files that will be transfered are checked */
typedef void (*SshFileCopyOverwriteDoneCallback)(void *context);

/* struct for handling overwriting */
struct SshFileCopyOverwriteFileListRec
{
  /* list which contains SshFileCopyOverwriteFile items */
  SshDlList file_list;

  /* name of the source current path that is beeing searhed */
  char *current_path;

  /* pointer to the source list that is beeing searched */
  SshDlList current_list;

  /* current destination directory that is beeing serached */
  SshFileCopyOverwriteDir current_dir;

  /* pointer to the SshFileClient */
  SshFileClient client;

  /* destination directory name */
  char *destination;

  /* destination slash type usually either "/" or "\\" */
  char *slash;

  /* callback that is called when file already exists in the destination end. */
  /* this callback MUST call ssh_file_copy_overwrite_file_name with the same  */
  /* SshFileCopyOverwriteFileList parameter that is passed to callback or */
  /* the file transfer is halted. */
  SshFileCopyOverwriteCallback overwrite_cb;

  /* callback that will be called when all files are checked */
  SshFileCopyOverwriteDoneCallback done_cb;

  /* context */
  void *context;

  /* Boolean value which tells if no files will be overwritten */
  Boolean no_to_all;
};

/* struct for handling file that will be transfered */
struct SshFileCopyOverwriteFileRec
{
  /* pointer to the parent list that containt SshFileCopyFile item */
  /* this is used when file will no be overwritten. */
  SshDlList parent_list;

  /* pointer to the SshFileCopyFile item */
  SshFileCopyFile file;

  /* name of the file. includes path */
  char *file_name;
};

/* struct for handling the current destination that is beeing searched */
struct SshFileCopyOverwriteDirRec
{
  /* list which contains file in the destination directory (type is char*) */
  SshDlList file_list;

  /* directory name */
  char *dir_name;

  /* directory handle */
  SshFileHandle handle;
};

/* following functions are just for internal use */

/* function for allocating and destroying structs */
SshFileCopyOverwriteDir ssh_fc_overwrite_dir_allocate(void);
SshFileCopyOverwriteFileList ssh_fc_overwrite_file_list_allocate(void);
SshFileCopyOverwriteFile ssh_fc_overwrite_file_allocate(void);
void ssh_fc_overwrite_dir_destroy(SshFileCopyOverwriteDir dir);
void ssh_fc_overwrite_file_list_destroy(SshFileCopyOverwriteFileList list);
void ssh_fc_overwrite_file_destroy(SshFileCopyOverwriteFile file);

/* the two function are used to get every file out from the original file_list */
void *ssh_fc_overwrite_list_mapcar_subdir(void *item, void *ctx);
void *ssh_fc_overwrite_list_mapcar(void *item, void *ctx);

/* helper functions that extract either the file name or the path name */
void ssh_fc_overwrite_get_path(char *file);
void ssh_fc_overwrite_get_filename(char *file);

/* function is called when the next location directory will be read */
void ssh_fc_overwrite_read_next_dir(void *context);

/* function is called when the next file in folder will be cheked for
   overwriting */
void ssh_fc_overwrite_next_file(SshFileCopyOverwriteFileList list);

/* this callback is called when we opened a destination directory 
   succesfully or not */
void ssh_fc_overwrite_opendir_cb(SshFileClientError error,
                              SshFileHandle handle,
                              void *context);

/* this callback is called when next file of the directory is read */
void ssh_fc_overwrite_readdir_cb(SshFileClientError error,
                              const char *name,
                              const char *long_name,
                              SshFileAttributes attrs,
                              void *context);

/* this callback is called when destination directory is closed */
void ssh_fc_overwrite_file_status_cb(SshFileClientError error,
                                     void *context);
/* actual interface starts here */

/* the function that start the checking of file that already exists in the
   destination end. 'client' is the SshFileClient of the destination.
   'file_list' is the same file list that ssh_file_copy_recurse_dirs returns
   in the callback. 'file_list' contains all the file that will be transfered.
   Type of the items in the 'file_list' is SshFileCopyFileListItem.
   'destination' is the name of the destination directory that files will be
   transfered to. It must included the whole path from the root down. 'slash'
   is use to separate windows and unix. It can be "\\" or "/". 'overwrite_cb'
   is called when file already exists. This callback MUST call
   ssh_file_copy_overwrite_file with the same 'file' and 'list' in order to
   continue checking the files. 'done_cb' is called when all file are checked.
   'context' is the context which user can defined and the same context is
   passed in the callbacks. */   
void ssh_file_copy_overwrite(SshFileClient client,
                             SshDlList file_list,
                             const char *destination,
                             const char *slash,
                             SshFileCopyOverwriteCallback overwrite_cb,
                             SshFileCopyOverwriteDoneCallback done_cb,
                             void *context);

/* This function determines what to do with the file that already exists.
   'list' must be the same list that is passed in the callback. 'file' must
   be the same file that is passed in the callback. 'overwrite' determines
   what to do with file. Possible options are overwrite it, do not overwrite
   it, overwrite everything or cancel the whole file transfer */
void ssh_file_copy_overwrite_file(SshFileCopyOverwriteFileList list, 
                                  SshFileCopyOverwriteFile file, 
                                  SshFileCopyOverwrite overwrite);

/* Returns the name of the 'file' */
const char *ssh_file_copy_overwrite_file_name(SshFileCopyOverwriteFile file);

#endif /* SSHFILECOPYOVERWRITE_H */
