/*
  sshfc_trcheck.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  States to check which attributes of a file can be changed with
  setstat.

  Created Wed Jan 17 11:21:34 2001.
*/

#include "ssh2includes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshfc_transferi.h"

#define SSH_DEBUG_MODULE "SshFCTransferCheck"

SSH_OLDFSM_STEP(transfer_check_setstat_flags)
{
  SSH_OLDFSM_GDATA(SshFileCopyTransferContext);
  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_permissions");
  gdata->attributes_flags = 0;
  return SSH_OLDFSM_CONTINUE;
}

void transfer_check_setstat_stat_source_cb(SshFileClientError error,
                                           SshFileAttributes attributes,
                                           void *context)
{
  SshOldFSMThread thread = (SshOldFSMThread) context;
  SshFileCopyError error_code;
  char *error_string;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  tdata->op_handle = NULL;

  switch (error)
    {
    case SSH_FX_OK:
      ssh_file_copy_file_register_attributes
        (tdata->source_file, ssh_file_attributes_dup(attributes));

      SSH_ASSERT(attributes->flags & SSH_FILEXFER_ATTR_PERMISSIONS);

      break;
    case SSH_FX_EOF:
      SSH_NOTREACHED;
      break;
    case SSH_FX_OP_UNSUPPORTED:
      /* This op should most definitely be supported. */
      SSH_NOTREACHED;

    case SSH_FX_NO_SUCH_FILE:
      error_code = SSH_FC_ERROR_NO_SUCH_FILE;
      ssh_dsprintf(&error_string, "%s: No such file or directory. ",
                   tdata->source_file_name);
      goto error_common;
    case SSH_FX_PERMISSION_DENIED:
      error_code = SSH_FC_ERROR_PERMISSION_DENIED;
      ssh_dsprintf(&error_string, "%s: Permission denied.",
                   tdata->source_file_name);
      goto error_common;
    case SSH_FX_FAILURE:
      /* Some error occurred. Drop file, and report an error. */
      error_code = SSH_FC_ERROR_FAILURE;
      ssh_dsprintf(&error_string, "%s: Undefined error occurred.",
                   tdata->source_file_name);
    error_common:
      /* Report error. */
      (*gdata->error_callback)(error_code, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      ssh_dllist_fw(tdata->current_location->file_list, 1);
      SSH_OLDFSM_SET_NEXT("fct_wake_parent_and_finish");
      break;
    case SSH_FX_BAD_MESSAGE:
      /* XXX we should exit the state machine, close connections
         etc. */
      (*gdata->completion_callback)(SSH_FC_ERROR_PROTOCOL_MISMATCH,
                                    "File transfer protocol mismatch.",
                                    gdata->callback_context);


      /* XXX Free everything, kill threads. */
      SSH_NOTREACHED;
      return;
    case SSH_FX_NO_CONNECTION:
      SSH_OLDFSM_SET_NEXT("fct_check_setstat_stat_source");
      SSH_TRACE(2, ("No connection yet. Waiting..."));
      ssh_register_timeout(SSH_FC_CONNECTION_WAIT_TIMEOUT,
                           connection_wait_timeout,
                           (void *)thread);
      return;
    case SSH_FX_CONNECTION_LOST:
      SSH_OLDFSM_SET_NEXT("fct_check_setstat_stat_source");
      SSH_TRACE(2, ("Connection down, re-establishing..."));
      gdata->source_conn_flags = SSH_FCC_DOWN;
      break;

      /* */
    case SSH_FX_OUT_OF_MEMORY:
      break;
    }

  SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_OLDFSM_STEP(transfer_check_setstat_stat_source)
{
  const char *basedir;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  SSH_PRECOND(tdata->source);

  SSH_FCT_CHECK_SRC_CONN;

  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_permissions");
  ssh_xfree(tdata->source_file_name);
  basedir = ssh_file_copy_file_get_name(tdata->source_dir);
  ssh_dsprintf(&tdata->source_file_name, "%s%s%s",
               *basedir ? basedir : "",
               *basedir && basedir[strlen(basedir) - 1] != '/' ? "/" : "",
               ssh_file_copy_file_get_name(tdata->source_file));

  SSH_OLDFSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat    \
                     (tdata->source->client,                    \
                      tdata->source_file_name,                  \
                      transfer_check_setstat_stat_source_cb,    \
                      thread));
}

void transfer_check_setstat_flags_cb(SshFileClientError error,
                                     void *context)
{
  SshOldFSMThread thread = (SshOldFSMThread) context;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  tdata->op_handle = NULL;

  SSH_DEBUG(4, ("Received error %d.", error));

  switch (error)
    {
    case SSH_FX_OK:
      /* Everything OK. */
      gdata->attributes_flags |= gdata->checking_flag;
      break;
    case SSH_FX_EOF:
      /* This should not happen, ever. */
      SSH_NOTREACHED;
    case SSH_FX_NO_SUCH_FILE:
      /* This should not happen, ever. */
      SSH_NOTREACHED;
      break;
    case SSH_FX_OP_UNSUPPORTED:
      /* This op should most definitely be supported. */
      SSH_NOTREACHED;

      /* Non-fatal errors. */
    case SSH_FX_PERMISSION_DENIED:
    case SSH_FX_FAILURE:
      break;

    case SSH_FX_NO_CONNECTION:
      SSH_OLDFSM_SET_NEXT(gdata->current_state);
      SSH_TRACE(2, ("No connection yet. Waiting..."));
      ssh_register_timeout(SSH_FC_CONNECTION_WAIT_TIMEOUT,
                           connection_wait_timeout,
                           (void *)thread);
      return;
    case SSH_FX_CONNECTION_LOST:
      SSH_OLDFSM_SET_NEXT(gdata->current_state);
      SSH_TRACE(2, ("Connection down, re-establishing..."));
      SSH_OLDFSM_THROW(gdata->dest_conn_thread, SSH_FCC_CONN_DOWN);
      gdata->dest_conn_flags = SSH_FCC_DOWN;
      break;

      /* Fatal errors. */
    case SSH_FX_BAD_MESSAGE:
      /* XXX we should exit the state machine, close connections
         etc. */
      (*gdata->completion_callback)(SSH_FC_ERROR_PROTOCOL_MISMATCH,
                                    "File transfer protocol mismatch.",
                                    gdata->callback_context);

      /* XXX Free everything, kill threads. */
      SSH_NOTREACHED;
      return;

      /* */
    case SSH_FX_OUT_OF_MEMORY:
      break;
    }

  SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_OLDFSM_STEP(transfer_check_setstat_flags_size)
{
  SshFileAttributes attrs;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  SSH_FCT_CHECK_DEST_CONN;

  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_uidgid");

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  if (!attrs || (attrs->flags != (SSH_FILEXFER_ATTR_SIZE |
                                  SSH_FILEXFER_ATTR_UIDGID |
                                  SSH_FILEXFER_ATTR_PERMISSIONS |
                                  SSH_FILEXFER_ATTR_ACMODTIME)))
    {
      SSH_OLDFSM_SET_NEXT("fct_check_setstat_stat_source");
      return SSH_OLDFSM_CONTINUE;
    }

  attrs->flags = gdata->checking_flag = SSH_FILEXFER_ATTR_SIZE;

  gdata->current_state = "fct_check_setstat_flags_size";

  SSH_OLDFSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat        \
                     (tdata->current_dest_file->handle,                 \
                      attrs,                                            \
                      transfer_check_setstat_flags_cb,                  \
                      thread));
}

SSH_OLDFSM_STEP(transfer_check_setstat_flags_uidgid)
{
  SshFileAttributes attrs;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);
  SSH_PRECOND(tdata->current_dest_file->handle);

  SSH_FCT_CHECK_DEST_CONN;

  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_permissions");

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);


  attrs->flags = gdata->checking_flag = SSH_FILEXFER_ATTR_UIDGID;

  gdata->current_state = "fct_check_setstat_flags_size";

  SSH_OLDFSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat        \
                     (tdata->current_dest_file->handle,                 \
                      attrs,                                            \
                      transfer_check_setstat_flags_cb,                  \
                      thread));
}

SSH_OLDFSM_STEP(transfer_check_setstat_flags_permissions)
{
  SshFileAttributes attrs;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);
  SSH_PRECOND(tdata->current_dest_file->handle);

  SSH_FCT_CHECK_DEST_CONN;

  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_acmodtime");

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  if (!attrs || !(attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS)
      || !(attrs->flags & SSH_FILEXFER_ATTR_ACMODTIME))
    {
      SSH_OLDFSM_SET_NEXT("fct_check_setstat_stat_source");
      return SSH_OLDFSM_CONTINUE;
    }

  attrs->flags = gdata->checking_flag = SSH_FILEXFER_ATTR_PERMISSIONS;

  gdata->current_state = "fct_check_setstat_flags_permissions";

  SSH_OLDFSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat        \
                     (tdata->current_dest_file->handle,                 \
                      attrs,                                            \
                      transfer_check_setstat_flags_cb,                  \
                      thread));
}

SSH_OLDFSM_STEP(transfer_check_setstat_flags_acmodtime)
{
  SshFileAttributes attrs;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);
  SSH_PRECOND(tdata->current_dest_file->handle);

  SSH_FCT_CHECK_DEST_CONN;

  SSH_OLDFSM_SET_NEXT("fct_check_setstat_flags_done");

  if (ssh_file_client_get_version(gdata->destination->client) == 0)
    {
      /* If server is old, this can't test can't be trusted. So let's
         just go with this one. */
      gdata->attributes_flags |= SSH_FILEXFER_ATTR_ACMODTIME;
      return SSH_OLDFSM_CONTINUE;
    }

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  attrs->flags = gdata->checking_flag = SSH_FILEXFER_ATTR_ACMODTIME;

  gdata->current_state = "fct_check_setstat_flags_acmodtime";

  SSH_OLDFSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat        \
                     (tdata->current_dest_file->handle,                 \
                      attrs,                                            \
                      transfer_check_setstat_flags_cb,                  \
                      thread));
}

SSH_OLDFSM_STEP(transfer_check_setstat_flags_done)
{
  SshFileAttributes attrs;
  char *error_message;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  SSH_PRECOND(tdata->current_dest_file->handle);

  SSH_OLDFSM_SET_NEXT(gdata->after_check_state);

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  attrs->flags = gdata->attributes_flags;


#if 0
  if (gdata->attributes_flags == (
                                  SSH_FILEXFER_ATTR_SIZE |
                                  SSH_FILEXFER_ATTR_UIDGID |
                                  SSH_FILEXFER_ATTR_PERMISSIONS |
                                  SSH_FILEXFER_ATTR_ACMODTIME))
    return SSH_OLDFSM_CONTINUE;
#else /* 0 or 1 */
    if ( (gdata->attributes_flags & SSH_FILEXFER_ATTR_ACMODTIME) &&
       (gdata->attributes_flags & SSH_FILEXFER_ATTR_PERMISSIONS) )
    return SSH_OLDFSM_CONTINUE;
#endif /* 0 or 1 */






  if (!gdata->attrs->preserve_attributes)
    return SSH_OLDFSM_CONTINUE;

  ssh_dsprintf(&error_message,
               "Following attribute(s) can't be changed on "
               "the remote file(s); %s%s%s%s",


#if 0
               (gdata->attributes_flags & SSH_FILEXFER_ATTR_SIZE ?
                "" : "size "),
               (gdata->attributes_flags & SSH_FILEXFER_ATTR_UIDGID ?
                "" : "uidgid "),
#else /* 0 or 1 */
               "", "",
#endif /* 0 or 1 */



               (gdata->attributes_flags & SSH_FILEXFER_ATTR_PERMISSIONS ?
                "" : "permissions "),
               (gdata->attributes_flags & SSH_FILEXFER_ATTR_ACMODTIME ?
                "" : "acmodtime"));

  (*gdata->error_callback)(SSH_FC_OK, error_message, gdata->callback_context);

  ssh_xfree(error_message);

  return SSH_OLDFSM_CONTINUE;
}
