/*
  sshfc_trcore.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  The core transfer step functions and callbacks.

  Created Wed Jan 17 09:58:53 2001.
*/

#include "ssh2includes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshfc_transferi.h"
#include "sshmatch.h"
#include "sshadt_list.h"
#include "sshoperation.h"

#define SSH_DEBUG_MODULE "SshFCTransferCore"

typedef struct TCRequestRec
{
  /* Operation handle for the ssh_file_client_* operations, so that we
     can abort them, if we get aborted. */
  SshOperationHandle op_handle;

  /* offset, where the read/write started. */
  off_t req_offset;
  /* Requested bytes. */
  size_t req_bytes;

  /* Associated buffer. */
  SshBuffer buffer;

  /* Associated thread. */
  SshOldFSMThread thread;
} *TCRequest, TCRequestStruct;

typedef struct TransferCoreThreadContextRec
{
  /* Source connection (needed for progress indicator). */
  SshFileCopyConnection source;

  SshFileCopyFile source_file;

  SshFileCopyFile dest_file;

  SshADTContainer req_queue;
  SshADTContainer in_progress_queue;

  Boolean dealing_with_source;

  off_t read_bytes;
  off_t written_bytes;

  SshOldFSMThread parent;

} *TransferCoreThreadContext;

void transfer_core_thread_destructor(void *tdata)
{
  TransferCoreThreadContext thread_context =
    (TransferCoreThreadContext) tdata;

  SSH_PRECOND(tdata);

  SSH_DEBUG(4, ("Destroying request queues.."));

  ssh_adt_destroy(thread_context->in_progress_queue);
  ssh_adt_destroy(thread_context->req_queue);
}

void tc_req_destroy(void *obj, void *context)
{
  TCRequest req = (TCRequest) obj;

  SSH_DEBUG(6, ("Destroying request."));
  if (req->buffer)
    {
      ssh_buffer_free(req->buffer);
      req->buffer = NULL;
    }
  ssh_xfree(req);
}

void tc_inprogress_req_destroy(void *obj, void *context)
{
  TCRequest req = (TCRequest) obj;

  SSH_DEBUG(5, ("Destroying request in progress."));

  ssh_operation_abort(req->op_handle);
  req->op_handle = NULL;

  tc_req_destroy(req, context);
}

void tc_buffer_destroy(void *obj, void *context)
{
  SshBuffer buf = (SshBuffer) obj;

  SSH_DEBUG(4, ("Freeing buffer from queue..."));

  ssh_buffer_free(buf);
}

/* The transfer loop. */
SSH_OLDFSM_STEP(transfer_start)
{
  TransferCoreThreadContext reader_tdata, writer_tdata;

  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferThreadContext);

  /* Allocate buffers etc. */
  if (!gdata->in_blocking)
    gdata->in_blocking = ssh_oldfsm_condition_create(gdata->fsm);
  if (!gdata->out_blocking)
    gdata->out_blocking = ssh_oldfsm_condition_create(gdata->fsm);

  SSH_TRACE(2, ("Starting transfer for file %s, destination %s",
                ssh_file_copy_file_get_name(tdata->source_file),
                ssh_file_copy_file_get_name(tdata->current_dest_file)));

  gdata->reader = SSH_OLDFSM_FORK(sizeof(*reader_tdata), "fctc_shred",
                               NULL, transfer_core_thread_destructor);

  ssh_oldfsm_set_thread_name(gdata->reader,
                          "reader");

  gdata->writer = SSH_OLDFSM_FORK(sizeof(*writer_tdata), "fctc_write_out",
                               NULL, transfer_core_thread_destructor);

  ssh_oldfsm_set_thread_name(gdata->writer, "writer");

  reader_tdata = ssh_oldfsm_get_tdata(gdata->reader);
  writer_tdata = ssh_oldfsm_get_tdata(gdata->writer);

  memset(reader_tdata, 0, sizeof(*reader_tdata));
  memset(writer_tdata, 0, sizeof(*writer_tdata));

  reader_tdata->source = writer_tdata->source = tdata->source;
  reader_tdata->source_file = writer_tdata->source_file = tdata->source_file;
  reader_tdata->dest_file = writer_tdata->dest_file = tdata->current_dest_file;

  reader_tdata->req_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                                   SSH_ADT_DESTROY,
                                                   tc_req_destroy,
                                                   SSH_ADT_ARGS_END);
  reader_tdata->in_progress_queue =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           tc_inprogress_req_destroy,
                           SSH_ADT_ARGS_END);

  writer_tdata->req_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                                   SSH_ADT_DESTROY,
                                                   tc_req_destroy,
                                                   SSH_ADT_ARGS_END);
  writer_tdata->in_progress_queue =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           tc_inprogress_req_destroy,
                           SSH_ADT_ARGS_END);

  /* If not already allocated, allocate buffer queue. */
  if (!gdata->buf_queue)
    gdata->buf_queue = ssh_adt_create_generic(SSH_ADT_LIST,
                                              SSH_ADT_DESTROY,
                                              tc_buffer_destroy,
                                              SSH_ADT_ARGS_END);

  /* XXX Clear buffers in buffer queue. */

  reader_tdata->parent = writer_tdata->parent = thread;

  if (gdata->progress_callback)
    {
      ssh_time_measure_reset(gdata->transfer_timer);
      ssh_time_measure_start(gdata->transfer_timer);

      /* Initialize timeout for progress callback. */
      ssh_register_timeout(1L, 0L, progress_timeout_cb, gdata);
    }

  SSH_OLDFSM_SET_NEXT("fct_chmod_dest_after");

  return SSH_OLDFSM_SUSPENDED;
}

/* "Shred" a file to multiple requests. */
SSH_OLDFSM_STEP(transfer_shred)
{
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferCoreThreadContext);
  TCRequest new_req = NULL;
  SshFileAttributes attrs;
  off_t bytes = 0L;

  attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  SSH_ASSERT(attrs);
  SSH_ASSERT(attrs->flags & SSH_FILEXFER_ATTR_SIZE);
  SSH_ASSERT(gdata->attrs->max_buf_size);

  gdata->byte_difference = 0;

  for (bytes = 0L; bytes < attrs->size;)
    {
      new_req = ssh_xcalloc(1, sizeof(*new_req));
      new_req->req_offset = bytes;
      new_req->req_bytes = gdata->attrs->max_buf_size < attrs->size - bytes ?
        gdata->attrs->max_buf_size : attrs->size - bytes;
      bytes += new_req->req_bytes;
      SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                   new_req) != SSH_ADT_INVALID);
    }

  SSH_OLDFSM_SET_NEXT("fctc_read_in");

  return SSH_OLDFSM_CONTINUE;
}

void read_in_cb(SshFileClientError error, const unsigned char *data,
                size_t len, void *context)
{
  TCRequest req = (TCRequest) context;
  SshOldFSMThread thread = req->thread;
  SshFileCopyError error_code;
  char *error_string;
  unsigned char *start;
  unsigned char *end;
  char *ext;
  char temp;
  int dest_length;
  int source_length;
  int byte_difference = 0;
  Boolean match;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferCoreThreadContext);
  SSH_PRECOND(thread);
  req->op_handle = NULL;

  switch (error)
    {
    case SSH_FX_OK:
      SSH_DEBUG(7, ("Got %ld bytes of data.", len));

      /* ascii file transfer */
      if (gdata->transfer_mode == SSH_FC_TRANSFER_MODE_ASCII)
        {
          /* don't do any parsing if newlines are same */
          if (strcmp(gdata->source_newline, gdata->dest_newline) != 0)
            {
              dest_length = strlen(gdata->dest_newline);
              source_length = strlen(gdata->source_newline);
              start = (unsigned char *)data;
              while (start - data < len)
                {
                  end = strstr(start, gdata->source_newline);
                  if (end != NULL && end - data < len)
                    {
                      ssh_xbuffer_append(req->buffer, start, end - start);
                    }
                  else
                    {
                      ssh_xbuffer_append(req->buffer,
                                         start,
                                         len - (start - data));
                      break;
                    }
                  if (end - start > 1 && start[end - start - 1] == '\r' &&
                      strcmp(gdata->dest_newline, "\r\n") == 0 &&
                      strcmp(gdata->source_newline, "\n") == 0)
                    {
                      ssh_xbuffer_append(req->buffer, "\n", 1);
                    }
                  else
                    {
                      ssh_xbuffer_append(req->buffer,
                                         gdata->dest_newline,
                                         dest_length);
                      byte_difference += source_length - dest_length;
                    }
                  start = end + source_length;
                }
            }
          else
            {
              /* newlines did match */
              ssh_xbuffer_append(req->buffer, data, len);
            }

        }
      else if (gdata->transfer_mode == SSH_FC_TRANSFER_MODE_AUTO)
        {
          /* don't do any parsing if newlines are same */
          if (strcmp(gdata->source_newline, gdata->dest_newline) != 0)
            {
              ext = strrchr(ssh_file_copy_file_get_name(tdata->source_file),
                            '.');
              if (ext != NULL)
                {
                  ext++;
                  start = gdata->ascii_extensions;
                  match = FALSE;
                  while (TRUE)
                    {
                      end = strchr(start, ',');
                      if (end != NULL)
                        {
                          temp = gdata->ascii_extensions[end - gdata->ascii_extensions];
                          gdata->ascii_extensions[end - gdata->ascii_extensions] = '\0';

                          /* remove characters before the last dot from the extension */
                          while (strchr(start, '.') != NULL)
                            start = strchr(start, '.') + 1;

                          match = ssh_match_pattern(ext, start);
                          gdata->ascii_extensions[end - gdata->ascii_extensions] = temp;
                          if (match)
                            break;
                        }
                      else
                        {
                          match = ssh_match_pattern(ext, start);
                          break;
                        }
                      start = end + 1;
                    }

                  if (match)
                    {
                      dest_length = strlen(gdata->dest_newline);
                      source_length = strlen(gdata->source_newline);
                      start = (unsigned char *)data;
                      while (start - data < len)
                        {
                          end = strstr(start, gdata->source_newline);
                          if (end != NULL && end - data < len)
                            {
                              ssh_xbuffer_append(req->buffer,
                                                 start,
                                                 end - start);
                            }
                          else
                            {
                              ssh_xbuffer_append(req->buffer,
                                                 start,
                                                 len - (start - data));
                              break;
                            }
                          if (end - start > 1 && start[end - start - 1] == '\r' &&
                              strcmp(gdata->dest_newline, "\r\n") == 0 &&
                              strcmp(gdata->source_newline, "\n") == 0)
                            {
                              ssh_xbuffer_append(req->buffer, "\n", 1);
                            }
                          else
                            {
                              ssh_xbuffer_append(req->buffer,
                                                 gdata->dest_newline,
                                                 dest_length);
                              byte_difference += source_length - dest_length;
                            }
                          start = end + source_length;
                        }
                    }
                  else
                    {
                      /* file extension didn't match */
                      ssh_xbuffer_append(req->buffer, data, len);
                    }
                }
              else
                {
                  /* file extension list is empty */
                  ssh_xbuffer_append(req->buffer, data, len);
                }
            }
          else
            {
              /* newlines did match */
              ssh_xbuffer_append(req->buffer, data, len);
            }
        }
      else
        {
          /* BINARY MODE */
          ssh_xbuffer_append(req->buffer, data, len);
        }

      if (gdata->transfer_mode != SSH_FC_TRANSFER_MODE_BINARY)
      {
        /* Change the offset of the current request depending on how the
           previous requests have changed the offset. */
        req->req_offset -= gdata->byte_difference;

        /* Add the current request difference to the total difference. */
        gdata->byte_difference += byte_difference;
      }

      if (req->req_bytes > len)
        {
          TCRequest new_req;
          /* If we get fewer bytes than we requested, we schedule a
             new request for the missing bit, and wake up the reader. */
          new_req = ssh_xcalloc(1, sizeof(*new_req));
          new_req->req_offset = req->req_offset + len;
          new_req->req_bytes = req->req_bytes - len;
          SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                       new_req) != SSH_ADT_INVALID);
        }
      req->req_bytes = len;
      tdata->read_bytes += len;

      {
        TransferCoreThreadContext writer_tdata =
          ssh_oldfsm_get_tdata(gdata->writer);
        ssh_adt_detach(tdata->in_progress_queue,
                       ssh_adt_get_handle_to(tdata->in_progress_queue, req));
        SSH_DEBUG(7, ("Adding a write request to writer's request queue."));
        SSH_VERIFY(ssh_adt_insert_to(writer_tdata->req_queue,
                                     SSH_ADT_BEGINNING, req) !=
                   SSH_ADT_INVALID);
        ssh_oldfsm_continue(gdata->writer);
      }
      return;
    case SSH_FX_EOF:
      error_code = SSH_FC_ERROR;
      ssh_dsprintf(&error_string, "%s: got EOF reading file.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      (*gdata->error_callback)(error_code, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      ssh_xfree(req);
      return;
    case SSH_FX_OP_UNSUPPORTED:
      error_code = SSH_FC_ERROR;
      ssh_dsprintf(&error_string, "%s: got SSH_FX_OP_UNSUPPORTED (should "
                   "not happen).",
                   ssh_file_copy_file_get_name(tdata->source_file));
      break;
    case SSH_FX_NO_SUCH_FILE:
      error_code = SSH_FC_ERROR_NO_SUCH_FILE;
      ssh_dsprintf(&error_string, "%s: No such file or directory.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      break;
    case SSH_FX_PERMISSION_DENIED:
      error_code = SSH_FC_ERROR_PERMISSION_DENIED;
      ssh_dsprintf(&error_string, "%s: Permission denied.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      goto error_common;
      break;
    case SSH_FX_BAD_MESSAGE:
      break;
    case SSH_FX_FAILURE:
      error_code = SSH_FC_ERROR_FAILURE;
      ssh_dsprintf(&error_string, "%s: Undefined error occurred.",
                   ssh_file_copy_file_get_name(tdata->source_file));
    error_common:
      /* Report error. */
      (*gdata->error_callback)(error_code, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      ssh_xfree(req);
      /* XXX Cancel rest of the requests. */
      /* XXX Finish the threads. */
      break;
    case SSH_FX_NO_CONNECTION:
      SSH_TRACE(2, ("No connection yet. Waiting..."));
      ssh_register_timeout(SSH_FC_CONNECTION_WAIT_TIMEOUT,
                           connection_wait_timeout,
                           (void *)thread);
      break;
    case SSH_FX_CONNECTION_LOST:
      SSH_TRACE(2, ("Connection down, re-establishing..."));
      SSH_OLDFSM_THROW(gdata->source_conn_thread, SSH_FCC_CONN_DOWN);
      gdata->source_conn_flags = SSH_FCC_DOWN;
      break;

      /* */
    case SSH_FX_OUT_OF_MEMORY:
      break;
    }

  ssh_xfree(req);
  ssh_oldfsm_continue(thread);
}

SSH_OLDFSM_STEP(transfer_read_in)
{
  SshBuffer buf;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferCoreThreadContext);
  TCRequest req;
  TransferCoreThreadContext writer_tdata = ssh_oldfsm_get_tdata(gdata->writer);

  SSH_FCT_CHECK_SRC_CONN;

  if (writer_tdata->written_bytes >=
      (ssh_file_copy_file_get_attributes(tdata->source_file))->size)
    {
      SSH_OLDFSM_CONDITION_SIGNAL(gdata->out_blocking);
      gdata->reader = NULL;
      return SSH_OLDFSM_FINISH;
    }

  if (ssh_adt_num_objects(tdata->req_queue) == 0)
    {
      SSH_DEBUG(8, ("No more requests at this time."));
      SSH_OLDFSM_CONDITION_WAIT(gdata->in_blocking);
    }

  if (!ssh_adt_num_objects(gdata->buf_queue))
    {
      SSH_ASSERT(gdata->attrs->max_buffers);
      if (gdata->num_bufs < gdata->attrs->max_buffers)
        {
          buf = ssh_xbuffer_allocate();
          gdata->num_bufs++;
        }
      else
        {
          SSH_OLDFSM_CONDITION_WAIT(gdata->in_blocking);
        }
    }
  else
    {
      buf = ssh_adt_detach_from(gdata->buf_queue, SSH_ADT_END);
    }

  req = ssh_adt_detach_from(tdata->req_queue, SSH_ADT_END);

  SSH_ASSERT(req != SSH_ADT_INVALID);

  req->buffer = buf;
  req->thread = thread;

  SSH_DEBUG(6, ("Reading %ld bytes from offset %ld.",
                req->req_offset, req->req_bytes));

  SSH_VERIFY(ssh_adt_insert_to(tdata->in_progress_queue, SSH_ADT_BEGINNING,
                               req) != SSH_ADT_INVALID);

  /* Result comes in a data callback. */
  req->op_handle = ssh_file_client_read
    (ssh_file_copy_file_get_handle(tdata->source_file),
     req->req_offset, req->req_bytes,
     read_in_cb, (void *)req);

  return SSH_OLDFSM_CONTINUE;
}

void write_out_cb(SshFileClientError error, void *context)
{
  TCRequest req = (TCRequest) context;
  SshOldFSMThread thread = req->thread;
  SshFileCopyError error_code;
  char *error_string;
  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferCoreThreadContext);
  SSH_PRECOND(thread);
  req->op_handle = NULL;

  switch (error)
    {
    case SSH_FX_OK:
      ssh_buffer_clear(req->buffer);
      tdata->written_bytes += req->req_bytes;
      ssh_oldfsm_continue(gdata->reader);
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                   req->buffer) != SSH_ADT_INVALID);
      ssh_xfree(req);
      return;
    case SSH_FX_EOF:
      error_code = SSH_FC_ERROR;
      ssh_dsprintf(&error_string, "%s: got EOF writing file.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      (*gdata->error_callback)(error_code, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      ssh_xfree(req);
      return;
    case SSH_FX_OP_UNSUPPORTED:
      error_code = SSH_FC_ERROR;
      ssh_dsprintf(&error_string, "%s: got SSH_FX_OP_UNSUPPORTED (should "
                   "not happen).",
                   ssh_file_copy_file_get_name(tdata->source_file));
      goto error_common;
    case SSH_FX_NO_SUCH_FILE:
      error_code = SSH_FC_ERROR_NO_SUCH_FILE;
      ssh_dsprintf(&error_string, "%s: No such file or directory.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      goto error_common;
    case SSH_FX_PERMISSION_DENIED:
      error_code = SSH_FC_ERROR_PERMISSION_DENIED;
      ssh_dsprintf(&error_string, "%s: Permission denied.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      goto error_common;
    case SSH_FX_BAD_MESSAGE:
      error_code = SSH_FC_ERROR_FAILURE;
      ssh_dsprintf(&error_string, "%s: FileXFer protocol error.",
                   ssh_file_copy_file_get_name(tdata->source_file));
      goto error_common;
    case SSH_FX_FAILURE:
      error_code = SSH_FC_ERROR_FAILURE;
      ssh_dsprintf(&error_string, "%s: Undefined error occurred.",
                   ssh_file_copy_file_get_name(tdata->source_file));
    error_common:
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(gdata->buf_queue, SSH_ADT_BEGINNING,
                                   req->buffer) != SSH_ADT_INVALID);
      /* Report error. */
      (*gdata->error_callback)(error_code, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      /* XXX Cancel rest of the requests. */
      /* XXX Finish the threads. */
      break;
    case SSH_FX_NO_CONNECTION:
      SSH_TRACE(2, ("No connection yet. Waiting..."));
      ssh_register_timeout(SSH_FC_CONNECTION_WAIT_TIMEOUT,
                           connection_wait_timeout,
                           (void *)thread);
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                   req) != SSH_ADT_INVALID);
      req = NULL;
      break;
    case SSH_FX_CONNECTION_LOST:
      SSH_TRACE(2, ("Connection down, re-establishing..."));
      SSH_OLDFSM_THROW(gdata->source_conn_thread, SSH_FCC_CONN_DOWN);
      gdata->source_conn_flags = SSH_FCC_DOWN;
      ssh_adt_detach(tdata->in_progress_queue,
                     ssh_adt_get_handle_to(tdata->in_progress_queue, req));
      SSH_VERIFY(ssh_adt_insert_to(tdata->req_queue, SSH_ADT_BEGINNING,
                                   req) != SSH_ADT_INVALID);
      req = NULL;
      break;

      /* */
    case SSH_FX_OUT_OF_MEMORY:
      break;
    }

  ssh_xfree(req);
  ssh_oldfsm_continue(thread);
}

SSH_OLDFSM_STEP(transfer_write_out)
{
  TCRequest req;

  SSH_OLDFSM_DATA(SshFileCopyTransferContext, TransferCoreThreadContext);

  SSH_FCT_CHECK_DEST_CONN;

  if (tdata->written_bytes >= tdata->source_file->attributes->size)
    {
      if (gdata->progress_callback)
        {

          SshUInt64 seconds;
          /* Unregister any progress indicator timeouts. Call the
             progress-callback directly to indicate that the
             transfer ended. */
          ssh_cancel_timeouts(progress_timeout_cb, gdata);
          ssh_time_measure_stop(gdata->transfer_timer);
          ssh_time_measure_get_value(gdata->transfer_timer,
                                     &seconds, NULL);
          (*gdata->progress_callback)(tdata->source,
                                      tdata->source_file,
                                      gdata->destination,
                                      tdata->dest_file,
                                      gdata->reader ?
                                      ((TransferCoreThreadContext)
                                       ssh_oldfsm_get_tdata(gdata->reader))->
                                      read_bytes
                                      : tdata->written_bytes,
                                      tdata->written_bytes,
                                      seconds,
                                      gdata->callback_context);
        }
      gdata->writer = NULL;
      SSH_OLDFSM_SET_NEXT("fctc_wake_parent_and_finish");
      return SSH_OLDFSM_CONTINUE;
    }

  if (ssh_adt_num_objects(tdata->req_queue) == 0)
    {
      SSH_DEBUG(8, ("No more requests at this time."));
      SSH_OLDFSM_CONDITION_WAIT(gdata->out_blocking);
    }

  req = ssh_adt_detach_from(tdata->req_queue, SSH_ADT_END);
  req->thread = thread;

  SSH_ASSERT(req != SSH_ADT_INVALID);

  SSH_DEBUG(6, ("Writing %ld bytes to offset %ld.",
                ssh_buffer_len(req->buffer), req->req_offset));

  SSH_VERIFY(ssh_adt_insert_to(tdata->in_progress_queue, SSH_ADT_BEGINNING,
                               req) != SSH_ADT_INVALID);

  /* Result comes in a status callback, thus written bytes has to be
     stored. */
  req->op_handle = ssh_file_client_write
    (ssh_file_copy_file_get_handle(tdata->dest_file),
     req->req_offset, ssh_buffer_ptr(req->buffer),
     ssh_buffer_len(req->buffer),
     write_out_cb, (void *)req);

  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(transfer_readwrite_error)
{
  SSH_OLDFSM_SET_NEXT("fctc_wake_parent_and_finish");
  return SSH_OLDFSM_CONTINUE;
}

SSH_OLDFSM_STEP(transfer_core_wake_parent_and_finish)
{
  SSH_OLDFSM_TDATA(TransferCoreThreadContext);

  if (tdata->parent)
    ssh_oldfsm_continue(tdata->parent);

  return SSH_OLDFSM_FINISH;
}

void progress_timeout_cb(void *context)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext) context;
  TransferCoreThreadContext reader_tdata;
  TransferCoreThreadContext writer_tdata;
  SshUInt64 seconds;

  reader_tdata = gdata->reader ? ssh_oldfsm_get_tdata(gdata->reader) : NULL;
  writer_tdata = ssh_oldfsm_get_tdata(gdata->writer);

  ssh_time_measure_get_value(gdata->transfer_timer,
                             &seconds, NULL);
  (*gdata->progress_callback)(writer_tdata->source,
                              writer_tdata->source_file,
                              gdata->destination,
                              writer_tdata->dest_file,
                              reader_tdata ?
                              reader_tdata->read_bytes
                              : writer_tdata->written_bytes,
                              writer_tdata->written_bytes,
                              seconds,
                              gdata->callback_context);

  ssh_register_timeout(1L, 0L, progress_timeout_cb, gdata);
}
