/*

  sshftpfilter.c

  Authors:
        Tomi Salo <ttsalo@ssh.com>
        Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

/*

  This module implements ftp protocol forwarding.

*/

#include "ssh2includes.h"
#include "sshfilterstream.h"
#include "sshtcp.h"
#include "sshcommon.h"
#include "sshftpfilter.h"
#include "sshappcommon.h"
#include "sshchtcpfwd.h"

#define SSH_DEBUG_MODULE "SshFtpFilter"

/* Finds the end of the command line in the buffer, starting from offset.
   If there is no valid command line (ending in \r\n), returns zero. CR and
   LF are counted into the length of the line. */

unsigned int
ssh_ftp_command_line_length(SshBuffer data,
                            size_t offset)
{
  unsigned char *data_ptr;
  size_t total_len, len = 0;
  
  total_len = ssh_buffer_len(data) - offset;
  data_ptr = ssh_buffer_ptr(data) + offset;

  /* First check that there is enough data for any command line at all */
  if (offset >= ssh_buffer_len(data) || total_len <= 1)
    return 0;

  while (len < total_len - 1)
    {
      if (data_ptr[len] == '\r' && data_ptr[len + 1] == '\n')
        return len + 2;
      len++;
    }

  /* No \r\n sequence found, no complete command line in the buffer */
  return 0;
}

/* Parses a line and reports whether it gives the parameters for a
   data connection. Returns FALSE if parsing failed. If successful,
   returns TRUE notes whether or not this was a passive-mode request
   in *passive_return and writes the six parameter values in
   values_return. */
Boolean
ssh_ftp_parse_open(SshBuffer data,
                   size_t offset,
                   size_t line_len,
                   Boolean *passive_return,
                   unsigned int values_return[6])
{
  char *data_ptr, *cmd_line, *parms;
  int result = 0;

  data_ptr = ssh_buffer_ptr(data) + offset;

  cmd_line = ssh_xmalloc(line_len + 1);
  strncpy(cmd_line, data_ptr, line_len);
  cmd_line[line_len] = 0;
  
  if (!strncmp(data_ptr, "PORT ", 5))
    {
      SSH_DEBUG(5, ("Detected PORT in ftp command stream."));
      *passive_return = FALSE;
      parms = strstr(cmd_line, " ");
      if (parms == NULL)
        goto error;
      parms++;
    }
  else
    {
      if (!strncmp(data_ptr, "227 ", 4))
        {
          SSH_DEBUG(5, ("Detected response to PASV in ftp command stream."));
          *passive_return = TRUE;
          parms = strstr(cmd_line, "(");
          if (parms == NULL)
            goto error;
          parms++;
        }
      else
        {
          ssh_xfree(cmd_line);
          return FALSE;
        }
    }

  result = sscanf(parms, "%d,%d,%d,%d,%d,%d", &values_return[0],
                  &values_return[1], &values_return[2],
                  &values_return[3], &values_return[4], &values_return[5]);

  if (result < 6)
    goto error;
  
  SSH_DEBUG(7, ("Decoded %d,%d,%d,%d,%d,%d from ftp command.",
                values_return[0], values_return[1], values_return[2],
                values_return[3], values_return[4], values_return[5]));
  
  ssh_xfree(cmd_line);
  return TRUE;

error:
  SSH_DEBUG(3, ("Malformed ftp protocol command."));
  ssh_xfree(cmd_line);
  return FALSE;
}

/* Checks whether the given command line (starting at offset, length
   line_len) is a command for opening a data connection. If so,
   returns the port number, writes the connection parameters in
   values_return and and marks in passive_return whether the command
   was PORT or an answer to PASV. */
unsigned int
ssh_ftp_detect_open(SshBuffer data,
                    size_t offset,
                    size_t line_len,
                    Boolean *passive_return,
                    unsigned int values_return[6])
{
  if (ssh_ftp_parse_open(data, offset, line_len, passive_return,
                         values_return) == FALSE)
    return 0;
  else
    return values_return[4] * 256 + values_return[5];
}

/* Replaces the port portion of the sextuplet of numbers on command line. */
/* Actually this function constructs a whole new command line and
   replaces the old one with it. */
/* Returns the length of the new command line. */
unsigned int
ssh_ftp_replace_port(SshBuffer data,
                     size_t offset,
                     size_t line_len,
                     unsigned int new_port)
{
  unsigned int values[6], new_line_len;
  char *new_line, *data_ptr, *tmp_str;
  Boolean passive_return;
  int size_change;
  size_t total_len;

  if (ssh_ftp_parse_open(data, offset, line_len, &passive_return,
                         values) == FALSE)
    return 0;

  data_ptr = ssh_buffer_ptr(data) + offset;
  total_len = ssh_buffer_len(data) - offset;
  new_line = ssh_xmalloc(160);

  values[4] = new_port / 256;
  values[5] = new_port % 256;

  /* We'll bind the local end of the tunnel created for the
     data channel always into the localhost. */
  if (passive_return == FALSE)
    {
      ssh_snprintf(new_line, 160, "PORT 127,0,0,1,%d,%d\r\n",
                   values[4], values[5]);
    }
  else
    {
      ssh_snprintf(new_line, 160, "227 Entering Passive Mode "
                   "(127,0,0,1,%d,%d)\r\n",
                   values[4], values[5]);
    }

  new_line_len = strlen(new_line);
  size_change = new_line_len - line_len;

  if (size_change == 0)
    {
      /* The new line is as long as the old one, so just replace it */
      strncpy(data_ptr, new_line, strlen(new_line));
      return new_line_len;
    }

  if (size_change < 0)
    {
      /* The new line is shorter than the old one. Copy it into the
         buffer, move the rest and then shrink the buffer. */
      strncpy(data_ptr, new_line, strlen(new_line));
      memmove(data_ptr + new_line_len, data_ptr + line_len,
              total_len - line_len);
      ssh_buffer_consume_end(data, -size_change);
      return new_line_len;
    }

  /* The new line is longer than the old one. Enlarge the buffer, move the
     rest of the buffer and copy the new line in. */
  ssh_xbuffer_append_space(data, (unsigned char **)&tmp_str, size_change);
  memmove(data_ptr + new_line_len, data_ptr + line_len, total_len - line_len);
  strncpy(data_ptr, new_line, strlen(new_line));
  return new_line_len;
}

/* Filter function for processing data connection opening requests
   arriving to this end. The requests will be handled by opening a
   local forwarding for the connection. Whether the request is going
   from server to client or client to server is not important. */

void
ssh_ftp_output_filter(void *context,
                      SshFilterGetCB get_data,
                      SshFilterCompletionCB completed,
                      void *internal_context)
{
  SshFtpFilter filter = (SshFtpFilter)context;
  size_t len;
  unsigned int port, values[6];
  Boolean passive;
  SshBuffer data;
  size_t offset;
  Boolean eof_received;

  (*get_data)(internal_context, &data, &offset, &eof_received);

  if (eof_received)
    {
      if (ssh_buffer_len(data) > offset)
        (*completed)(internal_context,
                     SSH_FILTER_ACCEPT(ssh_buffer_len(data) - offset));
      else
        (*completed)(internal_context, SSH_FILTER_HOLD);
    }
  
  if ((len = ssh_ftp_command_line_length(data, offset)) > 0)
    {
      SSH_DEBUG(8, ("Accepting a command line of %d bytes.", len));

      port = ssh_ftp_detect_open(data, offset, len, &passive, values);

      if (port > 0)
        {
          /* Caught a command for data connection opening. Open a
             local forwarding. */
          char *address_to_bind = "127.0.0.1",
            port_to_bind[16], connect_to_host[32], connect_to_port[16];
          unsigned int local_port = port; /* Let's try the port the other side
                                             is using... */
          do
            {
              ssh_snprintf(port_to_bind, 16, "%d", local_port);
              ssh_snprintf(connect_to_host, 32, "%d.%d.%d.%d",
                           values[0], values[1],
                           values[2], values[3]);
              ssh_snprintf(connect_to_port, 16, "%d", port);
              local_port++; /* XXX should be randomized instead? */
            }
          while (ssh_channel_start_local_tcp_forward(filter->common,
                                                     address_to_bind,
                                                     port_to_bind,
                                                     connect_to_host,
                                                     connect_to_port,
                                                     "ftp-data") == FALSE &&
                 local_port < 65535);

          /* Forwarding initiated successfully. Spoof the port. */
          if (local_port != 65535)
            {
              len = ssh_ftp_replace_port(data, offset, len, --local_port);
              SSH_DEBUG(4, ("Initiated a local forwarding from local "
                            "port %s to port %s on remote host",
                            port_to_bind, connect_to_port));
            }
          else
            SSH_DEBUG(4, ("Local forwarding from local port %s to "
                          "port %s on remote host failed",
                          port_to_bind, connect_to_port));
        }

      (*completed)(internal_context, SSH_FILTER_ACCEPT(len));
      return;
    }
  
  (*completed)(internal_context, SSH_FILTER_HOLD);
}

/* Completion function for the opening of a remote forwarding from the
   input_filter function. */
void
ssh_ftp_remotefwd_cb(Boolean ok, void *context)
{
  SshFtpFilter filter = (SshFtpFilter)context;
  int len;

  if (ok == TRUE)
    {
      /* Remote forwarding has been created, modify and accept the ftp
         command */
      SSH_DEBUG(7, ("Remote forwarding for ftp data channel created "
                    "successfully."));
      filter->remote_fwd_creation_in_progress = FALSE;
      len = ssh_ftp_replace_port
        (filter->buffer, filter->input_offset, 
         ssh_ftp_command_line_length(filter->buffer,
                                     filter->input_offset),
         filter->local_port);
      (*filter->input_filter_completion)(filter->input_filter_internal_context,
                                         SSH_FILTER_ACCEPT(len));
      SSH_DEBUG(7, ("Accepting %d bytes after successful remote fwd "
                    "creation", len));
    }
  else
    {
      if (filter->local_port >= 65535)
        {
          /* Failed too many times, give up, accept the command,
             ftp-client will fail to establish data connection */
          SSH_DEBUG(7, ("Creating a remote forwarding for ftp data channel "
                        "failed, giving up."));
          len = ssh_ftp_command_line_length(filter->buffer,
                                            filter->input_offset);
          filter->remote_fwd_creation_in_progress = FALSE;
          (*filter->input_filter_completion)
            (filter->input_filter_internal_context,
             SSH_FILTER_ACCEPT(len));
          SSH_DEBUG(7, ("Accepting %d bytes after unsuccessful remote "
                        "fwd creation", len));
        }
      else
        {
          /* Failed, try again with different listener port number. */
          char port_to_bind[16];
          filter->local_port++;
          SSH_DEBUG(7, ("Creating a remote forwarding for ftp data channel "
                        "failed, trying again (port %d->%d).",
                        filter->local_port - 1, filter->local_port));
          
          ssh_snprintf(port_to_bind, 16, "%d", filter->local_port);
          
          ssh_channel_start_remote_tcp_forward(filter->common, "127.0.0.1",
                                               port_to_bind,
                                               filter->connect_to_host,
                                               filter->connect_to_port,
                                               "ftp-data",
                                               ssh_ftp_remotefwd_cb,
                                               filter);
        }
    }
}

/* Filter function for processing data connection opening requests
   originating from this end. The requests will be handled by opening a
   remote forwarding for the connection. Whether the request is going
   from server to client or client to server is not important. */

void
ssh_ftp_input_filter(void *context,
                     SshFilterGetCB get_data,
                     SshFilterCompletionCB completed,
                     void *internal_context)
{
  SshFtpFilter filter = (SshFtpFilter)context;
  size_t len, current_offset;
  unsigned int port, values[6];
  Boolean passive;
  SshBuffer data;
  size_t offset;
  Boolean eof_received;

  (*get_data)(internal_context, &data, &offset, &eof_received);

  if (eof_received)
    {
      if (ssh_buffer_len(data) > offset)
        (*completed)(internal_context,
                     SSH_FILTER_ACCEPT(ssh_buffer_len(data) - offset));
      else
        (*completed)(internal_context, SSH_FILTER_HOLD);
    }
  
  current_offset = offset;

  if ((len = ssh_ftp_command_line_length(data, offset)) > 0)
    {
      /* We can't touch anything while the forwarding is being created */
      if (filter->remote_fwd_creation_in_progress == TRUE)
        {
          SSH_DEBUG(5, ("Entered again even though remote fwd creation is "
                        "in progress."));
          (*completed)(internal_context, SSH_FILTER_HOLD);
          return;
        }
      
      SSH_DEBUG(8, ("Accepting a command line of %d bytes.", len));
      
      port = ssh_ftp_detect_open(data, current_offset, len, &passive, values);
      
      if (port > 0)
        {
          /* Caught a command for data connection opening. Try to open
             a remote forwarding. */
          char port_to_bind[16];

          filter->local_port = port;
          
          ssh_snprintf(port_to_bind, 16, "%d", filter->local_port);
          ssh_snprintf(filter->connect_to_host, 32, "%d.%d.%d.%d",
                       values[0], values[1],
                       values[2], values[3]);
          ssh_snprintf(filter->connect_to_port, 16, "%d", port);

          /* Store the information concerning the acceptance of the command */
          filter->buffer = data;
          filter->remote_fwd_creation_in_progress = TRUE;
          
          ssh_channel_start_remote_tcp_forward(filter->common,
                                               "127.0.0.1",
                                               port_to_bind,
                                               filter->connect_to_host,
                                               filter->connect_to_port,
                                               "ftp-data",
                                               ssh_ftp_remotefwd_cb,
                                               filter);
          /* Store the relevant filter data and call the completion later */
          filter->input_filter_internal_context = internal_context;
          filter->input_offset = offset;
          filter->input_filter_completion = completed;
          return;
        }

      /* Not interested in the command, let it through */
      (*completed)(internal_context, SSH_FILTER_ACCEPT(len));
      return;
    }

  /* No complete command line seen, hold the data and wait for more */
  (*completed)(internal_context, SSH_FILTER_HOLD);
  return;
}

void
ssh_ftp_filter_destroy(void *context)
{
  SshFtpFilter filter = (SshFtpFilter)context;

  SSH_DEBUG(1, ("Destroying the ftp filter."));
  /* Check if there is open data forward and cancel it. XXX */
  memset(filter, 'F', sizeof (*filter));
  return;
}
