/*

  sshrcmd.c

  Author:
        Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/











#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "sshrcmd.h"

#ifndef SSH_SECURE_SHELL_PROGRAM
#define SSH_SECURE_SHELL_PROGRAM "/usr/local/bin/ssh2"
#endif /* ! SSH_SECURE_SHELL_PROGRAM */

#ifndef STDIN_FILENO
#define STDIN_FILENO 1
#endif /* ! STDIN_FILENO */

#ifndef STDOUT_FILENO
#define STDOUT_FILENO 1
#endif /* ! STDOUT_FILENO */

#ifndef STDERR_FILENO
#define STDERR_FILENO 1
#endif /* ! STDERR_FILENO */

pid_t ssh_run_remote_command(const char *remote_host,
                             const char *remote_port,
                             const char *remote_user,
                             const char *remote_command,
                             int *fd_stdin,
                             int *fd_stdout,
                             int *fd_stderr)
{
  int i, pipe_in[2], pipe_out[2], pipe_err[2], reserved[3];
  char *ssh_argv[64];
  pid_t pid;

  /* Ensure that we don't get fd's 0, 1, or 2 from this function. */
  if (pipe(&(reserved[0])) < 0)
    {
      return -1;
    }
  close(reserved[1]);
  if (pipe(&(reserved[1])) < 0)
    {
      close(reserved[0]);
      return -1;
    }
  /* Reserve stdin. */
  if (pipe(pipe_in) < 0)
    {
      close(reserved[0]);
      close(reserved[1]);
      close(reserved[2]);
      return -1;
    }
  /* Reserve stdout. */
  if (pipe(pipe_out) < 0)
    {
      close(reserved[0]);
      close(reserved[1]);
      close(reserved[2]);
      close(pipe_in[0]);
      close(pipe_in[1]);
      return -1;
    }
  /* Reserve stderr. */
  if (pipe(pipe_err) < 0)
    {
      close(reserved[0]);
      close(reserved[1]);
      close(reserved[2]);
      close(pipe_in[0]);
      close(pipe_in[1]);
      close(pipe_out[0]);
      close(pipe_out[1]);
      return -1;
    }
  /* Close the temp file descriptors. */
  close(reserved[0]);
  close(reserved[1]);
  close(reserved[2]);
  /* Fork the child. */
  pid = fork();
  if (pid < 0)
    {
      /* Fork failed */
      close(pipe_in[0]);
      close(pipe_in[1]);
      close(pipe_out[0]);
      close(pipe_out[1]);
      close(pipe_err[0]);
      close(pipe_err[1]);
      return -1;
    }
  if (pid > 0)
    {
      /* We are the parent.  Closing the pipe endpoints used by the child. */
      close(pipe_in[1]);
      close(pipe_out[1]);
      close(pipe_err[1]);
      /* Passing the file descriptors to the caller or closing them if
         caller doesn't want them. */
      if (fd_stdin)
        {
          *fd_stdin = pipe_in[0];
        }
      else
        {
          close(pipe_in[0]);
        }
      if (fd_stdout)
        {
          *fd_stdout = pipe_out[0];
        }
      else
        {
          close(pipe_out[0]);
        }
      if (fd_stderr)
        {
          *fd_stderr = pipe_err[0];
        }
      else
        {
          close(pipe_err[0]);
        }
      return pid;
    }
  /* We are the child.  Closing the pipe endpoints used by the parent. */
  close(pipe_in[0]);
  close(pipe_out[0]);
  close(pipe_err[0]);
  /* Make file descriptors 0, 1, and 2 copies of the pipe endpoint. */
  if (dup2(pipe_in[1], STDIN_FILENO) < 0)
    {
#ifdef DEBUG
      fprintf(stderr, "dup2() failed (%s:%u).\n", __FILE__, __LINE__);
#endif /* DEBUG */
      exit(-1);
    }
  else
    {
      close(pipe_in[1]);
    }
  if (dup2(pipe_out[1], STDOUT_FILENO) < 0)
    {
#ifdef DEBUG
      fprintf(stderr, "dup2() failed (%s:%u).\n", __FILE__, __LINE__);
#endif /* DEBUG */
      exit(-1);
    }
  else
    {
      close(pipe_out[1]);
    }
  if (dup2(pipe_err[1], STDERR_FILENO) < 0)
    {
#ifdef DEBUG
      fprintf(stderr, "dup2() failed (%s:%u).\n", __FILE__, __LINE__);
#endif /* DEBUG */
      exit(-1);
    }
  else
    {
      close(pipe_err[1]);
    }
  i = 0;
  /* Build argument vector. */
  ssh_argv[i++] = SSH_SECURE_SHELL_PROGRAM;
  /* Set possible remote part. */
  if (remote_port)
    {
      ssh_argv[i++] = "-p";
      ssh_argv[i++] = strdup(remote_port);
      if (! (ssh_argv[i - 1]))
        {
          exit(-1);
        }
    }
  /* Set possible remote user. */
  if (remote_user)
    {
      ssh_argv[i++] = "-l";
      ssh_argv[i++] = strdup(remote_user);
      if (! (ssh_argv[i - 1]))
        {
          exit(-1);
        }
    }
  /* Don't ask passwords or passphrases from the user. */
  ssh_argv[i++] = "-o";
  ssh_argv[i++] = "BatchMode yes";
  /* Don't ask host key decisions from the user. */
  ssh_argv[i++] = "-o";
  ssh_argv[i++] = "StrictHostKeyChecking yes";
  /* Don't forward X11. */
  ssh_argv[i++] = "-x";
  /* Don't forward authentication agent. */
  ssh_argv[i++] = "-a";
  /* Don't recognize an escape character. */
  ssh_argv[i++] = "-e";
  ssh_argv[i++] = "none";
  ssh_argv[i++] = strdup(remote_host);
  if (! (ssh_argv[i - 1]))
    {
      exit(-1);
    }
  ssh_argv[i++] = strdup(remote_command);
  if (! (ssh_argv[i - 1]))
    {
      exit(-1);
    }
  ssh_argv[i++] = NULL;
#ifdef DEBUG
  {
    int j;
    for (j = 0; j < i; j++)
      {
        fprintf(stderr, "argv[%d] = \"%s\"\n", j, ssh_argv[i]);
      }
  }
#endif /* DEBUG */
  execvp(ssh_argv[0], ssh_argv);
  exit(-1);
  /*NOTREACHED*/
}

#if 0

#include <sys/wait.h>

int main(int argc, char **argv)
{
  char *host, *cmd, buf[8];
  pid_t pid;
  ssize_t len;
  int fd_out, i, status;
  

  if (argc < 3)
    exit(-1);
  host = argv[1];
  for (i = 2; i < argc; i++)
    {
      if (i == 2)
        {
          cmd = strdup(argv[i]);
          if (! cmd)
            exit(-1);
        }
      else
        {
          cmd = (char *)realloc(cmd, strlen(cmd) + strlen(argv[i]) + 2);
          if (! cmd)
            exit(-1);
          cmd = strcat(cmd, " ");
          cmd = strcat(cmd, argv[i]);
        }
    }
  pid = ssh_run_remote_command(host,
                               NULL,
                               NULL,
                               cmd,
                               NULL,
                               &fd_out,
                               NULL);
  if (pid < 0)
    {
      exit(-1);
    }
  while ((len = read(fd_out, buf, sizeof (buf))) > 0)
    {
      write(1, buf, (size_t)len);
    }
  if (pid == waitpid(pid, &status, 0))
    {
      exit(status);
    }
  else
    {
      exit(-1);
    }
}
#endif /* 1 || 0 */

/* eof (sshrcmd.c) */
