/*

  sshstdiofilter.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#include "ssh2includes.h"
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#include "ssh1proto.h"
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
#include "sshfilterstream.h"
#include "sshtcp.h"
#include "sshcommon.h"
#include "sshstdiofilter.h"
#include "sshtty.h"
#include "sshappcommon.h"
#include "sshclient.h"
#include "sshencode.h"
#include "sshdsprintf.h"

#define SSH_DEBUG_MODULE "SshStdIOFilter"

#define STATE_BASIC     0
#define STATE_HAVE_CR   1
#define STATE_HAVE_ESC  2
#define STATE_LINEMODE  3

#define SSH_STDIO_FILTER_MAX_BUFFER_SIZE 2048

static int stdio_filter_state = STATE_BASIC;

void ssh_cancel_nonblocking(void);
void ssh_set_nonblocking(void);

#if 1
void ssh_buffer_consume_middle(SshBuffer buffer, size_t offset, size_t bytes);

void ssh_buffer_consume_middle(SshBuffer buffer, size_t offset, size_t bytes)
{
    if (bytes + offset > buffer->end - buffer->offset)
        ssh_fatal("buffer_consume_middle trying to get too many bytes");

    memmove(buffer->buf + buffer->offset + offset,
            buffer->buf + buffer->offset + offset + bytes,
            buffer->end - buffer->offset - offset - bytes);
    buffer->end -= bytes;
}
#endif

void ssh_escape_char_help(int esc_char)
{
  clearerr(stderr);        /* XXX */
  ssh_leave_raw_mode(-1);
  fprintf(stderr, "\n");
  ssh2_version(NULL);
  fprintf(stderr,
          "  Supported escape sequences:\n");
  fprintf(stderr,
          "  %c.  - terminate connection\n",
          esc_char);
  fprintf(stderr,
          "  %c^Z - suspend ssh\n",
          esc_char);
  fprintf(stderr,
          "  %c#  - list forwarded connections\n",
          esc_char);
  fprintf(stderr,
          "  %c-  - disable escape character uncancellably\n",
          esc_char);
  fprintf(stderr,
          "  %c?  - this message\n",
          esc_char);
  fprintf(stderr,
          "  %c%c  - send the escape character by typing it twice\n",
          esc_char, esc_char);
  fprintf(stderr,
          "  %cV   - dump version information to stderr\n", esc_char);
  fprintf(stderr,
          "  %cs   - dump statistics and connection information to stderr\n",
          esc_char);
  fprintf(stderr,
          "  %cr   - initiate rekey immediately with old algorithms (ie. "
          "changes\n"
          "         encryption and integrity keys)\n", esc_char);
  fprintf(stderr,
          "  %cl   - go to line mode (keystrokes are gathered to an internal\n"
          "         buffer, and will be output all at once when you enter\n"
          "         a newline)\n", esc_char);
  fprintf(stderr,
          "  (Note that escapes are only recognized immediately "
          "after newline.)\n");
  ssh_enter_raw_mode(-1, FALSE);
}

void ssh_cancel_nonblocking()
{
#if defined(O_NONBLOCK) && !defined(O_NONBLOCK_BROKEN)
  (void)fcntl(0, F_SETFL, 0);
  (void)fcntl(1, F_SETFL, 0);
#else /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
  (void)fcntl(0, F_SETFL, 0);
  (void)fcntl(1, F_SETFL, 0);
#endif /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
}

void ssh_set_nonblocking()
{
#if defined(O_NONBLOCK) && !defined(O_NONBLOCK_BROKEN)
  (void)fcntl(0, F_SETFL, O_NONBLOCK);
  (void)fcntl(1, F_SETFL, O_NONBLOCK);
#else /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
  (void)fcntl(0, F_SETFL, O_NDELAY);
  (void)fcntl(1, F_SETFL, O_NDELAY);
#endif /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
}

void ssh_escape_char_quit(int esc_char)
{
  /* Exit non-blocking raw mode. */
  ssh_leave_raw_mode(-1);
  ssh_cancel_nonblocking();

  /* Display the escape sequence. */
  fprintf(stderr, "%c.\n", esc_char);

  /* Forcibly exit the application immediately. */
  exit(0);
}

void ssh_escape_char_suspend(int esc_char)
{
#ifdef SIGWINCH
  struct winsize oldws, newws;
#endif /* SIGWINCH */

  /* Clear errors on stdout. */
  clearerr(stderr);        /* XXX */

  /* Exit non-blocking raw mode. */
  ssh_leave_raw_mode(-1);
  ssh_cancel_nonblocking();

  /* Print the escape sequence. */
  fprintf(stderr, "%c^Z\n", esc_char);

#ifdef SIGWINCH
  /* Save old window size. */
  ioctl(fileno(stdin), TIOCGWINSZ, &oldws);
#endif /* SIGWINCH */

  /* Send the suspend signal to the program
     itself. */
  kill(getpid(), SIGTSTP);

  /* Restore non-blocking raw mode. */
  ssh_set_nonblocking();
  ssh_enter_raw_mode(-1, FALSE);

#ifdef SIGWINCH
  /* Check if the window size has changed. */
  if (ioctl(fileno(stdin), TIOCGWINSZ, &newws) >= 0 &&
      (oldws.ws_row != newws.ws_row || oldws.ws_col != newws.ws_col ||
       oldws.ws_xpixel != newws.ws_xpixel || 
       oldws.ws_ypixel != newws.ws_ypixel))
    kill(getpid(), SIGWINCH);
#endif /* SIGWINCH */
}

void ssh_escape_char_list_connections(int esc_char, SshClient client)
{
  SshTcpForwardActive tmp_active_ptr;
  SshCommon common;

  clearerr(stderr);        /* XXX */
  ssh_leave_raw_mode(-1);
  ssh_cancel_nonblocking();

  common = client->common;
  
  fprintf(stderr, "\r\n");
  if (ssh_adt_num_objects(common->active_local_forwards) > 0)
    {      
      SshADTHandle h;
      
      fprintf(stderr, "Active local forwards:\r\n");
      for (h = ssh_adt_enumerate_start(common->active_local_forwards);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(common->active_local_forwards, h))
        {
          tmp_active_ptr = ssh_adt_get(common->active_local_forwards, h);

          fprintf(stderr, "  %s : %s : %s from %s\r\n",
                  tmp_active_ptr->port,
                  tmp_active_ptr->connect_to_host,
                  tmp_active_ptr->connect_to_port,
                  tmp_active_ptr->connect_from_host);
        }
    }
  else
    {
      fprintf(stderr, "No active local forwards.\r\n");
    }
  
  if (ssh_adt_num_objects(common->active_remote_forwards) > 0)
    {      
      SshADTHandle h;
      
      fprintf(stderr, "Active remote forwards:\r\n");
      for (h = ssh_adt_enumerate_start(common->active_remote_forwards);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(common->active_remote_forwards, h))
        {
          tmp_active_ptr = ssh_adt_get(common->active_remote_forwards, h);

          fprintf(stderr, "  %s : %s : %s from %s\r\n",
                  tmp_active_ptr->port,
                  tmp_active_ptr->connect_to_host,
                  tmp_active_ptr->connect_to_port,
                  tmp_active_ptr->connect_from_host);
        }
    }
  else
    {
      fprintf(stderr, "No active remote forwards.\r\n");
    }
  
  if (ssh_adt_num_objects(common->active_x11_forwards) > 0)
    {      
      SshADTHandle h;
      
      fprintf(stderr, "Active X11 forwards:\r\n");
      for (h = ssh_adt_enumerate_start(common->active_x11_forwards);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(common->active_x11_forwards, h))
        {
          tmp_active_ptr = ssh_adt_get(common->active_x11_forwards, h);

          fprintf(stderr, "  X11 forward from %s\r\n",
                  tmp_active_ptr->connect_from_host);
        }
    }
  else
    {
      fprintf(stderr, "No active X11 forwards.\r\n");
    }

  ssh_set_nonblocking();
  ssh_enter_raw_mode(-1, FALSE);
}

void ssh_escape_char_dump_version(int esc_char)
{
  clearerr(stderr);        /* XXX */
  ssh_leave_raw_mode(-1);
  fprintf(stderr, "\n");
  ssh2_version(NULL);
  ssh_enter_raw_mode(-1, FALSE);
}

void ssh_escape_char_dump_statistics(int esc_char, SshClient client)
{


  SshTransportStatisticsStruct stats;
  char *local_version_string = NULL;
  /* XXX this should be somewhere else. */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif /* MAXHOSTNAMELEN */
  char local_host_name[MAXHOSTNAMELEN + 1];
  
  SSH_PRECOND(client && client->config && client->common);
  clearerr(stderr);        /* XXX */
  ssh_leave_raw_mode(-1);
  fprintf(stderr, "\n");

#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
  if (client->ssh1_protocol_handle == NULL)
    {
      ssh_transport_get_statistics_tr(client->tr_context, &stats);
    }
  else
    {
      ssh1_get_statistics(client->ssh1_protocol_handle, &stats);
    }
#else /* WITH_INTERNAL_SSH1_EMULATION */
  ssh_transport_get_statistics_tr(client->tr_context, &stats);
#endif /* WITH_INTERNAL_SSH1_EMULATION */


#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */

  ssh_tcp_get_host_name(local_host_name, MAXHOSTNAMELEN);
  
  if (client->config->host_to_connect)
    fprintf(stderr, "remote host: %s\n", client->config->host_to_connect);
  if (local_host_name)
    fprintf(stderr, "local host: %s\n", local_host_name);
  
  ssh_dsprintf(&local_version_string,
               (client->config->ssh1compatibility ? 
                "SSH-1.99-%.200s" : "SSH-2.0-%.200s"),
               SSH2_PROTOCOL_VERSION_STRING);
  fprintf(stderr, "remote version: %s\n", client->remote_version);
  fprintf(stderr, "local version:  %s\n", local_version_string);
  ssh_xfree(local_version_string);
  fprintf(stderr, "compressed bytes in: %ld\n",
          stats.compressed_incoming_bytes);
  fprintf(stderr, "uncompressed bytes in: %ld\n",
          stats.uncompressed_incoming_bytes);
  fprintf(stderr, "compressed bytes out: %ld\n",
          stats.compressed_outgoing_bytes);
  fprintf(stderr, "uncompressed bytes out: %ld\n",
          stats.uncompressed_outgoing_bytes);
  fprintf(stderr, "packets in: %ld\n",
          stats.incoming_packets);
  fprintf(stderr, "packets out: %ld\n",
          stats.outgoing_packets);
  fprintf(stderr, "rekeys: %ld\n", stats.rekeys);
  
  fprintf(stderr, "\nAlgorithms:\n");

  fprintf(stderr, "Chosen key exchange algorithm: %s\n", stats.kex_name);
  fprintf(stderr, "Chosen host key algorithm: %s\n", stats.host_key_name);
  fprintf(stderr, "Common host key algorithms: %s\n", stats.host_key_names);
  fprintf(stderr, "\nAlgorithms client to server:\n");
  fprintf(stderr, "Cipher: %s\n", stats.cipher_c_to_s);
  fprintf(stderr, "MAC: %s\n", stats.mac_c_to_s);
  fprintf(stderr, "Compression: %s\n", stats.compression_c_to_s);
  fprintf(stderr, "\nAlgorithms server to client:\n");
  fprintf(stderr, "Cipher: %s\n", stats.cipher_s_to_c);
  fprintf(stderr, "MAC: %s\n", stats.mac_s_to_c);
  fprintf(stderr, "Compression: %s\n", stats.compression_s_to_c);
  
  ssh_enter_raw_mode(-1, FALSE);

}

void ssh_escape_char_request_rekey(int esc_char, SshClient client)
{
  SSH_PRECOND(client && client->common);
  clearerr(stderr);        /* XXX */
  ssh_leave_raw_mode(-1);
  fprintf(stderr, "\n");
#ifdef WITH_INTERNAL_SSH1_EMULATION
  if (client->ssh1_protocol_handle != NULL)
    {
      fprintf(stderr, "No rekeying in ssh1 emulation.\n");
    }
  else
#endif /* WITH_INTERNAL_SSH1_EMULATION */
    {
      ssh_common_request_rekey(client->common);
      fprintf(stderr, "Rekeying.\n");
    }
  ssh_enter_raw_mode(-1, FALSE);
}

void ssh_stdio_output_filter(void *context,
                             SshFilterGetCB get_data,
                             SshFilterCompletionCB completed,
                             void *internal_context)
{
  size_t offset, received_len;
  SshBuffer data;
  Boolean eof_received;

  (*get_data)(internal_context, &data, &offset, &eof_received);

  received_len = ssh_buffer_len(data) - offset;

  if (received_len == 0)
    {
      (*completed)(internal_context, SSH_FILTER_HOLD);
      return;
    }
  
  (*completed)(internal_context, SSH_FILTER_ACCEPT(received_len));
}

void ssh_stdio_input_filter(void *context,
                            SshFilterGetCB get_data,
                            SshFilterCompletionCB completed,
                            void *internal_context)
{
  size_t offset, received_len;
  SshBuffer data;
  Boolean eof_received;

  unsigned char *ucp;
  int i, c, e;
  SshClient client = (SshClient) context;
  
  (*get_data)(internal_context, &data, &offset, &eof_received);

  if (!context)
    {
      (*completed)(internal_context, SSH_FILTER_SHORTCIRCUIT);
      return;
    }
  e = *((unsigned char *)client->config->escape_char);
  if (!e)
    {
      (*completed)(internal_context, SSH_FILTER_SHORTCIRCUIT);
      return;
    }
  /* Compute the start and length of data that we have in the filter. */
  ucp = ssh_buffer_ptr(data);
  ucp += offset;
  received_len = ssh_buffer_len(data) - offset;

  if (received_len == 0)
    {
      (*completed)(internal_context, SSH_FILTER_HOLD);
      return;
    }

  for (i = 0; i < received_len; i++) {
    c = ucp[i];
    switch (stdio_filter_state) {
    case STATE_BASIC:
      if ((c == '\n') || (c == '\r'))
        stdio_filter_state = STATE_HAVE_CR;
      break;
    case STATE_HAVE_CR:
      if (c == e) {
        stdio_filter_state = STATE_HAVE_ESC;
        ssh_buffer_consume_middle(data, offset + i, 1);
        received_len--; /* Buffer is now shorter    */
        i--;            /* Re-exam current position */
      } else {
        if ((c != '\n') && (c != '\r'))
          stdio_filter_state = STATE_BASIC;
      }
      break;
    case STATE_LINEMODE:
      if ((c == '\n') || (c == '\r') ||
          (ssh_buffer_len(client->stdio_filter_buffer) >=
           SSH_STDIO_FILTER_MAX_BUFFER_SIZE))
        {
          size_t stored_len;
          
          SSH_TRACE(3, ("Going to normal mode."));
          /* Go back to normal state. */
          /* Copy stuff from internal buffer to filter buffer to
             be sent downstream. */
          stdio_filter_state = STATE_BASIC;
          stored_len = ssh_buffer_len(client->stdio_filter_buffer);
          ssh_xbuffer_append(client->stdio_filter_buffer,
                             ssh_buffer_ptr(data) + offset + i,
                             ssh_buffer_len(data) - offset - i);
          ssh_buffer_consume_end(data, ssh_buffer_len(data) - offset -i);
          ssh_xbuffer_append(data, ssh_buffer_ptr(client->stdio_filter_buffer),
                             ssh_buffer_len(client->stdio_filter_buffer));
          /* Clear sensitive data from the buffer (in linemode the
             stuff entered is usually password and such). */
          memset(ssh_buffer_ptr(client->stdio_filter_buffer), 0,
                 ssh_buffer_len(client->stdio_filter_buffer));
          received_len += stored_len;
          /* Re-examine current position. */
          i--;
          /* XXX Empty internal buffer. */
          ssh_buffer_clear(client->stdio_filter_buffer);
        }
      else
        {
          SSH_TRACE(3, ("Processed."));
          
          /* Remove character from data buffer, and append to internal
             buffer. */
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_xbuffer_append(client->stdio_filter_buffer,
                             (unsigned char *)&c, 1);
        }
      break;
    case STATE_HAVE_ESC:
      switch (c)
        {
        case '?':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_help(e);
          break;
        case '.':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_quit(e);
          break;
        case 26: /* ^Z */
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_suspend(e);
          break;
        case '#':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_list_connections(e, client);
          break;
        case '-':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          (*completed)(internal_context, SSH_FILTER_SHORTCIRCUIT);
          return;
        case 'V':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_dump_version(e);
          break;
        case 's':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_dump_statistics(e, client);
          break;
        case 'r':
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          ssh_escape_char_request_rekey(e, client);
          break;
        case 'l':
          /* Line mode. We will exit line mode on CR, and then all
             data typed by the user is sent in one blob. */
          ssh_buffer_consume_middle(data, offset + i, 1);
          received_len--; /* Buffer is now shorter    */
          i--;            /* Re-exam current position */
          stdio_filter_state = STATE_LINEMODE;
          fprintf(stderr, "\r\n<in linemode>\r\n");
          continue;
        default:
          if (c != e)
            {
              ssh_buffer_consume_middle(data, offset + i, 1);
              received_len--; /* Buffer is now shorter    */
              i--;            /* Re-exam current position */
            }
          break;
        }
      stdio_filter_state = STATE_BASIC;
      break;
    default:
      ssh_fatal("Unknown state in stdio filter.");
    }
  }

  (*completed)(internal_context, SSH_FILTER_ACCEPT(received_len));
}

void ssh_stdio_filter_destroy(void *context)
{
    return;
}
