/*

  sshuserfiles.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Simple functions that update user's files. These are not platform-dependent.

*/

#include "sshincludes.h"
#include "sshuserfiles.h"
#include "sshencode.h"
#include "sshuser.h"
#include "sshconfig.h"
#include "sshbase64.h"
#include "sshmiscstring.h"
#include "sshdsprintf.h"
#include "sshuserfile.h"
#include "sshappcommon.h"






#define SSH_DEBUG_MODULE "SshUserFiles"


/* Creates a new random number generator, and loads the random seed file
   into the generator. */

void ssh_randseed_open(SshUser user, SshConfig config)
{
  /* Load the random seed file and mix it into the generator. */
  ssh_randseed_load(user, config);
  return;
}

/* Return a pointer to user's ssh2 directory.  The directory is
   created if `create_if_needed' is TRUE.  Return NULL on failure. The
   caller is responsible for freeing the returned value with ssh_xfree
   when no longer needed. */

char *ssh_userdir(SshUser user, SshConfig config, Boolean create_if_needed)
{





  char *sshdir;
  char *ret = NULL;
  struct stat st;
  /* create the .ssh2 directory name */
  sshdir = ssh_user_conf_dir(user, config);
  ret = sshdir;

  ssh_userfile_init(ssh_user_name(user), ssh_user_uid(user),
                    ssh_user_gid(user), NULL, NULL);
  if (ssh_userfile_stat(ssh_user_uid(user), sshdir, &st) < 0)
    {
      if (create_if_needed)
        {
          if (ssh_userfile_mkdir(ssh_user_uid(user), sshdir, 0755) < 0)
            {
              SSH_DEBUG(2, ("could not create user's ssh directory %s",
                            sshdir));
              ssh_xfree(sshdir);
              ret = NULL;
            }
        }
      else
        {
          ssh_xfree(sshdir);
          ret = NULL;
        }
    }
  ssh_userfile_uninit();
  return ret;

}

/* Forms an absolute path from `file', returning an xmallocated
   string. */
char *ssh_userdir_prefix(SshUser user, SshConfig config,
                         const char *file, const char *default_prefix,
                         Boolean create_if_needed)
{
  char *prefix = NULL, *result = NULL;
  
  if (ssh_app_is_file_name_absolute(file))
    return ssh_xstrdup(file);

  prefix = ssh_userdir(user, config, FALSE);
  if (ssh_user_uid(user) != 0 && prefix)
    {
      ssh_xdsprintf(&result, "%s/%s", prefix, file);
      ssh_xfree(prefix);
      return result;
    }
  else
    { 
      ssh_xfree(prefix);
      if (!default_prefix)
        return NULL;
      ssh_xdsprintf(&result, "%s/%s", default_prefix, file);
      return result;
    }
}

char *ssh_user_conf_dir(SshUser user, SshConfig config)
{
  /* Get template from config data */
  const char *conf_dir = (config && config->user_conf_dir) ?
                         config->user_conf_dir : SSH_USER_CONFIG_DIRECTORY;
  return ssh_user_build_dir(user, conf_dir);
}

char *ssh_user_build_dir(SshUser user, const char *path_template)
{



  const char *user_str;
  char *conf_dir;
  unsigned long x;
  char buf[32];
  char *tmp;

  /* Replace %D with home directory */
  user_str = ssh_user_dir(user);
  conf_dir = ssh_replace_in_string(path_template,
                              "%D", (user_str != NULL) ? user_str : "");

  /* Ssh_Replace %U with user login name */

  user_str = ssh_user_name(user);



  tmp = ssh_replace_in_string(conf_dir, 
                              "%U", (user_str != NULL) ? user_str : "");
  ssh_xfree(conf_dir);
  conf_dir = tmp;

  /* Ssh_Replace %IU with user ID number */
  x = (unsigned long)ssh_user_uid(user);
  ssh_snprintf(buf, sizeof (buf), "%lu", x);
  tmp = ssh_replace_in_string(conf_dir,
                              "%IU", (user_str != NULL) ? user_str : "");
  ssh_xfree(conf_dir);
  conf_dir = tmp;

  /* Ssh_Replace %IG with group ID number */
  x = (unsigned long)ssh_user_gid(user);
  ssh_snprintf(buf, sizeof (buf), "%lu", x);
  tmp = ssh_replace_in_string(conf_dir,
                              "%IG", (user_str != NULL) ? user_str : "");
  ssh_xfree(conf_dir);
  conf_dir = tmp;

  return conf_dir;

}
