/*

  sshkeyfile.h

  Authors:
        Tatu Ylnen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Simple functions that update user's (key)files.

*/

#include "sshincludes.h"
#include "sshkeyfile.h"
#include "sshuserfile.h"
#include "sshkeyblob2.h"
#include "sshkeyblob1.h"
#include "ssh2pubkeyencode.h"
#include "sshbase64.h"

#define SSH_DEBUG_MODULE "SshKeyFile"

unsigned long ssh2_key_blob_read(SshUser user, const char *fname,
                                 Boolean try_convert_ssh1_cert,
                                 char **comment,
                                 unsigned char **blob,
                                 size_t *bloblen, void *context)
{
  unsigned char *pkeypem = NULL;
  size_t pkeylen;
  unsigned long kind;


  ssh_userfile_init(ssh_user_name(user), ssh_user_uid(user),
                    ssh_user_gid(user), NULL, NULL);

  
  if (ssh_blob_read(user, fname, &pkeypem, &pkeylen, context))
    {

      ssh_userfile_uninit();

      return SSH_KEY_MAGIC_FAIL;
    }
  kind = ssh2_key_blob_decode(pkeypem, pkeylen,
                              try_convert_ssh1_cert,
                              comment, blob, bloblen);

  /* Just put the filename into comment string for ssh1 keys. */
  if (comment &&
      (kind == SSH_KEY_MAGIC_SSH1_PRIVATE ||
       kind == SSH_KEY_MAGIC_SSH1_PRIVATE_ENCRYPTED))
    *comment = ssh_xstrdup(fname);

  ssh_userfile_uninit();

  return kind;
}

Boolean
ssh2_key_blob_write(SshUser user, const char *fname, mode_t mode,
                    unsigned long magic,
                    const char *comment, const unsigned char *key,
                    size_t keylen, void *context)
{
  char *subject = NULL;
  unsigned char *encoded;
  size_t encoded_len;


  if (user && ssh_user_name(user))
    subject = ssh_xstrdup(ssh_user_name(user));
  else
    subject = NULL;


  if (ssh2_key_blob_encode(magic,
                           subject, comment, key, keylen,
                           &encoded, &encoded_len))
    {
      Boolean rv = ssh_blob_write(user, fname, mode,
                                  encoded, encoded_len,
                                  context);
      ssh_xfree(encoded);
      return rv;
    }
  return FALSE;
}

/* Read a public key from a file. Return NULL on failure. */

SshPublicKey ssh_pubkey_read(SshUser user, const char *fname, char **comment,
                             void *context)
{
  unsigned char *pkeybuf;
  size_t pkeylen;
  unsigned long magic;
  SshPublicKey pubkey = NULL;

  pkeybuf = NULL;
  magic = ssh2_key_blob_read(user,
                             fname,
                             FALSE,
                             comment,
                             &pkeybuf,
                             &pkeylen,
                             context);

  if ((magic != SSH_KEY_MAGIC_PUBLIC) &&
      (magic != SSH_KEY_MAGIC_SSH1_PUBLIC))
    goto fail;

  if (magic == SSH_KEY_MAGIC_PUBLIC)
    {
      pubkey = ssh_decode_pubkeyblob(pkeybuf, pkeylen);
      if (pubkey == NULL)
        goto fail;
    }
  else if (magic == SSH_KEY_MAGIC_SSH1_PUBLIC)
    {
      if (comment)
        ssh_xfree(*comment);

      if (ssh1_decode_pubkeyblob(pkeybuf, pkeylen, comment,
                                 &pubkey) != SSH_CRYPTO_OK)
        goto fail;
    }

  return pubkey;

  /* cleanup */
fail:
  if (pkeybuf != NULL)
    {
      memset(pkeybuf, 0, pkeylen);
      ssh_xfree(pkeybuf);
    }
  return NULL;
}

/* Write a public key to a file. Returns TRUE on error. */

Boolean ssh_pubkey_write(SshUser user, const char *fname, const char *comment,
                         SshPublicKey key, void *context)
{
  unsigned char *pkeybuf;
  size_t pkeylen;
  Boolean ret;

  if ((pkeylen = ssh_encode_pubkeyblob(key, &pkeybuf)) == 0)
    return TRUE;

  ret = ssh2_key_blob_write(user, fname, 0644,
                           SSH_KEY_MAGIC_PUBLIC,
                           comment, pkeybuf, pkeylen, context);
  memset(pkeybuf, 0, pkeylen);
  ssh_xfree(pkeybuf);

  return ret;
}


/* Read a private key from a file. Returns SSH_PRIVKEY_OK on
   success. */

SshPrivKeyResult ssh_privkey_read(SshUser user,
                                  const char *fname,
                                  const char *passphrase,
                                  char **comment,
                                  SshPrivateKey *return_prvkey)
{
  SshCryptoStatus code;
  SshPrivateKey privkey;
  unsigned char *pkeybuf;
  size_t pkeylen;
  unsigned long magic;

  pkeybuf = NULL;
  magic = ssh2_key_blob_read(user,
                             fname,
                             FALSE,
                             comment,
                             &pkeybuf,
                             &pkeylen,
                             NULL);

  if (magic == SSH_KEY_MAGIC_FAIL)
    return SSH_PRIVKEY_KEY_UNREADABLE;

  if ((magic != SSH_KEY_MAGIC_PRIVATE) &&
      (magic != SSH_KEY_MAGIC_PRIVATE_ENCRYPTED) &&
      (magic != SSH_KEY_MAGIC_SSH1_PRIVATE) &&
      (magic != SSH_KEY_MAGIC_SSH1_PRIVATE_ENCRYPTED))
    goto fail;

  switch(magic)
    {
    case SSH_KEY_MAGIC_PRIVATE:
      if ((code = ssh_private_key_import_with_passphrase(pkeybuf,
                                                         pkeylen,
                                                         "",
                                                         &privkey)
           != SSH_CRYPTO_OK))
        {
          ssh_warning("ssh_privkey_read: %s.",
                      ssh_crypto_status_message(code));
          goto fail;
        }
      break;

    case SSH_KEY_MAGIC_PRIVATE_ENCRYPTED:
      if ((code = ssh_private_key_import_with_passphrase(pkeybuf,
                                                         pkeylen,
                                                         passphrase,
                                                         &privkey)
           != SSH_CRYPTO_OK))
        {
#if 0
          ssh_warning("ssh_privkey_read: %s.",
                      ssh_crypto_status_message(code));
#endif
          goto fail;
        }
      break;

    case SSH_KEY_MAGIC_SSH1_PRIVATE:
      if (comment)
        {
          ssh_xfree(*comment);
          *comment = NULL;
        }

      if (ssh1_decode_privkeyblob(pkeybuf, pkeylen, "", comment,
                                  &privkey) != SSH_CRYPTO_OK)
        {
          if (comment)
            *comment = NULL;
          goto fail;
        }
      break;

    case SSH_KEY_MAGIC_SSH1_PRIVATE_ENCRYPTED:
      if (comment)
        {
          ssh_xfree(*comment);
          *comment = NULL;
        }
      if (ssh1_decode_privkeyblob(pkeybuf, pkeylen, passphrase, comment,
                                  &privkey) != SSH_CRYPTO_OK)
        {
          if (comment)
            *comment = NULL;
          goto fail;
        }
      break;

    default:
      goto fail;
    }

  /* cleanup */

  memset(pkeybuf, 0, pkeylen);
  ssh_xfree(pkeybuf);

  *return_prvkey = privkey;
  return SSH_PRIVKEY_OK;

fail:
  if (pkeybuf != NULL)
    {
      memset(pkeybuf, 0, pkeylen);
      ssh_xfree(pkeybuf);
    }
  *return_prvkey = NULL;
  return SSH_PRIVKEY_FAILURE;
}

/* Write a private key to a file with a passphrase. Return TRUE on error. */

Boolean ssh_privkey_write(SshUser user,
                          const char *fname, const char *passphrase,
                          const char *comment,
                          SshPrivateKey key,
                          void *context)
{
  unsigned char *pkeybuf;
  size_t pkeylen;
  SshCryptoStatus code;
  Boolean ret;

  if ((code = ssh_private_key_export_with_passphrase(key,
                                                     SSH_PASSPHRASE_CIPHER,
                                                     passphrase,
                                                     &pkeybuf, &pkeylen))
      != SSH_CRYPTO_OK)
    {
      ssh_warning("ssh_privkey_write: %s.", ssh_crypto_status_message(code));
      return (int) code;
    }

  ret = ssh2_key_blob_write(user, fname, 0600,
                            SSH_KEY_MAGIC_PRIVATE_ENCRYPTED,
                            comment, pkeybuf, pkeylen, context);
  memset(pkeybuf, 0, pkeylen);
  ssh_xfree(pkeybuf);

  return ret;
}

/* Generate a name string from any blob.  String consists of
   caller given string and space and sha1 hash of the blob in hex.
   String is allocated with ssh_xmalloc. */
char *ssh_generate_name_from_blob(char *name,
                                  unsigned char *blob, size_t bloblen)
{
  SshHash hash;
  char *buf;
  unsigned char *digest;
  size_t len, namelen;
  int i;

  if (!name)
    name = "???";
  if (ssh_hash_allocate("sha1", &hash) != SSH_CRYPTO_OK)
      return ssh_xstrdup(name);
  namelen = strlen(name);
  ssh_hash_update(hash, blob, bloblen);
  len = ssh_hash_digest_length(hash);
  digest = ssh_xmalloc(len);
  ssh_hash_final(hash, digest);
  ssh_hash_free(hash);
  buf = ssh_xmalloc(namelen + 1 + (len * 2) + 1);
  strncpy(buf, name, namelen);
  buf[namelen] = ' ';
  for (i = 0; i < len; i++)
    ssh_snprintf(&(buf[namelen + 1 + (i * 2)]), 3, "%02x", digest[i]);
  ssh_xfree(digest);
  return buf;
}

/* Reads a blob into a buffer. Return TRUE on failure.  The caller must free
   `*blob' with ssh_xfree when no longer needed. */

Boolean ssh_blob_read(SshUser user, const char *fname, unsigned char **blob,
                      size_t *bloblen, void *context)
{
  SshUserFile f;
  unsigned char *data;
  struct stat st;
  size_t datalen;
  int flags = O_RDONLY;





  *bloblen = 0;
  *blob = NULL;


  ssh_userfile_init(ssh_user_name(user), ssh_user_uid(user),
                    ssh_user_gid(user), NULL, NULL);


  if (ssh_userfile_stat(ssh_user_uid(user), fname, &st) < 0)
    {
      SSH_DEBUG(2, ("file %s does not exist.", fname));      

      ssh_userfile_uninit();

      return TRUE;
    }

  datalen = st.st_size;
  data = ssh_xmalloc(datalen);

  if ((f = ssh_userfile_open(ssh_user_uid(user), fname, flags, 0)) == NULL)
    {
      SSH_DEBUG(2, ("Could not open %s.", fname));
      ssh_xfree(data);

      ssh_userfile_uninit();

      return TRUE;
    }

  if (ssh_userfile_read(f, data, datalen) != datalen)
    {
      SSH_DEBUG(2, ("Error while reading %s.", fname));
      memset(data, 0, datalen);
      ssh_xfree(data);
      ssh_userfile_close(f);

      ssh_userfile_uninit();

      return TRUE;
    }

  ssh_userfile_close(f);
  *blob = data;
  *bloblen = datalen;

  ssh_userfile_uninit();


  return FALSE;
}


/* Reads a blob in PEM format into a buffer and decodes it. Returns
   TRUE on failure.  The caller must free `*blob' with ssh_xfree when
   no longer needed. */
Boolean ssh_blob_read_pem(SshUser user, const char *fname, unsigned char **blob,
                          size_t *bloblen, void *context)
{
  unsigned char *tmp, *tmp2;
  size_t len, start, end;

  if (ssh_blob_read(user, fname, &tmp, &len, context))
    return TRUE;

  if (!ssh_base64_remove_headers(tmp, len, &start, &end))
    {
      ssh_xfree(tmp);
      return TRUE;
    }

  tmp2 = ssh_base64_remove_whitespace(tmp + start, end - start);
  *blob = ssh_base64_to_buf(tmp2, &len);
  *bloblen = len;

  ssh_xfree(tmp);
  ssh_xfree(tmp2);

  return FALSE;
}


/* Write a blob. Return TRUE on failure. */

Boolean ssh_blob_write(SshUser user, const char *fname, mode_t mode,
                       const unsigned char *blob, size_t bloblen,
                       void *context)
{
  SshUserFile f;
  int flags = O_WRONLY | O_CREAT | O_TRUNC;





  if ((f = ssh_userfile_open(ssh_user_uid(user), fname, flags,
                         mode)) == NULL)
    {
      SSH_DEBUG(2, ("could not open %s.", fname));
      return TRUE;
    }

  if(ssh_userfile_write(f, blob, bloblen) != bloblen)
    {
      SSH_DEBUG(2, ("failed to write %s.", fname));
      return TRUE;
    }

  ssh_userfile_close(f);

  return FALSE;
}
