/*
  File: sshprvkey.c

  Authors:
        Tero T Mononen <tmo@ssh.fi>

  Description:
        Routines to decode and encode various private key formats
        understood by the SSH library.

  Copyright:
        Copyright (c) 1999-2001 SSH Communications Security, Finland
        All rights reserved
*/

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshprvkey.h"






#include "sshkeyblob1.h"
#include "sshkeyblob2.h"

#define SSH_DEBUG_MODULE "SshSKB"

struct SshSkbTypeNameMap {
  SshSKBType type;
  char *name;
};

static struct SshSkbTypeNameMap type_name_map[] = {
  { SSH_SKB_SSH, "ssh-crypto-library-private-key@ssh.com" },
  { SSH_SKB_SSH_1, "secure-shell-1-private-key@ssh.com" },
  { SSH_SKB_SSH_2, "secure-shell-2-private-key@ssh.com" },
  { SSH_SKB_SSH_X509, "x509-raw-private-key@ssh.com" },
  { SSH_SKB_PKCS1, "pkcs1" },
  { SSH_SKB_PKCS8, "pkcs8" },
  { SSH_SKB_PKCS8_SHROUDED, "pkcs8-shrouded" },
  { SSH_SKB_PKCS12_BROWSER_KEY, "pkcs12" },
  { SSH_SKB_UNKNOWN, NULL }
};

SshCryptoStatus
ssh_skb_get_info(const unsigned char *data, size_t len,
                 unsigned char **unarmored_data, size_t *unarmored_len,
                 SshSKBType *kind, char **comment)
{
  unsigned long magic;
  SshPrivateKey prv = NULL;
  unsigned char *blob = NULL, *tmp;
  size_t bloblen = 0;
  char *tmpcomment = NULL;

  if (comment)
    *comment = NULL;




















































  tmp = ssh_xmemdup(data, len);
  magic = ssh2_key_blob_decode(tmp, len, FALSE, &tmpcomment, &blob, &bloblen);

  if (magic == SSH_KEY_MAGIC_SSH1_PRIVATE ||
      magic == SSH_KEY_MAGIC_SSH1_PRIVATE_ENCRYPTED)
    {
      if (kind)
        *kind = SSH_SKB_SSH_1;
      if (unarmored_len)
        *unarmored_len = bloblen;
      if (unarmored_data)
        {
          *unarmored_data = blob;
        }
      else
        {
          memset(blob, 0, bloblen);
          ssh_xfree(blob);
        }
      goto success;
    }

  if (magic == SSH_KEY_MAGIC_PRIVATE ||
      magic == SSH_KEY_MAGIC_PRIVATE_ENCRYPTED)
    {
      if (kind)
        *kind = SSH_SKB_SSH_2;
      if (unarmored_len)
        *unarmored_len = bloblen;
      if (unarmored_data)
        {
          *unarmored_data = blob;
        }
      else
        {
          memset(blob, 0, bloblen);
          ssh_xfree(blob);
        }
      goto success;
    }

  if (ssh_private_key_import_with_passphrase(data, len, "", &prv)
      != SSH_CRYPTO_CORRUPTED_KEY_FORMAT)
    {
      if (kind)
        *kind = SSH_SKB_SSH;
      if (unarmored_len)
        *unarmored_len = len;
      if (unarmored_data)
        *unarmored_data = ssh_xmemdup(data, len);
      goto success;
    }

  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;

 success:
  if (comment)
    *comment = tmpcomment;
  if (prv)
    ssh_private_key_free(prv);
  return SSH_CRYPTO_OK;
}




















































































































































SshCryptoStatus
ssh_skb_decode(SshSKBType kind,
               const unsigned char *data, size_t len,
               Boolean password_is_passphrase,
               const unsigned char *password, size_t password_len,
               SshPrivateKey *key)
{
  unsigned char *blob, *tmp;
  size_t bloblen;




  switch (kind)
    {
    case SSH_SKB_SSH:
      if (password_is_passphrase)
        return ssh_private_key_import_with_passphrase(data, len,
                                                      (password ?
                                                       (const char *)password :
                                                       ""),
                                                      key);
      else
        return SSH_CRYPTO_INVALID_PASSPHRASE;

    case SSH_SKB_SSH_1:
      if (password_is_passphrase)
        return ssh1_decode_privkeyblob(data, len, password ?
                                       (const char *)password : "",
                                       NULL, key);
      else
        return SSH_CRYPTO_INVALID_PASSPHRASE;

    case SSH_SKB_SSH_2:
      tmp = ssh_xmemdup(data, len);
      switch (ssh2_key_blob_decode(tmp, len, FALSE,
                                   NULL, &blob, &bloblen))
        {
        case SSH_KEY_MAGIC_PRIVATE_ENCRYPTED:
          return ssh_private_key_import_with_passphrase(blob, bloblen,
                                                        password ?
                                                        (const char *)password:
                                                        "",
                                                        key);
        case SSH_KEY_MAGIC_PRIVATE:
          return ssh_private_key_import_with_passphrase(blob, bloblen,
                                                        "",
                                                        key);
        default:
          /* It can still be unarmored key.  Let's try. */
          return ssh_private_key_import_with_passphrase(
                         data, len,
                         ((password_is_passphrase && password) ?
                          (const char *)password :
                          ""),
                         key);
        }
      break;

    case SSH_SKB_SSH_X509:







    case SSH_SKB_PKCS1:







    case SSH_SKB_PKCS8_SHROUDED:















    case SSH_SKB_PKCS8:







    case SSH_SKB_PKCS12_BROWSER_KEY:







      
    case SSH_SKB_UNKNOWN:
      break;
    }
  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
}


SshCryptoStatus
ssh_skb_encode(SshSKBType kind,
               const SshPrivateKey key,
               const char *subject, const char *comment,
               const char *cipher,
               Boolean password_is_passphrase,
               const unsigned char *password, size_t password_len,
               unsigned char **data, size_t *len)
{
  unsigned char *blob;
  size_t bloblen;

  switch (kind)
    {
    case SSH_SKB_SSH:
      if (password_is_passphrase)
        return ssh_private_key_export_with_passphrase(key,
                                                      cipher,
                                                      password ?
                                                      (const char *)password :
                                                      "",
                                                      data, len);

      else
        return SSH_CRYPTO_INVALID_PASSPHRASE;

    case SSH_SKB_SSH_1:
      return SSH_CRYPTO_UNSUPPORTED;

    case SSH_SKB_SSH_2:
      if (ssh_private_key_export_with_passphrase(key,
                                                 cipher,
                                                 password ?
                                                 (const char *)password :
                                                 "",
                                                 &blob, &bloblen)
          == SSH_CRYPTO_OK)
        {
          if (ssh2_key_blob_encode(SSH_KEY_MAGIC_PRIVATE,
                                   subject, comment, blob, bloblen,
                                   data, len) == TRUE)
            return SSH_CRYPTO_OK;
        }
      return SSH_CRYPTO_INVALID_PASSPHRASE;

    case SSH_SKB_SSH_X509:







    case SSH_SKB_PKCS1:







    case SSH_SKB_PKCS8_SHROUDED:










    case SSH_SKB_PKCS8:






      
    case SSH_SKB_PKCS12_BROWSER_KEY:



      
    case SSH_SKB_UNKNOWN:
      break;
    }
  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
}

/* Returns information about the key type. *Needs_secret is set to TRUE
   if decoding if the blob requires some secret  code.
   *secret_is_passphrase is set to TRUE if the secret is
   passphrase instead of password. Returns TRUE on success and FALSE
   otherwise. */
Boolean ssh_skb_get_type_info(SshSKBType type,
                              Boolean *needs_secret,
                              Boolean *secret_is_passphrase,
                              const char **key_type_name_ret)
{
  /* Table of key types and their 'properties' */
  static const struct SshSKBPropertiesRec
    {
      SshSKBType type;
      Boolean needs_secret;
      Boolean is_passphrase;
      const char *name;
    } ssh_skb_properties[] =
      {
        { SSH_SKB_UNKNOWN, FALSE, FALSE, "Unkown"},
        { SSH_SKB_SSH, TRUE, FALSE, "SSH Key"},
        { SSH_SKB_SSH_1, TRUE, FALSE, "SSH 1 Key"},
        { SSH_SKB_SSH_2, TRUE, TRUE, "SSH 2 Key"},
        { SSH_SKB_SSH_X509, FALSE, FALSE, "SSH X509 Key"},
        { SSH_SKB_PKCS1, FALSE, FALSE, "PKCS-1 Key"},
        { SSH_SKB_PKCS8, FALSE, FALSE, "PKCS-8 Key"},
        { SSH_SKB_PKCS8_SHROUDED, TRUE, TRUE, "Shrouded PKCS-8"},
        { SSH_SKB_PKCS12_BROWSER_KEY, TRUE, TRUE, "PKCS#12 key"}
      };
  int i, l;
  /* Find the right type. */
  l = sizeof(ssh_skb_properties) / sizeof(struct SshSKBPropertiesRec);
  for (i = 0; i < l; i++)
    {
      if (type == ssh_skb_properties[i].type)
        {
          /* Type found. */
          if (needs_secret)
            *needs_secret = ssh_skb_properties[i].needs_secret;
          if (secret_is_passphrase)
            *secret_is_passphrase = ssh_skb_properties[i].is_passphrase;
          if (key_type_name_ret)
            *key_type_name_ret = ssh_skb_properties[i].name;
          return TRUE;
        }
    }
  /* Type was not found. */
  return FALSE;
}

/* Maps type identifier to canonical name that can be used in protocols. */
const char *ssh_skb_type_to_name(SshSKBType kind)
{
  int i;

  for (i = 0; type_name_map[i].name != NULL; i++)
    {
      if (kind == type_name_map[i].type)
        return type_name_map[i].name;
    }
  return NULL;
}

/* Maps canonical name to type identifier. */
SshSKBType ssh_skb_name_to_type(const char *name)
{
  int i;

  for (i = 0; type_name_map[i].name != NULL; i++)
    {
      if (strcasecmp(name, type_name_map[i].name) == 0)
        return type_name_map[i].type;
    }
  return SSH_SKB_UNKNOWN;
}
