/*

  Authors: Timo J. Rinne <tri@ssh.com>
           Sami Lehtinen <sjl@ssh.com>
           
  Copyright (C) 1996,2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Simple Namelist. Compute the section between two name lists, SSHv2
  style.

  Inspired by namelist in sshcrypto.

*/
#include "sshincludes.h"
#include "sshcrypt.h"
#ifdef SSHDIST_CRYPT_COMPRESS
#include "sshcryptomisc/bufzip.h"
#endif /* SSHDIST_CRYPT_COMPRESS */

/* Simple ways of travelling the namelist. */
int ssh_snlist_name_len(const char *namelist)
{
  int i;
  if (namelist == NULL)
    return 0;
  for (i = 0; namelist[i] != ',' && namelist[i] != '\0'; i++)
    ;
  return i;
}

char *ssh_snlist_get_name(const char *namelist)
{
  int len = ssh_snlist_name_len(namelist);
  char *name = NULL;

  if (len > 0)
    name = ssh_xmemdup(namelist, len);
  return name;
}

const char *ssh_snlist_step_forward(const char *namelist)
{
  int len = ssh_snlist_name_len(namelist);

  if (len > 0)
    {
      if (namelist[len] != '\0')
        return namelist + len + 1;
    }

  return NULL;
}

char *ssh_snlist_intersection(const char *src1, const char *src2)
{
  int total_len1, total_len2, name_len1, name_len2, max_len1, max_len2;
  Boolean prev;
  const char *tmp;
  char *dest, *dest_start;

  /* Set up the destination buffer. */

  prev = FALSE;
  if ((dest = ssh_xmalloc(strlen(src1) + 1)) == NULL)
    return NULL;
  dest_start = dest;

  /* Start looping the two namelists. And seek for names of same
     length and only then compare. */

  for (total_len1 = 0, max_len1 = strlen(src1), max_len2 = strlen(src2);
       total_len1 < max_len1;)
    {
      /* Get name lenght */
      name_len1 = ssh_snlist_name_len(src1);

      /* Start inner loop */
      for (tmp = src2, total_len2 = 0; total_len2 < max_len2; )
        {
          name_len2 = ssh_snlist_name_len(tmp);

          if (name_len2 == name_len1)
            {
              if (memcmp(src1, tmp, name_len1) == 0)
                {
                  if (prev)
                    *dest++ = ',';
                  prev = TRUE;
                  memcpy(dest, src1, name_len1);
                  dest += name_len1;
                  break;
                }
            }
          total_len2 += name_len2;

          /* Tricky part is to notice that we need to check for terminating
             zero, and quit if found. */
          tmp += name_len2;
          if (*tmp == '\0')
            break;
          /* Not zero so get past that comma. */
          tmp++;
        }

      total_len1 += name_len1;

      src1 += name_len1;
      if (*src1 == '\0')
        break;
      src1++;
    }
  /* In any case place zero terminator to the namelist. */
  *dest = '\0';
  return dest_start;
}

/* Crypto library specific routines. */

#ifdef SSHDIST_CRYPT_GENCIPH
char *ssh_snlist_intersection_cipher(const char *src)
{
  char *buffer, *result = NULL;

  if ((buffer = ssh_cipher_get_supported()) != NULL)
    {
      result = ssh_snlist_intersection(src, buffer);
      ssh_xfree(buffer);
    }
  return result;
}
#endif /* SSHDIST_CRYPT_GENCIPH */

#ifdef SSHDIST_CRYPT_GENPKCS
char *ssh_snlist_intersection_public_key(const char *src)
{
  char *buffer, *result = NULL;

  if ((buffer = ssh_public_key_get_supported()) != NULL)
    {
      result = ssh_snlist_intersection(src, buffer);
      ssh_xfree(buffer);
    }
  return result;
}
#endif /* SSHDIST_CRYPT_GENPKCS */

#ifdef SSHDIST_CRYPT_GENMAC
char *ssh_snlist_intersection_mac(const char *src)
{
  char *buffer, *result = NULL;

  if ((buffer = ssh_mac_get_supported()) != NULL)
    {
      result = ssh_snlist_intersection(src, buffer);
      ssh_xfree(buffer);
    }
  return result;
}
#endif /* SSHDIST_CRYPT_GENMAC */

#ifdef SSHDIST_CRYPT_GENHASH
char *ssh_snlist_intersection_hash(const char *src)
{
  char *buffer, *result = NULL;

  if ((buffer = ssh_hash_get_supported()) != NULL)
    {
      result = ssh_snlist_intersection(src, buffer);
      ssh_xfree(buffer);
    }
  return result;
}
#endif /* SSHDIST_CRYPT_GENHASH */

/* Compression library specific routines. */

#ifdef SSHDIST_CRYPT_COMPRESS
char *ssh_snlist_intersection_compression(const char *src)
{
  char *buffer, *result = NULL;

  if ((buffer = ssh_compress_get_supported()) != NULL)
    {
      result = ssh_snlist_intersection(src, buffer);
      ssh_xfree(buffer);
    }
  return result;
}
#endif /* SSHDIST_CRYPT_COMPRESS */
