#include "sshincludes.h"
#include "sshencode.h"
#include "sshbuffer.h"
#include "sshgetput.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshhash/md5.h"
#include "sshcryptocore/namelist.h"

/* Magic numbers used for validy checks in SSH public key
   exported octet formats. */
#define SSH_PK_GROUP_RANDOMIZER_MAGIC 0x4c9356fe
#define SSH_PK_GROUP_MAGIC            0x89578271
#define SSH_PUBLIC_KEY_MAGIC          0x65c8b28a
#define SSH_PRIVATE_KEY_MAGIC         0x3f6ff9eb

/* Returns atleast one randomizer if buffer is long enough, else some
   appropriate error message.

   Output buffer contains magic cookie which is computed either with a
   hash function or some other means. Other very suitable possibility
   is to add the parameter information into the buffer. */

SshCryptoStatus
ssh_pk_group_export_randomizers(SshPkGroup group,
                                unsigned char **buf, size_t *buf_length)
{
  SshBufferStruct buffer;
  unsigned char *tmp;
  size_t tmplen;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;

  if (group->type->pk_group_export_randomizer == NULL)
    return SSH_CRYPTO_UNSUPPORTED;

  ssh_buffer_init(&buffer);

  /* Put magic and reserve space for total length */
  if (ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32,
                        (SshUInt32) SSH_PK_GROUP_RANDOMIZER_MAGIC,
                        SSH_FORMAT_UINT32, (SshUInt32) 0,
                        SSH_FORMAT_END) > 0)
    {
      while (TRUE)
        {
          if ((*group->type->pk_group_export_randomizer)(group, &tmp, &tmplen))
            {
              if (ssh_encode_buffer(&buffer,
                                    SSH_FORMAT_UINT32_STR, tmp, tmplen,
                                    SSH_FORMAT_END) == 0)
                {
                  ssh_buffer_uninit(&buffer);
                  return status;
                }
            }
          else
            break;
        }

      *buf_length = ssh_buffer_len(&buffer);
      if ((*buf = ssh_memdup(ssh_buffer_ptr(&buffer), *buf_length)) != NULL)
        {
          SSH_PUT_32BIT((*buf) + 4, *buf_length);
          status = SSH_CRYPTO_OK;
        }
    }
  ssh_buffer_uninit(&buffer);
  return status;
}

/* Add randomizers to randomizer list. */
SshCryptoStatus
ssh_pk_group_import_randomizers(SshPkGroup group,
                                const unsigned char *buf, size_t buf_length)
{
  SshInt32 total_length, length;
  SshUInt32 magic;
  int buf_consumed = 0, len_decoded = 0;

  if (group->type->pk_group_import_randomizer == NULL)
    return SSH_CRYPTO_UNSUPPORTED;

  if ((buf_consumed = ssh_decode_array(buf, buf_length,
                                       SSH_FORMAT_UINT32, &magic,
                                       SSH_FORMAT_UINT32, &total_length,
                                       SSH_FORMAT_END)) == 0)
    return SSH_CRYPTO_OPERATION_FAILED;

  if (magic != SSH_PK_GROUP_RANDOMIZER_MAGIC)
    {
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  total_length -= 8;

  while (total_length > 0)
    {
      if ((len_decoded =
           ssh_decode_array(buf + buf_consumed, buf_length - buf_consumed,
                            SSH_FORMAT_UINT32, &length,
                            SSH_FORMAT_END)) == 0)
        {
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      buf_consumed += len_decoded;
      if ((*group->type->pk_group_import_randomizer)(group->context,
                                                     buf + buf_consumed,
                                                     length) == FALSE)
        {
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      buf_consumed += length;
      total_length -= (length + 4);
    }

  return SSH_CRYPTO_OK;
}

/* Pk group format:

   uint32   magic
   uint32   total length
   uint32   group type name length n
   n bytes  group type name (contains also information on schemes)

   uint32   type specific part length n
   n bytes  type specific part
   */

SshCryptoStatus
ssh_pk_group_export(SshPkGroup group,
                    unsigned char **buf, size_t *buf_length)
{
  SshBufferStruct buffer;
  unsigned char *tmp;
  size_t tmplen;
  char *name;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;

  if (group->type->pk_group_export == NULL)
    return SSH_CRYPTO_UNSUPPORTED;

  ssh_buffer_init(&buffer);

  name = ssh_pk_group_name(group);
  if (!name ||
      ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32, (SshUInt32) SSH_PK_GROUP_MAGIC,
                        SSH_FORMAT_UINT32, (SshUInt32) 0,
                        SSH_FORMAT_UINT32_STR, name, strlen(name),
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      return status;
    }
  ssh_free(name);

  if (!(*group->type->pk_group_export)(group->context, &tmp, &tmplen))
    {
      ssh_buffer_uninit(&buffer);
      return status;
    }

  if (ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32_STR, tmp, tmplen,
                        SSH_FORMAT_END) == 0)
    {
      ssh_free(tmp);
      return status;
    }
  ssh_free(tmp);

  *buf_length = ssh_buffer_len(&buffer);
  if ((*buf = ssh_memdup(ssh_buffer_ptr(&buffer), *buf_length)) != NULL)
    {
      SSH_PUT_32BIT((*buf) + 4, *buf_length);
      status = SSH_CRYPTO_OK;
    }
  ssh_buffer_uninit(&buffer);
  return status;
}

SshCryptoStatus
ssh_pk_group_import(const unsigned char *buf, size_t buf_length,
                    SshPkGroup *group)
{
  SshBufferStruct buffer;
  SshUInt32 magic, total_length, length;
  size_t key_type_len;
  char *key_type, *name;
  const SshPkAction *action;
  void *scheme;
  SshPkGroup pk_group;
  SshNameTree tree;
  SshNameNode node;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;
  unsigned int i;

  ssh_buffer_init(&buffer);
  if (ssh_buffer_append(&buffer, buf, buf_length) != SSH_BUFFER_OK)
    return SSH_CRYPTO_OPERATION_FAILED;

  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &magic,
                        SSH_FORMAT_UINT32, &total_length,
                        SSH_FORMAT_UINT32_STR,
                        &key_type, &key_type_len,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if (magic != SSH_PK_GROUP_MAGIC)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  ssh_ntree_allocate(&tree);
  if (tree == NULL)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  ssh_ntree_parse(key_type, tree);
  node = ssh_ntree_get_root(tree);

  /* Find correct key type. We could use action lists now, but for
     simplicity don't. However with some other formats action lists
     and ssh_pk_group_generate might allow simpler implementation. */
  name = ssh_nnode_get_identifier(node);
  for (i = 0, pk_group = NULL;
       name && ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name;
       i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, name) == 0)
        {
          /* Allocate */
          if ((pk_group = ssh_malloc(sizeof(*pk_group))) != NULL)
            {
              pk_group->type = ssh_pk_type_slots[i];
              pk_group->diffie_hellman = NULL;
              break;
            }
        }
    }
  ssh_free(name);

  if (pk_group == NULL)
    {
      ssh_free(key_type);
      ssh_ntree_free(tree);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  status = SSH_CRYPTO_OK;

  /* Check the name tree for schemes. */
  node = ssh_nnode_get_child(node);
  while (node)
    {
      name = ssh_nnode_get_identifier(node);
      action = ssh_pk_find_scheme_action(pk_group->type->action_list,
                                         name,
                                         SSH_PK_FLAG_PK_GROUP);
      if (!action)
        {
          ssh_free(name);
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }
      scheme = ssh_pk_find_generic(name, action->type, action->type_size);
      ssh_free(name);

      if (scheme == NULL)
        {
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }
      status = ssh_pk_group_set_scheme(pk_group, scheme, action->scheme_flag);
      if (status != SSH_CRYPTO_OK)
        {
          break;
        }
      node = ssh_nnode_get_parent(node);
      if (node)
        node = ssh_nnode_get_next(node);
    }
  ssh_ntree_free(tree);

  if (status != SSH_CRYPTO_OK)
    {
      ssh_free(pk_group);
      ssh_buffer_uninit(&buffer);
      return status;
    }

  /* Read the final part and generate internal context. */
  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &length,
                        SSH_FORMAT_END) == 0 ||
      length > ssh_buffer_len(&buffer))
    {
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      ssh_free(pk_group);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if ((*pk_group->type->pk_group_import)(ssh_buffer_ptr(&buffer),
                                         length,
                                         &pk_group->context) == FALSE)
    {
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      ssh_free(pk_group);
      return SSH_CRYPTO_OPERATION_FAILED;
    }
  ssh_free(key_type);
  ssh_buffer_uninit(&buffer);

  /* Set for output. */
  *group = pk_group;
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_public_key_export(SshPublicKey key,
                      unsigned char **buf, size_t *length_return)
{
  SshBufferStruct buffer;
  unsigned char *tmp;
  size_t tmplen;
  char *name;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;

  if (key->type->public_key_export == NULL)
    return SSH_CRYPTO_UNSUPPORTED;

  ssh_buffer_init(&buffer);

  name = ssh_public_key_name(key);
  if (!name ||
      ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32, (SshUInt32) SSH_PUBLIC_KEY_MAGIC,
                        SSH_FORMAT_UINT32, (SshUInt32) 0,
                        SSH_FORMAT_UINT32_STR, name, strlen(name),
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(name);
      return status;
    }
  ssh_free(name);

  if ((*key->type->public_key_export)(key->context,
                                      &tmp, &tmplen) == FALSE)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if (ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32_STR, tmp, tmplen,
                        SSH_FORMAT_END) == 0)
    {
      ssh_free(tmp);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }
  ssh_free(tmp);

  /* Get the buffer information. */
  *length_return = ssh_buffer_len(&buffer);
  if ((*buf = ssh_memdup(ssh_buffer_ptr(&buffer), *length_return)) != NULL)
    {
      SSH_PUT_32BIT(*buf + 4, *length_return);
      status = SSH_CRYPTO_OK;
    }
  ssh_buffer_uninit(&buffer);

  return status;
}

SshCryptoStatus
ssh_public_key_export_canonical(SshPublicKey key,
                                unsigned char **buf, size_t *length_return)
{
  SshBufferStruct buffer;
  unsigned char *tmp;
  size_t tmplen;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;

  if (key->type->public_key_export == NULL)
      return SSH_CRYPTO_UNSUPPORTED;

  /* Encoding of the public key, in SSH format. */

  if (key == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  if (key->type == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;

  if (!(*key->type->public_key_export)(key->context, &tmp, &tmplen))
    return SSH_CRYPTO_OPERATION_FAILED;

  ssh_buffer_init(&buffer);

  /* Build the blob. */
  if (ssh_encode_buffer(&buffer,
                        SSH_FORMAT_UINT32, (SshUInt32) SSH_PUBLIC_KEY_MAGIC,
                        SSH_FORMAT_UINT32, (SshUInt32) 0,
                        SSH_FORMAT_UINT32_STR, key->type->name,
                        strlen(key->type->name),
                        SSH_FORMAT_UINT32_STR, tmp, tmplen,
                        SSH_FORMAT_END) == 0)
    {
      ssh_free(tmp);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }
  ssh_free(tmp);

  /* Get the buffer information. */
  *length_return = ssh_buffer_len(&buffer);
  if ((*buf = ssh_memdup(ssh_buffer_ptr(&buffer), *length_return)) != NULL)
    {
      SSH_PUT_32BIT(*buf + 4, *length_return);
      status = SSH_CRYPTO_OK;
    }

  ssh_buffer_uninit(&buffer);
  return status;
}

/* Export a private key. */

SshCryptoStatus
ssh_private_key_export_internal(SshPrivateKey key,
                                const char *cipher_name,
                                const unsigned char *cipher_key,
                                size_t cipher_keylen,
                                unsigned char **bufptr,
                                size_t *length_return,
                                Boolean expand_key)
{
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;
  SshBufferStruct buffer, encrypted;
  unsigned char byte;
  unsigned char *buf;
  size_t buf_length;
  SshCipher cipher;
  char *name;

  /* Check key len and expansion flag. */
  buf_length = ssh_cipher_get_key_length(cipher_name);
  buf        = NULL;
  if (buf_length == 0)
    buf_length = 32;

  if (buf_length > cipher_keylen || expand_key)
    {
      /* Expand encryption key. */
      if ((buf = ssh_malloc(buf_length)) != NULL)
        {
          ssh_hash_expand_key_internal(buf, buf_length,
                                       cipher_key, cipher_keylen,
                                       NULL, 0,
                                       &ssh_hash_md5_def);
        }
      else
        buf_length = 0;

      cipher_key    = buf;
      cipher_keylen = buf_length;
    }

  /* Allocate cipher. */
  if (cipher_keylen == 0 ||
      (status = ssh_cipher_allocate(cipher_name,
                                    cipher_key, cipher_keylen,
                                    TRUE,
                                    &cipher)) != SSH_CRYPTO_OK)
    {
      ssh_free(buf);
      return status;
    }
  /* Free the key buffer if it exists. */
  ssh_free(buf);

  /* Generate private key blob. */

  if (key->type->private_key_export == NULL ||
      (*key->type->private_key_export)(key->context,
                                       &buf, &buf_length) == FALSE)
    {
      ssh_cipher_free(cipher);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  /* Use buffer to append data. */
  ssh_buffer_init(&encrypted);

  ssh_encode_buffer(&encrypted,
                    SSH_FORMAT_UINT32_STR, buf, buf_length,
                    SSH_FORMAT_END);

  /* Free exact private key information. */
  memset(buf, 0, buf_length);
  ssh_free(buf);

  /* Add some padding. */
  while ((ssh_buffer_len(&encrypted) % ssh_cipher_get_block_length(cipher)) !=
         0)
    {
      byte = ssh_random_get_byte();
      ssh_buffer_append(&encrypted, &byte, 1);
    }

  /* Encrypt buffer. */
  if (ssh_cipher_transform(cipher, ssh_buffer_ptr(&encrypted),
                           ssh_buffer_ptr(&encrypted),
                           ssh_buffer_len(&encrypted)) != SSH_CRYPTO_OK)
    {
      ssh_buffer_uninit(&encrypted);
      ssh_cipher_free(cipher);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  /* Free cipher. */
  ssh_cipher_free(cipher);

  /* Initialize the actual private key buffer. */
  ssh_buffer_init(&buffer);

  name = ssh_private_key_name(key);

  ssh_encode_buffer(&buffer,
                    SSH_FORMAT_UINT32, (SshUInt32) SSH_PRIVATE_KEY_MAGIC,
                    SSH_FORMAT_UINT32, (SshUInt32) 0,
                    SSH_FORMAT_UINT32_STR,
                    name, strlen(name),
                    SSH_FORMAT_UINT32_STR,
                    cipher_name, strlen(cipher_name),
                    SSH_FORMAT_UINT32_STR,
                    ssh_buffer_ptr(&encrypted), ssh_buffer_len(&encrypted),
                    SSH_FORMAT_END);

  ssh_free(name);
  /* Free encrypted buffer. */
  ssh_buffer_uninit(&encrypted);

  /* Get the buffer information. */
  *length_return = ssh_buffer_len(&buffer);
  if ((*bufptr = ssh_memdup(ssh_buffer_ptr(&buffer), *length_return)) != NULL)
    {
      SSH_PUT_32BIT(*bufptr + 4, *length_return);
      status = SSH_CRYPTO_OK;
    }
  /* Free buffer. */
  ssh_buffer_uninit(&buffer);

  return status;
}

SshCryptoStatus
ssh_private_key_import_internal(const unsigned char *buf, size_t len,
                                const unsigned char *cipher_key,
                                size_t cipher_keylen,
                                SshPrivateKey *key,
                                Boolean expand_key)
{
  SshBufferStruct buffer;
  SshUInt32 pk_magic, pk_length, length, tmp_length;
  char *key_type, *cipher_name, *name;
  unsigned char *tmp;
  size_t tmplen;
  unsigned int i;
  SshPrivateKey private_key;
  SshCipher cipher;
  SshCryptoStatus status = SSH_CRYPTO_OPERATION_FAILED;
  const SshPkAction *action;
  void *scheme;
  SshNameTree tree;
  SshNameNode node, child;
  SshNameTreeStatus nstat;

  ssh_buffer_init(&buffer);
  if (ssh_buffer_append(&buffer, buf, len) != SSH_BUFFER_OK ||
      ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &pk_magic,
                        SSH_FORMAT_UINT32, &pk_length,
                        SSH_FORMAT_UINT32_STR, &key_type, NULL,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      return status;
    }

  if (pk_magic != SSH_PRIVATE_KEY_MAGIC || pk_length < 8)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_CORRUPTED_KEY_FORMAT;
    }

  ssh_ntree_allocate(&tree);
  if (!tree)
    {
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return status;
    }

  nstat = ssh_ntree_parse(key_type, tree);
  if (nstat != SSH_NTREE_OK)
    {
      ssh_buffer_uninit(&buffer);
      ssh_ntree_free(tree);
      ssh_free(key_type);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  node = ssh_ntree_get_root(tree);
  if (node == NULL)
    {
      ssh_ntree_free(tree);
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  /* Find correct key type. */
  name = ssh_nnode_get_identifier(node);
  for (i = 0, private_key = NULL;
       ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name;
       i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, name) == 0)
        {
          /* Initialize private key. */
          if ((private_key = ssh_malloc(sizeof(*private_key))) != NULL)
            {
              private_key->type = ssh_pk_type_slots[i];
              private_key->signature = NULL;
              private_key->encryption = NULL;
              private_key->diffie_hellman = NULL;
            }
          break;
        }
    }
  ssh_free(name);

  if (private_key == NULL)
    {
      ssh_ntree_free(tree);
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  /* Run through all preselected schemes in the group_type. */
  node = ssh_nnode_get_child(node);
  while (node)
    {
      name = ssh_nnode_get_identifier(node);
      action =
        ssh_pk_find_scheme_action(private_key->type->action_list,
                                  name,
                                  SSH_PK_FLAG_PRIVATE_KEY);
      ssh_free(name);
      if (!action)
        {
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }

      child = ssh_nnode_get_child(node);
      if (child == NULL)
        name = ssh_strdup(SSH_PK_USUAL_NAME);
      else
        name = ssh_nnode_get_identifier(child);

      scheme = ssh_pk_find_generic(name, action->type, action->type_size);
      ssh_free(name);
      if (scheme == NULL)
        {
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }

      /* Set the corresponding scheme to the group. */
      status = ssh_private_key_set_scheme(private_key, scheme,
                                          action->scheme_flag);

      if (status != SSH_CRYPTO_OK)
        break;

      /* Move to the next scheme. */
      node = ssh_nnode_get_next(node);
    }
  ssh_ntree_free(tree);
  ssh_free(key_type);

  if (status != SSH_CRYPTO_OK)
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(private_key);
      return status;
    }

  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32_STR, &cipher_name, NULL,
                        SSH_FORMAT_UINT32, &length,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  tmp = NULL;
  tmplen = ssh_cipher_get_key_length(cipher_name);
  if (tmplen == 0)
    tmplen = 32;

  /* Check key len and expansion flag. */
  if (tmplen > cipher_keylen || expand_key)
    {
      /* Expand encryption key. */
      if ((tmp = ssh_malloc(tmplen)) != NULL)
        ssh_hash_expand_key_internal(tmp, tmplen,
                                     cipher_key, cipher_keylen, NULL, 0,
                                     &ssh_hash_md5_def);
      else
        tmplen = 0;

      cipher_key = tmp;
      cipher_keylen = tmplen;
    }

  /* Allocate cipher. */
  if (cipher_keylen == 0 ||
      (status = ssh_cipher_allocate(cipher_name,
                                    cipher_key, cipher_keylen, FALSE,
                                    &cipher)) != SSH_CRYPTO_OK)
    {
      ssh_free(cipher_name);
      ssh_buffer_uninit(&buffer);
      ssh_free(tmp);
      ssh_free(private_key);
      return status;
    }

  ssh_free(tmp);
  ssh_free(cipher_name);
  if (ssh_cipher_transform(cipher,
                           ssh_buffer_ptr(&buffer), ssh_buffer_ptr(&buffer),
                           length) != SSH_CRYPTO_OK)
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(private_key);
      ssh_cipher_free(cipher);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  /* Free cipher immediately. */
  ssh_cipher_free(cipher);

  /* Algorithm specific part. */
  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &tmp_length,
                        SSH_FORMAT_END) == 0 ||
      tmp_length > ssh_buffer_len(&buffer))
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(private_key);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if (private_key->type->private_key_import == NULL ||
      !(*private_key->type->private_key_import)(ssh_buffer_ptr(&buffer),
                                                tmp_length,
                                                &(private_key->context)))
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(private_key);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  *key = private_key;
  ssh_buffer_uninit(&buffer);
  return SSH_CRYPTO_OK;
}

/* Functions that are used from outside. These tell whether one wants to
   expand the key here or not. */

SshCryptoStatus
ssh_private_key_import(const unsigned char *buf, size_t len,
                       const unsigned char *cipher_key, size_t cipher_keylen,
                       SshPrivateKey *key)
{
  return ssh_private_key_import_internal(buf, len,
                                         cipher_key, cipher_keylen,
                                         key, FALSE);
}

SshCryptoStatus
ssh_private_key_export(SshPrivateKey key,
                       const char *cipher_name,
                       const unsigned char *cipher_key, size_t cipher_keylen,
                       unsigned char **bufptr,
                       size_t *length_return)
{
  return ssh_private_key_export_internal(key, cipher_name,
                                         cipher_key, cipher_keylen,
                                         bufptr, length_return, FALSE);
}


SshCryptoStatus
ssh_private_key_import_with_passphrase(const unsigned char *buf, size_t len,
                                       const char *passphrase,
                                       SshPrivateKey *key)
{
  return ssh_private_key_import_internal(buf, len,
                                         (unsigned char *) passphrase,
                                         strlen(passphrase),
                                         key, TRUE);
}

SshCryptoStatus
ssh_private_key_export_with_passphrase(SshPrivateKey key,
                                       const char *cipher_name,
                                       const char *passphrase,
                                       unsigned char **bufptr,
                                       size_t *length_return)
{
  if (strcmp(passphrase, "") == 0)
    cipher_name = "none";
  return ssh_private_key_export_internal(key, cipher_name,
                                         (unsigned char *) passphrase,
                                         strlen(passphrase),
                                         bufptr, length_return, TRUE);
}

/* Key format looks like:

   32bit    magic
   32bit    total length
   32bit    key type name length n
   n bytes  key type name (no zero terminator) (with schemes)

   32bit    algorithm specific part length n
   n bytes  algorithm specific part

   One should note, that the following key will be identical to the
   one inputed. Also it would be possible, by extending nametree
   system, to actually output the public key in ASCII. */

SshCryptoStatus
ssh_public_key_import(const unsigned char *buf, size_t len,
                      SshPublicKey *key)
{
  SshBufferStruct buffer;
  SshUInt32 pk_magic, pk_length, length;
  char *key_type, *name;
  const SshPkAction *action;
  void *scheme;
  SshPublicKey public_key;
  unsigned int i;
  SshCryptoStatus status;
  SshNameTree tree;
  SshNameNode node, child;
  SshNameTreeStatus nstat;

  ssh_buffer_init(&buffer);
  ssh_buffer_append(&buffer, buf, len);

  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &pk_magic,
                        SSH_FORMAT_UINT32, &pk_length,
                        SSH_FORMAT_UINT32_STR, &key_type, NULL,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  if (pk_magic != SSH_PUBLIC_KEY_MAGIC || pk_length < 8)
    {
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  ssh_ntree_allocate(&tree);
  if (tree == NULL)
    {
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  nstat = ssh_ntree_parse(key_type, tree);
  if (nstat != SSH_NTREE_OK)
    {
      ssh_ntree_free(tree);
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  node = ssh_ntree_get_root(tree);
  if (node == NULL)
    {
      ssh_ntree_free(tree);
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  /* Find correct key type. Done here, because we don't want to
     overuse vararg lists. */
  name = ssh_nnode_get_identifier(node);
  for (i = 0, public_key = NULL;
       ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name;
       i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, name) == 0)
        {
          /* Initialize public key. */
          if ((public_key = ssh_malloc(sizeof(*public_key))) != NULL)
            {
              public_key->type = ssh_pk_type_slots[i];
              public_key->signature = NULL;
              public_key->encryption = NULL;
              public_key->diffie_hellman = NULL;
              break;
            }
        }
    }
  ssh_free(name);

  if (public_key == NULL)
    {
      ssh_ntree_free(tree);
      ssh_free(key_type);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  status = SSH_CRYPTO_OK;
  /* Run through all preselected schemes in the group_type. */
  node = ssh_nnode_get_child(node);
  while (node)
    {
      name = ssh_nnode_get_identifier(node);
      action =
        ssh_pk_find_scheme_action(public_key->type->action_list,
                                  name,
                                  SSH_PK_FLAG_PUBLIC_KEY);
      ssh_free(name);
      if (!action)
        {
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }
      child = ssh_nnode_get_child(node);
      if (child == NULL)
        name = ssh_strdup(SSH_PK_USUAL_NAME);
      else
        name = ssh_nnode_get_identifier(child);

      scheme = ssh_pk_find_generic(name, action->type, action->type_size);
      ssh_free(name);
      if (scheme == NULL)
        {
          status = SSH_CRYPTO_SCHEME_UNKNOWN;
          break;
        }
      /* Set the corresponding scheme to the group. */
      status = ssh_public_key_set_scheme(public_key, scheme,
                                         action->scheme_flag);
      if (status != SSH_CRYPTO_OK)
        {
          break;
        }
      /* Move to the next scheme. */
      node = ssh_nnode_get_next(node);
    }

  ssh_ntree_free(tree);
  ssh_free(key_type);
  if (status != SSH_CRYPTO_OK)
    {
      ssh_buffer_uninit(&buffer);
      ssh_free(public_key);
      return status;
    }

  if (ssh_decode_buffer(&buffer,
                        SSH_FORMAT_UINT32, &length,
                        SSH_FORMAT_END) == 0 ||
      length > ssh_buffer_len(&buffer))
    {
      ssh_free(public_key);
      ssh_buffer_uninit(&buffer);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  if (public_key->type->public_key_import == NULL ||
      !(*public_key->type->public_key_import)(ssh_buffer_ptr(&buffer),
                                              length,
                                              &(public_key->context)))
    {
      ssh_buffer_uninit(&buffer);
      *key = NULL;
      ssh_free(public_key);
      return SSH_CRYPTO_OPERATION_FAILED;
    }

  *key = public_key;
  ssh_buffer_uninit(&buffer);
  return SSH_CRYPTO_OK;
}
