/*
  File: ssh-pk-prv-define.c

  Authors:
        Mika Kojo <mkojo@ssh.fi>
        Tero T Mononen <tmo@ssh.fi>

  Description:
        Routine for generating or defining private keys depending on
        how it is compiled.

  Copyright:
        Copyright (c) 2001 SSH Communications Security Corp, Finland.
        All rights reserved.
*/

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshcryptocore/namelist.h"

#define SSH_DEBUG_MODULE "SshCryptoPK"

/* Private key generation or definition and selection of used schemes
   (although this can be done also with
   ssh_private_key_select_scheme(...)  interface, which is probably
   more suitable).

   We use vararg lists, although not easy to debug they make this
   interface very flexible (atleast considering these few algorithm
   families). */

SshCryptoStatus
#ifdef GENERATOR
ssh_private_key_generate(SshPrivateKey *key, const char *key_type, ...)
#else
ssh_private_key_define(SshPrivateKey *key, const char *key_type, ...)
#endif

{
  SshCryptoStatus status = SSH_CRYPTO_UNKNOWN_KEY_TYPE;
  SshPrivateKey private_key;
  SshPkGroup group;
  const SshPkAction *action;
  SshPkFormat format;
  SshNameTree tree;
  SshNameNode node, child;
  SshNameTreeStatus nstat;
  const char *name, *r;
  char consumed[128], *tmp;
  void *scheme, *context, *wrapper;
  unsigned int i;
  va_list ap;

  /* Parse given group type. */
  ssh_ntree_allocate(&tree);
  if (!tree)
    {
      return SSH_CRYPTO_NO_MEMORY;
    }

  if (((nstat = ssh_ntree_parse(key_type, tree)) != SSH_NTREE_OK) ||
      ((node = ssh_ntree_get_root(tree)) == NULL) ||
      ((tmp = ssh_nnode_get_identifier(node)) == NULL))
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
    }

  for (i = 0;
       ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name;
       i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, tmp) != 0)
        continue;

      ssh_free(tmp);
#ifdef GENERATOR
      if (ssh_pk_type_slots[i]->private_key_action_generate == NULL)
        continue;
#endif
      /* Type matches (and if this is call to generate function, the
         generator is defined for the key type) i.e. we've found our
         key type, so continue with finding schemes and parameters. */

      node = ssh_nnode_get_child(node);

      /* Allocate private key context. */
      if ((private_key = ssh_malloc(sizeof(*private_key))) == NULL)
        {
          ssh_ntree_free(tree);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      private_key->type = ssh_pk_type_slots[i];

      /* Clear pointers. */
      private_key->signature = NULL;
      private_key->encryption = NULL;
      private_key->diffie_hellman = NULL;

      /* Initialize actions, and verify that context was allocated. */
      context = (*private_key->type->private_key_action_init)();
      if (context == NULL)
        {
          ssh_free(private_key);
          va_end(ap);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      status = SSH_CRYPTO_OK;
      /* Run through all preselected schemes in the group_type. */
      while (node)
        {
          if ((tmp = ssh_nnode_get_identifier(node)) == NULL)
            {
              status = SSH_CRYPTO_OPERATION_FAILED;
              break;
            }

          action =
            ssh_pk_find_scheme_action(private_key->type->action_list,
                                      tmp,
                                      SSH_PK_FLAG_PRIVATE_KEY);
          ssh_free(tmp);

          if (!action)
            {
              status = SSH_CRYPTO_SCHEME_UNKNOWN;
              break;
            }

          if ((child = ssh_nnode_get_child(node)) == NULL)
            tmp = ssh_strdup(SSH_PK_USUAL_NAME);
          else
            tmp = ssh_nnode_get_identifier(child);

          scheme = ssh_pk_find_generic(tmp, action->type, action->type_size);
          ssh_free(tmp);
          if (scheme == NULL)
            {
              status = SSH_CRYPTO_SCHEME_UNKNOWN;
              break;
            }

          /* Call action_scheme if not set to NULL. */
          if (((SshPkGen *)scheme)->action_scheme != NULL)
            (*((SshPkGen *)scheme)->action_scheme)(context);

          /* Set the corresponding scheme to the group. */
          status = ssh_private_key_set_scheme(private_key, scheme,
                                              action->scheme_flag);

          if (status != SSH_CRYPTO_OK)
            {
              break;
            }
          /* Move to the next scheme. */
          node = ssh_nnode_get_next(node);
        }

      ssh_ntree_free(tree);
      if (status != SSH_CRYPTO_OK)
        {
          (*private_key->type->private_key_action_free)(context);
          ssh_free(private_key);
          va_end(ap);
          return status;
        }

      /* Parse vararg list. */
      consumed[0] = '\000';
      while (TRUE)
        {
          va_start(ap, key_type);
          PROCESS(ap, consumed);

          format = va_arg(ap, SshPkFormat);
          strcat(consumed, "i");
          if (format == SSH_PKF_END)
            break;

          /* Search name from command lists. */
          action = ssh_pk_find_action(format,
                                      private_key->type->action_list,
                                      SSH_PK_FLAG_PRIVATE_KEY);
          if (!action)
            {
              (*private_key->type->private_key_action_free)(context);
              ssh_free(private_key);
              va_end(ap);
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          /* Supported only scheme selection and special operations. */
          switch (action->flags & (SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_SPECIAL))
            {
            case SSH_PK_FLAG_SCHEME:
              name = va_arg(ap, const char *);
              strcat(consumed, "p");
              scheme = ssh_pk_find_generic(name, action->type,
                                           action->type_size);
              if (scheme == NULL)
                {
                  (*private_key->type->private_key_action_free)(context);
                  ssh_free(private_key);
                  va_end(ap);
                  return SSH_CRYPTO_SCHEME_UNKNOWN;
                }

              /* Call the action_scheme function here if not
                 NULL. */
              if (((SshPkGen *)scheme)->action_scheme != NULL)
                (*((SshPkGen *)scheme)->action_scheme)(context);

              /* Set the corresponding scheme. */
              status = ssh_private_key_set_scheme(private_key, scheme,
                                                  action->scheme_flag);
              if (status != SSH_CRYPTO_OK)
                {
                  (*private_key->type->private_key_action_free)(context);
                  ssh_free(private_key);
                  va_end(ap);
                  return status;
                }
              break;

            case SSH_PK_FLAG_SPECIAL:
              /* Assume we don't use wrappings. */
              wrapper = NULL;
              if (action->flags & SSH_PK_FLAG_WRAPPED)
                {
                  /* We assume that parameters are wrapped over group
                     structure. */
                  group = va_arg(ap, SshPkGroup);
                  strcat(consumed, "p");

                  wrapper = group->context;
                  /* For compatibility set also the Diffie-Hellman
                     field.  */
                  private_key->diffie_hellman = group->diffie_hellman;
                }

              r = (*action->action_put)(context, ap, wrapper, format);
              if (r == NULL)
                {
                  (*private_key->type->private_key_action_free)(context);
                  ssh_free(private_key);
                  va_end(ap);
                  return SSH_CRYPTO_LIBRARY_CORRUPTED;
                }
              else
                strcat(consumed, r);
              break;

            default:
              SSH_NOTREACHED;
              break;
            }
          va_end(ap);
        }

      /* Make the key and remove context. (One could incorporate
         making and freeing, however this way things seem to work
         also). */
#ifdef GENERATOR
      private_key->context =
        (*private_key->type->private_key_action_generate)(context);
#else
      private_key->context =
        (*private_key->type->private_key_action_define)(context);
#endif

      (*private_key->type->private_key_action_free)(context);

      /* Quit unhappily. */
      if (private_key->context == NULL)
        {
          ssh_free(private_key);
          va_end(ap);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      /* Quit happily. */
      *key = private_key;
      va_end(ap);

      return SSH_CRYPTO_OK;
    }
  ssh_ntree_free(tree);
  va_end(ap);

  return SSH_CRYPTO_UNKNOWN_KEY_TYPE;
}
