/*

sshproxykey.c

Author: Vesa Suontama <vsuontam@ssh.fi>
        Jukka Aittokallio <jai@ssh.fi>

Copyright (c) 1999-2001 SSH Communications Security Corp, Helsinki, Finland
              All rights reserved

*/

#include "sshincludes.h"
#include "sshproxykey.h"
#include "sshpk.h"
#include "sshrgf.h"

#define SSH_DEBUG_MODULE "SshProxyKey"

#define KEY_SIZE_TO_BYTES(x) ((((x) + 7) >> 3))

/* Declare the proxy key defined later on this file. */
const SshPkType ssh_proxy_key_pr_key_type;

Boolean ssh_proxy_key_register(void)
{
  static Boolean registered = FALSE;

  if (registered) return TRUE;

  if (ssh_pk_provider_register(&ssh_proxy_key_pr_key_type) != SSH_CRYPTO_OK)
  {
    SSH_DEBUG(0, ("Could not register procy key type"));
    return FALSE;
  }

  registered = TRUE;
  return TRUE;
}

typedef struct ProxyKeyRec {
  SshUInt32 key_size;
  SshProxyKeyOpCB operation_cb;
  void *context;
} *ProxyKey;

typedef struct ProxyKeySignContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  Boolean verify_done;
  ProxyKey key;
  SshRGFHash hash;
  unsigned char *raw_data;
  size_t raw_data_len;
  SshPrivateKeySignCB callback;
  void *context;
} *ProxyKeySignContext;

void ssh_proxy_key_sign_abort(void *context)
{
  ProxyKeySignContext ctx = context;

  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx->raw_data);
  ssh_free(ctx);
}


void ssh_proxy_key_sign_free(void *context)
{
  ProxyKeySignContext ctx = context;
  ssh_operation_unregister(ctx->op);
  ssh_proxy_key_sign_abort(ctx);
}


void ssh_proxy_key_sign_op_done(const unsigned char *data,
                                size_t data_len,
                                void *context)
{
  ProxyKeySignContext sign_ctx = context;

  sign_ctx->sub_op = NULL;

  (*sign_ctx->callback)(data ? SSH_CRYPTO_OK : SSH_CRYPTO_OPERATION_FAILED,
                        data, data_len,
                        sign_ctx->context);

  ssh_proxy_key_sign_free(sign_ctx);
}

SshOperationHandle
ssh_proxy_key_sign_async(const void *private_key,
                         SshRGFHash hash,
                         SshPrivateKeySignCB callback,
                         void *context)
{
  ProxyKeySignContext sign_ctx;
  SshOperationHandle sub_op;
  ProxyKey key = (ProxyKey)private_key;

  if ((sign_ctx = ssh_calloc(1, sizeof(*sign_ctx))) != NULL)
    {
      sign_ctx->callback = callback;
      sign_ctx->context = context;
      sign_ctx->hash = hash;
      sign_ctx->key = (ProxyKey)private_key;
      sign_ctx->raw_data_len = KEY_SIZE_TO_BYTES(sign_ctx->key->key_size);
      sign_ctx->raw_data = ssh_malloc(sign_ctx->raw_data_len);

      if (sign_ctx->raw_data == NULL ||
          ssh_rgf_hash_sign(hash,
                            sign_ctx->raw_data,
                            sign_ctx->raw_data_len) != SSH_RGF_OK)
        {
          (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
          ssh_proxy_key_sign_free(sign_ctx);
          return NULL;
        }
      sign_ctx->op = ssh_operation_register(ssh_proxy_key_sign_abort,
                                            sign_ctx);
      sub_op = (*key->operation_cb)(sign_ctx->raw_data,
                                    sign_ctx->raw_data_len,
                                    ssh_proxy_key_sign_op_done,
                                    sign_ctx,
                                    key->context);

      if (sub_op)
        {
          sign_ctx->sub_op = sub_op;
          return sign_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
      return NULL;
    }
}

typedef struct ProxyKeyDecryptContextRec {
  SshOperationHandle op;
  SshOperationHandle sub_op;
  Boolean verify_done;
  ProxyKey key;
  SshRGFHash hash;
  SshPrivateKeyDecryptCB callback;
  unsigned char *raw_data;
  size_t raw_data_len;
  void *context;
} *ProxyKeyDecryptContext;

void ssh_proxy_key_decrypt_abort(void *context)
{
  ProxyKeyDecryptContext ctx = context;
  ssh_operation_abort(ctx->sub_op);
  ssh_free(ctx->raw_data);
  ssh_free(ctx);

}


void ssh_proxy_key_decrypt_free(void *context)
{
  ProxyKeyDecryptContext ctx = context;

  ssh_operation_unregister(ctx->op);
  ssh_proxy_key_decrypt_abort(context);
}


void ssh_proxy_key_decrypt_op_done(const unsigned char *data,
                                   size_t data_len,
                                   void *context)
{
  ProxyKeyDecryptContext decrypt_ctx = context;
  unsigned char *output;
  size_t output_len;

  decrypt_ctx->sub_op = NULL;

  if (data == NULL)
    {
      (*decrypt_ctx->callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, decrypt_ctx->context);
      ssh_proxy_key_decrypt_free(decrypt_ctx);
    }
  else
    {
      if (ssh_rgf_hash_decrypt(decrypt_ctx->hash,
                               data,
                               data_len,
                               KEY_SIZE_TO_BYTES(decrypt_ctx->key->key_size),
                               &output,
                               &output_len) != SSH_RGF_OK)
        {
          (*decrypt_ctx->callback)(SSH_CRYPTO_OPERATION_FAILED,
                                   NULL, 0,
                                   decrypt_ctx->context);
          ssh_proxy_key_decrypt_free(decrypt_ctx);
          return;
        }
      (*decrypt_ctx->callback)(SSH_CRYPTO_OK,
                               output, output_len,
                               decrypt_ctx->context);
      ssh_free(output);
      ssh_proxy_key_decrypt_free(decrypt_ctx);
    }
}

SshOperationHandle
ssh_proxy_key_decrypt_async(const void *private_key,
                            const unsigned char *ciphertext,
                            size_t ciphertext_len,
                            SshRGFHash hash,
                            SshPrivateKeyDecryptCB callback,
                            void *context)
{
  ProxyKeyDecryptContext decrypt_ctx;
  SshOperationHandle sub_op;
  ProxyKey key = (ProxyKey)private_key;

  if ((decrypt_ctx = ssh_calloc(1, sizeof(*decrypt_ctx))) != NULL)
    {
      decrypt_ctx->callback = callback;
      decrypt_ctx->context = context;
      decrypt_ctx->key = (ProxyKey)private_key;
      decrypt_ctx->hash = hash;
      decrypt_ctx->raw_data = ssh_memdup(ciphertext, ciphertext_len);
      decrypt_ctx->raw_data_len = ciphertext_len;

      decrypt_ctx->op = ssh_operation_register(ssh_proxy_key_decrypt_abort,
                                               decrypt_ctx);

      sub_op = (*key->operation_cb)(ciphertext, ciphertext_len,
                                    ssh_proxy_key_decrypt_op_done,
                                    decrypt_ctx,
                                    key->context);
      if (sub_op)
        {
          decrypt_ctx->sub_op = sub_op;
          return decrypt_ctx->op;
        }
      return NULL;
    }
  else
    {
      (*callback)(SSH_CRYPTO_OPERATION_FAILED, NULL, 0, context);
      return NULL;
    }
}

typedef struct ProxyKeyActionRec {
  ProxyKey proxykey;
} *ProxyKeyAction;

void *ssh_proxy_key_action_init(void)
{
  return ssh_calloc(1, sizeof(struct ProxyKeyActionRec));
}

const char *ssh_proxy_key_action_put(void *context,
                                     va_list ap,
                                     void *input_context,
                                     SshPkFormat format)
{
  ProxyKeyAction ctx = context;
  char *r;

  r = "p";
  switch (format)
    {
    case SSH_PKF_EXTERNALKEY:
      ctx->proxykey = va_arg(ap, void *);
      r = "p";
      break;
    default:
      r = NULL;
    }
  return r;
}


void *ssh_proxy_key_action_make(void *context)
{
  ProxyKeyAction act = context;
  return act->proxykey;
}

void ssh_proxy_key_action_free(void *context)
{
  ssh_free(context);
}

void ssh_proxy_key_free(void *private_key)
{
  ProxyKey ctx = private_key;
  ssh_free(ctx);
}

void ssh_proxy_key_derive_public_key(const void *private_key,
                                  void **public_key)
{
  *public_key = NULL;
}

void ssh_proxy_key_copy(void *op_src, void **op_dest)
{
  ProxyKey ret, src;

  src = op_src;
  ret = ssh_memdup(op_src, sizeof(struct ProxyKeyRec));
  *op_dest = ret;
}

size_t ssh_proxy_key_max_signature_input_len(const void *private_key)
{
  return (size_t)-1;
}

size_t ssh_proxy_key_max_signature_output_len(const void *private_key)
{
  ProxyKey ctx = (ProxyKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}

size_t ssh_proxy_key_max_signature_unhash_input_len(const void *private_key)
{
  ProxyKey ctx = (ProxyKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}

size_t ssh_proxy_key_max_decrypt_input_len(const void *private_key)
{
  ProxyKey ctx = (ProxyKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}

size_t ssh_proxy_key_max_decrypt_output_len(const void *private_key)
{
  ProxyKey ctx = (ProxyKey)private_key;
  return KEY_SIZE_TO_BYTES(ctx->key_size);
}



const SshPkSignature ssh_proxy_key_signature_schemes[] =
{
  { "rsa-pkcs1-sha1", NULL,
    &ssh_rgf_pkcs1_sha1_def,
    ssh_proxy_key_max_signature_input_len,
    ssh_proxy_key_max_signature_output_len,
    NULL,
    NULL,
    NULL,
    ssh_proxy_key_sign_async },
  { "rsa-pkcs1-md5", NULL,
    &ssh_rgf_pkcs1_md5_def,
    ssh_proxy_key_max_signature_input_len,
    ssh_proxy_key_max_signature_output_len,
    NULL,
    NULL,
    NULL,
    ssh_proxy_key_sign_async },
  { "rsa-pkcs1-none", NULL,
    &ssh_rgf_pkcs1_none_def,
    ssh_proxy_key_max_signature_unhash_input_len,
    ssh_proxy_key_max_signature_output_len,
    NULL,
    NULL,
    NULL,
    ssh_proxy_key_sign_async },
  { NULL }
};

const SshPkEncryption ssh_proxy_key_encryption_schemes[] =
{
  { "rsa-pkcs1-none", NULL,
    &ssh_rgf_pkcs1_none_def,
    ssh_proxy_key_max_decrypt_input_len,
    ssh_proxy_key_max_decrypt_output_len,
    NULL,
    ssh_proxy_key_decrypt_async,
    NULL,
    NULL,
    NULL },
  { NULL }
};



const SshPkAction ssh_proxy_key_pk_actions[] =
{

  { SSH_PKF_KEY_TYPE, NULL,
    SSH_PK_FLAG_KEY_TYPE | SSH_PK_FLAG_PRIVATE_KEY,
    SSH_PK_SCHEME_NONE, 0, NULL },

  { SSH_PKF_SIGN, "sign",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY,
    SSH_PK_SCHEME_SIGN,
    sizeof(SshPkSignature),
    ssh_proxy_key_signature_schemes, NULL },

  { SSH_PKF_ENCRYPT, "encrypt",
    SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_PRIVATE_KEY,
    SSH_PK_SCHEME_ENCRYPT,
    sizeof(SshPkEncryption),
    ssh_proxy_key_encryption_schemes, NULL },

  { SSH_PKF_EXTERNALKEY, NULL,
    SSH_PK_FLAG_SPECIAL | SSH_PK_FLAG_PRIVATE_KEY,
    SSH_PK_SCHEME_NONE, 0,
    NULL,
    ssh_proxy_key_action_put,
    NULL },


  { SSH_PKF_END }
};

const SshPkType ssh_proxy_key_pr_key_type =
{
  "proxy:if-modn",
  ssh_proxy_key_pk_actions,

  NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL,

  NULL, NULL, NULL,

  NULL,
  NULL,
  NULL,
  NULL,
  NULL,

  NULL,

  ssh_proxy_key_action_init,
  ssh_proxy_key_action_make,
  NULL,
  ssh_proxy_key_action_free,

  NULL,
  NULL,
  ssh_proxy_key_free,
  ssh_proxy_key_derive_public_key,
  ssh_proxy_key_copy,
  NULL,
  NULL,
};



/* Create a proxy key. The returned key is of type key_type,
   which must be "if-modn" (which is RSA) at the moment.

   Calls the operation_cb with the data that is being operated when
   the library is performing crypto operations with the returned proxy key.

   The proxy key is freed with ssh_private_key_free. It is error to free a key
   that is currently being used.
*/
SshPrivateKey ssh_private_key_create_proxy(const char *key_type,
                                           SshUInt32 size_in_bits,
                                           SshProxyKeyOpCB operation_cb,
                                           void *context)
{
  SshPrivateKey key;
  ProxyKey proxykey;

  if (ssh_proxy_key_register() == FALSE)
    return NULL;

  if ((proxykey = ssh_calloc(1, sizeof(*proxykey))) != NULL)
    {
      proxykey->context = context;
      proxykey->operation_cb = operation_cb;

      proxykey->key_size = size_in_bits;
      if (ssh_private_key_define(&key,
                                 "proxy:if-modn"
                                 "{sign{rsa-pkcs1-sha1},"
                                 "encrypt{rsa-pkcs1-none}}",
                                 SSH_PKF_EXTERNALKEY, proxykey,
                                 SSH_PKF_END) != SSH_CRYPTO_OK)
        return NULL;
      return key;
    }
  else
    return NULL;
}
