#include "sshhash/sshhash.h"
#include "sshrgf.h"


/* By definition this function updates the allocated Hash context
   by the input data. The data may be either usual update data, which
   should be handled by the Hash function or the resulting digest computed
   "outside".

   In the latter case the "for_digest" is set to TRUE. If such a case
   happens the update function may either reject the operation or
   function in a such a way that the resulting digest is equal to the
   input. The input data in all cases does not need freeing. */

#define SSH_RGF_HASH_UPDATE_FUNC(func_name) \
Boolean func_name (SshRGFHash hash, Boolean for_digest, \
                   const unsigned char *data, size_t data_len)

#define SSH_RGF_HASH_FREE_FUNC(func_name) \
void func_name (SshRGFHash hash)

#define SSH_RGF_HASH_DIGEST_LENGTH_FUNC(func_name) \
size_t func_name (SshRGFHash hash)

#define SSH_RGF_HASH_ASN1_OID_FUNC(func_name) \
const char * func_name (SshRGFHash hash)

#define SSH_RGF_HASH_FINALIZE_FUNC(func_name) \
void func_name (SshRGFHash hash, unsigned char *digest)

/* The RGF hash function free. */
typedef void (*SshRGFHashFree)(SshRGFHash hash);
/* The RGF hash function update. */
typedef Boolean (*SshRGFHashUpdate)(SshRGFHash hash, Boolean for_digest,
                                    const unsigned char *data,
                                    size_t data_len);
/* Return the number of octets needed to represent the digest. */
typedef size_t (*SshRGFHashDigestLength)(SshRGFHash hash);
/* Return the computed digest. */
typedef void (*SshRGFHashFinalize)(SshRGFHash hash, unsigned char *digest);
/* Return a ASN.1 oid, or NULL if not available. */
typedef const char * (*SshRGFHashAsn1Oid)(SshRGFHash hash);


typedef struct SshRGFHashDefRec
{
  /* Signature redundancy generation functions need to be able to
     handle arbitrary length messages to be signed. That is, we need
     to have a similar architecture to hash functions. */

  SshRGFHashFree         rgf_hash_free;
  SshRGFHashUpdate       rgf_hash_update;
  SshRGFHashDigestLength rgf_hash_digest_length;
  SshRGFHashFinalize     rgf_hash_finalize;
  SshRGFHashAsn1Oid      rgf_hash_asn1_oid;

  char *asn1_oid;

  /* Following context area, and beyond, is reserved for future use. */
  void *context;
} *SshRGFHashDef, SshRGFHashDefStruct;

/* The RGF hash function context. This structure can be created only
   by the RGF function "rgf_hash_allocate" and can be freed only by
   "rgf_sign" and "rgf_verify". */
struct SshRGFHashRec
{
  /* The RGF hash definition. */
  const SshRGFHashDefStruct *def;

  /* The RGF method definition. */
  const SshRGFDefStruct *rgf_def;

  /* The hash definition. */
  const SshHashDefStruct *hash_def;

  /* The standard area for precomputed digest. */
  const unsigned char *precomp_digest;
  size_t               precomp_digest_length;

  /* The state context. */
  void          *context;

  /* Context space for the caller context. */
  const void    *caller_context;
};

/* The RGF hash function allocation. */
typedef SshRGFHash (*SshRGFHashAllocate)(const SshRGFHashDefStruct *def,
                                         const SshHashDefStruct *hash_def,
                                         const void *context);

/* Remark. Some cryptosystems make use only of the SshRGFSign, and
   it should perhaps be written so as to have a suggested output_msg_len
   and it should return a variably allocated output_msg. This would
   make life a bit nicer for e.g. DSS. */

/* The RGF signature and verification functions. */
typedef SshRGFStatus (*SshRGFSign)(SshRGFHash     hash,
                                   unsigned char *output_msg,
                                   size_t         output_msg_len);
typedef SshRGFStatus (*SshRGFVerify)(SshRGFHash   hash,
                                     const unsigned char *decrypted_signature,
                                     size_t decrypted_signature_len,
                                     size_t max_output_msg_len,
                                     unsigned char      **output_msg,
                                     size_t              *output_msg_len);

typedef void (*SshRGFCtxFree)(const void *context);


/* The RGF encryption and decryption functions. */
typedef SshRGFStatus (*SshRGFEncrypt)(const unsigned char *msg,
                                      size_t           msg_len,
                                      const SshRGFDefStruct *rgf,
                                      unsigned char   *output_msg,
                                      size_t           output_msg_len);
typedef SshRGFStatus (*SshRGFDecrypt)(const unsigned char *decrypted_msg,
                                      size_t           decrypted_msg_len,
                                      size_t           max_output_msg_len,
                                      const SshRGFDefStruct *rgf,
                                      unsigned char  **output_msg,
                                      size_t          *output_msg_len);

struct SshRGFDefRec
{
  /* The unique name of this RGF for purposes of allocation. It this
     definition is given by caller to the cryptolibrary then this name
     shall not be used and may be set to NULL. */
  const char *name;

  /* Creating the redundancy. */

  /* Encryption and decryption redundancy generation functions. */

  SshRGFEncrypt rgf_encrypt;
  SshRGFDecrypt rgf_decrypt;

  SshRGFSign    rgf_sign;
  SshRGFVerify  rgf_verify;

  /* The cleaning routine. This function is not necessarily needed if
     the context points to a static data structure.

     Remark. It is not guaranteed at this point that freeing will work
     correctly (e.g. there would not be memory leaks). */
  SshRGFCtxFree rgf_ctx_free;
  SshRGFHashAllocate rgf_hash_allocate;

  /* The definition of the "hash" function for the RGF. */
  const SshRGFHashDefStruct *rgf_hash_def;

  /* The usual hash-function context. */
  const SshHashDefStruct *hash_def;

  /* Additional, future, context space. */
  const void *context;
};


#define SSH_RGF_HASH_ALLOCATE_FUNC(func_name) \
SshRGFHash  func_name (const SshRGFHashDefStruct *def, \
                       const SshHashDefStruct *hash_def, \
                       const void *context)

#define SSH_RGF_ENCRYPT_FUNC(func_name) \
SshRGFStatus func_name (const unsigned char   *msg, \
                        size_t                 msg_len, \
                        const SshRGFDefStruct *rgf, \
                        unsigned char         *output_msg, \
                        size_t                 output_msg_len)

#define SSH_RGF_DECRYPT_FUNC(func_name) \
SshRGFStatus func_name (const unsigned char   *decrypted_msg, \
                        size_t                 decrypted_msg_len, \
                        size_t                 max_output_msg_len, \
                        const SshRGFDefStruct *rgf, \
                        unsigned char        **output_msg, \
                        size_t                *output_msg_len)

#define SSH_RGF_SIGN_FUNC(func_name) \
SshRGFStatus func_name (SshRGFHash             hash, \
                        unsigned char         *output_msg, \
                        size_t                 output_msg_len)

#define SSH_RGF_VERIFY_FUNC(func_name) \
SshRGFStatus func_name (SshRGFHash             hash, \
                        const unsigned char   *decrypted_signature, \
                        size_t                 decrypted_signature_len, \
                        size_t                 max_output_msg_len, \
                        unsigned char        **output_msg, \
                        size_t                *output_msg_len)

#define SSH_RGF_CTX_FREE_FUNC(func_name) \
void func_name (const void *context)
