/*

  sshrgf.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (c) 1999-2001 SSH Communications Security, Finland
  All rights reserved.

  Created Thu Dec  9 21:25:47 1999.

  */

/* A library for redundancy generation functions for specific
   algorithms. */

#include "sshincludes.h"
#include "pkcs1.h"
#include "sshcrypt.h"
#include "sshmp.h"
#include "sshgenmp.h"
#include "sshrgf.h"
#include "sshrgf-internal.h"
#include "sshpk.h"

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
#include "sshasn1.h"
#include "rsa.h"
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

#define SSH_DEBUG_MODULE "SshCryptoRGF"

#define SSH_RGF_MAXLEN  0xffffffff

/* Some useful generic hash definitions. */

SSH_RGF_HASH_ALLOCATE_FUNC(ssh_rgf_std_hash_allocate)
{
  SshRGFHash created;

  if (def == NULL || hash_def == NULL)
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash definition."));
      return NULL;
    }

  if ((created = ssh_calloc(1, sizeof(*created))) != NULL)
    {
      created->def = def;
      created->hash_def = hash_def;
      if ((created->context = ssh_hash_allocate_internal(hash_def)) == NULL)
        {
          ssh_free(created);
          return NULL;
        }
      ssh_hash_reset(created->context);
    }
  return created;
}

SSH_RGF_HASH_FREE_FUNC(ssh_rgf_std_hash_free)
{
  ssh_hash_free(hash->context);
  ssh_free(hash);
}

SSH_RGF_HASH_UPDATE_FUNC(ssh_rgf_std_hash_update)
{
  if (hash->context == NULL)
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash state."));
      return FALSE;
    }

  /* Handle the case when possibly setting the finalized digest
     beforehand. */
  if (for_digest)
    {
      if (ssh_hash_digest_length(hash->context) == data_len)
        {
          /* This does not allocate new space for the data. */
          hash->precomp_digest        = data;
          hash->precomp_digest_length = data_len;
          return TRUE;
        }
      return FALSE;
    }

  if (hash->precomp_digest)
    return FALSE;

  ssh_hash_update(hash->context, data, data_len);
  return TRUE;
}

SSH_RGF_HASH_DIGEST_LENGTH_FUNC(ssh_rgf_std_hash_digest_length)
{
  if (hash->context == NULL)
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash state."));
      return 0;
    }
  return ssh_hash_digest_length(hash->context);
}

SSH_RGF_HASH_FINALIZE_FUNC(ssh_rgf_std_hash_finalize)
{
  if (hash->context == NULL)
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash state."));
      return;
    }
  if (hash->precomp_digest)
    memcpy(digest, hash->precomp_digest, hash->precomp_digest_length);
  else
    ssh_hash_final(hash->context, digest);
}

SSH_RGF_HASH_ASN1_OID_FUNC(ssh_rgf_std_hash_asn1_oid)
{
  if (hash->context == NULL)
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash state."));
      return NULL;
    }
  return ssh_hash_asn1_oid(hash->context);
}

/* With no hashing. */

SSH_RGF_HASH_ALLOCATE_FUNC(ssh_rgf_none_hash_allocate)
{
  SshRGFHash created = NULL;

  if (def)
    {
      if ((created = ssh_calloc(1, sizeof(*created))) != NULL)
        {
          created->def      = def;
          created->hash_def = NULL;
        }
    }
  else
    {
      SSH_DEBUG(SSH_D_ERROR, ("No hash definition."));
    }
  return created;
}

SSH_RGF_HASH_FREE_FUNC(ssh_rgf_none_hash_free)
{
  ssh_free(hash);
}

SSH_RGF_HASH_UPDATE_FUNC(ssh_rgf_none_hash_update)
{
  hash->precomp_digest        = data;
  hash->precomp_digest_length = data_len;
  return TRUE;
}

SSH_RGF_HASH_DIGEST_LENGTH_FUNC(ssh_rgf_none_hash_digest_length)
{
  if (hash->precomp_digest == NULL)
    return SSH_RGF_MAXLEN;
  return hash->precomp_digest_length;
}

SSH_RGF_HASH_FINALIZE_FUNC(ssh_rgf_none_hash_finalize)
{
  if (hash->precomp_digest)
    memcpy(digest, hash->precomp_digest, hash->precomp_digest_length);
}

SSH_RGF_HASH_ASN1_OID_FUNC(ssh_rgf_none_hash_asn1_oid)
{
  return NULL;
}

/* Basic hash definitions. */
const SshRGFHashDefStruct ssh_rgf_std_hash_def =
{
  ssh_rgf_std_hash_free,
  ssh_rgf_std_hash_update,
  ssh_rgf_std_hash_digest_length,
  ssh_rgf_std_hash_finalize,
  ssh_rgf_std_hash_asn1_oid,
  NULL,
  NULL
};

const SshRGFHashDefStruct ssh_rgf_none_hash_def =
{
  ssh_rgf_none_hash_free,
  ssh_rgf_none_hash_update,
  ssh_rgf_none_hash_digest_length,
  ssh_rgf_none_hash_finalize,
  ssh_rgf_none_hash_asn1_oid,
  NULL,
  NULL
};

/* Basic redundancy function implementations. */

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
/* RSA PKCS-1 v1.5 */

/* Some useful routines doing the dirty work. */
SSH_RGF_ENCRYPT_FUNC(ssh_rgf_pkcs1_encrypt)
{
  SshMPIntStruct t1, t2;
  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Remark. This function could be written to run without the integer
     arithmetic, but the padding routines use them so... */

  /* Convert the message into an integer. */
  ssh_buf_to_mp(&t1, msg, msg_len);

  /* Now handle the padding. */
  ssh_pkcs1_pad(&t2, &t1, (unsigned int)msg_len, 2, output_msg_len);

  /* Convert back to an plaintext buffer. */
  ssh_mp_to_buf(output_msg, output_msg_len, &t2);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  return SSH_RGF_OK;
}

SSH_RGF_DECRYPT_FUNC(ssh_rgf_pkcs1_decrypt)
{
  SshMPIntStruct t1;
  unsigned char *buf;
  size_t         buf_len;

  if ((buf = ssh_malloc(max_output_msg_len)) == NULL)
    {
      return SSH_RGF_OP_FAILED;
    }

  ssh_mp_init(&t1);
  ssh_buf_to_mp(&t1, decrypted_msg, decrypted_msg_len);
  if (ssh_pkcs1_unpad(2, &t1, buf, max_output_msg_len, &buf_len) == FALSE)
    {
      ssh_free(buf);
      ssh_mp_clear(&t1);
      return SSH_RGF_OP_FAILED;
    }

  /* Return the unpadded msg. */
  *output_msg     = buf;
  *output_msg_len = buf_len;

  ssh_buf_to_mp(&t1, decrypted_msg, decrypted_msg_len);
  ssh_mp_clear(&t1);
  return SSH_RGF_OK;
}

static SshRGFStatus
rgf_pkcs1_sign(Boolean do_padding,
               SshRGFHash hash,
               unsigned char *output_msg, size_t output_msg_len)
{
  unsigned char *ber_data;
  size_t         ber_data_len;
  unsigned char  *digest;
  const char    *oid;
  size_t         digest_len;
  SshMPIntStruct t1, t2;

  if (hash->context == NULL)
    return SSH_RGF_OP_FAILED;

  if ((oid = (*hash->def->rgf_hash_asn1_oid)(hash)) == NULL)
    return SSH_RGF_OP_FAILED;

  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if ((digest = ssh_malloc(digest_len)) == NULL)
    return SSH_RGF_OP_FAILED;

  (*hash->def->rgf_hash_finalize)(hash, digest);

  if (!ssh_pkcs1_wrap(oid, digest, digest_len, &ber_data, &ber_data_len))
    {
      ssh_free(digest);
      return SSH_RGF_OP_FAILED;
    }
  ssh_free(digest);

  if (do_padding)
    {
      ssh_mp_init(&t1);
      ssh_mp_init(&t2);

      /* Unlinearize. */
      ssh_buf_to_mp(&t1, ber_data, ber_data_len);
      ssh_free(ber_data);

      /* Pad data as pkcs1 suggests. */
      if (output_msg_len < ber_data_len + 1)
        {
          ssh_mp_clear(&t1);
          ssh_mp_clear(&t2);
          return SSH_RGF_OP_FAILED;
        }

      ssh_pkcs1_pad(&t2, &t1, (unsigned int)ber_data_len, 1, output_msg_len);
      ssh_mp_to_buf(output_msg, output_msg_len, &t2);

      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
    }
  else
    {
      if (output_msg_len < ber_data_len)
        {
          ssh_xfree(ber_data);
          return SSH_RGF_OP_FAILED;
        }
      memcpy(output_msg, ber_data, ber_data_len);
      ssh_xfree(ber_data);
    }
  return SSH_RGF_OK;
}

SSH_RGF_SIGN_FUNC(ssh_rgf_pkcs1_nopad_sign)
{
  return rgf_pkcs1_sign(FALSE, hash, output_msg, output_msg_len);
}


SSH_RGF_SIGN_FUNC(ssh_rgf_pkcs1_sign)
{
  return rgf_pkcs1_sign(TRUE, hash, output_msg, output_msg_len);
}

static SshRGFStatus
rgf_pkcs1_verify(Boolean do_unpad,
                 SshRGFHash             hash,
                 const unsigned char   *decrypted_signature,
                 size_t                 decrypted_signature_len,
                 size_t                 max_output_msg_len,
                 unsigned char        **output_msg,
                 size_t                *output_msg_len)
{
  unsigned char *ber_buf, *temp_buf;
  SshMPIntStruct t1;
  unsigned char *oid;
  const char *hash_oid;
  unsigned char *digest;
  size_t digest_len, return_len;
  Boolean rv;

  if (hash->context == NULL)
    return SSH_RGF_OP_FAILED;

  *output_msg     = NULL;
  *output_msg_len = 0;

  /* Decode the msg. */
  if ((ber_buf = ssh_malloc(max_output_msg_len)) == NULL)
    return SSH_RGF_OP_FAILED;

  ssh_mp_init(&t1);
  ssh_buf_to_mp(&t1, decrypted_signature, decrypted_signature_len);

  if (do_unpad)
    {
      rv = ssh_pkcs1_unpad(1, &t1, ber_buf, max_output_msg_len, &return_len);
      if (!rv)
        {
          ssh_free(ber_buf);
          ssh_mp_clear(&t1);
          goto failed;
        }
    }

  /* Clear the multiple precision integer. */
  ssh_mp_clear(&t1);

  rv = ssh_pkcs1_unwrap(ber_buf, return_len, &oid, &temp_buf, &return_len);
  ssh_free(ber_buf);

  if (!rv)
    goto failed;

  /* Hash. */
  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if ((digest = ssh_malloc(digest_len)) == NULL)
    {
      ssh_free(oid);
      ssh_free(temp_buf);
      goto failed;
    }
  (*hash->def->rgf_hash_finalize)(hash, digest);
  hash_oid = (*hash->def->rgf_hash_asn1_oid)(hash);

  /* Compare. */
  if (hash_oid == NULL || digest_len != return_len)
    {
      ssh_free(oid);
      ssh_free(temp_buf);
      ssh_free(digest);
      goto failed;
    }

  if (memcmp(digest, temp_buf, digest_len) == 0 && strcmp(hash_oid, oid) == 0)
    rv = TRUE;
  else
    rv = FALSE;

  ssh_free(digest);
  ssh_free(oid);
  ssh_free(temp_buf);

failed:
  if (rv == TRUE)
    return SSH_RGF_OK;
  return SSH_RGF_OP_FAILED;
}

SSH_RGF_VERIFY_FUNC(ssh_rgf_pkcs1_nopad_verify)
{
  return rgf_pkcs1_verify(FALSE, hash, decrypted_signature,
                          decrypted_signature_len,
                          max_output_msg_len,
                          output_msg,
                          output_msg_len);
}

SSH_RGF_VERIFY_FUNC(ssh_rgf_pkcs1_verify)
{
  return rgf_pkcs1_verify(TRUE, hash, decrypted_signature,
                          decrypted_signature_len,
                          max_output_msg_len,
                          output_msg,
                          output_msg_len);
}

SSH_RGF_SIGN_FUNC(ssh_rgf_pkcs1_sign_nohash)
{
  SshMPIntStruct t1, t2;
  unsigned char *digest;
  size_t digest_length;

  digest_length = (*hash->def->rgf_hash_digest_length)(hash);
  if (digest_length == SSH_RGF_MAXLEN ||
      (digest = ssh_malloc(digest_length)) == NULL)
    return SSH_RGF_OP_FAILED;

  (*hash->def->rgf_hash_finalize)(hash, digest);

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Convert to integer. */
  ssh_buf_to_mp(&t1, digest, digest_length);
  ssh_free(digest);

  /* Pad data. */
  ssh_pkcs1_pad(&t2, &t1, (unsigned int)digest_length, 1, output_msg_len);
  ssh_mp_to_buf(output_msg, output_msg_len, &t2);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  return SSH_RGF_OK;
}

SSH_RGF_VERIFY_FUNC(ssh_rgf_pkcs1_verify_nohash)
{
  SshMPIntStruct t1;
  unsigned char *buf;
  size_t return_len;
  unsigned char *digest;
  size_t digest_length;

  *output_msg     = NULL;
  *output_msg_len = 0;

  /* Allocate a suitable decoding buffer. */
  if ((buf = ssh_malloc(max_output_msg_len)) == NULL)
    return SSH_RGF_OP_FAILED;

  ssh_mp_init(&t1);
  ssh_buf_to_mp(&t1, decrypted_signature, decrypted_signature_len);

  /* Unpad. */
  if (ssh_pkcs1_unpad(1, &t1, buf, max_output_msg_len, &return_len) == FALSE)
    {
      ssh_free(buf);
      ssh_mp_clear(&t1);
      ssh_rgf_hash_free(hash);
      return SSH_RGF_OP_FAILED;
    }

  ssh_mp_clear(&t1);

  digest_length = (*hash->def->rgf_hash_digest_length)(hash);
  if ((digest = ssh_malloc(digest_length)) == NULL)
    {
      ssh_free(buf);
      return SSH_RGF_OP_FAILED;
    }
  (*hash->def->rgf_hash_finalize)(hash, digest);

  if (digest_length != return_len ||
      memcmp(digest, buf, digest_length) != 0)
    {
      ssh_free(digest);
      ssh_free(buf);
      return SSH_RGF_OP_FAILED;
    }
  ssh_free(digest);
  ssh_free(buf);
  return SSH_RGF_OK;
}

/* RSA PKCS-1 v2.0 */

SSH_RGF_ENCRYPT_FUNC(ssh_rgf_pkcs1v2_encrypt)
{
  unsigned char *param;
  size_t param_len;

  if (rgf->hash_def == NULL)
    return SSH_RGF_OP_FAILED;

  param = ssh_rsa_pkcs1v2_default_explicit_param(rgf->hash_def, &param_len);
  if (param == NULL)
    return SSH_RGF_OP_FAILED;

  if (output_msg_len == 0)
    return SSH_RGF_OP_FAILED;

  /* Initialize the highest octet. */
  output_msg[0] = 0;
  if (ssh_rsa_oaep_encode_with_mgf1(rgf->hash_def,
                                    msg, msg_len,
                                    param, param_len,
                                    output_msg+1, output_msg_len-1) == FALSE)
    {
      ssh_free(param);
      return SSH_RGF_OP_FAILED;
    }
  ssh_free(param);

  return SSH_RGF_OK;
}

SSH_RGF_DECRYPT_FUNC(ssh_rgf_pkcs1v2_decrypt)
{
  unsigned char *param;
  size_t param_len;

  if (rgf->hash_def == NULL)
    return SSH_RGF_OP_FAILED;

  /* Find params. */
  param = ssh_rsa_pkcs1v2_default_explicit_param(rgf->hash_def, &param_len);
  if (param == NULL)
    return SSH_RGF_OP_FAILED;

  if (decrypted_msg_len == 0 ||
      decrypted_msg[0] != 0)
    return SSH_RGF_OP_FAILED;

  /* Apply the OAEP decoding. */
  if (ssh_rsa_oaep_decode_with_mgf1(rgf->hash_def,
                                    decrypted_msg+1, decrypted_msg_len-1,
                                    param, param_len,
                                    output_msg, output_msg_len) == FALSE)
    {
      ssh_free(param);
      return SSH_RGF_OP_FAILED;
    }
  ssh_free(param);
  return SSH_RGF_OK;
}

#ifdef SSHDIST_CRYPT_SHA
/* RSA PKCS-1 v1.5 */

extern const SshHashDefStruct ssh_hash_sha_def;

const SshRGFDefStruct ssh_rgf_pkcs1_sha1_def =
{
  "pkcs1-sha1",
  ssh_rgf_pkcs1_encrypt,
  ssh_rgf_pkcs1_decrypt,
  ssh_rgf_pkcs1_sign,
  ssh_rgf_pkcs1_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_sha_def,
  NULL
};
#endif /* SSHDIST_CRYPT_SHA */

#ifdef SSHDIST_CRYPT_MD5

extern const SshHashDefStruct ssh_hash_md5_def;

const SshRGFDefStruct ssh_rgf_pkcs1_md5_def =
{
  "pkcs1-md5",
  ssh_rgf_pkcs1_encrypt,
  ssh_rgf_pkcs1_decrypt,
  ssh_rgf_pkcs1_sign,
  ssh_rgf_pkcs1_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_md5_def,
  NULL
};

#endif /* SSHDIST_CRYPT_MD5 */




















const SshRGFDefStruct ssh_rgf_pkcs1_none_def =
{
  "pkcs1-none",
  ssh_rgf_pkcs1_encrypt,
  ssh_rgf_pkcs1_decrypt,
  ssh_rgf_pkcs1_sign_nohash,
  ssh_rgf_pkcs1_verify_nohash,
  NULL,
  ssh_rgf_none_hash_allocate,
  &ssh_rgf_none_hash_def,
  NULL,
  NULL
};

#ifdef SSHDIST_CRYPT_SHA

/* RSA PKCS-1 v2.0 */

const SshRGFDefStruct ssh_rgf_pkcs1v2_sha1_def =
{
  "pkcs1v2-sha1",
  ssh_rgf_pkcs1v2_encrypt,
  ssh_rgf_pkcs1v2_decrypt,
  ssh_rgf_pkcs1_sign,
  ssh_rgf_pkcs1_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_sha_def,
  NULL
};
#endif /* SSHDIST_CRYPT_SHA */

#ifdef SSHDIST_CRYPT_MD5

const SshRGFDefStruct ssh_rgf_pkcs1v2_md5_def =
{
  "pkcs1v2-md5",
  ssh_rgf_pkcs1v2_encrypt,
  ssh_rgf_pkcs1v2_decrypt,
  ssh_rgf_pkcs1_sign,
  ssh_rgf_pkcs1_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_md5_def,
  NULL
};
#endif /* SSHDIST_CRYPT_MD5 */


















const SshRGFDefStruct ssh_rgf_pkcs1v2_none_def =
{
  "pkcs1v2-none",
  ssh_rgf_pkcs1v2_encrypt,
  ssh_rgf_pkcs1v2_decrypt,
  ssh_rgf_pkcs1_sign_nohash,
  ssh_rgf_pkcs1_verify_nohash,
  NULL,
  ssh_rgf_none_hash_allocate,
  &ssh_rgf_none_hash_def,
  NULL,
  NULL
};
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */

/* A generic routines that can be used with many cryptosystems with
   little redundancy management. These include e.g. the DSA algorithm.

   Common idea with all the methods is that they basically do not
   do any redundancy related operations. For example, they just hash
   the message for signature using one of the standard hash functions.
   They do not pad the digest before signing, usually because these
   methods include the digest into the cryptosystem in more complicated
   manner than RSA does, for example.
   */

SSH_RGF_ENCRYPT_FUNC(ssh_rgf_std_encrypt)
{
  if (msg_len > output_msg_len)
    return SSH_RGF_OP_FAILED;

  /* Zero the output msg. */
  memset(output_msg, 0, output_msg_len);
  memcpy(output_msg + (output_msg_len - msg_len), msg, msg_len);

  return SSH_RGF_OK;
}

SSH_RGF_DECRYPT_FUNC(ssh_rgf_std_decrypt)
{
  if (decrypted_msg_len > max_output_msg_len)
    return SSH_RGF_OP_FAILED;

  if ((*output_msg = ssh_memdup(decrypted_msg, decrypted_msg_len)) != NULL)
    *output_msg_len = decrypted_msg_len;
  else
    {
      *output_msg_len = 0;
      return SSH_RGF_OP_FAILED;
    }
  return SSH_RGF_OK;
}

SSH_RGF_SIGN_FUNC(ssh_rgf_std_sign)
{
  unsigned char  *digest;
  size_t         digest_len;

  if (hash->context == NULL)
    return SSH_RGF_OP_FAILED;

  /* Create the digest. */
  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if (digest_len == SSH_RGF_MAXLEN ||
      (digest = ssh_malloc(digest_len)) == NULL)
    return SSH_RGF_OP_FAILED;

  (*hash->def->rgf_hash_finalize)(hash, digest);

  /* Now check whether we can output the digest or not. */
  if (digest_len > output_msg_len)
    {
      ssh_free(digest);
      return SSH_RGF_OP_FAILED;
    }
  memset(output_msg, 0, output_msg_len);
  memcpy(output_msg + (output_msg_len - digest_len), digest, digest_len);
  ssh_free(digest);

  return SSH_RGF_OK;
}

SSH_RGF_SIGN_FUNC(ssh_rgf_std_sign_no_hash)
{
  const unsigned char *msg;
  size_t msg_len;
  unsigned char *digest;
  size_t digest_len;

  /* Create the digest. */
  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if (digest_len == SSH_RGF_MAXLEN ||
      (digest = ssh_malloc(digest_len)) == NULL)
    return SSH_RGF_OP_FAILED;
  (*hash->def->rgf_hash_finalize)(hash, digest);

  msg     = digest;
  msg_len = digest_len;

  if (msg_len > output_msg_len)
    {
      msg += (msg_len - output_msg_len);
      msg_len = output_msg_len;
    }

  memset(output_msg, 0, output_msg_len);
  memcpy(output_msg + (output_msg_len - msg_len), msg, msg_len);
  ssh_free(digest);
  return SSH_RGF_OK;
}

SSH_RGF_VERIFY_FUNC(ssh_rgf_std_verify)
{
  unsigned char  *digest;
  size_t         digest_len;

  *output_msg     = NULL;
  *output_msg_len = 0;

  if (hash->context == NULL)
    return SSH_RGF_OP_FAILED;

  /* Create the digest. */
  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if (digest_len == SSH_RGF_MAXLEN ||
      (digest = ssh_malloc(digest_len)) == NULL)
    return SSH_RGF_OP_FAILED;
  (*hash->def->rgf_hash_finalize)(hash, digest);

  if (digest_len != decrypted_signature_len ||
      memcmp(decrypted_signature, digest, digest_len) != 0)
    {
      ssh_free(digest);
      return SSH_RGF_OP_FAILED;
    }
  ssh_free(digest);
  return SSH_RGF_OK;
}

SSH_RGF_VERIFY_FUNC(ssh_rgf_std_verify_no_hash)
{
  SshRGFStatus rv = SSH_RGF_OK;
  unsigned char *digest;
  size_t digest_len;

  *output_msg     = NULL;
  *output_msg_len = 0;

  /* Create the digest. */
  digest_len = (*hash->def->rgf_hash_digest_length)(hash);
  if (digest_len == SSH_RGF_MAXLEN ||
      (digest = ssh_malloc(digest_len)) == NULL)
    return SSH_RGF_OP_FAILED;
  (*hash->def->rgf_hash_finalize)(hash, digest);

  /* Check the validity. */
  if (digest_len != decrypted_signature_len ||
      memcmp(digest, decrypted_signature, digest_len) != 0)
    rv = SSH_RGF_OP_FAILED;

  ssh_free(digest);
  return rv;
}

#ifdef SSHDIST_CRYPT_SHA

const SshRGFDefStruct ssh_rgf_std_sha1_def =
{
  "std-sha1",
  ssh_rgf_std_encrypt,
  ssh_rgf_std_decrypt,
  ssh_rgf_std_sign,
  ssh_rgf_std_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_sha_def,
  NULL
};

const SshRGFDefStruct ssh_rgf_pkcs1_nopad_sha1_def =
{
  "pkcs1nopad-sha1",
  ssh_rgf_std_encrypt,
  ssh_rgf_std_decrypt,
  ssh_rgf_pkcs1_nopad_sign,
  ssh_rgf_pkcs1_nopad_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_sha_def,
  NULL
};

#endif /* SSHDIST_CRYPT_SHA */

#ifdef SSHDIST_CRYPT_MD5

const SshRGFDefStruct ssh_rgf_std_md5_def =
{
  "std-md5",
  ssh_rgf_std_encrypt,
  ssh_rgf_std_decrypt,
  ssh_rgf_std_sign,
  ssh_rgf_std_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_md5_def,
  NULL
};

const SshRGFDefStruct ssh_rgf_pkcs1_nopad_md5_def =
{
  "pkcs1nopad-md5",
  ssh_rgf_std_encrypt,
  ssh_rgf_std_decrypt,
  ssh_rgf_pkcs1_nopad_sign,
  ssh_rgf_pkcs1_nopad_verify,
  NULL,
  ssh_rgf_std_hash_allocate,
  &ssh_rgf_std_hash_def,
  &ssh_hash_md5_def,
  NULL
};


#endif /* SSHDIST_CRYPT_MD5 */


















const SshRGFDefStruct ssh_rgf_dummy_def =
{
  "std-dummy",
  ssh_rgf_std_encrypt,
  ssh_rgf_std_decrypt,
  ssh_rgf_std_sign_no_hash,
  ssh_rgf_std_verify_no_hash,
  NULL,
  ssh_rgf_none_hash_allocate,
  &ssh_rgf_none_hash_def,
  NULL,
  NULL
};

/* TODO. More redundancy functions and similar constructs. */

/* The RGF manipulation functions. */

SshRGFHash
ssh_rgf_hash_allocate(const SshRGFDefStruct *rgf_def)
{
  SshRGFHash rgf;

  if (rgf_def->rgf_hash_allocate == NULL ||
      rgf_def->rgf_hash_def == NULL)
    return NULL;

  rgf = (*rgf_def->rgf_hash_allocate)(rgf_def->rgf_hash_def,
                                      rgf_def->hash_def,
                                      rgf_def->context);
  if (rgf)
    rgf->rgf_def = rgf_def;

  return rgf;
}

void
ssh_rgf_hash_update(SshRGFHash hash,
                    const unsigned char *data, size_t data_len)
{
  if (hash->def->rgf_hash_update == NULL)
    ssh_fatal("ssh_rgf_hash_update: no update function defined.");
  (*hash->def->rgf_hash_update)(hash, FALSE,
                                data, data_len);
}

Boolean
ssh_rgf_hash_update_with_digest(SshRGFHash hash,
                                const unsigned char *data, size_t data_len)
{
  if (hash->def->rgf_hash_update == NULL)
    ssh_fatal("ssh_rgf_hash_update_with_digest: no update function defined.");
  return (*hash->def->rgf_hash_update)(hash, TRUE,
                                       data, data_len);
}

SshHash
ssh_rgf_hash_derive(SshRGFHash hash)
{
  /* Check whether the conversion is possible. */
  if (hash->hash_def == NULL)
    return NULL;
  return ssh_hash_allocate_internal(hash->hash_def);
}

void
ssh_rgf_hash_free(SshRGFHash hash)
{
  if (hash)
    {
      if (hash->def->rgf_hash_free == NULL)
        ssh_fatal("ssh_rgf_hash_free: no hash free function defined.");
      (*hash->def->rgf_hash_free)(hash);
    }
}

SshRGFStatus
ssh_rgf_hash_encrypt(SshRGFHash             hash,
                     const unsigned char   *msg,
                     size_t                 msg_len,
                     unsigned char         *output_msg,
                     size_t                 output_msg_len)
{
  SshRGFStatus status = SSH_RGF_OP_FAILED;

  if (hash->rgf_def->rgf_encrypt)
    status = (*hash->rgf_def->rgf_encrypt)(msg, msg_len,
                                           hash->rgf_def,
                                           output_msg, output_msg_len);

  ssh_rgf_hash_free(hash);
  return status;
}

SshRGFStatus
ssh_rgf_hash_decrypt(SshRGFHash             hash,
                     const unsigned char   *decrypted_msg,
                     size_t                 decrypted_msg_len,
                     size_t                 max_output_msg_len,
                     unsigned char        **output_msg,
                     size_t                *output_msg_len)
{
  SshRGFStatus status = SSH_RGF_OP_FAILED;

  if (hash->rgf_def->rgf_decrypt)
    status = (*hash->rgf_def->rgf_decrypt)(decrypted_msg, decrypted_msg_len,
                                           max_output_msg_len,
                                           hash->rgf_def,
                                           output_msg, output_msg_len);
  ssh_rgf_hash_free(hash);
  return status;
}

SshRGFStatus
ssh_rgf_hash_sign(SshRGFHash                hash,
                  unsigned char            *output_msg,
                  size_t                    output_msg_len)
{
  SshRGFStatus status = SSH_RGF_OP_FAILED;

  if (hash->rgf_def->rgf_sign)
    status = (*hash->rgf_def->rgf_sign)(hash, output_msg, output_msg_len);
  ssh_rgf_hash_free(hash);
  return status;
}

SshRGFStatus
ssh_rgf_hash_verify(SshRGFHash             hash,
                    const unsigned char   *decrypted_signature,
                    size_t                 decrypted_signature_len,
                    size_t                 max_output_msg_len,
                    unsigned char        **output_msg,
                    size_t                *output_msg_len)
{
  SshRGFStatus status = SSH_RGF_OP_FAILED;

  if (hash->rgf_def->rgf_verify)
    status = (*hash->rgf_def->rgf_verify)(hash,
                                          decrypted_signature,
                                          decrypted_signature_len,
                                          max_output_msg_len,
                                          output_msg, output_msg_len);
  ssh_rgf_hash_free(hash);
  return status;
}

size_t ssh_rgf_hash_digest_length(SshRGFHash hash)
{
  if (hash->def->rgf_hash_digest_length)
    return (*hash->def->rgf_hash_digest_length)(hash);
  else
    return 0;
}
/* rgf.c */
