/*

  gentest.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1996, 2000, 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Fri Nov  1 05:37:55 1996 [mkojo]

  Testing those gen- prefixed files.

  */

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshtimemeasure.h"
#include "readfile.h"
#include "sshmp.h"
#include "sshdsprintf.h"
#include "sshcryptocore/namelist.h"
#include "t-gentest.h"

/****************** Hash tests. ***********************/

void test(void)
{

#if 0
  int i;
  /* This short piece of test code is for our previous SHA-1 bug. It
     was cunning enough not to show up in NIST examples (even
     partitioned).  However the following test detects it.

     Bug can be detected by having 128 bytes of data to be
     hashed. This data is divided in two parts first part having 1
     byte and rest 127 bytes. These bytes should not all be the same.

     Then update the hash context with both parts in correct order
     (first the 1 byte and then the rest 127 bytes). Compare this with
     the hash output of straigh hashing of 128 original bytes. If
     result is not equal then this error (or some other) was detected.
     */

  SshHash hash;
  unsigned char digest[128];


  ssh_hash_allocate("sha1", &hash);

  ssh_hash_reset(hash);
  for (i = 0; i < 128; i++)
    digest[i] = 0;
  digest[127] = 1;

  ssh_hash_update(hash, digest, 1);
  ssh_hash_update(hash, digest + 1, 127);

  ssh_hash_final(hash, digest);

  ssh_hash_reset(hash);

  for (i = 0; i < 20; i++)
    printf("%02x", digest[i]);
  printf("\n");

  for (i = 0; i < 128; i++)
    digest[i] = 0;
  digest[127] = 1;

  ssh_hash_update(hash, digest, 128);

  ssh_hash_final(hash, digest);

  for (i = 0; i < 20; i++)
    printf("%02x", digest[i]);
  printf("\n");

  ssh_hash_free(hash);

  exit(1);

#endif
}

#ifdef SSHDIST_CRYPT_GENHASH

void hash_random_tests(void)
{
  char *namelist = ssh_hash_get_supported();
  const char *tmp_namelist = namelist;
  char *hash_name = NULL;
  unsigned char buf[SSH_MAX_HASH_DIGEST_LENGTH],
    *buf2;
  SshHash hash;
  struct SshTimeMeasureRec tmit = SSH_TIME_MEASURE_INITIALIZER;
  int i, len;

  while ((hash_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      if (ssh_hash_allocate(hash_name, &hash) != SSH_CRYPTO_OK)
        ssh_fatal("error: hash allocate %s failed.", hash_name);

      /* Put here some tests. */

      len = 1000;
    retry:
      buf2 = ssh_xmalloc(len);
      for (i = 0; i < len; i++)
        buf2[i] = i & 0xff;

      ssh_time_measure_reset(&tmit);
      ssh_time_measure_start(&tmit);

      for (i = 0; i < 1024; i++)
        ssh_hash_update(hash, buf2, len);

      ssh_time_measure_stop(&tmit);

      ssh_hash_final(hash, buf);

      ssh_xfree(buf2);

      if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN && len < 10000000)
        {
          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 10)
            {
              len *= 128;
            }
          else
            {
              len = (int) (len * TEST_TIME_OPTIMUM
                           / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              len |= 0x3ff;
              len++;
            }
          if (verbose)
            printf("  - %s was too fast, retrying...\n", hash_name);
          goto retry;
        }

      if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
        printf("%s -- " TEST_FMT " KBytes/sec\n",
               hash_name, ((double)len) /
               ((double) ssh_time_measure_get(&tmit,
                                              SSH_TIME_GRANULARITY_MICROSECOND)
                / 1000000.0));
      else
        printf("  - timing could not be performed for %s.\n", hash_name);


      /* Put here some tests. */

      ssh_xfree(hash_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);

      ssh_hash_free(hash);
    }

  ssh_xfree(namelist);
}

/* Read the file. */
void hash_static_tests(void)
{
  char hash_name[256];
  unsigned char *str;
  unsigned char buf[SSH_MAX_HASH_DIGEST_LENGTH];
  SshHash hash = NULL;
  size_t len;
  int i;
  RFStatus status;
#define HASH_IGNORE 0
#define HASH_INPUT 1
#define HASH_OUTPUT 2
  unsigned int state = HASH_IGNORE;

  status = ssh_t_read_init(filename);
  if (status != RF_READ)
    ssh_fatal("error: hash.tests file not available or corrupted.");

  while (status != RF_EMPTY)
    {
      status = ssh_t_read_token(&str, &len);
      switch (status)
        {
        case RF_LABEL:
          /* Delete the old hash context. */
          if (hash)
            {
              ssh_hash_free(hash);
              hash = NULL;
            }
          if (len > 255)
            ssh_fatal("error: hash name too long.");
          memcpy(hash_name, str, len);
          hash_name[len] = '\0';

          if (ssh_hash_supported(hash_name))
            {
              if (ssh_hash_allocate(hash_name, &hash) != SSH_CRYPTO_OK)
                ssh_fatal("error: hash allocate %s failed.", hash_name);
              state = HASH_INPUT;
            }
          else
            {
              ssh_debug("hash %s not supported", hash_name);
              state = HASH_IGNORE;
            }
          break;
        case RF_HEX:
        case RF_ASCII:
          switch (state)
            {
            case HASH_INPUT:
              ssh_hash_reset(hash);
              ssh_hash_update(hash, str, len);
              state = HASH_OUTPUT;
              break;
            case HASH_OUTPUT:
              ssh_hash_final(hash, buf);
              if (len != ssh_hash_digest_length(hash))
                ssh_fatal("error: file digest length incorrect.");

              if (memcmp(str, buf, ssh_hash_digest_length(hash)) != 0)
                {
                  printf("Wrong digest: ");
                  for (i = 0; i < ssh_hash_digest_length(hash); i++)
                    {
                      printf("%02x", buf[i]);
                    }
                  printf("\nShould be digest: ");
                  for (i = 0; i < ssh_hash_digest_length(hash); i++)
                    {
                      printf("%02x", str[i]);
                    }
                  printf("\n");
                  ssh_fatal("error: %s failed.", hash_name);
                }
              state = HASH_INPUT;
              break;
            case HASH_IGNORE:
              break;
            default:
              ssh_fatal("error: unknown hash flag (%d).", state);
              break;
            }

          break;
        case RF_EMPTY:
          break;
        default:
          ssh_fatal("error: file error or corrupted (%d).", status);
          break;
        }
    }

  ssh_t_close();

  if (hash)
    ssh_hash_free(hash);
}

void hash_static_tests_do(void)
{
  char *namelist = ssh_hash_get_supported();
  const char *tmp_namelist = namelist;
  char *hash_name = NULL;
  unsigned char buf[SSH_MAX_HASH_DIGEST_LENGTH], *buf2;
  SshHash hash;
  int i, j, len;
  RFStatus status;

  status = ssh_t_write_init("hash.tests.created");
  if (status != RF_WRITE)
    ssh_fatal("error: file hash.tests.created could not be created.");

  ssh_t_write_token(RF_LINEFEED, NULL, 0);
  ssh_t_write_token(RF_COMMENT, filename, strlen(filename));

  while ((hash_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      if (ssh_hash_allocate(hash_name, &hash) != SSH_CRYPTO_OK)
        ssh_fatal("error: hash allocate %s failed.", hash_name);

      /* Put here some tests. */

      ssh_t_write_token(RF_LINEFEED, NULL, 0);
      ssh_t_write_token(RF_LABEL, (unsigned char *) hash_name,
                        strlen(hash_name));
      ssh_t_write_token(RF_LINEFEED, NULL, 0);

      for (i = 0; i < 64; i++)
        {
          buf2 = (unsigned char *) "first input then digest";
          ssh_t_write_token(RF_COMMENT, buf2, strlen((char *) buf2));

          len = i + 10;
          buf2 = ssh_xmalloc(len);
          for (j = 0; j < len; j++)
            buf2[j] = ssh_random_get_byte();

          ssh_t_write_token(RF_HEX, buf2, len);
          ssh_t_write_token(RF_LINEFEED, NULL, 0);

          ssh_hash_reset(hash);
          ssh_hash_update(hash, buf2, len);
          ssh_hash_final(hash, buf);

          ssh_t_write_token(RF_HEX, buf, ssh_hash_digest_length(hash));
          ssh_t_write_token(RF_LINEFEED, NULL, 0);

          ssh_xfree(buf2);
        }

      /* Put here some tests. */

      ssh_xfree(hash_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);

      ssh_hash_free(hash);
    }

  ssh_t_write_token(RF_LINEFEED, NULL, 0);
  ssh_t_write_token(RF_COMMENT, filename, strlen(filename));
  ssh_t_write_token(RF_LINEFEED, NULL, 0);

  ssh_t_close();

  ssh_xfree(namelist);
}

void test_hash(int flag)
{
  ssh_snprintf(filename, sizeof(filename), "%s/hash.tests", srcpath);

  printf(" - random tests (with timing).\n");
  hash_random_tests();

  printf(" - generating static test cases.\n");
  hash_static_tests_do();
  printf(" - running static tests.\n");
  hash_static_tests();
}

#endif /* SSHDIST_CRYPT_GENHASH */
