/*

  gentest.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1996, 2000, 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Fri Nov  1 05:37:55 1996 [mkojo]

  Testing those gen- prefixed files.

  */
#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshtimemeasure.h"
#include "readfile.h"
#include "sshmp.h"
#include "sshdsprintf.h"
#include "sshcryptocore/namelist.h"
#include "t-gentest.h"

/*********************** MAC tests. *****************************/

void mac_random_tests(void)
{
  char *namelist = ssh_mac_get_supported();
  const char *tmp_namelist = namelist;
  char *mac_name = NULL;
  unsigned char *key;
  SshUInt32 keylen;
  unsigned char *buf;
  unsigned char *buf2;
  SshMac mac;
  struct SshTimeMeasureRec tmit = SSH_TIME_MEASURE_INITIALIZER;
  int i, len;

  while ((mac_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      keylen = (SshUInt32)ssh_random_get_byte();
      key = ssh_xmalloc(keylen);

      for (i = 0; i < keylen; i++)
        key[i] = ssh_random_get_byte();

      if (ssh_mac_allocate(mac_name, key, keylen, &mac) != SSH_CRYPTO_OK)
        ssh_fatal("error: mac allocate %s failed.", mac_name);

      ssh_xfree(key);

      buf = ssh_xmalloc(ssh_mac_length(mac));

      len = 1000;
    retry:
      buf2 = ssh_xmalloc(len);

      for (i = 0; i < len; i++)
        buf2[i] = (i & 0xff);

      /* Put here some tests. */

      ssh_mac_start(mac);

      ssh_time_measure_reset(&tmit);
      ssh_time_measure_start(&tmit);

      for (i = 0; i < 1024; i++)
        {
          ssh_mac_update(mac, buf2, len);
        }

      ssh_time_measure_stop(&tmit);

      ssh_xfree(buf2);

      if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) <= TEST_TIME_MIN && len < 10000000)
        {
          if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_MILLISECOND) < 10)
            {
              len *= 128;
            }
          else
            {
              len = (int) (len * TEST_TIME_OPTIMUM
                           / ssh_time_measure_get(
                                        &tmit,
                                        SSH_TIME_GRANULARITY_MILLISECOND));
              len |= 0x3ff;
              len++;
            }
          if (verbose)
            printf("  - %s was too fast, retrying...\n", mac_name);
          goto retry;
        }

      if (ssh_time_measure_get(&tmit, SSH_TIME_GRANULARITY_SECOND) >= TEST_TIME_MIN)
        printf("%s -- " TEST_FMT " KBytes/sec\n",
               mac_name, ((double)len) /
               ((double) ssh_time_measure_get(&tmit,
                                              SSH_TIME_GRANULARITY_MICROSECOND)
                / 1000000.0));
      else
        printf("  - timing could not be performed for %s.\n", mac_name);


      ssh_mac_final(mac, buf);

      /* Put here some tests. */

      ssh_xfree(buf);
      ssh_xfree(mac_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);

      ssh_mac_free(mac);
    }
  ssh_xfree(namelist);
}

void mac_static_tests(void)
{
  char mac_name[256];
  unsigned char *buf = NULL;
  unsigned char *str;
  size_t len;
  int i;
  SshMac mac = NULL;
  RFStatus status;
#define MAC_IGNORE 0
#define MAC_READ_KEY 1
#define MAC_OUTPUT   2
#define MAC_INPUT    3
  unsigned int state = MAC_IGNORE;

  status = ssh_t_read_init(filename);
  if (status != RF_READ)
    ssh_fatal("error: file mac.tests not available.");

  while (status != RF_EMPTY)
    {
      status = ssh_t_read_token(&str, &len);
      switch (status)
        {
        case RF_LABEL:
          if (mac != NULL)
            {
              ssh_mac_free(mac);
              ssh_xfree(buf);
            }

          if (len > 256)
            ssh_fatal("error: mac name too long.");
          memcpy(mac_name, str, len);
          mac_name[len] = '\0';

          if (ssh_mac_supported(mac_name))
            state = MAC_READ_KEY;
          else
            {
              ssh_debug("mac %s not supported", mac_name);
              state = MAC_IGNORE;
            }
          break;
        case RF_HEX:
        case RF_ASCII:
          switch (state)
            {
            case MAC_READ_KEY:
              if (ssh_mac_allocate(mac_name, str, len, &mac) != SSH_CRYPTO_OK)
                ssh_fatal("error: mac allocate %s failed.", mac_name);

              buf = ssh_xmalloc(ssh_mac_length(mac));

              state = MAC_INPUT;
              break;
            case MAC_INPUT:
              ssh_mac_start(mac);
              ssh_mac_update(mac, str, len);
              state = MAC_OUTPUT;
              break;
            case MAC_OUTPUT:
              ssh_mac_final(mac, buf);

              if (len < ssh_mac_length(mac))
                ssh_fatal("error: file mac output too short.");

              if (memcmp(str, buf, ssh_mac_length(mac)) != 0)
                {
                  printf("Wrong digest: ");
                  for (i = 0; i < ssh_mac_length(mac); i++)
                    {
                      printf("%02x", buf[i]);
                    }
                  printf("\nShould be digest: ");
                  for (i = 0; i < ssh_mac_length(mac); i++)
                    {
                      printf("%02x", str[i]);
                    }
                  printf("\n");
                  ssh_fatal("error: mac %s failed.", mac_name);
                }

              state = MAC_INPUT;
              break;
            case MAC_IGNORE:
              break;
            default:
              ssh_fatal("error: unknown state (%d).", state);
              break;
            }
        case RF_EMPTY:
          break;
        default:
          ssh_fatal("error: file corrupted (%d).", status);
          break;
        }
    }

  ssh_t_close();
  ssh_mac_free(mac);
  ssh_xfree(buf);
}

void mac_static_tests_do(void)
{
  char *namelist = ssh_mac_get_supported();
  const char *tmp_namelist = namelist;
  char *mac_name = NULL;
  unsigned char *key;
  size_t keylen;
  unsigned char *buf;
  unsigned char *buf2;
  SshMac mac;
  int i, j, k, len;
  RFStatus status;

  status = ssh_t_write_init("mac.tests.created");
  if (status != RF_WRITE)
    ssh_fatal("error: could not open mac.tests.created for writing.");

  ssh_t_write_token(RF_LINEFEED, NULL, 0);
  ssh_t_write_token(RF_COMMENT, filename, strlen(filename));
  ssh_t_write_token(RF_LINEFEED, NULL, 0);

  while ((mac_name = ssh_name_list_get_name(tmp_namelist)) != NULL)
    {
      ssh_t_write_token(RF_COMMENT, (unsigned char *) mac_name,
                        strlen(mac_name));
      for (k = 0; k < 16; k++)
        {
          keylen = 1 + (ssh_random_get_byte() & 31);
          key = ssh_xmalloc(keylen);
          
          for (i = 0; i < keylen; i++)
            key[i] = ssh_random_get_byte();

          if (ssh_mac_allocate(mac_name, key, keylen, &mac) != SSH_CRYPTO_OK)
            ssh_fatal("error: mac allocate %s failed.", mac_name);

          ssh_t_write_token(RF_LINEFEED, NULL, 0);
          ssh_t_write_token(RF_LABEL, (unsigned char *) mac_name,
                            strlen(mac_name));

          ssh_t_write_token(RF_HEX, key, keylen);
          ssh_t_write_token(RF_LINEFEED, NULL, 0);

          ssh_xfree(key);

          for (j = 0; j < 8; j++)
            {

              buf = ssh_xmalloc(ssh_mac_length(mac));

              len = j*2 + 10;
              buf2 = ssh_xmalloc(len);

              for (i = 0; i < len; i++)
                buf2[i] = ssh_random_get_byte();

              ssh_t_write_token(RF_HEX, buf2, len);

              /* Put here some tests. */

              ssh_mac_start(mac);

              ssh_mac_update(mac, buf2, len);
              ssh_xfree(buf2);

              ssh_mac_final(mac, buf);

              ssh_t_write_token(RF_HEX, buf, ssh_mac_length(mac));

              ssh_t_write_token(RF_LINEFEED, NULL, 0);

              /* Put here some tests. */

              ssh_xfree(buf);
            }
          ssh_mac_free(mac);
        }
      ssh_xfree(mac_name);
      tmp_namelist = ssh_name_list_step_forward(tmp_namelist);
    }
  ssh_xfree(namelist);

  ssh_t_close();
}

void test_mac(int flag)
{
  ssh_snprintf(filename, sizeof(filename), "%s/mac.tests", srcpath);

  printf(" - random tests (with timing).\n");
  mac_random_tests();

  printf(" - creating static test cases.\n");
  mac_static_tests_do();
  printf(" - running static tests.\n");
  mac_static_tests();
}
