/*

  sshmp.h

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1996-2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Fri Jan 16 18:23:36 1998 [mkojo]

  */

#ifndef SSHMP_H
#define SSHMP_H

/************************************************************************/
/* General definitions.                                                 */

/* The pseudo-random number generator. Please recall that this is not a
   cryptographically secure generator. It is mainly used at places where
   large number of pseudo-random numbers are needed at quick interval.

   Indeed, all routines in the mathematics library utilize this random
   number generator, if they use any. To create cryptographical
   randomness use some external means to generate it and introduce it
   to the computations through the high-level interface.
*/
#include "sshrand.h"

#include "sshbuffer.h"


/************************************************************************/
/* Mathematic library specific definitions.                             */

/* Mathematics library special type definitions. */
#include "sshmp-types.h"

/* The kernel definitions and primitives. */
#include "sshmp-kernel.h"

/* This defines the use of workspace. That is, the workspace will be
   allocated (to minimize the needed allocations in modular arithmetic)
   to the modulus structure.

   You can of course undefine it, and then most allocation will be handled
   dynamically when doing computations. */
#define SSHMATH_USE_WORKSPACE

/************************************************************************/
/* Integers.                                                            */

/* Library for large integers. Most libraries requiring support for
   large integers do not need the code below. */
#include "sshmp-integer.h"

int
ssh_mprz_encode_rendered(unsigned char *buf, size_t len,
                         SshMPInteger mp);
int
ssh_mprz_decode_rendered(const unsigned char *buf, size_t len,
                         SshMPInteger mp);

int
ssh_mprz_encode_ssh2style(SshMPIntegerConst mp,
                          unsigned char *buf, size_t len);
int
ssh_mprz_decode_ssh2style(const unsigned char *buf, size_t len,
                          SshMPInteger mp);

/************************************************************************/
/* Arithmetic.                                                          */

#ifdef SSHDIST_MATH_SIEVE
/* The prime number sieve. This is required by certain arithmetical
   functions. */
#include "sshsieve.h"
#endif /* SSHDIST_MATH_SIEVE */

#ifdef SSHDIST_MATH_ARITHMETIC
/* Arithmetic of numbers. */
#include "sshmp-arithmetic.h"
#endif /* SSHDIST_MATH_ARITHMETIC */

/************************************************************************/
/* Routines for residue rings.                                          */

#ifdef SSHDIST_MATH_2ADIC
/* 2-adic integers. */
#include "sshmp-2adic.h"
#endif /* SSHDIST_MATH_2ADIC */
#ifdef SSHDIST_MATH_MONTGOMERY
/* Montgomery representation. */
#include "sshmp-montgomery.h"
#endif /* SSHDIST_MATH_MONTGOMERY */

#ifdef SSHDIST_MATH_INTMOD
/* Library for integers modulo N. */
#include "sshmp-intmod.h"
#endif /* SSHDIST_MATH_INTMOD */

/************************************************************************/
/* Numbers.                                                             */














/************************************************************************/
/* Miscellaneous, but important.                                        */

/* Convenience abstractions. */
#include "sshmp-convabs.h"

#ifdef SSHDIST_MATH_POWM
/* Convenient functions for modular exponentiation, used extensively
   in the older versions. It is possible that this interface will be
   moved later to the "legacy" department. */
#include "sshmp-powm.h"
#endif /* SSHDIST_MATH_POWM */

/************************************************************************/
/* Legacy.                                                              */

#ifdef SSHDIST_MATH_COMPAT
/* Compatibility definitions. Most functions of the previous versions
   work directly. There exists functions that are not supported in
   the current version, but which were present in the previous versions.

   Care has been taken to make the compiler indicate some reason for
   the failure to compile code when it is due to missing support for
   certain functionality.
*/
#include "sshmp-compat.h"
#endif /* SSHDIST_MATH_COMPAT */

/************************************************************************/
/* Function fields (and rings).                                         */




















/* Future work, more general function field code. */

/************************************************************************/
/* Elliptic curves (and rings).                                         */





/* XX Initial conversion from the old code. Some modifications are to
   be still made. */








#endif /* SSHMP_H */
