/*

  sshadt_array.c

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 1999, 2000, 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Thu Oct 21 13:35:14 1999.

  */

#include "sshincludes.h"
#include "sshadt_i.h"
#include "sshadt_array.h"
#include "sshadt_array_i.h"
#include "sshadt_std_i.h"
#include "sshdebug.h"

#define SSH_DEBUG_MODULE "SshADTArray"

#define ROOT(c)     ((SshADTArrayRoot *)(c->container_specific))

/* We need to add and subtract one because otherwise the index zero
   would actually correspond to SSH_ADT_INVALID, which would be
   confusing. */
#define PTRTOUINT(ptr) ((((unsigned char *)(ptr)) - ((unsigned char *)0)) - 1)
#define UINTTOPTR(i) (&(((unsigned char *)0)[i + 1]))

static Boolean init(SshADTContainer c)
{
  /* Handles are of type (void *)idx.  */
  SSH_VERIFY(sizeof(void *) >= sizeof(unsigned int));

  /* The dynamic array type has zero-sized headers so they should
     not be `contained'. */
  SSH_ASSERT(!(c->flags & SSH_ADT_FLAG_CONTAINED_HEADER) &&
             "See documentation (array specifics section).");

  if ((c->container_specific = ssh_malloc(sizeof(*ROOT(c)))) == NULL)
    return FALSE;

  ROOT(c)->array = NULL;
  ROOT(c)->array_size = 0;

  return TRUE;
}

/* $$METHOD(array, clear) */
static void clear(SshADTContainer c)
{
  int i;
  for (i = 0; i < ROOT(c)->array_size; i++)
    {
      if (ROOT(c)->array[i] != NULL)
        ssh_adt_delete(c, UINTTOPTR(i));
    }
}

/* $$METHOD(pq, clear) */
static void pq_clear(SshADTContainer c)
{
  SshADTHandle h = UINTTOPTR(0);

  while (c->num_objects > 0)
    {
      ssh_adt_delete(c, h);
    }
}

/* $$METHOD(array, container_init) */
/* $$METHOD(pq, container_init) */
SSH_ADT_STD_INIT(container_init, init(c))

static void uninit(SshADTContainer c)
{
  clear(c);
  ssh_free(ROOT(c)->array);
  ssh_free(ROOT(c));
}

static void pq_uninit(SshADTContainer c)
{
  pq_clear(c);
  ssh_free(ROOT(c)->array);
  ssh_free(ROOT(c));
}

/* $$METHOD(array, destr) */
SSH_ADT_STD_DESTROY(destr, uninit(c);)

/* $$METHOD(pq, destr) */
SSH_ADT_STD_DESTROY(pq_destr, pq_uninit(c);)

/* After calling initialize_cell(c, n), the nth (0-based) cell is
   valid and empty. The array is expanded if necessary, and if there
   was an object at the nth cell, the object has been deleted.

   In usermode or if enough memory is available, initialize_cell
   always succeeds and returns TRUE.  In kernel mode and if
   ssh_calloc returns NULL, the new cell is not allocated and
   FALSE is returned. */
static Boolean initialize_cell(SshADTContainer c, unsigned int idx)
{
  size_t i, old_size, new_size;
  void **array;

  if (ROOT(c)->array_size <= idx)
    {
      old_size = ROOT(c)->array_size;
      new_size = idx + (idx / 4) + 1;

      if ((array =
           ssh_realloc(ROOT(c)->array,
                       old_size * sizeof(ROOT(c)->array[0]),
                       new_size * sizeof(ROOT(c)->array[0]))) == NULL)
        return FALSE;

      ROOT(c)->array = array;

      for (i = old_size; i < new_size ; i++)
        array[i] = NULL;

      ROOT(c)->array_size = new_size;
      return TRUE;
    }
  else
    {
      if (ROOT(c)->array[idx] != NULL)
        ssh_adt_delete(c, UINTTOPTR(idx));
    }

  return TRUE;
}

static Boolean empty_idx(SshADTContainer c,
                         SshADTAbsoluteLocation location,
                         unsigned int *idx)
{
  SSH_ASSERT(SSH_ADT_IS_INDEX(location));
  *idx = SSH_ADT_GET_INDEX(location);
  return initialize_cell(c, *idx);
}

/* $$METHOD(array, insert_to) */
static SshADTHandle insert_to(SshADTContainer c,
                              SshADTAbsoluteLocation location,
                              void *object)
{
  unsigned int idx;
  SshADTHandle h;

  if (!empty_idx(c, location, &idx))
    return SSH_ADT_INVALID;

  ROOT(c)->array[idx] = object;
  c->num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

static void percolate_up(SshADTContainer c, int idx)
{
  int parent;
  int cmp;
  void *tmp;

  while (idx > 0)
    {
      parent = (idx - 1) >> 1;
      SSH_ADT_CALL_APP_MANDATORY(c, compare,
                                 (ROOT(c)->array[idx],
                                  ROOT(c)->array[parent],
                                  SSH_ADT_APPCTX(c)),
                                 cmp);
      SSH_DEBUG(9, ("Comp %d %d == %d\n", idx, parent, cmp));
      if (cmp >= 0) return;
      tmp = ROOT(c)->array[parent];
      ROOT(c)->array[parent] = ROOT(c)->array[idx];
      ROOT(c)->array[idx] = tmp;
      idx = parent;
    }
}

static void percolate_down(SshADTContainer c, int idx)
{
  int left, right, child;
  int cmp, child_cmp;
  void *tmp;

  while (left = (idx << 1) + 1,
         right = left + 1,
         left < c->num_objects)
    {
      if (right < c->num_objects)
        {
          SSH_ADT_CALL_APP_MANDATORY(c, compare,
                                     (ROOT(c)->array[left],
                                      ROOT(c)->array[right],
                                      SSH_ADT_APPCTX(c)),
                                     child_cmp);
        }
      else
        {
          child_cmp = -1; /* use the left branch anyway */
        }
      if (child_cmp < 0) child = left; else child = right;

      SSH_ADT_CALL_APP_MANDATORY(c, compare,
                                 (ROOT(c)->array[idx],
                                  ROOT(c)->array[child],
                                  SSH_ADT_APPCTX(c)),
                                 cmp);
      if (cmp <= 0) return;
      tmp = ROOT(c)->array[child];
      ROOT(c)->array[child] = ROOT(c)->array[idx];
      ROOT(c)->array[idx] = tmp;

      idx = child;
    }
}

/* $$METHOD(pq, insert_to) */
static SshADTHandle pq_insert_to(SshADTContainer c,
                                 SshADTAbsoluteLocation location,
                                 void *object)
{
  unsigned int idx;

  SSH_ASSERT(location == SSH_ADT_DEFAULT);

  idx = c->num_objects; /* Add to the end. */
  if (!initialize_cell(c, idx))
    return SSH_ADT_INVALID;

  ROOT(c)->array[idx] = object;
  c->num_objects++;
  percolate_up(c, idx);
  return UINTTOPTR(0);
}

/* $$METHOD(array, get_handle_to_location) */
static SshADTHandle get_handle_to_location(SshADTContainer c,
                                           SshADTAbsoluteLocation location)
{
  unsigned int idx = SSH_ADT_GET_INDEX(location);

  if (idx >= ROOT(c)->array_size) return SSH_ADT_INVALID;
  else return UINTTOPTR(idx);
}

/* $$METHOD(pq, get_handle_to_location) */
static SshADTHandle pq_get_handle_to_location(SshADTContainer c,
                                              SshADTAbsoluteLocation location)
{
  SSH_ASSERT(location == SSH_ADT_DEFAULT);

  if (c->num_objects == 0) return SSH_ADT_INVALID;
  else return UINTTOPTR(0);
}

/* $$METHOD(array, alloc_n_to) */
static SshADTHandle alloc_n_to(SshADTContainer c,
                               SshADTAbsoluteLocation location,
                               size_t size)
{
  unsigned int idx;
  SshADTHandle h;
  void *newp;

  if (!empty_idx(c, location, &idx))
    return SSH_ADT_INVALID;
  if ((ROOT(c)->array[idx] = newp = ssh_malloc(size)) == NULL)
    return SSH_ADT_INVALID;

  c->num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, init, (newp, size, SSH_ADT_APPCTX(c)));
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

/* $$METHOD(array, put_n_to) */
static SshADTHandle put_n_to(SshADTContainer c,
                             SshADTAbsoluteLocation location,
                             size_t size,
                             void *object)
{
  unsigned int idx;
  SshADTHandle h;
  void *newp;

  if (!empty_idx(c, location, &idx))
    return SSH_ADT_INVALID;
  if ((ROOT(c)->array[idx] = newp = ssh_malloc(size)) == NULL)
    return SSH_ADT_INVALID;

  c->num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, copy, (newp, size, object, SSH_ADT_APPCTX(c)));
  SSH_ADT_CALL_HOOK(c, insert, h);
  return h;
}

/* $$METHOD(pq, alloc_n_to) */
static SshADTHandle pq_alloc_n_to(SshADTContainer c,
                                  SshADTAbsoluteLocation location,
                                  size_t size)
{
  unsigned int idx;
  SshADTHandle h;
  void *newp;

  SSH_ASSERT(location == SSH_ADT_DEFAULT);

  idx = c->num_objects;
  if (!initialize_cell(c, idx))
    return SSH_ADT_INVALID;
  if ((ROOT(c)->array[idx] = newp = ssh_malloc(size)) == NULL)
    return SSH_ADT_INVALID;

  c->num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, init, (newp, size, SSH_ADT_APPCTX(c)));
  percolate_up(c, idx);
  return h;
}

/* $$METHOD(pq, put_n_to) */
static SshADTHandle pq_put_n_to(SshADTContainer c,
                                SshADTAbsoluteLocation location,
                                size_t size,
                                void *object)
{
  unsigned int idx;
  SshADTHandle h;
  void *newp;

  SSH_ASSERT(location == SSH_ADT_DEFAULT);

  idx = c->num_objects;
  if (!initialize_cell(c, idx))
    return SSH_ADT_INVALID;
  if ((ROOT(c)->array[idx] = newp = ssh_malloc(size)) == NULL)
    return SSH_ADT_INVALID;

  c->num_objects++;
  h = UINTTOPTR(idx);
  SSH_ADT_CALL_APP(c, copy, (newp, size, object, SSH_ADT_APPCTX(c)));
  percolate_up(c, idx);
  return h;
}

/* $$METHOD(array, get) */
/* $$METHOD(pq, get) */
static void *get(SshADTContainer c, SshADTHandle h)
{
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(idx < ROOT(c)->array_size);
  return ROOT(c)->array[idx];
}

/* $$METHOD(array, num_objects) */
/* $$METHOD(pq,    num_objects) */
SSH_ADT_STD_NUM_OBJECTS(num_objects)

/* $$METHOD(array, get_handle_to) */
/* $$METHOD(pq,    get_handle_to) */
static SshADTHandle get_handle_to(SshADTContainer c, void *object)
{
  unsigned int i;
  for (i = 0; i < ROOT(c)->array_size; i++)
    {
      if (ROOT(c)->array[i] == object)
        return UINTTOPTR(i);
    }
  return SSH_ADT_INVALID;
}

/* $$METHOD(array, enumerate_start) */
/* $$METHOD(pq, enumerate_start) */
static SshADTHandle enum_start(SshADTContainer c)
{
  if (c->num_objects == 0) return SSH_ADT_INVALID;
  else return UINTTOPTR(0);
}

/* $$METHOD(array, enumerate_next) */
static SshADTHandle enum_next(SshADTContainer c, SshADTHandle h)
{
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  idx++;
  if (idx >= ROOT(c)->array_size) return SSH_ADT_INVALID;
  else return UINTTOPTR(idx);
}

/* $$METHOD(pq, enumerate_next) */
static SshADTHandle pq_enum_next(SshADTContainer c, SshADTHandle h)
{
  unsigned int idx = PTRTOUINT(h);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  idx++;
  if (idx >= c->num_objects) return SSH_ADT_INVALID;
  else return UINTTOPTR(idx);
}

/* $$METHOD(array, detach) */
static void *detach(SshADTContainer c, SshADTHandle h)
{
  void *object;
  unsigned int idx = PTRTOUINT(h);

  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(idx < ROOT(c)->array_size);

  SSH_ADT_CALL_HOOK(c, detach, h);

  c->num_objects--;

  object = ROOT(c)->array[idx];
  ROOT(c)->array[idx] = NULL;
  return object;
}

/* $$METHOD(pq, detach) */
static void *pq_detach(SshADTContainer c, SshADTHandle h)
{
  void *object;
  unsigned int idx = PTRTOUINT(h);

  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(idx < ROOT(c)->array_size);

  SSH_ADT_CALL_HOOK(c, detach, h);

  object = ROOT(c)->array[idx];

  c->num_objects--;
  ROOT(c)->array[idx] = ROOT(c)->array[c->num_objects];
  ROOT(c)->array[c->num_objects] = NULL;
  percolate_down(c, idx);

  return object;
}

/* $$METHOD(array, reallocate) */
/* $$METHOD(pq, reallocate) */
static void *reallocate(SshADTContainer c, void *object, size_t new_size)
{
  ssh_fatal("Realloc does not work with arrays.  Please complain.");
  return NULL;
}

/* $$METHOD(array, delet) */
/* $$METHOD(pq, delet) */
static void delet(SshADTContainer c, SshADTHandle handle)
{
  void *object = ssh_adt_detach_i(c, handle);
  SSH_ADT_CALL_APP(c, destr, (object, SSH_ADT_APPCTX(c)));
  if (c->flags & SSH_ADT_FLAG_ALLOCATE)
    {
      ssh_free(object);
    }
}

SshADTStaticData ssh_adt_array_static_data =
{
  {
    /* $$METHODS(array) */
    /* DO NOT EDIT THIS, edit METHODS.h and 
       the method implementations above instead. */
    container_init, /* container_init */
    clear, /* clear */
    destr, /* destr */
    NULL, /* insert_at */
    insert_to, /* insert_to */
    NULL, /* alloc_n_at */
    alloc_n_to, /* alloc_n_to */
    NULL, /* put_n_at */
    put_n_to, /* put_n_to */
    get, /* get */
    num_objects, /* num_objects */
    get_handle_to, /* get_handle_to */
    get_handle_to_location, /* get_handle_to_location */
    NULL, /* next */
    NULL, /* previous */
    enum_start, /* enumerate_start */
    enum_next, /* enumerate_next */
    NULL, /* get_handle_to_equal */
    reallocate, /* reallocate */
    detach, /* detach */
    delet, /* delet */
    NULL, /* map_lookup */
    NULL, /* map_attach */
    /* $$ENDMETHODS */
  },
  0,
  0
};

SshADTStaticData ssh_adt_pq_static_data =
{
  {
    /* $$METHODS(pq) */
    /* DO NOT EDIT THIS, edit METHODS.h and 
       the method implementations above instead. */
    container_init, /* container_init */
    pq_clear, /* clear */
    pq_destr, /* destr */
    NULL, /* insert_at */
    pq_insert_to, /* insert_to */
    NULL, /* alloc_n_at */
    pq_alloc_n_to, /* alloc_n_to */
    NULL, /* put_n_at */
    pq_put_n_to, /* put_n_to */
    get, /* get */
    num_objects, /* num_objects */
    get_handle_to, /* get_handle_to */
    pq_get_handle_to_location, /* get_handle_to_location */
    NULL, /* next */
    NULL, /* previous */
    enum_start, /* enumerate_start */
    pq_enum_next, /* enumerate_next */
    NULL, /* get_handle_to_equal */
    reallocate, /* reallocate */
    pq_detach, /* detach */
    delet, /* delet */
    NULL, /* map_lookup */
    NULL, /* map_attach */
    /* $$ENDMETHODS */
  },
  0,
  0
};


const SshADTContainerType ssh_adt_array_type = &ssh_adt_array_static_data;
const SshADTContainerType ssh_adt_priority_queue_type = &ssh_adt_pq_static_data;
