/*

  sshadt_conv.c

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 2000, 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Wed Sep 20 05:00:29 2000.

  */

#include "sshincludes.h"
#include "sshdebug.h"
#include "sshadt_i.h"
#include "sshadt_map.h"
#include "sshadt_strmap.h"
#include "sshadt_intmap.h"
#include "sshadt_xmap.h"

#define SSH_DEBUG_MODULE "SshADTConv"



/**************************************** Callbacks that are frequently used */

void ssh_adt_callback_destroy_free(void *obj, void *context)
{
  ssh_free(obj);
}

void ssh_adt_callback_destroy_free_null(void *obj, void *context)
{
  if (obj == NULL) return;
  ssh_free(obj);
}

int ssh_adt_callback_compare_str(const void *obj1, const void *obj2, void *ctx)
{
  const char *s1 = obj1, *s2 = obj2;
  return strcmp(s1, s2);
}

void *ssh_adt_callback_duplicate_str(const void *obj, void *ctx)
{
  const char *s = obj;
  return ssh_strdup(s);
}

SshUInt32 ssh_adt_callback_hash_str(const void *obj, void *ctx)
{
  SshUInt32 hash = 0;
  const char *c = obj;
  while (*c)
    {
      hash = (hash << 17) + (hash >> 15) + *c;
      c++; /* I dislike c++. */
    }
  return hash;
}

int ssh_adt_callback_compare_int(const void *obj1, const void *obj2,
                                 void *context)
{
  if (*(SshInt32*)obj1 < *(SshInt32*)obj2) return -1;
  if (*(SshInt32*)obj1 > *(SshInt32*)obj2) return 1;
  return 0;
}

SshUInt32 ssh_adt_callback_hash_int(const void *obj, void *ctx)
{
  return *(SshUInt32 *)obj;
}


/******************************************** Maps from strings to something */

SshADTContainer ssh_adt_create_strmap(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_HASH, ssh_adt_callback_hash_str,
                                SSH_ADT_DUPLICATE,
                                        ssh_adt_callback_duplicate_str,
                                SSH_ADT_DESTROY, ssh_adt_callback_destroy_free,
                                SSH_ADT_COMPARE, ssh_adt_callback_compare_str,
                                SSH_ADT_ARGS_END);
}

SshADTHandle ssh_adt_strmap_add(SshADTContainer c, const char *key,
                                void *value)
{
  SshADTHandle h;
  SSH_ASSERT(!(ssh_adt_strmap_exists(c, key)));
  h = ssh_adt_duplicate(c, (void *)key);
  SSH_ASSERT(ssh_adt_strmap_exists(c, key));
  ssh_adt_map_attach(c, h, value);
  return h;
}

void ssh_adt_strmap_remove(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  if (h != SSH_ADT_INVALID)
    {
      ssh_adt_delete(c, h);
    }
}

void ssh_adt_strmap_set(SshADTContainer c, const char *key, void *value)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  ssh_adt_map_attach(c, h, value);
}

void *ssh_adt_strmap_get(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  if (h == SSH_ADT_INVALID)
    return NULL;
  return ssh_adt_map_lookup(c, h);
}

Boolean ssh_adt_strmap_exists(SshADTContainer c, const char *key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, (void *)key);
  return (h != SSH_ADT_INVALID);
}

/******************************************* Maps from integers to something */

SshADTContainer ssh_adt_create_intmap(void)
{
  return ssh_adt_create_generic(SSH_ADT_MAP,
                                SSH_ADT_HASH, ssh_adt_callback_hash_int,
                                SSH_ADT_COMPARE, ssh_adt_callback_compare_int,
                                SSH_ADT_SIZE, sizeof(SshUInt32),
                                SSH_ADT_ARGS_END);
}

SshADTHandle ssh_adt_intmap_add(SshADTContainer c, SshUInt32 key,
                                void *value)
{
  SshADTHandle h;
  SSH_ASSERT(!(ssh_adt_intmap_exists(c, key)));
  h = ssh_adt_put(c, &key);
  SSH_ASSERT(ssh_adt_intmap_exists(c, key));
  ssh_adt_map_attach(c, h, value);
  return h;
}

void ssh_adt_intmap_remove(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  if (h != SSH_ADT_INVALID)
    {
      ssh_adt_delete(c, h);
    }
}

void ssh_adt_intmap_set(SshADTContainer c, SshUInt32 key, void *value)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  ssh_adt_map_attach(c, h, value);
}

void *ssh_adt_intmap_get(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  if (h == SSH_ADT_INVALID)
    return NULL;
  return ssh_adt_map_lookup(c, h);
}

Boolean ssh_adt_intmap_exists(SshADTContainer c, SshUInt32 key)
{
  SshADTHandle h = ssh_adt_get_handle_to_equal(c, &key);
  return (h != SSH_ADT_INVALID);
}

/************************************************ The abstract map interface */

/* for a key previously not in the map, insert the key and map it.  */
SshADTHandle ssh_adt_xmap_add(SshADTContainer c, void *key, void *value)
{
  SSH_ASSERT(c->static_data->methods.map_attach);
  SSH_ASSERT(c->static_data->methods.map_lookup);

  {
    SshADTHandle h;
    SSH_ASSERT(!(ssh_adt_xmap_exists(c, key)));

    if (c->flags & SSH_ADT_FLAG_ALLOCATE)   /* liballoc */
      h = ssh_adt_put(c, key);
    else                                    /* voidptr */
      h = ssh_adt_duplicate(c, key);

    SSH_ASSERT(ssh_adt_xmap_exists(c, key));
    ssh_adt_map_attach(c, h, value);
    return h;
  }
}

/* for any existing key from the map, delete the key.  if existing,
   the pointer to the value / image is dropped, and the MapDestroy
   callback can be used to free it.  */
void ssh_adt_xmap_remove(SshADTContainer c, void *key)
{
  SSH_ASSERT(c->static_data->methods.map_attach);
  SSH_ASSERT(c->static_data->methods.map_lookup);

  {
    SshADTHandle h = ssh_adt_get_handle_to_equal(c, key);
    if (h != SSH_ADT_INVALID)
      ssh_adt_delete(c, h);
  }
}

/* for a key already in the map, update the value.  the old value
   pointer is dropped, and the MapDropRef callback can be used to free
   it.  */
void ssh_adt_xmap_set(SshADTContainer c, void *key, void *value)
{
  SSH_ASSERT(c->static_data->methods.map_attach);
  SSH_ASSERT(c->static_data->methods.map_lookup);

  {
    SshADTHandle h = ssh_adt_get_handle_to_equal(c, key);
    SSH_ASSERT(h != SSH_ADT_INVALID);

    ssh_free(ssh_adt_get(c, h));     /* free old value */
    ssh_adt_map_attach(c, h, value);   /* attach new value */
    ssh_adt_map_changed(c, h);                 /* relocate */
  }
}

/* get the value to a key or NULL if key is invalid.  */
void *ssh_adt_xmap_get(SshADTContainer c, void *key)
{
  SSH_ASSERT(c->static_data->methods.map_attach);
  SSH_ASSERT(c->static_data->methods.map_lookup);

  {
    SshADTHandle h = ssh_adt_get_handle_to_equal(c, key);
    if (h == SSH_ADT_INVALID)
      return NULL;
    else
      return ssh_adt_map_lookup(c, h);
  }
}

/* check whether a key exists in the map.  */
Boolean ssh_adt_xmap_exists(SshADTContainer c, void *key)
{
  SSH_ASSERT(c->static_data->methods.map_attach);
  SSH_ASSERT(c->static_data->methods.map_lookup);

  {
    return (ssh_adt_get_handle_to_equal(c, key) != SSH_ADT_INVALID);
  }
}
