/*
  File: sshbuffer.c

  Authors:
        Tatu Ylonen <ylo@ssh.fi>
        Tero T Mononen <tmo@ssh.fi>

  Description:
        Functions for manipulating fifo buffers (that can grow if needed).
        Based on the Tatu Ylonen's implementation from 1995

  Copyright:
        Copyright (c) 2000 SSH Communications Security Corp, Finland.
        All rights reserved.
        Original was copywritten by Tatu Ylonen <ylo@cs.hut.fi>.
*/

#include "sshincludes.h"
#include "sshbuffer.h"
#include "sshdebug.h"

#define SSH_DEBUG_MODULE "SshBuffer"

#define SSH_BUFFER_BASE_SIZE    1024

/* Allocates a new buffer. */
SshBuffer ssh_buffer_allocate(void)
{
  SshBuffer buffer = ssh_malloc(sizeof(*buffer));

  if (buffer)
    {
      ssh_buffer_init(buffer);
      buffer->dynamic = TRUE;
    }
  return buffer;
}

/* Zeroes and frees the buffer. */

void ssh_buffer_free(SshBuffer buffer)
{
  SSH_ASSERT(buffer);
  SSH_ASSERT(buffer->dynamic);

  ssh_buffer_uninit(buffer);
  ssh_free(buffer);
}

/* Initializes the buffer structure. */

void ssh_buffer_init(SshBuffer buffer)
{
  SSH_ASSERT(buffer);

  buffer->offset = 0;
  buffer->end = 0;
  buffer->dynamic = FALSE;
  buffer->alloc = SSH_BUFFER_BASE_SIZE;
  buffer->buf = NULL;
}

/* Frees any memory used for the buffer. */

void ssh_buffer_uninit(SshBuffer buffer)
{
  SSH_ASSERT(buffer);

  if (buffer->buf)
    {
      memset(buffer->buf, 0, buffer->alloc);
      ssh_free(buffer->buf);
    }
}

/* Clears any data from the buffer, making it empty.  This does not actually
   zero the memory. */

void ssh_buffer_clear(SshBuffer buffer)
{
  SSH_ASSERT(buffer);

  buffer->offset = 0;
  buffer->end = 0;
}

/* Appends data to the buffer, expanding it if necessary. */

SshBufferStatus
ssh_buffer_append(SshBuffer buffer, const unsigned char *data, size_t len)
{
  unsigned char *cp;
  SshBufferStatus status = SSH_BUFFER_OK;

  SSH_ASSERT(buffer);

  if ((status = ssh_buffer_append_space(buffer, &cp, len)) == SSH_BUFFER_OK)
    {
      if (len > 0)
        memcpy(cp, data, len);
    }
  return status;
}

/* Appends space to the buffer, expanding the buffer if necessary.
   This does not actually copy the data into the buffer, but instead
   returns a pointer to the allocated region. */

SshBufferStatus
ssh_buffer_append_space(SshBuffer buffer, unsigned char **datap, size_t len)
{
  unsigned char *tmp;
  size_t incr;

  SSH_ASSERT(buffer);

  /* Now allocate the buffer space if not done already. */
  if (buffer->buf == NULL)
    {
      if ((buffer->buf = ssh_malloc(buffer->alloc)) == NULL)
        return SSH_BUFFER_ERROR;
    }

  /* If the buffer is empty, start using it from the beginning. */
  if (buffer->offset == buffer->end)
    {
      buffer->offset = 0;
      buffer->end = 0;
    }

 restart:
  /* If there is enough space to store all data, store it now. */
  if (buffer->end + len < buffer->alloc)
    {
      *datap = buffer->buf + buffer->end;
      buffer->end += len;
      return SSH_BUFFER_OK;
    }

  /* If the buffer is quite empty, but all data is at the end, move the
     data to the beginning and retry. */
  if (buffer->offset > buffer->alloc / 2)
    {
      memmove(buffer->buf, buffer->buf + buffer->offset,
              buffer->end - buffer->offset);
      buffer->end -= buffer->offset;
      buffer->offset = 0;
      goto restart;
    }

  /* Increase the size of the buffer and retry. */
  if (len > SSH_BUFFER_BASE_SIZE)
    incr = len;
  else
    incr = len + SSH_BUFFER_BASE_SIZE;

  tmp = ssh_realloc(buffer->buf, buffer->alloc, buffer->alloc + incr);
  if (tmp)
    {
      buffer->buf = tmp;
      buffer->alloc += incr;
      goto restart;
    }
  else
    return SSH_BUFFER_ERROR;
}

/* Appends NUL-terminated C-strings <...> to the buffer.  The argument
   list must be terminated with a NULL pointer. */

SshBufferStatus ssh_buffer_append_cstrs(SshBuffer buffer, ...)
{
  va_list ap;
  SshBufferStatus status = SSH_BUFFER_OK;

  va_start(ap, buffer);

  status = ssh_buffer_append_cstrs_va(buffer, ap);
  
  va_end(ap);
  return status;
}

SshBufferStatus ssh_buffer_append_cstrs_va(SshBuffer buffer, va_list ap)
{
  char *str;
  SshBufferStatus status = SSH_BUFFER_OK;

  while (status == SSH_BUFFER_OK && (str = va_arg(ap, char *)) != NULL)
    status = ssh_buffer_append(buffer, (unsigned char *) str, strlen(str));

  return status;
}

/* Returns the number of bytes of data in the buffer. */

size_t ssh_buffer_len(const SshBuffer buffer)
{
  SSH_ASSERT(buffer);
  SSH_ASSERT(buffer->offset <= buffer->end);

  return buffer->end - buffer->offset;
}

/* Consumes the given number of bytes from the beginning of the buffer. */

void ssh_buffer_consume(SshBuffer buffer, size_t bytes)
{
  if (bytes > buffer->end - buffer->offset)
    ssh_fatal("buffer_consume trying to get more bytes than in buffer");
  buffer->offset += bytes;
}

/* Consumes the given number of bytes from the end of the buffer. */

void ssh_buffer_consume_end(SshBuffer buffer, size_t bytes)
{
  if (bytes > buffer->end - buffer->offset)
    ssh_fatal("buffer_consume_end trying to get more bytes than in buffer");
  buffer->end -= bytes;
}

/* Returns a pointer to the first used byte in the buffer. */

unsigned char *ssh_buffer_ptr(const SshBuffer buffer)
{
  SSH_ASSERT(buffer);
  SSH_ASSERT(buffer->offset <= buffer->end);

  if (buffer->buf == NULL || ssh_buffer_len(buffer) == 0)
    return NULL;

  return buffer->buf + buffer->offset;
}
