/*

  sshoperation.c

  Author: Tero Kivinen <kivinen@ssh.fi>
          Timo J. Rinne <tri@ssh.fi>

  Copyright (c) 1999, 2001 SSH Communications Security
                     All rights reserved

  Generic asynchronous call interface.  Registering, aborting and finishing
  asynchronous calls.

*/

/*
 * $Id: sshoperation.c,v 1.6 2001/07/25 08:07:13 tmo Exp $
 * $Log: sshoperation.c,v $ * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * $EndLog$
 */

#include "sshincludes.h"
#include "sshoperation.h"

/* The destructor record that is used to save attached destructors for
   the handle. */
typedef struct SshOperationDestructorRec
{
  /* The link to  the next destructor. */
  struct SshOperationDestructorRec *next;

  /* Callback and the context for the callback. */
  SshOperationDestructorCB destructor_cb;
  void *context;
} *SshOperationDestructor, SshOperationDestructorStruct;

/* The implementation of SshOperationHandle. */
struct SshOperationHandleRec {
  SshOperationAbortCB abort_cb;
  void *operation_context;
  SshOperationDestructor destructor_list;
  Boolean in_abort;
};


/* Registeres a callback to be called when a handle is unregistered or
   aborted. If this function is called with a NULL operation handle
   the destructor_cb is called inside the call. This functionality is
   usefull for modules that pass handles from lower modules to higher
   modules. Usually the middle module needs to know, when the
   asynchronous call ends, so that it may free the allocated data that it
   has passed as parameters to lower modules.

   This function can be called multiple times for the same handle,
   even with the same callback and context. The callbacks are simply
   queued, and called whenever the ssh_operation_unregister is
   called. The destructor functions are called in the reverse order
   than they are attached to the handle. */
void ssh_operation_attach_destructor(SshOperationHandle handle,
                                     SshOperationDestructorCB destructor_cb,
                                     void *context)
{
  SshOperationDestructor dest;
  /* The simple case. */
  if (handle == NULL)
    {
      (*destructor_cb)(context);
    }
  else if (destructor_cb)
    {
      if ((dest = ssh_calloc(1, sizeof(*dest))) != NULL)
        {
          dest->destructor_cb = destructor_cb;
          dest->context = context;
          dest->next = handle->destructor_list;
          handle->destructor_list = dest;
        }
      else
        (*destructor_cb)(context);
    }
}

/* Calls destructors for a handle, when it is being unregistered or
   aborted. */
static void ssh_operation_call_destructors(SshOperationHandle handle)
{
  SshOperationDestructor next, dest = handle->destructor_list;

  while(dest)
    {
      (*dest->destructor_cb)(dest->context);
      next = dest->next;
      ssh_free(dest);
      dest = next;
    }
}


/* Register started asynchronous call and corresponding abort callback. This
   call is called by the function that starts the real operation, and the
   handle must be returned to the caller of the function, so it can then abort
   the operation using the ssh_operation_abort. */
SshOperationHandle ssh_operation_register(SshOperationAbortCB abort_cb,
                                          void *operation_context)
{
  SshOperationHandle handle;

  if ((handle = ssh_calloc(1, sizeof(*handle))) != NULL)
    {
      handle->abort_cb = abort_cb;
      handle->operation_context = operation_context;
      handle->in_abort = FALSE;
    }
  return handle;
}

/* Unregister the handle. The abort callback will not be called, and the handle
   is invalid after this (i.e any other part of the code must not call
   ssh_operation_abort using the handle). */
void ssh_operation_unregister(SshOperationHandle handle)
{
  if (handle != NULL)
    {
      /* If we are inside the abort call, ignore the unregister calls, as the
         abort will unregister the handle for us. */
      if (handle->in_abort == TRUE)
        {
          return;
        }
      ssh_operation_call_destructors(handle);
    }

  ssh_free(handle);
}

/* Return operation context pointer from the given operation handle. */
void *ssh_operation_get_context(SshOperationHandle handle)
{
  return handle->operation_context;
}

/* Call abort callback of the pending asynchronous call. The abort callback
   will then abort the operation if it is possible, and free the context
   associated to it. Note, after this call then, the handle is invalid, and no
   other calls can be made using it. The abort callback will also make sure
   that no other callbacks are called associated with this asynchronous call.
   */
void ssh_operation_abort(SshOperationHandle handle)
{
  if (handle != NULL)
    {
      handle->in_abort = TRUE;
      (*handle->abort_cb)(handle->operation_context);
      ssh_operation_call_destructors(handle);
      handle->in_abort = FALSE;
      ssh_free(handle);
    }
}


/* eof (sshoperation.c) */
