/*

  Author: Tomi Salo <ttsalo@ssh.fi>
          Tatu Ylonen <ylo@ssh.fi>

  Copyright (c) 1996-2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Unix implementation of the UDP communications interface.

  */

#include "sshincludes.h"
#include "sshdebug.h"
#include "sshudp.h"
#include "sshtcp.h"
#include "sshtimeouts.h"
#include "ssheloop.h"
#include "sshinet.h"

#define SSH_DEBUG_MODULE "SshUdp"

#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#else /* Some old linux systems at least have in_system.h instead. */
#include <netinet/in_system.h>
#endif /* HAVE_NETINET_IN_SYSTM_H */
#if !defined(__PARAGON__)
#include <netinet/ip.h>
#endif /* !__PARAGON__ */
#include <arpa/inet.h>

#if defined(HAVE_SOCKADDR_IN6_STRUCT) && defined(WITH_IPV6)
/* Currently, we include the IPv6 code only if we have the
   `sockaddr_in6' structure. */
#define SSH_HAVE_IPV6
#ifdef IPV6_JOIN_GROUP
#define SSH_HAVE_IPV6_MULTICAST
#endif /* IPV6_JOIN_GROUP */
#endif /* HAVE_SOCKADDR_IN6_STRUCT && WITH_IPV6 */

/* Internal representation of Listener structure, not exported */
struct SshUdpListenerRec
{
  int sock;
  Boolean ipv6;
  SshUdpListener sibling;
  SshUdpCallback callback;
  void *context;
  Boolean connected;
  Boolean keep_nonblocking;
};

/* Internal function; converts ip address from string format to in_addr. */
static Boolean ssh_string_to_in_addr(const char *s, struct in_addr *addr)
{
  size_t four = 4;
  return ssh_inet_strtobin(s, (unsigned char *)&addr->s_addr, &four);
}

#ifdef SSH_HAVE_IPV6
/* Internal function; converts IPv6 address from string format to in_addr. */
static Boolean ssh_string_to_in6_addr(const char *s, struct in6_addr *addr)
{
  size_t size = 16;
  return ssh_inet_strtobin(s, (unsigned char *)&addr->s6_addr, &size);
}
#endif /* SSH_HAVE_IPV6 */

void ssh_socket_udp_io_cb(unsigned int events, void *context)
{
  SshUdpListener listener = (SshUdpListener)context;

  if (events & SSH_IO_READ)
    {
      /* Call the callback to inform about a received packet or
         notification. */
      if (listener->callback)
        (*listener->callback)(listener, listener->context);
    }
}

/* Set the common (both IPv4 and IPv6) socket options for the UDP
   listener `listener'. */

static void ssh_udp_set_common_socket_options(SshUdpListener listener)
{
#ifdef SO_REUSEADDR
  {
    int value;

    value = 1;
    if (setsockopt(listener->sock, SOL_SOCKET, SO_REUSEADDR, (void *) &value,
                   sizeof(value)) == -1)
      {
        SSH_DEBUG(SSH_D_FAIL,
                  ("ssh_udp_set_common_socket_options: setsockopt " \
                   "SO_REUSEADDR failed: %s", strerror(errno)));
      }
  }
#endif /* SO_REUSEADDR */
#ifdef SO_REUSEPORT
  {
    int value;

    value = 1;
    if (setsockopt(listener->sock, SOL_SOCKET, SO_REUSEPORT, (void *) &value,
                   sizeof(value)) == -1)
      {
        SSH_DEBUG(SSH_D_FAIL,
                  ("ssh_udp_set_common_socket_options: setsockopt " \
                   "SO_REUSEPORT failed: %s", strerror(errno)));
      }
  }
#endif /* SO_REUSEPORT */
}

/* Set more common (both IPv4 and IPv6) socket options for the UDP
   listener `listener'.  These are set after the socket is bound to IP
   addresses. */

static void ssh_udp_set_more_common_socket_options(SshUdpListener listener)
{
#ifdef SO_SNDBUF
  {
    int buf_len;

    buf_len = 65535;
    if (setsockopt(listener->sock, SOL_SOCKET, SO_SNDBUF, &buf_len,
                   sizeof(int)) == -1)
      {
        SSH_DEBUG(2, ("ssh_udp_set_more_common_socket_options: " \
                      "setsockopt SO_SNDBUF failed: %s", strerror(errno)));
      }
  }
#endif /* SO_SNDBUF */

#ifdef SO_RCVBUF
  {
    int buf_len;

    buf_len = 65535;
    if (setsockopt(listener->sock, SOL_SOCKET, SO_RCVBUF, &buf_len,
                   sizeof(int)) == -1)
      {
        SSH_DEBUG(2, ("ssh_udp_set_more_common_socket_options: " \
                      "setsockopt SO_RCVBUF failed: %s", strerror(errno)));
      }
  }
#endif /* SO_RCVBUF */
}

/* Creates an IPv4 UDP listener. */

static SshUdpListener ssh_udp_make_ip4_listener(const char *local_address,
                                                const char *local_port,
                                                const char *remote_address,
                                                const char *remote_port,
                                                SshUdpCallback callback,
                                                void *context)
{
  SshUdpListener listener;
  struct sockaddr_in sinaddr;
  int ret, port;

  /* Allocate and initialize the listener context. */
  listener = ssh_xcalloc(1, sizeof(*listener));
  listener->ipv6 = FALSE;
  listener->context = context;
  listener->callback = callback;
  listener->connected = FALSE;
  listener->keep_nonblocking = FALSE;

  /* Create the socket. */
  listener->sock = socket(AF_INET, SOCK_DGRAM, 0);
  if (listener->sock == -1)
    {
      ssh_xfree(listener);
      return NULL;
    }

  /* Set the common socket options. */
  ssh_udp_set_common_socket_options(listener);

  if (local_address != NULL || local_port != NULL)
    {
      /* Initialize the address structure for the local address. */
      memset(&sinaddr, 0, sizeof(sinaddr));
      sinaddr.sin_family = AF_INET;

      if (local_port != NULL)
        {
          /* Look up the service name for the local port. */
          port = ssh_tcp_get_port_by_service(local_port, "udp");
          if (port == -1)
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
          sinaddr.sin_port = htons(port);
        }

      if (local_address != NULL && !SSH_IS_IPADDR_ANY(local_address))
        {
          /* Decode the IP address.  Host names are not accepted. */
          if (!ssh_string_to_in_addr(local_address, &sinaddr.sin_addr))
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
        }
      ret = bind(listener->sock, (struct sockaddr *)&sinaddr,
                 sizeof(sinaddr));
      if (ret == -1)
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_make_ip4_listener: " \
                                 " bind failed: %s", strerror(errno)));
          close(listener->sock);
          ssh_xfree(listener);
          return NULL;
        }
    }

  if (remote_address != NULL || remote_port != NULL)
    {
      /* Initialize the address structure for the remote address. */
      memset(&sinaddr, 0, sizeof(sinaddr));
      sinaddr.sin_family = AF_INET;

      if (remote_port != NULL)
        {
          /* Look up the service name for the remote port. */
          port = ssh_tcp_get_port_by_service(remote_port, "udp");
          if (port == -1)
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
          sinaddr.sin_port = htons(port);
        }

      if (remote_address != NULL)
        {
          /* Decode the IP address.  Host names are not accepted. */
          if (!ssh_string_to_in_addr(remote_address, &sinaddr.sin_addr))
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
        }

      /* Mark the socket to be connected */
      listener->connected = TRUE;

      /* Connect the socket, so that we will receive unreachable
         notifications. */
      ret = connect(listener->sock, (struct sockaddr *)&sinaddr,
                    sizeof(sinaddr));
      if (ret == -1)
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_make_ip4_listener: connect failed: "\
                                 " %s", strerror(errno)));
          close(listener->sock);
          ssh_xfree(listener);
          return NULL;
        }
    }

  /* Set more common UDP socket options. */
  ssh_udp_set_more_common_socket_options(listener);

  /* Socket creation succeeded. Do the event loop stuff */
  ssh_io_register_fd(listener->sock, ssh_socket_udp_io_cb, (void *)listener);
  ssh_io_set_fd_request(listener->sock, callback ? SSH_IO_READ : 0);

  return listener;
}

#ifdef SSH_HAVE_IPV6
/* Creates an IPv6 UDP listener. */

static SshUdpListener ssh_udp_make_ip6_listener(const char *local_address,
                                                const char *local_port,
                                                const char *remote_address,
                                                const char *remote_port,
                                                SshUdpCallback callback,
                                                void *context)
{
  SshUdpListener listener;
  struct sockaddr_in6 sinaddr;
  int ret, port;

  /* Allocate and initialize the listener context. */
  listener = ssh_xcalloc(1, sizeof(*listener));
  listener->ipv6 = TRUE;
  listener->context = context;
  listener->callback = callback;
  listener->connected = FALSE;
  listener->keep_nonblocking = FALSE;

  /* Create the socket. */
  listener->sock = socket(AF_INET6, SOCK_DGRAM, 0);
  if (listener->sock == -1)
    {
      ssh_xfree(listener);
      return NULL;
    }

  /* Set the common socket options. */
  ssh_udp_set_common_socket_options(listener);

  if (local_address != NULL || local_port != NULL)
    {
      /* Initialize the address structure for the local address. */
      memset(&sinaddr, 0, sizeof(sinaddr));
      sinaddr.sin6_family = AF_INET6;

      if (local_port != NULL)
        {
          /* Look up the service name for the local port. */
          port = ssh_tcp_get_port_by_service(local_port, "udp");
          if (port == -1)
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
          sinaddr.sin6_port = htons(port);
        }

      if (local_address != NULL && !SSH_IS_IPADDR_ANY(local_address))
        {
          /* Decode the IP address.  Host names are not accepted. */
          if (!ssh_string_to_in6_addr(local_address, &sinaddr.sin6_addr))
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
        }
      ret = bind(listener->sock, (struct sockaddr *)&sinaddr,
                 sizeof(sinaddr));
      if (ret == -1)
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_make_ip6_listener: bind failed: %s",
                                 strerror(errno)));
          close(listener->sock);
          ssh_xfree(listener);
          return NULL;
        }
    }

  if (remote_address != NULL || remote_port != NULL)
    {
      /* Initialize the address structure for the remote address. */
      memset(&sinaddr, 0, sizeof(sinaddr));
      sinaddr.sin6_family = AF_INET6;

      if (remote_port != NULL)
        {
          /* Look up the service name for the remote port. */
          port = ssh_tcp_get_port_by_service(remote_port, "udp");
          if (port == -1)
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
          sinaddr.sin6_port = htons(port);
        }

      if (remote_address != NULL)
        {
          /* Decode the IP address.  Host names are not accepted. */
          if (!ssh_string_to_in6_addr(remote_address, &sinaddr.sin6_addr))
            {
              close(listener->sock);
              ssh_xfree(listener);
              return NULL;
            }
        }

      /* Mark the socket to be connected */
      listener->connected = TRUE;

      /* Connect the socket, so that we will receive unreachable
         notifications. */
      ret = connect(listener->sock, (struct sockaddr *)&sinaddr,
                    sizeof(sinaddr));
      if (ret == -1)
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_make_ip6_listener: connect failed: "\
                                 "%s", strerror(errno)));
          close(listener->sock);
          ssh_xfree(listener);
          return NULL;
        }
    }

  /* Set more common UDP socket options. */
  ssh_udp_set_more_common_socket_options(listener);

  /* Socket creation succeeded. Do the event loop stuff */
  ssh_io_register_fd(listener->sock, ssh_socket_udp_io_cb, (void *)listener);
  ssh_io_set_fd_request(listener->sock, callback ? SSH_IO_READ : 0);

  return listener;
}
#endif /* SSH_HAVE_IPV6 */

/* Creates a listener for sending and receiving UDP packets.  The listener is
   connected if remote_address is non-NULL.  Connected listeners may receive
   notifications about the destination host/port being unreachable.
     local_address    local address for sending; SSH_IPADDR_ANY chooses
                      automatically
     local_port       local port for receiving udp packets
     remote_address   specifies the remote address for this listener
                      is non-NULL.  If specified, unreachable notifications
                      may be received for packets sent to the address.
     remote_port      remote port for packets sent using this listener, or NULL
     callback         function to call when packet or notification available
     context          argument to pass to the callback. */

SshUdpListener ssh_udp_make_listener(const char *local_address,
                                     const char *local_port,
                                     const char *remote_address,
                                     const char *remote_port,
                                     SshUdpCallback callback,
                                     void *context)
{
  SshUdpListener listener4 = NULL;
#ifdef SSH_HAVE_IPV6
  SshUdpListener listener6 = NULL;
#endif /* SSH_HAVE_IPV6 */

  SSH_DEBUG(SSH_D_HIGHSTART, ("Making UDP listener"));

  /* Let's determine the type of listener to create. */
  if (local_address && !SSH_IS_IPADDR_ANY(local_address))
    {
      SshIpAddrStruct ipaddr;

      /* We are creating only an IPv4 or an IPv6 listener. */
      if (!ssh_ipaddr_parse(&ipaddr, local_address))
        /* Malformed address. */
        return NULL;

      if (SSH_IP_IS4(&ipaddr))
        {
          SSH_DEBUG(SSH_D_HIGHSTART,
                    ("Making IPv4 only UDP listener for address %@",
                     ssh_ipaddr_render, &ipaddr));
          return ssh_udp_make_ip4_listener(local_address, local_port,
                                           remote_address, remote_port,
                                           callback, context);
        }
      else
        {
#ifdef SSH_HAVE_IPV6
          SSH_DEBUG(SSH_D_HIGHSTART,
                    ("Making IPv6 only UDP listener for address %@",
                     ssh_ipaddr_render, &ipaddr));
          return ssh_udp_make_ip6_listener(local_address, local_port,
                                           remote_address, remote_port,
                                           callback, context);
#else /* not  SSH_HAVE_IPV6 */
          SSH_DEBUG(SSH_D_HIGHSTART,
                    ("IPv6 is not supported on this platform"));
          return NULL;
#endif /* not SSH_HAVE_IPV6 */
        }
    }

  /* Create a dual listener for both IPv4 and IPv6. */
  SSH_DEBUG(SSH_D_HIGHSTART, ("Making IPv4 and IPv6 UDP listeners"));

  listener4 = ssh_udp_make_ip4_listener(local_address, local_port,
                                        remote_address, remote_port,
                                        callback, context);
  if (listener4 == NULL)
    return NULL;

#ifdef SSH_HAVE_IPV6
  /* Try to create an IPv6 listener.  It is ok if this fails since
     there seems to be systems which do not support IPv6 although they
     know the in6 structures. */
  listener6 = ssh_udp_make_ip6_listener(local_address, local_port,
                                        remote_address, remote_port,
                                        callback, context);
  if (listener6 != NULL)
    {
      /* We managed to make them both. */
      listener4->sibling = listener6;
    }
#endif /* SSH_HAVE_IPV6 */

  return listener4;
}

/* Destroys the udp listener. */

void ssh_udp_destroy_listener(SshUdpListener listener)
{
  if (listener->sibling)
    ssh_udp_destroy_listener(listener->sibling);

  ssh_io_unregister_fd(listener->sock, listener->keep_nonblocking);
  close(listener->sock);
  ssh_xfree(listener);
}

/* Ask for permission to send broadcast packets */
void ssh_udp_set_broadcasting(SshUdpListener listener, Boolean allow)
{
  int option;

  if (allow)
    option = 1;
  else
    option = 0;

#ifdef SO_BROADCAST
  while (listener)
    {
      if (setsockopt(listener->sock, SOL_SOCKET, SO_BROADCAST, &option,
                     sizeof(int)) == -1)
        {
          SSH_DEBUG(SSH_D_FAIL,
                    ("ssh_udp_set_broadcasting: setsockopt SO_BROADCAST "
                     "failed: %s", strerror(errno)));
        }
      listener = listener->sibling;
    }
#endif /* SO_BROADCAST */
}

/* Add membership to given multicast group */
SshUdpError ssh_udp_multicast_add_membership(SshUdpListener listener,
                                      const char *group_to_join,
                                      const char *interface_to_join)
{
  for(; listener; listener = listener->sibling)
    {
#ifdef SSH_HAVE_IPV6_MULTICAST
      if (listener->ipv6)
        {
          struct ipv6_mreq mreq6;
          size_t size = 16;

          memset(&mreq6, 0, sizeof(mreq6));

          if (!ssh_inet_strtobin(group_to_join,
                                 (unsigned char *)&(mreq6.ipv6mr_multiaddr.
                                                    s6_addr), &size) ||
              size != 16)
            continue;
          if (interface_to_join && !SSH_IS_IPADDR_ANY(interface_to_join))
            {
              /* XXX, add this later */
            }
          if (!setsockopt(listener->sock,
                          IPPROTO_IPV6,
                          IPV6_JOIN_GROUP,
                          &mreq6,
                          sizeof(mreq6)))
            {
              return SSH_UDP_OK;
            }
        }
      else
#endif /* SSH_HAVE_IPV6_MUILTICAST */
        {
#ifdef IP_ADD_MEMBERSHIP
          struct ip_mreq mreq;
          size_t size = 4;

          memset(&mreq, 0, sizeof(mreq));

          if (!ssh_inet_strtobin(group_to_join,
                                 (unsigned char *)&(mreq.imr_multiaddr.
                                                    s_addr), &size) ||
              size != 4)
            continue;

          if (interface_to_join && !SSH_IS_IPADDR_ANY(interface_to_join))
            {
              if (!ssh_inet_strtobin(interface_to_join,
                                     (unsigned char *)&(mreq.imr_interface.
                                                        s_addr), &size) ||
                  size != 4)
                continue;
            }
          if (!setsockopt(listener->sock,
                          IPPROTO_IP,
                          IP_ADD_MEMBERSHIP,
                          &mreq,
                          sizeof(mreq)))
            {
              return SSH_UDP_OK;
            }
#else /* IP_ADD_MEMBERSHIP */
          continue;
#endif /* IP_ADD_MEMBERSHIP */
        }
    }
  return SSH_UDP_INVALID_ARGUMENTS;
}

/* Drop membership to given multicast group */
SshUdpError ssh_udp_multicast_drop_membership(SshUdpListener listener,
                                              const char *group_to_drop,
                                              const char *interface_to_drop)
{
  for(; listener; listener = listener->sibling)
    {
#ifdef SSH_HAVE_IPV6_MULTICAST
      if (listener->ipv6)
        {
          struct ipv6_mreq mreq6;
          size_t size = 16;

          memset(&mreq6, 0, sizeof(mreq6));

          if (!ssh_inet_strtobin(group_to_drop,
                                 (unsigned char *)&(mreq6.ipv6mr_multiaddr.
                                                    s6_addr), &size) ||
              size != 16)
            continue;
          if (interface_to_drop && !SSH_IS_IPADDR_ANY(interface_to_drop))
            {
              /* XXX, add this later */
            }
          setsockopt(listener->sock,
                     IPPROTO_IPV6,
                     IPV6_LEAVE_GROUP,
                     &mreq6,
                     sizeof(mreq6));
        }
      else
#endif /* SSH_HAVE_IPV6_MULTICAST */
        {
#ifdef IP_DROP_MEMBERSHIP
          struct ip_mreq mreq;
          size_t size = 4;

          memset(&mreq, 0, sizeof(mreq));

          if (!ssh_inet_strtobin(group_to_drop,
                                 (unsigned char *)&(mreq.imr_multiaddr.
                                                    s_addr), &size) ||
              size != 4)
            continue;

          if (interface_to_drop && !SSH_IS_IPADDR_ANY(interface_to_drop))
            {
              if (!ssh_inet_strtobin(interface_to_drop,
                                     (unsigned char *)&(mreq.imr_interface.
                                                        s_addr), &size) ||
                  size != 4)
                continue;
            }
          setsockopt(listener->sock,
                     IPPROTO_IP,
                     IP_DROP_MEMBERSHIP,
                     &mreq,
                     sizeof(mreq));
#else /* IP_DROP_MEMBERSHIP */
          continue;
#endif /* IP_DROP_MEMBERSHIP */
        }
    }
  return SSH_UDP_OK;
}

/* Set multicast hop count limit (ttl). This affects when sending multicast
   traffic from the socket. You don't need to be part of the multicast
   group to send packets to it. */
SshUdpError ssh_udp_multicast_hops(SshUdpListener listener, int hop_limit)
{
  /* XXX */
  ssh_fatal("Not yet implemented XXX!");
  return SSH_UDP_OK;
}

/* Enable/disable multicast looping in the local host. If this is enabled then
   multicast sent from this socket is looped back inside the machine to all
   sockets that are member of the multicast group. Normally this is enabled,
   which means that all processes (including you) in this host that are part of
   the group can also hear your tranmissions. If you are sure that you are only
   member in this host you can disable this saving you time to process your own
   multicast packets. */
SshUdpError ssh_udp_multicast_loopback(SshUdpListener listener,
                                       Boolean loopback_enabled)
{
  /* XXX */
  ssh_fatal("Not yet implemented XXX!");
  return SSH_UDP_OK;
}

/* Select outbound interface for the multicast traffic. The interface_to_use is
   the ip address of the interface to use or SSH_IPADDR_ANY meaning that
   routing table is used to select suitable interface to where the multicast
   traffic is sent out. The default is SSH_IPADDR_ANY. */
SshUdpError ssh_udp_multicast_interface(SshUdpListener listener,
                                        const char *interface_to_use)
{
  /* XXX */
  ssh_fatal("Not yet implemented XXX!");
  return SSH_UDP_OK;
}


/* Reads the received packet or notification from the listener.  This
   function should be called from the listener callback.  This can be
   called multiple times from a callback; each call will read one more
   packet or notification from the listener until no more are
   available. */

SshUdpError ssh_udp_read(SshUdpListener listener,
                         char *remote_address, size_t remote_address_len,
                         char *remote_port, size_t remote_port_len,
                         unsigned char *datagram_buffer,
                         size_t datagram_buffer_len,
                         size_t *datagram_len_return)
{
  size_t ret;
  struct sockaddr_in from_addr4;
#ifdef SSH_HAVE_IPV6
  struct sockaddr_in6 from_addr6;
#endif /* SSH_HAVE_IPV6 */
  struct sockaddr *from_addr = NULL;
  int port = 0;
#ifdef HAVE_POSIX_STYLE_SOCKET_PROTOTYPES
  size_t fromlen = 0L, fromlen_min = 0L;
#else /* HAVE_POSIX_STYLE_SOCKET_PROTOTYPES */
  int fromlen = 0, fromlen_min = 0;
#endif /* HAVE_POSIX_STYLE_SOCKET_PROTOTYPES */
  SshIpAddrStruct ipaddr;

  if (datagram_len_return)
    *datagram_len_return = 0;

#ifdef SSH_HAVE_IPV6
  if (listener->ipv6)
    {
      from_addr = (struct sockaddr *) &from_addr6;
      fromlen = sizeof(from_addr6);
      fromlen_min = sizeof(from_addr6);
    }
#endif /* SSH_HAVE_IPV6 */
  if (!listener->ipv6)
    {
      from_addr = (struct sockaddr *) &from_addr4;
      fromlen = sizeof(from_addr4);
      fromlen_min = sizeof(from_addr4);
    }

  ret = recvfrom(listener->sock, (void *)datagram_buffer, datagram_buffer_len,
                 0, from_addr, &fromlen);
  if (ret == (size_t)-1)
    {
      SSH_DEBUG(SSH_D_UNCOMMON, ("Read result %ld, error = %s", (long)ret,
                                 strerror(errno)));
      switch (errno)
        {
#ifdef EHOSTDOWN
        case EHOSTDOWN:
#endif /* EHOSTDOWN */
#ifdef EHOSTUNREACH
        case EHOSTUNREACH:
#endif /* EHOSTUNREACH */
          return SSH_UDP_HOST_UNREACHABLE;

#ifdef ECONNREFUSED
        case ECONNREFUSED:
#endif /* ECONNREFUSED */
#ifdef ENOPROTOOPT
        case ENOPROTOOPT:
#endif /* ENOPROTOOPT */
          return SSH_UDP_PORT_UNREACHABLE;
        default:
          return SSH_UDP_NO_DATA;
        }
    }
  SSH_DEBUG(SSH_D_NICETOKNOW, ("Read got %ld bytes", (long)ret));

  /* XXX __linux__ IPv6:

  Issue:
   - stupid glibc _and_ kernel _both_ have their own sockaddr_in6.

   Funnily enough, the one in glibc is bigger _and_ used by us,
   _but_ the kernel returns shorter one.

   Therefore, we cannot do size checks although stuff we're interested
   in is in the beginning and available in both.. hopefully. */


#ifndef __linux__
  if (fromlen >= fromlen_min)
#endif /* __linux__ */
    {
      /* Format port number in user buffer. */
      if (remote_port != NULL)
        {
#ifdef SSH_HAVE_IPV6
          if (listener->ipv6)
            port = ntohs(from_addr6.sin6_port);
#endif /* SSH_HAVE_IPV6 */
          if (!listener->ipv6)
            port = ntohs(from_addr4.sin_port);
          ssh_snprintf(remote_port, remote_port_len, "%d", port);
        }

      /* Format source address in user buffer. */
      if (remote_address != NULL)
        {
#ifdef SSH_HAVE_IPV6
          if (listener->ipv6)
            {
              SSH_IP6_DECODE(&ipaddr, &from_addr6.sin6_addr.s6_addr);

#ifdef __linux__
              {
                Boolean is_ipv4_mapped_address = TRUE;
                int i;

                /* IPv6 allows for mapping of ipv4 addresses
                   directly to IPv6 scope. For IKE purposes the
                   addresses are _not_ the same, and therefore
                   for now we simply change the addresses to IPv4
                   when they are really IPv4 - that is, match
                   mask

                   ::FFFF:0:0/96
                */
                for (i = 0 ; i < 10 ; i++)
                  if (ipaddr.addr_data[i])
                    {
                      is_ipv4_mapped_address = FALSE;
                      break;
                    }
                for ( ; i < 11 ; i++)
                  if (ipaddr.addr_data[i] != 0xff)
                    {
                      is_ipv4_mapped_address = FALSE;
                      break;
                    }
                if (is_ipv4_mapped_address)
                  SSH_IP4_DECODE(&ipaddr,
                                 &ipaddr.addr_data[12]);
              }
#endif /* __linux__ */
            }
#endif /* SSH_HAVE_IPV6 */
          if (!listener->ipv6)
            SSH_IP4_DECODE(&ipaddr, &from_addr4.sin_addr.s_addr);

          ssh_ipaddr_print(&ipaddr, remote_address, remote_address_len);
        }
    }

  /* Return the length of the received packet. */
  if (datagram_len_return)
    *datagram_len_return = ret;

  return SSH_UDP_OK;
}

/* This sends udp datagram to remote destination. This call always success, or
   the if not then datagram is silently dropped (udp is not reliable anyways */

void ssh_udp_send(SshUdpListener listener,
                  const char *remote_address, const char *remote_port,
                  const unsigned char *datagram_buffer, size_t datagram_len)
{
  struct sockaddr *to_addr;
#ifdef HAVE_POSIX_STYLE_SOCKET_PROTOTYPES
  size_t to_addr_len;
#else /* HAVE_POSIX_STYLE_SOCKET_PROTOTYPES */
  int to_addr_len;
#endif /* HAVE_POSIX_STYLE_SOCKET_PROTOTYPES */
  int port = 0;
  SshIpAddrStruct ipaddr;

  SSH_DEBUG(SSH_D_NICETOKNOW, ("Send %ld bytes", (long)datagram_len));

  if (listener->connected)
    {
      if (remote_port != NULL)
        SSH_DEBUG(SSH_D_FAIL,
                  ("ssh_udp_send: Remote port number `%s' specified for "
                   "connected socket, ignored", remote_port));
      if (remote_address != NULL)
        SSH_DEBUG(SSH_D_FAIL,
                  ("ssh_udp_send: Remote address `%s' specified for "
                   "connected socket, ignored", remote_address));

      /* Send the packet to the connected socket. */
      if (send(listener->sock, (void *)datagram_buffer, datagram_len,
               0) == -1)
        SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: send failed: %s",
                               strerror(errno)));
      return;
    }

  /* Decode the port number if given. */
  if (remote_port != NULL)
    {
      port = ssh_tcp_get_port_by_service(remote_port, "udp");
      if (port == -1)
        {
          SSH_DEBUG(2, ("ssh_udp_send: bad port %s", remote_port));
          return;
        }
    }

  /* Decode the destination address if given. */
  memset(&ipaddr, 0, sizeof(ipaddr));
  if (remote_address != NULL)
    {
      /* First check if it is already an ip address. */
      if (!ssh_ipaddr_parse(&ipaddr, remote_address))
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: bad address %s",
                                 remote_address));
          return;
        }
    }

  if (SSH_IP_IS6(&ipaddr))
    {
      /* IPv6 addresses. */
#ifdef SSH_HAVE_IPV6
      struct sockaddr_in6 to_addr6;

      /* Do we have an IPv6 listener? */
      if (listener->ipv6)
        ;
      else if (listener->sibling && listener->sibling->ipv6)
        listener = listener->sibling;
      else
        {
          /* We do not have it. */
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: no IPv4 listener"));
          return;
        }

      memset(&to_addr6, 0, sizeof(to_addr6));
      to_addr6.sin6_family = AF_INET6;
      to_addr6.sin6_port = htons(port);

      if (remote_address
          && !ssh_string_to_in6_addr(remote_address, &to_addr6.sin6_addr))
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: bad address %s",
                                 remote_address));
          return;
        }

      to_addr = (struct sockaddr *) &to_addr6;
      to_addr_len = sizeof(to_addr6);

#else /* not SSH_HAVE_IPV6 */
      SSH_DEBUG(SSH_D_FAIL, ("IPv6 is not supported on this platform"));
      return;
#endif /* SSH_HAVE_IPV6 */
    }
  else
    {
      /* IPv4 and unspecified remote address cases. */
      struct sockaddr_in to_addr4;

      memset(&to_addr4, 0, sizeof(to_addr));
      to_addr4.sin_family = AF_INET;
      to_addr4.sin_port = htons(port);

      if (remote_address
          &&  !ssh_string_to_in_addr(remote_address, &to_addr4.sin_addr))
        {
          SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: bad address %s",
                                 remote_address));
          return;
        }

      to_addr = (struct sockaddr *) &to_addr4;
      to_addr_len = sizeof(to_addr4);
    }

  /* Send the packet. */
  if (sendto(listener->sock, (void *)datagram_buffer, datagram_len, 0,
             to_addr, to_addr_len) == -1)
    SSH_DEBUG(SSH_D_FAIL, ("ssh_udp_send: sendto failed: %s",
                           strerror(errno)));
}

void ssh_udp_listener_mark_forked(SshUdpListener listener)
{
  SSH_PRECOND(listener);

  listener->keep_nonblocking = TRUE;
  if (listener->sibling)
    listener->sibling->keep_nonblocking = TRUE;
}
