/*

sshencode.c

Author: Tero Kivinen <kivinen@ssh.fi>
        Tatu Ylonen <ylo@ssh.fi>

Copyright (c) 1997, 2001 SSH Communications Security, Finland
                   All rights reserved

Functions for encoding/decoding binary data.

*/

#include "sshincludes.h"
#include "sshbuffer.h"
#include "sshgetput.h"
#include "sshencode.h"

#define SSH_DEBUG_MODULE "SshEncode"

static Boolean do_expand(SshBuffer b, size_t r)
{
  void *tmp;

  if (!b->buf || ((b->end + r) > b->alloc))
    {
      if (b->dynamic)
        {
          if (!b->buf)
            {
              b->alloc = b->end + r;
              b->buf = ssh_malloc(b->alloc);
              return (b->buf != NULL);
            }
          else
            {
              size_t nsize = b->end + r + 512;
              SSH_DEBUG(20, ("Re-allocating buffer 0x%lx. Old size = %d, new "
                             "size = %d.", b, b->alloc, nsize));
              tmp = ssh_realloc(b->buf, b->alloc, nsize);
              if (tmp)
                {
                  b->buf = tmp;
                  b->alloc = nsize;
                  return TRUE;
                }
              else
                {
                  ssh_free(b->buf);
                  b->buf = NULL;
                  return FALSE;
                }
            }
        }
      else
        return FALSE;
    }
  else
    return TRUE;
}

#define EXPAND(b, r)            \
 do {                           \
  if (do_expand(b, r) == 0)     \
    return 0;                   \
 } while (0)

static size_t
encode_buffer_va_internal(SshBuffer buffer, va_list ap)
{
  SshEncodingFormat format;
  unsigned int intvalue;
  SshUInt64 u64;
  SshUInt32 u32;
  size_t i, orig_offset;
  Boolean b;
  const unsigned char *p;
  SshEncodeDatum fn;
  void *datum;

  /* If the buffer is empty, start using it from the beginning. */
  if (buffer->offset == buffer->end)
    {
      buffer->offset = 0;
      buffer->end = 0;
    }

  orig_offset = buffer->end;
  for (;;)
    {
      format = va_arg(ap, SshEncodingFormat);
      switch (format)
        {
        case SSH_FORMAT_UINT32_STR:
          p = va_arg(ap, unsigned char *);
          i = va_arg(ap, size_t);
          EXPAND(buffer, 4 + i);

          SSH_PUT_32BIT(buffer->buf + buffer->end, i);
          memcpy(buffer->buf + buffer->end + 4, p, i);
          buffer->end += 4 + i;
          break;

        case SSH_FORMAT_DATA:
          p = va_arg(ap, unsigned char *);
          i = va_arg(ap, size_t);
          EXPAND(buffer, i);

          memcpy(buffer->buf + buffer->end, p, i);
          buffer->end += i;
          break;

        case SSH_FORMAT_BOOLEAN:
          b = va_arg(ap, Boolean);
          EXPAND(buffer, 1);

          buffer->buf[buffer->end++] = (unsigned char)((b == 0) ? 0 : 1);
          break;

        case SSH_FORMAT_CHAR:
          intvalue = va_arg(ap, unsigned int);
          EXPAND(buffer, 1);

          buffer->buf[buffer->end++] = (unsigned char)intvalue;
          break;

        case SSH_FORMAT_UINT32:
          u32 = va_arg(ap, SshUInt32);
          EXPAND(buffer, 4);

          SSH_PUT_32BIT(buffer->buf + buffer->end, u32);
          buffer->end += 4;
          break;

        case SSH_FORMAT_UINT64:
          u64 = va_arg(ap, SshUInt64);
          EXPAND(buffer, 8);

          SSH_PUT_64BIT(buffer->buf + buffer->end, u64);
          buffer->end += 8;
          break;

        case SSH_FORMAT_SPECIAL:
          fn = va_arg(ap, SshEncodeDatum);
          datum = va_arg(ap, void *);

        retry:
          i = (*fn)(buffer->buf + buffer->end,
                    buffer->alloc - buffer->end,
                    datum);
          if (i > buffer->alloc - buffer->end)
            {
              /* If it obeys ssh_snprintf renderer semantics returning
                 1 + len in case of not fitting, optimize and expand
                 by 128 instead. */

              if (i == 1 + buffer->alloc - buffer->end)
                i = 128;

              EXPAND(buffer, i); /* expand it enough, i - alloc - end
                                    would do here. */
              goto retry;
            }

          buffer->end += i;
          break;

        case SSH_FORMAT_END:
          /* Return the number of bytes added. */
          return buffer->end - orig_offset;

        default:
          ssh_free(buffer->buf);
          buffer->buf = NULL;
          return 0;
        }
    }
}

size_t ssh_encode_buffer_va(SshBuffer buffer, va_list ap)
{
  size_t bytes;
  Boolean dynamic;

  dynamic = buffer->dynamic;
  buffer->dynamic = TRUE;
  bytes = encode_buffer_va_internal(buffer, ap);
  buffer->dynamic = dynamic;

  return bytes;
}

size_t ssh_encode_buffer(SshBuffer buffer, ...)
{
  size_t bytes;
  Boolean dynamic;
  va_list ap;

  dynamic = buffer->dynamic;
  buffer->dynamic = TRUE;

  va_start(ap, buffer);
  bytes = encode_buffer_va_internal(buffer, ap);
  va_end(ap);

  buffer->dynamic = dynamic;
  return bytes;
}

size_t ssh_encode_array_va(unsigned char *buf, size_t buflen, va_list ap)
{
  SshBufferStruct buffer;

  buffer.buf = buf;
  buffer.alloc = buflen;
  buffer.end = buffer.offset = 0;
  buffer.dynamic = FALSE;

  return encode_buffer_va_internal(&buffer, ap);
}

size_t ssh_encode_array(unsigned char *buf, size_t buflen, ...)
{
  va_list ap;
  size_t bytes;

  va_start(ap, buflen);
  bytes = ssh_encode_array_va(buf, buflen, ap);
  va_end(ap);

  return bytes;
}

size_t ssh_encode_array_alloc(unsigned char **buf_return, ...)
{
  size_t bytes;
  SshBufferStruct buffer;
  va_list ap;

  buffer.alloc = 512;
  buffer.buf = ssh_malloc(buffer.alloc);
  buffer.end = buffer.offset = 0;
  buffer.dynamic = TRUE;

  va_start(ap, buf_return);
  bytes = encode_buffer_va_internal(&buffer, ap);
  va_end(ap);

  if (buf_return != NULL)
    {
      *buf_return = buffer.buf;
    }
  return bytes;
}

size_t ssh_encode_array_alloc_va(unsigned char **buf_return, va_list ap)
{
  size_t bytes;
  SshBufferStruct buffer;

  buffer.alloc = 512;
  buffer.buf = ssh_malloc(buffer.alloc);
  buffer.end = buffer.offset = 0;
  buffer.dynamic = TRUE;

  bytes = encode_buffer_va_internal(&buffer, ap);

  if (buf_return != NULL)
    {
      *buf_return = buffer.buf;
    }
  return bytes;
}

/* Allocates a buffer of the given size with ssh_malloc and record it
   into allocsp[num_allocs_p], possibly expanding array allocsp to do
   this. Advance num_allocs_p.

   Return NULL in case of memory allocation failure, in which case
   free the array at allocsp as well. */

static unsigned char *
decode_alloc(unsigned int *num_allocs_p, unsigned char ***allocsp,
             size_t size)
{
  unsigned char *p, **tmpa;

  /* Check if we need to enlarge the pointer array.  We enlarge it in chunks
     of 16 pointers. */
  if (*num_allocs_p == 0)
    {
      if ((tmpa = ssh_malloc(16 * sizeof(unsigned char *))) == NULL)
        goto failure;
      else
        *allocsp = tmpa;
    }
  else
    {
      if (*num_allocs_p % 16 == 0)
        {
          if ((tmpa = ssh_realloc(*allocsp,
                                  *num_allocs_p,
                                  (*num_allocs_p + 16) *
                                  sizeof(unsigned char *)))
              == NULL)
            goto failure;
          else
            *allocsp = tmpa;
        }
    }

  if ((p = ssh_malloc(size)) == NULL)
    goto failure;

  /* Store it in the array. */
  (*allocsp)[*num_allocs_p] = p;
  (*num_allocs_p)++;
  return p;

 failure:
  if (*num_allocs_p)
    ssh_free(*allocsp);
  *allocsp = NULL;

  return NULL;
}

size_t ssh_decode_array_va(const unsigned char *buf, size_t len, va_list ap)
{
  SshEncodingFormat format;
  unsigned long longvalue;
  SshUInt64 *u64p;
  SshUInt32 *u32p;
  Boolean *bp;
  size_t size, *sizep;
  unsigned int *uip;
  unsigned char *p, **pp;
  const unsigned char **cpp;
  size_t offset;
  unsigned int i, num_allocs;
  unsigned char **allocs;
  SshDecodeDatum fn;
  void **datump;

  offset = 0;
  num_allocs = 0;

  for (;;)
    {
      /* Get the next format code. */
      format = va_arg(ap, SshEncodingFormat);

      switch (format)
        {
        case SSH_FORMAT_UINT32_STR:
          pp = va_arg(ap, unsigned char **);
          sizep = va_arg(ap, size_t *);

          /* Check if the buffer can fit the 32 bit string length,
             and grab it.*/
          if (len - offset < 4)
            goto fail;
          longvalue = SSH_GET_32BIT(buf + offset);
          offset += 4;

          /* Check if the buffer can fit the string data and get it. */
          if (longvalue > len - offset)
            goto fail;

          /* Store length if requested. */
          if (sizep != NULL)
            *sizep = longvalue;

          /* Retrieve the data if requested. */
          if (pp != NULL)
            {
              *pp = decode_alloc(&num_allocs, &allocs, (size_t)longvalue + 1);
              if (!*pp)
                  goto fail;
              memcpy(*pp, buf + offset, (size_t)longvalue);
              (*pp)[longvalue] = '\0';
            }

          /* Consume the data. */
          offset += longvalue;
          break;

        case SSH_FORMAT_UINT32_STR_NOCOPY:
          /* Get length and data pointers. */
          cpp = va_arg(ap, const unsigned char **);
          sizep = va_arg(ap, size_t *);

          /* Decode string length and skip the length. */
          if (len - offset < 4)
            goto fail;
          longvalue = SSH_GET_32BIT(buf + offset);
          offset += 4;

          /* Check that the string is all in the buffer. */
          if (longvalue > len - offset)
            goto fail;

          /* Store length and data if requested. */
          if (sizep != NULL)
            *sizep = longvalue;
          if (cpp != NULL)
            *cpp = buf + offset;

          /* Consume the data. */
          offset += longvalue;
          break;

        case SSH_FORMAT_DATA:
          p = va_arg(ap, unsigned char *);
          size = va_arg(ap, size_t);
          if (len - offset < size)
            goto fail;
          if (p)
            memcpy(p, buf + offset, size);
          offset += size;
          break;

        case SSH_FORMAT_BOOLEAN:
          bp = va_arg(ap, Boolean *);
          if (len - offset < 1)
            goto fail;
          if (bp != NULL)
            *bp = buf[offset] != 0;
          offset++;
          break;

        case SSH_FORMAT_CHAR:
          uip = va_arg(ap, unsigned int *);
          if (len - offset < 1)
            goto fail;
          if (uip)
            *uip = buf[offset];
          offset++;
          break;

        case SSH_FORMAT_UINT32:
          u32p = va_arg(ap, SshUInt32 *);
          if (len - offset < 4)
            goto fail;
          if (u32p)
            *u32p = SSH_GET_32BIT(buf + offset);
          offset += 4;
          break;

        case SSH_FORMAT_UINT64:
          u64p = va_arg(ap, SshUInt64 *);
          if (len - offset < 8)
            goto fail;
          if (u64p)
            *u64p = SSH_GET_64BIT(buf + offset);
          offset += 8;
          break;

        case SSH_FORMAT_SPECIAL:
          fn = va_arg(ap, SshDecodeDatum);
          datump = va_arg(ap, void **);

          size = (*fn)(buf + offset, len - offset, datump);
          if (size > len - offset)
            goto fail;
          /* XXX: nasty feature here. Actually encoding may fail after
             this tag has been processed, and now it is up the caller
             to free datump allocated by the decoder (as we do not
             know its structure. (as you see, strings allocated here
             are freed at failure handler but datums are not). */
          offset += size;
          break;

        case SSH_FORMAT_END:
          /* Free the allocs array. */
          if (num_allocs > 0)
            ssh_xfree(allocs);
          /* Return the number of bytes consumed. */
          return offset;

        default:
          SSH_DEBUG(SSH_D_ERROR, ("invalid format code %d", (int) format));
          return 0;
        }
    }
  /*NOTREACHED*/

 fail:
  /* An error was encountered.  Free all allocated memory and return zero. */
  for (i = 0; i < num_allocs; i++)
    ssh_xfree(allocs[i]);
  if (i > 0)
    ssh_xfree(allocs);
  return 0;
}

size_t ssh_decode_array(const unsigned char *buf, size_t len, ...)
{
  va_list ap;
  size_t bytes;

  va_start(ap, len);
  bytes = ssh_decode_array_va(buf, len, ap);
  va_end(ap);

  return bytes;
}

size_t ssh_decode_buffer_va(SshBuffer buffer, va_list ap)
{
  size_t bytes;

  bytes = ssh_decode_array_va(ssh_buffer_ptr(buffer),
                              ssh_buffer_len(buffer),
                              ap);

  ssh_buffer_consume(buffer, bytes);
  return bytes;
}

size_t ssh_decode_buffer(SshBuffer buffer, ...)
{
  va_list ap;
  size_t bytes;

  va_start(ap, buffer);
  bytes = ssh_decode_array_va(ssh_buffer_ptr(buffer),
                              ssh_buffer_len(buffer),
                              ap);
  va_end(ap);

  ssh_buffer_consume(buffer, bytes);
  return bytes;
}
