/*

  t-addr.c

  Author: Santeri Paavolainen <santtu@ssh.fi>


  Copyright (C) 1999 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

 */

#include "sshincludes.h"
#include "sshinet.h"
#include "sshtimemeasure.h"

#define SEQ(A,B) if (strcmp((A),(B))) { fprintf(stderr, "%s: [%d] %s <=> %s\n", av[0], __LINE__, (A), (B)); exit(1); } else if (verbose_flag) { fprintf(stderr, "%s: %s <=> %s ok\n", av[0], (A), (B)); }
#define FUT(X) if (!(X)) { fprintf(stderr, "%s: [%d] %s == FALSE\n", av[0], __LINE__, #X); exit(1); } else if (verbose_flag) { fprintf(stderr, "%s: %s ok\n", av[0], #X);  }
#define ITERS 100000

static Boolean verbose_flag = FALSE;

Boolean mask_equal_1(SshIpAddr a, SshIpAddr m);

Boolean mask_equal_1(SshIpAddr a, SshIpAddr m)
{
  int i, j;

  if (a->type != m->type)
    return FALSE;
  i = m->mask_len / 8;
  if (i > SSH_IP_ADDR_LEN(a))
    return FALSE;
      
  if (memcmp(a->addr_data, m->addr_data, i) != 0)
    return FALSE;
  j = m->mask_len % 8;
  if (j % 8 == 0)
    return TRUE;
  if ((a->addr_data[i] ^ m->addr_data[i]) &
      "\x80\xc0\xe0\xf0\xf8\xfc\xfe"[j - 1])
    return FALSE;
  return TRUE;
}

static unsigned long speed_test(SshIpAddr ip1, SshIpAddr ip2,
                                Boolean (*func)(SshIpAddr, SshIpAddr),
                                const char * name, Boolean ret)
{
  unsigned char buf[512 * 1024];
  Boolean b;
  int i;
  long msec;
  SshTimeMeasure tm;
  
  for (i = 0; i < sizeof(buf); i++)
    {
      buf[i] = i ^ buf[i];
    }
    
  tm = ssh_time_measure_allocate();
  ssh_time_measure_start(tm);

  for (i = 0; i < ITERS; i++)
    {
      b = (func)(ip1, ip2);
      if (b != ret)
        {
          fprintf(stderr, "Error in %s return value incorrect\n",
                  name);
          exit(1);
        }
    }
  ssh_time_measure_stop(tm);

  msec = (long)ssh_time_measure_get(tm, SSH_TIME_GRANULARITY_MICROSECOND);
  printf("count=%d %32s = %-5s: %ld microseconds\n",
         i, name, b ? "TRUE" : "FALSE", msec);
  
  for (i = 0; i < sizeof(buf); i++)
    {
      buf[i] = i ^ buf[i];
    }

  return msec;
}

void print_bits(const char * head,
                const void * data_raw,
                unsigned int bytes)
{
  int i, b, n;
  const unsigned char * data = (const unsigned char *)data_raw;

  fprintf(stderr, "%s = ", head);

  for (i = 0; i < bytes; i++)
    {
      b = data[i];
      for (n = 7; n >= 0; n--)
        fprintf(stderr, "%c", (b & (1 << n)) ? '1' : '0');
      if ((i + 1) < bytes)
        fprintf(stderr, ".");
    }
  
  fprintf(stderr, "\n");
}
                 

void speed(void)
{
  SshIpAddrStruct ip1, ip2;
  unsigned long t1, t3;
  
  ssh_ipaddr_parse(&ip1, "10.1.0.5");
  ssh_ipaddr_parse_with_mask(&ip2, "10.1.0.0/16", NULL);
  
  /************************************************************************/
  printf("With ipv4 and /16 network\n");
  t1 = speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", TRUE);
  t3 = speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                  TRUE);

  ssh_ipaddr_parse(&ip1, "10.1.0.5");
  ssh_ipaddr_parse_with_mask(&ip2, "10.1.0.5/32", NULL);
  printf("With ipv4 and /32 network\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", TRUE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   TRUE);
  
  ssh_ipaddr_parse(&ip1, "10.1.2.5");
  ssh_ipaddr_parse_with_mask(&ip2, "10.1.0.0/24", NULL);
  printf("With ipv4 and /24 network, failing\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", FALSE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   FALSE);

  ssh_ipaddr_parse(&ip1, "11.1.2.5");
  ssh_ipaddr_parse_with_mask(&ip2, "10.1.0.0/24", NULL);
  printf("With ipv4 and /24 network, failing immediately\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", FALSE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   FALSE);

  ssh_ipaddr_parse(&ip1, "f000:1234:5678:9abc:deff:edcb:a987:6543");
  ssh_ipaddr_parse_with_mask(&ip2, "f000:1234:5678:9abc:deff:edcb:a987:6543/124",
                             NULL);
  printf("With ipv6 and /124 network\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", TRUE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   TRUE);
  
  ssh_ipaddr_parse(&ip1, "f000:1234:5678:9abc:deff:edcb:a987:6543");
  ssh_ipaddr_parse_with_mask(&ip2, "f000:1234:5678:9abc:deff:edcb:a987:6543/64",
                             NULL);
  printf("With ipv6 and /64 network\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", TRUE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   TRUE);

  ssh_ipaddr_parse(&ip1, "f000:1234:5678:9abc:deff:edcb:a987:6543");
  ssh_ipaddr_parse_with_mask(&ip2, "f000:1234:5678:9abc:deff:edcb:a987:6543/16",
                             NULL);
  printf("With ipv6 and /16 network\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", TRUE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   TRUE);
  
  ssh_ipaddr_parse(&ip1, "f000:1234:5678:9abc:deff:edcb:a987:6543");
  ssh_ipaddr_parse_with_mask(&ip2, "f000:1234:5678:9abc:deff:fdcb:a987:6543/96",
                             NULL);
  printf("With ipv6 and /96 network, failing at the end\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", FALSE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   FALSE);

  ssh_ipaddr_parse(&ip1, "f000:1234:5678:9abc:deff:edcb:a987:6543");
  ssh_ipaddr_parse_with_mask(&ip2, "f000:2234:5678:9abc:deff:edcb:a987:6543/96",
                             NULL);
  printf("With ipv6 and /96 network, failing at the beginning\n");
  t1 += speed_test(&ip1, &ip2, mask_equal_1, "mask_equal_1", FALSE);
  t3 += speed_test(&ip1, &ip2, ssh_ipaddr_mask_equal, "ssh_ipaddr_mask_equal",
                   FALSE);
  
  /************************************************************************/
  
  printf("mask_equal_t1 / ssh_ipaddr_mask_equal = %f\n",
         (double) t1 / (double) t3);
}

/*

   ipv6 address parsing test strings:

   - string to parse
   - whether it is supposed to succeed in parsing (TRUE) or fail (FALSE)
   - if parsing should be true, address to compare against

*/

static struct {
  char                * str;
  Boolean             ok;
  unsigned char       addr[16];
} ipv6_test[] = {
  { "fedc:ba98:7654:3210:fedc:ba98:7654:3210", TRUE,
    { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
      0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10, }, },
  { "FEDC:BA98:7654:3210:FEDC:BA98:7654:3210", TRUE,
    { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
      0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10, }, },
  { "1080:0:0:0:8:800:200c:417a", TRUE,
    { 0x10, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x08, 0x08, 0x00, 0x20, 0x0c, 0x41, 0x7a, }, },
  { "ff01:0:0:0:0:0:43:0", TRUE,
    { 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, }, },
  { "1080::8:800:200c:417a", TRUE,
    { 0x10, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x08, 0x08, 0x00, 0x20, 0x0c, 0x41, 0x7a, }, },
  { "ff01::43", TRUE,
    { 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, }, },
  { "::1", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1, }, },
  { "::", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::13.1.68.3", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x0d, 0x01, 0x44, 0x03, }, },
  { "::ffff:129.144.52.18", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0xff, 0xff, 0x81, 0x90, 0x34, 0x12, }, },
  { "f01::192.168.2.69", TRUE,
    { 0x0f, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0xc0, 0xa8, 0x02, 0x45, }, },
  { "f01:abcd:def0:1111:2222:3333:4444:192.168.2.69", FALSE,
    { 0x0f, 0x01, 0xab, 0xcd, 0xde, 0xf0, 0x11, 0x11,
      0x22, 0x22, 0x33, 0x33, 0x44, 0x44, 0x00, 0x00, }, },
  { ":1223", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "1224:", FALSE,
    { 0x12, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { ":", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "4324:5456", FALSE,
    { 0x43, 0x24, 0x54, 0x56, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::1::4::5", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, }, },
  { "::54.fds.34.dsacf", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::44.3", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "dumby:gumby:humby::1", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "abc.def.ghi.j", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "abc.def.ghi.jk", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "abc.def.ghi.jkl", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "abc.def.ghi.jklm", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "abc.def.ghi.jklmn", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  
  /* Actually, as we know that ipv6 parser uses ipv4 parser
     internally to parse ipv4 parts, we here actually test the ipv4
     parser .. */
  
  { "::1..3.4", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::fo.def.5.4", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::199.994.122.0", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::199.-1.122.0", FALSE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, }, },
  { "::122.0", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00, }, },
  { "::255.255.255.255", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, }, },
  { "::192.99.195.255", TRUE,
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0xc0, 0x63, 0xc3, 0xff, }, },
};

typedef struct SshNetMaskTestRec {
  char *netmask;
  char *ip;
  Boolean inside;
} *SshNetMaskTest, SshNetMaskTestStruct;

SshNetMaskTestStruct netmask_tests[] = {
  { "10.1.1.0/24", "10.1.1.1", TRUE }, 
  { "10.1.1.0/24", "10.1.1.0", TRUE }, 
  { "10.1.1.0/24", "10.1.1.255", TRUE }, 
  { "10.1.1.0/24", "10.1.1.128", TRUE }, 
  { "10.1.1.0/25", "10.1.1.0", TRUE }, 
  { "10.1.1.0/25", "10.1.1.127", TRUE }, 
  { "10.1.1.0/25", "10.1.1.128", FALSE }, 
  { "10.1.1.0/25", "10.1.1.255", FALSE }, 

  { "123.45.67.89/32", "123.45.67.89", TRUE }, 
  { "123.45.67.89/31", "123.45.67.88", TRUE }, 
  { "123.45.67.89/30", "123.45.67.88", TRUE }, 
  { "123.45.67.89/29", "123.45.67.88", TRUE }, 
  { "123.45.67.89/28", "123.45.67.80", TRUE }, 
  { "123.45.67.89/27", "123.45.67.64", TRUE }, 
  { "123.45.67.89/26", "123.45.67.64", TRUE }, 
  { "123.45.67.89/25", "123.45.67.0", TRUE }, 
  { "123.45.67.89/24", "123.45.67.0", TRUE }, 
  { "123.45.67.89/23", "123.45.66.0", TRUE }, 
  { "123.45.67.89/22", "123.45.64.0", TRUE }, 
  { "123.45.67.89/21", "123.45.64.0", TRUE }, 
  { "123.45.67.89/20", "123.45.64.0", TRUE }, 
  { "123.45.67.89/19", "123.45.64.0", TRUE }, 
  { "123.45.67.89/18", "123.45.64.0", TRUE }, 
  { "123.45.67.89/17", "123.45.0.0", TRUE }, 
  { "123.45.67.89/16", "123.45.0.0", TRUE }, 
  { "123.45.67.89/15", "123.44.0.0", TRUE }, 
  { "123.45.67.89/14", "123.44.0.0", TRUE }, 
  { "123.45.67.89/13", "123.40.0.0", TRUE }, 
  { "123.45.67.89/12", "123.32.0.0", TRUE }, 
  { "123.45.67.89/11", "123.32.0.0", TRUE }, 
  { "123.45.67.89/10", "123.0.0.0", TRUE }, 
  { "123.45.67.89/9", "123.0.0.0", TRUE }, 
  { "123.45.67.89/8", "123.0.0.0", TRUE }, 
  { "123.45.67.89/7", "122.0.0.0", TRUE }, 
  { "123.45.67.89/6", "120.0.0.0", TRUE }, 
  { "123.45.67.89/5", "120.0.0.0", TRUE }, 
  { "123.45.67.89/4", "112.0.0.0", TRUE }, 
  { "123.45.67.89/3", "96.0.0.0", TRUE }, 
  { "123.45.67.89/2", "64.0.0.0", TRUE }, 
  { "123.45.67.89/1", "0.0.0.0", TRUE }, 
  { "123.45.67.89/0", "0.0.0.0", TRUE }, 

  { "123.45.67.89/32", "123.45.67.89", TRUE }, 
  { "123.45.67.88/31", "123.45.67.89", TRUE }, 
  { "123.45.67.88/30", "123.45.67.89", TRUE }, 
  { "123.45.67.88/29", "123.45.67.89", TRUE }, 
  { "123.45.67.80/28", "123.45.67.89", TRUE }, 
  { "123.45.67.64/27", "123.45.67.89", TRUE }, 
  { "123.45.67.64/26", "123.45.67.89", TRUE }, 
  { "123.45.67.0/25", "123.45.67.89", TRUE }, 
  { "123.45.67.0/24", "123.45.67.89", TRUE }, 
  { "123.45.66.0/23", "123.45.67.89", TRUE }, 
  { "123.45.64.0/22", "123.45.67.89", TRUE }, 
  { "123.45.64.0/21", "123.45.67.89", TRUE }, 
  { "123.45.64.0/20", "123.45.67.89", TRUE }, 
  { "123.45.64.0/19", "123.45.67.89", TRUE }, 
  { "123.45.64.0/18", "123.45.67.89", TRUE }, 
  { "123.45.0.0/17", "123.45.67.89", TRUE }, 
  { "123.45.0.0/16", "123.45.67.89", TRUE }, 
  { "123.44.0.0/15", "123.45.67.89", TRUE }, 
  { "123.44.0.0/14", "123.45.67.89", TRUE }, 
  { "123.40.0.0/13", "123.45.67.89", TRUE }, 
  { "123.32.0.0/12", "123.45.67.89", TRUE }, 
  { "123.32.0.0/11", "123.45.67.89", TRUE }, 
  { "123.0.0.0/10", "123.45.67.89", TRUE }, 
  { "123.0.0.0/9", "123.45.67.89", TRUE }, 
  { "123.0.0.0/8", "123.45.67.89", TRUE }, 
  { "122.0.0.0/7", "123.45.67.89", TRUE }, 
  { "120.0.0.0/6", "123.45.67.89", TRUE }, 
  { "120.0.0.0/5", "123.45.67.89", TRUE }, 
  { "112.0.0.0/4", "123.45.67.89", TRUE }, 
  { "96.0.0.0/3", "123.45.67.89", TRUE }, 
  { "64.0.0.0/2", "123.45.67.89", TRUE }, 
  { "0.0.0.0/1", "123.45.67.89", TRUE }, 
  { "0.0.0.0/0", "123.45.67.89", TRUE }, 

  { "123.45.67.89/32", "123.45.67.88", FALSE }, 
  { "123.45.67.89/31", "123.45.67.87", FALSE }, 
  { "123.45.67.89/30", "123.45.67.87", FALSE }, 
  { "123.45.67.89/29", "123.45.67.87", FALSE }, 
  { "123.45.67.89/28", "123.45.67.79", FALSE }, 
  { "123.45.67.89/27", "123.45.67.63", FALSE }, 
  { "123.45.67.89/26", "123.45.67.63", FALSE }, 
  { "123.45.67.89/25", "123.45.66.0", FALSE }, 
  { "123.45.67.89/24", "123.45.66.0", FALSE }, 
  { "123.45.67.89/23", "123.45.65.0", FALSE }, 
  { "123.45.67.89/22", "123.45.63.0", FALSE }, 
  { "123.45.67.89/21", "123.45.63.0", FALSE }, 
  { "123.45.67.89/20", "123.45.63.0", FALSE }, 
  { "123.45.67.89/19", "123.45.63.0", FALSE }, 
  { "123.45.67.89/18", "123.45.63.0", FALSE }, 
  { "123.45.67.89/17", "123.44.0.0", FALSE }, 
  { "123.45.67.89/16", "123.44.0.0", FALSE }, 
  { "123.45.67.89/15", "123.43.0.0", FALSE }, 
  { "123.45.67.89/14", "123.43.0.0", FALSE }, 
  { "123.45.67.89/13", "123.39.0.0", FALSE }, 
  { "123.45.67.89/12", "123.31.0.0", FALSE }, 
  { "123.45.67.89/11", "123.31.0.0", FALSE }, 
  { "123.45.67.89/10", "122.0.0.0", FALSE }, 
  { "123.45.67.89/9", "122.0.0.0", FALSE }, 
  { "123.45.67.89/8", "122.0.0.0", FALSE }, 
  { "123.45.67.89/7", "121.0.0.0", FALSE }, 
  { "123.45.67.89/6", "119.0.0.0", FALSE }, 
  { "123.45.67.89/5", "119.0.0.0", FALSE }, 
  { "123.45.67.89/4", "111.0.0.0", FALSE }, 
  { "123.45.67.89/3", "95.0.0.0", FALSE }, 
  { "123.45.67.89/2", "63.0.0.0", FALSE }, 
  { "123.45.67.89/1", "128.0.0.0", FALSE }, 

  { "128.128.128.128/32", "128.128.128.127", FALSE },
  { "128.128.128.128/32", "128.128.128.128", TRUE },
  { "128.128.128.128/32", "128.128.128.129", FALSE },
  { "128.128.128.128/31", "128.128.128.127", FALSE },
  { "128.128.128.128/31", "128.128.128.128", TRUE },
  { "128.128.128.128/31", "128.128.128.129", TRUE },
  { "128.128.128.128/31", "128.128.128.130", FALSE },
  { "128.128.128.128/30", "128.128.128.127", FALSE },
  { "128.128.128.128/30", "128.128.128.128", TRUE },
  { "128.128.128.128/30", "128.128.128.131", TRUE },
  { "128.128.128.128/30", "128.128.128.132", FALSE },
  { "128.128.128.128/29", "128.128.128.127", FALSE },
  { "128.128.128.128/29", "128.128.128.128", TRUE },
  { "128.128.128.128/29", "128.128.128.135", TRUE },
  { "128.128.128.128/29", "128.128.128.136", FALSE },
  { "128.128.128.128/28", "128.128.128.127", FALSE },
  { "128.128.128.128/28", "128.128.128.128", TRUE },
  { "128.128.128.128/28", "128.128.128.143", TRUE },
  { "128.128.128.128/28", "128.128.128.144", FALSE },
  { "128.128.128.128/27", "128.128.128.127", FALSE },
  { "128.128.128.128/27", "128.128.128.128", TRUE },
  { "128.128.128.128/27", "128.128.128.159", TRUE },
  { "128.128.128.128/27", "128.128.128.160", FALSE },
  { "128.128.128.128/26", "128.128.128.127", FALSE },
  { "128.128.128.128/26", "128.128.128.128", TRUE },
  { "128.128.128.128/26", "128.128.128.191", TRUE },
  { "128.128.128.128/26", "128.128.128.192", FALSE },
  { "128.128.128.128/25", "128.128.128.127", FALSE },
  { "128.128.128.128/25", "128.128.128.128", TRUE },
  { "128.128.128.128/25", "128.128.128.255", TRUE },
  { "128.128.128.128/25", "128.128.129.0", FALSE },

  { "128.128.128.0/24", "128.128.127.0", FALSE },
  { "128.128.128.0/24", "128.128.128.0", TRUE },
  { "128.128.128.0/24", "128.128.129.0", FALSE },
  { "128.128.128.0/23", "128.128.127.0", FALSE },
  { "128.128.128.0/23", "128.128.128.0", TRUE },
  { "128.128.128.0/23", "128.128.129.0", TRUE },
  { "128.128.128.0/23", "128.128.130.0", FALSE },
  { "128.128.128.0/22", "128.128.127.0", FALSE },
  { "128.128.128.0/22", "128.128.128.0", TRUE },
  { "128.128.128.0/22", "128.128.131.0", TRUE },
  { "128.128.128.0/22", "128.128.132.0", FALSE },
  { "128.128.128.0/21", "128.128.127.0", FALSE },
  { "128.128.128.0/21", "128.128.128.0", TRUE },
  { "128.128.128.0/21", "128.128.135.0", TRUE },
  { "128.128.128.0/21", "128.128.136.0", FALSE },
  { "128.128.128.0/20", "128.128.127.0", FALSE },
  { "128.128.128.0/20", "128.128.128.0", TRUE },
  { "128.128.128.0/20", "128.128.143.0", TRUE },
  { "128.128.128.0/20", "128.128.144.0", FALSE },
  { "128.128.128.0/19", "128.128.127.0", FALSE },
  { "128.128.128.0/19", "128.128.128.0", TRUE },
  { "128.128.128.0/19", "128.128.159.0", TRUE },
  { "128.128.128.0/19", "128.128.160.0", FALSE },
  { "128.128.128.0/18", "128.128.127.0", FALSE },
  { "128.128.128.0/18", "128.128.128.0", TRUE },
  { "128.128.128.0/18", "128.128.191.0", TRUE },
  { "128.128.128.0/18", "128.128.192.0", FALSE },
  { "128.128.128.0/17", "128.128.127.0", FALSE },
  { "128.128.128.0/17", "128.128.128.0", TRUE },
  { "128.128.128.0/17", "128.128.255.0", TRUE },
  { "128.128.128.0/17", "128.129.0.0", FALSE },

  { "128.128.0.0/16", "128.127.0.0", FALSE },
  { "128.128.0.0/16", "128.128.0.0", TRUE },
  { "128.128.0.0/16", "128.129.0.0", FALSE },
  { "128.128.0.0/15", "128.127.0.0", FALSE },
  { "128.128.0.0/15", "128.128.0.0", TRUE },
  { "128.128.0.0/15", "128.129.0.0", TRUE },
  { "128.128.0.0/15", "128.130.0.0", FALSE },
  { "128.128.0.0/14", "128.127.0.0", FALSE },
  { "128.128.0.0/14", "128.128.0.0", TRUE },
  { "128.128.0.0/14", "128.131.0.0", TRUE },
  { "128.128.0.0/14", "128.132.0.0", FALSE },
  { "128.128.0.0/13", "128.127.0.0", FALSE },
  { "128.128.0.0/13", "128.128.0.0", TRUE },
  { "128.128.0.0/13", "128.135.0.0", TRUE },
  { "128.128.0.0/13", "128.136.0.0", FALSE },
  { "128.128.0.0/12", "128.127.0.0", FALSE },
  { "128.128.0.0/12", "128.128.0.0", TRUE },
  { "128.128.0.0/12", "128.143.0.0", TRUE },
  { "128.128.0.0/12", "128.144.0.0", FALSE },
  { "128.128.0.0/11", "128.127.0.0", FALSE },
  { "128.128.0.0/11", "128.128.0.0", TRUE },
  { "128.128.0.0/11", "128.159.0.0", TRUE },
  { "128.128.0.0/11", "128.160.0.0", FALSE },
  { "128.128.0.0/10", "128.127.0.0", FALSE },
  { "128.128.0.0/10", "128.128.0.0", TRUE },
  { "128.128.0.0/10", "128.191.0.0", TRUE },
  { "128.128.0.0/10", "128.192.0.0", FALSE },
  { "128.128.0.0/9", "128.127.0.0", FALSE },
  { "128.128.0.0/9", "128.128.0.0", TRUE },
  { "128.128.0.0/9", "128.255.0.0", TRUE },
  { "128.128.0.0/9", "129.0.0.0", FALSE },

  { "128.0.0.0/8", "127.0.0.0", FALSE },
  { "128.0.0.0/8", "128.0.0.0", TRUE },
  { "128.0.0.0/8", "129.0.0.0", FALSE },
  { "128.0.0.0/7", "127.0.0.0", FALSE },
  { "128.0.0.0/7", "128.0.0.0", TRUE },
  { "128.0.0.0/7", "129.0.0.0", TRUE },
  { "128.0.0.0/7", "130.0.0.0", FALSE },
  { "128.0.0.0/6", "127.0.0.0", FALSE },
  { "128.0.0.0/6", "128.0.0.0", TRUE },
  { "128.0.0.0/6", "131.0.0.0", TRUE },
  { "128.0.0.0/6", "132.0.0.0", FALSE },
  { "128.0.0.0/5", "127.0.0.0", FALSE },
  { "128.0.0.0/5", "128.0.0.0", TRUE },
  { "128.0.0.0/5", "135.0.0.0", TRUE },
  { "128.0.0.0/5", "136.0.0.0", FALSE },
  { "128.0.0.0/4", "127.0.0.0", FALSE },
  { "128.0.0.0/4", "128.0.0.0", TRUE },
  { "128.0.0.0/4", "143.0.0.0", TRUE },
  { "128.0.0.0/4", "144.0.0.0", FALSE },
  { "128.0.0.0/3", "127.0.0.0", FALSE },
  { "128.0.0.0/3", "128.0.0.0", TRUE },
  { "128.0.0.0/3", "159.0.0.0", TRUE },
  { "128.0.0.0/3", "160.0.0.0", FALSE },
  { "128.0.0.0/2", "127.0.0.0", FALSE },
  { "128.0.0.0/2", "128.0.0.0", TRUE },
  { "128.0.0.0/2", "191.0.0.0", TRUE },
  { "128.0.0.0/2", "192.0.0.0", FALSE },
  { "128.0.0.0/1", "127.0.0.0", FALSE },
  { "128.0.0.0/1", "128.0.0.0", TRUE },
  { "128.0.0.0/1", "255.0.0.0", TRUE },
  { "128.0.0.0/1", "127.0.0.0", FALSE },

  { "::10.1.1.0/120", "10.1.1.1", TRUE }, 
  { "::10.1.1.0/120", "10.1.1.0", TRUE }, 
  { "::10.1.1.0/120", "10.1.1.255", TRUE }, 
  { "::10.1.1.0/120", "10.1.1.128", TRUE }, 
  { "::10.1.1.0/121", "10.1.1.0", TRUE }, 
  { "::10.1.1.0/121", "10.1.1.127", TRUE }, 
  { "::10.1.1.0/121", "10.1.1.128", FALSE }, 
  { "::10.1.1.0/121", "10.1.1.255", FALSE }, 
  { "::123.45.67.89/128", "123.45.67.89", TRUE }, 
  { "::123.45.67.89/127", "123.45.67.88", TRUE }, 
  { "::123.45.67.89/126", "123.45.67.88", TRUE }, 
  { "::123.45.67.89/125", "123.45.67.88", TRUE }, 
  { "::123.45.67.89/124", "123.45.67.80", TRUE }, 
  { "::123.45.67.89/123", "123.45.67.64", TRUE }, 
  { "::123.45.67.89/122", "123.45.67.64", TRUE }, 
  { "::123.45.67.89/121", "123.45.67.0", TRUE }, 
  { "::123.45.67.89/120", "123.45.67.0", TRUE }, 
  { "::123.45.67.89/119", "123.45.66.0", TRUE }, 
  { "::123.45.67.89/118", "123.45.64.0", TRUE }, 
  { "::123.45.67.89/117", "123.45.64.0", TRUE }, 
  { "::123.45.67.89/116", "123.45.64.0", TRUE }, 
  { "::123.45.67.89/115", "123.45.64.0", TRUE }, 
  { "::123.45.67.89/114", "123.45.64.0", TRUE }, 
  { "::123.45.67.89/113", "123.45.0.0", TRUE }, 
  { "::123.45.67.89/112", "123.45.0.0", TRUE }, 
  { "::123.45.67.89/111", "123.44.0.0", TRUE }, 
  { "::123.45.67.89/110", "123.44.0.0", TRUE }, 
  { "::123.45.67.89/109", "123.40.0.0", TRUE }, 
  { "::123.45.67.89/108", "123.32.0.0", TRUE }, 
  { "::123.45.67.89/107", "123.32.0.0", TRUE }, 
  { "::123.45.67.89/106", "123.0.0.0", TRUE }, 
  { "::123.45.67.89/105", "123.0.0.0", TRUE }, 
  { "::123.45.67.89/104", "123.0.0.0", TRUE }, 
  { "::123.45.67.89/103", "122.0.0.0", TRUE }, 
  { "::123.45.67.89/102", "120.0.0.0", TRUE }, 
  { "::123.45.67.89/101", "120.0.0.0", TRUE }, 
  { "::123.45.67.89/100", "112.0.0.0", TRUE }, 
  { "::123.45.67.89/99", "96.0.0.0", TRUE }, 
  { "::123.45.67.89/98", "64.0.0.0", TRUE }, 
  { "::123.45.67.89/97", "0.0.0.0", TRUE }, 
  { "::123.45.67.89/96", "0.0.0.0", TRUE }, 

  { "::123.45.67.89/128", "123.45.67.89", TRUE }, 
  { "::123.45.67.88/127", "123.45.67.89", TRUE }, 
  { "::123.45.67.88/126", "123.45.67.89", TRUE }, 
  { "::123.45.67.88/125", "123.45.67.89", TRUE }, 
  { "::123.45.67.80/124", "123.45.67.89", TRUE }, 
  { "::123.45.67.64/123", "123.45.67.89", TRUE }, 
  { "::123.45.67.64/122", "123.45.67.89", TRUE }, 
  { "::123.45.67.0/121", "123.45.67.89", TRUE }, 
  { "::123.45.67.0/120", "123.45.67.89", TRUE }, 
  { "::123.45.66.0/119", "123.45.67.89", TRUE }, 
  { "::123.45.64.0/118", "123.45.67.89", TRUE }, 
  { "::123.45.64.0/117", "123.45.67.89", TRUE }, 
  { "::123.45.64.0/116", "123.45.67.89", TRUE }, 
  { "::123.45.64.0/115", "123.45.67.89", TRUE }, 
  { "::123.45.64.0/114", "123.45.67.89", TRUE }, 
  { "::123.45.0.0/113", "123.45.67.89", TRUE }, 
  { "::123.45.0.0/112", "123.45.67.89", TRUE }, 
  { "::123.44.0.0/111", "123.45.67.89", TRUE }, 
  { "::123.44.0.0/110", "123.45.67.89", TRUE }, 
  { "::123.40.0.0/109", "123.45.67.89", TRUE }, 
  { "::123.32.0.0/108", "123.45.67.89", TRUE }, 
  { "::123.32.0.0/107", "123.45.67.89", TRUE }, 
  { "::123.0.0.0/106", "123.45.67.89", TRUE }, 
  { "::123.0.0.0/105", "123.45.67.89", TRUE }, 
  { "::123.0.0.0/104", "123.45.67.89", TRUE }, 
  { "::122.0.0.0/103", "123.45.67.89", TRUE }, 
  { "::120.0.0.0/102", "123.45.67.89", TRUE }, 
  { "::120.0.0.0/101", "123.45.67.89", TRUE }, 
  { "::112.0.0.0/100", "123.45.67.89", TRUE }, 
  { "::96.0.0.0/99", "123.45.67.89", TRUE }, 
  { "::64.0.0.0/98", "123.45.67.89", TRUE }, 
  { "::0.0.0.0/97", "123.45.67.89", TRUE }, 
  { "::0.0.0.0/96", "123.45.67.89", TRUE }, 

  { "::123.45.67.89/128", "123.45.67.88", FALSE }, 
  { "::123.45.67.89/127", "123.45.67.87", FALSE }, 
  { "::123.45.67.89/126", "123.45.67.87", FALSE }, 
  { "::123.45.67.89/125", "123.45.67.87", FALSE }, 
  { "::123.45.67.89/124", "123.45.67.79", FALSE }, 
  { "::123.45.67.89/123", "123.45.67.63", FALSE }, 
  { "::123.45.67.89/122", "123.45.67.63", FALSE }, 
  { "::123.45.67.89/121", "123.45.66.0", FALSE }, 
  { "::123.45.67.89/120", "123.45.66.0", FALSE }, 
  { "::123.45.67.89/119", "123.45.65.0", FALSE }, 
  { "::123.45.67.89/118", "123.45.63.0", FALSE }, 
  { "::123.45.67.89/117", "123.45.63.0", FALSE }, 
  { "::123.45.67.89/116", "123.45.63.0", FALSE }, 
  { "::123.45.67.89/115", "123.45.63.0", FALSE }, 
  { "::123.45.67.89/114", "123.45.63.0", FALSE }, 
  { "::123.45.67.89/113", "123.44.0.0", FALSE }, 
  { "::123.45.67.89/112", "123.44.0.0", FALSE }, 
  { "::123.45.67.89/111", "123.43.0.0", FALSE }, 
  { "::123.45.67.89/110", "123.43.0.0", FALSE }, 
  { "::123.45.67.89/109", "123.39.0.0", FALSE }, 
  { "::123.45.67.89/108", "123.31.0.0", FALSE }, 
  { "::123.45.67.89/107", "123.31.0.0", FALSE }, 
  { "::123.45.67.89/106", "122.0.0.0", FALSE }, 
  { "::123.45.67.89/105", "122.0.0.0", FALSE }, 
  { "::123.45.67.89/104", "122.0.0.0", FALSE }, 
  { "::123.45.67.89/103", "121.0.0.0", FALSE }, 
  { "::123.45.67.89/102", "119.0.0.0", FALSE }, 
  { "::123.45.67.89/101", "119.0.0.0", FALSE }, 
  { "::123.45.67.89/100", "111.0.0.0", FALSE }, 
  { "::123.45.67.89/99", "95.0.0.0", FALSE }, 
  { "::123.45.67.89/98", "63.0.0.0", FALSE }, 
  { "::123.45.67.89/97", "128.0.0.0", FALSE },

  { "::128.128.128.128/128", "128.128.128.127", FALSE },
  { "::128.128.128.128/128", "128.128.128.128", TRUE },
  { "::128.128.128.128/128", "128.128.128.129", FALSE },
  { "::128.128.128.128/127", "128.128.128.127", FALSE },
  { "::128.128.128.128/127", "128.128.128.128", TRUE },
  { "::128.128.128.128/127", "128.128.128.129", TRUE },
  { "::128.128.128.128/127", "128.128.128.130", FALSE },
  { "::128.128.128.128/126", "128.128.128.127", FALSE },
  { "::128.128.128.128/126", "128.128.128.128", TRUE },
  { "::128.128.128.128/126", "128.128.128.131", TRUE },
  { "::128.128.128.128/126", "128.128.128.132", FALSE },
  { "::128.128.128.128/125", "128.128.128.127", FALSE },
  { "::128.128.128.128/125", "128.128.128.128", TRUE },
  { "::128.128.128.128/125", "128.128.128.135", TRUE },
  { "::128.128.128.128/125", "128.128.128.136", FALSE },
  { "::128.128.128.128/124", "128.128.128.127", FALSE },
  { "::128.128.128.128/124", "128.128.128.128", TRUE },
  { "::128.128.128.128/124", "128.128.128.143", TRUE },
  { "::128.128.128.128/124", "128.128.128.144", FALSE },
  { "::128.128.128.128/123", "128.128.128.127", FALSE },
  { "::128.128.128.128/123", "128.128.128.128", TRUE },
  { "::128.128.128.128/123", "128.128.128.159", TRUE },
  { "::128.128.128.128/123", "128.128.128.160", FALSE },
  { "::128.128.128.128/122", "128.128.128.127", FALSE },
  { "::128.128.128.128/122", "128.128.128.128", TRUE },
  { "::128.128.128.128/122", "128.128.128.191", TRUE },
  { "::128.128.128.128/122", "128.128.128.192", FALSE },
  { "::128.128.128.128/121", "128.128.128.127", FALSE },
  { "::128.128.128.128/121", "128.128.128.128", TRUE },
  { "::128.128.128.128/121", "128.128.128.255", TRUE },
  { "::128.128.128.128/121", "128.128.129.0", FALSE },

  { "::128.128.128.0/120", "128.128.127.0", FALSE },
  { "::128.128.128.0/120", "128.128.128.0", TRUE },
  { "::128.128.128.0/120", "128.128.129.0", FALSE },
  { "::128.128.128.0/119", "128.128.127.0", FALSE },
  { "::128.128.128.0/119", "128.128.128.0", TRUE },
  { "::128.128.128.0/119", "128.128.129.0", TRUE },
  { "::128.128.128.0/119", "128.128.130.0", FALSE },
  { "::128.128.128.0/118", "128.128.127.0", FALSE },
  { "::128.128.128.0/118", "128.128.128.0", TRUE },
  { "::128.128.128.0/118", "128.128.131.0", TRUE },
  { "::128.128.128.0/118", "128.128.132.0", FALSE },
  { "::128.128.128.0/117", "128.128.127.0", FALSE },
  { "::128.128.128.0/117", "128.128.128.0", TRUE },
  { "::128.128.128.0/117", "128.128.135.0", TRUE },
  { "::128.128.128.0/117", "128.128.136.0", FALSE },
  { "::128.128.128.0/116", "128.128.127.0", FALSE },
  { "::128.128.128.0/116", "128.128.128.0", TRUE },
  { "::128.128.128.0/116", "128.128.143.0", TRUE },
  { "::128.128.128.0/116", "128.128.144.0", FALSE },
  { "::128.128.128.0/115", "128.128.127.0", FALSE },
  { "::128.128.128.0/115", "128.128.128.0", TRUE },
  { "::128.128.128.0/115", "128.128.159.0", TRUE },
  { "::128.128.128.0/115", "128.128.160.0", FALSE },
  { "::128.128.128.0/114", "128.128.127.0", FALSE },
  { "::128.128.128.0/114", "128.128.128.0", TRUE },
  { "::128.128.128.0/114", "128.128.191.0", TRUE },
  { "::128.128.128.0/114", "128.128.192.0", FALSE },
  { "::128.128.128.0/113", "128.128.127.0", FALSE },
  { "::128.128.128.0/113", "128.128.128.0", TRUE },
  { "::128.128.128.0/113", "128.128.255.0", TRUE },
  { "::128.128.128.0/113", "128.129.0.0", FALSE },

  { "::128.128.0.0/112", "128.127.0.0", FALSE },
  { "::128.128.0.0/112", "128.128.0.0", TRUE },
  { "::128.128.0.0/112", "128.129.0.0", FALSE },
  { "::128.128.0.0/111", "128.127.0.0", FALSE },
  { "::128.128.0.0/111", "128.128.0.0", TRUE },
  { "::128.128.0.0/111", "128.129.0.0", TRUE },
  { "::128.128.0.0/111", "128.130.0.0", FALSE },
  { "::128.128.0.0/110", "128.127.0.0", FALSE },
  { "::128.128.0.0/110", "128.128.0.0", TRUE },
  { "::128.128.0.0/110", "128.131.0.0", TRUE },
  { "::128.128.0.0/110", "128.132.0.0", FALSE },
  { "::128.128.0.0/109", "128.127.0.0", FALSE },
  { "::128.128.0.0/109", "128.128.0.0", TRUE },
  { "::128.128.0.0/109", "128.135.0.0", TRUE },
  { "::128.128.0.0/109", "128.136.0.0", FALSE },
  { "::128.128.0.0/108", "128.127.0.0", FALSE },
  { "::128.128.0.0/108", "128.128.0.0", TRUE },
  { "::128.128.0.0/108", "128.143.0.0", TRUE },
  { "::128.128.0.0/108", "128.144.0.0", FALSE },
  { "::128.128.0.0/107", "128.127.0.0", FALSE },
  { "::128.128.0.0/107", "128.128.0.0", TRUE },
  { "::128.128.0.0/107", "128.159.0.0", TRUE },
  { "::128.128.0.0/107", "128.160.0.0", FALSE },
  { "::128.128.0.0/106", "128.127.0.0", FALSE },
  { "::128.128.0.0/106", "128.128.0.0", TRUE },
  { "::128.128.0.0/106", "128.191.0.0", TRUE },
  { "::128.128.0.0/106", "128.192.0.0", FALSE },
  { "::128.128.0.0/105", "128.127.0.0", FALSE },
  { "::128.128.0.0/105", "128.128.0.0", TRUE },
  { "::128.128.0.0/105", "128.255.0.0", TRUE },
  { "::128.128.0.0/105", "129.0.0.0", FALSE },

  { "::128.0.0.0/104", "127.0.0.0", FALSE },
  { "::128.0.0.0/104", "128.0.0.0", TRUE },
  { "::128.0.0.0/104", "129.0.0.0", FALSE },
  { "::128.0.0.0/103", "127.0.0.0", FALSE },
  { "::128.0.0.0/103", "128.0.0.0", TRUE },
  { "::128.0.0.0/103", "129.0.0.0", TRUE },
  { "::128.0.0.0/103", "130.0.0.0", FALSE },
  { "::128.0.0.0/102", "127.0.0.0", FALSE },
  { "::128.0.0.0/102", "128.0.0.0", TRUE },
  { "::128.0.0.0/102", "131.0.0.0", TRUE },
  { "::128.0.0.0/102", "132.0.0.0", FALSE },
  { "::128.0.0.0/101", "127.0.0.0", FALSE },
  { "::128.0.0.0/101", "128.0.0.0", TRUE },
  { "::128.0.0.0/101", "135.0.0.0", TRUE },
  { "::128.0.0.0/101", "136.0.0.0", FALSE },
  { "::128.0.0.0/100", "127.0.0.0", FALSE },
  { "::128.0.0.0/100", "128.0.0.0", TRUE },
  { "::128.0.0.0/100", "143.0.0.0", TRUE },
  { "::128.0.0.0/100", "144.0.0.0", FALSE },
  { "::128.0.0.0/99", "127.0.0.0", FALSE },
  { "::128.0.0.0/99", "128.0.0.0", TRUE },
  { "::128.0.0.0/99", "159.0.0.0", TRUE },
  { "::128.0.0.0/99", "160.0.0.0", FALSE },
  { "::128.0.0.0/98", "127.0.0.0", FALSE },
  { "::128.0.0.0/98", "128.0.0.0", TRUE },
  { "::128.0.0.0/98", "191.0.0.0", TRUE },
  { "::128.0.0.0/98", "192.0.0.0", FALSE },
  { "::128.0.0.0/97", "127.0.0.0", FALSE },
  { "::128.0.0.0/97", "128.0.0.0", TRUE },
  { "::128.0.0.0/97", "255.0.0.0", TRUE },
  { "::128.0.0.0/97", "127.0.0.0", FALSE },

  { "::10.1.1.0/120", "::10.1.1.1", TRUE }, 
  { "::10.1.1.0/120", "::10.1.1.0", TRUE }, 
  { "::10.1.1.0/120", "::10.1.1.255", TRUE }, 
  { "::10.1.1.0/120", "::10.1.1.128", TRUE }, 
  { "::10.1.1.0/121", "::10.1.1.0", TRUE }, 
  { "::10.1.1.0/121", "::10.1.1.127", TRUE }, 
  { "::10.1.1.0/121", "::10.1.1.128", FALSE }, 
  { "::10.1.1.0/121", "::10.1.1.255", FALSE }, 
  { "::123.45.67.89/128", "::123.45.67.89", TRUE }, 
  { "::123.45.67.89/127", "::123.45.67.88", TRUE }, 
  { "::123.45.67.89/126", "::123.45.67.88", TRUE }, 
  { "::123.45.67.89/125", "::123.45.67.88", TRUE }, 
  { "::123.45.67.89/124", "::123.45.67.80", TRUE }, 
  { "::123.45.67.89/123", "::123.45.67.64", TRUE }, 
  { "::123.45.67.89/122", "::123.45.67.64", TRUE }, 
  { "::123.45.67.89/121", "::123.45.67.0", TRUE }, 
  { "::123.45.67.89/120", "::123.45.67.0", TRUE }, 
  { "::123.45.67.89/119", "::123.45.66.0", TRUE }, 
  { "::123.45.67.89/118", "::123.45.64.0", TRUE }, 
  { "::123.45.67.89/117", "::123.45.64.0", TRUE }, 
  { "::123.45.67.89/116", "::123.45.64.0", TRUE }, 
  { "::123.45.67.89/115", "::123.45.64.0", TRUE }, 
  { "::123.45.67.89/114", "::123.45.64.0", TRUE }, 
  { "::123.45.67.89/113", "::123.45.0.0", TRUE }, 
  { "::123.45.67.89/112", "::123.45.0.0", TRUE }, 
  { "::123.45.67.89/111", "::123.44.0.0", TRUE }, 
  { "::123.45.67.89/110", "::123.44.0.0", TRUE }, 
  { "::123.45.67.89/109", "::123.40.0.0", TRUE }, 
  { "::123.45.67.89/108", "::123.32.0.0", TRUE }, 
  { "::123.45.67.89/107", "::123.32.0.0", TRUE }, 
  { "::123.45.67.89/106", "::123.0.0.0", TRUE }, 
  { "::123.45.67.89/105", "::123.0.0.0", TRUE }, 
  { "::123.45.67.89/104", "::123.0.0.0", TRUE }, 
  { "::123.45.67.89/103", "::122.0.0.0", TRUE }, 
  { "::123.45.67.89/102", "::120.0.0.0", TRUE }, 
  { "::123.45.67.89/101", "::120.0.0.0", TRUE }, 
  { "::123.45.67.89/100", "::112.0.0.0", TRUE }, 
  { "::123.45.67.89/99", "::96.0.0.0", TRUE }, 
  { "::123.45.67.89/98", "::64.0.0.0", TRUE }, 
  { "::123.45.67.89/97", "::0.0.0.0", TRUE }, 
  { "::123.45.67.89/96", "::0.0.0.0", TRUE }, 

  { "::123.45.67.89/128", "::123.45.67.89", TRUE }, 
  { "::123.45.67.88/127", "::123.45.67.89", TRUE }, 
  { "::123.45.67.88/126", "::123.45.67.89", TRUE }, 
  { "::123.45.67.88/125", "::123.45.67.89", TRUE }, 
  { "::123.45.67.80/124", "::123.45.67.89", TRUE }, 
  { "::123.45.67.64/123", "::123.45.67.89", TRUE }, 
  { "::123.45.67.64/122", "::123.45.67.89", TRUE }, 
  { "::123.45.67.0/121", "::123.45.67.89", TRUE }, 
  { "::123.45.67.0/120", "::123.45.67.89", TRUE }, 
  { "::123.45.66.0/119", "::123.45.67.89", TRUE }, 
  { "::123.45.64.0/118", "::123.45.67.89", TRUE }, 
  { "::123.45.64.0/117", "::123.45.67.89", TRUE }, 
  { "::123.45.64.0/116", "::123.45.67.89", TRUE }, 
  { "::123.45.64.0/115", "::123.45.67.89", TRUE }, 
  { "::123.45.64.0/114", "::123.45.67.89", TRUE }, 
  { "::123.45.0.0/113", "::123.45.67.89", TRUE }, 
  { "::123.45.0.0/112", "::123.45.67.89", TRUE }, 
  { "::123.44.0.0/111", "::123.45.67.89", TRUE }, 
  { "::123.44.0.0/110", "::123.45.67.89", TRUE }, 
  { "::123.40.0.0/109", "::123.45.67.89", TRUE }, 
  { "::123.32.0.0/108", "::123.45.67.89", TRUE }, 
  { "::123.32.0.0/107", "::123.45.67.89", TRUE }, 
  { "::123.0.0.0/106", "::123.45.67.89", TRUE }, 
  { "::123.0.0.0/105", "::123.45.67.89", TRUE }, 
  { "::123.0.0.0/104", "::123.45.67.89", TRUE }, 
  { "::122.0.0.0/103", "::123.45.67.89", TRUE }, 
  { "::120.0.0.0/102", "::123.45.67.89", TRUE }, 
  { "::120.0.0.0/101", "::123.45.67.89", TRUE }, 
  { "::112.0.0.0/100", "::123.45.67.89", TRUE }, 
  { "::96.0.0.0/99", "::123.45.67.89", TRUE }, 
  { "::64.0.0.0/98", "::123.45.67.89", TRUE }, 
  { "::0.0.0.0/97", "::123.45.67.89", TRUE }, 
  { "::0.0.0.0/96", "::123.45.67.89", TRUE }, 

  { "::123.45.67.89/128", "::123.45.67.88", FALSE }, 
  { "::123.45.67.89/127", "::123.45.67.87", FALSE }, 
  { "::123.45.67.89/126", "::123.45.67.87", FALSE }, 
  { "::123.45.67.89/125", "::123.45.67.87", FALSE }, 
  { "::123.45.67.89/124", "::123.45.67.79", FALSE }, 
  { "::123.45.67.89/123", "::123.45.67.63", FALSE }, 
  { "::123.45.67.89/122", "::123.45.67.63", FALSE }, 
  { "::123.45.67.89/121", "::123.45.66.0", FALSE }, 
  { "::123.45.67.89/120", "::123.45.66.0", FALSE }, 
  { "::123.45.67.89/119", "::123.45.65.0", FALSE }, 
  { "::123.45.67.89/118", "::123.45.63.0", FALSE }, 
  { "::123.45.67.89/117", "::123.45.63.0", FALSE }, 
  { "::123.45.67.89/116", "::123.45.63.0", FALSE }, 
  { "::123.45.67.89/115", "::123.45.63.0", FALSE }, 
  { "::123.45.67.89/114", "::123.45.63.0", FALSE }, 
  { "::123.45.67.89/113", "::123.44.0.0", FALSE }, 
  { "::123.45.67.89/112", "::123.44.0.0", FALSE }, 
  { "::123.45.67.89/111", "::123.43.0.0", FALSE }, 
  { "::123.45.67.89/110", "::123.43.0.0", FALSE }, 
  { "::123.45.67.89/109", "::123.39.0.0", FALSE }, 
  { "::123.45.67.89/108", "::123.31.0.0", FALSE }, 
  { "::123.45.67.89/107", "::123.31.0.0", FALSE }, 
  { "::123.45.67.89/106", "::122.0.0.0", FALSE }, 
  { "::123.45.67.89/105", "::122.0.0.0", FALSE }, 
  { "::123.45.67.89/104", "::122.0.0.0", FALSE }, 
  { "::123.45.67.89/103", "::121.0.0.0", FALSE }, 
  { "::123.45.67.89/102", "::119.0.0.0", FALSE }, 
  { "::123.45.67.89/101", "::119.0.0.0", FALSE }, 
  { "::123.45.67.89/100", "::111.0.0.0", FALSE }, 
  { "::123.45.67.89/99", "::95.0.0.0", FALSE }, 
  { "::123.45.67.89/98", "::63.0.0.0", FALSE }, 
  { "::123.45.67.89/97", "::128.0.0.0", FALSE },

  { "::128.128.128.128/128", "::128.128.128.127", FALSE },
  { "::128.128.128.128/128", "::128.128.128.128", TRUE },
  { "::128.128.128.128/128", "::128.128.128.129", FALSE },
  { "::128.128.128.128/127", "::128.128.128.127", FALSE },
  { "::128.128.128.128/127", "::128.128.128.128", TRUE },
  { "::128.128.128.128/127", "::128.128.128.129", TRUE },
  { "::128.128.128.128/127", "::128.128.128.130", FALSE },
  { "::128.128.128.128/126", "::128.128.128.127", FALSE },
  { "::128.128.128.128/126", "::128.128.128.128", TRUE },
  { "::128.128.128.128/126", "::128.128.128.131", TRUE },
  { "::128.128.128.128/126", "::128.128.128.132", FALSE },
  { "::128.128.128.128/125", "::128.128.128.127", FALSE },
  { "::128.128.128.128/125", "::128.128.128.128", TRUE },
  { "::128.128.128.128/125", "::128.128.128.135", TRUE },
  { "::128.128.128.128/125", "::128.128.128.136", FALSE },
  { "::128.128.128.128/124", "::128.128.128.127", FALSE },
  { "::128.128.128.128/124", "::128.128.128.128", TRUE },
  { "::128.128.128.128/124", "::128.128.128.143", TRUE },
  { "::128.128.128.128/124", "::128.128.128.144", FALSE },
  { "::128.128.128.128/123", "::128.128.128.127", FALSE },
  { "::128.128.128.128/123", "::128.128.128.128", TRUE },
  { "::128.128.128.128/123", "::128.128.128.159", TRUE },
  { "::128.128.128.128/123", "::128.128.128.160", FALSE },
  { "::128.128.128.128/122", "::128.128.128.127", FALSE },
  { "::128.128.128.128/122", "::128.128.128.128", TRUE },
  { "::128.128.128.128/122", "::128.128.128.191", TRUE },
  { "::128.128.128.128/122", "::128.128.128.192", FALSE },
  { "::128.128.128.128/121", "::128.128.128.127", FALSE },
  { "::128.128.128.128/121", "::128.128.128.128", TRUE },
  { "::128.128.128.128/121", "::128.128.128.255", TRUE },
  { "::128.128.128.128/121", "::128.128.129.0", FALSE },

  { "::128.128.128.0/120", "::128.128.127.0", FALSE },
  { "::128.128.128.0/120", "::128.128.128.0", TRUE },
  { "::128.128.128.0/120", "::128.128.129.0", FALSE },
  { "::128.128.128.0/119", "::128.128.127.0", FALSE },
  { "::128.128.128.0/119", "::128.128.128.0", TRUE },
  { "::128.128.128.0/119", "::128.128.129.0", TRUE },
  { "::128.128.128.0/119", "::128.128.130.0", FALSE },
  { "::128.128.128.0/118", "::128.128.127.0", FALSE },
  { "::128.128.128.0/118", "::128.128.128.0", TRUE },
  { "::128.128.128.0/118", "::128.128.131.0", TRUE },
  { "::128.128.128.0/118", "::128.128.132.0", FALSE },
  { "::128.128.128.0/117", "::128.128.127.0", FALSE },
  { "::128.128.128.0/117", "::128.128.128.0", TRUE },
  { "::128.128.128.0/117", "::128.128.135.0", TRUE },
  { "::128.128.128.0/117", "::128.128.136.0", FALSE },
  { "::128.128.128.0/116", "::128.128.127.0", FALSE },
  { "::128.128.128.0/116", "::128.128.128.0", TRUE },
  { "::128.128.128.0/116", "::128.128.143.0", TRUE },
  { "::128.128.128.0/116", "::128.128.144.0", FALSE },
  { "::128.128.128.0/115", "::128.128.127.0", FALSE },
  { "::128.128.128.0/115", "::128.128.128.0", TRUE },
  { "::128.128.128.0/115", "::128.128.159.0", TRUE },
  { "::128.128.128.0/115", "::128.128.160.0", FALSE },
  { "::128.128.128.0/114", "::128.128.127.0", FALSE },
  { "::128.128.128.0/114", "::128.128.128.0", TRUE },
  { "::128.128.128.0/114", "::128.128.191.0", TRUE },
  { "::128.128.128.0/114", "::128.128.192.0", FALSE },
  { "::128.128.128.0/113", "::128.128.127.0", FALSE },
  { "::128.128.128.0/113", "::128.128.128.0", TRUE },
  { "::128.128.128.0/113", "::128.128.255.0", TRUE },
  { "::128.128.128.0/113", "::128.129.0.0", FALSE },

  { "::128.128.0.0/112", "::128.127.0.0", FALSE },
  { "::128.128.0.0/112", "::128.128.0.0", TRUE },
  { "::128.128.0.0/112", "::128.129.0.0", FALSE },
  { "::128.128.0.0/111", "::128.127.0.0", FALSE },
  { "::128.128.0.0/111", "::128.128.0.0", TRUE },
  { "::128.128.0.0/111", "::128.129.0.0", TRUE },
  { "::128.128.0.0/111", "::128.130.0.0", FALSE },
  { "::128.128.0.0/110", "::128.127.0.0", FALSE },
  { "::128.128.0.0/110", "::128.128.0.0", TRUE },
  { "::128.128.0.0/110", "::128.131.0.0", TRUE },
  { "::128.128.0.0/110", "::128.132.0.0", FALSE },
  { "::128.128.0.0/109", "::128.127.0.0", FALSE },
  { "::128.128.0.0/109", "::128.128.0.0", TRUE },
  { "::128.128.0.0/109", "::128.135.0.0", TRUE },
  { "::128.128.0.0/109", "::128.136.0.0", FALSE },
  { "::128.128.0.0/108", "::128.127.0.0", FALSE },
  { "::128.128.0.0/108", "::128.128.0.0", TRUE },
  { "::128.128.0.0/108", "::128.143.0.0", TRUE },
  { "::128.128.0.0/108", "::128.144.0.0", FALSE },
  { "::128.128.0.0/107", "::128.127.0.0", FALSE },
  { "::128.128.0.0/107", "::128.128.0.0", TRUE },
  { "::128.128.0.0/107", "::128.159.0.0", TRUE },
  { "::128.128.0.0/107", "::128.160.0.0", FALSE },
  { "::128.128.0.0/106", "::128.127.0.0", FALSE },
  { "::128.128.0.0/106", "::128.128.0.0", TRUE },
  { "::128.128.0.0/106", "::128.191.0.0", TRUE },
  { "::128.128.0.0/106", "::128.192.0.0", FALSE },
  { "::128.128.0.0/105", "::128.127.0.0", FALSE },
  { "::128.128.0.0/105", "::128.128.0.0", TRUE },
  { "::128.128.0.0/105", "::128.255.0.0", TRUE },
  { "::128.128.0.0/105", "::129.0.0.0", FALSE },

  { "::128.0.0.0/104", "::127.0.0.0", FALSE },
  { "::128.0.0.0/104", "::128.0.0.0", TRUE },
  { "::128.0.0.0/104", "::129.0.0.0", FALSE },
  { "::128.0.0.0/103", "::127.0.0.0", FALSE },
  { "::128.0.0.0/103", "::128.0.0.0", TRUE },
  { "::128.0.0.0/103", "::129.0.0.0", TRUE },
  { "::128.0.0.0/103", "::130.0.0.0", FALSE },
  { "::128.0.0.0/102", "::127.0.0.0", FALSE },
  { "::128.0.0.0/102", "::128.0.0.0", TRUE },
  { "::128.0.0.0/102", "::131.0.0.0", TRUE },
  { "::128.0.0.0/102", "::132.0.0.0", FALSE },
  { "::128.0.0.0/101", "::127.0.0.0", FALSE },
  { "::128.0.0.0/101", "::128.0.0.0", TRUE },
  { "::128.0.0.0/101", "::135.0.0.0", TRUE },
  { "::128.0.0.0/101", "::136.0.0.0", FALSE },
  { "::128.0.0.0/100", "::127.0.0.0", FALSE },
  { "::128.0.0.0/100", "::128.0.0.0", TRUE },
  { "::128.0.0.0/100", "::143.0.0.0", TRUE },
  { "::128.0.0.0/100", "::144.0.0.0", FALSE },
  { "::128.0.0.0/99", "::127.0.0.0", FALSE },
  { "::128.0.0.0/99", "::128.0.0.0", TRUE },
  { "::128.0.0.0/99", "::159.0.0.0", TRUE },
  { "::128.0.0.0/99", "::160.0.0.0", FALSE },
  { "::128.0.0.0/98", "::127.0.0.0", FALSE },
  { "::128.0.0.0/98", "::128.0.0.0", TRUE },
  { "::128.0.0.0/98", "::191.0.0.0", TRUE },
  { "::128.0.0.0/98", "::192.0.0.0", FALSE },
  { "::128.0.0.0/97", "::127.0.0.0", FALSE },
  { "::128.0.0.0/97", "::128.0.0.0", TRUE },
  { "::128.0.0.0/97", "::255.0.0.0", TRUE },
  { "::128.0.0.0/97", "::127.0.0.0", FALSE },

  { "10.1.1.0/24", "::10.1.1.1", TRUE }, 
  { "10.1.1.0/24", "::10.1.1.0", TRUE }, 
  { "10.1.1.0/24", "::10.1.1.255", TRUE }, 
  { "10.1.1.0/24", "::10.1.1.128", TRUE }, 
  { "10.1.1.0/25", "::10.1.1.0", TRUE }, 
  { "10.1.1.0/25", "::10.1.1.127", TRUE }, 
  { "10.1.1.0/25", "::10.1.1.128", FALSE }, 
  { "10.1.1.0/25", "::10.1.1.255", FALSE }, 
  { "123.45.67.89/32", "::123.45.67.89", TRUE }, 
  { "123.45.67.89/31", "::123.45.67.88", TRUE }, 
  { "123.45.67.89/30", "::123.45.67.88", TRUE }, 
  { "123.45.67.89/29", "::123.45.67.88", TRUE }, 
  { "123.45.67.89/28", "::123.45.67.80", TRUE }, 
  { "123.45.67.89/27", "::123.45.67.64", TRUE }, 
  { "123.45.67.89/26", "::123.45.67.64", TRUE }, 
  { "123.45.67.89/25", "::123.45.67.0", TRUE }, 
  { "123.45.67.89/24", "::123.45.67.0", TRUE }, 
  { "123.45.67.89/23", "::123.45.66.0", TRUE }, 
  { "123.45.67.89/22", "::123.45.64.0", TRUE }, 
  { "123.45.67.89/21", "::123.45.64.0", TRUE }, 
  { "123.45.67.89/20", "::123.45.64.0", TRUE }, 
  { "123.45.67.89/19", "::123.45.64.0", TRUE }, 
  { "123.45.67.89/18", "::123.45.64.0", TRUE }, 
  { "123.45.67.89/17", "::123.45.0.0", TRUE }, 
  { "123.45.67.89/16", "::123.45.0.0", TRUE }, 
  { "123.45.67.89/15", "::123.44.0.0", TRUE }, 
  { "123.45.67.89/14", "::123.44.0.0", TRUE }, 
  { "123.45.67.89/13", "::123.40.0.0", TRUE }, 
  { "123.45.67.89/12", "::123.32.0.0", TRUE }, 
  { "123.45.67.89/11", "::123.32.0.0", TRUE }, 
  { "123.45.67.89/10", "::123.0.0.0", TRUE }, 
  { "123.45.67.89/9", "::123.0.0.0", TRUE }, 
  { "123.45.67.89/8", "::123.0.0.0", TRUE }, 
  { "123.45.67.89/7", "::122.0.0.0", TRUE }, 
  { "123.45.67.89/6", "::120.0.0.0", TRUE }, 
  { "123.45.67.89/5", "::120.0.0.0", TRUE }, 
  { "123.45.67.89/4", "::112.0.0.0", TRUE }, 
  { "123.45.67.89/3", "::96.0.0.0", TRUE }, 
  { "123.45.67.89/2", "::64.0.0.0", TRUE }, 
  { "123.45.67.89/1", "::0.0.0.0", TRUE }, 
  { "123.45.67.89/0", "::0.0.0.0", TRUE }, 

  { "123.45.67.89/32", "::123.45.67.89", TRUE }, 
  { "123.45.67.88/31", "::123.45.67.89", TRUE }, 
  { "123.45.67.88/30", "::123.45.67.89", TRUE }, 
  { "123.45.67.88/29", "::123.45.67.89", TRUE }, 
  { "123.45.67.80/28", "::123.45.67.89", TRUE }, 
  { "123.45.67.64/27", "::123.45.67.89", TRUE }, 
  { "123.45.67.64/26", "::123.45.67.89", TRUE }, 
  { "123.45.67.0/25", "::123.45.67.89", TRUE }, 
  { "123.45.67.0/24", "::123.45.67.89", TRUE }, 
  { "123.45.66.0/23", "::123.45.67.89", TRUE }, 
  { "123.45.64.0/22", "::123.45.67.89", TRUE }, 
  { "123.45.64.0/21", "::123.45.67.89", TRUE }, 
  { "123.45.64.0/20", "::123.45.67.89", TRUE }, 
  { "123.45.64.0/19", "::123.45.67.89", TRUE }, 
  { "123.45.64.0/18", "::123.45.67.89", TRUE }, 
  { "123.45.0.0/17", "::123.45.67.89", TRUE }, 
  { "123.45.0.0/16", "::123.45.67.89", TRUE }, 
  { "123.44.0.0/15", "::123.45.67.89", TRUE }, 
  { "123.44.0.0/14", "::123.45.67.89", TRUE }, 
  { "123.40.0.0/13", "::123.45.67.89", TRUE }, 
  { "123.32.0.0/12", "::123.45.67.89", TRUE }, 
  { "123.32.0.0/11", "::123.45.67.89", TRUE }, 
  { "123.0.0.0/10", "::123.45.67.89", TRUE }, 
  { "123.0.0.0/9", "::123.45.67.89", TRUE }, 
  { "123.0.0.0/8", "::123.45.67.89", TRUE }, 
  { "122.0.0.0/7", "::123.45.67.89", TRUE }, 
  { "120.0.0.0/6", "::123.45.67.89", TRUE }, 
  { "120.0.0.0/5", "::123.45.67.89", TRUE }, 
  { "112.0.0.0/4", "::123.45.67.89", TRUE }, 
  { "96.0.0.0/3", "::123.45.67.89", TRUE }, 
  { "64.0.0.0/2", "::123.45.67.89", TRUE }, 
  { "0.0.0.0/1", "::123.45.67.89", TRUE }, 
  { "0.0.0.0/0", "::123.45.67.89", TRUE }, 

  { "123.45.67.89/32", "::123.45.67.88", FALSE }, 
  { "123.45.67.89/31", "::123.45.67.87", FALSE }, 
  { "123.45.67.89/30", "::123.45.67.87", FALSE }, 
  { "123.45.67.89/29", "::123.45.67.87", FALSE }, 
  { "123.45.67.89/28", "::123.45.67.79", FALSE }, 
  { "123.45.67.89/27", "::123.45.67.63", FALSE }, 
  { "123.45.67.89/26", "::123.45.67.63", FALSE }, 
  { "123.45.67.89/25", "::123.45.66.0", FALSE }, 
  { "123.45.67.89/24", "::123.45.66.0", FALSE }, 
  { "123.45.67.89/23", "::123.45.65.0", FALSE }, 
  { "123.45.67.89/22", "::123.45.63.0", FALSE }, 
  { "123.45.67.89/21", "::123.45.63.0", FALSE }, 
  { "123.45.67.89/20", "::123.45.63.0", FALSE }, 
  { "123.45.67.89/19", "::123.45.63.0", FALSE }, 
  { "123.45.67.89/18", "::123.45.63.0", FALSE }, 
  { "123.45.67.89/17", "::123.44.0.0", FALSE }, 
  { "123.45.67.89/16", "::123.44.0.0", FALSE }, 
  { "123.45.67.89/15", "::123.43.0.0", FALSE }, 
  { "123.45.67.89/14", "::123.43.0.0", FALSE }, 
  { "123.45.67.89/13", "::123.39.0.0", FALSE }, 
  { "123.45.67.89/12", "::123.31.0.0", FALSE }, 
  { "123.45.67.89/11", "::123.31.0.0", FALSE }, 
  { "123.45.67.89/10", "::122.0.0.0", FALSE }, 
  { "123.45.67.89/9", "::122.0.0.0", FALSE }, 
  { "123.45.67.89/8", "::122.0.0.0", FALSE }, 
  { "123.45.67.89/7", "::121.0.0.0", FALSE }, 
  { "123.45.67.89/6", "::119.0.0.0", FALSE }, 
  { "123.45.67.89/5", "::119.0.0.0", FALSE }, 
  { "123.45.67.89/4", "::111.0.0.0", FALSE }, 
  { "123.45.67.89/3", "::95.0.0.0", FALSE }, 
  { "123.45.67.89/2", "::63.0.0.0", FALSE }, 
  { "123.45.67.89/1", "::128.0.0.0", FALSE },

  { "128.128.128.128/32", "::128.128.128.127", FALSE },
  { "128.128.128.128/32", "::128.128.128.128", TRUE },
  { "128.128.128.128/32", "::128.128.128.129", FALSE },
  { "128.128.128.128/31", "::128.128.128.127", FALSE },
  { "128.128.128.128/31", "::128.128.128.128", TRUE },
  { "128.128.128.128/31", "::128.128.128.129", TRUE },
  { "128.128.128.128/31", "::128.128.128.130", FALSE },
  { "128.128.128.128/30", "::128.128.128.127", FALSE },
  { "128.128.128.128/30", "::128.128.128.128", TRUE },
  { "128.128.128.128/30", "::128.128.128.131", TRUE },
  { "128.128.128.128/30", "::128.128.128.132", FALSE },
  { "128.128.128.128/29", "::128.128.128.127", FALSE },
  { "128.128.128.128/29", "::128.128.128.128", TRUE },
  { "128.128.128.128/29", "::128.128.128.135", TRUE },
  { "128.128.128.128/29", "::128.128.128.136", FALSE },
  { "128.128.128.128/28", "::128.128.128.127", FALSE },
  { "128.128.128.128/28", "::128.128.128.128", TRUE },
  { "128.128.128.128/28", "::128.128.128.143", TRUE },
  { "128.128.128.128/28", "::128.128.128.144", FALSE },
  { "128.128.128.128/27", "::128.128.128.127", FALSE },
  { "128.128.128.128/27", "::128.128.128.128", TRUE },
  { "128.128.128.128/27", "::128.128.128.159", TRUE },
  { "128.128.128.128/27", "::128.128.128.160", FALSE },
  { "128.128.128.128/26", "::128.128.128.127", FALSE },
  { "128.128.128.128/26", "::128.128.128.128", TRUE },
  { "128.128.128.128/26", "::128.128.128.191", TRUE },
  { "128.128.128.128/26", "::128.128.128.192", FALSE },
  { "128.128.128.128/25", "::128.128.128.127", FALSE },
  { "128.128.128.128/25", "::128.128.128.128", TRUE },
  { "128.128.128.128/25", "::128.128.128.255", TRUE },
  { "128.128.128.128/25", "::128.128.129.0", FALSE },

  { "128.128.128.0/24", "::128.128.127.0", FALSE },
  { "128.128.128.0/24", "::128.128.128.0", TRUE },
  { "128.128.128.0/24", "::128.128.129.0", FALSE },
  { "128.128.128.0/23", "::128.128.127.0", FALSE },
  { "128.128.128.0/23", "::128.128.128.0", TRUE },
  { "128.128.128.0/23", "::128.128.129.0", TRUE },
  { "128.128.128.0/23", "::128.128.130.0", FALSE },
  { "128.128.128.0/22", "::128.128.127.0", FALSE },
  { "128.128.128.0/22", "::128.128.128.0", TRUE },
  { "128.128.128.0/22", "::128.128.131.0", TRUE },
  { "128.128.128.0/22", "::128.128.132.0", FALSE },
  { "128.128.128.0/21", "::128.128.127.0", FALSE },
  { "128.128.128.0/21", "::128.128.128.0", TRUE },
  { "128.128.128.0/21", "::128.128.135.0", TRUE },
  { "128.128.128.0/21", "::128.128.136.0", FALSE },
  { "128.128.128.0/20", "::128.128.127.0", FALSE },
  { "128.128.128.0/20", "::128.128.128.0", TRUE },
  { "128.128.128.0/20", "::128.128.143.0", TRUE },
  { "128.128.128.0/20", "::128.128.144.0", FALSE },
  { "128.128.128.0/19", "::128.128.127.0", FALSE },
  { "128.128.128.0/19", "::128.128.128.0", TRUE },
  { "128.128.128.0/19", "::128.128.159.0", TRUE },
  { "128.128.128.0/19", "::128.128.160.0", FALSE },
  { "128.128.128.0/18", "::128.128.127.0", FALSE },
  { "128.128.128.0/18", "::128.128.128.0", TRUE },
  { "128.128.128.0/18", "::128.128.191.0", TRUE },
  { "128.128.128.0/18", "::128.128.192.0", FALSE },
  { "128.128.128.0/17", "::128.128.127.0", FALSE },
  { "128.128.128.0/17", "::128.128.128.0", TRUE },
  { "128.128.128.0/17", "::128.128.255.0", TRUE },
  { "128.128.128.0/17", "::128.129.0.0", FALSE },

  { "128.128.0.0/16", "::128.127.0.0", FALSE },
  { "128.128.0.0/16", "::128.128.0.0", TRUE },
  { "128.128.0.0/16", "::128.129.0.0", FALSE },
  { "128.128.0.0/15", "::128.127.0.0", FALSE },
  { "128.128.0.0/15", "::128.128.0.0", TRUE },
  { "128.128.0.0/15", "::128.129.0.0", TRUE },
  { "128.128.0.0/15", "::128.130.0.0", FALSE },
  { "128.128.0.0/14", "::128.127.0.0", FALSE },
  { "128.128.0.0/14", "::128.128.0.0", TRUE },
  { "128.128.0.0/14", "::128.131.0.0", TRUE },
  { "128.128.0.0/14", "::128.132.0.0", FALSE },
  { "128.128.0.0/13", "::128.127.0.0", FALSE },
  { "128.128.0.0/13", "::128.128.0.0", TRUE },
  { "128.128.0.0/13", "::128.135.0.0", TRUE },
  { "128.128.0.0/13", "::128.136.0.0", FALSE },
  { "128.128.0.0/12", "::128.127.0.0", FALSE },
  { "128.128.0.0/12", "::128.128.0.0", TRUE },
  { "128.128.0.0/12", "::128.143.0.0", TRUE },
  { "128.128.0.0/12", "::128.144.0.0", FALSE },
  { "128.128.0.0/11", "::128.127.0.0", FALSE },
  { "128.128.0.0/11", "::128.128.0.0", TRUE },
  { "128.128.0.0/11", "::128.159.0.0", TRUE },
  { "128.128.0.0/11", "::128.160.0.0", FALSE },
  { "128.128.0.0/10", "::128.127.0.0", FALSE },
  { "128.128.0.0/10", "::128.128.0.0", TRUE },
  { "128.128.0.0/10", "::128.191.0.0", TRUE },
  { "128.128.0.0/10", "::128.192.0.0", FALSE },
  { "128.128.0.0/9", "::128.127.0.0", FALSE },
  { "128.128.0.0/9", "::128.128.0.0", TRUE },
  { "128.128.0.0/9", "::128.255.0.0", TRUE },
  { "128.128.0.0/9", "::129.0.0.0", FALSE },

  { "128.0.0.0/8", "::127.0.0.0", FALSE },
  { "128.0.0.0/8", "::128.0.0.0", TRUE },
  { "128.0.0.0/8", "::129.0.0.0", FALSE },
  { "128.0.0.0/7", "::127.0.0.0", FALSE },
  { "128.0.0.0/7", "::128.0.0.0", TRUE },
  { "128.0.0.0/7", "::129.0.0.0", TRUE },
  { "128.0.0.0/7", "::130.0.0.0", FALSE },
  { "128.0.0.0/6", "::127.0.0.0", FALSE },
  { "128.0.0.0/6", "::128.0.0.0", TRUE },
  { "128.0.0.0/6", "::131.0.0.0", TRUE },
  { "128.0.0.0/6", "::132.0.0.0", FALSE },
  { "128.0.0.0/5", "::127.0.0.0", FALSE },
  { "128.0.0.0/5", "::128.0.0.0", TRUE },
  { "128.0.0.0/5", "::135.0.0.0", TRUE },
  { "128.0.0.0/5", "::136.0.0.0", FALSE },
  { "128.0.0.0/4", "::127.0.0.0", FALSE },
  { "128.0.0.0/4", "::128.0.0.0", TRUE },
  { "128.0.0.0/4", "::143.0.0.0", TRUE },
  { "128.0.0.0/4", "::144.0.0.0", FALSE },
  { "128.0.0.0/3", "::127.0.0.0", FALSE },
  { "128.0.0.0/3", "::128.0.0.0", TRUE },
  { "128.0.0.0/3", "::159.0.0.0", TRUE },
  { "128.0.0.0/3", "::160.0.0.0", FALSE },
  { "128.0.0.0/2", "::127.0.0.0", FALSE },
  { "128.0.0.0/2", "::128.0.0.0", TRUE },
  { "128.0.0.0/2", "::191.0.0.0", TRUE },
  { "128.0.0.0/2", "::192.0.0.0", FALSE },
  { "128.0.0.0/1", "::127.0.0.0", FALSE },
  { "128.0.0.0/1", "::128.0.0.0", TRUE },
  { "128.0.0.0/1", "::255.0.0.0", TRUE },
  { "128.0.0.0/1", "::127.0.0.0", FALSE },

  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "128.128.128.128", TRUE },
  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "129.128.128.128", FALSE },
  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "130.128.128.128", TRUE },
  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "131.128.128.128", FALSE },
  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "132.128.128.128", TRUE },
  { "128.0.0.0/8,130.0.0.0/8,132.0.0.0/8", "133.128.128.128", FALSE },

  { "f000:1234:5678:9abc:deff:edcb:a987:6543/124",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", TRUE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6540/124",
    "f000:1234:5678:9abc:deff:edcb:a987:6540", TRUE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6540/124",
    "f000:1234:5678:9abc:deff:edcb:a987:653f", FALSE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6540/124",
    "f000:1234:5678:9abc:deff:edcb:a987:6550", FALSE },

  { "f000:1234:5678:9abc:deff:edcb:a987:6543/128",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", TRUE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6542/127",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", TRUE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6542/127",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", TRUE },
  { "f000:1234:5678:9abc:deff:edcb:a987:6542/127",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", TRUE },

  { "8000:1234:5678:9abc:deff:edcb:a987:6543/128",
    "8000:1234:5678:9abc:deff:edcb:a987:6542", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/128",
    "8000:1234:5678:9abc:deff:edcb:a987:6543", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/128",
    "8000:1234:5678:9abc:deff:edcb:a987:6543", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/128",
    "8000:1234:5678:9abc:deff:edcb:a987:6544", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/127",
    "8000:1234:5678:9abc:deff:edcb:a987:6541", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/127",
    "8000:1234:5678:9abc:deff:edcb:a987:6542", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/127",
    "8000:1234:5678:9abc:deff:edcb:a987:6543", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/127",
    "8000:1234:5678:9abc:deff:edcb:a987:6544", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/126",
    "8000:1234:5678:9abc:deff:edcb:a987:653f", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/126",
    "8000:1234:5678:9abc:deff:edcb:a987:6540", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/126",
    "8000:1234:5678:9abc:deff:edcb:a987:6543", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/126",
    "8000:1234:5678:9abc:deff:edcb:a987:6544", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/125",
    "8000:1234:5678:9abc:deff:edcb:a987:653f", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/125",
    "8000:1234:5678:9abc:deff:edcb:a987:6540", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/125",
    "8000:1234:5678:9abc:deff:edcb:a987:6547", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/125",
    "8000:1234:5678:9abc:deff:edcb:a987:6548", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/124",
    "8000:1234:5678:9abc:deff:edcb:a987:653f", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/124",
    "8000:1234:5678:9abc:deff:edcb:a987:6540", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/124",
    "8000:1234:5678:9abc:deff:edcb:a987:654f", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/124",
    "8000:1234:5678:9abc:deff:edcb:a987:6550", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/123",
    "8000:1234:5678:9abc:deff:edcb:a987:653f", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/123",
    "8000:1234:5678:9abc:deff:edcb:a987:6540", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/123",
    "8000:1234:5678:9abc:deff:edcb:a987:655f", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/123",
    "8000:1234:5678:9abc:deff:edcb:a987:6560", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/122",
    "8000:1234:5678:9abc:deff:edcb:a987:653f", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/122",
    "8000:1234:5678:9abc:deff:edcb:a987:6540", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/122",
    "8000:1234:5678:9abc:deff:edcb:a987:657f", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/122",
    "8000:1234:5678:9abc:deff:edcb:a987:6580", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/121",
    "8000:1234:5678:9abc:deff:edcb:a987:64ff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/121",
    "8000:1234:5678:9abc:deff:edcb:a987:6500", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/121",
    "8000:1234:5678:9abc:deff:edcb:a987:657f", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/121",
    "8000:1234:5678:9abc:deff:edcb:a987:6580", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/120",
    "8000:1234:5678:9abc:deff:edcb:a987:64ff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/120",
    "8000:1234:5678:9abc:deff:edcb:a987:6500", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/120",
    "8000:1234:5678:9abc:deff:edcb:a987:65ff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/120",
    "8000:1234:5678:9abc:deff:edcb:a987:6600", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/119",
    "8000:1234:5678:9abc:deff:edcb:a987:63ff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/119",
    "8000:1234:5678:9abc:deff:edcb:a987:6400", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/119",
    "8000:1234:5678:9abc:deff:edcb:a987:65ff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/119",
    "8000:1234:5678:9abc:deff:edcb:a987:6600", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/118",
    "8000:1234:5678:9abc:deff:edcb:a987:63ff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/118",
    "8000:1234:5678:9abc:deff:edcb:a987:6400", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/118",
    "8000:1234:5678:9abc:deff:edcb:a987:67ff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/118",
    "8000:1234:5678:9abc:deff:edcb:a987:6800", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/117",
    "8000:1234:5678:9abc:deff:edcb:a987:5fff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/117",
    "8000:1234:5678:9abc:deff:edcb:a987:6000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/117",
    "8000:1234:5678:9abc:deff:edcb:a987:67ff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/117",
    "8000:1234:5678:9abc:deff:edcb:a987:6800", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/116",
    "8000:1234:5678:9abc:deff:edcb:a987:5fff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/116",
    "8000:1234:5678:9abc:deff:edcb:a987:6000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/116",
    "8000:1234:5678:9abc:deff:edcb:a987:6fff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/116",
    "8000:1234:5678:9abc:deff:edcb:a987:7000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/115",
    "8000:1234:5678:9abc:deff:edcb:a987:5fff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/115",
    "8000:1234:5678:9abc:deff:edcb:a987:6000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/115",
    "8000:1234:5678:9abc:deff:edcb:a987:7fff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/115",
    "8000:1234:5678:9abc:deff:edcb:a987:8000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/114",
    "8000:1234:5678:9abc:deff:edcb:a987:3fff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/114",
    "8000:1234:5678:9abc:deff:edcb:a987:4000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/114",
    "8000:1234:5678:9abc:deff:edcb:a987:7fff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/114",
    "8000:1234:5678:9abc:deff:edcb:a987:8000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/113",
    "8000:1234:5678:9abc:deff:edcb:a986:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/113",
    "8000:1234:5678:9abc:deff:edcb:a987:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/113",
    "8000:1234:5678:9abc:deff:edcb:a987:7fff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/113",
    "8000:1234:5678:9abc:deff:edcb:a987:8000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/112",
    "8000:1234:5678:9abc:deff:edcb:a986:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/112",
    "8000:1234:5678:9abc:deff:edcb:a987:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/112",
    "8000:1234:5678:9abc:deff:edcb:a987:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/112",
    "8000:1234:5678:9abc:deff:edcb:a988:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/111",
    "8000:1234:5678:9abc:deff:edcb:a985:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/111",
    "8000:1234:5678:9abc:deff:edcb:a986:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/111",
    "8000:1234:5678:9abc:deff:edcb:a987:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/111",
    "8000:1234:5678:9abc:deff:edcb:a988:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/110",
    "8000:1234:5678:9abc:deff:edcb:a983:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/110",
    "8000:1234:5678:9abc:deff:edcb:a984:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/110",
    "8000:1234:5678:9abc:deff:edcb:a987:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/110",
    "8000:1234:5678:9abc:deff:edcb:a988:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/109",
    "8000:1234:5678:9abc:deff:edcb:a97f:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/109",
    "8000:1234:5678:9abc:deff:edcb:a980:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/109",
    "8000:1234:5678:9abc:deff:edcb:a987:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/109",
    "8000:1234:5678:9abc:deff:edcb:a988:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/108",
    "8000:1234:5678:9abc:deff:edcb:a97f:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/108",
    "8000:1234:5678:9abc:deff:edcb:a980:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/108",
    "8000:1234:5678:9abc:deff:edcb:a98f:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/108",
    "8000:1234:5678:9abc:deff:edcb:a990:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/107",
    "8000:1234:5678:9abc:deff:edcb:a97f:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/107",
    "8000:1234:5678:9abc:deff:edcb:a980:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/107",
    "8000:1234:5678:9abc:deff:edcb:a99f:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/107",
    "8000:1234:5678:9abc:deff:edcb:a9a0:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/106",
    "8000:1234:5678:9abc:deff:edcb:a97f:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/106",
    "8000:1234:5678:9abc:deff:edcb:a980:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/106",
    "8000:1234:5678:9abc:deff:edcb:a9bf:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/106",
    "8000:1234:5678:9abc:deff:edcb:a9c0:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/105",
    "8000:1234:5678:9abc:deff:edcb:a97f:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/105",
    "8000:1234:5678:9abc:deff:edcb:a980:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/105",
    "8000:1234:5678:9abc:deff:edcb:a9ff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/105",
    "8000:1234:5678:9abc:deff:edcb:aa00:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/104",
    "8000:1234:5678:9abc:deff:edcb:a8ff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/104",
    "8000:1234:5678:9abc:deff:edcb:a900:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/104",
    "8000:1234:5678:9abc:deff:edcb:a9ff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/104",
    "8000:1234:5678:9abc:deff:edcb:aa00:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/103",
    "8000:1234:5678:9abc:deff:edcb:a7ff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/103",
    "8000:1234:5678:9abc:deff:edcb:a800:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/103",
    "8000:1234:5678:9abc:deff:edcb:a9ff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/103",
    "8000:1234:5678:9abc:deff:edcb:aa00:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/102",
    "8000:1234:5678:9abc:deff:edcb:a7ff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/102",
    "8000:1234:5678:9abc:deff:edcb:a800:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/102",
    "8000:1234:5678:9abc:deff:edcb:abff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/102",
    "8000:1234:5678:9abc:deff:edcb:ac00:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/101",
    "8000:1234:5678:9abc:deff:edcb:a7ff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/101",
    "8000:1234:5678:9abc:deff:edcb:a800:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/101",
    "8000:1234:5678:9abc:deff:edcb:afff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/101",
    "8000:1234:5678:9abc:deff:edcb:b000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/100",
    "8000:1234:5678:9abc:deff:edcb:9fff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/100",
    "8000:1234:5678:9abc:deff:edcb:a000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/100",
    "8000:1234:5678:9abc:deff:edcb:afff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/100",
    "8000:1234:5678:9abc:deff:edcb:b000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/99",
    "8000:1234:5678:9abc:deff:edcb:9fff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/99",
    "8000:1234:5678:9abc:deff:edcb:a000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/99",
    "8000:1234:5678:9abc:deff:edcb:bfff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/99",
    "8000:1234:5678:9abc:deff:edcb:c000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/98",
    "8000:1234:5678:9abc:deff:edcb:7fff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/98",
    "8000:1234:5678:9abc:deff:edcb:8000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/98",
    "8000:1234:5678:9abc:deff:edcb:bfff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/98",
    "8000:1234:5678:9abc:deff:edcb:c000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/97",
    "8000:1234:5678:9abc:deff:edcb:7fff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/97",
    "8000:1234:5678:9abc:deff:edcb:8000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/97",
    "8000:1234:5678:9abc:deff:edcb:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/97",
    "8000:1234:5678:9abc:deff:edcc:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/96",
    "8000:1234:5678:9abc:deff:edca:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/96",
    "8000:1234:5678:9abc:deff:edcb:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/96",
    "8000:1234:5678:9abc:deff:edcb:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/96",
    "8000:1234:5678:9abc:deff:edcc:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/95",
    "8000:1234:5678:9abc:deff:edc9:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/95",
    "8000:1234:5678:9abc:deff:edca:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/95",
    "8000:1234:5678:9abc:deff:edcb:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/95",
    "8000:1234:5678:9abc:deff:edcc:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/94",
    "8000:1234:5678:9abc:deff:edc7:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/94",
    "8000:1234:5678:9abc:deff:edc8:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/94",
    "8000:1234:5678:9abc:deff:edcb:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/94",
    "8000:1234:5678:9abc:deff:edcc:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/93",
    "8000:1234:5678:9abc:deff:edc7:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/93",
    "8000:1234:5678:9abc:deff:edc8:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/93",
    "8000:1234:5678:9abc:deff:edcf:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/93",
    "8000:1234:5678:9abc:deff:edd0:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/92",
    "8000:1234:5678:9abc:deff:edbf:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/92",
    "8000:1234:5678:9abc:deff:edc0:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/92",
    "8000:1234:5678:9abc:deff:edcf:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/92",
    "8000:1234:5678:9abc:deff:edd0:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/91",
    "8000:1234:5678:9abc:deff:edbf:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/91",
    "8000:1234:5678:9abc:deff:edc0:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/91",
    "8000:1234:5678:9abc:deff:eddf:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/91",
    "8000:1234:5678:9abc:deff:ede0:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/90",
    "8000:1234:5678:9abc:deff:edbf:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/90",
    "8000:1234:5678:9abc:deff:edc0:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/90",
    "8000:1234:5678:9abc:deff:edff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/90",
    "8000:1234:5678:9abc:deff:ee00:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/89",
    "8000:1234:5678:9abc:deff:ed7f:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/89",
    "8000:1234:5678:9abc:deff:ed80:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/89",
    "8000:1234:5678:9abc:deff:edff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/89",
    "8000:1234:5678:9abc:deff:ee00:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/88",
    "8000:1234:5678:9abc:deff:ecff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/88",
    "8000:1234:5678:9abc:deff:ed00:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/88",
    "8000:1234:5678:9abc:deff:edff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/88",
    "8000:1234:5678:9abc:deff:ee00:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/87",
    "8000:1234:5678:9abc:deff:ebff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/87",
    "8000:1234:5678:9abc:deff:ec00:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/87",
    "8000:1234:5678:9abc:deff:edff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/87",
    "8000:1234:5678:9abc:deff:ee00:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/86",
    "8000:1234:5678:9abc:deff:ebff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/86",
    "8000:1234:5678:9abc:deff:ec00:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/86",
    "8000:1234:5678:9abc:deff:efff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/86",
    "8000:1234:5678:9abc:deff:f000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/85",
    "8000:1234:5678:9abc:deff:e7ff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/85",
    "8000:1234:5678:9abc:deff:e800:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/85",
    "8000:1234:5678:9abc:deff:efff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/85",
    "8000:1234:5678:9abc:deff:f000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/84",
    "8000:1234:5678:9abc:deff:dfff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/84",
    "8000:1234:5678:9abc:deff:e000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/84",
    "8000:1234:5678:9abc:deff:efff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/84",
    "8000:1234:5678:9abc:deff:f000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/83",
    "8000:1234:5678:9abc:deff:dfff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/83",
    "8000:1234:5678:9abc:deff:e000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/83",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/83",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/82",
    "8000:1234:5678:9abc:deff:bfff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/82",
    "8000:1234:5678:9abc:deff:c000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/82",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/82",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/81",
    "8000:1234:5678:9abc:deff:7fff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/81",
    "8000:1234:5678:9abc:deff:8000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/81",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/81",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/80",
    "8000:1234:5678:9abc:defe:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/80",
    "8000:1234:5678:9abc:deff:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/80",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/80",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/79",
    "8000:1234:5678:9abc:defd:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/79",
    "8000:1234:5678:9abc:defe:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/79",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/79",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/78",
    "8000:1234:5678:9abc:defb:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/78",
    "8000:1234:5678:9abc:defc:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/78",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/78",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/77",
    "8000:1234:5678:9abc:def7:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/77",
    "8000:1234:5678:9abc:def8:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/77",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/77",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/76",
    "8000:1234:5678:9abc:deef:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/76",
    "8000:1234:5678:9abc:def0:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/76",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/76",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/75",
    "8000:1234:5678:9abc:dedf:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/75",
    "8000:1234:5678:9abc:dee0:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/75",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/75",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/74",
    "8000:1234:5678:9abc:debf:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/74",
    "8000:1234:5678:9abc:dec0:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/74",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/74",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/73",
    "8000:1234:5678:9abc:de7f:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/73",
    "8000:1234:5678:9abc:de80:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/73",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/73",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/72",
    "8000:1234:5678:9abc:ddff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/72",
    "8000:1234:5678:9abc:de00:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/72",
    "8000:1234:5678:9abc:deff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/72",
    "8000:1234:5678:9abc:df00:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/71",
    "8000:1234:5678:9abc:ddff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/71",
    "8000:1234:5678:9abc:de00:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/71",
    "8000:1234:5678:9abc:dfff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/71",
    "8000:1234:5678:9abc:e000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/70",
    "8000:1234:5678:9abc:dbff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/70",
    "8000:1234:5678:9abc:dc00:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/70",
    "8000:1234:5678:9abc:dfff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/70",
    "8000:1234:5678:9abc:e000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/69",
    "8000:1234:5678:9abc:d7ff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/69",
    "8000:1234:5678:9abc:d800:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/69",
    "8000:1234:5678:9abc:dfff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/69",
    "8000:1234:5678:9abc:e000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/68",
    "8000:1234:5678:9abc:cfff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/68",
    "8000:1234:5678:9abc:d000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/68",
    "8000:1234:5678:9abc:dfff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/68",
    "8000:1234:5678:9abc:e000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/67",
    "8000:1234:5678:9abc:bfff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/67",
    "8000:1234:5678:9abc:c000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/67",
    "8000:1234:5678:9abc:dfff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/67",
    "8000:1234:5678:9abc:e000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/66",
    "8000:1234:5678:9abc:bfff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/66",
    "8000:1234:5678:9abc:c000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/66",
    "8000:1234:5678:9abc:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/66",
    "8000:1234:5678:9abd:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/65",
    "8000:1234:5678:9abc:7fff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/65",
    "8000:1234:5678:9abc:8000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/65",
    "8000:1234:5678:9abc:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/65",
    "8000:1234:5678:9abd:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/64",
    "8000:1234:5678:9abb:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/64",
    "8000:1234:5678:9abc:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/64",
    "8000:1234:5678:9abc:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/64",
    "8000:1234:5678:9abd:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/63",
    "8000:1234:5678:9abb:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/63",
    "8000:1234:5678:9abc:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/63",
    "8000:1234:5678:9abd:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/63",
    "8000:1234:5678:9abe:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/62",
    "8000:1234:5678:9abb:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/62",
    "8000:1234:5678:9abc:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/62",
    "8000:1234:5678:9abf:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/62",
    "8000:1234:5678:9ac0:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/61",
    "8000:1234:5678:9ab7:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/61",
    "8000:1234:5678:9ab8:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/61",
    "8000:1234:5678:9abf:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/61",
    "8000:1234:5678:9ac0:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/60",
    "8000:1234:5678:9aaf:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/60",
    "8000:1234:5678:9ab0:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/60",
    "8000:1234:5678:9abf:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/60",
    "8000:1234:5678:9ac0:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/59",
    "8000:1234:5678:9a9f:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/59",
    "8000:1234:5678:9aa0:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/59",
    "8000:1234:5678:9abf:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/59",
    "8000:1234:5678:9ac0:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/58",
    "8000:1234:5678:9a7f:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/58",
    "8000:1234:5678:9a80:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/58",
    "8000:1234:5678:9abf:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/58",
    "8000:1234:5678:9ac0:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/57",
    "8000:1234:5678:9a7f:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/57",
    "8000:1234:5678:9a80:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/57",
    "8000:1234:5678:9aff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/57",
    "8000:1234:5678:9b00:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/56",
    "8000:1234:5678:99ff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/56",
    "8000:1234:5678:9a00:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/56",
    "8000:1234:5678:9aff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/56",
    "8000:1234:5678:9b00:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/55",
    "8000:1234:5678:99ff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/55",
    "8000:1234:5678:9a00:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/55",
    "8000:1234:5678:9bff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/55",
    "8000:1234:5678:9c00:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/54",
    "8000:1234:5678:97ff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/54",
    "8000:1234:5678:9800:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/54",
    "8000:1234:5678:9bff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/54",
    "8000:1234:5678:9c00:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/53",
    "8000:1234:5678:97ff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/53",
    "8000:1234:5678:9800:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/53",
    "8000:1234:5678:9fff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/53",
    "8000:1234:5678:a000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/52",
    "8000:1234:5678:8fff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/52",
    "8000:1234:5678:9000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/52",
    "8000:1234:5678:9fff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/52",
    "8000:1234:5678:a000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/51",
    "8000:1234:5678:7fff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/51",
    "8000:1234:5678:8000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/51",
    "8000:1234:5678:9fff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/51",
    "8000:1234:5678:a000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/50",
    "8000:1234:5678:7fff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/50",
    "8000:1234:5678:8000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/50",
    "8000:1234:5678:bfff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/50",
    "8000:1234:5678:c000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/49",
    "8000:1234:5678:7fff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/49",
    "8000:1234:5678:8000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/49",
    "8000:1234:5678:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/49",
    "8000:1234:5679:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/48",
    "8000:1234:5677:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/48",
    "8000:1234:5678:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/48",
    "8000:1234:5678:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/48",
    "8000:1234:5679:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/47",
    "8000:1234:5677:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/47",
    "8000:1234:5678:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/47",
    "8000:1234:5679:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/47",
    "8000:1234:567a:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/46",
    "8000:1234:5677:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/46",
    "8000:1234:5678:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/46",
    "8000:1234:567b:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/46",
    "8000:1234:567c:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/45",
    "8000:1234:5677:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/45",
    "8000:1234:5678:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/45",
    "8000:1234:567f:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/45",
    "8000:1234:5680:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/44",
    "8000:1234:566f:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/44",
    "8000:1234:5670:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/44",
    "8000:1234:567f:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/44",
    "8000:1234:5680:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/43",
    "8000:1234:565f:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/43",
    "8000:1234:5660:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/43",
    "8000:1234:567f:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/43",
    "8000:1234:5680:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/42",
    "8000:1234:563f:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/42",
    "8000:1234:5640:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/42",
    "8000:1234:567f:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/42",
    "8000:1234:5680:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/41",
    "8000:1234:55ff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/41",
    "8000:1234:5600:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/41",
    "8000:1234:567f:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/41",
    "8000:1234:5680:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/40",
    "8000:1234:55ff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/40",
    "8000:1234:5600:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/40",
    "8000:1234:56ff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/40",
    "8000:1234:5700:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/39",
    "8000:1234:55ff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/39",
    "8000:1234:5600:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/39",
    "8000:1234:57ff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/39",
    "8000:1234:5800:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/38",
    "8000:1234:53ff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/38",
    "8000:1234:5400:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/38",
    "8000:1234:57ff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/38",
    "8000:1234:5800:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/37",
    "8000:1234:4fff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/37",
    "8000:1234:5000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/37",
    "8000:1234:57ff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/37",
    "8000:1234:5800:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/36",
    "8000:1234:4fff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/36",
    "8000:1234:5000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/36",
    "8000:1234:5fff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/36",
    "8000:1234:6000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/35",
    "8000:1234:3fff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/35",
    "8000:1234:4000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/35",
    "8000:1234:5fff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/35",
    "8000:1234:6000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/34",
    "8000:1234:3fff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/34",
    "8000:1234:4000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/34",
    "8000:1234:7fff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/34",
    "8000:1234:8000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/33",
    "8000:1233:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/33",
    "8000:1234:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/33",
    "8000:1234:7fff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/33",
    "8000:1234:8000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/32",
    "8000:1233:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/32",
    "8000:1234:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/32",
    "8000:1234:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/32",
    "8000:1235:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/31",
    "8000:1233:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/31",
    "8000:1234:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/31",
    "8000:1235:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/31",
    "8000:1236:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/30",
    "8000:1233:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/30",
    "8000:1234:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/30",
    "8000:1237:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/30",
    "8000:1238:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/29",
    "8000:122f:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/29",
    "8000:1230:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/29",
    "8000:1237:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/29",
    "8000:1238:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/28",
    "8000:122f:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/28",
    "8000:1230:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/28",
    "8000:123f:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/28",
    "8000:1240:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/27",
    "8000:121f:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/27",
    "8000:1220:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/27",
    "8000:123f:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/27",
    "8000:1240:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/26",
    "8000:11ff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/26",
    "8000:1200:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/26",
    "8000:123f:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/26",
    "8000:1240:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/25",
    "8000:11ff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/25",
    "8000:1200:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/25",
    "8000:127f:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/25",
    "8000:1280:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/24",
    "8000:11ff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/24",
    "8000:1200:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/24",
    "8000:12ff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/24",
    "8000:1300:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/23",
    "8000:11ff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/23",
    "8000:1200:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/23",
    "8000:13ff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/23",
    "8000:1400:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/22",
    "8000:0fff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/22",
    "8000:1000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/22",
    "8000:13ff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/22",
    "8000:1400:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/21",
    "8000:0fff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/21",
    "8000:1000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/21",
    "8000:17ff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/21",
    "8000:1800:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/20",
    "8000:0fff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/20",
    "8000:1000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/20",
    "8000:1fff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/20",
    "8000:2000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/19",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/19",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/19",
    "8000:1fff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/19",
    "8000:2000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/18",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/18",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/18",
    "8000:3fff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/18",
    "8000:4000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/17",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/17",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/17",
    "8000:7fff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/17",
    "8000:8000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/16",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/16",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/16",
    "8000:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/16",
    "8001:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/15",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/15",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/15",
    "8001:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/15",
    "8002:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/14",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/14",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/14",
    "8003:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/14",
    "8004:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/13",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/13",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/13",
    "8007:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/13",
    "8008:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/12",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/12",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/12",
    "800f:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/12",
    "8010:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/11",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/11",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/11",
    "801f:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/11",
    "8020:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/10",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/10",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/10",
    "803f:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/10",
    "8040:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/9",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/9",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/9",
    "807f:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/9",
    "8080:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/8",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/8",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/8",
    "80ff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/8",
    "8100:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/7",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/7",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/7",
    "81ff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/7",
    "8200:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/6",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/6",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/6",
    "83ff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/6",
    "8400:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/5",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/5",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/5",
    "87ff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/5",
    "8800:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/4",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/4",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/4",
    "8fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/4",
    "9000:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/3",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/3",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/3",
    "9fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/3",
    "a000:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/2",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/2",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/2",
    "bfff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/2",
    "c000:0000:0000:0000:0000:0000:0000:0000", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/1",
    "7fff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", FALSE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/1",
    "8000:0000:0000:0000:0000:0000:0000:0000", TRUE }, 
  { "8000:1234:5678:9abc:deff:edcb:a987:6543/1",
    "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", TRUE }
};

typedef struct SshCompareTestRec {
  char *ip1;
  char *ip2;
  int value;
} *SshCompareTest, SshCompareTestStruct;

SshCompareTestStruct compare_tests[] = {
  { "0.0.0.0", "0.0.0.0", 0 },
  { "0.0.0.0", "0.0.0.1", -1 },
  { "0.0.0.1", "0.0.0.0", 1 },

  { "255.255.255.0", "255.255.255.0", 0 },
  { "255.255.255.0", "255.255.255.1", -1 },
  { "255.255.255.1", "255.255.255.0", 1 },

  { "128.128.128.0", "128.128.128.0", 0 },
  { "128.128.128.0", "128.128.128.1", -1 },
  { "128.128.128.1", "128.128.128.0", 1 },

  { "123.45.67.0", "123.45.67.0", 0 },
  { "123.45.67.0", "123.45.67.1", -1 },
  { "123.45.67.1", "123.45.67.0", 1 },

  { "52.22.12.0", "52.22.12.0", 0 },
  { "52.22.12.0", "52.22.12.1", -1 },
  { "52.22.12.1", "52.22.12.0", 1 },

  { "0.0.0.0", "0.0.0.0", 0 },
  { "0.0.0.0", "0.0.1.0", -1 },
  { "0.0.1.0", "0.0.0.0", 1 },

  { "255.255.0.255", "255.255.0.255", 0 },
  { "255.255.0.255", "255.255.1.255", -1 },
  { "255.255.1.255", "255.255.0.255", 1 },

  { "128.128.0.128", "128.128.0.128", 0 },
  { "128.128.0.128", "128.128.1.128", -1 },
  { "128.128.1.128", "128.128.0.128", 1 },

  { "123.45.0.67", "123.45.0.67", 0 },
  { "123.45.0.67", "123.45.1.67", -1 },
  { "123.45.1.67", "123.45.0.67", 1 },

  { "52.22.0.12", "52.22.0.12", 0 },
  { "52.22.0.12", "52.22.1.12", -1 },
  { "52.22.1.12", "52.22.0.12", 1 },

  { "::0.0.0.0", "0.0.0.0", 0 },
  { "::0.0.0.0", "0.0.0.1", -1 },
  { "::0.0.0.1", "0.0.0.0", 1 },

  { "::255.255.255.0", "255.255.255.0", 0 },
  { "::255.255.255.0", "255.255.255.1", -1 },
  { "::255.255.255.1", "255.255.255.0", 1 },

  { "::128.128.128.0", "128.128.128.0", 0 },
  { "::128.128.128.0", "128.128.128.1", -1 },
  { "::128.128.128.1", "128.128.128.0", 1 },

  { "::123.45.67.0", "123.45.67.0", 0 },
  { "::123.45.67.0", "123.45.67.1", -1 },
  { "::123.45.67.1", "123.45.67.0", 1 },

  { "::52.22.12.0", "52.22.12.0", 0 },
  { "::52.22.12.0", "52.22.12.1", -1 },
  { "::52.22.12.1", "52.22.12.0", 1 },

  { "::0.0.0.0", "0.0.0.0", 0 },
  { "::0.0.0.0", "0.0.1.0", -1 },
  { "::0.0.1.0", "0.0.0.0", 1 },

  { "::255.255.0.255", "255.255.0.255", 0 },
  { "::255.255.0.255", "255.255.1.255", -1 },
  { "::255.255.1.255", "255.255.0.255", 1 },

  { "::128.128.0.128", "128.128.0.128", 0 },
  { "::128.128.0.128", "128.128.1.128", -1 },
  { "::128.128.1.128", "128.128.0.128", 1 },

  { "::123.45.0.67", "123.45.0.67", 0 },
  { "::123.45.0.67", "123.45.1.67", -1 },
  { "::123.45.1.67", "123.45.0.67", 1 },

  { "::52.22.0.12", "52.22.0.12", 0 },
  { "::52.22.0.12", "52.22.1.12", -1 },
  { "::52.22.1.12", "52.22.0.12", 1 },

  { "0.0.0.0", "::0.0.0.0", 0 },
  { "0.0.0.0", "::0.0.0.1", -1 },
  { "0.0.0.1", "::0.0.0.0", 1 },

  { "255.255.255.0", "::255.255.255.0", 0 },
  { "255.255.255.0", "::255.255.255.1", -1 },
  { "255.255.255.1", "::255.255.255.0", 1 },

  { "128.128.128.0", "::128.128.128.0", 0 },
  { "128.128.128.0", "::128.128.128.1", -1 },
  { "128.128.128.1", "::128.128.128.0", 1 },

  { "123.45.67.0", "::123.45.67.0", 0 },
  { "123.45.67.0", "::123.45.67.1", -1 },
  { "123.45.67.1", "::123.45.67.0", 1 },

  { "52.22.12.0", "::52.22.12.0", 0 },
  { "52.22.12.0", "::52.22.12.1", -1 },
  { "52.22.12.1", "::52.22.12.0", 1 },

  { "0.0.0.0", "::0.0.0.0", 0 },
  { "0.0.0.0", "::0.0.1.0", -1 },
  { "0.0.1.0", "::0.0.0.0", 1 },

  { "255.255.0.255", "::255.255.0.255", 0 },
  { "255.255.0.255", "::255.255.1.255", -1 },
  { "255.255.1.255", "::255.255.0.255", 1 },

  { "128.128.0.128", "::128.128.0.128", 0 },
  { "128.128.0.128", "::128.128.1.128", -1 },
  { "128.128.1.128", "::128.128.0.128", 1 },

  { "123.45.0.67", "::123.45.0.67", 0 },
  { "123.45.0.67", "::123.45.1.67", -1 },
  { "123.45.1.67", "::123.45.0.67", 1 },

  { "52.22.0.12", "::52.22.0.12", 0 },
  { "52.22.0.12", "::52.22.1.12", -1 },
  { "52.22.1.12", "::52.22.0.12", 1 },

  { "::0.0.0.0", "::0.0.0.0", 0 },
  { "::0.0.0.0", "::0.0.0.1", -1 },
  { "::0.0.0.1", "::0.0.0.0", 1 },

  { "::255.255.255.0", "::255.255.255.0", 0 },
  { "::255.255.255.0", "::255.255.255.1", -1 },
  { "::255.255.255.1", "::255.255.255.0", 1 },

  { "::128.128.128.0", "::128.128.128.0", 0 },
  { "::128.128.128.0", "::128.128.128.1", -1 },
  { "::128.128.128.1", "::128.128.128.0", 1 },

  { "::123.45.67.0", "::123.45.67.0", 0 },
  { "::123.45.67.0", "::123.45.67.1", -1 },
  { "::123.45.67.1", "::123.45.67.0", 1 },

  { "::52.22.12.0", "::52.22.12.0", 0 },
  { "::52.22.12.0", "::52.22.12.1", -1 },
  { "::52.22.12.1", "::52.22.12.0", 1 },

  { "::0.0.0.0", "::0.0.0.0", 0 },
  { "::0.0.0.0", "::0.0.1.0", -1 },
  { "::0.0.1.0", "::0.0.0.0", 1 },

  { "::255.255.0.255", "::255.255.0.255", 0 },
  { "::255.255.0.255", "::255.255.1.255", -1 },
  { "::255.255.1.255", "::255.255.0.255", 1 },

  { "::128.128.0.128", "::128.128.0.128", 0 },
  { "::128.128.0.128", "::128.128.1.128", -1 },
  { "::128.128.1.128", "::128.128.0.128", 1 },

  { "::123.45.0.67", "::123.45.0.67", 0 },
  { "::123.45.0.67", "::123.45.1.67", -1 },
  { "::123.45.1.67", "::123.45.0.67", 1 },

  { "::52.22.0.12", "::52.22.0.12", 0 },
  { "::52.22.0.12", "::52.22.1.12", -1 },
  { "::52.22.1.12", "::52.22.0.12", 1 },

  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6543", 0 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6542", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6533", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6443", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a977:6543", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:eccb:a987:6543", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:8abc:deff:edcb:a987:6543", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "e000:1234:5678:9abc:deff:edcb:a987:6543", 1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6544", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a987:6553", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:a988:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:edcb:b987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:deff:eddb:a987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9abc:df00:edcb:a987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5678:9adc:deff:edcb:a987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:1234:5679:9abc:deff:edcb:a987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f000:2234:5678:9abc:deff:edcb:a987:6543", -1 },
  { "f000:1234:5678:9abc:deff:edcb:a987:6543",
    "f100:1234:5678:9abc:deff:edcb:a987:6543", -1 }
};


int main (int ac, char *av[])
{
  SshIpAddrStruct   addr, addr2, addr3;
  char        * addr_1 = "192.168.2.69";
  char        * mask_1 = "192.168.2.69/24";
  char        *mask_2 = "255.255.255.0";
  unsigned char        buf[512], buf2[512];
  int i, k, n, alen;
  Boolean     ret, b, b2;
  SshUInt32 masklen, masklen2;

  if (ac > 1 && strcmp(av[1], "-v") == 0)
    verbose_flag = TRUE;

  srandom((SshUInt32)ssh_time()); 

  /* addr_1 */
  FUT(ssh_ipaddr_parse(&addr, addr_1));
  ssh_ipaddr_print(&addr, (char *)buf, sizeof(buf));
  SEQ(addr_1, (char *)buf);

  /* mask_1 */
  FUT(ssh_ipaddr_parse_with_mask(&addr, mask_1, NULL));
  ssh_ipaddr_print_with_mask(&addr, (char *)buf, sizeof(buf));
  SEQ(mask_1, (char *)buf);

  FUT(SSH_IP_MASK_LEN(&addr) == 24);

    /* addr_1 and mask_1 compares */
  FUT(ssh_ipaddr_parse_with_mask(&addr2, addr_1, "32"));
  FUT(SSH_IP_MASK_LEN(&addr2) == 32);

  FUT(SSH_IP_EQUAL(&addr, &addr2));
  FUT(ssh_ipaddr_mask_equal(&addr2, &addr));

    /* addr_1/mask_2 */
  FUT(ssh_ipaddr_parse_with_mask(&addr, addr_1, mask_2));
  FUT(SSH_IP_MASK_LEN(&addr) == 24);

  fprintf(stderr, "%s: ipv4 tests ok\n", av[0]);

  speed();

  /* test ipv6 parsing */
  for (i = 0; i < sizeof(ipv6_test) / sizeof(*ipv6_test); i++)
    {
      unsigned char binbuf[16];
      size_t len;

      ret = ssh_ipaddr_parse(&addr, ipv6_test[i].str);
      
      if (ret != ipv6_test[i].ok)
        {
          fprintf(stderr, "%s: ssh_ipaddr_parse(&addr, \"%s\") == %d != %d\n",
                  av[0], ipv6_test[i].str, ret, ipv6_test[i].ok);
          return 1;
        }
      else if (verbose_flag)
        {
          fprintf(stderr, "%s: \"%s\" parsed %s, ok\n",
                  av[0], ipv6_test[i].str, ret == TRUE ? "valid" : "invalid");
        }
      
      if (!SSH_IP_IS6(&addr))
        {
          fprintf(stderr, "%s: ssh_ipaddr_parse(&addr, \"%s\"), "
                  "SSH_IP_IS6(&addr) == FALSE\n", av[0], ipv6_test[i].str);
          return 1;
        }
      
      if (ret)
        {
          for (n = 0; n < 16; n++)
            {
              if (addr.addr_data[n] != ipv6_test[i].addr[n])
                {
                  fprintf(stderr, "%s: ssh_ipaddr_parse(&addr, \"%s\"), "
                          "mismatch at addr byte %d, 0x%x != 0x%x\n",
                          av[0], ipv6_test[i].str, n, addr.addr_data[n],
                          ipv6_test[i].addr[n]);
                  return 1;
                }
            }
        }
      
      len = sizeof(binbuf);
      ret = ssh_inet_strtobin(ipv6_test[i].str, binbuf, &len);
      
      if (ret != ipv6_test[i].ok)
        {
          fprintf(stderr, "%s: ssh_inet_strtobin(\"%s\", buf, 16) == %d != %d\n",
                  av[0], ipv6_test[i].str, ret, ipv6_test[i].ok);
          return 1;
        }
      else if (verbose_flag)
        {
          fprintf(stderr, "%s: \"%s\" parsed %s, ok\n",
                  av[0], ipv6_test[i].str, ret == TRUE ? "valid" : "invalid");
        }
      
      if (!SSH_IP_IS6(&addr))
        {
          fprintf(stderr, "%s: ssh_ipaddr_parse(&addr, \"%s\"), "
                  "SSH_IP_IS6(&addr) == FALSE\n", av[0], ipv6_test[i].str);
          return 1;
        }
      
      if (ret)
        {
          for (n = 0; n < 16; n++)
            {
              if (binbuf[n] != ipv6_test[i].addr[n])
                {
                  fprintf(stderr, "%s: ssh_ipaddr_parse(&addr, \"%s\"), "
                          "mismatch at addr byte %d, 0x%x != 0x%x\n",
                          av[0], ipv6_test[i].str, n, binbuf[n],
                          ipv6_test[i].addr[n]);
                  return 1;
                }
            }
        }
      
      if (ipv6_test[i].ok)
        {
          ssh_ipaddr_print(&addr, (char *)buf, sizeof(buf));
          
          /*fprintf(stderr, "\"%s\" as \"%s\"\n", ipv6_test[i].str, buf);*/
          
          if (ssh_ipaddr_parse(&addr2, (char *)buf) == FALSE)
            {
              fprintf(stderr, "%s: \"%s\" could not be parsed"
                      " (it was produced with ssh_ipaddr_print)\n", av[0], buf);
              return 1;
            }
          
          ssh_ipaddr_print(&addr2, (char *)buf2, sizeof(buf2));
          
          if (strcmp((char *)buf, (char *)buf2) != 0)
            {
              fprintf(stderr, "%s: \"%s\" not parsed and re-printed same, "
                      "was \"%s\"\n", av[0], buf, buf2);
              return 1;
            }
        }
      
      /*fprintf(stderr, "\"%s\": ok\n", ipv6_test[i].str);*/
    }
  
  fprintf(stderr, "%s: ipv6 tests ok\n", av[0]);
  
  /* Test ssh_ipaddr_merge_bits (we know the implementation -- it is
     sufficient to test it as ipv6) */
  
#define MAXMERGEITER    10000

  for (i = 0; i < MAXMERGEITER ; i++)
    {
      for (n = 0; n < 16; n++)
        {
#if 1
          buf[n] = random() % 0xff;
          buf2[n] = random() % 0xff;
          masklen = random();
          masklen2 = random();
#else
          buf[n] = 0xff;
          buf2[n] = 0x00;
#endif
        }
      
      if (i < MAXMERGEITER/2)
        {
          SSH_IP6_MASK_DECODE(&addr, buf, masklen % 129);
          SSH_IP6_MASK_DECODE(&addr2, buf2, masklen2 % 129);
          n = i % 129;
          alen = 128;
        }
      else
        {
          SSH_IP4_MASK_DECODE(&addr, buf, masklen % 33);
          SSH_IP4_MASK_DECODE(&addr2, buf2, masklen % 33);
          n = i % 33;
          alen = 32;
        }

      if ((b = ssh_ipaddr_mask_equal(&addr, &addr2)) != 
          (b2 = mask_equal_1(&addr, &addr2)))
        {
          char buf[512];

          ssh_snprintf(buf, sizeof(buf),
                       "ssh_ipaddr_mask_equal(%@,%@) == %s when should be %s\n",
                        ssh_ipaddr_render, &addr,
                       ssh_ipaddr_render, &addr2,
                       b ? "TRUE" : "FALSE",
                       b2 ? "TRUE" : "FALSE");
          
          fprintf(stderr, "%s", buf);
          
          ssh_ipaddr_mask_equal(&addr, &addr2);
          
          print_bits(" addr", addr.addr_data, SSH_IP_ADDR_LEN(&addr));
          print_bits("addr2", addr2.addr_data, SSH_IP_ADDR_LEN(&addr2));
          
          return 1;
        }

      ssh_ipaddr_merge_bits(&addr3, &addr, n, &addr2);
      
      for (k = 0; k < n; k++)
        {
          if ((addr3.addr_data[k / 8] & (1 << (7 - k % 8))) !=
              (addr.addr_data[k / 8] & (1 << (7 - k % 8))))
            {
              fprintf(stderr, "Mismatch in bit %d, byte %d: %#x vs. %#x, "
                      "n=%d\n", k, k / 8,
                      addr3.addr_data[k / 8],
                      addr.addr_data[k / 8], n);
              
              fprintf(stderr, "        ");
              for (i = 0; i < (k + (k / 8)); i++)
                {
                  fputc(' ', stderr);
                }
              fprintf(stderr, "x\n");
              
              print_bits(" addr", addr.addr_data, 16);
              print_bits("addr2", addr2.addr_data, 16);
              print_bits("addr3", addr3.addr_data, 16);
              
              return 1;
            }
        }

      for (;k < alen; k++)
        {
          if ((addr3.addr_data[k / 8] & (1 << (7 - k % 8))) !=
              (addr2.addr_data[k / 8] & (1 << (7 - k % 8))))
            {
              fprintf(stderr, "Mismatch in bit %d, byte %d: %#x vs. %#x, "
                      "n=%d\n", k, k / 8,
                      addr3.addr_data[k / 8],
                      addr2.addr_data[k / 8], n);
              
              fprintf(stderr, "        ");
              for (i = 0; i < (k + (k / 8)); i++)
                {
                  fputc(' ', stderr);
                }
              fprintf(stderr, "x\n");
              
              print_bits(" addr", addr.addr_data, 16);
              print_bits("addr2", addr2.addr_data, 16);
              print_bits("addr3", addr3.addr_data, 16);
              
              return 1;
            }
        }

      if (verbose_flag)
        {
          ssh_snprintf((char *)buf, sizeof(buf), "%s: %@ |%d %@ ok\n",
                       av[0],
                       ssh_ipaddr_render,&addr, 
                       n,
                       ssh_ipaddr_render, &addr2);
          
          fprintf(stderr, "%s", buf);
        }
    }
  
  fprintf(stderr, "%s: ssh_ipaddr_merge_bits/ssh_ipaddr_mask_equal tests ok\n",
          av[0]);

  for (i = 0; i < sizeof(netmask_tests) / sizeof(*netmask_tests); i++)
    {
      ret = ssh_inet_compare_netmask(netmask_tests[i].netmask,
                                     netmask_tests[i].ip);
      if (ret != netmask_tests[i].inside)
        {
          fprintf(stderr, "%s: ssh_inet_compare_netmask(\"%s\", \"%s\") "
                  "=> %d != %d\n",
                  av[0], netmask_tests[i].netmask,
                  netmask_tests[i].ip, ret, netmask_tests[i].inside);
          exit(1);
        }
    }
  fprintf(stderr, "%s: ssh_inet_compare_netmask tests ok\n", av[0]);
  
  for (i = 0; i < sizeof(compare_tests) / sizeof(*compare_tests); i++)
    {
      ret = ssh_inet_ip_address_compare(compare_tests[i].ip1,
                                        compare_tests[i].ip2);
      if (ret != compare_tests[i].value)
        {
          fprintf(stderr, "%s: ssh_inet_ip_address_compare(\"%s\", \"%s\") "
                  "=> %d != %d\n",
                  av[0], compare_tests[i].ip1,
                  compare_tests[i].ip2, ret, compare_tests[i].value);
          exit(1);
        }
    }
  fprintf(stderr, "%s: ssh_inet_ip_address_compare tests ok\n", av[0]);
  
  /* Test ssh_ipproto_render */
  for (i = 0; ssh_ip_protocol_id_keywords[i].name; i++)
    {
      ssh_snprintf((char *)buf, sizeof(buf), "%@",
                   ssh_ipproto_render, ssh_ip_protocol_id_keywords[i].code);

      if (strcmp(ssh_ip_protocol_id_keywords[i].name, (char *)buf))
        {
          fprintf(stderr, "Protocol %lu is not rendered as `%s', but `%s'\n",
                  ssh_ip_protocol_id_keywords[i].code,
                  ssh_ip_protocol_id_keywords[i].name,
                  buf);
        }

      if (verbose_flag)
        {
          fprintf(stderr, "Protocol %lu = %s\n",
                  ssh_ip_protocol_id_keywords[i].code,
                  buf);
        }
    }
  
  fprintf(stderr, "%s: ssh_ipproto_render tests ok\n", av[0]);
  
  return 0;
}
