/*

  t-adt-avltree.c

  Author: Matthias Fischmann <fis@ssh.fi>

  Copyright (c) 2000, 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Tue Apr 10 20:50:46 2001.

  */

#include "sshincludes.h"
#include "sshdebug.h"
#include "sshadt.h"
#include "sshadt_avltree.h"
#include "sshadt_ranges.h"
#include "sshadt_list.h"
#include "sshadt_map.h"
#include "sshadt_xmap.h"

#define SSH_DEBUG_MODULE "SshADTAvlTest"

static const int num_elements = (int)1e4;


typedef struct {
  int i;
  SshADTHeaderStruct h;
} AvlElementStruct,  *AvlElement;


/************************************************* container generation glue */

static int cmp(const void *o1, const void *o2, void *ctx)
{
  int val, i1, i2;

  SSH_DEBUG(9, ("in"));

  SSH_ASSERT(o1 != NULL);
  SSH_ASSERT(o2 != NULL);

  i1 = ((AvlElement)o1)->i;
  i2 = ((AvlElement)o2)->i;
  val = i1 - i2;

  SSH_DEBUG(9, ("out [%i - %i = %i]", i1, i2, val));

  return val;
}


/* user-allocated memory, user-administrated headers */

static SshADTContainer create_um_uh(SshADTContainerType t)
{
  return ssh_adt_create_generic
    (t,
     SSH_ADT_COMPARE, cmp,
     SSH_ADT_HEADER, SSH_ADT_OFFSET_OF(AvlElementStruct, h),
     SSH_ADT_ARGS_END);
}

static SshADTHandle inject_um(SshADTContainer c, void *o)
{
  return ssh_adt_insert(c, o);
  /* (this works for both um_uh and um_ah) */
}

/* user-allocated memory, adt-administrated headers */

static SshADTContainer create_um_ah(SshADTContainerType t)
{
  return ssh_adt_create_generic
    (t,
     SSH_ADT_COMPARE, cmp,
     SSH_ADT_ARGS_END);
}

/* adt-allocated memory, user-administraded headers */

static SshADTContainer create_am_uh(SshADTContainerType t)
{
  return ssh_adt_create_generic
    (t,
     SSH_ADT_COMPARE, cmp,
     SSH_ADT_SIZE, sizeof(AvlElementStruct),
     SSH_ADT_HEADER, SSH_ADT_OFFSET_OF(AvlElementStruct, h),
     SSH_ADT_ARGS_END);
}

static SshADTHandle inject_am(SshADTContainer c, void *o)
{
  return ssh_adt_put(c, o);
}

/* adt-allocated memory, adt-administraded headers */

static SshADTContainer create_am_ah(SshADTContainerType t)
{
  return ssh_adt_create_generic
    (t,
     SSH_ADT_COMPARE, cmp,
     SSH_ADT_SIZE, sizeof(AvlElementStruct),
     SSH_ADT_ARGS_END);
}


/******************************************************* contents generation */


/* Generate a list of random numbers that forms a deterministic
   basis for all tree operations.  */

static SshADTContainer mk_contents(void);
static SshADTContainer mk_contents()
{
  int i;
  AvlElement e;
  SshADTContainer list;

  /* perhaps be spontaneous, perhaps not (not that important anyway) */
  {
    FILE *fp;
    int i, c = 1347;
    if ((fp = fopen("/dev/urandom", "r")))
      for (i = 0; i < 10; i++) c *= fgetc(fp);
    srand(c);
  }

  list = ssh_adt_create_generic
    (SSH_ADT_LIST, SSH_ADT_COMPARE, cmp, SSH_ADT_ARGS_END);

  for (i = 0; i < num_elements; i++)
    {
      e = ssh_xmalloc(sizeof(*e));
      e->i = rand() % num_elements;
      SSH_DEBUG(9, ("#%i=%i.", i, e->i));
      ssh_adt_insert(list, e);
    }

  return list;
}


/* Compare a tree agains sorted input list.  */

static void check_sanity_enumerate(SshADTContainer c, SshADTContainer sorted)
{
  SshADTHandle h, i;
  int a, b;

  h = ssh_adt_enumerate_start(c);
  i = ssh_adt_enumerate_start(sorted);

  while(h != SSH_ADT_INVALID && i != SSH_ADT_INVALID)
    {
      a = ((AvlElement)ssh_adt_get(c, h))->i;
      b = *((int *)ssh_adt_get(sorted, i));
      SSH_DEBUG(9, ("%i(tree)=%i(list).", a, b));
      SSH_ASSERT(a == b);

      h = ssh_adt_enumerate_next(c, h);
      i = ssh_adt_enumerate_next(sorted, i);
    }

  SSH_ASSERT(h == SSH_ADT_INVALID && i == SSH_ADT_INVALID);
}


/********************************************** The actual algorithmic stuff */

Boolean tree(SshADTContainer (*create)(SshADTContainerType),
             SshADTHandle (*inject)(SshADTContainer, void *),
             SshADTContainer raw, SshADTContainer sorted,
             Boolean can_detach)
{
  SshADTContainer c;
  SshADTHandle h;

  SSH_DEBUG(4, ("in"));

  SSH_DEBUG(4, ("create empty tree."));
  c = create(SSH_ADT_AVLTREE);

  SSH_DEBUG(4, ("inject."));
  h = ssh_adt_enumerate_start(raw);
  while(h != SSH_ADT_INVALID)
    {
      inject(c, ssh_adt_get(raw, h));
      h = ssh_adt_enumerate_next(raw, h);
    }

  SSH_DEBUG(4, ("enumerate."));
#if 0
  ssh_adt_avltree_int_dump(5, c);
#endif
  check_sanity_enumerate(c, sorted);

  SSH_DEBUG(4, ("remove and reinsert at random."));
  {
    int i, n, odds, num_rounds;
    SshADTAbsoluteLocation l;
    AvlElement e;
    SshADTContainer cache, raw2;

    num_rounds = 500;

    cache = ssh_adt_create_generic
      (SSH_ADT_LIST, SSH_ADT_COMPARE, cmp, SSH_ADT_ARGS_END);
    raw2 = ssh_adt_duplicate_container(raw);

    for (i = 0; i <= num_rounds; i++)
      {
        /* Look up random element in contents.  Detach it from the
           tree and insert it into the cache.  */

        l = SSH_ADT_INDEX(rand() % ssh_adt_num_objects(raw2));
        h = ssh_adt_get_handle_to_location(raw2, l);
        e = ssh_adt_detach(raw2, h);

        SSH_DEBUG(5, ("detaching object #%i=%i.", l, e));
        h = ssh_adt_get_handle_to_equal(c, e);
        SSH_DEBUG(5, ("get_handle_to_equal found h=%i.", h));
        SSH_ASSERT(h != NULL);

        if (can_detach)
          {
            e = ssh_adt_detach(c, h);
          }
        else  /* automatic memory - we need to copy the data out by hand.  */
          {
            memcpy(e, ssh_adt_get(c, h), sizeof(*e));
            ssh_adt_delete(c, h);
          }

        ssh_adt_insert(cache, e);

        /* If the cache has grown big enough, empty it into the tree.  */

        n = ssh_adt_num_objects(cache);
        if (n > 0)
          {
            odds = num_elements / (3 * n);
            if (odds == 0 ||                    /* if the cache is `full'... */
                rand() % odds == 0 ||  /* ... or we feel like emptying it... */
                i == num_rounds)          /* ... or this is the last round.  */
              {
                SSH_DEBUG(5, ("reinjecting %i objects from cache.", n));

                while(TRUE)
                  {
                    h = ssh_adt_get_handle_to_location(cache, SSH_ADT_DEFAULT);
                    if (h == SSH_ADT_INVALID)
                      break;
                    e = ssh_adt_detach(cache, h);
                    ssh_adt_insert(raw2, e);
                    inject(c, e);
                  }
              }
          }
      }

    SSH_ASSERT(ssh_adt_num_objects(cache) == 0);
    ssh_adt_destroy(cache);
    ssh_adt_destroy(raw2);
  };

  SSH_DEBUG(4, ("enumerate again."));
  check_sanity_enumerate(c, sorted);

  SSH_DEBUG(4, ("destroy."));
  ssh_adt_destroy(c);

  SSH_DEBUG(4, ("out"));
  return TRUE;
}


/******************************************************************** ranges */


void ranges_test_glb()
{
  SshADTContainer c;
  SshADTHandle h;
  AvlElement e;

  SSH_DEBUG(4, ("in"));

  c = create_um_uh(SSH_ADT_RANGES);

  SSH_DEBUG(5, ("feeding."));
  e = ssh_xmalloc(sizeof(*e)); e->i = 18; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i =  2; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 26; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 48; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 46; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 16; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 29; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 26; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 12; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 40; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 40; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 48; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 47; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 18; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 14; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i =  5; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 33; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 27; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 31; inject_um(c, e);
  e = ssh_xmalloc(sizeof(*e)); e->i = 38; inject_um(c, e);

  SSH_DEBUG(5, ("reading."));
  e = ssh_xmalloc(sizeof(*e)); e->i = 0; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h == SSH_ADT_INVALID);
  e = ssh_xmalloc(sizeof(*e)); e->i = 1; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h == SSH_ADT_INVALID);
  e = ssh_xmalloc(sizeof(*e)); e->i = 2; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 2);
  e = ssh_xmalloc(sizeof(*e)); e->i = 3; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 2);
  e = ssh_xmalloc(sizeof(*e)); e->i = 4; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 2);
  e = ssh_xmalloc(sizeof(*e)); e->i = 5; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 6; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 7; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 8; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 9; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 10; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 11; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 5);
  e = ssh_xmalloc(sizeof(*e)); e->i = 12; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 12);
  e = ssh_xmalloc(sizeof(*e)); e->i = 13; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 12);
  e = ssh_xmalloc(sizeof(*e)); e->i = 14; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 14);
  e = ssh_xmalloc(sizeof(*e)); e->i = 15; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 14);
  e = ssh_xmalloc(sizeof(*e)); e->i = 16; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 16);
  e = ssh_xmalloc(sizeof(*e)); e->i = 17; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 16);
  e = ssh_xmalloc(sizeof(*e)); e->i = 18; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 19; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 20; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 21; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 22; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 23; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 24; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 25; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 18);
  e = ssh_xmalloc(sizeof(*e)); e->i = 26; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 26);
  e = ssh_xmalloc(sizeof(*e)); e->i = 27; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 27);
  e = ssh_xmalloc(sizeof(*e)); e->i = 28; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 27);
  e = ssh_xmalloc(sizeof(*e)); e->i = 29; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 29);
  e = ssh_xmalloc(sizeof(*e)); e->i = 30; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 29);
  e = ssh_xmalloc(sizeof(*e)); e->i = 31; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 31);
  e = ssh_xmalloc(sizeof(*e)); e->i = 32; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 31);
  e = ssh_xmalloc(sizeof(*e)); e->i = 33; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 33);
  e = ssh_xmalloc(sizeof(*e)); e->i = 34; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 33);
  e = ssh_xmalloc(sizeof(*e)); e->i = 35; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 33);
  e = ssh_xmalloc(sizeof(*e)); e->i = 36; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 33);
  e = ssh_xmalloc(sizeof(*e)); e->i = 37; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 33);
  e = ssh_xmalloc(sizeof(*e)); e->i = 38; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 38);
  e = ssh_xmalloc(sizeof(*e)); e->i = 39; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 38);
  e = ssh_xmalloc(sizeof(*e)); e->i = 40; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 41; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 42; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 43; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 44; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 45; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 40);
  e = ssh_xmalloc(sizeof(*e)); e->i = 46; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 46);
  e = ssh_xmalloc(sizeof(*e)); e->i = 47; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 47);
  e = ssh_xmalloc(sizeof(*e)); e->i = 48; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 48);
  e = ssh_xmalloc(sizeof(*e)); e->i = 49; h = ssh_adt_get_handle_to_glb(c, e);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  SSH_ASSERT(*(int *)ssh_adt_get(c, h) == 48);

  SSH_DEBUG(5, ("killing."));
  ssh_adt_destroy(c);

  SSH_DEBUG(4, ("out"));
  return;
}



Boolean ranges(SshADTContainer (*create)(SshADTContainerType),
               SshADTHandle (*inject)(SshADTContainer, void *))
{
  SshADTContainer c;
  SshADTHandle h;
  AvlElement e;
#ifdef DEBUG_LIGHT
  int debug = 8;
#endif /* DEBUG_LIGHT */

  /* ssh_debug_set_level_string("*=8"); */

  SSH_DEBUG(4, ("in"));

  SSH_DEBUG(4, ("create empty tree."));
  c = create(SSH_ADT_RANGES);

  SSH_DEBUG(4, ("test low level interface."));

#ifdef DEBUG_LIGHT
# define DUMP_RANGES                                                          \
  {                                                                           \
    SSH_DEBUG(debug, ("> dump_ranges <"));                                    \
    h = ssh_adt_enumerate_start(c);                                           \
    while(h != SSH_ADT_INVALID)                                               \
      {                                                                       \
        int k = *(int *)ssh_adt_get(c, h);                                    \
        int *v = ssh_adt_map_lookup(c, h);                                    \
        SSH_DEBUG(debug, (">  %i %i.", k, (int)v));                           \
        h = ssh_adt_enumerate_next(c, h);                                     \
      }                                                                       \
    SSH_DEBUG(debug, ("> <"));                                                \
  }
#else /* DEBUG_LIGHT */
# define DUMP_RANGES                                                          \
  {                                                                           \
    SSH_DEBUG(debug, ("> dump_ranges <"));                                    \
  }
#endif /* DEBUG_LIGHT */

# define INSERT(ik, v)                                                        \
  {                                                                           \
    e = ssh_xmalloc(sizeof(*e));                                              \
    e->i = (ik);                                                              \
    inject(c, e);                                                             \
    h = ssh_adt_get_handle_to_equal(c, e);                                    \
    SSH_ASSERT(h);                                                            \
    ssh_adt_map_attach(c, h, (void *)(v));                                    \
  }

#ifdef DELETE
#undef DELETE
#endif /* DELETE */

# define DELETE(ik)                                                           \
  {                                                                           \
    e = ssh_xmalloc(sizeof(*e));                                              \
    e->i = (ik);                                                              \
    h = ssh_adt_get_handle_to_equal(c, e);                                    \
    SSH_ASSERT(h);                                                            \
    ssh_xfree(e);                                                             \
    ssh_adt_delete(c, h);                                                     \
  }

#ifdef DEBUG_LIGHT
# define CK(ik, expect)                                                       \
  {                                                                           \
    e = ssh_xmalloc(sizeof(*e));                                              \
    e->i = (ik);                                                              \
    h = ssh_adt_get_handle_to_glb(c, e);                                      \
    if (h == NULL)                                                            \
      {                                                                       \
        SSH_DEBUG(debug, ("%7i ==  (NULL)", (ik)));                           \
        SSH_ASSERT(expect == (int)NULL);                                      \
      }                                                                       \
    else                                                                      \
      {                                                                       \
        int find = (int)ssh_adt_map_lookup(c, h);                             \
        SSH_DEBUG(debug, ("%7i == %7i", (ik), find));                         \
        SSH_ASSERT(expect == find);                                           \
      }                                                                       \
  }
#else /* DEBUG_LIGHT */
# define CK(ik, expect)                                                       \
  {                                                                           \
    e = ssh_xmalloc(sizeof(*e));                                              \
    e->i = (ik);                                                              \
    h = ssh_adt_get_handle_to_glb(c, e);                                      \
    if (h == NULL)                                                            \
      {                                                                       \
        SSH_DEBUG(debug, ("%7i ==  (NULL)", (ik)));                           \
        SSH_ASSERT(expect == (int)NULL);                                      \
      }                                                                       \
  }

#endif /* DEBUG_LIGHT */
  /* write */
  DUMP_RANGES;

  INSERT(20, 1);
  INSERT(50, 2);

  /* lookup */
  DUMP_RANGES;

  CK(-10, (int)NULL);
  CK(23, 1);
  CK(230, 2);

  /* stirr */
  DUMP_RANGES;  CK(-10, (int)NULL); CK(20, 1); CK(50, 2);

  INSERT(24, 1);
  DUMP_RANGES;
  CK(-10, (int)NULL); CK(20, 1); CK(50, 2);

  DELETE(20);
  DUMP_RANGES;
  CK(22, (int)NULL); CK(50, 2);

  INSERT(24, 1);
  DUMP_RANGES;
  CK(22, (int)NULL); CK(24, 1); CK(50, 2);

  INSERT(24, 2);
  DUMP_RANGES;
  CK(22, (int)NULL); CK(24, 2);

  INSERT(10, 8);
  DUMP_RANGES;
  CK(9, (int)NULL); CK(10, 8); CK(24, 2);

  INSERT(38, -1);
  DUMP_RANGES;
  CK(9, (int)NULL); CK(10, 8); CK(24, 2); CK(38, -1);

  INSERT(88, 6);
  DUMP_RANGES;
  CK(9, (int)NULL); CK(10, 8); CK(24, 2); CK(38, -1); CK(88, 6);

  DELETE(88);
  DUMP_RANGES;
  CK(9, (int)NULL); CK(10, 8); CK(24, 2); CK(38, -1);

  INSERT(24, 0);
  DUMP_RANGES;
  CK(9, (int)NULL); CK(10, 8); CK(24, 0); CK(38, -1);

  DELETE(10);
  DUMP_RANGES;
  CK(24, (int)NULL); CK(38, -1);

  INSERT(10, 8);
  DUMP_RANGES;
  CK(-1000, (int)NULL); CK(10, 8); CK(38, -1);

  INSERT(24, 0);
  DUMP_RANGES;
  CK(-1000, (int)NULL); CK(10, 8); CK(24, 0); CK(38, -1);

  DELETE(38);
  DUMP_RANGES;
  CK(-1000, (int)NULL); CK(10, 8); CK(24, 0);

  /* XXX - SSH_DEBUG(4, ("test high level interface.")); */

  SSH_DEBUG(4, ("done."));

  ssh_adt_destroy(c);
  return TRUE;
}

/* ***************************************************************** mappings */

typedef struct {
  int i;
  int ref_count;
} *AvlImage;

static void mappings_attach_cb(void *i, void *ctx)
{
  AvlImage img = (AvlImage)i;
  SSH_DEBUG(9, ("in"));

  SSH_ASSERT(img != NULL);
  /* (this is enforced by the container, but after all we are here to
     question its sanity...)  */

  SSH_DEBUG(9, ("%p: setting ref_count to %i.", img, img->ref_count + 1));
  img->ref_count++;

  SSH_DEBUG(9, ("out"));
}

static void mappings_detach_cb(void *i, void *ctx)
{
  AvlImage img = (AvlImage)i;
  SSH_DEBUG(9, ("in"));

  if (img != NULL)
    {
      SSH_ASSERT(img->ref_count > 0);
      if (img->ref_count == 1)
        {
          SSH_DEBUG(9, ("%p: this was the last reference.", img));
          ssh_xfree(img);
          SSH_DEBUG(9, ("free was successful."));
        }
      else
        {
          SSH_DEBUG(9, ("%p: setting ref_count to %i.",
                        img, img->ref_count - 1));
          img->ref_count--;
        }
    }

  SSH_DEBUG(9, ("out"));
}

Boolean mappings(void);
Boolean mappings()
{
# define  num_rounds  num_elements
# define  num_images  30

  SshADTContainer c;
  SshADTHandle h;
  AvlElement key;
  AvlImage img[num_images];
  int i, k;

  for (k = 0; k < 2; k++)
    {
      /* Now that we are at it, we can as well test a few other
         container types that support the generic mapping mechanism.  */

      SshADTContainerType t = NULL;

      switch (k)
        {
        case 0: SSH_DEBUG(4, ("(map)")); t = SSH_ADT_MAP; break;
        case 1: SSH_DEBUG(4, ("(tree)")); t = SSH_ADT_AVLTREE; break;
        }

      c = ssh_adt_create_generic
        (t,
         SSH_ADT_COMPARE, cmp,
         SSH_ADT_MAP_ATTACH, mappings_attach_cb,
         SSH_ADT_MAP_DETACH, mappings_detach_cb,
         SSH_ADT_HEADER, SSH_ADT_OFFSET_OF(AvlElementStruct, h),
         SSH_ADT_ARGS_END);

      for (i = 0; i < num_images; i++)
        {
          img[i] = ssh_xmalloc(sizeof(*img[i]));
          img[i]->i = i;
          img[i]->ref_count = 0;
        }

      /* Insert some random integer keys that map on themselves.  */
      for (i = 0; i < num_rounds; i++)
        {
          key = ssh_xmalloc(sizeof(*key));
          key->i = i;

          h = ssh_adt_insert(c, key);
          ssh_adt_map_attach(c, h, img[rand() % num_images]);
        }

      /* Destruction will hopefully not leave too many memory leaks.  */
      ssh_adt_destroy(c);
    }

  return TRUE;
}


/*************************************************************** performance */

Boolean performance(void);
Boolean performance()
{
  /* XXX */
  return TRUE;
}


/******************************************************************** driver */

int main(int argc, char **argv)
{
  SshADTContainer raw, sorted;
  Boolean val;

  ssh_debug_set_level_string("*=0");

  SSH_DEBUG(3, ("generate contents."));

  raw = mk_contents();
  sorted = ssh_adt_duplicate_container(raw);
  ssh_adt_list_sort(sorted);

  ssh_debug_set_level_string("*=3");


  /* standard trees */

  SSH_DEBUG(3, ("tree() / user memory / abstract headers."));
  val = tree(create_um_ah, inject_um, raw, sorted, TRUE);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("tree() / user memory / inlined headers."));
  val = tree(create_um_uh, inject_um, raw, sorted, TRUE);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("tree() / adt allocated memory / abstract headers."));
  val = tree(create_am_ah, inject_am, raw, sorted, FALSE);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("tree() / adt allocated memory / inlined headers"
                " (not the most useful mode ever)."));
  val = tree(create_am_uh, inject_am, raw, sorted, FALSE);
  SSH_ASSERT(val);


  /* tree mappings */

  SSH_DEBUG(3, ("mappings."));
  val = mappings();
  SSH_ASSERT(val);


  /* ranges */

  SSH_DEBUG(3, ("glb."));
  ranges_test_glb();

  SSH_DEBUG(3, ("ranges() / user memory / inlined headers."));
  val = ranges(create_um_uh, inject_um);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("ranges() / user memory / abstract headers."));
  val = ranges(create_um_ah, inject_um);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("ranges() / adt allocated memory / abstract headers."));
  val = ranges(create_am_ah, inject_am);
  SSH_ASSERT(val);

  SSH_DEBUG(3, ("ranges() / adt allocated memory / inlined headers"
                " (not the most useful mode ever)."));
  val = ranges(create_am_uh, inject_am);
  SSH_ASSERT(val);


  /* performance */

  SSH_DEBUG(3, ("quick & dirty performance test."));
  val = performance();
  SSH_ASSERT(val);

  return 0;
}
