/*
 *
 * t-localstreamclient.c
 *
 * Author: Markku Rossi <mtr@ssh.fi>
 *
 * Copyright (c) 2001 SSH Communications Security, Finland
 *               All rights reserved.
 *
 * What is this file for?
 *
 */

#define SSH_DEBUG_MODULE "TLocalStreamClient"

#include "sshincludes.h"
#include "sshdebug.h"
#include "ssheloop.h"
#include "sshbuffer.h"
#include "sshlocalstream.h"
#include "sshtimeouts.h"

static void local_connect_callback(SshStream stream, void *context);
static void stream_callback(SshStreamNotification notification, void *context);
static void timeout_callback(void *context);

static SshBuffer buffer;
static long microseconds;

size_t bytes = 20000;

int main(int argc, char *argv[])
{
  SSH_ASSERT(argv[1]);
  SSH_ASSERT(argv[2]);

  ssh_event_loop_initialize();

  buffer = ssh_buffer_allocate();
  microseconds = atol(argv[2]);

  if (argv[3])
    bytes = atoi(argv[3]);

  ssh_local_connect(argv[1],
                    local_connect_callback,
                    0);

  ssh_event_loop_run();

  ssh_cancel_timeouts(SSH_ALL_CALLBACKS,
                      SSH_ALL_CONTEXTS);

  ssh_buffer_free(buffer);

  ssh_event_loop_uninitialize();

  return 0;
}

static void local_connect_callback(SshStream stream, void *context)
{
  SSH_ASSERT(stream);

  ssh_stream_set_callback(stream,
                          stream_callback,
                          stream);

  stream_callback(SSH_STREAM_INPUT_AVAILABLE, stream);
  if (stream)
    stream_callback(SSH_STREAM_CAN_OUTPUT, stream);
}

static void stream_callback(SshStreamNotification notification, void *context)
{
  SshStream stream = context;
  char buf[4096];
  int i;

  SSH_ASSERT(stream);

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      do
        {
          memset(buf, 0, sizeof (buf));
          i = ssh_stream_read(stream,
                              buf,
                              sizeof (buf) - 1);
          if (0 == i)
            {
              /* EOF */
              ssh_stream_destroy(stream);
              ssh_cancel_timeouts(timeout_callback, stream);
              stream = NULL;
              return;
            }
          else if (i > 0)
            {
              fprintf(stderr, "%s", buf);
              if (i > bytes)
                {
                  ssh_stream_destroy(stream);
                  ssh_cancel_timeouts(timeout_callback, stream);
                  return;
                }
              bytes -= i;
            }
        } while (i > 0);
      break;

    case SSH_STREAM_CAN_OUTPUT:
      while (ssh_buffer_len(buffer) > 0)
        {
          i = ssh_stream_write(stream,
                               ssh_buffer_ptr(buffer),
                               ssh_buffer_len(buffer));
          if (i > 0)
            {
              ssh_buffer_consume(buffer, i);
            }
          else if (i == 0)
            {
              /* EOF */
              ssh_stream_destroy(stream);
              ssh_cancel_timeouts(timeout_callback, stream);
              stream = NULL;
              return;
            }
          else
            {
              /* Would block. */
              return;
            }
        }

      SSH_ASSERT(ssh_buffer_len(buffer) == 0);

      /* Order new data. */
      ssh_register_timeout(0,
                           microseconds,
                           timeout_callback,
                           stream);
      break;

    case SSH_STREAM_DISCONNECTED:
      ssh_event_loop_abort();
    }
}

static void timeout_callback(void *context)
{
  SshStream stream = context;
  char*     time_string = 0;


  SSH_ASSERT(stream);

  time_string = ssh_readable_time_string(ssh_time(),
                                         1);
  SSH_ASSERT(time_string);

  ssh_buffer_append(buffer,
                    "client: ",
                    strlen("client: "));
  ssh_buffer_append(buffer,
                    time_string,
                    strlen(time_string));
  ssh_buffer_append(buffer,
                    "\n",
                    strlen("\n"));

  ssh_xfree(time_string);

  stream_callback(SSH_STREAM_CAN_OUTPUT,
                  stream);

}
