/*
 *
 * Author: Tuomas A. Sirn <tuomas.siren@ssh.com>
 *
 * Copyright (c) 2000 SSH Communications Security Corporation
 */

#define SSH_DEBUG_MODULE "t-pty-redirect"

#include <sshincludes.h>
#include <sshgetopt.h>
#include <ssheloop.h>
#include <sshbuffer.h>
#include <sshunixptystream.h>
#include <sshfdstream.h>

typedef struct t_pty_redirect_context
{
  unsigned int  debug_global_level;
  char*         pdebug_level_string;
  uid_t         uid;
  gid_t         gid;
  char          apty[SSH_PTY_NAME_SIZE];
  SshStream     pptystream;
  SshBuffer     pptybuffer;
  SshStream     pstdiostream;
  SshBuffer     pstdiobuffer;
}* t_pty_redirect_context;

static void t_pty_redirect_usage(void);
static void t_pty_redirect_pty_stream_callback(SshStreamNotification    notification,
                                               void*                    pdata);
static void t_pty_redirect_stdio_stream_callback(SshStreamNotification  notification,
                                                 void*                  pdata);

int main(int argc, char* argv[])
{
  t_pty_redirect_context        pcontext = 0;
  SshGetOptData                 pgetoptdata = 0;
  int                           i;
  SshPtyStatus                  ptystatus;
  int                           argv_index;

  pcontext = ssh_xmalloc(sizeof (*pcontext));
  memset(pcontext, 0, sizeof (*pcontext));

  pgetoptdata = ssh_xmalloc(sizeof (*pgetoptdata));
  memset(pgetoptdata, 0, sizeof (*pgetoptdata));
  
  ssh_getopt_init_data(pgetoptdata);
  
  while ((i = ssh_getopt(argc, argv, "G:D:u:g:", pgetoptdata)) != -1)
    {
      switch (i)
        {
        case 'G':
          pcontext->debug_global_level = atoi(pgetoptdata->arg);
          break;
        case 'D':
          pcontext->pdebug_level_string = ssh_xstrdup(pgetoptdata->arg);
          break;
        case 'u':
          pcontext->uid = atoi(pgetoptdata->arg);
          break;
        case 'g':
          pcontext->gid = atoi(pgetoptdata->arg);
          break;
        default:
          SSH_NOTREACHED;
          break;
        }
    }

  argv_index = pgetoptdata->ind;

  ssh_xfree(pgetoptdata);

  if (argv_index == argc)
    {
      t_pty_redirect_usage();
      SSH_NOTREACHED;
    }

  ssh_event_loop_initialize();

  ptystatus = ssh_pty_allocate_and_fork(pcontext->uid,
                                        pcontext->gid,
                                        pcontext->apty,
                                        &pcontext->pptystream);

  if (SSH_PTY_ERROR == ptystatus)
    {
      ssh_fatal("%s: ssh_pty_allocate_and_fork() failed\n",
                SSH_DEBUG_MODULE);
      SSH_NOTREACHED;
    }

  if (SSH_PTY_PARENT_OK == ptystatus)
    {
      pcontext->pstdiostream = ssh_stream_fd_stdio();
      pcontext->pstdiobuffer = ssh_buffer_allocate();
      ssh_stream_set_callback(pcontext->pstdiostream,
                              t_pty_redirect_stdio_stream_callback,
                              pcontext);

      pcontext->pptybuffer = ssh_buffer_allocate();
      ssh_stream_set_callback(pcontext->pptystream,
                              t_pty_redirect_pty_stream_callback,
                              pcontext);
    }
  else
    {
      char*     ppath = 0;
      char**    ppargv = 0;

      ppath = ssh_xstrdup(argv[argv_index]);

      ppargv = ssh_xcalloc((argc - argv_index) + 1,
                           sizeof (char*));

      for (i = 0;  argv_index < argc; ++i, ++argv_index)
        {
          ppargv[i] = ssh_xstrdup(argv[argv_index]);
        }

      if (execv(ppath, ppargv) < 0)
        {
          ssh_fatal("%s: execv() failed\n", SSH_DEBUG_MODULE);
          SSH_NOTREACHED;
        }
    }

  ssh_event_loop_run();

  ssh_event_loop_uninitialize();

  ssh_buffer_free(pcontext->pptybuffer);
  ssh_xfree(pcontext->pdebug_level_string);
  ssh_xfree(pcontext);

  return 0;
}

static void t_pty_redirect_usage(void)
{
  fprintf(stderr,
          "%s: usage: %s [-G debug-global-level] [-D debug-level-string][-u uid] [-g gid] command [arguments]\n",
          SSH_DEBUG_MODULE,
          SSH_DEBUG_MODULE);
          
  exit(1);
  SSH_NOTREACHED;
}

static void t_pty_redirect_pty_stream_callback(SshStreamNotification    notification,
                                               void*                    pdata)
{
  t_pty_redirect_context        pcontext = pdata;
  char                          abuf[4096];
  int                           i;
  int                           j;
  
  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      while ((i = ssh_stream_read(pcontext->pptystream,
                                  abuf,
                                  4096)) > 0)
        {
          ssh_buffer_append(pcontext->pstdiobuffer,
                            abuf,
                            i);
          t_pty_redirect_stdio_stream_callback(SSH_STREAM_CAN_OUTPUT,
                                               pdata);
        }

      if (0 == i)
        {
          ssh_event_loop_abort();
        }
      break;
    case SSH_STREAM_CAN_OUTPUT:
      i = ssh_buffer_len(pcontext->pptybuffer);
      if (i > 0)
        {
          j = ssh_stream_write(pcontext->pptystream,
                               ssh_buffer_ptr(pcontext->pptybuffer),
                                              i);
          if (j > 0)
            {
              ssh_buffer_consume(pcontext->pptybuffer,
                                 j);
            }
        }
      break;
    case SSH_STREAM_DISCONNECTED:
      ssh_event_loop_abort();
      break;
    default:
      SSH_NOTREACHED;
      break;
    }
}

static void t_pty_redirect_stdio_stream_callback(SshStreamNotification  notification,
                                                 void*                  pdata)
{
  t_pty_redirect_context        pcontext = pdata;
  char                          abuf[4096];
  int                           i;
  int                           j;

  switch (notification)
    {
    case SSH_STREAM_INPUT_AVAILABLE:
      while ((i = ssh_stream_read(pcontext->pstdiostream,
                                  abuf,
                                  4096)) > 0)
        {
          ssh_buffer_append(pcontext->pptybuffer,
                            abuf,
                            i);
          t_pty_redirect_pty_stream_callback(SSH_STREAM_CAN_OUTPUT,
                                             pdata);
        }
      if (0 == i)
        {
          ssh_event_loop_abort();
        }
      break;
    case SSH_STREAM_CAN_OUTPUT:
      i = ssh_buffer_len(pcontext->pstdiobuffer);
      if (i > 0)
        {
          j = ssh_stream_write(pcontext->pstdiostream,
                               ssh_buffer_ptr(pcontext->pstdiobuffer),
                                              i);
          if (j > 0)
            {
              ssh_buffer_consume(pcontext->pstdiobuffer,
                                 j);
            }
        }
      break;
    case SSH_STREAM_DISCONNECTED:
      ssh_event_loop_abort();
      break;
    default:
      SSH_NOTREACHED;
      break;
    }
}
