/*
 * Author: Tero Kivinen <kivinen@iki.fi>
 *
 * Copyright (c) 1998 Tero Kivinen <kivinen@ssh.fi>, Espoo, Finland
 * Copyright (c) 1998 SSH Communications Security Oy <info@ssh.fi>
 *                   All rights reserved
 */
/*
 *        Program: Urlparse
 *        $Source: /ssh/CVS/src/lib/sshutil/tests/t-url.c,v $
 *        $Author: ksr $
 *
 *        Creation          : 10:45 Jul 10 1998 kivinen
 *        Last Modification : 21:09 Jun 29 2000 kivinen
 *        Last check in     : $Date: 2001/06/05 14:52:24 $
 *        Revision number   : $Revision: 1.12 $
 *        State             : $State: Exp $
 *        Version           : 1.186
 *        
 *
 *        Description       : Test program for library to parse urls
 */
/*
 * $Id: t-url.c,v 1.12 2001/06/05 14:52:24 ksr Exp $
 * $EndLog$
 */


/* #undef SSH_URL_PARSE_USE_RELAXED */
#define SSH_URL_PARSE_USE_RELAXED

#include "sshincludes.h"
#include "sshbuffer.h"
#include "sshurl.h"

#define FUNC_O "ssh_url_parse"
#define FUNC_M "ssh_url_parse_relaxed"

static char *parsefunc;

typedef struct TestUrlRec {
  const char *url;
  const char *scheme;
  const char *host;
  const char *username;
  const char *password;
  const char *port;
  const char *path;
  Boolean ok;
  Boolean gok;
} *TestUrl;

struct TestUrlRec tests[] = {
  { "http://www.ssh.fi/testing/host",
    "http", "www.ssh.fi", NULL, NULL, NULL, "testing/host", TRUE, TRUE },
  { "http://[0012123123/testing/host",
    "http", "[0012123123", NULL, NULL, NULL, "testing/host", TRUE, TRUE },
  { "http://[FEDC:BA98:7654:3210:FEDC:BA98:7654:3210]/testing/host",
    "http", "[FEDC:BA98:7654:3210:FEDC:BA98:7654:3210]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[fedc:ba98:7654:3210:fedc:ba98:7654:3210]/testing/host",
    "http", "[fedc:ba98:7654:3210:fedc:ba98:7654:3210]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[1080:0:0:0:8:800:200C:4171]/testing/host",
    "http", "[1080:0:0:0:8:800:200C:4171]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[3FFE:2A00:100:7031::1]/testing/host",
    "http", "[3FFE:2A00:100:7031::1]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[1080::8:800:200C:417A]/testing/host",
    "http", "[1080::8:800:200C:417A]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[::192.9.5.5]/testing/host",
    "http", "[::192.9.5.5]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[::FFFF:129.144.52.38]/testing/host",
    "http", "[::FFFF:129.144.52.38]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http://[2010:836B:4179::836B:4179]/testing/host",
    "http", "[2010:836B:4179::836B:4179]", NULL, NULL, NULL,
    "testing/host", TRUE, TRUE },
  { "http:///testing/host",
    "http", "", NULL, NULL, NULL, "testing/host", TRUE, TRUE },
  { "scheme1:///scheme2:///scheme3://data",
    "scheme1", "", NULL, NULL, NULL, "scheme2:///scheme3://data", TRUE, TRUE},
  { "ftp://kivinen:foobar@ftp.ssh.fi:21/hidden",
    "ftp", "ftp.ssh.fi", "kivinen", "foobar", "21", "hidden", TRUE, TRUE },
  { "ftp://kivinen:foobar@[::10.0.0.1]:21/hidden",
    "ftp", "[::10.0.0.1]", "kivinen", "foobar", "21", "hidden", TRUE, TRUE },
  { "ftp://kivinen:@ftp.ssh.fi:21/hidden",
    "ftp", "ftp.ssh.fi", "kivinen", "", "21", "hidden", TRUE, TRUE },
  { "ftp://kivinen@ftp.ssh.fi:21/hidden",
    "ftp", "ftp.ssh.fi", "kivinen", NULL, "21", "hidden", TRUE, TRUE },
  { "ftp://kivinen@ftp.ssh.fi:21/hidden",
    "ftp", "ftp.ssh.fi", "kivinen", NULL, "21", "hidden", TRUE, TRUE },
  { "scheme://username:password@host:2222/path",
    "scheme", "host", "username", "password", "2222", "path", TRUE, TRUE },
  { "scheme://username:password@[FEDC:BA98:7654:3210:FEDC:BA98:7654:3210]:2222/path",
    "scheme", "[FEDC:BA98:7654:3210:FEDC:BA98:7654:3210]", "username",
    "password", "2222", "path", TRUE, TRUE },
  { "scheme://username:password@host/path",
    "scheme", "host", "username", "password", NULL, "path", TRUE, TRUE },
  { "scheme://username@host:2222/path",
    "scheme", "host", "username", NULL, "2222", "path", TRUE, TRUE },
  { "scheme://username:@host:2222/path",
    "scheme", "host", "username", "", "2222", "path", TRUE, TRUE },
  { "scheme://:@host:2222/path",
    "scheme", "host", "", "", "2222", "path", TRUE, TRUE },
  { "scheme://:password@host:2222/path",
    "scheme", "host", "", "password", "2222", "path", TRUE, TRUE },
  { "scheme://host:2222/path",
    "scheme", "host", NULL, NULL, "2222", "path", TRUE, TRUE },
  { "//username:password@host:2222/path",
    NULL, "host", "username", "password", "2222", "path", TRUE, TRUE },
  { "scheme://username:password@host:2222",
    "scheme", "host", "username", "password", "2222", NULL, TRUE, TRUE },
  { "scheme://username:password@host",
    "scheme", "host", "username", "password", NULL, NULL, TRUE, TRUE },
  { "scheme://username:password@host/",
    "scheme", "host", "username", "password", NULL, "", TRUE, TRUE },
  { "scheme://host/path",
    "scheme", "host", NULL, NULL, NULL, "path", TRUE, TRUE },
  { "scheme://host",
    "scheme", "host", NULL, NULL, NULL, NULL, TRUE, TRUE },
  { "//host",
    NULL, "host", NULL, NULL, NULL, NULL, TRUE, TRUE },
  { "host",
    NULL, "host", NULL, NULL, NULL, NULL, FALSE, TRUE },
  { "/path",
    NULL, NULL, NULL, NULL, NULL, "path", TRUE, TRUE },
  { "",
    NULL, NULL, NULL, NULL, NULL, NULL, FALSE, FALSE },
  { "socks://muuri.ssh.fi:1080",
    "socks", "muuri.ssh.fi", NULL, NULL, "1080", NULL, TRUE, TRUE },
  { "scheme://usernam%65:pas%73word@h%6Fst:2222/%70ath",
    "scheme", "host", "username", "password", "2222", "path", TRUE, TRUE },
  { "scheme://username%40host:pass%3aword@%68%6F%73%74:2222/%70ath",
    "scheme", "host", "username@host", "pass:word", "2222", "path", TRUE, TRUE },
  { "www.ssh.fi",
    NULL, "www.ssh.fi", NULL, NULL, NULL, NULL, FALSE, TRUE },
  { "//www.ssh.fi:389",
    NULL, "www.ssh.fi", NULL, NULL, "389", NULL, TRUE, TRUE },
  { "www.ssh.fi:389",
    NULL, "www.ssh.fi", NULL, NULL, "389", NULL, FALSE, TRUE },
  { "//usr@www.ssh.fi:389",
    NULL, "www.ssh.fi", "usr", NULL, "389", NULL, TRUE, TRUE },
  { "file://netbsd",
    "file", "netbsd", NULL, NULL, NULL, NULL, TRUE, TRUE },
  { "http:index.html",
    "http", "index.html", NULL, NULL, NULL, NULL, FALSE, FALSE },
  { "mailto:kivinen@ssh.com",
    "mailto", "ssh.com", "kivinen", NULL, NULL, NULL, FALSE, TRUE },
  { "mailto:kivinen@ssh.com?subject=Subscribe&body=send%20current-issue", 
    "mailto", "ssh.com", "kivinen", NULL, NULL, "?subject=Subscribe&body=send current-issue", FALSE, TRUE },
  { "ldap://parketti.hel.fi.ssh.com:389",
    "ldap", "parketti.hel.fi.ssh.com", NULL, NULL, "389", NULL, TRUE, TRUE },
  { "ldap://parketti.hel.fi.ssh.com:389/duh",
    "ldap", "parketti.hel.fi.ssh.com", NULL, NULL, "389", "duh", TRUE, TRUE },
  { "www.google.com/search?q=\"body=send%20\"",
    NULL, "www.google.com", NULL, NULL, NULL, "search?q=\"body=send \"", FALSE, TRUE }

};

typedef struct TestFormItemRec {
  const char *key;
  const char *value;
} *TestFormItem;

typedef struct TestFormRec {
  const char *url;
  const char *path;
  Boolean ok;
  struct TestFormItemRec table[10];
} *TestForm;

struct TestFormRec form_tests[] = {
  { "/foo?a=b", "/foo", TRUE,
    { { "a", "b" } } },
  { "/foo?a=b&c=d", "/foo", TRUE,
    { { "a", "b" }, { "c", "d" } } },
  { "/aksjdklasjdlkasjdlkasjdkla?kukkuu=reset",
    "/aksjdklasjdlkasjdlkasjdkla", TRUE,
    { { "kukkuu", "reset" } } },
  { "!@#$%25^&*()_+][|\":%3f><,./'\\{}`1234567890-=?a=b&c=d&e=f",
    "!@#$%^&*()_ ][|\":?><,./'\\{}`1234567890-=", TRUE,
    { { "a", "b" }, { "c", "d" }, { "e", "f" } } },
  { "%20%21%22?kukkuu=reset&zappa=bar", " !\"", TRUE,
    { { "kukkuu", "reset" }, { "zappa", "bar" } } },
  { " %21\"?kukk%75u=re%73et&zap%70a=b%61r", " !\"", TRUE,
    { { "kukkuu", "reset" }, { "zappa", "bar" } } },
  { "/fo%3do?kuk%3dk%75u=re%73et&zap%70a=b%61r%3dfoo", "/fo=o", TRUE,
    { { "kuk=kuu", "reset" }, { "zappa", "bar=foo" } } },
  { "/fo%26o?kuk%26k%75u=re%73et&zap%70a=b%61r%26foo", "/fo&o", TRUE,
    { { "kuk&kuu", "reset" }, { "zappa", "bar&foo" } } },
  { "/foo?name=Tero%20&name=T%20&name=Kivinen", "/foo", TRUE,
    { { "name", "Tero \nT \nKivinen" } } },
  { "/foo?na%6de=Tero%20&nam%65=T%20&n%61me=Kivinen", "/foo", TRUE,
    { { "name", "Tero \nT \nKivinen" } } },
  { "/fo%xx?a=b&c=d", "/fo%xx", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/fo%3?a=b&c=d", "/fo%3", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/fo%?a=b&c=d", "/fo%", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/foo?&a=b&c=d", "/foo", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/foo?a=b&&c=d", "/foo", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/foo?a=b&c=d&", "/foo", FALSE,
    { { "a", "b" }, { "c", "d" } } },
  { "/foo?a%xx=b&c=d", "/foo", FALSE,
    { { "a%xx", "b" }, { "c", "d" } } },
  { "/foo?a%3=b&c=d", "/foo", FALSE,
    { { "a%3", "b" }, { "c", "d" } } },
  { "/foo?a%=b&c=d", "/foo", FALSE,
    { { "a%", "b" }, { "c", "d" } } },
  { "/foo?a=b&c=%xxd", "/foo", FALSE,
    { { "a", "b" }, { "c", "%xxd" } } },
  { "/foo?a=b&c=%3qd", "/foo", FALSE,
    { { "a", "b" }, { "c", "%3qd" } } },
  { "/foo?a=b&c=%qd", "/foo", FALSE,
    { { "a", "b" }, { "c", "%qd" } } },
  { "/foo?a=b&c=d%", "/foo", FALSE,
    { { "a", "b" }, { "c", "d%" } } },
  { "/foo?a=b&c=d%xx", "/foo", FALSE,
    { { "a", "b" }, { "c", "d%xx" } } },
  { "/foo?a=b&c=d%3", "/foo", FALSE,
    { { "a", "b" }, { "c", "d%3" } } },
  { "/foo?na%6de=Tero%20&nam%65=T%20&n%61me=Kivinen&bar=zappa", "/foo", TRUE,
    { { "name", "Tero \nT \nKivinen" }, { "bar", "zappa" } } },
  { "/foo?first+name=Tero+Tapani&Last%2bName=Kivinen%2b%2b", "/foo", TRUE,
    { { "first name", "Tero Tapani" }, { "Last+Name", "Kivinen++" } } }
};

void mapping_print(SshMapping mapping)
{
  int i = 0;
  char *key, *value;
  size_t key_len, value_len;

  ssh_mapping_reset_index(mapping);
  while (ssh_mapping_get_next_vl(mapping, (void *) &key, &key_len,
                                 (void *) &value, &value_len))
    {
      fprintf(stderr, "[%d] key[%d] = `%s', value[%d] = `%s'\n",
              i++, key_len, key, value_len, value);
    }
}

int main(int argc, char **argv)
{
  int i, j;
  char *scheme, *host, *port, *username, *password, *path, *key, *value;
  size_t path_length, key_len, value_len;
  SshMapping mapping;
  Boolean want_nulls;

  for(i = 0; i < sizeof(tests) / sizeof(*tests); i++)
    {
      want_nulls = FALSE;
      parsefunc = FUNC_O;

      if (ssh_url_parse_and_decode(tests[i].url, &scheme, &host,
                                   &port, &username, &password, &path))
        {
          if (!tests[i].ok)
            ssh_fatal("%s returned true, even if it should have " \
                      "failed,\nurl = %s", parsefunc, tests[i].url);

#ifdef SSH_URL_PARSE_USE_RELAXED
          want_nulls = FALSE;
          parsefunc = FUNC_M;
          
          if (ssh_url_parse_relaxed_and_decode(tests[i].url, &scheme, &host,
                                               &port, &username, &password,
                                               &path))
            {
              if (!tests[i].gok)
                ssh_fatal("%s returned true, even if it " \
                          "should have failed, url = %s", parsefunc, tests[i].url);
            }
          else
            {
              if (tests[i].gok)
                ssh_fatal("%s returned false, even if it " \
                          "should have succeeded, url = %s", parsefunc, tests[i].url);
              want_nulls = TRUE;
            }
#endif
        }
      else
        {
          /* We are about to fail, be prepared */
          want_nulls = TRUE;

          if (tests[i].ok)
            {
              ssh_fatal("%s returned false, even if it should have " \
                        "succeeded,\nurl = %s", parsefunc, tests[i].url);
            }
#ifdef SSH_URL_PARSE_USE_RELAXED
          want_nulls = FALSE;
          parsefunc = FUNC_M;

          if (ssh_url_parse_relaxed_and_decode(tests[i].url, &scheme, &host,
                                               &port, &username, &password,
                                               &path))
            {
              if (!tests[i].gok)
                ssh_fatal("%s returned true, even if it " \
                          "should have failed, url = %s", parsefunc, tests[i].url);
            }
          else
            {
              if (tests[i].gok)
                ssh_fatal("%s returned false, even if it " \
                          "should have succeeded, url = %s", parsefunc, tests[i].url);
              want_nulls = TRUE;
            }
#endif
        }

      if (want_nulls)
        {
#define CHECK(s) \
         if (s != NULL) \
          { \
            ssh_fatal("%s did not return NULL for %s, url = %s", \
                      parsefunc, s, #s, tests[i].s, tests[i].url); \
          }

          CHECK(scheme);
          CHECK(host);
          CHECK(port);
          CHECK(username);
          CHECK(password);
          CHECK(path);
#undef CHECK
        }
      else
        {
#define CHECK(s) \
          if (s == NULL && tests[i].s != NULL) \
          { \
            fprintf(stderr, "%s\n\tschema:%s, host:%s, port:%s, username:%s,\n\tpassword:%s, path:%s\n", \
                    tests[i].url, scheme, host, port, username, password, path); \
            ssh_fatal("%s returned NULL for %s, it should have returned\n'%s' for url = %s", parsefunc, #s, tests[i].s, tests[i].url); \
          } \
          if (s != NULL && tests[i].s == NULL) \
          { \
            fprintf(stderr, "%s\n\tschema:%s, host:%s, port:%s, username:%s,\n\tpassword:%s, path:%s\n", \
                    tests[i].url, scheme, host, port, username, password, path); \
            ssh_fatal("%s returned '%s' for %s, it should have returned\nNULL for url = '%s'", parsefunc, s, #s, tests[i].url); \
          } \
          if (s != NULL && strcmp(s, tests[i].s) != 0) \
          { \
            fprintf(stderr, "%s\n\tschema:%s, host:%s, port:%s, username:%s,\n\tpassword:%s, path:%s\n", \
                    tests[i].url, scheme, host, port, username, password, path); \
            ssh_fatal("%s returned '%s' for %s, it should have returned\n'%s' for url = %s", parsefunc, s, #s, tests[i].s, tests[i].url); \
          } 
    
          CHECK(scheme);
          CHECK(host);
          CHECK(port);
          CHECK(username);
          CHECK(password);
          CHECK(path);
#undef CHECK
        }
    }

  for(i = 0; i < sizeof(form_tests) / sizeof(*form_tests); i++)
    {
      if (ssh_url_parse_form(form_tests[i].url, &path, &path_length,
                             &mapping))
        {
          if (!form_tests[i].ok)
            ssh_fatal("ssh_url_parse_form returned TRUE, should have returned " \
                      "FALSE, url = %s",form_tests[i].url);
        }
      else
        {
          if (form_tests[i].ok)
            ssh_fatal("ssh_url_parse_form returned FALSE, should have returned " \
                      "TRUE, url = %s", form_tests[i].url);
        }
      if (strcmp(form_tests[i].path, path) != 0)
        ssh_fatal("ssh_url_parse_form path check failed, url = %s, path = %s, " \
                  "should be %s", form_tests[i].url, path, form_tests[i].path);
      for(j = 0; j < 10; j++)
        {
          if (form_tests[i].table[j].key == NULL ||
              form_tests[i].table[j].value == NULL)
            continue;
          if (!ssh_mapping_get_vl(mapping, form_tests[i].table[j].key,
                                  strlen(form_tests[i].table[j].key),
                                  (void *) &value, &value_len))
            {
              mapping_print(mapping);
              ssh_fatal("ssh_url_parse_form mapping check failed, cannot find " \
                        "key %s from the mapping, url = %s",
                        form_tests[i].table[j].key,
                        form_tests[i].url);
            }
          if (strcmp(value, form_tests[i].table[j].value) != 0)
            {
              ssh_fatal("ssh_url_parse_form mapping check failed, value for key " \
                        "%s from the mapping does not match, is = %s, should be " \
                        "= %s, url = %s",
                        form_tests[i].table[j].key,
                        value,
                        form_tests[i].table[j].value,
                        form_tests[i].url);
            }
        }

      ssh_mapping_reset_index(mapping);
      while (ssh_mapping_get_next_vl(mapping, (void *) &key, &key_len,
                                     (void *) &value, &value_len))
        {
          for(j = 0; j < 10; j++)
            {
              if (form_tests[i].table[j].key == NULL ||
                  form_tests[i].table[j].value == NULL)
                continue;
              if (strcmp(key, form_tests[i].table[j].key) == 0)
                break;
            }
          if (j == 10)
            {
              mapping_print(mapping);
              ssh_fatal("ssh_url_parse_form mapping check failed, found key %s" \
                        ", with value %s that should not exists, url = %s",
                        key, value, form_tests[i].url);
            }
        }
      ssh_url_free_mapping(mapping);
    }
  return 0;
}
