/*

  ssh-f-configd-log.c

  Author: Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Main module of Secure Shell Configuration Daemon for F-Secure FSMA.

*/

#include "sshincludes.h"
#include "ssh-f-configd.h"

#include <syslog.h>
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */

#define SSH_DEBUG_MODULE "Ssh-F-ConfigD-Log"
static int ssh_f_configd_log_severity(SshLogSeverity severity);
static int ssh_f_configd_log_facility(SshLogFacility facility);

/* Helper functions for server_ssh_log */
static int ssh_f_configd_log_severity(SshLogSeverity severity)
{
  switch(severity)
    {
    case SSH_LOG_INFORMATIONAL:
#ifdef LOG_INFO
      return LOG_INFO;
#endif /* LOG_INFO */
      break;
    case SSH_LOG_NOTICE:
#ifdef LOG_NOTICE
      return LOG_NOTICE;
#endif /* LOG_NOTICE */
      break;
    case SSH_LOG_WARNING:
#ifdef LOG_WARNING
      return LOG_WARNING;
#endif /* LOG_WARNING */
      break;
    case SSH_LOG_ERROR:
#ifdef LOG_ERR
      return LOG_ERR;
#endif /* LOG_ERR */
      break;
    case SSH_LOG_CRITICAL:
#ifdef LOG_CRIT
      return LOG_CRIT;
#endif /* LOG_CRIT */
      break;
    }
  return 1;
}

static int ssh_f_configd_log_facility(SshLogFacility facility)
{
  switch (facility)
    {
    case SSH_LOGFACILITY_AUTH:
    case SSH_LOGFACILITY_SECURITY:
#ifdef LOG_AUTH
      return LOG_AUTH;
#endif /* LOG_AUTH */
      break;
    case SSH_LOGFACILITY_DAEMON:
#ifdef LOG_DAEMON
      return LOG_DAEMON;
#endif /* LOG_DAEMON */
      break;
    case SSH_LOGFACILITY_USER:
#ifdef LOG_USER
      return LOG_USER;
#endif /* LOG_USER */
      break;
    case SSH_LOGFACILITY_MAIL:
#ifdef LOG_MAIL
      return LOG_MAIL;
#endif /* LOG_MAIL */
      break;
    case SSH_LOGFACILITY_LOCAL0:
#ifdef LOG_LOCAL0
      return LOG_LOCAL0;
#endif /* LOG_LOCAL0 */
      break;
    case SSH_LOGFACILITY_LOCAL1:
#ifdef LOG_LOCAL1
      return LOG_LOCAL1;
#endif /* LOG_LOCAL1 */
      break;
    case SSH_LOGFACILITY_LOCAL2:
#ifdef LOG_LOCAL2
      return LOG_LOCAL2;
#endif /* LOG_LOCAL2 */
      break;
    case SSH_LOGFACILITY_LOCAL3:
#ifdef LOG_LOCAL3
      return LOG_LOCAL3;
#endif /* LOG_LOCAL3 */
      break;
    case SSH_LOGFACILITY_LOCAL4:
#ifdef LOG_LOCAL4
      return LOG_LOCAL4;
#endif /* LOG_LOCAL4 */
      break;
    case SSH_LOGFACILITY_LOCAL5:
#ifdef LOG_LOCAL5
      return LOG_LOCAL5;
#endif /* LOG_LOCAL5 */
      break;
    case SSH_LOGFACILITY_LOCAL6:
#ifdef LOG_LOCAL6
      return LOG_LOCAL6;
#endif /* LOG_LOCAL6 */
      break;
    case SSH_LOGFACILITY_LOCAL7:
#ifdef LOG_LOCAL7
      return LOG_LOCAL7;
#endif /* LOG_LOCAL7 */
      break;
    }
  return 1;
}

void ssh_f_configd_log_stderr(SshLogFacility facility, 
                              SshLogSeverity severity, 
                              const char *msg, 
                              void *context)
{
  fprintf(stderr, "%s\n", msg);
}

void ssh_f_configd_log(SshLogFacility facility,
                       SshLogSeverity severity, 
                       const char *msg, 
                       void *context)
{
  static Boolean openlog_called = FALSE;
  static pid_t openlog_pid;
  static Boolean openlog_facility = 0;
  int syslog_facility;

  SshFConfigDConf conf = (SshFConfigDConf)context;

  if (openlog_called && (openlog_pid != getpid()))
    {
      closelog();
      openlog_called = FALSE;
    }
  if (! openlog_called)
    {
      openlog_facility = ssh_f_configd_log_facility(SSH_LOGFACILITY_DAEMON);
      openlog(conf->av0, LOG_PID, openlog_facility);
      openlog_pid = getpid();
      openlog_called = TRUE;
    }
  syslog_facility = ssh_f_configd_log_facility(facility);
  syslog((((syslog_facility != openlog_facility) ? syslog_facility : 0) |
          ssh_f_configd_log_severity(severity)),
         msg);
}

/* eof (ssh-f-configd-log.c) */
