/*

  sshdconfigure.c

  Author: Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  (Re-)configuration tools for programs.

*/

#include "sshincludes.h"
#include "sshdsprintf.h"

#include "ssh-f-configd.h"

#define SSH_DEBUG_MODULE "Ssh-F-ConfigD-Ssh-Configure"

void ssh_f_configd_ssh_config_data_clear(SshFConfigDConf conf,
                                         SshFConfigDProg prog)
{
  ssh_buffer_clear(prog->config_data);
}

void ssh_f_configd_ssh_config_data_begin(SshFConfigDConf conf,
                                         SshFConfigDProg prog)
{
  char *t;

  ssh_f_configd_ssh_config_data_clear(conf, prog);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "# This configuration file was automatically generated for `");
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      prog->name);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "'\n");
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "# by SSH-F-ConfigD.  Data was retrieved from FSMA server.\n");
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "# THIS FILE IS CENTRALLY MANAGED. DON'T EDIT THIS FILE.\n");
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "# Configuration generation started at ");
  t = ssh_readable_time_string(ssh_time(), FALSE);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      t);
  ssh_xfree(t);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      " UTC\n");
  return;
}

void ssh_f_configd_ssh_config_data_add_string(SshFConfigDConf conf,
                                              SshFConfigDProg prog,
                                              const char *str)
{
  ssh_xbuffer_append(prog->config_data, str, strlen(str));
  return;
}

void ssh_f_configd_ssh_config_data_add_param(SshFConfigDConf conf,
                                             SshFConfigDProg prog,
                                             const char *key,
                                             const char *arg)
{
  ssh_f_configd_ssh_config_data_add_string(conf, prog, key);
  ssh_f_configd_ssh_config_data_add_string(conf, prog, " ");
  ssh_f_configd_ssh_config_data_add_string(conf, prog, arg);
  ssh_f_configd_ssh_config_data_add_string(conf, prog, "\n");
  return;
}

void ssh_f_configd_ssh_config_data_end(SshFConfigDConf conf,
                                       SshFConfigDProg prog)
{
  char *t;

  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      "\n# Configuration generation finished at ");
  t = ssh_readable_time_string(ssh_time(), FALSE);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      t);
  ssh_xfree(t);
  ssh_f_configd_ssh_config_data_add_string(
      conf,
      prog,
      " UTC\n");
  return;
}

void ssh_f_configd_ssh_config_data_activate(SshFConfigDConf conf,
                                            SshFConfigDProg prog)
{
  FILE *f;

  ssh_f_configd_ssh_config_data_end(conf, prog);
  remove(prog->config_file_bak);
  rename(prog->config_file, prog->config_file_bak);
  f = fopen(prog->config_file, "w");
  if (f == NULL)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                  "%s: Can't open ssh configuration file \"%s\" for writing.",
                  conf->av0, prog->config_file);
      ssh_buffer_clear(prog->config_data);
      remove(prog->config_file);
      rename(prog->config_file_bak, prog->config_file);
      return;
    }
  if (fwrite(ssh_buffer_ptr(prog->config_data), 
             1,
             ssh_buffer_len(prog->config_data),
             f) != ssh_buffer_len(prog->config_data))
    {
      fclose(f);
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
          "%s: Can't write %u bytes to ssh configuration file \"%s\".",
          conf->av0,
          ssh_buffer_len(prog->config_data), 
          prog->config_file);
      ssh_buffer_clear(prog->config_data);
      remove(prog->config_file);
      rename(prog->config_file_bak, prog->config_file);
      return;
    }
  fclose(f);
  ssh_buffer_clear(prog->config_data);
  ssh_f_configd_ssh_reconfigure(conf, prog);
  return;
}

void ssh_f_configd_ssh_config_data_dump(SshFConfigDConf conf,
                                        SshFConfigDProg prog)
{
  SshADTHandle handle;
  SshFConfigDOidKeyPair pair;
  char *t;

  ssh_f_configd_ssh_config_data_begin(conf, prog);
  handle = ssh_adt_enumerate_start(prog->oid_map);
  while (handle != SSH_ADT_INVALID)
    {
      pair = ssh_adt_map_lookup(prog->oid_map, handle);
      if (pair->val_is_raw)
        {
          if (pair->val != NULL)
            {
              ssh_f_configd_ssh_config_data_add_string(conf, 
                                                       prog,
                                                       pair->val_raw);
            }
        }
      else
        {
          ssh_f_configd_ssh_config_data_add_string(conf, prog, "\n# OID: ");
          ssh_f_configd_ssh_config_data_add_string(conf, prog, pair->oid);
          if (pair->timestamp > 0)
            {
              ssh_f_configd_ssh_config_data_add_string(conf,  prog, "\n# ");
              if (pair->updated)
                {
                  ssh_f_configd_ssh_config_data_add_string(conf, 
                                                           prog, 
                                                           "Updated");
                }
              else
                {
                  ssh_f_configd_ssh_config_data_add_string(conf,
                                                           prog,
                                                           "Initialized");
                }
              ssh_f_configd_ssh_config_data_add_string(conf, prog, ": ");
              t = ssh_readable_time_string(pair->timestamp, FALSE);
              ssh_f_configd_ssh_config_data_add_string(conf, prog, t);
              ssh_xfree(t);
              ssh_f_configd_ssh_config_data_add_string(conf, prog, " UTC\n");
            }
          if (pair->val != NULL)
            {
              ssh_f_configd_ssh_config_data_add_param(conf,
                                                      prog,
                                                      pair->key,
                                                      pair->val);
            }
          else
            {
              ssh_f_configd_ssh_config_data_add_string(conf, 
                                                       prog,
                                                       "\n# `");
              ssh_f_configd_ssh_config_data_add_string(conf, prog, pair->key);
              ssh_f_configd_ssh_config_data_add_string(
                                                     conf, 
                                                     prog,
                                                     "' is not set by FSMA\n");
            }
        }
      handle = ssh_adt_enumerate_next(prog->oid_map, handle);
    }
  ssh_f_configd_ssh_config_data_activate(conf, prog);
  return;
}

SshFConfigDPolicyTransferStatus ssh_f_configd_config_data_collect_generic(
                                                        SshFConfigDConf conf,
                                                        SshFConfigDProg prog)
{
  SshADTHandle handle;
  SshFConfigDOidKeyPair pair;
  DfPolicyStatus dr;
  SSH_F_CONFIGD_CHAR_TYPE *val_buf;
  char *new_val;
  unsigned int val_int;
  DfPolicyVar var;
  enum DfPolicyType var_type;
  Boolean something_retrieved;
  Boolean something_changed;

  something_retrieved = FALSE;
  something_changed = FALSE;
  if (DfpCreatePolicyVar(&var) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpCreatePolicyVar fails.", conf->av0);
      ssh_f_configd_terminate(conf, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  val_buf = ssh_xcalloc(SSH_F_CONFIGD_POLICY_VAR_MAX_LEN, sizeof(val_buf[0]));
  handle = ssh_adt_enumerate_start(prog->oid_map);
  new_val = NULL;
  while (handle != SSH_ADT_INVALID)
    {
      pair = ssh_adt_map_lookup(prog->oid_map, handle);
      pair->val_is_raw = FALSE;
      dr = DfpGetPolicy(conf->fsma_session,
                        pair->oid,
                        &var);
      switch (dr)
        {
        case DFP_SUCCESS:
          if (pair->oid_handler != NULL)
            {
              switch ((*pair->oid_handler)(conf, prog, pair, var))
                {
                case SSH_F_CONFIGD_POLICY_CHANGED:
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                            "%s: OID %s (%s) in `%s' updated by OID handler.",
                            conf->av0, pair->oid, pair->key, prog->name);
                  something_retrieved = TRUE;
                  something_changed = TRUE;
                  break;

                case SSH_F_CONFIGD_POLICY_NOT_CHANGED:
                  something_retrieved = TRUE;
                  break;

                case SSH_F_CONFIGD_POLICY_TRANSFER_FAILED:
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                            "%s: OID %s (%s) in `%s' handler fails.",
                            conf->av0, pair->oid, pair->key, prog->name);
                  break;
                }
              goto next_oid;
            }
          if (DfpGetType(var, &var_type) == DFP_SUCCESS)
            {
              switch (var_type)
                {
                case DFP_STRING:
                  if (DfpGetString(var, 
                                   val_buf, 
                                   SSH_F_CONFIGD_POLICY_VAR_MAX_LEN) ==
                      DFP_SUCCESS)
                    {
                      if (strcmp(val_buf,
                                 SSH_F_CONFIGD_FSMA_STRING_VALUE_NONE) != 0)
                        {
                          ssh_dsprintf(&new_val, "\"%s\"", val_buf);
                        }
                      else
                        {
                          new_val = NULL;
                        }
                      something_retrieved = TRUE;
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_STRING OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_DISPLAYSTRING:
                  if (DfpGetString(var, 
                                   val_buf, 
                                   SSH_F_CONFIGD_POLICY_VAR_MAX_LEN) ==
                      DFP_SUCCESS)
                    {
                      if (strcmp(val_buf,
                                 SSH_F_CONFIGD_FSMA_STRING_VALUE_NONE) != 0)
                        {
                          ssh_dsprintf(&new_val, "\"%s\"", val_buf);
                        }
                      else
                        {
                          new_val = NULL;
                        }
                      something_retrieved = TRUE;
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_DISPLAYSTRING OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_ADDRESS:
                  if (DfpGetString(var, 
                                   val_buf, 
                                   SSH_F_CONFIGD_POLICY_VAR_MAX_LEN) ==
                      DFP_SUCCESS)
                    {
                      if (strcmp(val_buf,
                                 SSH_F_CONFIGD_FSMA_STRING_VALUE_NONE) != 0)
                        {
                          ssh_dsprintf(&new_val, "\"%s\"", val_buf);
                        }
                      else
                        {
                          new_val = NULL;
                        }
                      something_retrieved = TRUE;
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_ADDRESS OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_NUMBER:
                  if (DfpGetInt(var, &val_int) == DFP_SUCCESS)
                    {
                      ssh_dsprintf(&new_val, "%u", val_int);
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_NUMBER OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_COUNTER:
                  if (DfpGetInt(var, &val_int) == DFP_SUCCESS)
                    {
                      ssh_dsprintf(&new_val, "%u", val_int);
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_COUNTER OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_TICKS:
                  if (DfpGetInt(var, &val_int) == DFP_SUCCESS)
                    {
                      ssh_dsprintf(&new_val, "%u", val_int);
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: Can't retrieve DFP_TICKS OID %s (%s).",
                          conf->av0, pair->oid, pair->key);
                    }
                  break;

                case DFP_TABLE:
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Value type DFP_TABLE for OID %s (%s) not supported.",
                    conf->av0, pair->oid, pair->key);
                  break;

                default:
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                                "%s: Invalid value type for OID %s (%s).",
                                conf->av0, pair->oid, pair->key);
                }
            }
          else
            {
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                            "%s: Invalid value type for OID %s (%s).",
                            conf->av0, pair->oid, pair->key);
            }
          break;

        case DFP_ERR_NO_VALUE:
          break;

        case DFP_ERR_NO_SUCH_OID:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                        "%s: FSMA don't recognize OID %s (%s).",
                        conf->av0, pair->oid, pair->key);
          break;

        default:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Error %d in retrieving OID %s (%s).",
                        conf->av0, (int)dr, pair->oid, pair->key);
          break;
        }
      if ((pair->val == NULL) && (new_val == NULL))
        {
          pair->updated = FALSE;
        }
      else if (pair->val == NULL)
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                   "%s: OID %s (%s) in `%s' changed from UNDEFINED to \"%s\".",
                    conf->av0, pair->oid, pair->key, prog->name, new_val);
          something_retrieved = TRUE;
          something_changed = TRUE;
          pair->val = new_val;
          pair->timestamp = ssh_time();
          pair->updated = FALSE;
          new_val = NULL;
        }
      else if (new_val == NULL)
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                   "%s: OID %s (%s) in `%s' changed from \"%s\" to UNDEFINED.",
                    conf->av0, pair->oid, pair->key, prog->name, pair->val);
          something_changed = TRUE;
          ssh_xfree(pair->val);
          pair->val = NULL;
          pair->timestamp = 0;
          pair->updated = FALSE;
        }
      else if (strcmp(pair->val, new_val) != 0)
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
             "%s: OID %s (%s) in `%s' changed from \"%s\" to \"%s\".",
             conf->av0, pair->oid, pair->key, prog->name, pair->val, new_val);
          something_retrieved = TRUE;
          something_changed = TRUE;
          ssh_xfree(pair->val);
          pair->val = new_val;
          pair->timestamp = ssh_time();
          pair->updated = TRUE;
          new_val = NULL;
        }
      else
        {
          something_retrieved = TRUE;
          ssh_xfree(new_val);
          new_val = NULL;
        }
    next_oid:
      handle = ssh_adt_enumerate_next(prog->oid_map, handle);
    }
  ssh_xfree(val_buf);
  DfpClose(var);
  if (something_changed)
    return SSH_F_CONFIGD_POLICY_CHANGED;
  else if (something_retrieved)
    return SSH_F_CONFIGD_POLICY_NOT_CHANGED;
  else
    return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
  /*NOTREACHED*/;
}

void ssh_f_configd_ssh_config_data_process(SshFConfigDConf conf,
                                           SshFConfigDProg prog)
{
  SshFConfigDPolicyTransferStatus pr;

  switch (ssh_f_configd_config_data_collect_generic(conf, prog))
    {
    case SSH_F_CONFIGD_POLICY_CHANGED:
      SSH_DEBUG(5, ("Policy was retrieved and updated (%s).",
                    prog->name));
      prog->configuration_valid = TRUE;
      ssh_f_configd_ssh_config_data_dump(conf, prog);
      break;

    case SSH_F_CONFIGD_POLICY_NOT_CHANGED:
      SSH_DEBUG(5, ("Policy was retrieved but unchanged (%s).",
                    prog->name));
      prog->configuration_valid = TRUE;
      break;

    case SSH_F_CONFIGD_POLICY_TRANSFER_FAILED:
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Policy transfer failed (%s).",
                    conf->av0,
                    prog->name);
      break;
    }
  return;
}

void ssh_f_configd_ssh_pid_refresh(SshFConfigDConf conf,
                                   SshFConfigDProg prog)
{
  FILE *f;
  char buf[16], *tmp;
  pid_t pid;
  unsigned long x;

  prog->pid = 0;
  if (prog->pid_file == NULL)
    return;
  f = fopen(prog->pid_file, "r");
  if (f == NULL)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                    "%s: Can't open sshd pidfile \"%s\"", 
                    conf->av0, prog->pid_file);
      return;
    }
  tmp = fgets(buf, sizeof (buf), f);
  fclose(f);
  if (tmp == NULL)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                    "%s: Can't read sshd pidfile \"%s\"", 
                    conf->av0, prog->pid_file);
      return;
    }
  x = strtoul(tmp, NULL, 10);
  pid = (pid_t)x;
  if ((pid < 2) || (pid != x))
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Invalid pid in pidfile \"%s\"", 
                    conf->av0, prog->pid_file);
      return;
    }
  if (kill(pid, 0) != 0)
    {
      switch (errno)
        {
#ifdef ESRCH
        case ESRCH:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Process ID %u does not exist.", 
                        conf->av0, (unsigned int)pid);
          break;
#endif /* ESRCH */

#ifdef EPERM
        case EPERM:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Signaling process ID %u not allowed.", 
                        conf->av0, (unsigned int)pid);
          break;
#endif /* EPERM */

#ifdef EACCESS
        case EACCESS:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Signaling process ID %u not allowed.", 
                        conf->av0, (unsigned int)pid);
          break;
#endif /* EACCESS */
          
        default:
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Unable to signal process ID %u.", 
                        conf->av0, (unsigned int)pid);
        }
      return;
    }
  prog->pid = pid;
  SSH_DEBUG(7, ("sshd_pid=%u", (unsigned long)(prog->pid)));
  return;
}

void ssh_f_configd_ssh_reconfigure(SshFConfigDConf conf,
                                   SshFConfigDProg prog)
{
  ssh_f_configd_ssh_pid_refresh(conf, prog);
  if (prog->pid == 0)
    {
      if (prog->pid_file != NULL)
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Unable to find reconfigurable %s.", 
                        conf->av0,
                        prog->name);
        }
      return;
    }
  if (kill(prog->pid, SIGHUP) != 0)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                    "%s: Unable to reconfigure %s.", 
                    conf->av0,
                    prog->name);
      return;
    }
  SSH_DEBUG(5, ("%s[%u] reconfigured", 
                prog->name, 
                (unsigned long)(prog->pid)));
  return;
}

/* eof (sshdconfigure.c) */
