/*
 * gnome-print-ps2.c: A Postscript driver for GnomePrint based	in
 * gnome-print-pdf which was based on the PS driver by Raph Levien.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *   Chema Celorio <chema@celorio.com>
 *   Lauris Kaplinski <lauris@helixcode.com>
 *
 * Copyright 2000 Jose M Celorio and Ximian, Inc.
 *
 */

/* __FUNCTION__ is not defined in Irix according to David Kaelbling <drk@sgi.com>*/
#ifndef __GNUC__
  #define __FUNCTION__   ""
#endif
#define debug(section,str) if (FALSE) printf ("%s:%d (%s) %s\n", __FILE__, __LINE__, __FUNCTION__, str); 
	
#include "config.h"
#include <math.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_misc.h>
#include <libgnome/gnome-paper.h>
#include <libgnomeprint/gp-unicode.h>
#include <libgnomeprint/gnome-print-private.h>
#include <libgnomeprint/gnome-printer-private.h>
#include <libgnomeprint/gnome-print-ps2.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gnome-print-encode.h>
#include <libgnomeprint/gnome-print-encode-private.h>

#include <libgnomeprint/gnome-pgl-private.h>

#ifdef ENABLE_LIBGPA
#include <libgpa/gpa-printer.h>
#include <libgpa/gpa-settings.h>
#endif

#define EOL "\n"

typedef struct _GPPS2Font GPPS2Font;
typedef struct _GPPS2Page GPPS2Page;

static gchar *ps2_dict =
  "/|/def load def/,/load load" EOL
  "|/m/moveto , |/l/lineto , |/c/curveto , |/q/gsave ," EOL
  "|/Q/grestore , |/rg/setrgbcolor , |/J/setlinecap ," EOL
  "|/j/setlinejoin , |/w/setlinewidth , |/M/setmiterlimit ," EOL
  "|/d/setdash , |/i/pop , |/W/clip , |/W*/eoclip , |/n/newpath ," EOL
  "|/S/stroke , |/f/fill , |/f*/eofill , |/Tj/show , |/Tm/moveto ," EOL
  "|/FF/findfont ," EOL
  "|/h/closepath , |/cm/concat , |/rm/rmoveto , |/sp/strokepath ," EOL
  "|/SP/showpage , |/p/pop , |/EX/exch , |/DF/definefont , |" EOL
  /* Define the procedure for reencoding fonts */
  "/RE {dup length dict begin {1 index /FID ne {def} {p p} ifelse}forall" EOL
  "/Encoding ISOLatin1Encoding def currentdict end} |" EOL
  "/F {scalefont setfont} def" EOL;

/* These are the PostScript 35, assumed to be in the printer. */
static char *ps2_internal_fonts[] = {
	"AvantGarde-Book",
	"AvantGarde-BookOblique",
	"AvantGarde-Demi",
	"AvantGarde-DemiOblique",
	"Bookman-Demi",
	"Bookman-DemiItalic",
	"Bookman-Light",
	"Bookman-LightItalic",
	"Courier",
	"Courier-Bold",
	"Courier-BoldOblique",
	"Courier-Oblique",
	"Helvetica",
	"Helvetica-Bold",
	"Helvetica-BoldOblique",
	"Helvetica-Narrow",
	"Helvetica-Narrow-Bold",
	"Helvetica-Narrow-BoldOblique",
	"Helvetica-Narrow-Oblique",
	"Helvetica-Oblique",
	"NewCenturySchlbk-Bold",
	"NewCenturySchlbk-BoldItalic",
	"NewCenturySchlbk-Italic",
	"NewCenturySchlbk-Roman",
	"Palatino-Bold",
	"Palatino-BoldItalic",
	"Palatino-Italic",
	"Palatino-Roman",
	"Symbol",
	"Times-Bold",
	"Times-BoldItalic",
	"Times-Italic",
	"Times-Roman",
	"ZapfChancery-MediumItalic",
	"ZapfDingbats"
};
#define num_ps2_internal_fonts (sizeof (ps2_internal_fonts) / sizeof (ps2_internal_fonts[0]))

struct _GPPS2Font {
	GPPS2Font *next;
	GnomeFontFace *face;
	const gchar *psname;
	gchar *encodedname;
	gboolean internal;
	gboolean encoded;
};

struct _GPPS2Page {
	GPPS2Page *next;
	gchar *name;
	gint number;
	gboolean shown;
};

struct _GnomePrintPs2 {
	GnomePrintContext pc;

#ifdef ENABLE_LIBGPA	
	/* GpaPrinter */
	GpaPrinter *gpa_printer;
	GpaSettings *gpa_settings;
#endif

	GPPS2Font *fonts;

	const GnomeFont *private_font;
	gint private_font_flag;
        gdouble r, g, b;
	gint private_color_flag;

	GPPS2Page *pages;

	gint gsave_level;
	gint ps_level;
};

struct _GnomePrintPs2Class
{
	GnomePrintContextClass parent_class;
};

static void gnome_print_ps2_class_init (GnomePrintPs2Class *klass);
static void gnome_print_ps2_init (GnomePrintPs2 *PS2);
static void gnome_print_ps2_destroy (GtkObject *object);

static gint gnome_print_ps2_gsave (GnomePrintContext *pc);
static gint gnome_print_ps2_grestore (GnomePrintContext *pc);
static gint gnome_print_ps2_fill (GnomePrintContext *pc, ArtWindRule rule);
static gint gnome_print_ps2_clip (GnomePrintContext *pc, ArtWindRule rule);
static gint gnome_print_ps2_stroke (GnomePrintContext *pc);
static gint gnome_print_ps2_glyphlist (GnomePrintContext *pc, GnomeGlyphList *gl);
static gint gnome_print_ps2_grayimage (GnomePrintContext *pc, const char *data, gint width, gint height, gint rowstride);
static gint gnome_print_ps2_rgbimage (GnomePrintContext *pc, const char *data, gint width, gint height, gint rowstride);
static gint gnome_print_ps2_beginpage (GnomePrintContext *pc, const char *name);
static gint gnome_print_ps2_showpage (GnomePrintContext *pc);
static gint gnome_print_ps2_close (GnomePrintContext *pc);

static gint gp_ps2_set_color (GnomePrintContext *pc);
static gint gp_ps2_set_line (GnomePrintContext *pc);
static gint gp_ps2_set_dash (GnomePrintContext *pc);

static gint gp_ps2_set_color_private (GnomePrintContext *pc, gdouble r, gdouble g, gdouble b);
static gint gp_ps2_set_font_private (GnomePrintContext *pc, const GnomeFont *font);

static gint gp_ps2_encode_font (GnomePrintContext *pc, GPPS2Font *font, const gchar *privatename);
static GPPS2Font *gp_ps2_download_and_encode_font (GnomePrintContext *pc, const GnomeFontFace *face);
static gint gp_ps2_print_path (GnomePrintContext *pc, const GPPath *gppath);
static int gnome_print_ps2_image (GnomePrintContext *pc, const char *data, int width, int height, int rowstride, int bytes_per_pixel);

static gchar* gnome_print_ps2_get_date (void);


#ifdef ENABLE_LIBGPA
static gint gnome_print_ps2_get_level (GnomePrintPs2 *ps2);
#endif

static GnomePrintContextClass *parent_class;

GtkType
gnome_print_ps2_get_type (void)
{
	static GtkType ps2_type = 0;

	if (!ps2_type) {
		GtkTypeInfo ps2_info = {
			"GnomePrintps2",
			sizeof (GnomePrintPs2),
			sizeof (GnomePrintPs2Class),
			(GtkClassInitFunc)  gnome_print_ps2_class_init,
			(GtkObjectInitFunc) gnome_print_ps2_init,
			NULL, NULL, NULL
		};
		ps2_type = gtk_type_unique (gnome_print_context_get_type (), &ps2_info);
	}

	return ps2_type;
}

static void
gnome_print_ps2_class_init (GnomePrintPs2Class *klass)
{
	GtkObjectClass *object_class;
	GnomePrintContextClass *pc_class;

	object_class = (GtkObjectClass *) klass;
	pc_class = (GnomePrintContextClass *)klass;

	parent_class = gtk_type_class (gnome_print_context_get_type ());
	
	object_class->destroy = gnome_print_ps2_destroy;

	pc_class->gsave = gnome_print_ps2_gsave;
	pc_class->grestore = gnome_print_ps2_grestore;
	pc_class->fill = gnome_print_ps2_fill;
	pc_class->clip = gnome_print_ps2_clip;
	pc_class->stroke = gnome_print_ps2_stroke;
	pc_class->glyphlist = gnome_print_ps2_glyphlist;
	pc_class->grayimage = gnome_print_ps2_grayimage;
	pc_class->rgbimage = gnome_print_ps2_rgbimage;
	pc_class->beginpage = gnome_print_ps2_beginpage;
	pc_class->showpage = gnome_print_ps2_showpage;
	pc_class->close = gnome_print_ps2_close;
}

static void
gnome_print_ps2_init (GnomePrintPs2 *ps2)
{
	GPPS2Font *f;
	gint i;

	ps2->ps_level = 2;
	
	ps2->gsave_level = 0;

	ps2->fonts = NULL;

	for (i = 0; i < num_ps2_internal_fonts; i++) {
		f = g_new (GPPS2Font, 1);
		f->next = ps2->fonts;
		ps2->fonts = f;
		f->face = NULL;
		f->psname = ps2_internal_fonts[i];
		f->encodedname = NULL;
		f->internal = TRUE;
		f->encoded = FALSE;
	}
	ps2->private_font = NULL;
	ps2->private_font_flag = GP_GC_FLAG_UNSET;
	ps2->private_color_flag = GP_GC_FLAG_UNSET;

	ps2->pages = NULL;
}

static void
gnome_print_ps2_destroy (GtkObject *object)
{
	GnomePrintPs2 *ps2;

	ps2 = GNOME_PRINT_PS2 (object);

	while (ps2->pages) {
		GPPS2Page *p;
		p = ps2->pages;
		if (!p->shown) g_warning ("page %d was not shown", p->number);
		if (p->name) g_free (p->name);
		ps2->pages = p->next;
		g_free (p);
	}

	while (ps2->fonts) {
		GPPS2Font *f;
		f = ps2->fonts;
		if (f->face) gtk_object_unref (GTK_OBJECT (f->face));
		if (f->encodedname) g_free (f->encodedname);
		ps2->fonts = f->next;
		g_free (f);
	}

	if (ps2->private_font) gnome_font_unref (ps2->private_font);
	ps2->private_font = NULL;
	ps2->private_font_flag = GP_GC_FLAG_UNSET;
	ps2->private_color_flag = GP_GC_FLAG_UNSET;

	if (* GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static gint
gnome_print_ps2_gsave (GnomePrintContext *pc)
{
	((GnomePrintPs2 *) pc)->gsave_level += 1;

	return gnome_print_context_fprintf (pc, "q" EOL);
}

static gint
gnome_print_ps2_grestore (GnomePrintContext *pc)
{
	GnomePrintPs2 *ps2;

	ps2 = (GnomePrintPs2 *) pc;

	g_return_val_if_fail (ps2->gsave_level > 0, GNOME_PRINT_ERROR_UNKNOWN);

	ps2->gsave_level -= 1;
	ps2->private_font_flag = GP_GC_FLAG_UNSET;
	ps2->private_color_flag = GP_GC_FLAG_UNSET;

	return gnome_print_context_fprintf (pc, "Q" EOL);
}

static gint
gnome_print_ps2_fill (GnomePrintContext *pc, ArtWindRule rule)
{
	gint ret;

	g_return_val_if_fail (gp_gc_has_currentpath (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPATH);
	g_return_val_if_fail (gp_path_all_closed (gp_gc_get_currentpath (pc->gc)), GNOME_PRINT_ERROR_BADVALUE);

	gp_ps2_set_color (pc);

	ret = gp_ps2_print_path (pc, gp_gc_get_currentpath (pc->gc));

	if (rule == ART_WIND_RULE_NONZERO) {
		ret += gnome_print_context_fprintf (pc, "f" EOL);
	} else {
		ret += gnome_print_context_fprintf (pc, "f*" EOL);
	}

	return ret;
}

static gint
gnome_print_ps2_clip (GnomePrintContext *pc, ArtWindRule rule)
{
	gint ret;

	g_return_val_if_fail (gp_gc_has_currentpath (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPATH);
	g_return_val_if_fail (gp_path_all_closed (gp_gc_get_currentpath (pc->gc)), GNOME_PRINT_ERROR_BADVALUE);

	ret = gp_ps2_print_path (pc, gp_gc_get_currentpath (pc->gc));

	if (rule == ART_WIND_RULE_NONZERO) {
		ret += gnome_print_context_fprintf (pc, "W n" EOL);
	} else {
		ret += gnome_print_context_fprintf (pc, "W* n" EOL);
	}

	return ret;
}


static gint
gnome_print_ps2_stroke (GnomePrintContext *pc)
{
	gint ret;

	g_return_val_if_fail (gp_gc_has_currentpath (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPATH);

	gp_ps2_set_color (pc);
	gp_ps2_set_line (pc);
	gp_ps2_set_dash (pc);

	ret = gp_ps2_print_path (pc, gp_gc_get_currentpath (pc->gc));

	ret += gnome_print_context_fprintf (pc, "S" EOL);

	return ret;
}

static gint
gnome_print_ps2_glyphlist (GnomePrintContext *pc, GnomeGlyphList *gl)
{
	static gdouble id[] = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
	GnomePrintPs2 *ps2;
	GnomePosGlyphList *pgl;
	const gdouble *ctm;
	const ArtPoint *cp;
	gboolean identity;
	gdouble dx, dy;
	gint ret, s;

	ps2 = (GnomePrintPs2 *) pc;

	g_return_val_if_fail (gp_gc_has_currentpoint (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPOINT);

	ctm = gp_gc_get_ctm (pc->gc);
	cp = gp_gc_get_currentpoint (pc->gc);
	identity = art_affine_equal ((double *) ctm, (double *) id);

	if (!identity) {
		ret = gnome_print_context_fprintf (pc, "q" EOL);
		ret += gnome_print_context_fprintf (pc, "[%g %g %g %g %g %g]cm" EOL, ctm[0], ctm[1], ctm[2], ctm[3], cp->x, cp->y);
		dx = dy = 0.0;
	} else {
		dx = cp->x;
		dy = cp->y;
	}

	pgl = gnome_pgl_from_gl (gl, id, GNOME_PGL_RENDER_DEFAULT);

	for (s = 0; s < pgl->num_strings; s++) {
		GnomePosString * ps;
		gint i;

		ps = pgl->strings + s;

		ret += gp_ps2_set_font_private (pc, gnome_rfont_get_font (ps->rfont));
		ret += gp_ps2_set_color_private (pc,
						 ((ps->color >> 24) & 0xff) / 255.0,
						 ((ps->color >> 16) & 0xff) / 255.0,
						 ((ps->color >>  8) & 0xff) / 255.0);
		ret += gnome_print_context_fprintf (pc, "%g %g Tm" EOL, pgl->glyphs[ps->start].x + dx, pgl->glyphs[ps->start].y + dy);

		/* Build string */
		ret += gnome_print_context_fprintf (pc, "(");
		for (i = ps->start; i < ps->start + ps->length; i++) {
			gint glyph, page;
			glyph = pgl->glyphs[i].glyph & 0xff;
			page = (pgl->glyphs[i].glyph >> 8) & 0xff;
			ret += gnome_print_context_fprintf (pc, "\\%o\\%o", page, glyph);
		}
		ret += gnome_print_context_fprintf (pc, ")" EOL);

		/* Build array */
		ret += gnome_print_context_fprintf (pc, "[");
		for (i = ps->start + 1; i < ps->start + ps->length; i++) {
			ret += gnome_print_context_fprintf (pc, "%g %g ",
							    pgl->glyphs[i].x - pgl->glyphs[i-1].x,
							    pgl->glyphs[i].y - pgl->glyphs[i-1].y);
		}
		ret += gnome_print_context_fprintf (pc, "0 0] ");

		/* xyshow */
		ret += gnome_print_context_fprintf (pc, "xyshow" EOL);
	}

	if (!identity) {
		ret = gnome_print_context_fprintf (pc, "Q" EOL);
	}

	gnome_pgl_destroy (pgl);

	return 1;
}

static gint
gnome_print_ps2_grayimage (GnomePrintContext *pc, const char *data, gint width, gint height, gint rowstride)
{
	return gnome_print_ps2_image (pc, data, width, height, rowstride, 1);
}

static gint
gnome_print_ps2_rgbimage (GnomePrintContext *pc, const char *data, gint width, gint height, gint rowstride)
{
	return gnome_print_ps2_image (pc, data, width, height, rowstride, 3);
}

static gint
gnome_print_ps2_beginpage (GnomePrintContext *pc, const char *name)
{
	GnomePrintPs2 *ps2;
	GPPS2Page *p;
	gint number;

	ps2 = GNOME_PRINT_PS2 (pc);

	g_return_val_if_fail (!ps2->pages || ps2->pages->shown, GNOME_PRINT_ERROR_UNKNOWN);

	number = ps2->pages ? ps2->pages->number : 0;

	p = g_new (GPPS2Page, 1);
	p->next = ps2->pages;
	p->name = g_strdup (name);
	p->number = number + 1;
	p->shown = FALSE;
	ps2->pages = p;

	return gnome_print_context_fprintf (pc, "%%%%Page: %s\n", name);
}


static gint
gnome_print_ps2_showpage (GnomePrintContext *pc)
{
	GnomePrintPs2 *ps2;

	ps2 = GNOME_PRINT_PS2 (pc);

#if 0
	g_return_val_if_fail (ps2->pages, GNOME_PRINT_ERROR_UNKNOWN);
	g_return_val_if_fail (!ps2->pages->shown, GNOME_PRINT_ERROR_UNKNOWN);
#else
	if (!ps2->pages) {
		g_warning ("missing beginpage in print job");
	} else {
		g_return_val_if_fail (!ps2->pages->shown, GNOME_PRINT_ERROR_UNKNOWN);
	}
#endif
	g_return_val_if_fail (ps2->gsave_level == 0, GNOME_PRINT_ERROR_UNKNOWN);

	if (ps2->pages) ps2->pages->shown = TRUE;
	ps2->private_font_flag = GP_GC_FLAG_UNSET;
	ps2->private_color_flag = GP_GC_FLAG_UNSET;

	return gnome_print_context_fprintf (pc,"SP" EOL);
}

static gint
gnome_print_ps2_close (GnomePrintContext *pc)
{
	GnomePrintPs2 *ps2;
	gint ret = 0;

	ps2 = GNOME_PRINT_PS2 (pc);

	ret += gnome_print_context_fprintf (pc, "%%%%EOF" EOL);

	gnome_print_context_close_file (pc);
	
	return ret;
}

static gint
gp_ps2_set_color (GnomePrintContext *pc)
{
	return gp_ps2_set_color_private (pc, gp_gc_get_red (pc->gc), gp_gc_get_green (pc->gc), gp_gc_get_blue (pc->gc));
}

static gint
gp_ps2_set_line (GnomePrintContext *pc)
{
	gint ret;

	if (gp_gc_get_line_flag (pc->gc) == GP_GC_FLAG_CLEAR) return 0;

	ret = gnome_print_context_fprintf (pc, "%g w %i J %i j %g M" EOL,
					   gp_gc_get_linewidth (pc->gc),
					   gp_gc_get_linecap (pc->gc),
					   gp_gc_get_linejoin (pc->gc),
					   gp_gc_get_miterlimit (pc->gc));
	gp_gc_set_line_flag (pc->gc, GP_GC_FLAG_CLEAR);

	return ret;
}

static gint
gp_ps2_set_dash (GnomePrintContext *pc)
{
	const ArtVpathDash *dash;
	gint ret, i;

	if (gp_gc_get_dash_flag (pc->gc) == GP_GC_FLAG_CLEAR) return 0;

	dash = gp_gc_get_dash (pc->gc);
	ret = gnome_print_context_fprintf (pc, "[");
	for (i = 0; i < dash->n_dash; i++)
		ret += gnome_print_context_fprintf (pc, " %g", dash->dash[i]);
	ret += gnome_print_context_fprintf (pc, "]%g d" EOL, dash->n_dash > 0 ? dash->offset : 0.0);
	gp_gc_set_dash_flag (pc->gc, GP_GC_FLAG_CLEAR);

	return ret;
}

static gint
gp_ps2_set_color_private (GnomePrintContext *pc, gdouble r, gdouble g, gdouble b)
{
	GnomePrintPs2 *ps2;
	gint ret;

	ps2 = (GnomePrintPs2 *) pc;

	if ((ps2->private_color_flag == GP_GC_FLAG_CLEAR) && (r == ps2->r) && (g == ps2->g) && (b == ps2->b)) return 0;

	ret = gnome_print_context_fprintf (pc, "%.3g %.3g %.3g rg" EOL, r, g, b);

	ps2->r = r;
	ps2->g = g;
	ps2->b = b;
	ps2->private_color_flag = GP_GC_FLAG_CLEAR;

	return ret;
}

static gint
gp_ps2_set_font_private (GnomePrintContext *pc, const GnomeFont *font)
{
	GnomePrintPs2 *ps2;
	const GnomeFontFace *face;
	const gchar *name;
	GPPS2Font *f;
	gint ret;

	ps2 = (GnomePrintPs2 *) pc;

	if ((ps2->private_font_flag == GP_GC_FLAG_CLEAR) && (ps2->private_font == font)) return 0;

	face = gnome_font_get_face (font);
	name = gnome_font_face_get_ps_name (face);

	for (f = ps2->fonts; f != NULL; f = f->next) {
		if (!strcmp (name, f->psname)) {
			/* We found it */
			if (f->face == NULL) {
				f->face = (GnomeFontFace *) face;
				gnome_font_face_ref (face);
			}
			if (!f->encoded) {
				gp_ps2_encode_font (pc, f, f->psname);
			}
			break;
		}
	}
	if (f == NULL) {
		f = gp_ps2_download_and_encode_font (pc, face);
	}
	g_return_val_if_fail (f != NULL, GNOME_PRINT_ERROR_UNKNOWN);

	ret = gnome_print_context_fprintf (pc, "/%s FF %g F" EOL, f->encodedname, gnome_font_get_size (font));
	gnome_font_ref (font);
	if (ps2->private_font) gnome_font_unref (ps2->private_font);
	ps2->private_font = font;
	ps2->private_font_flag = GP_GC_FLAG_CLEAR;

	return ret;
}

#define gpcf gnome_print_context_fprintf

static gint
gp_ps2_encode_font (GnomePrintContext *pc, GPPS2Font *font, const gchar *privatename)
{
	GnomeFontFace * face;
	gint nglyphs, nfonts;
	gint i, j;

	/* Bitch 'o' bitches (Lauris) ! */

	face = font->face;
	g_return_val_if_fail (face != NULL, -1);
	g_return_val_if_fail (GNOME_IS_FONT_FACE (face), -1);
	font->encodedname = g_strdup_printf ("GnomeUni-%s" EOL, font->psname);

	nglyphs = gnome_font_face_get_num_glyphs (face);
	nfonts = (nglyphs + 255) >> 8;

	gpcf (pc, "32 dict begin" EOL);

	/* Common entries */
	gpcf (pc, "/FontType 0 def" EOL);
	gpcf (pc, "/FontMatrix [1 0 0 1 0 0] def" EOL);
	gpcf (pc, "/FontName /%s-Glyph-Composite def" EOL, font->psname);
	gpcf (pc, "/LanguageLevel 2 def" EOL);

	/* Type 0 entries */
	gpcf (pc, "/FMapType 2 def" EOL);

	/* Bitch 'o' bitches */
	gpcf (pc, "/FDepVector [" EOL);
	for (i = 0; i < nfonts; i++) {
		gpcf (pc, "/%s FF" EOL, privatename);
		gpcf (pc, "dup length dict begin {1 index /FID ne {def} {pop pop} ifelse} forall" EOL);
		gpcf (pc, "/Encoding [" EOL);
		for (j = 0; j < 256; j++) {
			gint glyph;
			glyph = 256 * i + j;
			if (glyph >= nglyphs) glyph = 0;
			gpcf (pc, ((j & 0x0f) == 0x0f) ? "/%s" EOL : "/%s ", gnome_font_face_get_glyph_ps_name (face, glyph));
		}
		gpcf (pc, "] def" EOL);
		gpcf (pc, "currentdict end /%s-Glyph-Page-%d exch definefont" EOL, font->psname, i);
	}
	gpcf (pc, "] def" EOL);
	gpcf (pc, "/Encoding [" EOL);
	for (i = 0; i < 256; i++) {
		gint fn;
		fn = (i < nfonts) ? i : 0;
		gpcf (pc, ((i & 0x0f) == 0x0f) ? "%d" EOL : "%d  ", fn);
	}
	gpcf (pc, "] def" EOL);
	gpcf (pc, "currentdict end" EOL);
	gpcf (pc, "/%s EX DF p" EOL, font->encodedname);

	font->encoded = TRUE;

	return 0;
}

#undef gpcf

static GPPS2Font *
gp_ps2_download_and_encode_font (GnomePrintContext *pc, const GnomeFontFace *face)
{
	GnomePrintPs2 *ps2;
	GPPS2Font *font;
	gchar *pfbname;
	gchar *pfa;

	ps2 = (GnomePrintPs2 *) pc;

	pfa = gnome_font_face_get_pfa (face);
	g_return_val_if_fail (pfa != NULL, NULL);
	gnome_print_context_fprintf (pc, "%s", pfa);

	font = g_new (GPPS2Font, 1);
	font->next = ps2->fonts;
	ps2->fonts = font;
	font->face = (GnomeFontFace *) face;
	gnome_font_face_ref (face);
	font->psname = gnome_font_face_get_ps_name (face);
	font->encodedname = NULL;
	font->internal = FALSE;
	font->encoded = FALSE;

	gtk_object_get (GTK_OBJECT (face), "pfbname", &pfbname, NULL);
	gp_ps2_encode_font (pc, font, pfbname);
	g_free (pfbname);

	return font;
}

GnomePrintPs2 *
gnome_print_ps2_new (GnomePrinter *printer)
{
	GnomePrintPs2 *ps2;
	GnomePrintContext *pc;
	gchar *date;
	gint ret = 0;

	g_return_val_if_fail (printer != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINTER (printer), NULL);
	
	ps2 = gtk_type_new (gnome_print_ps2_get_type ());

#ifdef ENABLE_LIBGPA
	g_return_val_if_fail (GPA_IS_PRINTER (printer->gpa_printer), NULL);
	/* FIXME: We take the first settings for now */
	g_return_val_if_fail (GPA_IS_SETTINGS (printer->gpa_settings), NULL);
	
	ps2->gpa_printer = printer->gpa_printer;
	ps2->gpa_settings = printer->gpa_settings;

	/* We don't need to ref the printer, because the settings should
	 * ref it (and unref it) but I don't think libgpa settings is
	 * refing the printer now. Just be safe now, but FIX gpa-settings
	 * so that it refs the printer. Chema
	 */
	gpa_printer_ref (printer->gpa_printer);
	gpa_settings_ref (printer->gpa_settings);

	ps2->ps_level = gnome_print_ps2_get_level (ps2);
	g_print ("PSLEVEL IS %i\n", ps2->ps_level);
	
#endif

	if (!gnome_print_context_open_file (GNOME_PRINT_CONTEXT (ps2), printer->filename))
		goto failure;

	pc = GNOME_PRINT_CONTEXT (ps2);

	date = gnome_print_ps2_get_date ();

	ret = gnome_print_context_fprintf (pc,
					   "%%!PS-Adobe-2.0\n"
					   "%%%% Creator: Gnome Print Version %s\n"
					   "%%%% DocumentName: %s\n"
					   "%%%% Author: %s\n"
					   "%%%% Pages: (atend)\n"
					   "%%%% Date: %s\n"
					   "%%%% driver : gnome-print-ps2\n"
					   "%%%% EndComments\n\n\n",
					   VERSION,
					   "Document Name Goes Here",
					   "Author Goes Here",
					   date);

	g_free (date);
	
	if ( ret < 0)
		goto failure;

	ret += gnome_print_context_fprintf (pc, ps2_dict);
	
	if ( ret < 0)
		goto failure;

	return ps2;

	failure:
	
	g_warning ("gnome_print_ps2_new: ps2 new failure ..\n");
	gtk_object_unref (GTK_OBJECT (ps2));
	return NULL;
}

static gint
gp_ps2_print_path (GnomePrintContext *pc, const GPPath *gppath)
{
	const ArtBpath *path;
	gboolean started, closed;
	
	path = gp_path_bpath (gppath);
	
	started = FALSE;
	closed = FALSE;
	for (; path->code != ART_END; path++)
		switch (path->code) {
		case ART_MOVETO_OPEN:
			if (started && closed) gnome_print_context_fprintf (pc, "h" EOL);
			closed = FALSE;
			started = FALSE;
			gnome_print_context_fprintf (pc, "%g %g m" EOL, path->x3, path->y3);
			break;
		case ART_MOVETO:
			if (started && closed) gnome_print_context_fprintf (pc, "h" EOL);
			closed = TRUE;
			started = TRUE;
			gnome_print_context_fprintf (pc, "%g %g m" EOL, path->x3, path->y3);
			break;
		case ART_LINETO:
			gnome_print_context_fprintf (pc, "%g %g l" EOL, path->x3, path->y3);
			break;
		case ART_CURVETO:
			gnome_print_context_fprintf (pc, "%g %g %g %g %g %g c" EOL,
						     path->x1, path->y1,
						     path->x2, path->y2,
						     path->x3, path->y3);
			break;
		default:
			g_warning ("Path structure is corrupted");
			return -1;
		}

	if (started && closed) gnome_print_context_fprintf (pc, "h" EOL);
	
	return 0;
}

static int
gnome_print_ps2_image (GnomePrintContext *pc, const char *data, int width, int height, int rowstride, int bytes_per_pixel)
{
	GnomePrintPs2 *ps2;
	const gdouble *ctm;
	gint ret, r;

	gchar *hex_data;
	gint data_size, data_size_real;

	ps2 = GNOME_PRINT_PS2 (pc);

	ctm = gp_gc_get_ctm (pc->gc);
	gnome_print_context_fprintf (pc, "q" EOL);
	gnome_print_context_fprintf (pc, "[%g %g %g %g %g %g]cm" EOL, ctm[0], ctm[1], ctm[2], ctm[3], ctm[4], ctm[5]);

	/* Image commands */
	ret = gnome_print_context_fprintf (pc, "/buf %d string def" EOL "%d %d 8" EOL, width * bytes_per_pixel, width, height);

	ret += gnome_print_context_fprintf (pc, "[%d 0 0 %d 0 %d]" EOL, width, -height, height);

	ret += gnome_print_context_fprintf (pc, "{ currentfile buf readhexstring pop }\n");

	if (bytes_per_pixel == 1) {
		ret += gnome_print_context_fprintf (pc, "image" EOL);
	} else if (bytes_per_pixel == 3) {
		ret += gnome_print_context_fprintf (pc, "false %d colorimage" EOL, bytes_per_pixel);
	}

	data_size = width * bytes_per_pixel;
	hex_data = g_new (gchar, gnome_print_encode_hex_wcs (data_size));

	for (r = 0; r < height; r++) {
		data_size_real = gnome_print_encode_hex (data + r * rowstride, hex_data, data_size);
		gnome_print_context_write_file (pc, hex_data, data_size_real);
		gnome_print_context_fprintf (pc, EOL);
	}

	g_free (hex_data);

	return 0;
}


/* Other stuff */


static gchar*
gnome_print_ps2_get_date (void)
{
	time_t clock;
	struct tm *now;
	gchar *date;

#ifdef ADD_TIMEZONE_STAMP
  extern char * tzname[];
	/* TODO : Add :
		 "[+-]"
		 "HH'" Offset from gmt in hours
		 "OO'" Offset from gmt in minutes
	   we need to use tz_time. but I don't
	   know how protable this is. Chema */
	gprint ("Timezone %s\n", tzname[0]);
	gprint ("Timezone *%s*%s*%li*\n", tzname[1], timezone);
#endif	

	debug (FALSE, "");

	clock = time (NULL);
	now = localtime (&clock);

	date = g_strdup_printf ("D:%04d%02d%02d%02d%02d%02d",
				now->tm_year + 1900,
				now->tm_mon + 1,
				now->tm_mday,
				now->tm_hour,
				now->tm_min,
				now->tm_sec);

	return date;
}

/* -------------------------- END: PUBLIC FUNCTIONS ------------------------ */
#if 0
static gint
gnome_print_ps2_error (gint fatal, const char *format, ...)
{
	va_list arguments;
	gchar *text;

	debug (FALSE, "");

	va_start (arguments, format);
	text = g_strdup_vprintf (format, arguments);
	va_end (arguments);
	
	g_warning ("Offending command [[%s]]", text);

	g_free (text);
	
	return -1;
}
#endif

#if 0
static GnomePrintPs2GraphicState *
gnome_print_ps2_graphic_state_current (GnomePrintPs2 *ps2, gint dirtyfy)
{
	GnomePrintPs2GraphicState *gs;

	debug (FALSE, "");

	g_return_val_if_fail (GNOME_IS_PRINT_PS2(ps2), NULL);

	gs = ps2->graphic_state;

	return gs;
}
	
static gint
gnome_print_ps2_dictionary  (GnomePrintContext *pc)
{
	gint ret = 0;
	
	ret += gnome_print_context_fprintf (pc,
					    "/|/def load def/,/load load" EOL
					    "|/m/moveto , |/l/lineto , |/c/curveto , |/q/gsave ," EOL
					    "|/Q/grestore , |/rg/setrgbcolor , |/J/setlinecap ," EOL
					    "|/j/setlinejoin , |/w/setlinewidth , |/M/setmiterlimit ," EOL
					    "|/d/setdash , |/i/pop , |/W/clip , |/W*/eoclip , |/n/newpath ," EOL
					    "|/S/stroke , |/f/fill , |/f*/eofill , |/Tj/show , |/Tm/moveto ," EOL
					    "|/FF/findfont ," EOL
					    "|/h/closepath , |/cm/concat , |/rm/rmoveto , |/sp/strokepath ," EOL
					    "|/SP/showpage , |/p/pop , |/EX/exch , |/DF/definefont , |" EOL
					    /* Define the procedure for reencoding fonts */
					    "/RE {dup length dict begin {1 index /FID ne {def} {p p} ifelse}forall" EOL
					    "/Encoding ISOLatin1Encoding def currentdict end} |" EOL
					    "/F {scalefont setfont} def" EOL);
	return ret;
}
																			
static void
gnome_print_ps2_graphic_state_free (GnomePrintPs2GraphicState *gs)
{
	debug (FALSE, "");

	g_free (gs);
}

static GnomePrintPs2GraphicState *
gnome_print_ps2_graphic_state_new (gint undefined)
{
	GnomePrintPs2GraphicState * state;
	gint def;

	debug (FALSE, "");

	if (undefined)
		def = 1;
	else
		def = 0;
	
	state = g_new (GnomePrintPs2GraphicState, 1);

	/* Font stuff */
	state->font_size = def;
	state->font_character_spacing = def;
	state->font_word_spacing = def;

	state->ps2_font_number = GNOME_PRINT_PS2_FONT_UNDEFINED;
	state->text_flag = FALSE;

	/* Text font stuff */
	state->text_font_handle = def;
	state->text_font_size = def;

	return state;
}

static GnomePrintPs2GraphicState *
gnome_print_ps2_graphic_state_text_set (GnomePrintContext *pc)
{
	GnomePrintPs2 *ps2;
	GnomePrintPs2GraphicState *gs;
	GnomePrintPs2GraphicState *gs_set;

	debug (FALSE, "");

	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), NULL);
	ps2 = GNOME_PRINT_PS2 (pc);
	g_return_val_if_fail (ps2 != NULL, NULL);
	
	gs     = ps2->graphic_state;
	gs_set = ps2->graphic_state_set;

	gnome_print_ps2_graphic_state_set_color (pc);
	
	return gs;
}
#endif

#if 0
static GnomePrintPs2GraphicState *
gnome_print_ps2_graphic_state_duplicate (GnomePrintPs2GraphicState *gs_in, GnomePrintPs2 *ps2)
{
	GnomePrintPs2GraphicState *gs_out;

	debug (FALSE, "");
	
	gs_out = g_new (GnomePrintPs2GraphicState, 1);
	
	memcpy (gs_out,
		gs_in,
		sizeof (GnomePrintPs2GraphicState));

	return gs_out;
}
#endif

#if 0
static gint
gnome_print_ps2_show_matrix_set (GnomePrintPs2 *ps2)
{
	GnomePrintContext *pc;
	const gdouble *ctm;
	gint ret = 0;

	pc = (GnomePrintContext *) ps2;
	ret += gnome_print_ps2_gsave (pc);

	ctm = gp_gc_get_ctm (pc->gc);

	ret += gnome_print_context_fprintf (pc, "[%g %g %g %g %g %g]cm" EOL,
					    ctm[0], ctm[1], ctm[2], ctm[3], ctm[4], ctm[5]);

	return ret;
}

static gint
gnome_print_ps2_show_matrix_restore (GnomePrintPs2 *ps2)
{
	return gnome_print_ps2_grestore ((GnomePrintContext *) ps2);
}

static gint
gnome_print_ps2_get_font_number (GnomePrintContext *pc, GnomeFont *gnome_font,
				 gboolean external)
{
	GnomePrintPs2 *ps2;
	GnomePrintPs2Font *font;
	gint n;
	gchar *font_name;

	debug (FALSE, "");

	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (GNOME_IS_FONT (gnome_font), -1);
	ps2 = GNOME_PRINT_PS2 (pc);
	g_return_val_if_fail (ps2 != NULL, -1);
	
	font_name = g_strdup (gnome_font_get_ps_name (gnome_font));

	for (n = 0; n < ps2->fonts_number; n++)
		if (!strcmp (font_name, ps2->fonts[n].font_name))
			break;

	if (n != ps2->fonts_number) {
		g_free (font_name);
		return n;
	}

	if (ps2->fonts_number == ps2->fonts_max) {
		ps2->fonts = g_realloc (ps2->fonts, sizeof (GnomePrintPs2Font) *
					(ps2->fonts_max += GNOME_PRINT_PS2_NUMBER_OF_ELEMENTS_GROW));
	}

	font = &ps2->fonts[ps2->fonts_number++];
	font->font_number   = ps2->fonts_number*2;
	gnome_font_ref (gnome_font);
	font->gnome_font    = gnome_font;
	
	font->font_name     = font_name;
	font->reencoded     = FALSE;
	font->external      = external;

	return ps2->fonts_number-1;
}

static gint
gnome_print_ps2_setfont (GnomePrintContext *pc, GnomeFont *font)
{
	GnomePrintPs2 *ps2;
	GnomePrintPs2GraphicState *gs;
	const char *fontname;
	gchar *pfbname;
	gint n;
	gboolean external = FALSE;

	debug (FALSE, "");

	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	ps2 = GNOME_PRINT_PS2 (pc);
	g_return_val_if_fail (ps2 != NULL, -1);
	g_return_val_if_fail (GNOME_IS_FONT (font), -1);

	fontname = gnome_font_get_ps_name (font);

#if 0	
	g_print ("Setting font [%s,%i]\n", fontname, GPOINTER_TO_INT (font));
#endif	

	for (n = 0; n < ps2->fonts_internal_number; n++)
		if (!strcmp (fontname, ps2->fonts_internal[n]))
			break;

	if (n == ps2->fonts_internal_number) {
		gtk_object_get (GTK_OBJECT (gnome_font_get_face (font)), "pfbname", &pfbname, NULL);
		for (n = 0; n < ps2->fonts_external_number; n++)
			if (!strcmp (pfbname, ps2->fonts_external[n]))
				break;

		if (n == ps2->fonts_external_number) {
			gchar *pfa = NULL;
			/* We need to make sure the PFA will be available */
			pfa = gnome_font_get_pfa (font);
			if (pfa == NULL) {
				g_warning ("Could not get the PFA for font %s\n", pfbname);
				g_free (pfbname);
				return -1;
			}
			g_free (pfa);
		}
		g_free (pfbname);
		external = TRUE;
	}

	gs = ps2->graphic_state;
	gs->font_size = gnome_font_get_size (font);
	gs->ps2_font_number = gnome_print_ps2_get_font_number (pc, font, external);

	/* Invalidate the current settings for gnome-text */
	gs->text_font_handle = 0;
	gs->text_font_size = 0;
	
	return 0;
}
#endif


#ifdef ENABLE_LIBGPA
static gint
gnome_print_ps2_get_level (GnomePrintPs2 *ps2)
{
	const gchar *temp;

	g_return_val_if_fail (GNOME_IS_PRINT_PS2 (ps2), 2);

	
	temp = gpa_settings_query_options (ps2->gpa_settings,
					   "PsLevel");

	if (temp == NULL)
		return 2;
	/* FIXME: for printers other than generic Postscript
		 we get the PsLevel from the values hash, not from
		 a selected option. Just keep this in mind */
	if (strcmp (temp, "PsLevel1") == 0)
		return 1;
	else if (strcmp (temp, "PsLevel2") == 0)
		return 2;
	else if (strcmp (temp, "PsLevel3") == 0)
		return 3;
	else
		g_warning ("Invalid PS Level %s", temp);
	
	return 2;
}
#endif

#if 0
static gint
gnome_print_ps2_show_sized (GnomePrintContext *pc, const char *text, int bytes)
{
	GnomePrintPs2 *ps2;
	GnomePrintPs2GraphicState *gs;
	const ArtPoint *cp;
	gint ret = 0;
	const char *p;
	const GnomeFontFace * face;

	debug (FALSE, "");

	ps2 = (GnomePrintPs2 *) pc;

	g_return_val_if_fail (GNOME_IS_PRINT_PS2 (ps2), -1);
	g_return_val_if_fail (ps2->fonts != NULL, -1);
	g_return_val_if_fail (gp_gc_has_currentpoint (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPOINT);

	gs = ps2->graphic_state;
	
	if (gs->ps2_font_number == GNOME_PRINT_PS2_FONT_UNDEFINED || gs->font_size == 0) {
		gnome_print_ps2_error (FALSE, "show, fontname or fontsize not defined.");
		return -1;
	}
	
	cp = gp_gc_get_currentpoint (pc->gc);

	ret += gnome_print_ps2_graphic_state_setfont (pc);

	g_return_val_if_fail (GNOME_IS_FONT (ps2->fonts[gs->ps2_font_number].gnome_font), -1);

	face = gnome_font_get_face (ps2->fonts[gs->ps2_font_number].gnome_font);

	g_return_val_if_fail (GNOME_IS_FONT_FACE (face), -1);

	ret += gnome_print_context_fprintf (pc, "%g %g Tm" EOL, cp->x, cp->y);

	gnome_print_ps2_graphic_state_text_set (pc);
	gnome_print_ps2_show_matrix_set (ps2);
	
	/* I don't like the fact that we are writing one letter at time */
	if (gnome_print_context_fprintf (pc, "(") < 0)
		return -1;

	for (p = text; p && p < (text + bytes); p = g_utf8_next_char (p)) {
		gunichar u;
		gint g, glyph, page;

		u = g_utf8_get_char (p);
		g = gnome_font_face_lookup_default (face, u);
		glyph = g & 0xff;
		page = (g >> 8) & 0xff;

		if (gnome_print_context_fprintf (pc,"\\%03o\\%03o", page, glyph) < 0)
			return -1;
	}

	ret += gnome_print_context_fprintf (pc, ") Tj" EOL);
	ret += gnome_print_ps2_show_matrix_restore (ps2);

	return ret;
}

static int
gnome_print_ps2_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
	/* new */
	GnomePrintPs2 *ps2;
	GnomePrintPs2GraphicState *gs;

	GnomeTextFontHandle font_handle, font_handle_last;
	gint font_size, font_size_last;

	gint current_x_scale;
	gboolean open;

	const ArtPoint *cp;
	ArtPoint point;

	gint ret = 0;
	/* new */


	int i;
	int attr_idx;
	double scale_factor;
	GnomeTextGlyphAttrEl *attrs = line->attrs;
	int x;
	int glyph;

	g_return_val_if_fail (gp_gc_has_currentpoint (pc->gc), GNOME_PRINT_ERROR_NOCURRENTPOINT);

	/* new */
	debug (FALSE, "");

	g_warning ("using textline\n");

	ps2 = GNOME_PRINT_PS2 (pc);

	gs = ps2->graphic_state;

	/* Set the initial point */
	cp = gp_gc_get_currentpoint (pc->gc);
	art_affine_point (&point, cp, gp_gc_get_ctm (pc->gc));
	/* CRASH CRASH CRASH CRASH 
	   ret += gnome_print_ps2_graphic_state_setfont (pc);
	*/
		
	ret += gnome_print_context_fprintf (pc,
					    "%g %g Tm" EOL,
					    point.x,
					    point.y);
	/* new */

	font_handle = gs->text_font_handle;
	font_handle_last = font_handle;
	font_size   = gs->text_font_size;
	font_size_last = font_size;

	current_x_scale = 1000;
	scale_factor = font_size * current_x_scale * 1e-9 * GNOME_TEXT_SCALE;

	open = 0;
	x = 0;
	attr_idx = 0;
	for (i = 0; i < line->n_glyphs; i++) {
		while (attrs[attr_idx].glyph_pos == i) {
			switch (attrs[attr_idx].attr) {
	    case GNOME_TEXT_GLYPH_FONT:
		    font_handle = attrs[attr_idx].attr_val;
		    break;
			case GNOME_TEXT_GLYPH_SIZE:
				font_size = attrs[attr_idx].attr_val;
				scale_factor = font_size * current_x_scale * 1e-9 * GNOME_TEXT_SCALE;
				break;
			default:
				break;
			}
			attr_idx++;
		}

		if (font_size != font_size_last ||
		    font_handle != font_handle_last)
		{
#ifdef VERBOSE
			g_print ("cur_size = %d, expands to %g\n",
				 cur_size, ps->current_font_size);
#endif
			if (open)
				gnome_print_context_fprintf (pc, ") Tj" EOL);
			{
				GnomeFont *font;
				font = gnome_font_face_get_font (gnome_text_get_font (font_handle), font_size * 0.001);
				gnome_print_ps2_setfont_raw (pc, font);
				gnome_font_unref (font);
			}
			open = 0;
			font_size_last = font_size;
			font_handle_last = font_handle;
		}
#ifdef VERBOSE
		g_print ("x = %d, glyph x = %d\n",
			 x, line->glyphs[i].x);
#endif
		if (abs (line->glyphs[i].x - x) > 1)
		{
			gnome_print_context_fprintf (pc, "%s%g 0 rm" EOL,
						     open ? ") Tj " : "",
						     ((line->glyphs[i].x - x) * 1.0 /
						      GNOME_TEXT_SCALE));
			open = 0;
			x = line->glyphs[i].x;
		}
		glyph = line->glyphs[i].glyph_num;
		if (!open)
			gnome_print_context_fprintf (pc, "(");
		if (glyph >= ' ' && glyph < 0x7f)
			if (glyph == '(' || glyph == ')' || glyph == '\\')
				gnome_print_context_fprintf (pc, "\\%c", glyph);
			else
				gnome_print_context_fprintf (pc, "%c", glyph);
		else
			gnome_print_context_fprintf (pc, "\\%03o", glyph);
		open = 1;
		x += floor (gnome_text_get_width (font_handle, glyph) * scale_factor + 0.5);
	}

	if (open)
		gnome_print_context_fprintf (pc, ") Tj" EOL);

	gs->text_font_handle = font_handle;
	gs->text_font_size   = font_size;

	return 0;
}
#endif


