/*
  gyrus-main-app.c

  GYRUS -- GNOME Cyrus Administrator Main Application Class.

  Copyright (C) 2003-2004 Alejandro Valdes J.
  Copyright (C) 2003-2004 Jorge Bustos B.
  Copyright (C) 2003-2004 Claudio Saavedra V.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  
*/

#include <libbonoboui.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libgnomeui/libgnomeui.h>

#include "gyrus-i18n.h"
#include "gyrus-dialog-mailbox-new.h"
#include "gyrus-dialog-find-mailbox.h"
#include "gyrus-main-app.h"
#include "gyrus-session.h"
#include "gyrus-preferences.h"
#include "gyrus-admin-mailbox.h"

static void gyrus_main_app_class_init (GyrusMainAppClass *class);
static void gyrus_main_app_init (GyrusMainApp *app);
static void gyrus_main_app_finalize (GObject *object);
static void gyrus_main_app_destroy (GtkObject *object);

static void gyrus_main_app_about (void);

static BonoboWindowClass *parent_class = NULL;

/*** Class Callbacks ***/

static void
gyrus_main_app_on_file_open_session (BonoboUIComponent *component,
				     gpointer user_data,
				     const char *cname)
{
	gyrus_session_show_sessions (NULL, user_data);
}

static void
gyrus_main_app_on_file_exit (BonoboUIComponent *component,
			     gpointer user_data,
			     const char *cname)
{
	gtk_main_quit ();
}

static void
gyrus_main_app_on_edit_find (BonoboUIComponent *component,
			     gpointer user_data,
			     const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);

	GtkWidget *dialog =
		gyrus_dialog_find_mailbox_new ();
	
	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (user_data));
	gyrus_dialog_find_mailbox_set_admin (GYRUS_DIALOG_FIND_MAILBOX (dialog),
					     gyrus_main_app_get_current_admin (app));
	
	gtk_widget_show (dialog);
}

static void
gyrus_main_app_on_edit_mailbox_add (BonoboUIComponent *component,
				    gpointer user_data,
				    const char *cname)
{
	GtkWidget *dialog =
		gyrus_dialog_mailbox_new_new ();
	gtk_widget_show (dialog);
	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (user_data));
	
}

void
gyrus_admin_delete_dialog_on_response (GtkDialog *dialog,
				       gint result,
				       gpointer user_data)
{
	GyrusAdmin * admin = GYRUS_ADMIN (user_data);

	if (result == GTK_RESPONSE_YES) {
		gchar *mailbox = gyrus_admin_get_selected_mailbox (admin);
		gyrus_admin_mailbox_delete_all (admin, mailbox);
		gyrus_admin_refresh_users_list (admin);
		gyrus_admin_mailbox_clear_info (admin);
		if (GYRUS_IS_MAIN_APP (admin->priv->parent_app))
			gyrus_main_app_menu_item_set_sensitive
				(GYRUS_MAIN_APP (admin->priv->parent_app),
				 "/commands/EditMailboxRemove",
				 FALSE);
		g_free (mailbox);
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
gyrus_main_app_on_edit_mailbox_remove (BonoboUIComponent *component,
				       gpointer user_data,
				       const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);
	GyrusAdmin * admin = gyrus_main_app_get_current_admin (app);

	gchar *mailbox = gyrus_admin_get_selected_mailbox (admin);

	GtkWidget *dialog = gtk_message_dialog_new
		(GTK_WINDOW (app), GTK_DIALOG_DESTROY_WITH_PARENT,
		 GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
		 _("Really delete mailbox '%s' and all of its submailboxes?"),
		 mailbox);
	
	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (gyrus_admin_delete_dialog_on_response),
			  admin);
	
	gtk_dialog_run (GTK_DIALOG (dialog));
	
	g_free (mailbox);
}

static void
gyrus_main_app_on_edit_preferences (BonoboUIComponent *component,
				    gpointer user_data,
				    const char *cname)
{
	gyrus_preferences_show_dialog ();
}

static void
gyrus_main_app_on_acl_entry_delete (BonoboUIComponent *component,
				    gpointer user_data,
				    const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);
	GyrusAdmin * admin = gyrus_main_app_get_current_admin (app);

	gyrus_admin_delete_selected_acl_entry (admin);
}

static void
gyrus_main_app_on_acl_entry_rename (BonoboUIComponent *component,
				    gpointer user_data,
				    const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);
	GyrusAdmin * admin = gyrus_main_app_get_current_admin (app);

	gyrus_admin_start_editing_selected_acl (admin);
}

static void
gyrus_main_app_on_acl_entry_new (BonoboUIComponent *component,
				 gpointer user_data,
				 const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);
	GyrusAdmin * admin = gyrus_main_app_get_current_admin (app);

	gyrus_admin_add_acl_entry (admin);
}

static void
gyrus_main_app_on_view_refresh (BonoboUIComponent *component,
				gpointer user_data,
				const char *cname)
{
	GyrusMainApp *app  = GYRUS_MAIN_APP (user_data);
	GyrusAdmin * admin = gyrus_main_app_get_current_admin (app);
	gyrus_admin_refresh_users_list (admin);
}

static void
gyrus_main_app_on_help_about (BonoboUIComponent *component,
				     gpointer user_data,
				     const char *cname)
{
	gyrus_main_app_about();
}

void
gyrus_main_app_make_sensitivity_consistent (GyrusMainApp *app)
{
	gint n_page;
	GyrusAdmin *admin;

	n_page = gtk_notebook_get_current_page
		(GTK_NOTEBOOK (app->priv->notebook));

	/* If there are no pages left in Notebook */
	if (n_page < 0) {
		gyrus_main_app_menu_set_sensitive (app, FALSE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryDelete", FALSE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryRename", FALSE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/EditMailboxRemove", FALSE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryNew", FALSE);
		return;
	}
	
	admin = GYRUS_ADMIN
		(gtk_notebook_get_nth_page (GTK_NOTEBOOK (app->priv->notebook),
					    n_page));
	
	if (gyrus_admin_is_connected (admin)) {
		gyrus_main_app_menu_set_sensitive (app, TRUE);
		if (gyrus_admin_acl_has_selection (admin)) {
			gyrus_main_app_menu_item_set_sensitive
				(app, "/commands/AclEntryDelete", TRUE);
			gyrus_main_app_menu_item_set_sensitive
				(app, "/commands/AclEntryRename", TRUE);
		} else {
			gyrus_main_app_menu_item_set_sensitive
				(app, "/commands/AclEntryDelete", FALSE);
			gyrus_main_app_menu_item_set_sensitive
				(app, "/commands/AclEntryRename", FALSE);
		}
	}
	else
		gyrus_main_app_menu_set_sensitive (app, FALSE);
}

static void
gyrus_main_app_on_button_close_page_clicked (GtkButton *button, gpointer data)
{
	gint page_number;
	GtkWidget    *admin   = GTK_WIDGET (data);
	gpointer     app_data = g_object_get_data (G_OBJECT (button),
						   "parent-app");
	GyrusMainApp *app     = GYRUS_MAIN_APP (app_data);
	
	page_number = gtk_notebook_page_num (GTK_NOTEBOOK
					     (app->priv->notebook), admin);
	gtk_notebook_remove_page (GTK_NOTEBOOK
				  (app->priv->notebook), page_number);

	gyrus_main_app_make_sensitivity_consistent (app);
}

void
gyrus_main_app_on_switch_page (GtkNotebook *notebook,
			       GtkNotebookPage *page,
			       guint page_num,
			       gpointer user_data)
{
	GyrusMainApp *app = GYRUS_MAIN_APP (user_data);


	GyrusAdmin *admin = GYRUS_ADMIN (gtk_notebook_get_nth_page (notebook,
								    page_num));
	
	gchar *mailbox;
	if (gyrus_admin_is_connected (admin))
		gyrus_main_app_menu_set_sensitive (app, TRUE);
	else
		gyrus_main_app_menu_set_sensitive (app, FALSE);

	if ((mailbox = gyrus_admin_get_selected_mailbox (admin)) != NULL) {
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/EditMailboxRemove", TRUE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryNew",
			 gyrus_admin_has_current_acl_access (admin));
		g_free (mailbox);
	}
	else {
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/EditMailboxRemove", FALSE);
			gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryNew", FALSE);
	}
	if (gyrus_admin_acl_has_selection (admin)) {
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryDelete", TRUE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryRename", TRUE);
	} else {
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryDelete", FALSE);
		gyrus_main_app_menu_item_set_sensitive
			(app, "/commands/AclEntryRename", FALSE);
	}
}

/*** Private Methods ***/

GType
gyrus_main_app_get_type (void)
{
	static GType type = 0;
  
	if (!type) {
		static const GTypeInfo info = {
				sizeof (GyrusMainAppClass),
				NULL,           /* base_init */
				NULL,           /* base_finalize */
				(GClassInitFunc) gyrus_main_app_class_init,
				NULL,           /* class_finalize */
				NULL,           /* class_data */
				sizeof (GyrusMainApp),
				0,              /* n_preallocs */
				(GInstanceInitFunc) gyrus_main_app_init,
				0
		};
		type = g_type_register_static (BONOBO_TYPE_WINDOW,
					       "GyrusMainApp",
					       &info, 0);
	}
	return type;
}

static void
gyrus_main_app_class_init (GyrusMainAppClass *class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (class);
	GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gobject_class->finalize = gyrus_main_app_finalize;
	object_class->destroy = gyrus_main_app_destroy;
}

static const BonoboUIVerb gyrus_main_app_verbs [] = {
	BONOBO_UI_VERB ("FileOpenSession", gyrus_main_app_on_file_open_session),
	BONOBO_UI_VERB ("FileExit",        gyrus_main_app_on_file_exit),
	BONOBO_UI_VERB ("EditMailboxAdd",  gyrus_main_app_on_edit_mailbox_add),
	BONOBO_UI_VERB ("EditMailboxRemove",
			gyrus_main_app_on_edit_mailbox_remove),
	BONOBO_UI_VERB ("EditFind",        gyrus_main_app_on_edit_find),
	BONOBO_UI_VERB ("EditPreferences", gyrus_main_app_on_edit_preferences),
	BONOBO_UI_VERB ("AclEntryNew",     gyrus_main_app_on_acl_entry_new),
	BONOBO_UI_VERB ("AclEntryRename",  gyrus_main_app_on_acl_entry_rename),
	BONOBO_UI_VERB ("AclEntryDelete",  gyrus_main_app_on_acl_entry_delete),
	BONOBO_UI_VERB ("ViewRefresh",     gyrus_main_app_on_view_refresh),
	BONOBO_UI_VERB ("HelpAbout",       gyrus_main_app_on_help_about),
	
	BONOBO_UI_VERB_END
};

static void
gyrus_main_app_init (GyrusMainApp *app)
{
	BonoboUIContainer *ui_container;
	BonoboUIComponent *ui_component;
	GtkWidget *main_vbox;
	GtkWidget *appbar;

	ui_container = bonobo_window_get_ui_container (BONOBO_WINDOW (app));
	bonobo_ui_engine_config_set_path (bonobo_window_get_ui_engine
					  (BONOBO_WINDOW (app)),
					  "/apps/gyrus/UIConfig/kvps");
	ui_component = bonobo_ui_component_new_default ();
	bonobo_ui_component_set_name (ui_component, "gyrus-menubar");
	bonobo_ui_component_set_container (ui_component,
					   BONOBO_OBJREF (ui_container), NULL);
	
	bonobo_ui_util_set_ui (ui_component, "",
			       GYRUS_SCHEMAS_DIR "/gyrus.xml",
			       "gyrus", NULL);

	bonobo_ui_component_add_verb_list_with_data (ui_component,
						     gyrus_main_app_verbs,
						     app);

	app->priv = g_new0 (GyrusMainAppPrivate, 1);
	gtk_window_set_title (GTK_WINDOW (app),
			      _("Cyrus IMAP Administrator"));
	bonobo_window_set_name (BONOBO_WINDOW (app), "gyrus");
	main_vbox = gtk_vbox_new (FALSE, 0);
	app->priv->notebook = gtk_notebook_new ();

	g_signal_connect (G_OBJECT (app->priv->notebook), "switch-page",
			  G_CALLBACK (gyrus_main_app_on_switch_page), app);
	
	gtk_box_pack_start (GTK_BOX (main_vbox), app->priv->notebook,
			    TRUE, TRUE, 0);
	appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_USER);

	gtk_box_pack_start (GTK_BOX (main_vbox), appbar,
			    FALSE, FALSE, 0);
	bonobo_window_set_contents (BONOBO_WINDOW (app), main_vbox);

	gtk_widget_show_all (main_vbox);
}

static void
gyrus_main_app_finalize (GObject *object)
{
	GyrusMainApp *app;

	g_return_if_fail (GYRUS_IS_MAIN_APP(object));

	app = GYRUS_MAIN_APP (object);

	g_free (app->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gyrus_main_app_destroy (GtkObject *object)
{
	GyrusMainApp *app;
	g_return_if_fail (GYRUS_IS_MAIN_APP (object));
	app = GYRUS_MAIN_APP (object);

	GTK_OBJECT_CLASS (parent_class)->destroy (object);

}

GyrusAdmin *
gyrus_main_app_get_current_admin (GyrusMainApp *app)
{
	GyrusAdmin *admin;
	gint n;
	n = gtk_notebook_get_current_page (GTK_NOTEBOOK (app->priv->notebook));
	admin = GYRUS_ADMIN (gtk_notebook_get_nth_page
			     (GTK_NOTEBOOK (app->priv->notebook),
			      n));

	return admin;
}

static void
gyrus_main_app_about (void)
{
	static GtkWidget *about_box = NULL;
	GdkPixbuf *pixbuf;
	gchar *logo_gyrus;

	const gchar *authors[] = {
		"Alejandro Vald" "\xC3\xA9" "s Jim" "\xC3\xA9"
		"nez <alejandro@gnome.cl>",
		"Jorge Bustos Bustos <jorge@gnome.cl>",
		"Claudio Saavedra "
		"<csaavedra@alumnos.utalca.cl>",
		NULL
	};

	const gchar *documentors[] = { NULL };

	const gchar *translators = _("translators");

	if (about_box != NULL){
		gtk_window_present(GTK_WINDOW(about_box));
		return;
	}

	logo_gyrus = g_strdup_printf ("%s%c%s", GYRUS_PIXMAPS_DIR,
	                              G_DIR_SEPARATOR, "logo_gyrus.png");
	                              
	pixbuf = gdk_pixbuf_new_from_file(logo_gyrus, NULL);
	
	g_free (logo_gyrus);

	about_box = gnome_about_new (_("GNOME Cyrus Administrator"),
				     VERSION,
				     _("(c) 2003 GNOME Foundation"),
				     _("Administration tool for Cyrus IMAP servers."),
				     authors, documentors,
				     strcmp (translators, "translators") != 0 ? translators : NULL,
				     pixbuf);

	g_signal_connect (G_OBJECT(about_box), "destroy",
			  G_CALLBACK(gtk_widget_destroyed), &about_box);
	
	gtk_widget_show(about_box);

	g_object_unref(pixbuf);
}

GtkWidget *
gyrus_main_app_create_label_from_admin (GyrusMainApp *app, GyrusAdmin *admin)
{
	GtkWidget *hbox;
	GtkWidget *image;
	GtkWidget *button;
	GtkWidget *label;

	const gchar *session_name;
	
	/* create a text label */
	session_name = gyrus_admin_get_current_session_name (admin);
	label = gtk_label_new (session_name ? session_name : "");
	
	/* create a button with a close icon */
	image = gtk_image_new_from_stock (GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	button = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_widget_set_size_request (button, 20, 20);
	gtk_container_add (GTK_CONTAINER (button), image);
	g_object_set_data (G_OBJECT (button), "parent-app", app);

	/* create the container for both widgets*/
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);

	gtk_widget_show_all (hbox);

	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (gyrus_main_app_on_button_close_page_clicked),
			  admin);

	return hbox;
}

/*** Public Methods ****/
GtkWidget *
gyrus_main_app_new (void)
{
	GyrusMainApp *app;
	app = g_object_new (GYRUS_TYPE_MAIN_APP, NULL);
	return GTK_WIDGET (app);
}

GtkWidget *
gyrus_main_app_append_page (GyrusMainApp *app, GyrusSession *session)
{
	g_return_val_if_fail (GYRUS_IS_MAIN_APP (app), NULL);
	GtkWidget *hbox_label;
	gint page_num;
	GtkWidget *admin = gyrus_admin_new (session);
	gyrus_admin_set_parent_app (GYRUS_ADMIN (admin),
				    (gpointer) app);
	
	gtk_notebook_append_page (GTK_NOTEBOOK (app->priv->notebook),
				  admin, NULL);
	
	hbox_label = gyrus_main_app_create_label_from_admin
		(app, GYRUS_ADMIN (admin));
	gtk_notebook_set_tab_label (GTK_NOTEBOOK (app->priv->notebook),
				    admin, hbox_label);

	page_num = gtk_notebook_page_num (GTK_NOTEBOOK (app->priv->notebook),
					  admin);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (app->priv->notebook),
				       page_num);
	
/*	gyrus_main_app_menu_set_sensitive (app, TRUE); */
	return admin;
}

void
gyrus_main_app_menu_item_set_sensitive (GyrusMainApp *app,
					const gchar *command,
					gboolean sensitive)
{
	CORBA_char *state;
	BonoboUIEngine *ui_engine;
	gboolean invalid;
	
	ui_engine = bonobo_window_get_ui_engine (BONOBO_WINDOW (app));
	
	state = (CORBA_char *)
		bonobo_ui_engine_xml_get_prop (ui_engine, command,
					       "sensitive", &invalid);
	state[0] = (sensitive)? '1' : '0';

	bonobo_ui_engine_xml_set_prop (ui_engine, command, "sensitive",
				       state, "gyrus-menubar");

	CORBA_free (state);
}

void
gyrus_main_app_menu_set_sensitive (GyrusMainApp *app, gboolean sens)
{
	g_return_if_fail (GYRUS_IS_MAIN_APP (app));

	gyrus_main_app_menu_item_set_sensitive
		(app, "/commands/EditFind", sens);
	gyrus_main_app_menu_item_set_sensitive
		(app, "/commands/EditMailboxAdd", sens);
	gyrus_main_app_menu_item_set_sensitive
		(app, "/commands/ViewRefresh", sens);
}

/***** Above functions doesnt belong this file ******/

void
gyrus_application_window_destroyed_callback (GtkWindow *window,
					      gpointer userdata)
{
	gtk_main_quit();
}

int
main (int argc, char *argv[])
{
	GtkWidget *app;
	gtk_init (&argc, &argv);
	bindtextdomain (GETTEXT_PACKAGE, GNOME_LOCALE_DIR);
	textdomain (GETTEXT_PACKAGE);

	gnome_program_init ("gyrus", VERSION, LIBGNOMEUI_MODULE,
			    argc, argv,
			    GNOME_PARAM_APP_DATADIR, GYRUS_DATA_DIR, NULL);

	app = gyrus_main_app_new ();
	gtk_window_set_default_size (GTK_WINDOW (app), 600, 400);
	g_signal_connect (GTK_OBJECT (app), "delete-event",
			  G_CALLBACK
			  (gyrus_application_window_destroyed_callback),
			  NULL);

	gtk_widget_show (GTK_WIDGET (app));
	gtk_main ();

	return 0;
}
