/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */


/**
 *@file
 *The definition of the #MlViewNodeEditor widget.
 *
 *This widget edits a given xml node.
 *Some of it methods take an xmlNodePtr 
 *in parameter and allow the user to
 *interactively edit the xml node. 
 *It emits signal whenever the xml 
 *node being edited changes.
 *The user of this widget can thus 
 *catch and handle the emitted signals.
 *An xml node can have several types
 *(element node, text node, processing instruction node, comment node).
 *Each type of node is edited using what i 
 *call a view. As they are several types of nodes, they are several
 *types of view; one for each type of node. 
 *A particular view is a widget that can 
 *edit a particular type of xml node
 *Each view is abstracted by a structure 
 *that holds the view widget.
 *The #MlViewElementWidget contains all the
 *possible views it can handle. 
 *Whenever an xml node is to be edited, the proper view
 *is then selected and used to edit the node.
 *The selection of the right view is done in 
 *the method mlview_node_editor_edit_xml_node(). 
 *
 */
#define GTK_ENABLE_BROKEN
#include "mlview-node-editor.h"
#include "mlview-xml-document.h"
#include "mlview-attributes-list.h"
#include "mlview-namespace-editor.h"
#include "mlview-utils.h"

#include <libxml/tree.h>
#include <ctype.h>

typedef struct _XMLElementNodeView XMLElementNodeView;
typedef struct _XMLCommentNodeView XMLCommentNodeView;
typedef struct _XMLCDataSectionNodeView XMLCDataSectionNodeView;
typedef struct _XMLTextNodeView XMLTextNodeView;
typedef struct _XMLPINodeView XMLPINodeView;
typedef struct _XMLDocNodeView XMLDocNodeView;

/*===========================================================
 *Some structure used in the private data part of MlViewElementEditor.
 *===========================================================*/

/**
 *This structure represents the element node view 
 *
 */
struct _XMLElementNodeView {
        GtkVBox *vbox;

        /*a GtkEntry: contains the current element name */
        GtkEntry *name;

        /*a GtkCList: contains the attibutes list */
        MlViewAttributesList *attributes;

        /*The namespaces editor */
        MlViewNamespaceEditor *namespaces;

        guint name_changed_handler_id;
};


/**
 *The text node view 
 *
 */
struct _XMLTextNodeView {
        GtkVBox *vbox;
        GtkText *content;
};

/**
 *The comment node view. 
 *
 */
struct _XMLCommentNodeView {
        GtkVBox *vbox;
        GtkText *content;
};

/**
 *The cdata section node view. 
 *
 */
struct _XMLCDataSectionNodeView {
        GtkVBox *vbox;
        GtkText *content;
};

/**
 *The processing instruction node. 
 *
 */
struct _XMLPINodeView {
        GtkVBox *vbox;
        GtkEntry *name;
        GtkText *content;
        guint name_changed_handler_id;
};

struct _XMLDocNodeView {
        GtkVBox *vbox;

        /*the name of this document ... not mandatory */
        GtkEntry *name;

        guint name_changed_handler_id;

        /*wether this document is mandatory or not */
        GtkEntry *standalone;

        /*the xml version */
        GtkEntry *xml_version;

        /*the encoding of the on disk file */
        /*GtkEntry *external_encoding ; */
        GtkCombo *external_encoding;

        /*the external id of the external subset */
        GtkEntry *ext_subset_external_id;

        /*the system id of the external subset */
        GtkEntry *ext_subset_system_id;
};


/*===========================================================
 *The private part of the MlViewNodeEditor structure.
 *===========================================================*/
struct _MlViewNodeEditorPrivate {
        /*the left part of the MlViewNodeEditor widget */
        GtkVBox *left_margin;

        /*the combo that defines the type of element being edited */
        GtkEntry *xml_node_type;

        /*Right part of element editor. 
         *This widget shows a view appropriated
         *to the type of xml element
         *being edited. Let's say the xml node 
         *being edited is a text node, the view (widget) used to edit it
         *will be a GtkText. If the xml node 
         *being edited is an element node, the view used to edit it
         *will be more complex: it will be a widget 
         *that enable the user to edit the xml element name and the
         *it attributes ...
         */
        GtkNotebook *node_view;

        /*the current xml node being edited */
        xmlNode *curr_xml_node;

        MlViewXMLDocument *curr_xml_document;
        gboolean iseditable;

        /*the view used to edit an xml element node */
        XMLElementNodeView *element_node_view;

        /*the view used to edit an xml text node */
        XMLTextNodeView *text_node_view;

        /*the view used to edit an xml comment node */
        XMLCommentNodeView *comment_node_view;

        /*the view used to edit an xml cdata section node */
        XMLCDataSectionNodeView *cdata_section_node_view;

        /*the view used to edit an xml pi node */
        XMLPINodeView *pi_node_view;

        /*the view used to edit an xml doc node */
        XMLDocNodeView *doc_node_view;

        MlViewAppContext *app_context;
        guint xml_node_type_changed_handler_id;
        guint left_right_percentage;
};

/*
 *This enum defines the signals emited by MlViewNodeEditor.
 *It is used as the offset of the signal in 
 *the private p_table mlview_node_editor_signals.
 */
enum {
        ELEMENT_CHANGED,
        EDIT_STATE_CHANGED,
        ELEMENT_NAME_CHANGED,
        ELEMENT_ATTRIBUTE_CHANGED,
        ELEMENT_CONTENT_CHANGED,
        NUMBER_OF_SIGNALS
};

/*
 *This enum defines the notebook pages that hold
 *the possibles views of an xml node. It is used
 *as the offset of matching notebook pages.
 *Make sure that the order of the construction of 
 *the views (of the notebook pages)
 *(at the end of mlview_node_editor_init()) 
 *matches the order defined by this enum.
 */
enum {
        ELEMENT_NODE_VIEW_PAGE,
        TEXT_NODE_VIEW_PAGE,
        COMMENT_NODE_VIEW_PAGE,
        CDATA_SECTION_VIEW_PAGE,
        PI_NODE_VIEW_PAGE,
        DOC_NODE_VIEW_PAGE
};

/*the macro used to access the private part of the MlViewNodeEditor*/
#define PRIVATE(node_editor) (node_editor->private)

#define ELEMENT_NODE_VIEW(node_editor) (PRIVATE(node_editor)->element_node_view) /*acess the element node view ... */
#define TEXT_NODE_VIEW(node_editor)  (PRIVATE(node_editor)->text_node_view)
#define COMMENT_NODE_VIEW(node_editor) (PRIVATE(node_editor)->comment_node_view)
#define CDATA_SECTION_NODE_VIEW(node_editor) (PRIVATE(node_editor)->cdata_section_node_view)
#define PI_NODE_VIEW(node_editor) (PRIVATE(node_editor)->pi_node_view)
#define DOC_NODE_VIEW(node_editor) (PRIVATE(node_editor)->doc_node_view)
#define CURRENT_XML_NODE(node_editor) (PRIVATE(node_editor)->curr_xml_node)
#define IS_EDITABLE(node_editor) (PRIVATE(node_editor)->iseditable)
#define DEFAULT_LEFT_RIGHT_PERCENTAGE 20

/*private functions declarations*/
static void
 mlview_node_editor_xml_text_node_view_set_editable
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_xml_cdata_section_node_view_set_editable
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_xml_comment_node_view_set_editable
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_xml_pi_node_view_set_editable
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_build_xml_element_node_view
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_build_xml_text_node_view (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_build_xml_comment_node_view (MlViewNodeEditor
                                                 *
                                                 a_node_editor);

static void
 mlview_node_editor_build_xml_cdata_section_node_view
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_build_xml_pi_node_view (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_build_xml_doc_node_view (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_clear_xml_element_node_view
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_clear_xml_text_node_view (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_clear_xml_comment_node_view
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_clear_xml_cdata_section_node_view
        (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_clear_xml_pi_node_view (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_xml_element_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void
 mlview_node_editor_xml_text_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void
 mlview_node_editor_xml_comment_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void
 mlview_node_editor_xml_cdata_section_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void
 mlview_node_editor_xml_pi_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void
 mlview_node_editor_xml_doc_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node);


static void
 mlview_node_editor_class_init (MlViewNodeEditorClass * a_klass);

static void
 mlview_node_editor_init (MlViewNodeEditor * a_node_editor);

static void
 mlview_node_editor_destroy (GtkObject * a_object);

static void
 mlview_node_editor_construct (MlViewNodeEditor * a_node_editor,
                               MlViewAppContext * a_app_context);

static void
 mlview_node_editor_editable_state_changed_cb (GtkWidget * a_widget,
                                               MlViewNodeEditor *
                                               a_editor);

static void
 mlview_node_editor_name_changed_cb (GtkWidget * a_entry,
                                     MlViewNodeEditor *
                                     a_editor);

static void
 mlview_node_editor_attribute_changed_cb (MlViewAttributesList * a_attributes,
                                          gpointer a_editor);

static void
 mlview_node_editor_content_changed_cb (GtkText * a_content,
                                        MlViewNodeEditor *
                                        a_editor);

static void
 mlview_node_editor_configure_event_cb (GtkWidget * a_node_editor,
                                        GdkEventConfigure *
                                        a_event,
                                        gpointer a_data);

static void
 external_encoding_changed_cb (GtkEditable * a_encoding,
                               MlViewNodeEditor * a_editor);
/*private static data*/
static GtkHPanedClass *gv_parent_class;
static guint gv_mlview_node_editor_signals[NUMBER_OF_SIGNALS] =
        { 0 };



/*================================================================
 *private helper functions
 *===============================================================*/


/**
 *
 */
static void
mlview_node_editor_build_xml_element_node_view (MlViewNodeEditor
                                                * a_node_editor)
{
        XMLElementNodeView *view;
        MlViewNodeEditorPrivate *private_data;
        GtkWidget *table,
        *label = NULL;
        GtkWidget *frame = NULL,
                *vbox = NULL;

        const guchar *attributes_names_title =
                N_("Attribute names");
        const guchar *attributes_values_title =
                N_("Attribute values");

        g_return_if_fail (a_node_editor != NULL);


        if (PRIVATE (a_node_editor) == NULL) {

                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));
        }

        if (ELEMENT_NODE_VIEW (a_node_editor) == NULL) {

                ELEMENT_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof (XMLElementNodeView));

        } else {

                if (ELEMENT_NODE_VIEW (a_node_editor)->vbox !=
                    NULL) {

                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (TEXT_NODE_VIEW
                                  (a_node_editor)->vbox));
                }
        }

        view = ELEMENT_NODE_VIEW (a_node_editor);

        private_data = PRIVATE (a_node_editor);
        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));
        frame = gtk_frame_new (_("Element node"));
        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);
        vbox = gtk_vbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (frame), vbox);

        label = gtk_label_new (_("Element name"));

        view->name = GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_text (GTK_ENTRY (view->name), "");

        gtk_entry_set_editable (GTK_ENTRY (view->name),
                                private_data->iseditable);

        table = gtk_table_new (1, 2, TRUE);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, FALSE, FALSE, 0);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   label, 0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->name),
                                   1, 2, 0, 1);

        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "changed",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_node_editor);

        /*init the attributes and the namespace edition areas */
        table = gtk_table_new (1, 2, TRUE);

        frame = gtk_frame_new (_("attributes edition"));

        view->attributes =
                MLVIEW_ATTRIBUTES_LIST
                (mlview_attributes_list_new
                 ((guchar *) attributes_names_title,
                  (guchar *) attributes_values_title,
                  PRIVATE (a_node_editor)->app_context));

        g_signal_connect (G_OBJECT (view->attributes),
                          "attribute-changed",
                          G_CALLBACK
                          (mlview_node_editor_attribute_changed_cb),
                          a_node_editor);

        gtk_container_add (GTK_CONTAINER (frame),
                           GTK_WIDGET (view->attributes));

        gtk_table_attach_defaults (GTK_TABLE (table), frame,
                                   0, 1, 0, 1);

        view->namespaces =
                MLVIEW_NAMESPACE_EDITOR
                (mlview_namespace_editor_new
                 (PRIVATE (a_node_editor)->app_context));

        frame = gtk_frame_new (_("namespaces edition"));

        gtk_container_add (GTK_CONTAINER (frame),
                           GTK_WIDGET (view->namespaces));

        gtk_table_attach_defaults (GTK_TABLE (table), frame,
                                   1, 2, 0, 1);

        /*FIXME: add signals on the namespaces editor */
        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *Creates the text node view. This view will be used to edit text elements. 
 *
 */
static void
 mlview_node_editor_build_xml_text_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLTextNodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;

        g_return_if_fail (a_node_editor != NULL);



        if (PRIVATE (a_node_editor) == NULL)
                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_node_editor);


        if (TEXT_NODE_VIEW (a_node_editor) == NULL)
                TEXT_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof (XMLElementNodeView));
        else if (TEXT_NODE_VIEW (a_node_editor)->vbox != NULL) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (TEXT_NODE_VIEW (a_node_editor)->vbox));
        }

        view = TEXT_NODE_VIEW (a_node_editor);
        frame = gtk_frame_new (_("Text Node"));

        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));
        gtk_box_pack_start
                (GTK_BOX (view->vbox), frame, TRUE, TRUE, 0);

        gtk_widget_show (frame);

        view->content = GTK_TEXT (gtk_text_new (NULL, NULL));

        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER
                           (scrolled_window),
                           GTK_WIDGET (view->content));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);

        g_signal_connect (G_OBJECT (view->content),
                          "changed",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_node_editor);

        gtk_widget_show (GTK_WIDGET (view->content));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *creates the xml comment node view. 
 *This view will be used by 
 *MlViewElement editor to
 *edit xml comment node. 
 *
 */
static void
 mlview_node_editor_build_xml_comment_node_view
        (MlViewNodeEditor * a_node_editor) {
        MlViewNodeEditorPrivate *private_data = NULL;
        XMLCommentNodeView *view = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;

        g_return_if_fail (a_node_editor != NULL);


        if (PRIVATE (a_node_editor) == NULL)
                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_node_editor);

        if (COMMENT_NODE_VIEW (a_node_editor) == NULL) {
                COMMENT_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof (XMLCommentNodeView));
        } else {
                if (COMMENT_NODE_VIEW (a_node_editor)->vbox !=
                    NULL) {
                        gtk_widget_destroy (GTK_WIDGET
                                            (COMMENT_NODE_VIEW
                                             (a_node_editor)->
                                             vbox));
                }
        }

        view = COMMENT_NODE_VIEW (a_node_editor);
        view->vbox = GTK_VBOX (gtk_vbox_new (TRUE, 0));
        view->content = GTK_TEXT (gtk_text_new (NULL, NULL));
        frame = gtk_frame_new (_("Comment node"));

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            GTK_WIDGET (frame), TRUE, TRUE, 0);

        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (view->content));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);

        g_signal_connect (G_OBJECT (view->content),
                          "changed",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_node_editor);

        gtk_widget_show_all (GTK_WIDGET (view->vbox));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Creates the xml cdata section view. 
 *This view will be used by 
 *MlViewNodeEditor to edit cdata section node.
 *
 */
static void
 mlview_node_editor_build_xml_cdata_section_node_view
        (MlViewNodeEditor * a_node_editor) {
        MlViewNodeEditorPrivate *private_data = NULL;
        XMLCDataSectionNodeView *view = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;

        g_return_if_fail (a_node_editor != NULL);

        if (PRIVATE (a_node_editor) == NULL)
                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_node_editor);
        if (CDATA_SECTION_NODE_VIEW (a_node_editor) == NULL) {
                CDATA_SECTION_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof
                                   (XMLCDataSectionNodeView));
        } else {
                if (CDATA_SECTION_NODE_VIEW (a_node_editor)->
                    vbox != NULL) {
                        gtk_widget_destroy (GTK_WIDGET
                                            (CDATA_SECTION_NODE_VIEW
                                             (a_node_editor)->
                                             vbox));
                }
        }

        view = CDATA_SECTION_NODE_VIEW (a_node_editor);
        view->vbox = GTK_VBOX (gtk_vbox_new (TRUE, 0));
        view->content = GTK_TEXT (gtk_text_new (NULL, NULL));
        frame = gtk_frame_new (_("CDATA Section node"));

        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);
        scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (view->content));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);

        g_signal_connect (G_OBJECT (view->content),
                          "changed",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_node_editor);
        gtk_widget_show_all (GTK_WIDGET (view->vbox));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *builds the xml processing instruction 
 *node view included in the current instance of a_node_editor. 
 *
 */
static void
 mlview_node_editor_build_xml_pi_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLPINodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL,
                *table = NULL,
                *label = NULL;

        g_return_if_fail (a_node_editor != NULL);

        if (PRIVATE (a_node_editor) == NULL)
                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));
        private_data = PRIVATE (a_node_editor);

        if (PI_NODE_VIEW (a_node_editor) == NULL) {
                PI_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof (XMLPINodeView));
        } else {
                if (PI_NODE_VIEW (a_node_editor)->vbox != NULL) {
                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (PI_NODE_VIEW (a_node_editor)->
                                  vbox));
                }
        }

        view = PI_NODE_VIEW (a_node_editor);
        frame = gtk_frame_new (_("PI Node"));
        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));

        label = gtk_label_new (_("PI node name"));
        view->name = GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_editable (view->name,
                                private_data->iseditable);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE
                                   (table),
                                   GTK_WIDGET (view->name), 1, 2,
                                   0, 1);

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            table, TRUE, TRUE, 0);

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            frame, TRUE, TRUE, 0);

        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "changed",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_node_editor);

        view->content = GTK_TEXT (gtk_text_new (NULL, NULL));

        gtk_text_set_editable (view->content,
                               private_data->iseditable);

        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add
                (GTK_CONTAINER (scrolled_window),
                 GTK_WIDGET (view->content));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);

        g_signal_connect (G_OBJECT (view->content),
                          "changed",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_node_editor);

        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Creates the view that can edit an xml doc node. 
 *
 */
static void
 mlview_node_editor_build_xml_doc_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLDocNodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *table = NULL,
                *label = NULL,
                *vbox = NULL;
        GList *available_encodings = NULL;

        g_return_if_fail (a_node_editor != NULL);

        if (PRIVATE (a_node_editor) == NULL) {
                PRIVATE (a_node_editor) = g_malloc0
                        (sizeof (MlViewNodeEditorPrivate));
        }

        private_data = PRIVATE (a_node_editor);

        if (DOC_NODE_VIEW (a_node_editor) == NULL) {
                DOC_NODE_VIEW (a_node_editor) =
                        g_malloc0 (sizeof (XMLDocNodeView));
        } else {
                if (DOC_NODE_VIEW (a_node_editor)->vbox != NULL) {
                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (DOC_NODE_VIEW (a_node_editor)->
                                  vbox));
                }
        }

        view = DOC_NODE_VIEW (a_node_editor);

        /*Set the decoration ... */
        frame = gtk_frame_new (_("document node"));

        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));

        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);

        vbox = gtk_vbox_new (FALSE, 0);

        gtk_container_add (GTK_CONTAINER (frame), vbox);

        /*The "name of the document" entry */
        view->name = GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->name,
                                private_data->iseditable);

        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "changed",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_node_editor);

        label = gtk_label_new (_("document name or uri:"));

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->name),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*the "standalone" combo ... */
        label = gtk_label_new (_("is document standalone:"));
        view->standalone = GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_editable (view->standalone, FALSE);

        table = gtk_table_new (1, 2, TRUE);
        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE
                                   (table),
                                   GTK_WIDGET (view->standalone),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox), table,
                            TRUE, TRUE, 0);

        /*The "xml version" entry ... */
        label = gtk_label_new (_("xml version:"));

        view->xml_version = GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->xml_version, FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               xml_version), 1,
                                   2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox), table,
                            TRUE, TRUE, 0);

        /*The "xml encoding" entry ... */
        label = gtk_label_new (_("xml file encoding:"));

        view->external_encoding = GTK_COMBO (gtk_combo_new ());
        g_return_if_fail (view->external_encoding);

        gtk_entry_set_editable (GTK_ENTRY
                                (view->external_encoding->entry),
                                FALSE);

        available_encodings =
                mlview_utils_get_available_encodings ();

        g_return_if_fail (available_encodings);

        gtk_combo_set_popdown_strings (view->external_encoding,
                                       available_encodings);
        g_signal_connect (G_OBJECT
                          (view->external_encoding->entry),
                          "changed",
                          G_CALLBACK
                          (external_encoding_changed_cb),
                          a_node_editor);

        table = gtk_table_new (1, 2, TRUE);

        g_return_if_fail (table);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               external_encoding),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*The external subset external is */
        label = gtk_label_new (_
                               ("external id of the external subset:"));

        view->ext_subset_external_id =
                GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->ext_subset_external_id,
                                FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               ext_subset_external_id),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*The external subset external id entry ... */
        label = gtk_label_new (_
                               ("system id of the external subset:"));

        view->ext_subset_system_id =
                GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->ext_subset_system_id,
                                FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               ext_subset_system_id),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*Show everything and attach it to the views */
        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Visually clears the xml element node view. 
 *
 */
static void
 mlview_node_editor_clear_xml_element_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLElementNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        view = ELEMENT_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        if (view->name_changed_handler_id != 0)
                g_signal_handler_block (G_OBJECT (view->name),
                                        view->
                                        name_changed_handler_id);

        gtk_entry_set_text (view->name, "");
        mlview_attributes_list_clear (view->attributes);

        if (view->name_changed_handler_id != 0)
                g_signal_handler_unblock (G_OBJECT (view->name),
                                          view->
                                          name_changed_handler_id);
}


/**
 *Clears visually (only) the text node view of the MlViewNodeEditor widget. 
 *
 */
static void
 mlview_node_editor_clear_xml_text_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLTextNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);

        view = TEXT_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);
/*
 *FIXME: remember to block the 
 *"text-changed" signal (if used) 
 *before clearing the text node view.
 */
        gtk_text_set_point (view->content, 0);
        gtk_text_forward_delete (view->content,
                                 gtk_text_get_length (view->
                                                      content));

}


/**
 *Clears visually (only) the comment node view of MlViewNodeEditor. 
 *
 */
static void
mlview_node_editor_clear_xml_comment_node_view (MlViewNodeEditor
                                                * a_node_editor)
{
        XMLCommentNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);

        view = COMMENT_NODE_VIEW (a_node_editor);

        g_return_if_fail (view != NULL);

        gtk_text_set_point (view->content, 0);
        gtk_text_forward_delete (view->content,
                                 gtk_text_get_length (view->
                                                      content));

}


/**
 *
 */
static void
 mlview_node_editor_clear_xml_cdata_section_node_view
        (MlViewNodeEditor * a_node_editor) {
        XMLCDataSectionNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        g_return_if_fail (PRIVATE (a_node_editor) != NULL);

        view = CDATA_SECTION_NODE_VIEW (a_node_editor);

        g_return_if_fail (view != NULL);

        gtk_text_set_point (view->content, 0);
        gtk_text_forward_delete (view->content,
                                 gtk_text_get_length (view->
                                                      content));
}

/**
 *Visually clears the current pi node view 
 *included in the current instance of MlViewNodeEditor. 
 *
 */
static void
mlview_node_editor_clear_xml_pi_node_view (MlViewNodeEditor *
                                           a_node_editor)
{
        XMLPINodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        g_return_if_fail (PRIVATE (a_node_editor) != NULL);

        view = PI_NODE_VIEW (a_node_editor);

        g_return_if_fail (view != NULL);

        /*clear the pi node content editor */
        gtk_text_set_point (view->content, 0);
        gtk_text_forward_delete (view->content,
                                 gtk_text_get_length (view->
                                                      content));

        /*clear the name */
        gtk_editable_delete_text (GTK_EDITABLE (view->name), 0,
                                  -1);
}


/**
 *
 */
static void
 mlview_node_editor_xml_element_node_view_set_editable
        (MlViewNodeEditor * a_node_editor) {
        XMLElementNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);

        view = ELEMENT_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        gtk_entry_set_editable (view->name,
                                PRIVATE (a_node_editor)->
                                iseditable);

        mlview_attributes_list_set_editable
                (view->attributes,
                 PRIVATE (a_node_editor)->iseditable);
}

/**
 *
 */
static void
 mlview_node_editor_xml_text_node_view_set_editable
        (MlViewNodeEditor * a_node_editor) {
        XMLTextNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        view = TEXT_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        gtk_text_set_editable
                (view->content,
                 PRIVATE (a_node_editor)->iseditable);
}

/**
 *
 */
static void
 mlview_node_editor_xml_cdata_section_node_view_set_editable
        (MlViewNodeEditor * a_node_editor) {
        XMLCDataSectionNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        view = CDATA_SECTION_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        gtk_text_set_editable (view->content,
                               PRIVATE (a_node_editor)->
                               iseditable);
}

/**
 *
 */
static void
 mlview_node_editor_xml_comment_node_view_set_editable
        (MlViewNodeEditor * a_node_editor) {
        XMLCommentNodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        view = COMMENT_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        gtk_text_set_editable (view->content,
                               PRIVATE (a_node_editor)->
                               iseditable);
}


/**
 *
 */
static void
 mlview_node_editor_xml_pi_node_view_set_editable
        (MlViewNodeEditor * a_node_editor) {
        XMLPINodeView *view = NULL;

        g_return_if_fail (a_node_editor != NULL);
        view = PI_NODE_VIEW (a_node_editor);
        g_return_if_fail (view != NULL);

        gtk_text_set_editable (view->content,
                               PRIVATE (a_node_editor)->
                               iseditable);
}


/**
 *Edits the xml node 
 *(which is supposed to be an xml element node) 
 *using the xml element view contained in 
 *the current instance of MlViewNodeEditor.
 *Note that if the xml node is not 
 *an element node, this function does nothing.
 *@param a_editor the "this" pointer of the current
 *instance of #MlViewNodeEditor.
 *@a_xml_doc the instance of #MlViewXMLDocument currently
 *being edited.
 *@a_node the node of a_xml_doc to edit.
 */
static void
 mlview_node_editor_xml_element_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLElementNodeView *editor_view = NULL;
        guchar *full_name = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;

        editor_private->curr_xml_document = a_xml_doc;

        if (a_node->type != XML_ELEMENT_NODE)
                return;

        editor_view = ELEMENT_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);

        status = mlview_xml_document_node_get_fqn (a_node,
                                                   ISO8859_1,
                                                   &full_name);
        g_return_if_fail (status == MLVIEW_OK);

        /*set the name of the element */

        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->
                                name_changed_handler_id);

        gtk_entry_set_text (editor_view->name, "");

        gtk_entry_set_text (GTK_ENTRY (editor_view->name),
                            full_name);

        if (full_name != NULL) {
                g_free (full_name);
                full_name = NULL;
        }

        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->
                                  name_changed_handler_id);

        /*edit the attributes */
        mlview_attributes_list_clear (editor_view->attributes);
        mlview_attributes_list_edit_xml_attributes (editor_view->
                                                    attributes,
                                                    a_node);

        /*edit the namespaces */
        mlview_namespace_editor_clear (editor_view->namespaces);
        mlview_namespace_editor_edit_node_visible_namespaces
                (editor_view->namespaces, a_node);
        gtk_notebook_set_page (editor_private->node_view,
                               ELEMENT_NODE_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("Element node"));
}


/**
 *
 */
static void
 mlview_node_editor_xml_text_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLTextNodeView *editor_view = NULL;
        guchar *isolat1_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        if (a_node->type != XML_TEXT_NODE)
                return;

        editor_view = TEXT_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);

        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        status = mlview_xml_document_node_get_content (a_node,
                                                       ISO8859_1,
                                                       &isolat1_content);
        g_return_if_fail (status == MLVIEW_OK);

        gtk_text_set_point (editor_view->content, 0);

        gtk_text_forward_delete (editor_view->content,
                                 gtk_text_get_length
                                 (editor_view->content));

        if (isolat1_content) {

                len = strlen (isolat1_content);
                gtk_text_insert (editor_view->content,
                                 NULL, NULL, NULL,
                                 isolat1_content, len);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               TEXT_NODE_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("Text node"));

/* release_resources:*/

        if (isolat1_content) {

                g_free (isolat1_content);
                isolat1_content = NULL;

        }
}

/**
 *static void mlview_node_editor_xml_comment_node_view_edit_xml_node:
 *Edits an xml comment node using a specific xml comment node view. If the xml node given in argument
 *is not an xml comment node, does nothing. 
 *
 */
static void
 mlview_node_editor_xml_comment_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLCommentNodeView *editor_view = NULL;
        guchar *isolat1_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        if (a_node->type != XML_COMMENT_NODE)
                return;

        editor_view = COMMENT_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*content = xmlNodeGetContent (a_node) ; */

        status = mlview_xml_document_node_get_content (a_node,
                                                       ISO8859_1,
                                                       &isolat1_content);

        g_return_if_fail (status == MLVIEW_OK);

        gtk_text_set_point (editor_view->content, 0);

        gtk_text_forward_delete (editor_view->content,
                                 gtk_text_get_length
                                 (editor_view->content));

        if (isolat1_content) {
                len = strlen (isolat1_content);
                gtk_text_insert (editor_view->content, NULL,
                                 NULL, NULL, isolat1_content,
                                 len);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               COMMENT_NODE_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("Comment node"));

/* release_resources:*/

        if (isolat1_content) {
                g_free (isolat1_content);
                isolat1_content = NULL;
        }
}

/**
 *Edits a xml cdata section node using the a specific xml cdata section view.
 *If the xml node given in argument is node a cdata section node, does nothing. 
 *
 */
static void
 mlview_node_editor_xml_cdata_section_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLCDataSectionNodeView *editor_view = NULL;
        guchar *isolat1_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_editor);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        editor_view = CDATA_SECTION_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view);

        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*content = xmlNodeGetContent (a_node) ; */
        status = mlview_xml_document_node_get_content (a_node,
                                                       ISO8859_1,
                                                       &isolat1_content);
        g_return_if_fail (status == MLVIEW_OK);

        gtk_text_set_point (editor_view->content, 0);

        gtk_text_forward_delete (editor_view->content,
                                 gtk_text_get_length
                                 (editor_view->content));

        if (isolat1_content) {

                len = strlen (isolat1_content);
                gtk_text_insert (editor_view->content, NULL,
                                 NULL, NULL, isolat1_content,
                                 len);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               CDATA_SECTION_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("CDATA Section node"));

/* release_resources:*/

        if (isolat1_content) {

                g_free (isolat1_content);
                isolat1_content = NULL;
        }
}


/**
 *Edits the xml pi node given in argument 
 *using a specific xml pi node view. If he xml node
 *given in argument is not a pi node, does nothing. 
 *
 */
static void
 mlview_node_editor_xml_pi_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLPINodeView *editor_view = NULL;
        guchar *content = NULL,
                *full_name = NULL;

        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        editor_view = PI_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        status = mlview_xml_document_node_get_name (a_node,
                                                    ISO8859_1,
                                                    &full_name);
        g_return_if_fail (status == MLVIEW_OK);

        /*get the name of the element. */
        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->
                                name_changed_handler_id);

        gtk_entry_set_text (editor_view->name, "");
        gtk_entry_set_text (GTK_ENTRY (editor_view->name),
                            full_name);

        if (full_name)
                g_free (full_name);

        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->
                                  name_changed_handler_id);

        /*edit the content */
        /*content = xmlNodeGetContent (a_node) ; */

        status = mlview_xml_document_node_get_content (a_node,
                                                       ISO8859_1,
                                                       &content);
        g_return_if_fail (status == MLVIEW_OK);

        gtk_text_set_point (editor_view->content, 0);
        gtk_text_forward_delete (editor_view->content,
                                 gtk_text_get_length
                                 (editor_view->content));

        if (content) {
                len = strlen (content);
                gtk_text_insert (editor_view->content, NULL,
                                 NULL, NULL, content, len);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               PI_NODE_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("PI node"));

        if (content) {
                g_free (content);
                content = NULL;
        }
}


/**
 *Edits the xml document given in argument using a specific view. If the xml node given in argument is not
 *a xml document node, does nothing. 
 *
 */
static void
 mlview_node_editor_xml_doc_node_view_edit_xml_node
        (MlViewNodeEditor * a_editor,
         MlViewXMLDocument * a_xml_doc, xmlNode * a_node) {
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLDocNodeView *editor_view = NULL;
        guchar *name = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        if (a_node->type != XML_DOCUMENT_NODE)
                return;

        editor_view = DOC_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*get the name of the element. */
        status = mlview_xml_document_node_get_name (a_node,
                                                    ISO8859_1,
                                                    &name);
        g_return_if_fail (status == MLVIEW_OK);

        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->
                                name_changed_handler_id);
        gtk_entry_set_text (editor_view->name, "");

        if (name != NULL)
                gtk_entry_set_text (GTK_ENTRY
                                    (editor_view->name), name);
        else
                gtk_entry_set_text (GTK_ENTRY
                                    (editor_view->name), "");

        if (name) {
                g_free (name);
                name = NULL;
        }

        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->
                                  name_changed_handler_id);

        /*get the standalone */
        if (((xmlDocPtr) a_node)->extSubset) {

                gtk_entry_set_text (editor_view->standalone,
                                    _("NO"));
                ((xmlDocPtr) a_node)->standalone = FALSE;

        } else {

                gtk_entry_set_text (editor_view->standalone,
                                    _("YES"));
                ((xmlDocPtr) a_node)->standalone = TRUE;

        }

        /*get the xml version */
        if (((xmlDocPtr) a_node)->version)
                gtk_entry_set_text (editor_view->xml_version,
                                    ((xmlDocPtr) a_node)->
                                    version);
        else
                gtk_entry_set_text (editor_view->xml_version,
                                    "1.0");

        /*get the external encoding */


        if (((xmlDocPtr) a_node)->encoding) {
                enum MlViewStatus status = MLVIEW_OK;

                /*
                 *If the encoding is supported
                 *by mlview and if it is not
                 *in the default list of supported encodings
                 *add it to that list.
                 */
                status = mlview_utils_add_supported_encoding
                        ((guchar *) ((xmlDocPtr) a_node)->
                         encoding);

                if (status ==
                    MLVIEW_ENCODING_NOT_SUPPORTED_ERROR) {

                        /*
                         *If the document has no encoding specified,
                         *set the default document encoding to UTF-8.
                         */
                        ((xmlDocPtr) a_node)->encoding =
                                xmlMemStrdup ("UTF-8");

                }

        } else {

                ((xmlDocPtr) a_node)->encoding =
                        xmlMemStrdup ("UTF-8");
        }

        g_return_if_fail (editor_view->external_encoding
                          && editor_view->external_encoding->
                          entry);

        gtk_signal_handler_block_by_data
                (GTK_OBJECT
                 (editor_view->external_encoding->entry),
                 a_editor);

        gtk_entry_set_text (GTK_ENTRY
                            (editor_view->external_encoding->
                             entry),
                            ((xmlDocPtr) a_node)->encoding);

        gtk_signal_handler_unblock_by_data
                (GTK_OBJECT
                 (editor_view->external_encoding->entry),
                 a_editor);

        /*get stuffs about the external subset */
        if (((xmlDocPtr) a_node)->extSubset
            && ((xmlDocPtr) a_node)->extSubset->ExternalID) {
                gtk_entry_set_text (editor_view->
                                    ext_subset_external_id,
                                    ((xmlDocPtr) a_node)->
                                    extSubset->ExternalID);
        }

        if (((xmlDocPtr) a_node)->extSubset
            && ((xmlDocPtr) a_node)->extSubset->SystemID) {
                gtk_entry_set_text (editor_view->
                                    ext_subset_system_id,
                                    ((xmlDocPtr) a_node)->
                                    extSubset->SystemID);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               DOC_NODE_VIEW_PAGE);

        gtk_entry_set_text (editor_private->xml_node_type,
                            _("xml document node"));
}




/***********************************************
 *private signal handlers
 *FIXME: recode everything below to 
 *support the private field in MlViewNodeEditor 
 *and the different views ...
 ************************************************/

/**
 *handler called when the edit state combo is activated. 
 *
 */
static void
mlview_node_editor_editable_state_changed_cb (GtkWidget *
                                              a_widget,
                                              MlViewNodeEditor *
                                              a_editor)
{
        g_return_if_fail (a_widget != NULL);
        g_return_if_fail (a_editor != NULL);

        mlview_node_editor_set_editable (a_editor,
                                         !(PRIVATE (a_editor)->
                                           iseditable));

}


/**
 *Called back when the element name entry changes.
 *Applies the change to the attached xml node and emits the
 *signal "element_name_changed" followed by the signal
 *"element_changed" if
 *and only if the element is an xml element node. 
 *
 */
static void
mlview_node_editor_name_changed_cb (GtkWidget * a_entry,
                                    MlViewNodeEditor * a_editor)
{


        g_return_if_fail (a_entry != NULL);
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->curr_xml_node !=
                          NULL);

        if (PRIVATE (a_editor)->curr_xml_node->type ==
            XML_ELEMENT_NODE
            || PRIVATE (a_editor)->curr_xml_node->type ==
            XML_PI_NODE) {
                guchar *full_name = NULL,
                        *local_name = NULL;
                xmlNs *ns = NULL;

                full_name = (guchar *)
                        gtk_entry_get_text (GTK_ENTRY (a_entry));


                if (full_name) {
                        mlview_utils_parse_full_name
                                (PRIVATE (a_editor)->
                                 curr_xml_node, full_name, &ns,
                                 &local_name);
                }

                if (ns != NULL) {
                        xmlSetNs (PRIVATE (a_editor)->
                                  curr_xml_node, ns);
                } else {
                        PRIVATE (a_editor)->curr_xml_node->ns =
                                NULL;
                }

                mlview_xml_document_set_node_name
                        (PRIVATE (a_editor)->curr_xml_document,
                         PRIVATE (a_editor)->curr_xml_node,
                         local_name, ISO8859_1, TRUE);

                gtk_signal_emit (GTK_OBJECT (a_editor),
                                 gv_mlview_node_editor_signals
                                 [ELEMENT_CHANGED]);

                if (local_name) {
                        g_free (local_name);
                        local_name = NULL;
                }

        }
}


/**
 *
 */
static void
mlview_node_editor_attribute_changed_cb (MlViewAttributesList *
                                         a_attributes,
                                         gpointer a_node_editor)
{
        MlViewNodeEditor *node_editor = NULL;

        g_return_if_fail (a_attributes);
        g_return_if_fail (a_node_editor);

        node_editor = MLVIEW_NODE_EDITOR (a_node_editor);
        gtk_signal_emit (GTK_OBJECT (node_editor),
                         gv_mlview_node_editor_signals
                         [ELEMENT_CHANGED]);
}

/**
 *A callback called when the user changes the content of the element.
 *Emits a signal "element_content_changed" of the MlViewNodeEditor class. 
 *
 */
static void
mlview_node_editor_content_changed_cb (GtkText * a_content,
                                       MlViewNodeEditor *
                                       a_editor)
{
        guchar *content = NULL;

        g_return_if_fail (a_content != NULL);
        g_return_if_fail (GTK_IS_TEXT (a_content));
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (PRIVATE (a_editor)->curr_xml_node !=
                          NULL);
        g_return_if_fail (PRIVATE (a_editor)->
                          curr_xml_document != NULL);

        content =
                gtk_editable_get_chars (GTK_EDITABLE (a_content),
                                        0, -1);

        mlview_xml_document_set_node_content
                (PRIVATE (a_editor)->curr_xml_document,
                 PRIVATE (a_editor)->curr_xml_node,
                 content, ISO8859_1, TRUE);

        g_signal_emit (G_OBJECT (a_editor),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CONTENT_CHANGED], 0, content);

        g_signal_emit (G_OBJECT (a_editor),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CHANGED], 0, content);

        g_free (content);
}


static void
mlview_node_editor_configure_event_cb (GtkWidget * a_node_editor,
                                       GdkEventConfigure *
                                       a_event, gpointer a_data)
{
        MlViewNodeEditor *node_editor;

        g_return_if_fail (a_node_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_node_editor)
                          != TRUE);
        g_return_if_fail (a_event->type != GDK_CONFIGURE);

        node_editor = MLVIEW_NODE_EDITOR (a_node_editor);

        mlview_node_editor_set_left_right_percentage
                (node_editor,
                 PRIVATE (node_editor)->left_right_percentage);
}


static void
external_encoding_changed_cb (GtkEditable * a_encoding,
                              MlViewNodeEditor * a_editor)
{
        guchar *encoding = NULL;

        g_return_if_fail (a_encoding
                          && GTK_IS_EDITABLE (a_encoding));
        g_return_if_fail (a_editor
                          && MLVIEW_IS_NODE_EDITOR (a_editor)
                          && PRIVATE (a_editor));

        if (PRIVATE (a_editor)->curr_xml_node == NULL
            || PRIVATE (a_editor)->curr_xml_node->doc == NULL)
                return;

        encoding = gtk_editable_get_chars (a_encoding, 0, -1);

        if (encoding == NULL)
                return;

        if (mlview_utils_is_encoding_supported (encoding) ==
            TRUE) {

                if (PRIVATE (a_editor)->curr_xml_node->doc->
                    encoding) {
                        xmlFree ((xmlChar *) PRIVATE (a_editor)->
                                 curr_xml_node->doc->encoding);
                }

                PRIVATE (a_editor)->curr_xml_node->doc->
                        encoding = xmlMemStrdup (encoding);
        } else {
                /*
                 *Libxml2 does not support the new encoding so ...
                 *why not setting the encoding to UTF-8 as default ?
                 */
        }

        if (encoding) {
                g_free (encoding);
                encoding = NULL;
        }
}

/*=========================================
 *private gtk object framework methods
 *=========================================*/

/**
 *Classical dynamic type allocator.
 */
guint
mlview_node_editor_get_type (void)
{

        static guint mlview_node_editor_type = 0;

        if (!mlview_node_editor_type) {
                static const GTypeInfo
                        mlview_node_editor_type_info = {
                        sizeof (MlViewNodeEditorClass),
                        NULL,   /* base_init */
                        NULL,   /* base_finalize */
                        (GClassInitFunc)
                                mlview_node_editor_class_init,
                        NULL,   /* class_finalize */
                        NULL,   /* class_data */
                        sizeof (MlViewNodeEditor),
                        0,
                        (GInstanceInitFunc)
                        mlview_node_editor_init
                };

                mlview_node_editor_type =
                        g_type_register_static (GTK_TYPE_HPANED,
                                                "MlViewNodeEditor",
                                                &mlview_node_editor_type_info,
                                                0);
        }
        return mlview_node_editor_type;
}

/**
 *classical class struct initialyzer. This code overrides the
 *destroy method of the GtkObjectClass with mlview_node_editor_destroy. 
 *It defines
 *also the signal which offsets are 
 *strored in the static gv_mlview_node_editor_signals. 
 *
 */
static void
mlview_node_editor_class_init (MlViewNodeEditorClass * a_klass)
{
        GtkObjectClass *object_class =
                (GtkObjectClass *) a_klass;
        gv_parent_class =
                gtk_type_class (gtk_hpaned_get_type ());

        /*overload the destroy method of object */
        object_class->destroy = mlview_node_editor_destroy;

        /*signal definitions */
        gv_mlview_node_editor_signals[ELEMENT_CHANGED] =
                g_signal_new ("element-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_NAME_CHANGED] =
                g_signal_new ("element-name-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_name_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_ATTRIBUTE_CHANGED]
                =
                g_signal_new ("element-attribute-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_attribute_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_CONTENT_CHANGED] =
                g_signal_new ("element-content-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_content_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[EDIT_STATE_CHANGED] =
                g_signal_new ("edit-state-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               edit_state_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        /* gtk_object_class_add_signals (object_class, 
           gv_mlview_node_editor_signals, 
           NUMBER_OF_SIGNALS) ; */

        /*no default signal handler for signal "document-changed" */
        a_klass->element_changed = NULL;
        a_klass->element_name_changed = NULL;
        a_klass->element_attribute_changed = NULL;
        a_klass->element_content_changed = NULL;
        a_klass->edit_state_changed = NULL;
}

static void
mlview_node_editor_construct (MlViewNodeEditor * a_node_editor,
                              MlViewAppContext * a_app_context)
{
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *label = NULL,
                *table = NULL;
        GtkWidget *combo = NULL;
        GList *list2 = NULL;

        g_return_if_fail (a_node_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_node_editor));
        g_return_if_fail (PRIVATE (a_node_editor) != NULL);

        private_data = PRIVATE (a_node_editor);
        private_data->curr_xml_node = NULL;
        private_data->iseditable = TRUE;
        private_data->app_context = a_app_context;

        private_data->left_margin =
                GTK_VBOX (gtk_vbox_new (FALSE, 0));

        private_data->node_view =
                GTK_NOTEBOOK (gtk_notebook_new ());

        gtk_notebook_set_show_tabs (private_data->node_view,
                                    FALSE);

        gtk_notebook_popup_disable (private_data->node_view);

        gtk_paned_add1 (GTK_PANED (a_node_editor),
                        GTK_WIDGET (private_data->left_margin));

        gtk_paned_add2 (GTK_PANED (a_node_editor),
                        GTK_WIDGET (private_data->node_view));

        /*build left margin ... */
        label = gtk_label_new (_("Node type"));
        private_data->xml_node_type =
                GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (GTK_ENTRY
                                (private_data->xml_node_type),
                                FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_box_pack_start (GTK_BOX (private_data->left_margin),
                            table, FALSE, FALSE, 0);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   label, 0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (private_data->
                                               xml_node_type), 1,
                                   2, 0, 1);

        gtk_widget_show_all (table);

        label = gtk_label_new (_("Edit State:"));

        list2 = g_list_append (list2, (gpointer)_("Editable"));

        list2 = g_list_append (list2, (gpointer) _("Read Only"));

        combo = gtk_combo_new ();

        gtk_combo_set_popdown_strings (GTK_COMBO (combo), list2);

        g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry),
                          "changed",
                          G_CALLBACK
                          (mlview_node_editor_editable_state_changed_cb),
                          (gpointer) a_node_editor);

        table = gtk_table_new (1, 2, TRUE);

        gtk_box_pack_start (GTK_BOX (private_data->left_margin),
                            table, FALSE, FALSE, 0);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   label, 0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   combo, 1, 2, 0, 1);

        gtk_widget_show_all (table);

        /*handle size modif of the window */
        g_signal_connect (G_OBJECT (a_node_editor),
                          "configure_event",
                          GTK_SIGNAL_FUNC
                          (mlview_node_editor_configure_event_cb),
                          NULL);

        /*
         *now that the mlview app context context has been 
         *passed to this object, can build
         *sub components that need that context.
         */
        mlview_node_editor_build_xml_element_node_view
                (a_node_editor);
        mlview_node_editor_build_xml_text_node_view
                (a_node_editor);
        mlview_node_editor_build_xml_comment_node_view
                (a_node_editor);
        mlview_node_editor_build_xml_cdata_section_node_view
                (a_node_editor);
        mlview_node_editor_build_xml_pi_node_view
                (a_node_editor);
        mlview_node_editor_build_xml_doc_node_view
                (a_node_editor);
        mlview_node_editor_set_editable (a_node_editor, TRUE);
}


/**
 *Classical instance initialyser.Creates the widget. 
 *
 */
static void
mlview_node_editor_init (MlViewNodeEditor * a_node_editor)
{
        if (PRIVATE (a_node_editor) == NULL)
                PRIVATE (a_node_editor) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

}

/*============================================
 *public exported methods
 *===========================================
 */

/**
 *Simple instance builder. 
 *
 */
GtkWidget *
mlview_node_editor_new (MlViewAppContext * a_app_context)
{
        MlViewNodeEditor *editor = NULL;

        editor = gtk_type_new (MLVIEW_TYPE_NODE_EDITOR);
        mlview_node_editor_construct (editor, a_app_context);

        return GTK_WIDGET (editor);
}


/**
 *mlview_node_editor_set_application_context 
 *
 */
void
mlview_node_editor_set_application_context (MlViewNodeEditor *
                                            a_node_editor,
                                            MlViewAppContext *
                                            a_app_context)
{
        g_return_if_fail (a_node_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_node_editor));
        g_return_if_fail (PRIVATE (a_node_editor) != NULL);

        PRIVATE (a_node_editor)->app_context = a_app_context;
}


MlViewAppContext *
mlview_node_editor_get_application_context (MlViewNodeEditor *
                                            a_node_editor)
{
        g_return_val_if_fail (a_node_editor != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_NODE_EDITOR
                              (a_node_editor), NULL);
        g_return_val_if_fail (PRIVATE (a_node_editor) != NULL,
                              NULL);

        return PRIVATE (a_node_editor)->app_context;
}

/**
 *xmlNodePtr mlview_node_editor_get_current_xml_node: 
 *
 *@return a pointer on the xmlNode being edited. 
 */
xmlNodePtr
mlview_node_editor_get_current_xml_node (MlViewNodeEditor *
                                         a_editor)
{
        g_return_val_if_fail (a_editor != NULL, NULL);
        return PRIVATE (a_editor)->curr_xml_node;
}


/**
 *Edits the node current node. 
 *
 */
void
mlview_node_editor_edit_xml_node (MlViewNodeEditor * a_editor,
                                  MlViewXMLDocument * a_xml_doc,
                                  xmlNode * a_node)
{
        /*some sanity checks */
        g_return_if_fail (a_editor != NULL
                          && MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));

        PRIVATE (a_editor)->curr_xml_node = a_node;
        g_return_if_fail (PRIVATE (a_editor)->curr_xml_node !=
                          NULL);

        switch (a_node->type) {
        case XML_ELEMENT_NODE:
                /*
                 *edit the element node using the 
                 *XMLElementNodeView of the element editor
                 */
                mlview_node_editor_xml_element_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_TEXT_NODE:
                /*
                 *show the text content using the 
                 *XMLTextNodeView of the element editor
                 */
                mlview_node_editor_xml_text_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_CDATA_SECTION_NODE:
                /*
                 *show the cdata using the 
                 *XMLCDataSectionNodeView of the element editor
                 */
                mlview_node_editor_xml_cdata_section_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_COMMENT_NODE:
                mlview_node_editor_xml_comment_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_DOCUMENT_NODE:
                mlview_node_editor_xml_doc_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_PI_NODE:
                mlview_node_editor_xml_pi_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        default:
                break;
        }
        /*FIXME: remove the provious children of right side of a_editor */
        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_editor)->node_view));

}                               /*end mlview_node_editor_edit_xml_node */


/**
 *Visualy (only) Clears the element editor. 
 *Does not free anything. 
 */
void
mlview_node_editor_clear (MlViewNodeEditor * a_editor)
{
        g_return_if_fail (a_editor != NULL
                          && MLVIEW_IS_NODE_EDITOR (a_editor));

        /*clear all the views */
        mlview_node_editor_clear_xml_element_node_view
                (a_editor);
        mlview_node_editor_clear_xml_text_node_view (a_editor);
        mlview_node_editor_clear_xml_comment_node_view
                (a_editor);
        mlview_node_editor_clear_xml_cdata_section_node_view
                (a_editor);
        mlview_node_editor_clear_xml_pi_node_view (a_editor);

}

/**
 *Sets the new editable state of the 
 *current instance of MlViewNodeEditor.
 *Actually, it the current view of the element 
 *editor that is set to the editable/non editable state.
 *Also emits the signal "edit state changes". 
 */
void
mlview_node_editor_set_editable (MlViewNodeEditor * a_editor,
                                 gboolean a_editable)
{
        MlViewNodeEditorPrivate *private_data = NULL;

        g_return_if_fail (a_editor != NULL
                          && MLVIEW_IS_NODE_EDITOR (a_editor));
        private_data = PRIVATE (a_editor);
        g_return_if_fail (private_data);

        private_data->iseditable = a_editable;

        mlview_node_editor_xml_element_node_view_set_editable
                (a_editor);
        mlview_node_editor_xml_text_node_view_set_editable
                (a_editor);
        mlview_node_editor_xml_cdata_section_node_view_set_editable
                (a_editor);
        mlview_node_editor_xml_comment_node_view_set_editable
                (a_editor);
        mlview_node_editor_xml_pi_node_view_set_editable
                (a_editor);

        gtk_signal_emit (GTK_OBJECT (a_editor),
                         gv_mlview_node_editor_signals
                         [EDIT_STATE_CHANGED]);
}


/**
 *
 */
void
mlview_node_editor_set_left_right_percentage (MlViewNodeEditor *
                                              a_node_editor,
                                              const guint
                                              a_percentage)
{
        guint separator_position;
        GtkWidget *top_level_widget;

        g_return_if_fail (a_node_editor != NULL
                          &&
                          MLVIEW_IS_NODE_EDITOR (a_node_editor));
        g_return_if_fail (PRIVATE (a_node_editor) != NULL);
        top_level_widget =
                gtk_widget_get_toplevel (GTK_WIDGET
                                         (a_node_editor));
        g_return_if_fail (top_level_widget != NULL);

        PRIVATE (a_node_editor)->left_right_percentage =
                a_percentage;
        separator_position =
                top_level_widget->allocation.width *
                a_percentage / 100;

        gtk_paned_set_position (GTK_PANED (a_node_editor),
                                separator_position);

        gtk_widget_show_all (GTK_WIDGET (a_node_editor));
}


/**
 *
 */
void
mlview_node_editor_destroy (GtkObject * a_object)
{
        MlViewNodeEditor *editor = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;

        g_return_if_fail (a_object != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_object));

        if (PRIVATE (MLVIEW_NODE_EDITOR (a_object)) == NULL)
                return;

        editor = MLVIEW_NODE_EDITOR (a_object);
        private_data = PRIVATE (editor);

        if (private_data->element_node_view != NULL) {
                g_free (private_data->element_node_view);
                private_data->element_node_view = NULL;

        }

        if (private_data->text_node_view != NULL) {
                g_free (private_data->text_node_view);
                private_data->text_node_view = NULL;

        }

        if (private_data->comment_node_view != NULL) {
                g_free (private_data->comment_node_view);
                private_data->comment_node_view = NULL;
        }

        if (private_data->cdata_section_node_view != NULL) {
                g_free (private_data->cdata_section_node_view);
                private_data->cdata_section_node_view = NULL;
        }

        if (private_data->pi_node_view != NULL) {
                g_free (private_data->pi_node_view);
                private_data->pi_node_view = NULL;
        }

        if (private_data->doc_node_view != NULL) {
                g_free (private_data->doc_node_view);
                private_data->doc_node_view = NULL;
        }


        if (PRIVATE (editor) != NULL) {
                /*
                 *this if clause should be the last one before 
                 * the one that follows
                 */
                g_free (PRIVATE (editor));
                PRIVATE (editor) = NULL;
        }

        /*
         *do not delete curr_xml_node 
         *cause it is to be freed by the MlViewTreeEditor
         */
        if (*GTK_OBJECT_CLASS (gv_parent_class)->destroy)
                (*GTK_OBJECT_CLASS (gv_parent_class)->
                 destroy) (a_object);
}
