/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#include "mlview-xml-document-view.h"

struct _MlViewXMLDocumentViewPrivate {
        /*The xml document the view is connected to */
        MlViewXMLDocument *mlview_xml_document;

        /*The name of this view */
        gchar *view_name;

        /*the interactif name edition widget */
        GtkDialog *name_edition_dialog;
};


/*signals*/
enum {
        NAME_CHANGED,
        /*must be the last one! */
        NUMBER_OF_SIGNALS
};

static GtkVBoxClass *parent_class = NULL;
static gint
        gv_mlview_xml_document_view_signals[NUMBER_OF_SIGNALS] =
        { 0 };

/*This table lists the callbacks of the xml document signals*/
static SignalCallbackDescriptor *gv_signals_fired_by_document =
        NULL;



static void
 mlview_xml_document_view_class_init (MlViewXMLDocumentViewClass * a_klass);

static void
 mlview_xml_document_view_init (MlViewXMLDocumentView * a_mlview_xml_doc);

static enum MLVIEW_XML_DOCUMENT_VIEW_STATUS
 mlview_xml_document_view_connect_to_document
        (MlViewXMLDocumentView * a_tree_view,
         MlViewXMLDocument * a_xml_doc);

static enum MLVIEW_XML_DOCUMENT_VIEW_STATUS
 mlview_xml_document_view_disconnect_from_document
        (MlViewXMLDocumentView * a_doc_view,
         MlViewXMLDocument * a_xml_doc);


static GtkWidget *build_name_edition_dialog (MlViewAppContext *
                                             a_app_context);

static void
 set_name_edition_widget_value (GtkWidget * a_edition_widget,
                                gchar * a_value);
static const guchar *get_name_edition_widget_value (GtkWidget *
                                                    a_edition_widget);


static const char *NAME_EDITION_ENTRY_FIELD_KEY =
        "name-edition-entry-field";

#define PRIVATE(object) ((object)->priv)


/****************************************************
 *Private (static) methods of this class.
 ****************************************************/

/*
 *mlview_xml_document_view_class_init:
 *@a_klass: the current class
 *
 *Private initializer of the class. Part of the Gtk typing system framework.
 */
static void
mlview_xml_document_view_class_init (MlViewXMLDocumentViewClass *
                                     a_klass)
{
        GtkObjectClass *object_class =
                (GtkObjectClass *) a_klass;

        g_return_if_fail (a_klass != NULL);
        parent_class = gtk_type_class (GTK_TYPE_VBOX);
        object_class->destroy = mlview_xml_document_view_destroy;

        /*now, create the signals */
        gv_mlview_xml_document_view_signals[NAME_CHANGED] =
                g_signal_new ("name-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewXMLDocumentViewClass,
                               name_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        /* gtk_object_class_add_signals (object_class, 
           gv_mlview_xml_document_view_signals, 
           NUMBER_OF_SIGNALS) ; */
        a_klass->name_changed = NULL;
}


/*
 *mlview_xml_document_view_init:
 *@a_xml_doc_view: the current instance of MlViewXMLDocumentView.
 *
 *The initialyzer of the MlViewXMLDocumentView object. 
 *Part of the gtk typing system
 *framework.
 */
static void
mlview_xml_document_view_init (MlViewXMLDocumentView *
                               a_xml_doc_view)
{
        g_return_if_fail (a_xml_doc_view != NULL);
        g_return_if_fail (MLVIEW_XML_DOCUMENT_VIEW
                          (a_xml_doc_view));

        g_return_if_fail (PRIVATE (a_xml_doc_view) == NULL);

        PRIVATE (a_xml_doc_view) =
                g_malloc0 (sizeof
                           (MlViewXMLDocumentViewPrivate));
}

/**
 *Connects the current view to the signals 
 *emitted by the document.
 *
 *After successfull completion of this function, 
 *each signal emitted by the document to notify
 *a change of its state will trigger a visual change
 *in the current view.
 *
 *The signals fired by the document are declared in the
 *global structure gv_signals_fired_by_document .
 *
 *Note that this function references (increases the ref count) of
 *the document a_xml_doc (calls mlview_xml_document_ref() ), so please, 
 *make sure the destructor of the current instance of #MlViewXMLDocumentView
 *calls mlview_xml_document_unref().
 *
 *@param a_doc_view the "this pointer".
 *@param a_xml_doc the document which signals are to be connected to.
 *@return the status.
 */
static enum MLVIEW_XML_DOCUMENT_VIEW_STATUS
 mlview_xml_document_view_connect_to_document
        (MlViewXMLDocumentView * a_doc_view,
         MlViewXMLDocument * a_xml_doc) {
        int i = 0;

        g_return_val_if_fail (a_doc_view != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_doc_view),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (a_xml_doc != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail
                (gv_signals_fired_by_document != NULL,
                 MLVIEW_XML_DOCUMENT_VIEW_DOC_SIGNAL_HANDLERS_NOT_SET);


        for (i = 0; gv_signals_fired_by_document[i].signal_name;
             i++) {

                if (!gv_signals_fired_by_document[i].
                    signal_handler)
                        continue;

                g_signal_connect
                        (G_OBJECT (a_xml_doc),
                         gv_signals_fired_by_document[i].
                         signal_name,
                         G_CALLBACK (gv_signals_fired_by_document
                                     [i].signal_handler),
                         a_doc_view);
        }

        mlview_xml_document_ref (a_xml_doc);

        return MLVIEW_XML_DOCUMENT_VIEW_OK;
}

static enum MLVIEW_XML_DOCUMENT_VIEW_STATUS
 mlview_xml_document_view_disconnect_from_document
        (MlViewXMLDocumentView * a_doc_view,
         MlViewXMLDocument * a_xml_doc) {
        int i = 0;

        g_return_val_if_fail (a_doc_view != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_doc_view),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);
        g_return_val_if_fail (a_xml_doc != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);
        g_return_val_if_fail (gv_signals_fired_by_document,
                              MLVIEW_XML_DOCUMENT_VIEW_DOC_SIGNAL_HANDLERS_NOT_SET);

        for (i = 0; gv_signals_fired_by_document[i].signal_name;
             i++) {

                if (!gv_signals_fired_by_document[i].
                    signal_handler)
                        continue;

                g_signal_handlers_disconnect_by_func
                        (G_OBJECT (a_xml_doc),
                         G_CALLBACK
                         (gv_signals_fired_by_document[i].
                          signal_handler), a_doc_view);
        }

        return MLVIEW_XML_DOCUMENT_VIEW_OK;
}


/**
 *Helper function. Builds the view name entry widget.
 *This widget embedds a GtkEntry widget wich actually holds the view name.
 *to get/set the that GtkEntry widget, you just have to invoke 
 *the gtk_object_get_data () or
 *gtk_object_set_data () using the key NAME_EDITION_ENTRY_FIELD_KEY. 
 *
 *@return the newly built name edition widget. 
 */
static GtkWidget *
build_name_edition_dialog (MlViewAppContext * a_app_context)
{
        GtkDialog *result = NULL;
        GtkEntry *name_entry_field = NULL;

        g_return_val_if_fail (a_app_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_app_context), NULL);

        result = GTK_DIALOG
                (gtk_dialog_new_with_buttons
                 (_("Type the name of the current view"), NULL,
                  GTK_DIALOG_MODAL, _("OK"), GTK_RESPONSE_ACCEPT,
                  _("Cancel"), GTK_RESPONSE_REJECT, NULL));

        name_entry_field = GTK_ENTRY (gtk_entry_new ());

        gtk_widget_show_all (GTK_WIDGET (name_entry_field));

        gtk_object_set_data (GTK_OBJECT (result),
                             NAME_EDITION_ENTRY_FIELD_KEY,
                             name_entry_field);

        gtk_box_pack_start_defaults (GTK_BOX (result->vbox),
                                     GTK_WIDGET
                                     (name_entry_field));

        gtk_widget_realize (GTK_WIDGET (result));
        mlview_app_context_set_window_icon (a_app_context,
                                            GTK_WINDOW (result));

        gtk_widget_show_all (GTK_WIDGET (result->vbox));

        return GTK_WIDGET (result);
}


/**
 *
 */
static void
set_name_edition_widget_value (GtkWidget * a_edition_widget,
                               gchar * a_value)
{
        GtkEntry *name_entry = NULL;

        g_return_if_fail (a_edition_widget != NULL);
        g_return_if_fail (GTK_IS_DIALOG (a_edition_widget));

        name_entry =
                gtk_object_get_data (GTK_OBJECT
                                     (a_edition_widget),
                                     NAME_EDITION_ENTRY_FIELD_KEY);

        g_return_if_fail (name_entry != NULL);
        g_return_if_fail (GTK_IS_ENTRY (name_entry));
        if (a_value)
                gtk_entry_set_text (name_entry, a_value);
        else
                gtk_entry_set_text (name_entry, "");
}


static const guchar *
get_name_edition_widget_value (GtkWidget * a_edition_widget)
{
        GtkEntry *name_entry = NULL;

        g_return_val_if_fail (a_edition_widget != NULL, NULL);
        g_return_val_if_fail (GTK_IS_DIALOG (a_edition_widget),
                              NULL);

        name_entry =
                gtk_object_get_data (GTK_OBJECT
                                     (a_edition_widget),
                                     NAME_EDITION_ENTRY_FIELD_KEY);

        g_return_val_if_fail (name_entry != NULL, NULL);
        g_return_val_if_fail (GTK_IS_ENTRY (name_entry), NULL);

        return gtk_entry_get_text (name_entry);
}


/*********************************************
 *Public methods of this class.
 *********************************************/

enum MLVIEW_XML_DOCUMENT_VIEW_STATUS
mlview_xml_document_view_construct (MlViewXMLDocumentView *
                                    a_doc_view,
                                    MlViewXMLDocument *
                                    a_mlview_xml_doc)
{
        g_return_val_if_fail (a_doc_view != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_doc_view),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (a_mlview_xml_doc != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_mlview_xml_doc),
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        g_return_val_if_fail (PRIVATE (a_doc_view) != NULL,
                              MLVIEW_XML_DOCUMENT_VIEW_BAD_PARAMETER);

        PRIVATE (a_doc_view)->mlview_xml_document =
                a_mlview_xml_doc;

        return mlview_xml_document_view_connect_to_document
                (a_doc_view, a_mlview_xml_doc);
}


MlViewXMLDocumentView *
mlview_xml_document_view_new (MlViewXMLDocument * a_mlview_doc)
{
        MlViewXMLDocumentView *result = NULL;

        g_return_val_if_fail (a_mlview_doc != NULL, NULL);

        result = g_object_new (MLVIEW_TYPE_XML_DOCUMENT_VIEW,
                               NULL);

        if (mlview_xml_document_view_construct
            (result, a_mlview_doc)
            == MLVIEW_XML_DOCUMENT_VIEW_OK) {
                return result;
        } else {
                return NULL;
        }
}


/**
 *References the current instance of #MlViewXMLDocumentView.
 *That is, increments the reference count associated to
 *the current instance of #MlViewXMLDocument.
 *
 *@param a_this the "this pointer".
 */
void
mlview_xml_document_view_ref (MlViewXMLDocumentView * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT_VIEW (a_this)
                          && PRIVATE (a_this));

        gtk_widget_ref (GTK_WIDGET (a_this));
}


/**
 *Unreferences the current instance of #MlViewXMLDocumentVIew.
 *That is, decrements the reference count associated to
 *to the current instance of #MlViewXMLDocument.
 *
 *@param a_this in out parameter. A pointer to the this pointer.
 *If the ref count reaches zero and if the current instance has been
 *deleted, *a_this is set to NULL.
 */
void
mlview_xml_document_view_unref (MlViewXMLDocumentView * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT_VIEW (a_this)
                          && PRIVATE (a_this));

        gtk_widget_unref (GTK_WIDGET (a_this));
}

/**
 *Associates an mlview xml document to the 
 *current instance of MlViewXMLDocumentView. 
 *
 *@param a_mlview_xml_doc the new mlview xml document 
 *to associate to the current instanceof MlViewXMLDocumentView. 
 *@param a_xml_doc_view the current instance of MlViewXMLDocumentView.
 *
 *
 */
void
mlview_xml_document_view_set_document (MlViewXMLDocumentView *
                                       a_xml_doc_view,
                                       MlViewXMLDocument *
                                       a_mlview_xml_doc)
{
        g_return_if_fail (a_xml_doc_view != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                          (a_xml_doc_view));
        g_return_if_fail (PRIVATE (a_xml_doc_view) != NULL);

        PRIVATE (a_xml_doc_view)->mlview_xml_document =
                a_mlview_xml_doc;
}


/**
 *
 *@param a_xml_doc_view the current instance of MlViewXMLDocumentView.
 *
 *
 */
MlViewXMLDocument *
mlview_xml_document_view_get_document (MlViewXMLDocumentView *
                                       a_xml_doc_view)
{
        g_return_val_if_fail (a_xml_doc_view != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_xml_doc_view), NULL);
        g_return_val_if_fail (PRIVATE (a_xml_doc_view) != NULL,
                              NULL);

        return PRIVATE (a_xml_doc_view)->mlview_xml_document;
}


/**
 *The type builder of this class. This method is part of the Gtk typing
 *system framework. 
 *
 */
gint
mlview_xml_document_view_get_type (void)
{
        static gint type_id = 0;

        if (type_id == 0) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewXMLDocumentViewClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_xml_document_view_class_init,
                        NULL, NULL,
                        sizeof (MlViewXMLDocumentView),
                        0,
                        (GInstanceInitFunc)
                        mlview_xml_document_view_init
                };
                type_id = g_type_register_static (GTK_TYPE_VBOX,
                                                  "MlViewXMLDocumentView",
                                                  &type_info, 0);
        }
        return type_id;
}

/**
 *Sets the document name to the new one. The new name is copied so that
 *the caller can free a_name if she wish. This function makes a_doc_view
 *emit the "name-changed" signal. 
 *
 *@param a_doc_view the document view.
 *@param a_name the new document name.
 *
 *
 */
void
mlview_xml_document_view_set_name (MlViewXMLDocumentView *
                                   a_doc_view, gchar * a_name)
{
        g_return_if_fail (a_doc_view != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                          (a_doc_view));
        g_return_if_fail (PRIVATE (a_doc_view) != NULL);

        if (PRIVATE (a_doc_view)->view_name) {
                g_free (PRIVATE (a_doc_view)->view_name);
                PRIVATE (a_doc_view)->view_name = NULL;
        }

        if (a_name)
                PRIVATE (a_doc_view)->view_name =
                        g_strdup (a_name);

        gtk_signal_emit (GTK_OBJECT (a_doc_view),
                         gv_mlview_xml_document_view_signals
                         [NAME_CHANGED]);
}

/**
 *
 *@param a_view the current xml document view.
 *
 *
 */
void
 mlview_xml_document_view_set_name_interactive
        (MlViewXMLDocumentView * a_view) {
        gint button = 0;
        guchar *view_name = NULL;
        MlViewAppContext *app_context = NULL;

        g_return_if_fail (a_view != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW (a_view));
        g_return_if_fail (PRIVATE (a_view) != NULL);
        g_return_if_fail (PRIVATE (a_view)->
                          mlview_xml_document != NULL);

        app_context =
                mlview_xml_document_get_app_context
                (PRIVATE (a_view)->mlview_xml_document);

        g_return_if_fail (app_context != NULL);

        if (PRIVATE (a_view)->name_edition_dialog == NULL) {
                PRIVATE (a_view)->name_edition_dialog =
                        GTK_DIALOG (build_name_edition_dialog
                                    (app_context));
        }

        g_return_if_fail (PRIVATE (a_view)->
                          name_edition_dialog != NULL);

        set_name_edition_widget_value
                (GTK_WIDGET
                 (PRIVATE (a_view)->name_edition_dialog),
                 mlview_xml_document_view_get_name (a_view));

        button = gtk_dialog_run (PRIVATE (a_view)->
                                 name_edition_dialog);

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*user clicked ok */
                view_name =
                        (guchar *) get_name_edition_widget_value
                        (GTK_WIDGET
                         (PRIVATE (a_view)->
                          name_edition_dialog));

                if (view_name != NULL)
                        mlview_xml_document_view_set_name
                                (a_view, view_name);
                break;
        default:
                break;
        }

        gtk_widget_hide (GTK_WIDGET
                         (PRIVATE (a_view)->
                          name_edition_dialog));
}


/**
 *
 *@param a_doc_view the current instance of MlViewXMLDocumentView.
 *
 *
 *@return a pointer to the name of the current xml document. 
 *Note the caller should neither 
 */
gchar *
mlview_xml_document_view_get_name (MlViewXMLDocumentView *
                                   a_doc_view)
{
        g_return_val_if_fail (a_doc_view != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_doc_view), NULL);
        g_return_val_if_fail (PRIVATE (a_doc_view) != NULL,
                              NULL);

        return PRIVATE (a_doc_view)->view_name;
}

/**
 *This functions is a startup init function. The does something only when it
 *is called for the first time. 
 *It sets the signal handlers of the  current view vis a vis
 *the underlying document. Subsequent calls just do nothing. 
 *
 *@param a_signals the signal handlers
 *
 *
 */
void
 mlview_xml_document_view_set_document_signal_handlers
        (SignalCallbackDescriptor * a_signals) {
        if (gv_signals_fired_by_document == NULL)
                gv_signals_fired_by_document = a_signals;
}

/**
 *
 *@param a_doc_view the current instance of MlViewXMLDocumentView.
 *
 *
 */
MlViewFileDescriptor
        * mlview_xml_document_view_get_file_descriptor
        (MlViewXMLDocumentView * a_doc_view) {
        g_return_val_if_fail (a_doc_view != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                              (a_doc_view), NULL);
        g_return_val_if_fail (PRIVATE (a_doc_view) != NULL,
                              NULL);

        return mlview_xml_document_get_file_descriptor
                (PRIVATE (a_doc_view)->mlview_xml_document);
}


/*
 *mlview_xml_docuement_view_destroy:
 *@a_object: the object to destroy.
 *
 *The destroyer of this object.
 */
void
mlview_xml_document_view_destroy (GtkObject *
                                  a_mlview_xml_doc_view)
{
        MlViewXMLDocumentView *mlview_xml_doc_view = NULL;
        MlViewXMLDocument *mlview_xml_doc = NULL;

        g_return_if_fail (a_mlview_xml_doc_view != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT_VIEW
                          (a_mlview_xml_doc_view));

        mlview_xml_doc_view =
                MLVIEW_XML_DOCUMENT_VIEW (a_mlview_xml_doc_view);
        g_return_if_fail (PRIVATE (mlview_xml_doc_view));

        mlview_xml_doc =
                mlview_xml_document_view_get_document
                (mlview_xml_doc_view);

        if (mlview_xml_doc)
                mlview_xml_document_view_disconnect_from_document
                        (mlview_xml_doc_view, mlview_xml_doc);

        if (PRIVATE (mlview_xml_doc_view)->view_name) {
                g_free (PRIVATE (mlview_xml_doc_view)->
                        view_name);
                PRIVATE (mlview_xml_doc_view)->view_name = NULL;
        }

        if (PRIVATE (mlview_xml_doc_view)) {
                g_free (PRIVATE (mlview_xml_doc_view));
                PRIVATE (mlview_xml_doc_view) = NULL;
        }

        if (mlview_xml_doc)
                mlview_xml_document_unref (mlview_xml_doc);

        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (*GTK_OBJECT_CLASS (parent_class)->destroy)
                        (a_mlview_xml_doc_view);
}
