/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYING file for copyright information.
 */
#include <string.h>
#include <libxml/tree.h>
#include <libxml/valid.h>

#include "mlview-xml-document.h"
#include "mlview-parsing-utils.h"
#include "mlview-file-selection.h"
#include "mlview-file-descriptor.h"
#include "mlview-marshal.h"

/**
 *@file
 *The definition of the #MlViewXMLDocument class.
 *
 *This is a wrapper of the "tree" module
 *of Daniel Veillard's libxml.
 *That way, all the real actions made on
 *the underlying xml data structure
 *are made through this class.
 *This class then emits signals to
 *notify its state to all the views that are
 *in relation with it instances. Of course, this
 *class should not know anything about the views that holds it ...
 *
 */

struct _MlViewXMLDocumentPrivate {
        /*
         *the file descriptor associated to
         *this mlview xml document.
         */
        MlViewFileDescriptor *file_desc;
        /*the underlying xml doc data structure */
        xmlDocPtr xml_doc;
        /*the current xml application context */
        MlViewAppContext *app_context;
        /*used in the dispose private method*/
        gboolean dispose_has_run;
        /**
         *A list of all the xml nodes
         *arranged in a linear way.
         *This is used by the search functions.
         */
        GList *nodes_list ;
        GHashTable *nodes_hash ;
        /*
         *a number that is incremented at each modification
         *on the tree.
         */
        gulong modif_sequence ;
        /*somewhere to store a given modif_sequence*/
        gulong last_modif_sequence ;
} ;

enum {
        DOCUMENT_CHANGED = 0,
        NODE_CUT,
        CHILD_NODE_ADDED,
        PREV_SIBLING_NODE_INSERTED,
        NEXT_SIBLING_NODE_INSERTED,
        CONTENT_CHANGED,
        NAME_CHANGED,
        NODE_CHANGED,
        NODE_ATTRIBUTE_ADDED,
        NODE_ATTRIBUTE_NAME_CHANGED,
        NODE_ATTRIBUTE_VALUE_CHANGED,
        NODE_ATTRIBUTE_REMOVED,
        NODE_NAMESPACE_ADDED,
        NODE_NAMESPACE_REMOVED,
        NODE_NAMESPACE_CHANGED,
        FILE_PATH_CHANGED,
        SEARCHED_NODE_FOUND, 
        NODE_SELECTED,
        NUMBER_OF_SIGNALS
};

#define PRIVATE(mlview_xml_doc) ((mlview_xml_doc)->priv)
#define MLVIEW_XML_DOCUMENT_DEFAULT_VALIDATION_ON FALSE ;


static void mlview_xml_document_do_interactive_validation_if_needed (MlViewXMLDocument * a_doc);

#define MLVIEW_XML_DOCUMENT_NODE_CLIPBOARD_SIZE 128
static GObjectClass *gv_parent_class = NULL;
static guint gv_signals[NUMBER_OF_SIGNALS] =
        { 0 };
static xmlNode
        * gv_clipboard[MLVIEW_XML_DOCUMENT_NODE_CLIPBOARD_SIZE] =
        { 0 };
static guint gv_clipboard_index = 0;
static guint gv_clipboard_ref_count = 0;


static void mlview_xml_document_destroy_clipboard (void);

static void mlview_xml_document_ref_clipboard (void);

static gboolean mlview_xml_document_unref_clipboard (void);

static void mlview_xml_document_class_init (MlViewXMLDocumentClass * a_klass);

static void mlview_xml_document_init (MlViewXMLDocument * a_xml_doc);

static void free_tree_list_cache (MlViewXMLDocument *a_this) ;

/*======================================================
 *private helper functions
 *=====================================================*/


/**
 *Destroys the instances of xmlNode
 *that are present in the clipboard.
 *
 */
static void
mlview_xml_document_destroy_clipboard (void)
{
        guint i = 0;

        for (i = 0; i < gv_clipboard_index; i++) {
                if (gv_clipboard[i]) {
                        xmlFreeNode (gv_clipboard[i]);
                        gv_clipboard[i] = NULL;
                }
        }

        gv_clipboard_index = 0;
}


/**
 *Increments the ref counter of the
 *clipboard.
 */
static void
mlview_xml_document_ref_clipboard (void)
{
        gv_clipboard_ref_count++;
}

/**
 *Decrements the ref count of the clipboard.
 *If the ref count reaches zero, the clipboard
 *if destroyed.
 *
 *@return TRUE if the clipboard got destroyed, FALSE
 *otherwise.
 */
static gboolean
mlview_xml_document_unref_clipboard (void)
{
        if (gv_clipboard_ref_count > 0)
                gv_clipboard_ref_count--;

        if (gv_clipboard_ref_count == 0) {
                mlview_xml_document_destroy_clipboard ();
                return TRUE;
        }

        return FALSE;
}


/**
 *Validates the document if the settings allows
 *it.
 *@param a_doc the "this" pointer.
 */
static void
mlview_xml_document_do_interactive_validation_if_needed (MlViewXMLDocument * a_doc) 
{
        struct MlViewAppSettings *settings = NULL ;

        g_return_if_fail (a_doc != NULL);
        g_return_if_fail (PRIVATE (a_doc) != NULL);
        g_return_if_fail (PRIVATE (a_doc)->xml_doc != NULL);

        if (PRIVATE (a_doc)->app_context) {
                settings = mlview_app_context_get_settings 
                        (PRIVATE (a_doc)->app_context) ;
                g_return_if_fail (settings) ;
        }

        if (settings->general.validation_is_on == TRUE) {
                mlview_xml_document_validate (a_doc);
        }
}


/***********************************************
 *private gtk framework methods						
 ***********************************************/

static void
mlview_xml_document_dispose (GObject * a_xml_doc)
{
        MlViewXMLDocument *xml_doc = NULL;

        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));

        xml_doc = MLVIEW_XML_DOCUMENT (a_xml_doc);

        g_return_if_fail (PRIVATE (xml_doc));
        g_return_if_fail (PRIVATE (xml_doc)->dispose_has_run ==
                          FALSE);

        if (PRIVATE (xml_doc)->file_desc != NULL) {
                mlview_file_descriptor_destroy (PRIVATE
                                                (xml_doc)->
                                                file_desc);
                PRIVATE (xml_doc)->file_desc = NULL;
        }

        /*
         *Free the nodes contained in the clipboard
         *before freeing the document that contain
         *them or else, the nodes will have a dangling
         *pointer to a freed document. That could lead
         *to invalid read or invalid write and thus to
         *stability problems.
         */
        mlview_xml_document_unref_clipboard ();
        if (PRIVATE (xml_doc)->xml_doc != NULL) {
                xmlFreeDoc (PRIVATE (xml_doc)->xml_doc);

                PRIVATE (xml_doc)->xml_doc = NULL;
        }
        if (PRIVATE (xml_doc)->nodes_list) {
                free_tree_list_cache (xml_doc) ;
        }
        PRIVATE (xml_doc)->dispose_has_run = TRUE;
}

static void
mlview_xml_document_finalize (GObject * a_object)
{
        MlViewXMLDocument *xml_document = NULL;

        g_return_if_fail (a_object);
        xml_document = MLVIEW_XML_DOCUMENT (a_object);
        g_return_if_fail (xml_document);
        g_return_if_fail (PRIVATE (xml_document));
        g_return_if_fail (PRIVATE (xml_document)->
                          dispose_has_run == TRUE);

        g_free (PRIVATE (xml_document));
        PRIVATE (xml_document) = NULL;
}

/**
 *Inits the MlViewXMLDocument class.
 *@param a_klass the class structure containing the vtable
 *and all the class wide data.
 */
static void
mlview_xml_document_class_init (MlViewXMLDocumentClass * a_klass)
{
        GObjectClass *object_class = NULL;

        g_return_if_fail (a_klass != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT_CLASS
                          (a_klass));

        object_class = G_OBJECT_CLASS (a_klass);
        gv_parent_class = g_type_class_peek_parent (a_klass);
        object_class->dispose = mlview_xml_document_dispose;
        object_class->finalize = mlview_xml_document_finalize;

        /*signal creation code */
        gv_signals[DOCUMENT_CHANGED] =
                g_signal_new ("document-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               document_changed), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);

        gv_signals[NODE_CUT] =
                g_signal_new ("node-cut",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass, node_cut),
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);

        gv_signals[CHILD_NODE_ADDED] =
                g_signal_new ("child-node-added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               child_node_added), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);

        gv_signals
                [PREV_SIBLING_NODE_INSERTED] =
                g_signal_new ("prev-sibling-node-inserted",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               prev_sibling_node_inserted), NULL,
                              NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);

        gv_signals
                [NEXT_SIBLING_NODE_INSERTED] =
                g_signal_new ("next-sibling-node-inserted",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               next_sibling_node_inserted), NULL,
                              NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);

        gv_signals[CONTENT_CHANGED] =
                g_signal_new ("content-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               content_changed), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NAME_CHANGED] =
                g_signal_new ("name-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               name_changed), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_CHANGED] =
                g_signal_new ("node-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_changed), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[FILE_PATH_CHANGED] =
                g_signal_new ("file-path-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               file_path_changed), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);

        gv_signals[SEARCHED_NODE_FOUND] =
                g_signal_new ("searched-node-found",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               searched_node_found), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[NODE_ATTRIBUTE_NAME_CHANGED] = 
                g_signal_new ("node-attribute-name-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_attribute_name_changed), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER) ;

        gv_signals[NODE_ATTRIBUTE_ADDED] = 
                g_signal_new ("node-attribute-added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_attribute_added), NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER) ;

        gv_signals[NODE_ATTRIBUTE_VALUE_CHANGED] =
                g_signal_new ("node-attribute-value-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_attribute_value_changed), 
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER) ;

        gv_signals[NODE_ATTRIBUTE_REMOVED] =
                g_signal_new ("node-attribute-removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_attribute_removed), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER) ;

        gv_signals[NODE_NAMESPACE_ADDED] =
                g_signal_new ("node-namespace-added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_namespace_added), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER) ;
        
        gv_signals[NODE_NAMESPACE_CHANGED] =
                g_signal_new ("node-namespace-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_namespace_changed), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER) ;
        
        gv_signals[NODE_NAMESPACE_REMOVED] =
                g_signal_new ("node-namespace-removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass,
                               node_namespace_removed), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER) ;

        gv_signals[NODE_SELECTED] =
                g_signal_new ("node-selected",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewXMLDocumentClass, node_selected),
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER) ;

        a_klass->document_changed = NULL;
        a_klass->node_cut = NULL;
        a_klass->child_node_added = NULL;
        a_klass->node_changed = NULL;
        a_klass->content_changed = NULL;
        a_klass->name_changed = NULL;
        a_klass->searched_node_found = NULL ;
}

/**
 *The init of the class instance.
 *@param a_xml_doc the newly built "this pointer".
 */
static void
mlview_xml_document_init (MlViewXMLDocument * a_xml_doc)
{
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));

        PRIVATE (a_xml_doc) =
                g_malloc0 (sizeof (MlViewXMLDocumentPrivate));
}

/**
 *Is called on the emission of the "document-changed" signal.
 *@param a_doc the instance of #MlViewXMLDocument that emitted
 *the signal.
 *@param a custom data pointed.
 */
static void
document_changed_cb (MlViewXMLDocument *a_doc,
                     gpointer a_data)
{
        g_return_if_fail (a_doc
                          && MLVIEW_IS_XML_DOCUMENT (a_doc)
                          && PRIVATE (a_doc)) ;
        PRIVATE (a_doc)->modif_sequence ++ ;
        if (PRIVATE (a_doc)->file_desc) {
                mlview_file_descriptor_update_modified_time
                        (PRIVATE (a_doc)->file_desc) ;
        }
}


/**
 *Builds a chained list where each element is an xml node.
 *the list is build by visiting the xml tree nodes in a depth first
 *scheme. During the visit, each node found is prepended to the list.
 *Yeah, we prepend each xml node to the list (not append) because
 *prepend is _much_ faster than append.
 *To walk the resulting tree, one must first, get the tail of
 *the list, and walk the list backward.
 *@param a_node the root of the xml tree to visit.
 *@param a_list out parameter. The list to build.
 *@param a_nodes_hash a lookup table built during the visit of
 *xml nodes. It contains pairs of (xml node/GList).
 *@return
 */
static enum MlViewStatus
build_tree_list_cache_real (xmlNode *a_node,
                            GList **a_list,
                            GHashTable **a_nodes_hash)
{
        GList *list = NULL ;
        GHashTable *hash = NULL ;
        xmlNode *cur_node = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;
        list = *a_list ;
        hash = *a_nodes_hash ;

        if (!hash) {
                hash = g_hash_table_new (g_direct_hash, 
                                         g_direct_equal) ;
                if (!hash) {
                        mlview_utils_trace_info
                                ("hash failed. "
                                 "System may be out of memory.") ;
                        return MLVIEW_OUT_OF_MEMORY_ERROR ;
                }
        }
        for (cur_node = a_node ;
             cur_node ;
             cur_node = cur_node->next) {
                list = g_list_prepend (list, cur_node) ;
                g_hash_table_insert (hash, cur_node, list) ;
                if (cur_node->children) {
                        status = build_tree_list_cache_real
                                (cur_node->children, 
                                 &list, &hash) ;
                        g_return_val_if_fail (status == MLVIEW_OK,
                                              status) ;
                }
        }
        *a_list = list ;
        *a_nodes_hash = hash ;
        return MLVIEW_OK ;
}

/**
 *Frees the list built by build_tree_list_cache_real()
 */
static void
free_tree_list_cache (MlViewXMLDocument *a_this)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && PRIVATE (a_this)) ;
        if (PRIVATE (a_this)->nodes_list) {
                g_list_free (PRIVATE (a_this)->nodes_list) ;
                PRIVATE (a_this)->nodes_list = NULL ;
        }
        if (PRIVATE (a_this)->nodes_hash) {
                g_hash_table_destroy (PRIVATE (a_this)->nodes_hash) ;
                PRIVATE (a_this)->nodes_hash = NULL ;
        }
}


/**
 *Builds an oredered list of all the xml node
 *of the current document.
 *This is used in the algorithm of "search node" feature.
 *@param a_this the current instance of #MlViewXMLDocument.
 */
static enum MlViewStatus
build_tree_list_cache (MlViewXMLDocument *a_this)
{
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this
                              && MLVIEW_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->xml_doc,
                              MLVIEW_BAD_PARAM_ERROR) ;
        if (!PRIVATE (a_this)->xml_doc->children)
                return MLVIEW_OK ;
        status = build_tree_list_cache_real
                (PRIVATE (a_this)->xml_doc->children,
                 &PRIVATE (a_this)->nodes_list,
                 &PRIVATE (a_this)->nodes_hash) ;
        g_return_val_if_fail (status == MLVIEW_OK, status) ;
        return status ;
}

/**
 *Searches a string in the various parts
 *of an xml node (name, content, attribute names, 
 *attribute values)
 *@param a_node the node to search in.
 *@param a_conf the search configuration (i.e: what string 
 *to search and where).
 *@return TRUE if the string has been found in this node.
 */
static gboolean
search_in_node (xmlNode *a_node,
                const struct SearchConfig *a_conf)
{
        gboolean node_matches = FALSE ;

        g_return_val_if_fail (a_node && a_conf, FALSE) ;

        if (a_node->type == XML_ELEMENT_NODE 
            && (a_conf->where & NODE_NAME)) {
                if (mlview_utils_strstr
                    (a_node->name, a_conf->search_string,
                     a_conf->ignore_case) == TRUE) {
                        return TRUE ;
                }
        }
        if (a_node->properties 
            && ((a_conf->where & NODE_ATTRIBUTE_NAME)
                ||(a_conf->where & NODE_ATTRIBUTE_VALUE))) {
                xmlAttr *cur_attr = NULL;
                xmlChar *cur_attr_val = NULL ;
                /*
                 *cycle through the attribute list
                 *to see if one matches.
                 */
                for (cur_attr = a_node->properties;
                     cur_attr; cur_attr = cur_attr->next) {
                        if (cur_attr->name
                            && (a_conf->where & NODE_ATTRIBUTE_NAME)
                            && mlview_utils_strstr
                            (cur_attr->name,
                             a_conf->search_string, 
                             a_conf->ignore_case) == TRUE) {
                                node_matches = TRUE ;
                                break ;
                        }
                        if (cur_attr->name
                            && (a_conf->where
                                & NODE_ATTRIBUTE_VALUE)
                            && (cur_attr_val 
                                = xmlGetProp (a_node,
                                              cur_attr->name))) {
                                if (mlview_utils_strstr
                                    (cur_attr_val,
                                     a_conf->search_string,
                                     a_conf->ignore_case) == TRUE) {
                                        node_matches = TRUE ;
                                }
                        }
                        /*
                         *free the result of xmlGetProp,
                         *if any.
                         */
                        if (cur_attr_val) {
                                xmlFree (cur_attr_val) ;
                                cur_attr_val = NULL ;
                        }
                        if (node_matches == TRUE)
                                break ;
                }
        }
        if (a_node->type == XML_TEXT_NODE
            && a_conf->where & NODE_CONTENT) {
                guchar *content = NULL ;
                content = xmlNodeGetContent (a_node) ;
                if (content 
                    && mlview_utils_strstr
                    (content, a_conf->search_string,
                     a_conf->ignore_case) == TRUE) {
                        node_matches = TRUE ;
                }
                if (content) {
                        xmlFree (content) ;
                        content = NULL ;
                }
        }
        return node_matches ;
}

/*****************
 *public methods *
 *****************/


/**
 *The actual #MlViewXMLDocument type ID builder.
 *@return the type ID of the #MlViewXMLDocument class.
 */
guint
mlview_xml_document_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewXMLDocumentClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_xml_document_class_init,
                        NULL, NULL,
                        sizeof (MlViewXMLDocument),
                        0,
                        (GInstanceInitFunc)
                        mlview_xml_document_init
                };
                type = g_type_register_static (G_TYPE_OBJECT,
                                               "MlViewXMLDocument",
                                               &type_info, 0);
        }

        return type;
}


/**
 *The class instance builder.
 *Builds a new instance of the #MlViewXMLDocument. Initialyzes
 *it with a native instance of xmlDoc * (see libxml2 to learn more
 *about xmlDoc*)
 *
 *@param a_xml_doc the native libxml2 xmlDocPtr.
 *@param a_context the mlview application context.
 *@return the newly created instance of MlViewXMLDocument.
 */
MlViewXMLDocument *
mlview_xml_document_new (xmlDocPtr a_xml_doc,
                         MlViewAppContext * a_context)
{
        MlViewXMLDocument *result = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlNodePtr xml_node;

        result = g_object_new (MLVIEW_TYPE_XML_DOCUMENT, NULL);

        xml_doc = a_xml_doc;

        if (xml_doc == NULL) {
                xml_doc = xmlNewDoc ("1.0");
                xml_node = xmlNewNode (NULL, "");
                xmlDocSetRootElement (xml_doc, xml_node);
        }

        PRIVATE (result)->xml_doc = xml_doc;
        PRIVATE (result)->app_context = a_context;

        mlview_xml_document_ref_clipboard ();

        g_signal_connect (G_OBJECT (result),
                          "document-changed",
                          G_CALLBACK (document_changed_cb),
                          NULL) ;        
        return result;
}


/**
 *Associates the current instance of 
 *#MlViewXMLDocument with a new instance
 *of #MlViewApplicationContext.
 *
 *@param a_doc the current xml document.
 *@param a_context the current application 
 *context to set to this component.
 */
void
mlview_xml_document_set_app_context (MlViewXMLDocument * a_doc,
                                     MlViewAppContext *a_context)
{
        g_return_if_fail (a_doc != NULL);
        g_return_if_fail (PRIVATE (a_doc) != NULL);
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));


        PRIVATE (a_doc)->app_context = a_context;
}


/**
 *A Getter of the application context associated to
 *this instance of #MlViewAppContext.
 *
 *@param a_doc the "this pointer".
 *@return the application context.
 */
MlViewAppContext *
mlview_xml_document_get_app_context (MlViewXMLDocument * a_doc)
{
        g_return_val_if_fail (a_doc, NULL);
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, NULL);

        return PRIVATE (a_doc)->app_context;
}

/************************************
 *clipboard methods.
 ***********************************/

/**
 *Stores the a copy of the xmlNode given in argument into the node clipboard.
 *The cilpboard is a circular buffer. It cannot be overflowed. 
 *
 *@param a_visual_node the pointer to 
 *xmlNode to store in the clipboard. If it is a 
 *NULL pointer, a NULL pointer is actually stored into clipboard.
 */
void
mlview_xml_document_copy_node_to_clipboard (xmlNode * a_xml_node,
                                            xmlDoc * a_doc)
{
        g_return_if_fail (a_xml_node != NULL);

        g_return_if_fail (sizeof (gv_clipboard) /
                          sizeof (xmlNodePtr) > 0);

        /*if buffer full, go to beginning of buffer. */
        if (gv_clipboard_index >=
            sizeof (gv_clipboard) / sizeof (xmlNodePtr)) {
                gv_clipboard_index = 0;
        }

        /*avoid mem leaks */
        if (gv_clipboard[gv_clipboard_index]) {
                xmlFreeNode (gv_clipboard[gv_clipboard_index]);
                gv_clipboard[gv_clipboard_index] = NULL;
        }

        gv_clipboard[gv_clipboard_index] =
                xmlDocCopyNode (a_xml_node, a_doc, TRUE);

        gv_clipboard_index++;
}


/**
 *Gets the last node stored in the node clipboard.
 *
 *@param a_xml_doc the "this pointer".
 *@return a copy of the 
 *xmlNode that has been previously stored. 
 */
xmlNode *
mlview_xml_document_get_node_from_clipboard (xmlDoc * a_xml_doc)
{
        xmlNodePtr xml_node = NULL;

        g_return_val_if_fail (sizeof (gv_clipboard) /
                              sizeof (xmlNodePtr) > 0, NULL);

        if (gv_clipboard_index >=
            sizeof (gv_clipboard) / sizeof (xmlNodePtr)) {
                gv_clipboard_index =
                        sizeof (gv_clipboard) /
                        sizeof (xmlNodePtr) - 1;
        }

        xml_node = gv_clipboard[gv_clipboard_index - 1];

        if (xml_node != NULL) {
                /*make a recursive copy of the node */
                xml_node =
                        xmlDocCopyNode (xml_node, a_xml_doc, 1);
        }

        return xml_node;
}


/*************************************
 *Other miscellaneous methods
 *************************************/

/**
 *Opens and loads the xml file contains in the filename given in parameter. 
 *
 *@param a_file_name the total file path of the file that contains 
 *the xml document to open
 *@param a_context the mlview application context.
 *the current instance of MlViewXMLDocument.
 *@return the newly loaded xml document.
 */
MlViewXMLDocument *
mlview_xml_document_open (gchar * a_file_name,
                          MlViewAppContext * a_context)
{
        MlViewXMLDocument *result = NULL;
        xmlDoc *xml_doc = NULL;

        xml_doc =
                mlview_parsing_utils_load_xml_file (a_file_name,
                                                    a_context);
        if (!xml_doc)
                return NULL ;
        result = mlview_xml_document_new (xml_doc, a_context);

        mlview_xml_document_do_interactive_validation_if_needed
                (result);
        mlview_xml_document_set_file_path (result, a_file_name);

        return result;
}


/**
 *Associates a Dtd to the current xml document.
 *If the document already contains
 *a Dtd, this functions displays a dialog box 
 *to ask the user if he wants to overide
 *that Dtd. Depending on the answer of the user, 
 *this function does the association
 *or not. 
 *
 *@param a_doc the current mlview xml document.
 *@return 0 if the association has been done, 
 *1 if the user cancelled the operations, 
 */
gint
mlview_xml_document_associate_dtd_interactive (MlViewXMLDocument* a_doc)
{
        MlViewExtSubsDef *external_subset_def =
                NULL;
        gint result = 0;

        g_return_val_if_fail (a_doc != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_doc)->app_context !=
                              NULL, -1);

        external_subset_def =
                mlview_parsing_utils_let_user_choose_a_dtd
                (PRIVATE (a_doc)->app_context,
                 (guchar*)_("Choose a Dtd to associate to the document"));

        if (external_subset_def) {
                result = mlview_xml_document_associate_dtd
                        (a_doc, external_subset_def);
                mlview_ext_subs_def_destroy
                        (external_subset_def);
                return result;

        } else {
                return 1;
        }
}


/**
 *Associates a the dtd designated by 
 *a_ext_subs_def to the current xml document.
 *Note that if the document contains is already 
 *associated to a dtd, the function asks
 *the user if she wants to overide it. 
 *If the user agrees, the functions frees the former
 *dtd and makes the new association. If not, nothing is done. 
 *
 *@return 0 if the association has been done, 1 
 *if the user cancelled the operations, 
 */
gint 
mlview_xml_document_associate_dtd (MlViewXMLDocument * a_doc,
                                   MlViewExtSubsDef * a_ext_subs_def) 
{
        xmlDtd *dtd = NULL;
        gboolean user_cancels = FALSE;

        g_return_val_if_fail (a_doc != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, -1);

        /*
         *if an external subset already exists, 
         *ask the user if she wants to overide it
         */
        if (PRIVATE (a_doc)->xml_doc
            && PRIVATE (a_doc)->xml_doc->extSubset) {
                GtkDialog *dialog = NULL;
                GtkWidget *label = NULL;
                gint button = 0;

                dialog = GTK_DIALOG
                        (gtk_dialog_new_with_buttons
                         (_("Document has already has a Dtd"),
                          NULL, GTK_DIALOG_MODAL, 
                          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 
                          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                          NULL));
                g_return_val_if_fail (dialog, -1) ;
                gtk_dialog_set_default_response 
                        (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT) ;
                label = gtk_label_new 
                        (_("This Document already has an associated Dtd.\nDo you want really want to associate another Dtd to this document?"));

                gtk_box_pack_start (GTK_BOX (dialog->vbox),
                                    label, TRUE, FALSE, 0);

                gtk_widget_show_all (dialog->vbox);
                button = gtk_dialog_run (dialog);

                switch (button) {
                case GTK_RESPONSE_ACCEPT: /*YES button */
                        break;
                default:       /*CANCEL Button, dialog box closed etc.. */
                        user_cancels = TRUE;
                        break;
                }
                gtk_widget_destroy (GTK_WIDGET (dialog)) ;
        }
        /*end if */
        if (user_cancels == TRUE)
                return 1;       /*user cancelled operations */

        /*if the user did not cancel the operations, let's do the association */

        /*let's parse the external subset. 
         *If parsing is OK, we associate that dtd to
         *the document.
         */
        if (a_ext_subs_def->system_id
            && strcmp (a_ext_subs_def->system_id, "")
            && (dtd = mlview_parsing_utils_load_a_dtd
                (a_ext_subs_def,
                 PRIVATE (a_doc)->app_context))) {

                if (PRIVATE (a_doc)->xml_doc->extSubset)
                        xmlFreeDtd (PRIVATE (a_doc)->xml_doc->
                                    extSubset);

                PRIVATE (a_doc)->xml_doc->extSubset = dtd;

                dtd->doc = PRIVATE (a_doc)->xml_doc;

                PRIVATE (a_doc)->xml_doc->standalone = 0;

                return 0;       /*dtd parsing OK, association OK */
        }

        return 2;               /*dtd parsing failed */
}


/**
 *Validates the document agains the dtd that is associated to it. 
 *
 *@param a_doc the document to validate.
 *@return O upon successfull completion, or an error code.
 */
gint
mlview_xml_document_validate (MlViewXMLDocument * a_doc)
{
        /*this defaul value means "there is no dtd to parse" */
        gint validity_status = -2;

        g_return_val_if_fail (a_doc != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, -1);
        g_return_val_if_fail (PRIVATE (a_doc)->xml_doc, -1);

        if (PRIVATE (a_doc)->xml_doc->extSubset
            || PRIVATE (a_doc)->xml_doc->intSubset) {

                validity_status =
                        mlview_parsing_utils_validate_dtd
                        (PRIVATE (a_doc)->xml_doc,
                         PRIVATE (a_doc)->xml_doc->extSubset,
                         PRIVATE (a_doc)->app_context);
        } else {

                mlview_app_context_warning
                        (PRIVATE (a_doc)->app_context,
                         _
                         ("No DTD was provided. Could not validate the document"));

                return validity_status;
        }

        /*If the document is valid, tell the user. */
        if (!validity_status)

                mlview_app_context_message (PRIVATE (a_doc)->
                                            app_context,
                                            _
                                            ("The Document is valid."));

        else
                mlview_app_context_warning (PRIVATE (a_doc)->
                                            app_context,
                                            _
                                            ("The Document is not valid!"));

        return validity_status;
}


/**
 *Saves the document.
 *
 *@param a_doc the current instance of MlViewXMLDocument
 *@param a_file_path the path where to save the document.
 *@param a_check_overwrt  whether to check and warn user
 *when the we are going to overwrite a file.
 *@return the number of bytes saved or -1 in case of error.
 */
gint
mlview_xml_document_save (MlViewXMLDocument * a_doc,
                          gchar * a_file_path,
                          gboolean a_check_overwrt)
{
        MlViewFileDescriptor *file_desc = NULL;
        guchar *cur_doc_file_path = NULL ;
        gboolean save_file = TRUE,
                is_reg_file = TRUE;

        g_return_val_if_fail (a_doc != NULL, -1) ;
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, -1) ;

        if (PRIVATE (a_doc)->file_desc) {
                cur_doc_file_path = mlview_file_descriptor_get_file_path
                        (PRIVATE (a_doc)->file_desc) ;
        }
        if (a_check_overwrt == TRUE 
            && (!cur_doc_file_path 
                || strcmp (cur_doc_file_path, a_file_path))) {
                file_desc =
                        mlview_file_descriptor_new (a_file_path);
                g_return_val_if_fail (file_desc, -1);
                
                if (!mlview_file_descriptor_is_regular_file
                    (file_desc, &is_reg_file)
                    && is_reg_file == TRUE) {
                        GtkDialog *ok_cancel_dialog = NULL;
                        GtkLabel *message = NULL;
                        gchar *str = NULL;

                        str = g_strdup_printf
                                (_("The file '%s' already exists.\n"
				   "Do you want to overwrite it?"),
                                 a_file_path);
			message =
                                GTK_LABEL (gtk_label_new (str));
                        ok_cancel_dialog =
                                GTK_DIALOG
                                (gtk_dialog_new_with_buttons
                                 (_("Save"), NULL,
                                  GTK_DIALOG_MODAL,
                                  GTK_STOCK_NO, GTK_RESPONSE_REJECT,
                                  GTK_STOCK_YES, GTK_RESPONSE_ACCEPT,
                                  NULL));
                        g_return_val_if_fail 
                                (ok_cancel_dialog, -1) ;
                        gtk_dialog_set_default_response 
                                (GTK_DIALOG (ok_cancel_dialog),
                                 GTK_RESPONSE_ACCEPT) ;
                        gtk_box_pack_start 
                                (GTK_BOX
                                 (GTK_DIALOG
                                  (ok_cancel_dialog)->vbox),
                                 GTK_WIDGET (message),
                                 TRUE, TRUE, 0);

                        gtk_widget_show_all
                                (GTK_WIDGET
                                 (GTK_DIALOG 
                                  (ok_cancel_dialog)->vbox));

                        if (str) {
                                g_free (str);
                                str = NULL;
                        }
                        if (ok_cancel_dialog) {
                                gint button = 0;

                                button = gtk_dialog_run
                                        (ok_cancel_dialog);

                                switch (button) {
                                case GTK_RESPONSE_ACCEPT: 
                                   /*The OK Button was pressed */
                                        save_file = TRUE;
                                        break;

                                case GTK_RESPONSE_REJECT: 
                               /*The cancel Button was pressed */
                                case GTK_RESPONSE_CLOSE:
                                default:
                                        save_file = FALSE;
                                        break;
                                }
                                gtk_widget_destroy 
                                        (GTK_WIDGET
                                         (ok_cancel_dialog));
                        }
                }
        }

        if (file_desc) {
                mlview_file_descriptor_destroy (file_desc);
                file_desc = NULL;
        }
        if (save_file == TRUE) {
                gint res = 0;
                xmlDoc *xml_doc = NULL;

                xml_doc =
                        mlview_xml_document_get_xml_document
                        (a_doc);
                g_return_val_if_fail (xml_doc, -1);

                res = mlview_parsing_utils_save_xml_doc
                        (xml_doc, a_file_path,
                         PRIVATE (a_doc)->app_context);

                if (res > 0)
                        mlview_xml_document_set_file_path 
                                (a_doc, a_file_path);

                return res;
        }

        return 0;
}


/**
 *Clones the current instance of MlViewXMLDocument. 
 *@param a_original the current instance of MlViewXMLDocument.
 *@return the cloned instance of #MlViewXMLDocument.
 */
MlViewXMLDocument *
mlview_xml_document_clone (MlViewXMLDocument * a_original)
{
        MlViewXMLDocument *result;
        MlViewFileDescriptor *file_desc;
        gchar *file_path;

        if (a_original == NULL)
                return NULL;
        file_desc =
                mlview_xml_document_get_file_descriptor
                (a_original);
        file_path =
                mlview_file_descriptor_get_file_path (file_desc);
        result = mlview_xml_document_open (file_path,
                                           PRIVATE (a_original)->
                                           app_context);
        return result;
}

/**
 *Sets the file path associated to the current instance
 *of MlViewXMLDocument.
 *@param a_xml_doc the "this" pointer of the current object.
 *@param a_file_path the new file path.
 *
 */
void
mlview_xml_document_set_file_path (MlViewXMLDocument * a_xml_doc,
                                   gchar * a_file_path)
{
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (PRIVATE (a_xml_doc) != NULL);

        if (PRIVATE (a_xml_doc)->file_desc == NULL) {
                PRIVATE (a_xml_doc)->file_desc =
                        mlview_file_descriptor_new (a_file_path);
        } else {
                mlview_file_descriptor_set_file_path
                        (PRIVATE (a_xml_doc)->file_desc,
                         a_file_path);
        }

        g_signal_emit (G_OBJECT (a_xml_doc),
                       gv_signals[FILE_PATH_CHANGED], 0);
}


/**
 *Gets the file path associated to the current instance of
 *#MlViewXMLDocument.
 *@param a_xml_doc the "this" pointer of the current
 *instance.
 *@return the file path. Note that the caller should not free
 *the returned string!
 */
gchar *
mlview_xml_document_get_file_path (MlViewXMLDocument * a_xml_doc)
{
        g_return_val_if_fail (a_xml_doc
                              && PRIVATE (a_xml_doc), NULL);

        if (PRIVATE (a_xml_doc)->file_desc == NULL) {
                return NULL;
        } else {
                return mlview_file_descriptor_get_file_path
                        (PRIVATE (a_xml_doc)->file_desc);
        }
}


/**
 *Adds a child node to the a_parent_node node. 
 *After adding the child node, this methods emits the following
 *signals: "child-node-added" and "document-changed".
 *
 *@param a_xml_node the node to add.
 *@param a_xml_doc the current instance of MlViewXMLDocument.
 *@param a_parent_node the parent of the node to add.
 *@return the newly added node or NULL if adding failed.
 */
xmlNode *
mlview_xml_document_add_child_node (MlViewXMLDocument *a_this,
                                    xmlNode * a_parent_xml_node,
                                    xmlNode * a_xml_node,
                                    gboolean a_subtree_required,
                                    gboolean a_emit_signal)
{
        xmlNode *result = NULL;
        struct MlViewAppSettings *settings = NULL ;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_this),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context, 
                              NULL);
        g_return_val_if_fail (a_parent_xml_node, NULL);        
        g_return_val_if_fail ((a_parent_xml_node->type ==
                               XML_ELEMENT_NODE)
                              || (a_parent_xml_node->type ==
                                  XML_DOCUMENT_NODE), NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);
        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;

        result = xmlAddChild (a_parent_xml_node, a_xml_node);
        g_return_val_if_fail (result != NULL, NULL);

        if (a_subtree_required == TRUE
            && settings->general.validation_is_on == TRUE
            && result->type == XML_ELEMENT_NODE) {
                enum MLVIEW_PARSING_UTILS_STATUS status;
                status = mlview_parsing_utils_build_required_attributes_list 
                        (PRIVATE (a_this)->app_context, result);
                status = mlview_parsing_utils_build_required_children_tree 
                        (PRIVATE (a_this)->app_context, &result);
        }

        /*emit the proper signals */
        if (a_emit_signal == TRUE) {
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[CHILD_NODE_ADDED], 0,
                         a_parent_xml_node, result);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return result;
}


/**
 *Cuts the node a_xml_node. The node is stored in the node clipboard
 *and then unlinked from the xml tree. 
 *After a successfull cut, this methods emits the following signals:
 *"node-cut" and "document-changed".
 *
 *@param a_this the current xml node
 *@return the xmlNode cut or NULL if something bad happened.
 */
xmlNode *
mlview_xml_document_cut_node (MlViewXMLDocument * a_this,
                              xmlNode * a_xml_node,
                              gboolean a_emit_signal)
{
        xmlNode *parent_node = NULL,
                *result = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_this),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this)->xml_doc !=
                              NULL, NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);

        g_return_val_if_fail (a_xml_node->doc
                              == PRIVATE (a_this)->xml_doc,
                              NULL);

        parent_node = a_xml_node->parent;

        g_return_val_if_fail (parent_node != NULL, NULL);

        mlview_xml_document_copy_node_to_clipboard
                (a_xml_node, PRIVATE (a_this)->xml_doc);

        xmlUnlinkNode (a_xml_node);
        result = a_xml_node;

        if (a_emit_signal == TRUE) {

                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CUT], 0, parent_node,
                               a_xml_node);

                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }

        return result;
}


/**
 *Sets a new content a_content to a_node.
 *After setting the content, this method emits the following
 *signals: "content-changed", "node-changed" and "document-changed".
 *
 *@param a_content the new content to set to a_node.
 *@param a_this the current xml document.
 *@param a_node the node to set the new content to.
 *
 *@return the a_node with the new content or NULL if something bad happened. 
 */
xmlNode *
mlview_xml_document_set_node_content (MlViewXMLDocument * a_this,
                                      xmlNode * a_node,
                                      guchar * a_content,
                                      enum MlViewEncoding a_enc,
                                      gboolean a_emit_signal)
{
        guchar *utf8buf = NULL;
        xmlNode *result = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_this), NULL);
        g_return_val_if_fail (a_node != NULL, NULL);

        if (a_enc == ISO8859_1) {
                enum MlViewStatus status = MLVIEW_OK;

                status = mlview_utils_isolat1_str_to_utf8
                        (a_content, &utf8buf);

                if (status != MLVIEW_OK) {
                        mlview_utils_trace_debug
                                ("utf8 conversion failed");
                        result = NULL;
                        goto release_resources;
                }

        } else if (a_enc == UTF8) {
                utf8buf = a_content ;
        } else {
                mlview_utils_trace_info ("encoding not supported.") ;
                result = NULL ;
                goto release_resources ;
        }

        xmlNodeSetContent (a_node, utf8buf);

        if (a_emit_signal) {

                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[CONTENT_CHANGED], 0, a_node);

                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED], 0, a_node);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }

        result = a_node;

 release_resources:       
        if (utf8buf && utf8buf != a_content) {
                g_free (utf8buf);
                utf8buf = NULL;
        }

        return result;
}

/**
 *Selects a node of the XML document.
 *This actually emits a "node-selected" signal only
 *It's up to the listeners of that signal to actually
 *do what's needed to select the node.
 */
void 
mlview_xml_document_select_node (MlViewXMLDocument *a_this,
                                 xmlNode *a_node)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && PRIVATE (a_this)
                          && a_node) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[NODE_SELECTED], 0,
                       a_node) ;
}

/**
 *Sets or reset the attribute value.
 *@param a_this the current instance of #MlViewXMLDocument.
 *@param a_node the xml node to consider.
 *@param a_name the name of the attribute
 *@param a_value the value of the attribute.
 *@param a_emit_signal if TRUE, the function emits the signals
 *"node-attribute-added", "node-attribute-value-changed", and
 *"node-changed" signals in case of successfull completion.
 *@return the xmlAttr * of the attribute set.
 *upon successfull completion, an error code
 *otherwise.
 */
xmlAttr *
mlview_xml_document_set_attribute (MlViewXMLDocument *a_this,
                                   xmlNode *a_node,
                                   const xmlChar* a_name,
                                   const xmlChar *a_value,
                                   gboolean a_emit_signal)
{
        gboolean attr_added = TRUE ;
        xmlChar *value = NULL ;
        xmlAttr *attr = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this)
                              && a_node && a_name && a_value,
                              NULL) ;

        if ((value = xmlGetProp (a_node, a_name))) {
                xmlFree (value) ;
                value = NULL ;
                attr_added = FALSE ;
        }
        attr = xmlSetProp (a_node, a_name, a_value) ;
        if (a_emit_signal == TRUE) {
                if (attr_added == TRUE) {
                        g_signal_emit (G_OBJECT (a_this),
                                       gv_signals[NODE_ATTRIBUTE_ADDED], 0,
                                       attr) ;
                }
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_ATTRIBUTE_VALUE_CHANGED], 0,
                               attr) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED], 0, a_node);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return attr ;
}

/**
 *Removes an attribute from the a node of the tree.
 *@param a_this the current instance of #MlViewXMLDocument.
 *@param a_node the node that holds the attribute to remove.
 *@param a_name the name of attribute to remove.
 *@param a_emit_signal if set to TRUE, the function emits the
 *"node-attribute-removed" signal.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_xml_document_remove_attribute (MlViewXMLDocument *a_this,
                                      xmlNode *a_node, 
                                      const xmlChar *a_name,
                                      gboolean a_emit_signal)
{
        xmlAttr *attr = NULL ;
        xmlChar *attr_name = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this)
                              && a_node && a_name,
                              MLVIEW_BAD_PARAM_ERROR) ;
        attr = xmlHasProp (a_node, a_name) ;
        if (attr) {
                /*
                 *a_name may be allocated on attr
                 *before removing (freeing) attr
                 *let's keep a copy of a_name.
                 */
                attr_name = xmlStrdup (a_name) ;
                if (!attr_name) {
                        mlview_utils_trace_info ("xmlStrdup failed. "
                                                 "system may be out of memory.") ;
                        return MLVIEW_OUT_OF_MEMORY_ERROR ;
                }
                xmlRemoveProp (attr) ;
                attr = NULL ;
                if (a_emit_signal == TRUE) {
                        g_signal_emit 
                                (G_OBJECT (a_this),
                                 gv_signals[NODE_ATTRIBUTE_REMOVED], 0,
                                 a_node, 
                                 attr_name/*use our copy of a_name here.*/) ;
                        xmlFree (attr_name) ;
                        attr_name = NULL ;
                        g_signal_emit
                                (G_OBJECT (a_this),
                                 gv_signals[NODE_CHANGED], 0,
                                 a_node) ;
                        g_signal_emit
                                (G_OBJECT (a_this),
                                 gv_signals[DOCUMENT_CHANGED], 0);
                }
        }
        return MLVIEW_OK ;
}

/**
 *Makes the attribute list of a given node equals the
 *attribute list given in parameter.
 *@param a_doc the current instance of #MlViewXMLDocument
 *@param a_node the xml node to consider.
 *@param a_nv_pair_list the list of attributes to synch with.
 *@return MLVIEW_OK upon successfull completion, an error code otherwise.
 */
enum MlViewStatus 
mlview_xml_document_synch_attributes (MlViewXMLDocument *a_doc,
                                      xmlNode *a_node,
                                      GList *a_nv_pair_list)
{
        struct NameValuePair *cur_nv_pair = NULL ;
        xmlAttr *cur_attr = NULL , *attr_to_remove = NULL;
        if (!a_nv_pair_list) {
                /*remove all the attributes from the xml node*/
                for (attr_to_remove = NULL, cur_attr = a_node->properties ; 
                     cur_attr ;
                     attr_to_remove = cur_attr, cur_attr = cur_attr->next) {
                        if (attr_to_remove) {
                                mlview_xml_document_remove_attribute
                                        (a_doc, a_node,
                                         attr_to_remove->name,
                                         TRUE) ;
                                attr_to_remove = NULL ;
                        }
                }
                if (attr_to_remove) {
                        mlview_xml_document_remove_attribute
                                (a_doc, a_node,
                                 attr_to_remove->name,
                                 TRUE) ;
                        attr_to_remove = NULL ;
                }
        } else {
                GList *cur_item = NULL ;
                /*add attributes that are to be added.*/
                for (cur_item = a_nv_pair_list ; cur_item ;
                     cur_item = cur_item->next) {
                        xmlChar *attr_value = NULL ;

                        cur_nv_pair = cur_item->data ;
                        if (!cur_nv_pair || !cur_nv_pair->name 
                            || !cur_nv_pair->name->str)
                                continue ;
                        attr_value = xmlGetProp 
                                (a_node,  cur_nv_pair->name->str) ;
                        
                        if (!attr_value 
                            || strcmp (attr_value,
                                       cur_nv_pair->value->str)) {
                                mlview_xml_document_set_attribute 
                                        (a_doc,  a_node,
                                         cur_nv_pair->name->str,
                                         cur_nv_pair->value->str,
                                         TRUE) ;
                        }
                        if (attr_value) {
                                xmlFree (attr_value) ;
                                attr_value = NULL ;
                        }
                }
        }
        /*now, remove attributes that are to be removed*/
        attr_to_remove = NULL ;
        for (cur_attr = a_node->properties ; 
             cur_attr ;
             cur_attr = cur_attr->next) {
                if (attr_to_remove) {
                        mlview_xml_document_remove_attribute
                                (a_doc, a_node,
                                 attr_to_remove->name,
                                 TRUE) ;
                        attr_to_remove = NULL ;
                }
                if (!cur_attr->name)
                        continue ;
                cur_nv_pair = 
                        mlview_utils_name_value_pair_list_lookup
                        (a_nv_pair_list, cur_attr->name) ;
                if (!cur_nv_pair) {
                        attr_to_remove = cur_attr ;
                }
        }
        if (attr_to_remove) {
                mlview_xml_document_remove_attribute
                        (a_doc, a_node, attr_to_remove->name, TRUE) ;
                attr_to_remove = NULL ;
        }

        return MLVIEW_OK ;
}

/**
 *Sets the name of the attribute.
 *@param a_this the document to consider.
 *@param a_attr the attribute which name to set.
 *@param a_name the new name of the attribute
 *@param a_emit_signal if TRUE, emits the signals
 *"node-attribute-name-changed" and "node-changed".
 *@return MLVIEW_OK upon successfull completion, an error code
 *otherwise.
 */
enum MlViewStatus 
mlview_xml_document_set_attribute_name (MlViewXMLDocument *a_this,
                                        xmlAttr *a_attr,
                                        const xmlChar *a_name,
                                        gboolean a_emit_signal)
{
        xmlNode * node = NULL ;
        xmlNs *ns = NULL ;
        guchar *local_name = NULL ;

        g_return_val_if_fail (a_this
                              && a_attr
                              && a_attr->parent
                              && a_name,
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        node = a_attr->parent ;
        g_return_val_if_fail  (node->type == XML_ELEMENT_NODE,
                               MLVIEW_BAD_PARAM_ERROR) ;
        mlview_utils_parse_full_name (node, a_name,
                                      &ns, &local_name) ;
        if (ns) {
                xmlSetNs ((xmlNode*)a_attr, ns) ;
        }
        xmlNodeSetName ((xmlNode*)a_attr, a_name) ;
        if (a_emit_signal == TRUE) {
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[NODE_ATTRIBUTE_NAME_CHANGED], 0,
                         a_attr) ;
                g_signal_emit 
                        (G_OBJECT (a_this),
                         gv_signals[NODE_CHANGED], 0,
                         node) ;
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }

        if (local_name) {
                g_free (local_name) ;
                local_name = NULL ;
        }
        return MLVIEW_OK ;
}

/**
 *Creates a new namespace on a given xml node.
 *@param a_this the current instance of #MlViewXMLDocument
 *@param a_node the node that holds the namespace
 *@param a_uri the uri part of the namespace.
 *@param a_prefix the prefix of the namespace.
 *@param a_emit_signal if TRUE, emits the signals
 *"node-namespace-added" and "node-changed" upon successful
 *completion.
 *@return a pointer to the newly created uri upon successful
 *completion, a NULL pointer otherwise.
 */
xmlNs *
mlview_xml_document_create_ns (MlViewXMLDocument *a_this,
                               xmlNode *a_node, 
                               guchar *a_uri,
                               guchar *a_prefix,
                               gboolean a_emit_signal)
{
        xmlNs *result = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this)
                              && a_node, NULL) ;
        if (a_prefix && !strcmp (a_prefix, ""))
                a_prefix = NULL ;
        result = xmlNewNs (a_node, a_uri, a_prefix) ;
        g_return_val_if_fail (result, NULL) ;
        result->_private = a_node ;
        if (a_emit_signal == TRUE) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_NAMESPACE_ADDED], 0,
                               a_node, result) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED], 0,
                               a_node) ;
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return result ;
}


/**
 *Removes a namespace defined on the xml node given
 *in argument.
 *@param a_this the current instance of #MlViewXMLDocument
 *@param a_ns the namespace to remove.
 *@param a_node the xml node that hosts the namespace.
 *@param a_emit_signal if TRUE, emits the signals
 *"node-namespace-removed" and "node-changed".
 *@return MLVIEW_OK upon successfull completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_xml_document_remove_ns (MlViewXMLDocument *a_this,
                               xmlNs *a_ns,
                               xmlNode *a_node,
                               gboolean a_emit_signal)
{
        xmlNs *ns = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        ns = xmlUnlinkNsDef (a_node, a_ns) ;
        if (!ns) {
                /*the ns is not defined on this xml node*/
                return MLVIEW_OK ;
        }
        if (a_emit_signal == TRUE) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_NAMESPACE_REMOVED],
                               0, a_node, ns) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED],
                               0, a_node) ;
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        if (ns) {
                xmlFreeNs (ns) ;
                ns = NULL ;
        }
        return MLVIEW_OK ;
}

/**
 *Sets the prefix/href of a given
 *namespace declaration.
 *@param a_this the current instance of
 *#MlViewXMLDocument
 *@param a_node the xml node that holds the xmlNs to modify.
 *@param a_ns the namespace definition to modify.
 *@param a_uri the uri of the namespace.
 *@param a_prefix the prefix of the namespace.
 *@param a_emit_signal if set to true, emits
 *signals "node-namespace-changed" and "node-changed" 
 *after successfull completion.
 *@return MLVIEW_OK upon successfull completion, an
 *error code otherwise.
 */
enum MlViewStatus
mlview_xml_document_set_ns (MlViewXMLDocument *a_this,
                            xmlNode *a_node,
                            xmlNs *a_ns,
                            xmlChar *a_uri,
                            xmlChar *a_prefix,
                            gboolean a_emit_signal)
{
        xmlNs *cur_ns = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && a_node
                              && a_ns
                              && a_uri,
                              MLVIEW_BAD_PARAM_ERROR) ;

        /*are we sure a_ns is a Ns defined on node a_node ?*/
        for (cur_ns = a_node->nsDef ;
             cur_ns; cur_ns = cur_ns->next) {
                if (cur_ns == a_ns)
                        break ;
        }
        if (cur_ns != a_ns) {
                mlview_utils_trace_info 
                        ("a_ns is not a namespace defined on node a_node") ;
                return MLVIEW_BAD_PARAM_ERROR ;
        }
        if (a_ns->href) {
                xmlFree ((void*)a_ns->href) ;
        }
        a_ns->href = xmlStrdup (a_uri) ;
        if (a_ns->prefix) {
                xmlFree ((void*)a_ns->prefix) ;
        }
        a_ns->prefix = xmlStrdup (a_prefix) ;
        /*emit the required signals*/
        if (a_emit_signal == TRUE) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_NAMESPACE_CHANGED],0,
                               a_node, a_ns) ;
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED], 0, 
                               a_node) ;
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return MLVIEW_OK ;
}

/**
 *Sets the name of a_node to a_name.
 *@param a_this the current xml document.
 *@param a_node the node to set the name on.
 *@param a_name the new name to set.
 *@param a_enc the encoding of "a_name". Better set
 *it to UTF8 and make sure a_name is encoded in utf8
 *@param a_emit_signal if TRUE, emits the signals 
 *"name-changed" and "node-changed" upon succesful completion.
 *@return a_node with it new name or NULL if something bad happened. 
 */
xmlNode *
mlview_xml_document_set_node_name (MlViewXMLDocument *a_this,
                                   xmlNode * a_node,
                                   guchar * a_name,
                                   enum MlViewEncoding a_enc,
                                   gboolean a_emit_signal)
{
        guchar *utf8_name = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_this), NULL);
        g_return_val_if_fail (a_node != NULL, NULL);

        if (a_enc == ISO8859_1) {
                status = mlview_utils_isolat1_str_to_utf8
                        (a_name, &utf8_name);

        } else if (a_enc == UTF8) {
                utf8_name = a_name;
        }
        xmlNodeSetName (a_node, utf8_name);
        if (a_emit_signal) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NAME_CHANGED], 0, a_node);
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[NODE_CHANGED], 0, a_node);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        if (utf8_name && utf8_name != a_name) {
                g_free (utf8_name);
                utf8_name = NULL;
        }
        return a_node;
}

/**
 *Gets the node name of a given XML node.
 *@param a_node the XML node to get the name from.
 *@param a_enc the encoding in with the name should be converted
 *to. (Note that xmlNode's name is natively stored in utf8).
 *@param a_outbuf out parameter. A pointer to where the node name will
 *be stored. *a_outbut must be freed by the user.
 *@return MLVIEW_OK upon successfull completion, an error code otherwise.
 */
enum MlViewStatus
mlview_xml_document_node_get_name (xmlNode * a_node,
                                   enum MlViewEncoding a_enc,
                                   guchar ** a_outbuf)
{
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_node != NULL
                              && (a_node->type ==
                                  XML_ELEMENT_NODE
                                  || a_node->type ==
                                  XML_DOCUMENT_NODE
                                  || a_node->type == XML_PI_NODE)
                              && a_outbuf != NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        if (a_node->name == NULL) {
                *a_outbuf = 0;
                return MLVIEW_OK;
        }

        if (a_enc == ISO8859_1) {
                status = mlview_utils_utf8_str_to_isolat1 
                        ((guchar *) a_node->name, a_outbuf);
                if (status != MLVIEW_OK)
                        return status;
                status = MLVIEW_OK;
        } else if (a_enc == UTF8) {
                *a_outbuf = g_strdup (a_node->name);
                status = MLVIEW_OK;
        } else {
                status = MLVIEW_UNKNOWN_ENCODING_ERROR;
        }
        return status;
}

/**
 *Gets the iso latin 1 string length of the fully qualified name of an
 *xml node.
 *@param a_node the XML node to consider.
 *@param a_len out parameter. A pointer to where to store the length.
 *a_len must be allocated by the caller.
 *@return MLVIEW_OK upon successfull completion, an error code otherwise.
 */
enum MlViewStatus
mlview_xml_document_node_get_fqn_len_as_isolat1 (xmlNode *a_node,
                                                 gint * a_len)
{
        enum MlViewStatus status = MLVIEW_OK;
        gchar *fqn = NULL;
        gint len = 0;

        g_return_val_if_fail (a_node && a_len,
                              MLVIEW_BAD_PARAM_ERROR);

        if (a_node->ns
            && a_node->ns->prefix
            && mlview_utils_is_white_string
            (a_node->ns->prefix) == FALSE) {

                fqn = g_strconcat (a_node->ns->prefix,
                                   ":", a_node->name, NULL);
        } else {
                fqn = g_strdup (a_node->name);
        }

        if (fqn == NULL || *fqn == 0) {
                len = 0;
                status = MLVIEW_OK;

        } else {
                status = mlview_utils_utf8_str_len_as_isolat1
                        (fqn, &len);
        }

        if (status == MLVIEW_OK)
                *a_len = len;

        if (fqn) {
                g_free (fqn);
                fqn = NULL;
        }

        return status;
}

/**
 *Gets the fully qualified name of a given xml Node.
 *@param a_node the XML node to consider.
 *@param a_enc the encoding in which the name is to be translated.
 *@param a_outbuf out parameter. A pointer to where the name is gonna
 *bo store. *a_outbut is to be freed by caller.
 *@return MLVIEW_OK upon successfull completion, an error code otherwise.
 */
enum MlViewStatus
mlview_xml_document_node_get_fqn (xmlNode * a_node,
                                  enum MlViewEncoding a_enc,
                                  guchar ** a_outbuf)
{
        enum MlViewStatus status = MLVIEW_OK;
        gchar *fqn = NULL;

        g_return_val_if_fail (a_node != NULL
                              && a_node->type == XML_ELEMENT_NODE
                              && a_outbuf != NULL,
                              MLVIEW_BAD_PARAM_ERROR);
        if (a_node->name == NULL) {
                *a_outbuf = 0;
                return MLVIEW_OK;
        }
        if (a_node->ns
            && a_node->ns->prefix
            && mlview_utils_is_white_string
            (a_node->ns->prefix) == FALSE) {
                fqn = g_strconcat (a_node->ns->prefix,
                                   ":", a_node->name, NULL);
        } else {
                fqn = g_strdup (a_node->name);
        }
        if (a_enc == ISO8859_1) {
                status = mlview_utils_utf8_str_to_isolat1 (fqn,
                                                           a_outbuf);
                if (status != MLVIEW_OK) {
                        goto release_resources;
                }
        } else if (a_enc == UTF8) {
                *a_outbuf = g_strdup (fqn);
        } else {
                status = MLVIEW_UNKNOWN_ENCODING_ERROR;
                goto release_resources;
        }

 release_resources:
        if (fqn) {
                g_free (fqn);
                fqn = NULL;
        }

        return status;
}

/**
 *Gets the content of a given xml node.
 *@param a_node the xml node to consider.
 *@param a_enc the encoding in which the content is to be translated to.
 *@param a_outbuf out parameter. A pointer to where the content must be
 *copied to. *a_outbuf must be freeed by caller.
 *@return MLVIEW_OK upon successful completion.
 */
enum MlViewStatus
mlview_xml_document_node_get_content (xmlNode * a_node,
                                      enum MlViewEncoding a_enc,
                                      guchar ** a_outbuf)
{
        gchar *utf8_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_val_if_fail (a_node != NULL
                              && a_outbuf != NULL,
                              MLVIEW_BAD_PARAM_ERROR);

        utf8_content = xmlNodeGetContent (a_node);

        if (utf8_content == NULL) {
                *a_outbuf = 0;
                return MLVIEW_OK;
        }

        if (a_enc == ISO8859_1) {
                status = mlview_utils_utf8_str_to_isolat1
                        (utf8_content, a_outbuf);
                if (status != MLVIEW_OK) {
                        goto release_resources;
                }

                status = MLVIEW_OK;

        } else if (a_enc == UTF8) {
                *a_outbuf = g_strdup (utf8_content);
                status = MLVIEW_OK;
        } else {
                status = MLVIEW_UNKNOWN_ENCODING_ERROR;
                goto release_resources;
        }

      release_resources:
        if (utf8_content) {
                g_free (utf8_content);
                utf8_content = NULL;
        }

        return status;
}


/**
 *Inserts a_sibling_node as the previous sibling node of a_xml_node. 
 *@param a_this the current xml document. Must be not NULL.
 *@param a_sibling_node a the new sibling node to insert. Must be not NULL.
 *@param a_xml_node the current reference node. Must be not NULL.
 *@param a_subtree_required if TRUE, add a subtree to a_sibling_node to
 *make the a_sibling_node validate.
 *@param a_emit_signal if TRUE, emits the signals "prev-sibling-node-inserted"
 *and "document-changed" upon successful completion.
 *@return the newly insterted sibling (a_sibling_node) or 
 *NULL if something BAD happened.
 */
xmlNode *
mlview_xml_document_insert_prev_sibling_node (MlViewXMLDocument *a_this, 
                                              xmlNode * a_sibling_node,
                                              xmlNode *a_xml_node,
                                              gboolean a_subtree_required,
                                              gboolean a_emit_signal)
{
        xmlNode *result = NULL;
        struct MlViewAppSettings *settings = NULL ;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context,
                              NULL);
        g_return_val_if_fail (a_sibling_node != NULL, NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (settings, NULL) ;

        result = xmlAddPrevSibling (a_sibling_node, a_xml_node) ;
        g_return_val_if_fail (result, NULL) ;

        if (a_subtree_required == TRUE
            && settings->general.validation_is_on
            && result->type == XML_ELEMENT_NODE) {
                enum MLVIEW_PARSING_UTILS_STATUS status;
                status = mlview_parsing_utils_build_required_attributes_list 
                        (PRIVATE (a_this)->app_context, result);

                status = mlview_parsing_utils_build_required_children_tree 
                        (PRIVATE (a_this)->app_context, &result);
        }

        if (a_emit_signal == TRUE) {
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[PREV_SIBLING_NODE_INSERTED], 0,
                         a_sibling_node, result);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return result;
}


/**
 *Inserts a_sibling_node as  the next sibling of a_xml_node. 
 *
 *@param a_xml_node the reference xml node.
 *@param a_this the current xml document.
 *@param a_sibling_node the new sibling node to insert.
 *@param a_subtree_required if TRUE, add a subtree to a_sibling_node to
 *make it validate against the DTD associated to the current xml document.
 *@param a_emit_signal if TRUE emits signals 
 *"next-sibling-node-inserted" and "document-changed" 
 *upon successful completion.
 *@return the newly inserted next sibling node (a_sibling_node) or NULL
 *if an error arises.
 */
xmlNode *
mlview_xml_document_insert_next_sibling_node (MlViewXMLDocument *a_this,
                                              xmlNode * a_sibling_node,
                                              xmlNode * a_xml_node,
                                              gboolean a_subtree_required,
                                              gboolean a_emit_signal)
{
        xmlNode *result = NULL;
        struct MlViewAppSettings *settings = NULL ;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context,
                              NULL);
        g_return_val_if_fail (a_sibling_node != NULL, NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (settings, NULL) ;
        result = xmlAddNextSibling (a_sibling_node, a_xml_node);
        g_return_val_if_fail (result, NULL) ;

        if (a_subtree_required == TRUE
            && settings
            && settings->general.validation_is_on == TRUE
            && result 
            && result->type == XML_ELEMENT_NODE) {
                        enum MLVIEW_PARSING_UTILS_STATUS status;
                        status = mlview_parsing_utils_build_required_attributes_list
                                (PRIVATE  (a_this)->app_context, 
                                 result);
                        status = mlview_parsing_utils_build_required_children_tree 
                                (PRIVATE (a_this)->app_context, 
                                 &result);
        }
        if (a_emit_signal == TRUE) {
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[NEXT_SIBLING_NODE_INSERTED], 0,
                         a_sibling_node, result);
                g_signal_emit
                        (G_OBJECT (a_this),
                         gv_signals[DOCUMENT_CHANGED], 0);
        }
        return result;
}


/**
 *Pastes a node (coming from the internal clipboard) as a child of
 *a given xml node.
 *@param the current xml document to consider.
 *@param a_parent_node the parent node of the xml node to paste.
 *@param a_emit_signal if TRUE emits the appropriate signals upon successful
 *completion. Actually emits the same signals as mlview_xml_document_add_child_node().
 */
void
mlview_xml_document_paste_node_as_child (MlViewXMLDocument *a_this,
                                         xmlNode * a_parent_node,
                                         gboolean a_emit_signal)
{
        xmlNode *xml_node = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT
                          (a_this));
        g_return_if_fail (PRIVATE (a_this));
        g_return_if_fail (a_parent_node != NULL);

        xml_node =
                mlview_xml_document_get_node_from_clipboard
                (PRIVATE (a_this)->xml_doc);
        g_return_if_fail (xml_node != NULL);
        mlview_xml_document_add_child_node (a_this,
                                            a_parent_node,
                                            xml_node, FALSE,
                                            a_emit_signal);
}

/**
 *Pastes a node (coming from the internal clipboard) as a sibling of
 *a given xml node.
 *@param the current xml document to consider.
 *@param a_parent_node the parent node of the xml node to paste.
 *@param a_emit_signal if TRUE emits the appropriate signals upon successful
 *completion. Actually emits the same signals as 
 *mlview_xml_document_insert_next_sibling_node() or
 *mlview_xml_document_insert_prev_sibling_node() .
 */
void
mlview_xml_document_paste_node_as_sibling (MlViewXMLDocument *a_this,
                                           xmlNode *a_parent_node,
                                           xmlNode *a_sibling_node,
                                           gboolean a_previous,
                                           gboolean a_emit_signal)
{
        xmlNode *xml_node = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_parent_node != NULL);
        g_return_if_fail (a_sibling_node != NULL);

        xml_node =
                mlview_xml_document_get_node_from_clipboard
                (PRIVATE (a_this)->xml_doc);

        g_return_if_fail (xml_node != NULL);

        if (a_previous == FALSE) {

                mlview_xml_document_insert_next_sibling_node
                        (a_this, a_sibling_node,
                         xml_node, FALSE, a_emit_signal);
        } else {
                mlview_xml_document_insert_prev_sibling_node
                        (a_this , a_sibling_node,
                         xml_node, FALSE, a_emit_signal);
        }
}


/**
 *Tests if the document needs to be saved,
 *(has been modified since the last save) or not.
 *@param a_doc the xml document to consider.
 *@return TRUE if the document needs to be saved, FALSE otherwise.
 */
gboolean
mlview_xml_document_needs_saving (MlViewXMLDocument * a_doc)
{
        MlViewFileDescriptor *file_desc;
        gboolean is_modified;

        g_return_val_if_fail (a_doc != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (a_doc) != NULL, FALSE);

        file_desc =
                mlview_xml_document_get_file_descriptor (a_doc);
        if (file_desc == NULL)
                return TRUE;

        if (mlview_file_descriptor_is_modified
            (file_desc, &is_modified))
                return FALSE;
        return is_modified;
}


/**
 *Increments the reference count associated to this instance.
 *To decrement that reference count, please use
 *mlview_xml_document_unref().
 *
 *@param a_this the "this pointer".
 */
void
mlview_xml_document_ref (MlViewXMLDocument * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && PRIVATE (a_this));

        g_object_ref (G_OBJECT (a_this));
}


/**
 *Decrements the reference count associated to the current
 *instance of #MlViewXMLDocument. If the reference count reaches
 *zero, the current instance of #MlViewXMLDocument is destroyed.
 *
 *@param a_this in out parameter. Points to the "this" pointer.
 *If the ref count reached zero and the current instance has been
 *destroyed, *a_this is set to NULL.
 */
void
mlview_xml_document_unref (MlViewXMLDocument * a_this)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && PRIVATE (a_this));

        g_object_unref (G_OBJECT (a_this));
}


/**
 *Gets the file descriptor asscociated to the current instance
 *of #MlViewXMLDocument.
 *
 *@param a_this the "this pointer".
 *@return the file descriptor associated to the document.
 */
MlViewFileDescriptor *
mlview_xml_document_get_file_descriptor (MlViewXMLDocument *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->file_desc;
}

/**
 *Sets the file descriptor associated to the document.
 *@param a_this the this pointer.
 *@param a_file_desc the new file descriptor to set.
 */
void
mlview_xml_document_set_file_descriptor (MlViewXMLDocument *a_this,
                                         MlViewFileDescriptor *a_file_desc)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->file_desc = a_file_desc;
}

/**
 *Gets the native underlaying libxml2 document.
 *@param a_this the "this pointer".
 *@return the underlying native libxml2 document.
 */
xmlDocPtr
mlview_xml_document_get_xml_document (MlViewXMLDocument *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->xml_doc;
}


/**
 *Sets a new underlying native libxml2 document.
 *@param a_doc the "this pointer".
 *@param a_mem_doc a pointer to the new xmlDoc to set.
 */
void
mlview_xml_document_set_xml_document (MlViewXMLDocument * a_doc,
                                      xmlDocPtr a_mem_doc)
{
        g_return_if_fail (a_doc != NULL);
        g_return_if_fail (PRIVATE (a_doc) != NULL);

        PRIVATE (a_doc)->xml_doc = a_mem_doc;
}

/**
 *Searches a string in the xml document tree.
 *@param a_this the current instance of #MlViewXMLDocument.
 *@param a_conf the configuration of what to search and where.
 *@param a_from the node to start the search from. If NULL, 
 *start the search from the document root node.
 *@param a_found out parameter. The node found, or NULL if nothing
 *has been found.
 *@param a_emit_signal if set to TRUE, this function emits the
 *"searched-node-found" signal if the node has been found.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_xml_document_search (MlViewXMLDocument *a_this,
                             const struct SearchConfig *a_conf,
                             xmlNode *a_from,
                             xmlNode **a_found,
                             gboolean a_emit_signal)
{
        xmlNode *xml_node = NULL ;
        GList *start_list = NULL, *cur = NULL ;
        gboolean node_matches = FALSE ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_XML_DOCUMENT (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->xml_doc
                              && a_conf
                              && a_found, MLVIEW_BAD_PARAM_ERROR) ;
        if (!PRIVATE (a_this)->xml_doc->children)
                return MLVIEW_OK ;

        if ((PRIVATE (a_this)->last_modif_sequence
            < PRIVATE (a_this)->modif_sequence)
            || !PRIVATE (a_this)->nodes_list) {
                if (PRIVATE (a_this)->nodes_list) {
                        free_tree_list_cache (a_this) ;
                }
                status = build_tree_list_cache (a_this) ;
                PRIVATE (a_this)->last_modif_sequence =
                        PRIVATE (a_this)->modif_sequence ;
        }
        if (a_from) {
                g_return_val_if_fail
                        (a_from->doc == PRIVATE (a_this)->xml_doc,
                         MLVIEW_BAD_PARAM_ERROR) ;
                if (a_from->type == XML_DOCUMENT_NODE) {
                       start_list  = g_hash_table_lookup
                               (PRIVATE (a_this)->nodes_hash,
                                PRIVATE (a_this)->xml_doc->children);
                } else {
                        start_list = g_hash_table_lookup
                                (PRIVATE (a_this)->nodes_hash,
                                 a_from) ;
                }
        } else {
                start_list = g_hash_table_lookup
                        (PRIVATE (a_this)->nodes_hash,
                         PRIVATE (a_this)->xml_doc->children) ;
        }
        g_return_val_if_fail (start_list, MLVIEW_BAD_PARAM_ERROR) ;
        if (a_conf->downward == TRUE) {
                start_list = start_list->prev ;
        } else {
                start_list = start_list->next ;
        }
        cur = start_list ;
        while (cur) {
                xml_node = (xmlNode *)cur->data ;
                node_matches = search_in_node 
                        (xml_node, a_conf) ;
                if (node_matches == TRUE) {
                        *a_found = xml_node ;
                        goto end ;
                }
                if (a_conf->downward == TRUE) {
                        cur = cur->prev ;
                } else {
                        cur = cur->next ;
                }
        }

 end:
        if (a_emit_signal == TRUE && *a_found) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[SEARCHED_NODE_FOUND], 0,
                               *a_found) ;
        }
        return MLVIEW_OK ;
}
