/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

/**
 *TODO: apply the editing transaction scheme initiated with
 *the text node view to all the types of views of the node editor.
 *This is urgent !!!!!!
 */

/**
 *@file
 *The definition of the #MlViewNodeEditor widget.
 *
 *This widget edits a given xml node.
 *Some of it methods take an xmlNodePtr 
 *in parameter and allow the user to
 *interactively edit the xml node. 
 *It emits signal whenever the xml 
 *node being edited changes.
 *The user of this widget can thus 
 *catch and handle the emitted signals.
 *An xml node can have several types
 *(element node, text node, processing instruction node, comment node).
 *Each type of node is edited using what I
 *call a view. As they are several types of nodes, they are several
 *types of view; one for each type of node. 
 *A particular view is a widget that can 
 *edit a particular type of xml node
 *Each view is abstracted by a structure 
 *that holds the view widget.
 *The #MlViewElementWidget contains all the
 *possible views it can handle. 
 *Whenever an xml node is to be edited, the proper view
 *is then selected and used to edit the node.
 *The selection of the right view is done in 
 *the method mlview_node_editor_edit_xml_node(). 
 *
 */
#include <ctype.h>
#include <string.h>
#include <libxml/tree.h>
#include "mlview-node-editor.h"
#include "mlview-xml-document.h"
#include "mlview-attrs-editor.h"
#include "mlview-ns-editor.h"
#include "mlview-utils.h"

typedef struct _XMLElementNodeView XMLElementNodeView;
typedef struct _XMLCommentNodeView XMLCommentNodeView;
typedef struct _XMLCDataSectionNodeView XMLCDataSectionNodeView;
typedef struct _XMLTextNodeView XMLTextNodeView;
typedef struct _XMLPINodeView XMLPINodeView;
typedef struct _XMLDocNodeView XMLDocNodeView;

/*===========================================================
 *Some structure used in the private data part of MlViewElementEditor.
 *===========================================================*/

/**
 *This structure represents the element node view 
 */
struct _XMLElementNodeView {
        GtkVBox *vbox;
        /*a GtkEntry: contains the current element name */
        GtkEntry *name;
        /*the widget that does attributes edition*/
        MlViewAttrsEditor *attrs_editor ;
        /*The namespaces editor */
        MlViewNSEditor *ns_editor ;
        guint name_changed_handler_id;
        gboolean started_editing_transaction ;
        xmlNode *transaction_node ;
};


/**
 *The text node view
 */
struct _XMLTextNodeView {
        GtkVBox *vbox ;
        GtkTextView *widget ;
        gboolean started_editing_transaction ;
        xmlNode *transaction_node ;
} ;

/**
 *The comment node view. 
 *
 */
struct _XMLCommentNodeView {
        GtkVBox *vbox;
        GtkTextView *widget ;
        gboolean started_editing_transaction ;
        xmlNode *transaction_node ;
};

/**
 *The cdata section node view. 
 *
 */
struct _XMLCDataSectionNodeView {
        GtkVBox *vbox;
        GtkTextView *widget ;
        gboolean started_editing_transaction ;
        xmlNode *transaction_node ;
};

/**
 *The processing instruction node view
 */
struct _XMLPINodeView {
        GtkVBox *vbox;
        GtkEntry *name;
        GtkTextView *widget ;
        guint name_changed_handler_id;
};

/**
 *The document node view.
 */
struct _XMLDocNodeView {
        GtkVBox *vbox;

        /*the name of this document ... not mandatory */
        GtkEntry *name;

        guint name_changed_handler_id;

        /*wether this document is mandatory or not */
        GtkEntry *standalone;

        /*the xml version */
        GtkEntry *xml_version;

        /*the encoding of the on disk file */
        /*GtkEntry *external_encoding ; */
        GtkCombo *external_encoding;

        /*the external id of the external subset */
        GtkEntry *ext_subset_external_id;

        /*the system id of the external subset */
        GtkEntry *ext_subset_system_id;
};


/*===========================================================
 *The private part of the MlViewNodeEditor structure.
 *===========================================================*/
struct _MlViewNodeEditorPrivate {
        /*the left part of the MlViewNodeEditor widget */
        GtkVBox *left_margin;

        /*Right part of element editor. 
         *This widget shows a view appropriated
         *to the type of xml element
         *being edited. Let's say the xml node 
         *being edited is a text node, the view (widget) used to edit it
         *will be a GtkText. If the xml node 
         *being edited is an element node, the view used to edit it
         *will be more complex: it will be a widget 
         *that enable the user to edit the xml element name and the
         *it attributes ...
         */
        GtkNotebook *node_view;

        /*the current xml node being edited */
        xmlNode *curr_xml_node;

        MlViewXMLDocument *curr_xml_document;

        /*the view used to edit an xml element node */
        XMLElementNodeView *element_node_view;

        /*the view used to edit an xml text node */
        XMLTextNodeView *text_node_view;

        /*the view used to edit an xml comment node */
        XMLCommentNodeView *comment_node_view;

        /*the view used to edit an xml cdata section node */
        XMLCDataSectionNodeView *cdata_section_node_view;

        /*the view used to edit an xml pi node */
        XMLPINodeView *pi_node_view;

        /*the view used to edit an xml doc node */
        XMLDocNodeView *doc_node_view;

        MlViewAppContext *app_context;
        guint xml_node_type_changed_handler_id;
        guint left_right_percentage;
        gboolean dispose_has_run ;
};

/*
 *This enum defines the signals emited by MlViewNodeEditor.
 *It is used as the offset of the signal in 
 *the private p_table mlview_node_editor_signals.
 */
enum {
        ELEMENT_CHANGED,
        EDIT_STATE_CHANGED,
        ELEMENT_NAME_CHANGED,
        ELEMENT_ATTRIBUTE_CHANGED,
        ELEMENT_CONTENT_CHANGED,
        NUMBER_OF_SIGNALS
};

/*
 *This enum defines the notebook pages that hold
 *the possibles views of an xml node. It is used
 *as the offset of matching notebook pages.
 *Make sure that the order of the construction of 
 *the views (of the notebook pages)
 *(at the end of mlview_node_editor_init()) 
 *matches the order defined by this enum.
 */
enum {
        ELEMENT_NODE_VIEW_PAGE,
        TEXT_NODE_VIEW_PAGE,
        COMMENT_NODE_VIEW_PAGE,
        CDATA_SECTION_VIEW_PAGE,
        PI_NODE_VIEW_PAGE,
        DOC_NODE_VIEW_PAGE
};

/*the macro used to access the private part of the MlViewNodeEditor*/
#define PRIVATE(node_editor) ((node_editor)->priv)

#define ELEMENT_NODE_VIEW(node_editor) (PRIVATE(node_editor)->element_node_view) /*acess the element node view ... */
#define TEXT_NODE_VIEW(node_editor)  (PRIVATE(node_editor)->text_node_view)
#define COMMENT_NODE_VIEW(node_editor) (PRIVATE(node_editor)->comment_node_view)
#define CDATA_SECTION_NODE_VIEW(node_editor) (PRIVATE(node_editor)->cdata_section_node_view)
#define PI_NODE_VIEW(node_editor) (PRIVATE(node_editor)->pi_node_view)
#define DOC_NODE_VIEW(node_editor) (PRIVATE(node_editor)->doc_node_view)
#define CURRENT_XML_NODE(node_editor) (PRIVATE(node_editor)->curr_xml_node)
#define DEFAULT_LEFT_RIGHT_PERCENTAGE 20

/*private functions declarations*/
static void mlview_node_editor_build_xml_element_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_build_xml_text_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_build_xml_comment_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_build_xml_cdata_section_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_build_xml_pi_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_build_xml_doc_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_clear_xml_element_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_clear_xml_text_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_clear_xml_comment_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_clear_xml_cdata_section_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_clear_xml_pi_node_view (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_xml_element_node_view_edit_xml_node (MlViewNodeEditor * a_editor, 
                                                                    MlViewXMLDocument * a_xml_doc, 
                                                                    xmlNode * a_node);

static void mlview_node_editor_xml_element_node_view_commit_edit_trans (MlViewNodeEditor *a_this) ;

static void mlview_node_editor_xml_text_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                                 MlViewXMLDocument * a_xml_doc, xmlNode * a_node);

static void mlview_node_editor_xml_text_node_view_commit_edit_trans (MlViewNodeEditor *a_this) ;

static void mlview_node_editor_xml_comment_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                                    MlViewXMLDocument * a_xml_doc, 
                                                                    xmlNode * a_node);

static void mlview_node_editor_xml_comment_node_view_commit_edit_trans (MlViewNodeEditor *a_this) ;

static void mlview_node_editor_xml_cdata_section_node_view_commit_edit_trans (MlViewNodeEditor *a_this) ;

static void mlview_node_editor_xml_cdata_section_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                                          MlViewXMLDocument * a_xml_doc, 
                                                                          xmlNode * a_node);

static void
mlview_node_editor_xml_pi_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                   MlViewXMLDocument * a_xml_doc, 
                                                   xmlNode * a_node);

static void mlview_node_editor_xml_doc_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                                MlViewXMLDocument * a_xml_doc,
                                                                xmlNode * a_node);


static void mlview_node_editor_class_init (MlViewNodeEditorClass * a_klass);

static void mlview_node_editor_init (MlViewNodeEditor * a_node_editor);

static void mlview_node_editor_construct (MlViewNodeEditor * a_node_editor,
                                          MlViewAppContext * a_app_context);

static void mlview_node_editor_commit_editing_transaction (MlViewNodeEditor *a_this) ;

static gboolean mlview_node_editor_name_changed_cb (GtkWidget * a_entry,
                                                GdkEventFocus *a_event,
                                                MlViewNodeEditor * a_editor) ;

static void mlview_node_editor_attribute_changed_cb (MlViewAttrsEditor * a_attr_editor,
                                                     gpointer a_editor);

static gboolean mlview_node_editor_content_changed_cb (GtkTextView * a_view,
                                                       GdkEventFocus *a_event,
                                                       MlViewNodeEditor *a_editor) ;

static void external_encoding_changed_cb (GtkEditable * a_encoding,
                                          MlViewNodeEditor * a_editor);

static void text_inserted_in_text_node_view_cb (GtkTextBuffer *a_text_buffer,
                                                GtkTextIter *a_iter,
                                                gchar *a_text,
                                                gint a_len,
                                                gpointer a_user_data) ;

static void text_inserted_in_element_name_cb (GtkEditable *a_editable,
                                              MlViewNodeEditor *a_this) ;

static void  text_inserted_in_comment_node_view_cb (GtkTextBuffer *a_text_buffer,
                                                    GtkTextIter *a_iter,
                                                    gchar *a_text,
                                                    gint a_len,
                                                    gpointer a_user_data) ;

static void  text_inserted_in_cdata_node_view_cb (GtkTextBuffer *a_text_buffer,
                                                  GtkTextIter *a_iter,
                                                  gchar *a_text,
                                                  gint a_len,
                                                  gpointer a_user_data) ;

static void mlview_node_editor_dispose (GObject *a_this) ;

static void mlview_node_editor_finalize (GObject *a_this) ;



/*private static data*/
static GtkHPanedClass *gv_parent_class;
static guint gv_mlview_node_editor_signals[NUMBER_OF_SIGNALS] ={ 0 };



/*================================================================
 *private helper functions
 *===============================================================*/

static void
mlview_node_editor_build_xml_element_node_view (MlViewNodeEditor* a_this)
{
        XMLElementNodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *table = NULL, *label = NULL;
        GtkWidget *frame = NULL, *vbox = NULL;

        const guchar *attributes_names_title =
                N_("Attribute names");
        const guchar *attributes_values_title =
                N_("Attribute values");
        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL) {
                PRIVATE (a_this) =
                        g_try_malloc 
                        (sizeof (MlViewNodeEditorPrivate));
                if (!PRIVATE (a_this)) {
                        mlview_utils_trace_info ("g_try_malloc failed") ;
                        return ;
                }
                memset (PRIVATE (a_this), 0, 
                        sizeof (MlViewNodeEditorPrivate)) ;
        }
        if (ELEMENT_NODE_VIEW (a_this) == NULL) {
                ELEMENT_NODE_VIEW (a_this) =
                        g_try_malloc (sizeof (XMLElementNodeView));
                if (!ELEMENT_NODE_VIEW (a_this)) {
                        mlview_utils_trace_info ("g_try_malloc failed") ;
                        return ;
                }
                memset (ELEMENT_NODE_VIEW (a_this), 0,
                        sizeof (XMLElementNodeView)) ;
        } else {
                if (ELEMENT_NODE_VIEW (a_this)->vbox !=
                    NULL) {
                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (TEXT_NODE_VIEW
                                  (a_this)->vbox));
                }
        }
        view = ELEMENT_NODE_VIEW (a_this);
        private_data = PRIVATE (a_this);
        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));
        frame = gtk_frame_new (_("Element node"));
        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);
        vbox = gtk_vbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (frame), vbox);
        label = gtk_label_new (_("Element name"));
        view->name = GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_text (GTK_ENTRY (view->name), "");
        table = gtk_table_new (1, 2, TRUE);
        gtk_box_pack_start (GTK_BOX (vbox),
                            table, FALSE, FALSE, 0);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   label, 0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->name),
                                   1, 2, 0, 1);
        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "focus-out-event",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_this);
        g_signal_connect (G_OBJECT (view->name),
                          "changed",
                          G_CALLBACK (text_inserted_in_element_name_cb),
                          a_this) ;
        
        /*init the attributes and the namespace edition areas */
        table = gtk_table_new (1, 2, TRUE);
        frame = gtk_frame_new (_("attributes edition"));
        view->attrs_editor = MLVIEW_ATTRS_EDITOR 
                (mlview_attrs_editor_new 
                 ((guchar*)attributes_names_title,
                  (guchar*)attributes_values_title,
                  PRIVATE (a_this)->app_context)) ;

        g_signal_connect (G_OBJECT (view->attrs_editor),
                          "attribute-changed",
                          G_CALLBACK
                          (mlview_node_editor_attribute_changed_cb),
                          a_this);

        gtk_container_add (GTK_CONTAINER (frame),
                           GTK_WIDGET (view->attrs_editor)) ;
        gtk_table_attach_defaults (GTK_TABLE (table), frame,
                                   0, 1, 0, 1);
        view->ns_editor = MLVIEW_NS_EDITOR 
                (mlview_ns_editor_new 
                 (PRIVATE (a_this)->app_context,
                  PRIVATE (a_this)->curr_xml_document)) ;
        frame = gtk_frame_new (_("namespaces edition"));

        gtk_container_add (GTK_CONTAINER (frame),
                           GTK_WIDGET (view->ns_editor));
        gtk_table_attach_defaults (GTK_TABLE (table), frame,
                                   1, 2, 0, 1);

        /*FIXME: add signals on the namespaces editor */
        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *Creates the text node view. This view will be used to edit text elements. 
 */
static void
mlview_node_editor_build_xml_text_node_view (MlViewNodeEditor * a_this) 
{
        XMLTextNodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;
        GtkTextBuffer *text_buffer = NULL ;

        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL)
                PRIVATE (a_this) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_this);

        if (TEXT_NODE_VIEW (a_this) == NULL)
                TEXT_NODE_VIEW (a_this) =
                        g_malloc0 (sizeof (XMLElementNodeView));
        else if (TEXT_NODE_VIEW (a_this)->vbox != NULL) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (TEXT_NODE_VIEW (a_this)->vbox));
        }

        view = TEXT_NODE_VIEW (a_this);
        frame = gtk_frame_new (_("Text Node"));

        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));
        gtk_box_pack_start
                (GTK_BOX (view->vbox), frame, TRUE, TRUE, 0);

        gtk_widget_show (frame);
        view->widget = GTK_TEXT_VIEW (gtk_text_view_new ()) ;
        scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (view->widget)) ;
        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);
        g_signal_connect (G_OBJECT (view->widget),
                          "focus-out-event",
                          G_CALLBACK 
                          (mlview_node_editor_content_changed_cb),
                          a_this) ;
        text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view->widget)) ;
        g_return_if_fail (text_buffer) ;
        g_signal_connect (G_OBJECT (text_buffer),
                          "insert-text",
                          G_CALLBACK (text_inserted_in_text_node_view_cb),
                          a_this) ;
        gtk_widget_show (GTK_WIDGET (view->widget));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *creates the xml comment node view. 
 *This view will be used by 
 *MlViewElement editor to
 *edit xml comment node. 
 */
static void
mlview_node_editor_build_xml_comment_node_view (MlViewNodeEditor * a_this) 
{
        MlViewNodeEditorPrivate *private_data = NULL;
        XMLCommentNodeView *view = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;
        GtkTextBuffer *text_buffer = NULL ;

        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL)
                PRIVATE (a_this) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_this);

        if (COMMENT_NODE_VIEW (a_this) == NULL) {
                COMMENT_NODE_VIEW (a_this) =
                        g_malloc0 (sizeof (XMLCommentNodeView));
        } else {
                if (COMMENT_NODE_VIEW (a_this)->vbox !=
                    NULL) {
                        gtk_widget_destroy (GTK_WIDGET
                                            (COMMENT_NODE_VIEW
                                             (a_this)->
                                             vbox));
                }
        }

        view = COMMENT_NODE_VIEW (a_this);
        view->vbox = GTK_VBOX (gtk_vbox_new (TRUE, 0));
        view->widget = GTK_TEXT_VIEW (gtk_text_view_new ()) ;
        frame = gtk_frame_new (_("Comment node"));

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            GTK_WIDGET (frame), TRUE, TRUE, 0);

        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (view->widget));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_connect (G_OBJECT (view->widget),
                          "focus-out-event",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_this);
        text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view->widget)) ;
        g_return_if_fail (text_buffer) ;
        g_signal_connect (G_OBJECT (text_buffer),
                          "insert-text",
                          G_CALLBACK (text_inserted_in_comment_node_view_cb),
                          a_this) ;
        gtk_widget_show_all (GTK_WIDGET (view->vbox));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Creates the xml cdata section view. 
 *This view will be used by 
 *MlViewNodeEditor to edit cdata section node.
 *
 */
static void
mlview_node_editor_build_xml_cdata_section_node_view (MlViewNodeEditor * a_this) 
{
        MlViewNodeEditorPrivate *private_data = NULL;
        XMLCDataSectionNodeView *view = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL;
        GtkTextBuffer *text_buffer = NULL ;

        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL)
                PRIVATE (a_this) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

        private_data = PRIVATE (a_this);
        if (CDATA_SECTION_NODE_VIEW (a_this) == NULL) {
                CDATA_SECTION_NODE_VIEW (a_this) =
                        g_malloc0 (sizeof
                                   (XMLCDataSectionNodeView));
        } else {
                if (CDATA_SECTION_NODE_VIEW (a_this)->
                    vbox != NULL) {
                        gtk_widget_destroy (GTK_WIDGET
                                            (CDATA_SECTION_NODE_VIEW
                                             (a_this)->
                                             vbox));
                }
        }

        view = CDATA_SECTION_NODE_VIEW (a_this);
        view->vbox = GTK_VBOX (gtk_vbox_new (TRUE, 0));        
        view->widget = GTK_TEXT_VIEW (gtk_text_view_new ());
        frame = gtk_frame_new (_("CDATA Section node"));

        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);
        scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scrolled_window),
                           GTK_WIDGET (view->widget));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_connect (G_OBJECT (view->widget),
                          "focus-out-event",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_this);
        g_signal_connect (G_OBJECT (text_buffer),
                          "insert-text",
                          G_CALLBACK (text_inserted_in_cdata_node_view_cb),
                          a_this) ;
        gtk_widget_show_all (GTK_WIDGET (view->vbox));
        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}

/**
 *builds the xml processing instruction 
 *node view included in the current instance of a_this. 
 *
 */
static void 
mlview_node_editor_build_xml_pi_node_view (MlViewNodeEditor * a_this) 
{
        XMLPINodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *scrolled_window = NULL,
                *table = NULL,
                *label = NULL;
        GtkTextBuffer *text_buffer = NULL ;

        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL)
                PRIVATE (a_this) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));
        private_data = PRIVATE (a_this);

        if (PI_NODE_VIEW (a_this) == NULL) {
                PI_NODE_VIEW (a_this) =
                        g_malloc0 (sizeof (XMLPINodeView));
        } else {
                if (PI_NODE_VIEW (a_this)->vbox != NULL) {
                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (PI_NODE_VIEW (a_this)->
                                  vbox));
                }
        }

        view = PI_NODE_VIEW (a_this);
        frame = gtk_frame_new (_("PI Node"));
        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));

        label = gtk_label_new (_("PI node name"));
        view->name = GTK_ENTRY (gtk_entry_new ());
        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE
                                   (table),
                                   GTK_WIDGET (view->name), 1, 2,
                                   0, 1);

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            table, TRUE, TRUE, 0);

        gtk_box_pack_start (GTK_BOX (view->vbox),
                            frame, TRUE, TRUE, 0);

        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "focus-out-event",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_this) ;

        view->widget = GTK_TEXT_VIEW (gtk_text_view_new ());

        scrolled_window = gtk_scrolled_window_new (NULL, NULL);

        gtk_container_add
                (GTK_CONTAINER (scrolled_window),
                 GTK_WIDGET (view->widget));

        gtk_container_add (GTK_CONTAINER (frame),
                           scrolled_window);
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_connect (G_OBJECT (view->widget),
                          "focus-out-event",
                          G_CALLBACK
                          (mlview_node_editor_content_changed_cb),
                          a_this);

        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Creates the view that can edit an xml doc node. 
 *
 */
static void
mlview_node_editor_build_xml_doc_node_view (MlViewNodeEditor * a_this) 
{
        XMLDocNodeView *view = NULL;
        MlViewNodeEditorPrivate *private_data = NULL;
        GtkWidget *frame = NULL,
                *table = NULL,
                *label = NULL,
                *vbox = NULL;
        GList *available_encodings = NULL;

        g_return_if_fail (a_this != NULL);

        if (PRIVATE (a_this) == NULL) {
                PRIVATE (a_this) = g_malloc0
                        (sizeof (MlViewNodeEditorPrivate));
        }

        private_data = PRIVATE (a_this);

        if (DOC_NODE_VIEW (a_this) == NULL) {
                DOC_NODE_VIEW (a_this) =
                        g_malloc0 (sizeof (XMLDocNodeView));
        } else {
                if (DOC_NODE_VIEW (a_this)->vbox != NULL) {
                        gtk_widget_destroy
                                (GTK_WIDGET
                                 (DOC_NODE_VIEW (a_this)->
                                  vbox));
                }
        }

        view = DOC_NODE_VIEW (a_this);

        /*Set the decoration ... */
        frame = gtk_frame_new (_("document node"));

        view->vbox = GTK_VBOX (gtk_vbox_new (FALSE, 0));

        gtk_box_pack_start (GTK_BOX (view->vbox), frame,
                            TRUE, TRUE, 0);

        vbox = gtk_vbox_new (FALSE, 0);

        gtk_container_add (GTK_CONTAINER (frame), vbox);

        /*The "name of the document" entry */
        view->name = GTK_ENTRY (gtk_entry_new ());
        view->name_changed_handler_id =
                g_signal_connect (G_OBJECT (view->name),
                                  "focus-out-event",
                                  G_CALLBACK
                                  (mlview_node_editor_name_changed_cb),
                                  a_this);

        label = gtk_label_new (_("document name or uri:"));

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->name),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*the "standalone" combo ... */
        label = gtk_label_new (_("is document standalone:"));
        view->standalone = GTK_ENTRY (gtk_entry_new ());
        gtk_entry_set_editable (view->standalone, FALSE);

        table = gtk_table_new (1, 2, TRUE);
        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);
        gtk_table_attach_defaults (GTK_TABLE
                                   (table),
                                   GTK_WIDGET (view->standalone),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox), table,
                            TRUE, TRUE, 0);

        /*The "xml version" entry ... */
        label = gtk_label_new (_("xml version:"));

        view->xml_version = GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->xml_version, FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               xml_version), 1,
                                   2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox), table,
                            TRUE, TRUE, 0);

        /*The "xml encoding" entry ... */
        label = gtk_label_new (_("xml file encoding:"));

        view->external_encoding = GTK_COMBO (gtk_combo_new ());
        g_return_if_fail (view->external_encoding);

        gtk_entry_set_editable (GTK_ENTRY
                                (view->external_encoding->entry),
                                FALSE);

        available_encodings =
                mlview_utils_get_available_encodings ();

        g_return_if_fail (available_encodings);

        gtk_combo_set_popdown_strings (view->external_encoding,
                                       available_encodings);
        g_signal_connect (G_OBJECT
                          (view->external_encoding->entry),
                          "changed",
                          G_CALLBACK
                          (external_encoding_changed_cb),
                          a_this);

        table = gtk_table_new (1, 2, TRUE);

        g_return_if_fail (table);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               external_encoding),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*The external subset external is */
        label = gtk_label_new (_
                               ("external id of the external subset:"));

        view->ext_subset_external_id =
                GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->ext_subset_external_id,
                                FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               ext_subset_external_id),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*The external subset external id entry ... */
        label = gtk_label_new (_
                               ("system id of the external subset:"));

        view->ext_subset_system_id =
                GTK_ENTRY (gtk_entry_new ());

        gtk_entry_set_editable (view->ext_subset_system_id,
                                FALSE);

        table = gtk_table_new (1, 2, TRUE);

        gtk_table_attach_defaults (GTK_TABLE (table), label,
                                   0, 1, 0, 1);

        gtk_table_attach_defaults (GTK_TABLE (table),
                                   GTK_WIDGET (view->
                                               ext_subset_system_id),
                                   1, 2, 0, 1);

        gtk_box_pack_start (GTK_BOX (vbox),
                            table, TRUE, TRUE, 0);

        /*Show everything and attach it to the views */
        gtk_widget_show_all (GTK_WIDGET (view->vbox));

        gtk_notebook_append_page (private_data->node_view,
                                  GTK_WIDGET (view->vbox), NULL);
}


/**
 *Visually clears the xml element node view. 
 *
 */
static void
mlview_node_editor_clear_xml_element_node_view (MlViewNodeEditor * a_this) 
{
        XMLElementNodeView *view = NULL;

        g_return_if_fail (a_this != NULL);
        view = ELEMENT_NODE_VIEW (a_this);
        g_return_if_fail (view != NULL);

        g_signal_handlers_block_by_func 
                (G_OBJECT (view->name),
                 mlview_node_editor_name_changed_cb,
                 a_this) ;
        gtk_entry_set_text (view->name, "");
        mlview_attrs_editor_clear (view->attrs_editor) ;

        g_signal_handlers_unblock_by_func
                (G_OBJECT (view->name),
                 mlview_node_editor_name_changed_cb,
                 a_this) ;
}


/**
 *Clears visually (only) the text node view of the MlViewNodeEditor widget. 
 */
static void
mlview_node_editor_clear_xml_text_node_view (MlViewNodeEditor * a_this) 
{
        XMLTextNodeView *view = NULL ;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;

        g_return_if_fail (a_this != NULL);

        view = TEXT_NODE_VIEW (a_this);
        g_return_if_fail (view != NULL);

        /*
         *FIXME: remember to block the
         *"text-changed" signal (if used)
         *before clearing the text node view.
         */
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
}


/**
 *Clears visually (only) the comment node view of MlViewNodeEditor. 
 *
 */
static void
mlview_node_editor_clear_xml_comment_node_view (MlViewNodeEditor* a_this)
{
        XMLCommentNodeView *view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;

        g_return_if_fail (a_this != NULL);

        view = COMMENT_NODE_VIEW (a_this);

        g_return_if_fail (view != NULL);
                
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
}

static void
mlview_node_editor_clear_xml_cdata_section_node_view (MlViewNodeEditor * a_this) 
{
        XMLCDataSectionNodeView *view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        view = CDATA_SECTION_NODE_VIEW (a_this);

        g_return_if_fail (view != NULL);

        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                            &iter2, 0) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
}

/**
 *Visually clears the current pi node view 
 *included in the current instance of MlViewNodeEditor. 
 *
 */
static void
mlview_node_editor_clear_xml_pi_node_view (MlViewNodeEditor *a_this)
{
        XMLPINodeView *view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        view = PI_NODE_VIEW (a_this);

        g_return_if_fail (view != NULL);

        /*clear the pi node content editor */
        text_buffer = gtk_text_view_get_buffer (view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                                &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                                &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;

        /*clear the name */
        gtk_editable_delete_text (GTK_EDITABLE (view->name), 0,
                                  -1);
        g_signal_handlers_unblock_by_func
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
}


/**
 *Edits the xml node 
 *(which is supposed to be an xml element node) 
 *using the xml element view contained in 
 *the current instance of MlViewNodeEditor.
 *Note that if the xml node is not 
 *an element node, this function does nothing.
 *@param a_editor the "this" pointer of the current
 *instance of #MlViewNodeEditor.
 *@a_xml_doc the instance of #MlViewXMLDocument currently
 *being edited.
 *@a_node the node of a_xml_doc to edit.
 */
static void
mlview_node_editor_xml_element_node_view_edit_xml_node (MlViewNodeEditor * a_editor,
                                                        MlViewXMLDocument * a_xml_doc, 
                                                        xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLElementNodeView *editor_view = NULL;
        guchar *full_name = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        editor_private = PRIVATE (a_editor);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;
        if (a_node->type != XML_ELEMENT_NODE)
                return;
        editor_view = ELEMENT_NODE_VIEW (a_editor);
        g_return_if_fail (editor_view != NULL);
        status = mlview_xml_document_node_get_fqn 
                (a_node, UTF8, &full_name);
        g_return_if_fail (status == MLVIEW_OK);

        /*set the name of the element */
        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->name_changed_handler_id);
        g_signal_handlers_block_by_func (G_OBJECT (editor_view->name),
                                         G_CALLBACK (text_inserted_in_element_name_cb),
                                         a_editor) ;
        gtk_entry_set_text (editor_view->name, "");
        gtk_entry_set_text (GTK_ENTRY (editor_view->name),
                            full_name);
        if (full_name != NULL) {
                g_free (full_name);
                full_name = NULL;
        }
        g_signal_handlers_unblock_by_func (G_OBJECT (editor_view->name),
                                           G_CALLBACK (text_inserted_in_element_name_cb),
                                           a_editor) ;
        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->name_changed_handler_id);

        /*edit the attributes */
        mlview_attrs_editor_clear (editor_view->attrs_editor) ;
        mlview_attrs_editor_edit_xml_attributes
                (editor_view->attrs_editor, 
                 a_xml_doc, a_node) ;
        /*edit the namespaces */
        mlview_ns_editor_clear (editor_view->ns_editor) ;
        mlview_ns_editor_edit_node_visible_namespaces 
                (editor_view->ns_editor, a_node) ;
        gtk_notebook_set_page (editor_private->node_view,
                               ELEMENT_NODE_VIEW_PAGE);
}


static void
mlview_node_editor_xml_text_node_view_edit_xml_node (MlViewNodeEditor * a_this,
                                                     MlViewXMLDocument * a_xml_doc, xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLTextNodeView *editor_view = NULL;
        guchar *utf8_content = NULL ;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_this
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && a_xml_doc
                          && MLVIEW_IS_XML_DOCUMENT (a_xml_doc)
                          && a_node
                          && PRIVATE (a_this));

        if (a_node->type != XML_TEXT_NODE)
                return;

        editor_view = TEXT_NODE_VIEW (a_this);
        g_return_if_fail (editor_view != NULL);

        editor_private = PRIVATE (a_this);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        status = mlview_xml_document_node_get_content (a_node, UTF8,
                                                       &utf8_content);
        g_return_if_fail (status == MLVIEW_OK);
        
        text_buffer = gtk_text_view_get_buffer (editor_view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        g_signal_handlers_block_by_func (G_OBJECT (text_buffer),
                                         G_CALLBACK (text_inserted_in_text_node_view_cb),
                                         a_this) ;                
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;

        if (utf8_content) {
                gtk_text_buffer_get_iter_at_offset 
                        (text_buffer, &iter1, 0) ;
                len = strlen (utf8_content);
                gtk_text_buffer_insert (text_buffer, &iter1, 
                                        utf8_content, len) ;
        }
        gtk_notebook_set_page (editor_private->node_view,
                               TEXT_NODE_VIEW_PAGE);

        g_signal_handlers_unblock_by_func (G_OBJECT (text_buffer),
                                           G_CALLBACK (text_inserted_in_text_node_view_cb),
                                           a_this) ;
        g_signal_handlers_unblock_by_func 
                (text_buffer, 
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
/* release_resources:*/        
        if (utf8_content) {
                g_free (utf8_content);
                utf8_content = NULL;
        }
}


/**
 *static void mlview_node_editor_xml_comment_node_view_edit_xml_node:
 *Edits an xml comment node using a specific xml comment node view. If the xml node given in argument
 *is not an xml comment node, does nothing. 
 *
 */
static void
mlview_node_editor_xml_comment_node_view_edit_xml_node (MlViewNodeEditor * a_this,
                                                        MlViewXMLDocument * a_xml_doc, 
                                                        xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLCommentNodeView *editor_view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *utf8_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_this));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (a_node->type != XML_COMMENT_NODE)
                return;

        editor_view = COMMENT_NODE_VIEW (a_this);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_this);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*content = xmlNodeGetContent (a_node) ; */

        status = mlview_xml_document_node_get_content (a_node, UTF8,
                                                       &utf8_content);

        g_return_if_fail (status == MLVIEW_OK);
        
        text_buffer = gtk_text_view_get_buffer (editor_view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        g_signal_handlers_block_by_func (G_OBJECT (text_buffer),
                                         G_CALLBACK (text_inserted_in_comment_node_view_cb),
                                         a_this) ;

        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;

        if (utf8_content) {
                len = strlen (utf8_content);
                gtk_text_buffer_get_iter_at_offset 
                        (text_buffer, &iter1, 0) ;
                gtk_text_buffer_insert (text_buffer, &iter1,
                                        utf8_content, len) ;
        }
        gtk_notebook_set_page (editor_private->node_view,
                               COMMENT_NODE_VIEW_PAGE);
        g_signal_handlers_unblock_by_func (G_OBJECT (text_buffer),
                                           G_CALLBACK (text_inserted_in_comment_node_view_cb),
                                           a_this) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
/* release_resources:*/

        if (utf8_content) {
                g_free (utf8_content);
                utf8_content = NULL;
        }
}

/**
 *Edits a xml cdata section node using the a specific xml cdata section view.
 *If the xml node given in argument is node a cdata section node, does nothing. 
 *
 */
static void
mlview_node_editor_xml_cdata_section_node_view_edit_xml_node (MlViewNodeEditor * a_this,
                                                              MlViewXMLDocument * a_xml_doc, 
                                                              xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLCDataSectionNodeView *editor_view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *utf8_content = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_this
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && a_xml_doc
                          && MLVIEW_IS_XML_DOCUMENT (a_xml_doc)
                          && a_node
                          && PRIVATE (a_this));

        editor_view = CDATA_SECTION_NODE_VIEW (a_this);
        g_return_if_fail (editor_view);

        editor_private = PRIVATE (a_this);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*content = xmlNodeGetContent (a_node) ; */
        status = mlview_xml_document_node_get_content (a_node, UTF8,
                                                       &utf8_content);
        g_return_if_fail (status == MLVIEW_OK);
        
        text_buffer = gtk_text_view_get_buffer 
                (editor_view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        g_signal_handlers_block_by_func (G_OBJECT (text_buffer),
                                         G_CALLBACK (text_inserted_in_cdata_node_view_cb),
                                         a_this) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;
        if (utf8_content) {
                len = strlen (utf8_content);
                gtk_text_buffer_get_iter_at_offset (text_buffer, 
                                            &iter1, 0) ;
                gtk_text_buffer_insert (text_buffer, &iter1,
                                        utf8_content, len) ;
        }
        gtk_notebook_set_page (editor_private->node_view,
                               CDATA_SECTION_VIEW_PAGE);
        g_signal_handlers_unblock_by_func (G_OBJECT (text_buffer),
                                           G_CALLBACK (text_inserted_in_cdata_node_view_cb),
                                           a_this) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
/* release_resources:*/
        if (utf8_content) {
                g_free (utf8_content);
                utf8_content = NULL;
        }
}


/**
 *Edits the xml pi node given in argument 
 *using a specific xml pi node view. If he xml node
 *given in argument is not a pi node, does nothing. 
 *
 */
static void
mlview_node_editor_xml_pi_node_view_edit_xml_node (MlViewNodeEditor * a_this,
                                                   MlViewXMLDocument * a_xml_doc, 
                                                   xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLPINodeView *editor_view = NULL;
        GtkTextBuffer *text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *content = NULL, *full_name = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gint len = 0;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_this));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        editor_view = PI_NODE_VIEW (a_this);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_this);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        status = mlview_xml_document_node_get_name (a_node, UTF8,
                                                    &full_name);
        g_return_if_fail (status == MLVIEW_OK);

        /*get the name of the element. */
        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->name_changed_handler_id);

        gtk_entry_set_text (editor_view->name, "");
        gtk_entry_set_text (GTK_ENTRY (editor_view->name),
                            full_name);

        if (full_name)
                g_free (full_name);

        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->
                                  name_changed_handler_id);

        /*edit the content */
        /*content = xmlNodeGetContent (a_node) ; */

        status = mlview_xml_document_node_get_content (a_node, UTF8,
                                                       &content);
        g_return_if_fail (status == MLVIEW_OK);
        
        text_buffer = gtk_text_view_get_buffer (editor_view->widget) ;
        g_signal_handlers_block_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                            &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset (text_buffer,
                                            &iter2, -1) ;
        gtk_text_buffer_delete (text_buffer, &iter1, &iter2) ;

        if (content) {
                len = strlen (content);
                gtk_text_buffer_get_iter_at_offset (text_buffer,
                                                    &iter1, 0) ;
                gtk_text_buffer_insert (text_buffer, &iter1, 
                                        content, len) ;
        }
        gtk_notebook_set_page (editor_private->node_view,
                               PI_NODE_VIEW_PAGE) ;
        g_signal_handlers_unblock_by_func 
                (G_OBJECT (text_buffer),
                 G_CALLBACK (mlview_node_editor_content_changed_cb),
                 a_this) ;
        if (content) {
                g_free (content);
                content = NULL;
        }
}


/**
 *Edits the xml document given in argument using a specific view. If the xml node given in argument is not
 *a xml document node, does nothing. 
 *
 */
static void
mlview_node_editor_xml_doc_node_view_edit_xml_node (MlViewNodeEditor * a_this,
                                                    MlViewXMLDocument * a_xml_doc, 
                                                    xmlNode * a_node) 
{
        MlViewNodeEditorPrivate *editor_private = NULL;
        XMLDocNodeView *editor_view = NULL;
        guchar *name = NULL;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_this));
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (a_node->type != XML_DOCUMENT_NODE)
                return;

        editor_view = DOC_NODE_VIEW (a_this);
        g_return_if_fail (editor_view != NULL);
        editor_private = PRIVATE (a_this);
        editor_private->curr_xml_node = a_node;
        editor_private->curr_xml_document = a_xml_doc;

        /*get the name of the element. */
        status = mlview_xml_document_node_get_name (a_node,
                                                    UTF8,
                                                    &name);
        g_return_if_fail (status == MLVIEW_OK);

        g_signal_handler_block (G_OBJECT (editor_view->name),
                                editor_view->
                                name_changed_handler_id);
        gtk_entry_set_text (editor_view->name, "");

        if (name != NULL)
                gtk_entry_set_text (GTK_ENTRY
                                    (editor_view->name), name);
        else
                gtk_entry_set_text (GTK_ENTRY
                                    (editor_view->name), "");

        if (name) {
                g_free (name);
                name = NULL;
        }

        g_signal_handler_unblock (G_OBJECT (editor_view->name),
                                  editor_view->
                                  name_changed_handler_id);

        /*get the standalone */
        if (((xmlDocPtr) a_node)->extSubset) {

                gtk_entry_set_text (editor_view->standalone,
                                    _("NO"));
                ((xmlDocPtr) a_node)->standalone = FALSE;

        } else {

                gtk_entry_set_text (editor_view->standalone,
                                    _("YES"));
                ((xmlDocPtr) a_node)->standalone = TRUE;

        }

        /*get the xml version */
        if (((xmlDocPtr) a_node)->version)
                gtk_entry_set_text (editor_view->xml_version,
                                    ((xmlDocPtr) a_node)->
                                    version);
        else
                gtk_entry_set_text (editor_view->xml_version,
                                    "1.0");

        /*get the external encoding */


        if (((xmlDocPtr) a_node)->encoding) {
                enum MlViewStatus status = MLVIEW_OK;

                /*
                 *If the encoding is supported
                 *by mlview and if it is not
                 *in the default list of supported encodings
                 *add it to that list.
                 */
                status = mlview_utils_add_supported_encoding
                        ((guchar *) ((xmlDocPtr) a_node)->
                         encoding);

                if (status ==
                    MLVIEW_ENCODING_NOT_SUPPORTED_ERROR) {

                        /*
                         *If the document has no encoding specified,
                         *set the default document encoding to UTF-8.
                         */
                        ((xmlDocPtr) a_node)->encoding =
                                xmlMemStrdup ("UTF-8");

                }

        } else {

                ((xmlDocPtr) a_node)->encoding =
                        xmlMemStrdup ("UTF-8");
        }

        g_return_if_fail (editor_view->external_encoding
                          && editor_view->external_encoding->
                          entry);

        gtk_signal_handler_block_by_data
                (GTK_OBJECT
                 (editor_view->external_encoding->entry),
                 a_this);

        gtk_entry_set_text (GTK_ENTRY
                            (editor_view->external_encoding->
                             entry),
                            ((xmlDocPtr) a_node)->encoding);

        gtk_signal_handler_unblock_by_data
                (GTK_OBJECT
                 (editor_view->external_encoding->entry),
                 a_this);

        /*get stuffs about the external subset */
        if (((xmlDocPtr) a_node)->extSubset
            && ((xmlDocPtr) a_node)->extSubset->ExternalID) {
                gtk_entry_set_text (editor_view->
                                    ext_subset_external_id,
                                    ((xmlDocPtr) a_node)->
                                    extSubset->ExternalID);
        }

        if (((xmlDocPtr) a_node)->extSubset
            && ((xmlDocPtr) a_node)->extSubset->SystemID) {
                gtk_entry_set_text (editor_view->
                                    ext_subset_system_id,
                                    ((xmlDocPtr) a_node)->
                                    extSubset->SystemID);
        }

        gtk_notebook_set_page (editor_private->node_view,
                               DOC_NODE_VIEW_PAGE);
}

static gboolean
mlview_node_editor_has_an_editing_transaction_started (MlViewNodeEditor *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_NODE_EDITOR (a_this),
                              FALSE) ;

        if (PRIVATE (a_this)->element_node_view->started_editing_transaction == TRUE
            || PRIVATE (a_this)->text_node_view->started_editing_transaction == TRUE
            || PRIVATE (a_this)->comment_node_view->started_editing_transaction == TRUE
            || PRIVATE (a_this)->cdata_section_node_view->started_editing_transaction == TRUE) {
                return TRUE ;
        }
        return FALSE ;
}

static void
mlview_node_editor_xml_text_node_view_commit_edit_trans (MlViewNodeEditor *a_this)
{
        GtkTextBuffer * text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *content = NULL;
        xmlNode *node = NULL ;

        g_return_if_fail (a_this
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)->curr_xml_node
                          && PRIVATE (a_this)->text_node_view
                          && PRIVATE (a_this)->text_node_view->transaction_node) ;

        if (PRIVATE (a_this)->text_node_view
            && PRIVATE (a_this)->text_node_view->started_editing_transaction == FALSE)
                return ;
        PRIVATE (a_this)->text_node_view->started_editing_transaction = FALSE ;
        node = PRIVATE (a_this)->text_node_view->transaction_node ;
        PRIVATE (a_this)->text_node_view->transaction_node = NULL ;
        text_buffer = gtk_text_view_get_buffer (PRIVATE (a_this)->text_node_view->widget) ;
        g_return_if_fail (text_buffer) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter2, -1) ;
        content =
                gtk_text_buffer_get_text (text_buffer, &iter1,
                                          &iter2, FALSE) ;
        mlview_xml_document_set_node_content
                (PRIVATE (a_this)->curr_xml_document,
                 node,
                 content, UTF8, TRUE);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CONTENT_CHANGED], 0, content);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CHANGED], 0, content);
        g_free (content);        
}

static void
mlview_node_editor_xml_element_node_view_commit_edit_trans (MlViewNodeEditor *a_this)
{
        guchar *full_name = NULL,
                *local_name = NULL;
        xmlNs *ns = NULL;
        xmlNode *node = NULL ;
        GtkEntry *entry = NULL ;

        g_return_if_fail (a_this && MLVIEW_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->curr_xml_document) ;

        g_return_if_fail (PRIVATE (a_this)->element_node_view) ;
        entry = PRIVATE (a_this)->element_node_view->name ;
        if (PRIVATE (a_this)->element_node_view->started_editing_transaction == FALSE)
                return ;
        g_return_if_fail (PRIVATE (a_this)->element_node_view->transaction_node) ;
        g_return_if_fail (PRIVATE (a_this)->element_node_view->transaction_node->type == XML_ELEMENT_NODE
                          || PRIVATE (a_this)->element_node_view->transaction_node->type == XML_PI_NODE) ;

        node = PRIVATE (a_this)->element_node_view->transaction_node ;

        full_name = (guchar *)
                gtk_entry_get_text (GTK_ENTRY (entry));
        if (full_name) {
                mlview_utils_parse_full_name
                        (node, full_name, &ns,
                         &local_name);
        }
        if (ns != NULL) {
                xmlSetNs (node, ns);
        } else {
                node->ns = NULL;
        }
        PRIVATE (a_this)->element_node_view->started_editing_transaction = FALSE ;
        PRIVATE (a_this)->element_node_view->transaction_node = NULL ;
        mlview_xml_document_set_node_name
                (PRIVATE (a_this)->curr_xml_document, node,
                 local_name, UTF8, TRUE);
        gtk_signal_emit (GTK_OBJECT (a_this),
                         gv_mlview_node_editor_signals
                         [ELEMENT_CHANGED]);
        if (local_name) {
                g_free (local_name);
                local_name = NULL;
        }

}

static void 
mlview_node_editor_xml_comment_node_view_commit_edit_trans (MlViewNodeEditor *a_this)
{
        GtkTextBuffer * text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *content = NULL;
        xmlNode *node = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)->curr_xml_node
                          && PRIVATE (a_this)->comment_node_view
                          && PRIVATE (a_this)->comment_node_view->transaction_node) ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)->curr_xml_node
                          && PRIVATE (a_this)->comment_node_view
                          && PRIVATE (a_this)->comment_node_view->transaction_node) ;

        if (PRIVATE (a_this)->comment_node_view
            && PRIVATE (a_this)->comment_node_view->started_editing_transaction == FALSE)
                return ;
        PRIVATE (a_this)->comment_node_view->started_editing_transaction = FALSE ;
        node = PRIVATE (a_this)->comment_node_view->transaction_node ;
        PRIVATE (a_this)->comment_node_view->transaction_node = NULL ;
        text_buffer = gtk_text_view_get_buffer (PRIVATE (a_this)->comment_node_view->widget) ;
        g_return_if_fail (text_buffer) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter2, -1) ;
        content =
                gtk_text_buffer_get_text (text_buffer, &iter1,
                                          &iter2, FALSE) ;
        mlview_xml_document_set_node_content
                (PRIVATE (a_this)->curr_xml_document,
                 node,
                 content, UTF8, TRUE);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CONTENT_CHANGED], 0, content);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CHANGED], 0, content);
        g_free (content);        
        
}

static void 
mlview_node_editor_xml_cdata_section_node_view_commit_edit_trans (MlViewNodeEditor *a_this)
{
        GtkTextBuffer * text_buffer = NULL ;
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        guchar *content = NULL;
        xmlNode *node = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)->curr_xml_node
                          && PRIVATE (a_this)->cdata_section_node_view
                          && PRIVATE (a_this)->cdata_section_node_view->transaction_node) ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)->curr_xml_node
                          && PRIVATE (a_this)->cdata_section_node_view
                          && PRIVATE (a_this)->cdata_section_node_view->transaction_node) ;

        if (PRIVATE (a_this)->cdata_section_node_view
            && PRIVATE (a_this)->cdata_section_node_view->started_editing_transaction == FALSE)
                return ;
        PRIVATE (a_this)->cdata_section_node_view->started_editing_transaction = FALSE ;
        node = PRIVATE (a_this)->cdata_section_node_view->transaction_node ;
        PRIVATE (a_this)->cdata_section_node_view->transaction_node = NULL ;
        text_buffer = gtk_text_view_get_buffer (PRIVATE (a_this)->cdata_section_node_view->widget) ;
        g_return_if_fail (text_buffer) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter2, -1) ;
        content =
                gtk_text_buffer_get_text (text_buffer, &iter1,
                                          &iter2, FALSE) ;
        mlview_xml_document_set_node_content
                (PRIVATE (a_this)->curr_xml_document,
                 node,
                 content, UTF8, TRUE);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CONTENT_CHANGED], 0, content);

        g_signal_emit (G_OBJECT (a_this),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CHANGED], 0, content);
        g_free (content);        
}
static void 
mlview_node_editor_commit_editing_transaction (MlViewNodeEditor *a_this)
{
        g_return_if_fail (a_this && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)) ;
        if (PRIVATE (a_this)->text_node_view
            && PRIVATE (a_this)->text_node_view->started_editing_transaction == TRUE) {
                mlview_node_editor_xml_text_node_view_commit_edit_trans 
                        (a_this) ;
        }
        if (PRIVATE (a_this)->element_node_view
            && PRIVATE (a_this)->element_node_view->started_editing_transaction == TRUE) {
                mlview_node_editor_xml_element_node_view_commit_edit_trans (a_this) ;
        }
        if (PRIVATE (a_this)->comment_node_view
            && PRIVATE (a_this)->comment_node_view->started_editing_transaction == TRUE) {
                mlview_node_editor_xml_comment_node_view_commit_edit_trans (a_this) ;
        }
        if (PRIVATE (a_this)->cdata_section_node_view
            && PRIVATE (a_this)->cdata_section_node_view->started_editing_transaction == TRUE) {
                mlview_node_editor_xml_cdata_section_node_view_commit_edit_trans (a_this) ;
        }
}

/***********************************************
 *private signal handlers
 *FIXME: recode everything below to 
 *support the private field in MlViewNodeEditor 
 *and the different views ...
 ************************************************/


/**
 *Called back when the element name entry changes.
 *Applies the change to the attached xml node and emits the
 *signal "element_name_changed" followed by the signal
 *"element_changed" if
 *and only if the element is an xml element node. 
 *
 */
static gboolean
mlview_node_editor_name_changed_cb (GtkWidget * a_entry,
                                    GdkEventFocus *a_event,
                                    MlViewNodeEditor * a_editor)
{

/*TODO: handle the PI node case !!*/
        g_return_val_if_fail (a_entry
                              && GTK_IS_ENTRY (a_entry)
                              && a_event
                              && a_editor 
                              && MLVIEW_IS_NODE_EDITOR (a_editor)
                              && PRIVATE (a_editor)
                              && PRIVATE (a_editor)->curr_xml_node
                              && ELEMENT_NODE_VIEW (a_editor),
                              FALSE) ;
        if (ELEMENT_NODE_VIEW (a_editor)->started_editing_transaction == FALSE)
                return FALSE ;
        mlview_node_editor_xml_element_node_view_commit_edit_trans (a_editor) ;

/*
        if (PRIVATE (a_editor)->curr_xml_node->type ==
            XML_ELEMENT_NODE
            || PRIVATE (a_editor)->curr_xml_node->type ==
            XML_PI_NODE) {
                guchar *full_name = NULL,
                        *local_name = NULL;
                xmlNs *ns = NULL;

                full_name = (guchar *)
                        gtk_entry_get_text (GTK_ENTRY (a_entry));
                if (full_name) {
                        mlview_utils_parse_full_name
                                (PRIVATE (a_editor)->
                                 curr_xml_node, full_name, &ns,
                                 &local_name);
                }
                if (ns != NULL) {
                        xmlSetNs (PRIVATE (a_editor)->
                                  curr_xml_node, ns);
                } else {
                        PRIVATE (a_editor)->curr_xml_node->ns =
                                NULL;
                }
                mlview_xml_document_set_node_name
                        (PRIVATE (a_editor)->curr_xml_document,
                         PRIVATE (a_editor)->curr_xml_node,
                         local_name, ISO8859_1, TRUE);
                gtk_signal_emit (GTK_OBJECT (a_editor),
                                 gv_mlview_node_editor_signals
                                 [ELEMENT_CHANGED]);
                if (local_name) {
                        g_free (local_name);
                        local_name = NULL;
                }
        }
*/
        return FALSE ;
}

static void
mlview_node_editor_attribute_changed_cb (MlViewAttrsEditor *a_attrs_editor,
                                         gpointer a_this)
{
        MlViewNodeEditor *node_editor = NULL;
        g_return_if_fail (a_attrs_editor 
                          && MLVIEW_IS_ATTRS_EDITOR (a_attrs_editor));
        g_return_if_fail (a_this);

        node_editor = MLVIEW_NODE_EDITOR (a_this);
        gtk_signal_emit (GTK_OBJECT (node_editor),
                         gv_mlview_node_editor_signals
                         [ELEMENT_CHANGED]);
}

static void
xml_doc_node_unselected_cb (MlViewXMLDocument *a_doc,
                             xmlNode *a_node,
                             gpointer a_user_data)
{
/*
        GtkTextIter iter1 = {0}, iter2 = {0} ;        
        GtkTextBuffer *text_buffer = NULL ;
        gchar *content = NULL ;
*/
        MlViewNodeEditor *editor = NULL ;

        if (a_node->type != XML_TEXT_NODE)
                return ;
        g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc) 
                          && a_node && a_user_data 
                          && MLVIEW_IS_NODE_EDITOR (a_user_data)) ;

        editor = MLVIEW_NODE_EDITOR (a_user_data) ;
        g_return_if_fail (editor && PRIVATE (editor)
                          && PRIVATE (editor)->text_node_view
                          && PRIVATE (editor)->text_node_view->widget) ;
        
        if (PRIVATE (editor)->text_node_view->started_editing_transaction == TRUE) {
                mlview_node_editor_xml_text_node_view_commit_edit_trans (editor) ;                
        }

        /*
        text_buffer = gtk_text_view_get_buffer 
                (GTK_TEXT_VIEW 
                 (PRIVATE (editor)->text_node_view->widget)) ;
        g_return_if_fail (text_buffer) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter2, -1) ;
        content =
                gtk_text_buffer_get_text (text_buffer, &iter1,
                                          &iter2, FALSE) ;        
        mlview_xml_document_set_node_content
                (a_doc,a_node,
                 content, UTF8, TRUE);
        if (content) {
                g_free (content) ;
                content = NULL ;
        }
        */
}

/**
 *A callback called when the user changes the content of the element.
 *Emits a signal "element_content_changed" of the MlViewNodeEditor class. 
 *
 */
static gboolean
mlview_node_editor_content_changed_cb (GtkTextView * a_view,
                                       GdkEventFocus *a_event,
                                       MlViewNodeEditor *a_editor)
{        
/*
        GtkTextIter iter1 = {0}, iter2 = {0} ;
        GtkTextBuffer *text_buffer = NULL ;
        guchar *content = NULL;
*/
        g_return_val_if_fail (GTK_IS_TEXT_VIEW (a_view)
                              && a_editor
                              && MLVIEW_IS_NODE_EDITOR (a_editor)
                              && PRIVATE (a_editor),
                              FALSE);

        if (mlview_node_editor_has_an_editing_transaction_started (a_editor) == FALSE) {
                return FALSE ;
        }
        mlview_node_editor_commit_editing_transaction (a_editor) ;

        /*
        text_buffer = gtk_text_view_get_buffer (a_view) ;
        g_return_val_if_fail (text_buffer, FALSE) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter1, 0) ;
        gtk_text_buffer_get_iter_at_offset 
                (text_buffer, &iter2, -1) ;
        content =
                gtk_text_buffer_get_text (text_buffer, &iter1,
                                          &iter2, FALSE) ;
        mlview_xml_document_set_node_content
                (PRIVATE (a_editor)->curr_xml_document,
                 PRIVATE (a_editor)->curr_xml_node,
                 content, UTF8, TRUE);

        g_signal_emit (G_OBJECT (a_editor),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CONTENT_CHANGED], 0, content);

        g_signal_emit (G_OBJECT (a_editor),
                       gv_mlview_node_editor_signals
                       [ELEMENT_CHANGED], 0, content);

        g_free (content);
        */
        return FALSE ;
}

static void
external_encoding_changed_cb (GtkEditable * a_encoding,
                              MlViewNodeEditor * a_editor)
{
        guchar *encoding = NULL;

        g_return_if_fail (a_encoding
                          && GTK_IS_EDITABLE (a_encoding));
        g_return_if_fail (a_editor
                          && MLVIEW_IS_NODE_EDITOR (a_editor)
                          && PRIVATE (a_editor));

        if (PRIVATE (a_editor)->curr_xml_node == NULL
            || PRIVATE (a_editor)->curr_xml_node->doc == NULL)
                return;

        encoding = gtk_editable_get_chars (a_encoding, 0, -1);

        if (encoding == NULL)
                return;

        if (mlview_utils_is_encoding_supported (encoding) ==
            TRUE) {

                if (PRIVATE (a_editor)->curr_xml_node->doc->
                    encoding) {
                        xmlFree ((xmlChar *) PRIVATE (a_editor)->
                                 curr_xml_node->doc->encoding);
                }

                PRIVATE (a_editor)->curr_xml_node->doc->
                        encoding = xmlMemStrdup (encoding);
        } else {
                /*
                 *Libxml2 does not support the new encoding so ...
                 *why not setting the encoding to UTF-8 as default ?
                 */
        }

        if (encoding) {
                g_free (encoding);
                encoding = NULL;
        }
}

static void
xml_doc_node_changed_cb (MlViewXMLDocument *a_this,
                         xmlNode *a_node,
                         MlViewNodeEditor *a_editor)
{
        g_return_if_fail (a_this 
                          && MLVIEW_IS_XML_DOCUMENT (a_this)
                          && a_node 
                          && MLVIEW_IS_NODE_EDITOR (a_editor)
                          && PRIVATE (a_editor)) ;

        
        if (PRIVATE (a_editor)->curr_xml_node != a_node)
                return ;
        mlview_node_editor_edit_xml_node (a_editor, a_this, a_node) ;
}

static void
xml_doc_node_selected_cb (MlViewXMLDocument *a_doc,
                          xmlNode *a_node,
                          MlViewNodeEditor *a_editor)
{
        g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
                          && a_node && a_editor
                          && MLVIEW_IS_NODE_EDITOR (a_editor)) ;

        mlview_node_editor_edit_xml_node (a_editor, a_doc,
                                          a_node) ;
}

static void 
text_inserted_in_text_node_view_cb (GtkTextBuffer *a_text_buffer,
                                    GtkTextIter *a_iter,
                                    gchar *a_text,
                                    gint a_len,
                                    gpointer a_user_data)
{
        MlViewNodeEditor *thiz = NULL ;

        g_return_if_fail (a_text_buffer
                          && GTK_IS_TEXT_BUFFER (a_text_buffer)
                          && a_iter
                          && a_user_data) ;

        thiz = MLVIEW_NODE_EDITOR (a_user_data) ;
        g_return_if_fail (thiz && PRIVATE (thiz)) ;

        if (a_text && a_len) {
                if (PRIVATE (thiz)->text_node_view) {
                        if (PRIVATE (thiz)->text_node_view->started_editing_transaction == FALSE) {
                                PRIVATE (thiz)->text_node_view->transaction_node = PRIVATE (thiz)->curr_xml_node ;
                                PRIVATE (thiz)->text_node_view->started_editing_transaction = TRUE ;
                        } else {
                                g_return_if_fail (PRIVATE (thiz)->text_node_view->transaction_node 
                                                  == PRIVATE (thiz)->curr_xml_node) ;
                        }
                }
        }
}

static void
text_inserted_in_element_name_cb (GtkEditable *a_editable,
                                  MlViewNodeEditor *a_this)
{
        g_return_if_fail (a_editable && GTK_IS_ENTRY (a_editable)) ;
        g_return_if_fail (a_this && MLVIEW_IS_NODE_EDITOR (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->element_node_view
                          && PRIVATE (a_this)->curr_xml_node) ;

        if (PRIVATE (a_this)->element_node_view->started_editing_transaction == TRUE)
                return ;
        
        PRIVATE (a_this)->element_node_view->started_editing_transaction = TRUE ;
        PRIVATE (a_this)->element_node_view->transaction_node = 
                PRIVATE (a_this)->curr_xml_node ;
}

static void 
text_inserted_in_comment_node_view_cb (GtkTextBuffer *a_text_buffer,
                                       GtkTextIter *a_iter,
                                       gchar *a_text,
                                       gint a_len,
                                       gpointer a_user_data)
{
        MlViewNodeEditor *thiz = NULL ;

        g_return_if_fail (a_text_buffer
                          && GTK_IS_TEXT_BUFFER (a_text_buffer)
                          && a_iter
                          && a_user_data) ;

        thiz = MLVIEW_NODE_EDITOR (a_user_data) ;
        g_return_if_fail (thiz && PRIVATE (thiz)) ;

        if (a_text && a_len) {
                if (PRIVATE (thiz)->text_node_view) {
                        if (PRIVATE (thiz)->comment_node_view->started_editing_transaction == FALSE) {
                                PRIVATE (thiz)->comment_node_view->transaction_node = PRIVATE (thiz)->curr_xml_node ;
                                PRIVATE (thiz)->comment_node_view->started_editing_transaction = TRUE ;
                        } else {
                                g_return_if_fail (PRIVATE (thiz)->comment_node_view->transaction_node 
                                                  == PRIVATE (thiz)->curr_xml_node) ;
                        }
                }
        }
}

static void 
text_inserted_in_cdata_node_view_cb (GtkTextBuffer *a_text_buffer,
                                     GtkTextIter *a_iter,
                                     gchar *a_text,
                                     gint a_len,
                                     gpointer a_user_data)
{
        MlViewNodeEditor *thiz = NULL ;

        g_return_if_fail (a_text_buffer
                          && GTK_IS_TEXT_BUFFER (a_text_buffer)
                          && a_iter
                          && a_user_data) ;

        thiz = MLVIEW_NODE_EDITOR (a_user_data) ;
        g_return_if_fail (thiz && PRIVATE (thiz)) ;

        if (a_text && a_len) {
                if (PRIVATE (thiz)->cdata_section_node_view) {
                        if (PRIVATE (thiz)->cdata_section_node_view->started_editing_transaction == FALSE) {
                                PRIVATE (thiz)->cdata_section_node_view->transaction_node = PRIVATE (thiz)->curr_xml_node ;
                                PRIVATE (thiz)->cdata_section_node_view->started_editing_transaction = TRUE ;
                        } else {
                                g_return_if_fail (PRIVATE (thiz)->cdata_section_node_view->transaction_node 
                                                  == PRIVATE (thiz)->curr_xml_node) ;
                        }
                }
        }
}

/*=========================================
 *private gtk object framework methods
 *=========================================*/

/**
 *Classical dynamic type allocator.
 */
guint
mlview_node_editor_get_type (void)
{

        static guint mlview_node_editor_type = 0;

        if (!mlview_node_editor_type) {
                static const GTypeInfo
                        mlview_node_editor_type_info = {
                        sizeof (MlViewNodeEditorClass),
                        NULL,   /* base_init */
                        NULL,   /* base_finalize */
                        (GClassInitFunc)
                                mlview_node_editor_class_init,
                        NULL,   /* class_finalize */
                        NULL,   /* class_data */
                        sizeof (MlViewNodeEditor),
                        0,
                        (GInstanceInitFunc)
                        mlview_node_editor_init
                };

                mlview_node_editor_type =
                        g_type_register_static (GTK_TYPE_HPANED,
                                                "MlViewNodeEditor",
                                                &mlview_node_editor_type_info,
                                                0);
        }
        return mlview_node_editor_type;
}

/**
 *classical class struct initialyzer. This code overrides the
 *destroy method of the GtkObjectClass with mlview_node_editor_destroy. 
 *It defines
 *also the signal which offsets are 
 *strored in the static gv_mlview_node_editor_signals. 
 *
 */
static void
mlview_node_editor_class_init (MlViewNodeEditorClass * a_klass)
{
        GObjectClass *gobject_class = NULL ;
        gv_parent_class =
                gtk_type_class (gtk_hpaned_get_type ());

        gobject_class = G_OBJECT_CLASS (a_klass) ;
        g_return_if_fail (gobject_class) ;

        /*overload the destroy method of object */
        gobject_class->dispose = mlview_node_editor_dispose ;
        gobject_class->finalize = mlview_node_editor_finalize ;

        /*signal definitions */
        gv_mlview_node_editor_signals[ELEMENT_CHANGED] =
                g_signal_new ("element-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_NAME_CHANGED] =
                g_signal_new ("element-name-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_name_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_ATTRIBUTE_CHANGED]
                =
                g_signal_new ("element-attribute-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_attribute_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[ELEMENT_CONTENT_CHANGED] =
                g_signal_new ("element-content-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               element_content_changed), NULL,
                              NULL, gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        gv_mlview_node_editor_signals[EDIT_STATE_CHANGED] =
                g_signal_new ("edit-state-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewNodeEditorClass,
                               edit_state_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              G_TYPE_NONE, 0, NULL);

        /* gtk_object_class_add_signals (object_class, 
           gv_mlview_node_editor_signals, 
           NUMBER_OF_SIGNALS) ; */

        /*no default signal handler for signal "document-changed" */
        a_klass->element_changed = NULL;
        a_klass->element_name_changed = NULL;
        a_klass->element_attribute_changed = NULL;
        a_klass->element_content_changed = NULL;
        a_klass->edit_state_changed = NULL;
}

static void
mlview_node_editor_dispose (GObject *a_this)
{
        MlViewNodeEditor *editor = NULL ;
        
        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)) ;
        editor = MLVIEW_NODE_EDITOR (a_this) ;
        g_return_if_fail (PRIVATE (editor)) ;
        if (PRIVATE (editor)->dispose_has_run == TRUE) {
                return ;
        }
        if (PRIVATE (editor)->element_node_view) {
                g_free (PRIVATE (editor)->element_node_view) ;
                PRIVATE (editor)->element_node_view = NULL ;
        }
        if (PRIVATE (editor)->text_node_view) {
                g_free (PRIVATE (editor)->text_node_view) ;
                PRIVATE (editor)->text_node_view = NULL ;
        }
        if (PRIVATE (editor)->comment_node_view) {
                g_free (PRIVATE (editor)->comment_node_view) ;
                PRIVATE (editor)->comment_node_view = NULL ;
        }
        if (PRIVATE (editor)->cdata_section_node_view) {
                g_free (PRIVATE (editor)->cdata_section_node_view) ;
                PRIVATE (editor)->cdata_section_node_view = NULL ;
        }
        if (PRIVATE (editor)->pi_node_view) {
                g_free (PRIVATE (editor)->pi_node_view) ;
                PRIVATE (editor)->pi_node_view = NULL ;
        }
        if (PRIVATE (editor)->doc_node_view) {
                g_free (PRIVATE (editor)->doc_node_view) ;
                PRIVATE (editor)->doc_node_view = NULL ;
        }
        PRIVATE (editor)->dispose_has_run = TRUE ;

        if (gv_parent_class 
            && G_OBJECT_CLASS (gv_parent_class)->dispose) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
        }
}

static void
mlview_node_editor_finalize (GObject *a_this)
{
        MlViewNodeEditor *editor = NULL ;
        
        g_return_if_fail (a_this && MLVIEW_NODE_EDITOR (a_this)) ;
        editor = MLVIEW_NODE_EDITOR (a_this) ;
        g_return_if_fail (PRIVATE (editor)) ;
        g_free (PRIVATE (editor)) ;
        PRIVATE (editor) = NULL ;

        if (gv_parent_class 
            && G_OBJECT_CLASS (gv_parent_class)->finalize) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

static void
mlview_node_editor_construct (MlViewNodeEditor * a_this,
                              MlViewAppContext * a_app_context)
{
        MlViewNodeEditorPrivate *private_data = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        private_data = PRIVATE (a_this);
        private_data->curr_xml_node = NULL;
        private_data->app_context = a_app_context;
        private_data->node_view =
                GTK_NOTEBOOK (gtk_notebook_new ());

        gtk_notebook_set_show_tabs (private_data->node_view,
                                    FALSE);

        gtk_notebook_popup_disable (private_data->node_view);
        gtk_paned_add1 (GTK_PANED (a_this),
                        GTK_WIDGET (private_data->node_view));

        /*
         *now that the mlview app context context has been 
         *passed to this object, can build
         *sub components that need that context.
         */
        mlview_node_editor_build_xml_element_node_view
                (a_this);
        mlview_node_editor_build_xml_text_node_view
                (a_this);
        mlview_node_editor_build_xml_comment_node_view
                (a_this);
        mlview_node_editor_build_xml_cdata_section_node_view
                (a_this);
        mlview_node_editor_build_xml_pi_node_view
                (a_this);
        mlview_node_editor_build_xml_doc_node_view
                (a_this);
}


/**
 *Classical instance initialyser.Creates the widget. 
 *
 */
static void
mlview_node_editor_init (MlViewNodeEditor * a_this)
{
        if (PRIVATE (a_this) == NULL)
                PRIVATE (a_this) =
                        g_malloc0 (sizeof
                                   (MlViewNodeEditorPrivate));

}

/*************************
 *public exported methods
 *************************/

/**
 *Default constructor.
 *@param a_app_context the application context.
 *@param a_doc #MlViewXMLDocument the document object model we
 *are editing.
 *@return the newly built widget (node editor) or NULL in case of
 *error.
 */
GtkWidget *
mlview_node_editor_new (MlViewAppContext * a_app_context,
                        MlViewXMLDocument *a_doc)
{
        MlViewNodeEditor *editor = NULL;

        editor = g_object_new (MLVIEW_TYPE_NODE_EDITOR, NULL);
        PRIVATE (editor)->curr_xml_document = a_doc ;
        mlview_node_editor_construct (editor, a_app_context);

        return GTK_WIDGET (editor);
}


/**
 *Sets a new application context the node editor.
 *@param a_this the current instance of #MlViewNodeEditor.
 *@param a_app_context the new application context.
 */
void
mlview_node_editor_set_application_context (MlViewNodeEditor *a_this,
                                            MlViewAppContext *a_app_context)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->app_context = a_app_context;
}

/**
 *Gets the application context used by the current node editor.
 *@param a_this the current instance of #MlViewNodeEditor.
 *@return the returned application context.
 */
MlViewAppContext *
mlview_node_editor_get_application_context (MlViewNodeEditor *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_NODE_EDITOR
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return PRIVATE (a_this)->app_context;
}

/**
 *Gets the current xml node being edited by the node editor.
 *@return a pointer on the xmlNode being edited. 
 */
xmlNodePtr
mlview_node_editor_get_current_xml_node (MlViewNodeEditor *a_editor)
{
        g_return_val_if_fail (a_editor != NULL, NULL);
        return PRIVATE (a_editor)->curr_xml_node;
}


/**
 *Edits an xml node. This is basically what this
 *widget is for.
 *@param a_editor the current instance of #MlViewNodeEditor.
 *@param a_xml_doc the document object model the node belongs to.
 *@param a_node xml node to edit.
 */
void
mlview_node_editor_edit_xml_node (MlViewNodeEditor * a_editor,
                                  MlViewXMLDocument * a_xml_doc,
                                  xmlNode * a_node)
{
        /*some sanity checks */
        g_return_if_fail (a_editor != NULL
                          && MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (MLVIEW_IS_NODE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));

        if (PRIVATE (a_editor)->curr_xml_node) {                
                /*commit pending editing transactions*/
                if (mlview_node_editor_has_an_editing_transaction_started (a_editor) == TRUE) {
                        mlview_node_editor_commit_editing_transaction (a_editor) ;
                }
        }
        PRIVATE (a_editor)->curr_xml_node = a_node;
        g_return_if_fail (PRIVATE (a_editor)->curr_xml_node !=
                          NULL);

        switch (a_node->type) {
        case XML_ELEMENT_NODE:
                /*
                 *edit the element node using the 
                 *XMLElementNodeView of the element editor
                 */
                mlview_node_editor_xml_element_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_TEXT_NODE:
                /*
                 *show the text content using the 
                 *XMLTextNodeView of the element editor
                 */
                mlview_node_editor_xml_text_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_CDATA_SECTION_NODE:
                /*
                 *show the cdata using the 
                 *XMLCDataSectionNodeView of the element editor
                 */
                mlview_node_editor_xml_cdata_section_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_COMMENT_NODE:
                mlview_node_editor_xml_comment_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_DOCUMENT_NODE:
                mlview_node_editor_xml_doc_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        case XML_PI_NODE:
                mlview_node_editor_xml_pi_node_view_edit_xml_node
                        (a_editor, a_xml_doc, a_node);
                break;

        default:
                break;
        }
        /*FIXME: remove the provious children of right side of a_editor */
        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_editor)->node_view));

}                               /*end mlview_node_editor_edit_xml_node */


/**
 *Visualy (only) Clears the element editor. 
 *Does not free anything. 
 *@param a_editor the current instance of #MlViewNodeEditor.
 */
void
mlview_node_editor_clear (MlViewNodeEditor * a_editor)
{
        g_return_if_fail (a_editor != NULL
                          && MLVIEW_IS_NODE_EDITOR (a_editor));

        /*clear all the views */
        mlview_node_editor_clear_xml_element_node_view
                (a_editor);
        mlview_node_editor_clear_xml_text_node_view (a_editor);
        mlview_node_editor_clear_xml_comment_node_view
                (a_editor);
        mlview_node_editor_clear_xml_cdata_section_node_view
                (a_editor);
        mlview_node_editor_clear_xml_pi_node_view (a_editor);

}

/**
 *Connects the signals of the document object model.
 *@param a_this the current instance of #MlViewNodeEditor
 *@param a_doc the documet object model to connect to.
 *@return MLVIEW_OK upon successful completion, an error code 
 *otherwise.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_node_editor_connect_to_doc (MlViewNodeEditor *a_this,
                                   MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_NODE_EDITOR (a_this)
                              && PRIVATE (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;
        
        g_signal_connect (G_OBJECT (a_doc),
                          "node-selected",
                          G_CALLBACK (xml_doc_node_selected_cb),
                          a_this) ; 

        g_signal_connect (G_OBJECT (a_doc),
                          "node-unselected",
                          G_CALLBACK (xml_doc_node_unselected_cb),
                          a_this) ; 

        g_signal_connect (G_OBJECT (a_doc),
                          "node-changed",
                          G_CALLBACK (xml_doc_node_changed_cb),
                          a_this) ;

        if (PRIVATE (a_this)->element_node_view)
                mlview_attrs_editor_connect_to_doc
                        (PRIVATE (a_this)->element_node_view->attrs_editor,
                         a_doc) ;
        mlview_ns_editor_connect_to_doc
                (PRIVATE (a_this)->element_node_view->ns_editor,
                 a_doc) ;
        return MLVIEW_OK ;
}

/**
 *Disconnect from the document object model we connected to using
 *mlview_node_editor_connect_to_doc().
 *@param a_this the current instance of #MlViewNodeEditor
 *@param a_doc the document object model to connect to.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_node_editor_disconnect_from_doc (MlViewNodeEditor *a_this,
                                        MlViewXMLDocument *a_doc)
{
        g_return_val_if_fail (a_this
                              && MLVIEW_IS_NODE_EDITOR (a_this)
                              && PRIVATE (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc)
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_handlers_disconnect_by_func 
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_changed_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_selected_cb),
                 a_this) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (a_doc),
                 G_CALLBACK (xml_doc_node_unselected_cb),
                 a_this) ;
        mlview_attrs_editor_disconnect_from_doc 
                (PRIVATE (a_this)->element_node_view->attrs_editor,
                 a_doc) ;
        mlview_ns_editor_disconnect_from_doc 
                (PRIVATE (a_this)->element_node_view->ns_editor,
                 a_doc) ;
        return MLVIEW_OK ;
}
