/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it 
 *and/or modify it under the terms of 
 *the GNU General Public License as published
 * by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will be useful, 
 *but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of MERCHANTABILITY or 
 *FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the GNU General Public License 
 *along with MlView; 
 *see the file COPYING. If not, write to the 
 *Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright informations.
 */

/**
 *@file
 *The definition of the #MlViewAppContext class.
 *
 *The context is the way by which the application container 
 *and the applications communicate (here, the application is MlViewEditor).
 *Both applications and the application container can store information in it.
 *The application container insures the consistency of the 
 *application context during the life time of the application. 
 *
 */
#include <stdarg.h>
/*#include <libgnomeui/libgnomeui.h>*/
#include <string.h>
#include <eel/eel-gconf-extensions.h>
#include <mlview-marshal.h>
#include "mlview-app-context.h"
#include "mlview-file-descriptor.h"
#include "mlview-utils.h"
#include "mlview-xml-document.h"

/*********************************
 *Gconf keys used throughout the
 *application:
 *********************************/

#define DEFAULT_TREE_EXPANSION_DEPTH_KEY "/apps/mlview/default-tree-expansion-depth"
#define VALIDATION_IS_ON_KEY "/apps/mlview/validation-is-on"
#define DEFAULT_EDITING_VIEW_TYPE_KEY "/apps/mlview/default-editing-view-type"
#define ENABLE_COMPLETION_BOX_KEY "/apps/mlview/enable-completion-box"

#define MAIN_WINDOW_WIDTH_KEY "/apps/mlview/sizes/main-window-width"
#define MAIN_WINDOW_HEIGHT_KEY "/apps/mlview/sizes/main-window-height"
#define TREE_EDITOR_SIZE_KEY "/apps/mlview/sizes/tree-editor-size"
#define COMPLETION_SIZE_KEY "/apps/mlview/sizes/completion-size"

static const gchar *tree_editors_node_colours_keys[NB_OF_TREE_NODE_COLOURS + 1] = 
{"/apps/mlview/colours/xml-element-node",
 "/apps/mlview/colours/xml-attribute-node",
 "/apps/mlview/colours/xml-text-node",
 "/apps/mlview/colours/xml-comment-node",
 "/apps/mlview/colours/xml-document-node",
 "/apps/mlview/colours/xml-pi-node",
 "/apps/mlview/colours/xml-dtd-node",
 "/apps/mlview/colours/xml-entity-decl-node",
 NULL} ;

/**
 *Some icons.
 */
#define ELEMENT_ICON      "mlview/xml-element-node.png"
#define OPEN_ELEMENT_ICON "mlview/xml-element-node-open.png"
#define TEXT_ICON         "mlview/xml-text-node.png"
#define ROOT_ICON         "mlview/xml-root.png"
#define OPEN_ROOT_ICON    "mlview/xml-root-open.png"
#define COMMENT_ICON      "mlview/xml-comment-node.png"
#define PI_ICON           "mlview/xml-pi-node.png"
#define ENTITY_REF_ICON   "mlview/xml-entity-ref-node.png"

enum NumberOfSignals {
        APPLICATION_INITIALIZED,
        CONTEXTUAL_MENU_REQUESTED,
        VIEW_SWAPPED,
        DOCUMENT_NAME_CHANGED,
        VIEW_UNDO_STATE_CHANGED,
        NUMBER_OF_SIGNALS
} ;

static guint gv_signals[NUMBER_OF_SIGNALS] = {0} ;

/**
 *The private data members of MlViewAppContext.
 */
struct _MlViewAppContextPrivate {

        /**The place to store some elements of the context.*/
        GHashTable *context_elements;

        /**
         *the application wide settings
         */
        struct MlViewAppSettings *settings ;

        /**A cache for the xpms. 
	 *We have choosen this method
	 *to enable pixmap designers to
	 *change the pixmaps and have
	 *mlview load the new pixmap dynamically.
	 */
        GHashTable *pixmaps_cache;

        /** 
	 *mlview application bitmaps cache.
	 *one the bitmap have been loaded from the disk, they
	 *are cached in this hash table.
	 */
        GHashTable *bitmaps_cache;

        /**
	 *The place where the error messages are
	 *put before being displayed.
	 */
        gchar *error_msg_buffer;

        /**
	 *The title of the error dialog box
	 */
        gchar *error_dialog_title;

        /**
	 *A cache for the instance of MlViewFileSelection
	 *used by instance of MlViewAppContext.
	 */
        MlViewFileSelection *file_sel;

        /**
         *A cache for the instance of GtkFileChooser (for gtk+ >=2.4)
         *components should use this instance of file chooser.         
         */
#ifdef MLVIEW_WITH_GTK_FILE_CHOOSER
        GtkFileChooser *file_chooser ;
#endif
        /**
	 *The xml catalog loaded by mlview
	 */
        xmlCatalog *xml_catalog;

        /*
         *the last attribute id number 
         *part generated by mlview
         */
        gint last_id;

        /**
         *the gconfclient cache
         */
        GConfClient * gconf_client ;
        gboolean dispose_has_run;

        /**
         *the refcount on 
         *the type icons
         */
        guint type_icons_refcnt;

        /**
         *the type icons
         */
        struct MlViewTypeIcons *type_icons;
};


/**
 *the macro to access the private part of MlViewAppContext object
 */
#define PRIVATE(mlv_app_context) (mlv_app_context->private)

static GObjectClass *gv_parent_class = NULL;

static void
mlview_app_context_class_init (MlViewAppContextClass * a_klass);

static void
mlview_app_context_init (MlViewAppContext * a_ctxt);

static void
mlview_app_context_finalize (GObject * a_this);

static void
mlview_app_context_dispose (GObject * a_object);

static GObject *mlview_app_context_new (void);

static void
pixmaps_cache_foreach_func (gchar * a_key,
                             GdkPixmap * a_value,
                             gchar * user_data);

static void
bitmaps_cache_foreach_func (gchar * a_key,
                            GdkBitmap * a_value,
                            gchar * user_data);

static void
mlview_app_context_gconf_notify_func (GConfClient *a_gconf_client,
                                      guint a_cnxid,
                                      GConfEntry *a_entry,
                                      gpointer a_user_data) ;

static gboolean is_a_gconf_entry_a_tree_editor_node_color (GConfEntry *a_entry,
                                                           enum MlViewTreeEditorsNodeColour *a_color) ;

/*=====================================================
 *Private methods imposed by the gtk object framework.
 *====================================================*/

/**
 *The standard type builder. 
 *@return the id of the type of
 *#MlViewAppContext.
 */
GType
mlview_app_context_get_type (void)
{
        static GType type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewAppContextClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_app_context_class_init,
                        NULL, NULL,
                        sizeof (MlViewAppContext),
                        0,
                        (GInstanceInitFunc)
                        mlview_app_context_init
                };
                type = g_type_register_static (G_TYPE_OBJECT,
                                               "MlViewAppContext",
                                               &type_info, 0);
        }
        return type;
}

/**
 *The standard class initialyzer. 
 *@param a_klass the vtable structure of MlViewAppContext.
 */
static void
mlview_app_context_class_init (MlViewAppContextClass * a_klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (a_klass);

        gv_parent_class = g_type_class_peek_parent (a_klass);
        g_return_if_fail (G_IS_OBJECT_CLASS (gv_parent_class));
        object_class->dispose = mlview_app_context_dispose;
        object_class->finalize = mlview_app_context_finalize;

        /*signals creation code*/
        gv_signals[APPLICATION_INITIALIZED] =
                g_signal_new ("application-initialized",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewAppContextClass,
                               application_initialized), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);

        gv_signals[CONTEXTUAL_MENU_REQUESTED] =
                g_signal_new ("contextual-menu-requested",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewAppContextClass,
                               contextual_menu_requested), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);


        gv_signals[VIEW_SWAPPED] =
                g_signal_new ("view-swapped",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewAppContextClass,
                               view_swapped), NULL, NULL,
                              mlview_marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER,
                              G_TYPE_POINTER);
        
        gv_signals[DOCUMENT_NAME_CHANGED] =
                g_signal_new ("document-name-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewAppContextClass, document_name_changed), 
                              NULL, NULL,
                              mlview_marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);

        gv_signals[VIEW_UNDO_STATE_CHANGED] =
                g_signal_new ("view-undo-state-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET
                              (MlViewAppContextClass,
                               view_undo_state_changed), NULL, NULL,
                              mlview_marshal_VOID__VOID,
                              G_TYPE_NONE, 0, NULL);
}


static void
pixmaps_cache_foreach_func (gchar * a_key,
                            GdkPixmap * a_value,
                            gchar * user_data)
{
        gdk_pixmap_unref (a_value);
}

static void
bitmaps_cache_foreach_func (gchar * a_key,
                            GdkBitmap * a_value,
                            gchar * user_data)
{
        gdk_bitmap_unref (a_value);
}


/*
 *The destroy method associated to this class. 
 *@param a_object the instance of MlViewAppContext to be destroyed.
 */
static void
mlview_app_context_dispose (GObject * a_object)
{
        MlViewAppContext *ctxt = NULL;

        g_return_if_fail (a_object);

        ctxt = MLVIEW_APP_CONTEXT (a_object);

        g_return_if_fail (PRIVATE (ctxt));
        g_return_if_fail (PRIVATE (ctxt)->dispose_has_run ==
                          FALSE);

        if (PRIVATE (ctxt)->context_elements) {
                g_hash_table_destroy (PRIVATE (ctxt)->
                                      context_elements);
        }
        if (PRIVATE (ctxt)->pixmaps_cache) {
                g_hash_table_foreach (PRIVATE (ctxt)->
                                      pixmaps_cache, (GHFunc)
                                      pixmaps_cache_foreach_func,
                                      NULL);
                PRIVATE (ctxt)->pixmaps_cache = NULL;
        }
        if (PRIVATE (ctxt)->file_sel) {
                gtk_widget_destroy (GTK_WIDGET
                                    (PRIVATE (ctxt)->file_sel));
                PRIVATE (ctxt)->file_sel = NULL;
        }
#ifdef MLVIEW_WITH_GTK_FILE_CHOOSER
        if (PRIVATE (ctxt)->file_chooser) {
                gtk_widget_destroy (GTK_WIDGET 
                                    (PRIVATE (ctxt)->file_chooser)) ;
                PRIVATE (ctxt)->file_chooser = NULL ;
        }
#endif
        if (PRIVATE (ctxt)->bitmaps_cache) {
                g_hash_table_foreach (PRIVATE (ctxt)->
                                      bitmaps_cache, (GHFunc)
                                      bitmaps_cache_foreach_func,
                                      NULL);
                PRIVATE (ctxt)->bitmaps_cache = NULL;
        }

        if (PRIVATE (ctxt)->xml_catalog) {
                xmlFreeCatalog (PRIVATE (ctxt)->xml_catalog);
                PRIVATE (ctxt)->xml_catalog = NULL;
        }

        if (PRIVATE (ctxt)->gconf_client) {
                g_object_unref
                        (G_OBJECT (PRIVATE (ctxt)->gconf_client)) ;
                PRIVATE (ctxt)->gconf_client = NULL ;
        }

        if (PRIVATE (ctxt)->settings) {
                g_free (PRIVATE (ctxt)->settings) ;
                PRIVATE (ctxt)->settings = NULL ;
        }
        mlview_utils_unref_available_encodings () ;
        PRIVATE (ctxt)->dispose_has_run = TRUE;
        if (gv_parent_class
            && G_OBJECT_CLASS (gv_parent_class)->dispose) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_object) ;
        }
}

/**
 *The finalize method. Frees the private data structure
 *of #MlViewAppContext.
 */
static void
mlview_app_context_finalize (GObject * a_this)
{
        MlViewAppContext *ctxt = MLVIEW_APP_CONTEXT (a_this);

        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (ctxt));

        g_return_if_fail (PRIVATE (ctxt)
                          && PRIVATE (ctxt)->dispose_has_run ==
                          TRUE);

        g_free (PRIVATE (ctxt));
        PRIVATE (ctxt) = NULL;

        if (gv_parent_class
            && G_OBJECT_CLASS (gv_parent_class)->finalize) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

/**
 *The MlViewAppContext instance initialyzer.
 *This methods allocates the MlViewAppContext object structure.
 *
 *@param a_ctxt the current instance of MlViewAppContext.
 */
static void
mlview_app_context_init (MlViewAppContext * a_ctxt)
{
        g_return_if_fail (a_ctxt != NULL);

        if (PRIVATE (a_ctxt) == NULL)
                PRIVATE (a_ctxt) =
                        g_malloc0 (sizeof
                                   (MlViewAppContextPrivate));

        if (PRIVATE (a_ctxt)->context_elements == NULL)
                PRIVATE (a_ctxt)->context_elements =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        if (PRIVATE (a_ctxt)->pixmaps_cache == NULL)
                PRIVATE (a_ctxt)->pixmaps_cache =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        if (PRIVATE (a_ctxt)->bitmaps_cache == NULL)
                PRIVATE (a_ctxt)->bitmaps_cache =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        /*
         *initialise the list of available encodings
         *supported by mlview and ref count it.
         */
        mlview_utils_init_available_encodings_list ();

        mlview_utils_ref_available_encodings ();

        PRIVATE (a_ctxt)->dispose_has_run = FALSE;
}


/**
 *The default constructor of MlViewAppContext.
 *
 *@return the newly created instance of MlViewAppContext.
 */
static GObject *
mlview_app_context_new (void)
{
        MlViewAppContext *result = NULL;

        result = g_object_new (MLVIEW_TYPE_APP_CONTEXT, NULL);

        return G_OBJECT (result);
}

/******************
 *Public methods
 ******************/

/**
 *The instance getter of the MlViewAppContext class.
 *Actually, if you want to use a 'singleton pattern', use this method.
 *All the invocations to this method will allways return the same object ;
 *
 *@return the singleton object. The firts time this method is called, creates
 *a new object and returns it. subsequent calls will always 
 *returns the same object.
 */
MlViewAppContext *
mlview_app_context_get_instance (void)
{
        static MlViewAppContext *app_context = NULL;

        if (!app_context) {
                app_context =
                        MLVIEW_APP_CONTEXT
                        (mlview_app_context_new ());
        }
        return app_context;
}


/**
 *a getter of the last attribute id automatically generated.
 *This method is used in the automatic generation of attributes ids.
 *
 *@param a_this: the current instance of MlViewAppContext.
 *
 *
 *@returns a pointer to the last id stored in the object a_this using
 *the method mlview_app_context_set_last_id() ; 
 *You should manipulate this pointer
 *with great care or else you can break the attribute id storage.
 */
gint *
mlview_app_context_get_last_id_ptr (MlViewAppContext *
                                    a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return &PRIVATE (a_this)->last_id;
}


/**
 *The getter of the last generated attribute id.
 *This method is used in the automatic generation of attributes ids.
 *
 *@param a_this the current instance of MlViewAppContext.
 *@return the last generated attribute id.
 */
gint
mlview_app_context_get_last_id (MlViewAppContext * a_this)
{
        g_return_val_if_fail (a_this != NULL, 0);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_this), 0);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              0);

        return PRIVATE (a_this)->last_id;
}


/**
 *A setter of the last generated attribut id.
 *This method is used in the automatic attribute id generation process.
 *
 *@param a_this the current instance of MlViewAppContext.
 *@param a_new_id the new id.
 */
void
mlview_app_context_set_last_id (MlViewAppContext * a_this,
                                gint a_new_id)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->last_id = a_new_id;
}


/**
 *Stores a key/value pair in the context.
 *This method is used tho stores objects in the context.
 *If you want to store strings, (for settings for ex) 
 *use mlview_app_context_set_settings_value() method instead.
 *
 *@param a_context the current instance of MlViewAppContext class.
 *@param a_element_name the key of the object to store. 
 *Note that this string is not copied so this
 *pointer must remain valid during all the lifetime of a_context. 
 *When a_context is destroyed, 
 *a_element_name is _not_ destroyed. 
 *So the caller must manage the life time of this string.
 *@param a_context_element the element to store in the context. 
 *This pointer is not destroyed upon
 *destruction of a_context. 
 *It is up to the caller to manage the life time of this pointer. If
 *it is NULL, a NULL value is stored.
 */
void
mlview_app_context_set_element (MlViewAppContext * a_context,
                                const gchar * a_element_name,
                                gpointer a_context_element)
{
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));
        g_return_if_fail (PRIVATE (a_context) != NULL);
        g_return_if_fail (a_element_name != NULL);

        if (PRIVATE (a_context)->context_elements == NULL)
                PRIVATE (a_context)->context_elements =
                        g_hash_table_new (g_str_hash,
                                          g_str_equal);

        g_hash_table_insert (PRIVATE (a_context)->
                             context_elements,
                             (guchar *) a_element_name,
                             a_context_element);
}


/**
 *Gets an object stored in the context using the 
 *mlview_app_context_set_element() method.
 *@param a_context the current instance of MlViewAppContext.
 *@param a_element_name the key of the object stored.
 *@return the element stored or NULL if nothing has been stored. 
 *If a NULL value has been stored
 *using this key, (using the mlview_app_context_set_element() method) 
 *a NULL value is also returned.
 */
gpointer
mlview_app_context_get_element (MlViewAppContext * a_context,
                                const gchar * a_element_name)
{
        g_return_val_if_fail (a_context != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT (a_context),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_context) != NULL, NULL);

        return g_hash_table_lookup (PRIVATE (a_context)->
                                    context_elements,
                                    a_element_name);
}

/**
 *Emits the "contextual-menu-requested" signal to notify that
 *an editing widget has been requested for a contextual menu.
 *@param a_this the current instance of #MlViewAppContext.
 *@param a_source the widget that received the contextual menu
 *request.
 *@param MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_app_context_notify_contextual_menu_request (MlViewAppContext *a_this,
                                                   GtkWidget *a_source_widget,
                                                   GdkEvent *a_event)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && a_source_widget
                              && GTK_IS_WIDGET (a_source_widget),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[CONTEXTUAL_MENU_REQUESTED], 0,
                       a_source_widget, a_event) ;

        return MLVIEW_OK ;
}

/**
 *Emits a "view-swapped" signal to notify a new view becoming
 *the "currently selected view"
 *@param a_this the current instance of #MlViewAppContext
 *@param a_old_view the previous "current selected view". Should
 *be an implementation of #MlViewIView
 *@param a_new_view the new "current selected view". Should be
 *an implementation of the #MlViewIView
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_app_context_notify_view_swapped (MlViewAppContext *a_this,
                                        gpointer a_old_view,
                                        gpointer a_new_view)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals [VIEW_SWAPPED], 0,
                       a_old_view, a_new_view) ;

        return MLVIEW_OK ;
}

/**
 *Notifies listeners that the application started.
 *This actually makes the #MlViewAppContext instance
 *to emit the "application-initialized" signal.
 *@param a_this the current instance of #MlViewAppContext
 *@return MLVIEW_OK upon succesful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_app_context_notify_application_initialized (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this
                              && MLVIEW_IS_APP_CONTEXT (a_this), 
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[APPLICATION_INITIALIZED], 0) ;
        return MLVIEW_OK ;
}

/**
 *Notifies listeners that the name of the document being edited has
 *changed.
 *@param a_this the current instance of #MlViewAppContext
 *@param a_doc the document which name has changed.
 *@return MLVIEW_OK upon successful completion.
 */
enum MlViewStatus 
mlview_app_context_notify_document_name_changed (MlViewAppContext *a_this,
                                                 gpointer a_doc)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (a_doc) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[DOCUMENT_NAME_CHANGED], 0,
                               a_doc) ;
        }
        return MLVIEW_OK ;
}


enum MlViewStatus 
mlview_app_context_notify_view_undo_state_changed (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this) ,
                              MLVIEW_BAD_PARAM_ERROR) ;

	g_signal_emit (G_OBJECT (a_this),
		       gv_signals[VIEW_UNDO_STATE_CHANGED], 0) ;
        return MLVIEW_OK ;
}

/**
 *Pushes a message on the app bar. 
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message (same parameter as printf)
 */
void
mlview_app_context_sbar_push_message (MlViewAppContext *a_context,
                                      const gchar * a_msg_format,
                                      ...)
{
        va_list params;
        gchar *msg = NULL;

        /*
         *FIXME: this is broken since I dumped
         *the GnomeApp stuffs.
         */
        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);
        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}


/**
 *Pops the last message pushed on the message stack.
 *The effect is that the message pushed before the 
 *last message has been pushed, will be
 *displayed. 
 *@param a_context the current instance of MlViewAppContext.
 */
void
mlview_app_context_sbar_pop_message (MlViewAppContext *
                                     a_context)
{
        /*
         *FIXME: this is broken since I dumped
         *the GnomeApp stuffs.
         */
}


/**
 *Sets the message that is displayed when no message is pushed on the
 *status bar message stack. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message
 *(as in the standard printf function)
 */
void
mlview_app_context_sbar_set_default_message (MlViewAppContext *a_context,
                                             const gchar *a_msg_format, ...)
{
        va_list params;
        gchar *msg = NULL;

        /*
         *FIXME: this is broken since I dumped
         *the GnomeApp stuffs.
         */
        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);


        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}


/**
 *Displays a message to the user. 
 *It opens a Modal dialog with the message on it.
 *The user just has to click OK. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the message (as in the printf funtion)
 */
void
mlview_app_context_message (MlViewAppContext * a_context,
                            const gchar * a_msg_format, ...)
{
        va_list params;
        gchar *msg = NULL;

        g_return_if_fail (a_context && a_msg_format) ;
        va_start (params, a_msg_format);
        msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        mlview_utils_display_message_dialog (a_context, msg) ;
        if (msg) {
                g_free (msg);
                msg = NULL;
        }
}

/**
 *Sets the title of the next error message that will be displayed by a call
 *to the function mlview_app_context_error() or 
 *mlview_app_context_display_buffered_error().
 *After a call to one of these two functions, 
 *all subsequent calls will display an error dialog
 *with no title. 
 *
 *@param a_title the new title of the error 
 *message dialog thatwill be used to display the 
 *content of the error message buffer.
 *look at definition of mlview_app_context_bufferize_error(). 
 *@param a_context the current application context.
 */
void
mlview_app_context_set_error_dialog_title (MlViewAppContext *a_context,
                                           const gchar * a_title)
{
        if (!a_context || !PRIVATE (a_context))
                return;

        if (PRIVATE (a_context)->error_dialog_title) {
                g_free (PRIVATE (a_context)->error_dialog_title);
                PRIVATE (a_context)->error_dialog_title = NULL;
        }

        if (a_title)
                PRIVATE (a_context)->error_dialog_title =
                        g_strdup (a_title);
        else
                PRIVATE (a_context)->error_dialog_title = NULL;
}


/**
 *Getter of error message dialog title.
 *@param a_context the current application context.
 *
 *@return a pointer to the title of the next 
 *error message that will be displayed. 
 */
gchar *
mlview_app_context_get_error_dialog_title (MlViewAppContext *a_context)
{
        if (!a_context || !PRIVATE (a_context))
                return NULL;

        return PRIVATE (a_context)->error_dialog_title;
}


/**
 *Bufferizes the error message given by a_msg_format and the other parameters.
 *The buffered error message can be displayed to the user by calling 
 *the function
 *@mlview_app_context_display_buffered_error() 
 *with the same a_context in parameter.
 *Note that if this function is called with a NULL a_context param, the
 *formated error message is sent to stderr and is not buffered. Subsquent call
 *to mlview_app_context_display_buffered_error() will not display anything. 
 *
 *@param a_msg_format the format string. 
 *Plays the same role asthe format string used in the function fprintf.
 *@param ... the parameters that may be used in the format string. 
 *@param a_context the current application context.
 *
 *
 */
void
mlview_app_context_bufferize_error (MlViewAppContext * a_context,
                                    const gchar * a_msg_format,
                                    ...)
{
        va_list params;
        gchar *temp_err_msg = NULL;

        va_start (params, a_msg_format);
        temp_err_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (temp_err_msg) {

                if (!PRIVATE (a_context)->error_msg_buffer)

                        PRIVATE (a_context)->error_msg_buffer =
                                g_strdup (temp_err_msg);
                else {
                        gchar *old_str =
                                PRIVATE (a_context)->
                                error_msg_buffer;

                        PRIVATE (a_context)->error_msg_buffer =
                                g_strconcat
                                (PRIVATE (a_context)->
                                 error_msg_buffer, temp_err_msg,
                                 NULL);
                        g_free (old_str);
                        old_str = NULL ;
                }
        } else if (temp_err_msg) {
                g_printerr (temp_err_msg);
        }
        if (temp_err_msg) {
                g_free (temp_err_msg);
                temp_err_msg = NULL ;
        }
}


/**
 *Displays the error message that has been buffered by calling the function
 *mlview_app_context_bufferize_error() with a_context in parameter.
 *If a_context is set to NULL,  this function does nothing. 
 *
 *@param a_context the current application context.
 */
void
mlview_app_context_display_buffered_error (MlViewAppContext *a_context)
{

        if (!a_context || !PRIVATE (a_context)
            || !PRIVATE (a_context)->error_msg_buffer)
                return;

        mlview_app_context_error (a_context,
                                  PRIVATE (a_context)->
                                  error_msg_buffer);

        g_free (PRIVATE (a_context)->error_msg_buffer);

        PRIVATE (a_context)->error_msg_buffer = NULL;
}

/**
 *Tells wether if there are error messages to be displayed or not.
 *Note the messages that may have been stored in the 
 *message buffer have been stored
 *using the function mlview_app_context_bufferize_error() .
 *
 *@param a_context the current instance of MlViewAppContext.
 *@return TRUE if the error buffer is empty and false if not. 
 */
gboolean
mlview_app_context_error_buffer_is_empty (MlViewAppContext *a_context)
{
        g_return_val_if_fail (a_context != NULL, TRUE);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT (a_context),
                              TRUE);
        g_return_val_if_fail (PRIVATE (a_context) != NULL, TRUE);

        return (PRIVATE (a_context)->error_msg_buffer == NULL
                || !strcmp (PRIVATE (a_context)->
                            error_msg_buffer, ""));
}


/**
 *Displays a warning message. It actually displays a modal 
 *dialog showing the message.
 *The user just has to click OK. 
 *If a_context is set to NULL, the message is sent to stderr. 
 *
 *@param a_msg_format the format of the warning message to display.
 *@param a_context the current application context. 
 *If NULL, stderr is used to display.
 */
void
mlview_app_context_warning (MlViewAppContext * a_context,
                            const gchar * a_msg_format, ...)
{
        va_list params;
        gchar *warning_msg = NULL;


        va_start (params, a_msg_format);
        warning_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        if (warning_msg) {
                mlview_utils_display_warning_dialog 
                        (a_context, warning_msg) ;
        } else if (warning_msg) {
                g_printerr (warning_msg);
        }

        if (warning_msg) {
                g_free (warning_msg);
                warning_msg = NULL ;
        }
}

/**
 *Displays a (fatal) error to the user. It actually 
 *shows a modal dialog box to the user.
 *The user just has to click OK. 
 *
 *@param a_context the current instance of MlViewAppContext.
 *@param a_msg_format the format of the 
 *message (as in the standard printf funtion)
 */
void
mlview_app_context_error (MlViewAppContext * a_context,
                          const gchar * a_msg_format, ...)
{
        va_list params;
        gchar *err_msg = NULL;

        g_return_if_fail (a_context) ;

        va_start (params, a_msg_format);
        err_msg = g_strdup_vprintf (a_msg_format, params);
        va_end (params);

        /*Add a title to the error message if needed */
        if (err_msg && PRIVATE (a_context)->error_dialog_title) {
                gchar *tmp_str = err_msg;

                err_msg = g_strconcat
                        (PRIVATE (a_context)->error_dialog_title,
                         "\n", tmp_str, NULL);
                g_free (tmp_str);

                /*delete the error dialog title */
                g_free (PRIVATE (a_context)->error_dialog_title);
                PRIVATE (a_context)->error_dialog_title = NULL;
        }

        if (err_msg) {
                mlview_utils_display_error_dialog (a_context,
                                                   err_msg);
        } else if (err_msg) {
                g_printerr (err_msg);
                return;
        }

        if (err_msg)
                g_free (err_msg);
}


/**
 *Tells if a gconf value (contained in a gconfentry) is
 *MlViewSettings::tree_editors.nodes_colour[i] .
 *If yes, tells what the color is.
 *@param a_entry the instance of GConfEntry to consider
 *@param a_color out parameter. The colour contained in the
 *gconfentry.
 */
static gboolean
is_a_gconf_entry_a_tree_editor_node_color (GConfEntry *a_entry,
                                           enum MlViewTreeEditorsNodeColour *a_colour)
{
        gchar *key = NULL ;
        gint i = 0 ;

        g_return_val_if_fail (a_entry && a_colour, FALSE) ;
        key = (gchar*) gconf_entry_get_key (a_entry) ;
        if (!key)
                return FALSE ;
        for (i = 0 ;
             i < NB_OF_TREE_NODE_COLOURS && tree_editors_node_colours_keys[i] ;
             i++) {
                if (!strcmp (tree_editors_node_colours_keys[i],
                             key)) {
                        *a_colour = i ;
                        return TRUE ;
                }
        }
        return FALSE ;
}

/**
 *A callback that get called by gconf
 *whenever a gconf key gets changed.
 *@param a_gconf_client the gconf client associated to
 *the mlview application.
 *@param a_cnxid the gconf notification connection id.
 *@param a_entry the gconf entry that has been changed.
 *@param a custom data passed to the notification init function.
 *In the current case, a_user_data is an instance of 
 *#MlViewAppContext.
 */
static void
mlview_app_context_gconf_notify_func (GConfClient *a_gconf_client,
                                      guint a_cnxid,
                                      GConfEntry *a_entry,
                                      gpointer a_user_data)
{
        GConfValue *value = NULL ;
        gchar *key = NULL ;
        MlViewAppContext *ctxt = a_user_data ;
        struct MlViewAppSettings *settings = NULL ;
        enum MlViewTreeEditorsNodeColour colour_item = 0 ;

        g_return_if_fail (a_gconf_client && a_entry) ;
        g_return_if_fail (ctxt && MLVIEW_IS_APP_CONTEXT (ctxt)) ;

        key = (gchar*) gconf_entry_get_key (a_entry) ;
        g_return_if_fail (key) ;
        settings = mlview_app_context_get_settings (ctxt) ;
        g_return_if_fail (settings) ;
        if (!strcmp (key,
                     DEFAULT_TREE_EXPANSION_DEPTH_KEY)) {
                value = gconf_entry_get_value (a_entry) ;
                g_return_if_fail
                        (value  && value->type == GCONF_VALUE_INT) ;
                settings->tree_editors.default_doc_expansion_depth =
                        gconf_value_get_int (value) ;
        } else if (!strcmp (key,
                            VALIDATION_IS_ON_KEY)) {
                value = gconf_entry_get_value (a_entry) ;
                g_return_if_fail
                        (value  && value->type == GCONF_VALUE_BOOL) ;
                settings->general.validation_is_on =
                        gconf_value_get_bool (value) ;
        } else if (!strcmp (key,
                            ENABLE_COMPLETION_BOX_KEY)) {
                value = gconf_entry_get_value (a_entry);
                g_return_if_fail
                        (value && value->type == GCONF_VALUE_BOOL);
                settings->general.enable_completion_box = 
                        gconf_value_get_bool (value);
        } else if (is_a_gconf_entry_a_tree_editor_node_color
                   (a_entry, &colour_item) == TRUE) {
                settings->tree_editors.nodes_colours[colour_item] 
                        = eel_gconf_get_string (key) ;
        } else if (!strcmp (key, DEFAULT_EDITING_VIEW_TYPE_KEY)) {
                settings->general.default_editing_view_type =
                        eel_gconf_get_string (key) ;
        } else if (!strcmp (key, MAIN_WINDOW_WIDTH_KEY)) {
		value = gconf_entry_get_value (a_entry);
		settings->sizes.main_window_width =
			gconf_value_get_int (value);
	} else if (!strcmp (key, MAIN_WINDOW_HEIGHT_KEY)) {
		value = gconf_entry_get_value (a_entry);
		settings->sizes.main_window_height =
			gconf_value_get_int (value);
	} else if (!strcmp (key, TREE_EDITOR_SIZE_KEY)) {
		value = gconf_entry_get_value (a_entry);
		settings->sizes.tree_editor_size =
			gconf_value_get_int (value);
	} else if (!strcmp (key, COMPLETION_SIZE_KEY)) {
		value = gconf_entry_get_value (a_entry);
		settings->sizes.completion_size =
			gconf_value_get_int (value);
	}



}

/**
 *Loads all the gconf keys defined by the MlView application.
 *@param a_this the current instance of the #MlViewAppContext.
 *@return MLVIEW_OK upon successfull completion, an error code
 *otherwise.
 */
static enum MlViewStatus
mlview_app_context_load_gconf_settings (MlViewAppContext *a_this)
{
        gint i = 0 ;
        struct MlViewAppSettings *settings = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        settings = PRIVATE (a_this)->settings ;
        g_return_val_if_fail (settings, MLVIEW_ERROR) ;
        /*make sure we our gconf client gets initialized*/
        PRIVATE (a_this)->gconf_client =
                mlview_app_context_get_gconf_client (a_this) ;
        g_return_val_if_fail (PRIVATE (a_this)->gconf_client, 
                              MLVIEW_ERROR) ;
        /*
         *walk through the settings data structure 
         *and fill all its fields.
         */
        settings->tree_editors.default_doc_expansion_depth =
                eel_gconf_get_integer
                (DEFAULT_TREE_EXPANSION_DEPTH_KEY) ;
        settings->general.validation_is_on =
                eel_gconf_get_boolean (VALIDATION_IS_ON_KEY);
        settings->general.enable_completion_box = 
                eel_gconf_get_boolean (ENABLE_COMPLETION_BOX_KEY);
        settings->general.default_editing_view_type =
                eel_gconf_get_string (DEFAULT_EDITING_VIEW_TYPE_KEY) ;
	settings->sizes.main_window_width =
		eel_gconf_get_integer (MAIN_WINDOW_WIDTH_KEY);
	settings->sizes.main_window_height =
		eel_gconf_get_integer (MAIN_WINDOW_HEIGHT_KEY);
	settings->sizes.tree_editor_size =
		eel_gconf_get_integer (TREE_EDITOR_SIZE_KEY);
	settings->sizes.completion_size =
		eel_gconf_get_integer (COMPLETION_SIZE_KEY);
        for (i = 0 ;
             i < NB_OF_TREE_NODE_COLOURS && tree_editors_node_colours_keys[i];
             i++) {
                settings->tree_editors.nodes_colours[i] =
                        eel_gconf_get_string 
                        (tree_editors_node_colours_keys[i]) ;
        }
        return MLVIEW_OK ;
}

/**
 *Gets the gconf client associated to the mlview application.
 *The user doesn't have to care about either 
 *notification configuration,
 *or even the destruction of the returned gconf client.
 *@param a_this the current instance of #MlViewAppContext.
 *@return the gconf client associated to the current instance
 *of the mlview application.
 */
GConfClient *
mlview_app_context_get_gconf_client (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this),
                              NULL) ;

        if (!PRIVATE (a_this)->gconf_client) {
                PRIVATE (a_this)->gconf_client =
                        eel_gconf_client_get_global () ;
                g_return_val_if_fail
                        (PRIVATE (a_this)->gconf_client,
                         NULL) ;
                g_object_ref (G_OBJECT 
                              (PRIVATE (a_this)->gconf_client)) ;
                gconf_client_add_dir
                        (PRIVATE (a_this)->gconf_client,
                         "/apps/mlview", GCONF_CLIENT_PRELOAD_NONE,
                         NULL) ;
                gconf_client_notify_add
                        (PRIVATE (a_this)->gconf_client,
                         "/apps/mlview",
                         mlview_app_context_gconf_notify_func,
                         a_this, NULL, NULL) ;
        }
        return PRIVATE (a_this)->gconf_client ;
}

/**
 *Gets the settings data structure associated to
 *the application context.
 *@param a_this the current instance of #MlViewAppContext.
 *@return the settings data structure. The user must _NOT_
 *free the returned pointer.
 */
struct MlViewAppSettings *
mlview_app_context_get_settings (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this),
                              NULL) ;
        if (!PRIVATE (a_this)->settings) {
                PRIVATE (a_this)->settings =
                        g_try_malloc
                        (sizeof (struct MlViewAppSettings)) ;
                if (!PRIVATE (a_this)->settings) {
                        mlview_utils_trace_debug
                                ("g_try_malloc () failed") ;
                        return NULL ;
                }
                memset (PRIVATE (a_this)->settings, 0,
                        sizeof (struct MlViewAppSettings)) ;
                mlview_app_context_load_gconf_settings (a_this) ;
        }
        return PRIVATE (a_this)->settings ;
}

/**
 * Save main window size settings
 */
void
mlview_app_context_save_window_state (MlViewAppContext *a_this, gint width, gint height)
{
	 g_return_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this));
	
	gconf_client_set_int (
			PRIVATE (a_this)->gconf_client,
			MAIN_WINDOW_WIDTH_KEY,
			width,
			NULL);

	gconf_client_set_int (
			PRIVATE (a_this)->gconf_client,
			MAIN_WINDOW_HEIGHT_KEY,
			height,
			NULL);
}

/**
 * Save TreeView paned positions state
 */
void
mlview_app_context_save_treeview_state (
				MlViewAppContext *a_this,
				gint treeview_height,
				gint completion_box_size)
{
	g_return_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this));
	
	gconf_client_set_int (
			PRIVATE (a_this)->gconf_client,
			TREE_EDITOR_SIZE_KEY,
			treeview_height,
			NULL);

	gconf_client_set_int (
			PRIVATE (a_this)->gconf_client,
			COMPLETION_SIZE_KEY,
			completion_box_size,
			NULL);
}
	
/**
 *Gets the file selection dialog box associated to the mlview application.
 *The first call to this method creates the file selection. Subsequent
 *calls to this method returns a pointer to the same object, making these calls
 *very fast.
 *
 *@param a_this: the current xml document considered.
 *@param a_title the title of the mlview file selector.
 *@returns the file selection associated to this mlview app context.
 */
MlViewFileSelection *
mlview_app_context_get_file_selector (MlViewAppContext *a_this,
                                      const gchar * a_title)
{

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        if (!PRIVATE (a_this)->file_sel) {
                PRIVATE (a_this)->file_sel =
                        MLVIEW_FILE_SELECTION
                        (mlview_file_selection_new ());
        }

        if (a_title && strcmp (a_title, ""))
                gtk_window_set_title
                        (GTK_WINDOW
                         (PRIVATE (a_this)->file_sel),
                         a_title);

        return PRIVATE (a_this)->file_sel;
}


#ifdef MLVIEW_WITH_GTK_FILE_CHOOSER
GtkFileChooser *
mlview_app_context_get_file_chooser (MlViewAppContext *a_this,
                                     const gchar *a_title, MlViewFileChooserMode a_mode)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_APP_CONTEXT (a_this)
                              && PRIVATE (a_this),
                              NULL) ;

        /*if (!PRIVATE (a_this)->file_chooser) {*/
	GtkWidget * parent_window = NULL ;

	parent_window = mlview_app_context_get_element (a_this,
							"MlViewMainWindow") ;

	PRIVATE (a_this)->file_chooser = GTK_FILE_CHOOSER (gtk_file_chooser_dialog_new
							   (a_title,
							    GTK_WINDOW (parent_window),
							    GTK_FILE_CHOOSER_ACTION_OPEN, 
							    GTK_STOCK_CANCEL, 
							    GTK_RESPONSE_CANCEL, 
							    (a_mode == MlViewFileChooserOpenMode) ? GTK_STOCK_OPEN : GTK_STOCK_SAVE,
							    GTK_RESPONSE_OK,
							    NULL));
	g_return_val_if_fail (PRIVATE (a_this)->file_chooser != NULL, NULL) ;
	g_signal_connect (G_OBJECT (PRIVATE (a_this)->file_chooser),
			  "delete-event",
			  G_CALLBACK (gtk_widget_hide_on_delete),
			  NULL) ;
	gtk_window_set_modal (GTK_WINDOW (PRIVATE (a_this)->file_chooser),
			      TRUE) ;
	gtk_file_chooser_set_local_only 
		(PRIVATE (a_this)->file_chooser, TRUE);
	gtk_file_chooser_set_select_multiple 
		(GTK_FILE_CHOOSER (PRIVATE (a_this)->file_chooser),
		 FALSE);
        /*}*/
        g_return_val_if_fail (PRIVATE (a_this)->file_chooser != NULL, NULL) ;
        gtk_window_set_title (GTK_WINDOW (PRIVATE (a_this)->file_chooser),
                              a_title) ;
	if (a_mode == MlViewFileChooserOpenMode)
		gtk_file_chooser_set_action (PRIVATE (a_this)->file_chooser, GTK_FILE_CHOOSER_ACTION_OPEN);
	else
		gtk_file_chooser_set_action (PRIVATE (a_this)->file_chooser, GTK_FILE_CHOOSER_ACTION_SAVE);

        return PRIVATE (a_this)->file_chooser ;
}
#endif

/**
 *Associates an xml catalog to the current instance of MlViewAppContext.
 *When a_this is destroyed, 
 *the instance of xml catalog is _NOT_ destroyed.
 *@param a_xml_catalog the new xml catalog
 *@param a_this the current instance of MlViewAppContext.
 */
void
mlview_app_context_set_xml_catalog (MlViewAppContext *a_this,
                                    xmlCatalog * a_xml_catalog)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->xml_catalog = a_xml_catalog;
}

/**
 *Unloads the type icons used to represent different types
 *of XML nodes in some components of mlview.
 *@param a_this the current instance of MlViewAppContext.
 */
static void
mlview_app_context_unload_type_icons (MlViewAppContext *a_this)
{
        struct MlViewTypeIcons *icons = NULL;

        g_return_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this));
        g_return_if_fail (PRIVATE (a_this));

        icons = PRIVATE (a_this)->type_icons;

        if (icons) {
                if (icons->element) {
                        g_object_unref (G_OBJECT (icons->element));
                        icons->element = NULL ;
                }
                if (icons->open_element) {
                        g_object_unref (G_OBJECT (icons->open_element));
                        icons->open_element = NULL ;
                }
                if (icons->text) {
                        g_object_unref (G_OBJECT (icons->text));
                        icons->text = NULL ;
                }
                if (icons->root) {
                        g_object_unref (G_OBJECT (icons->root));
                        icons->root = NULL ;
                }
                if (icons->open_root) {
                        g_object_unref (G_OBJECT (icons->open_root));
                        icons->open_root = NULL ;
                }
                if (icons->comment) {
                        g_object_unref (G_OBJECT (icons->comment));
                        icons->comment = NULL ;
                }
                if (icons->pi) {
                        g_object_unref (G_OBJECT (icons->pi));
                        icons->pi = NULL ;
                }
                if (icons->entity_ref) {
                        g_object_unref (G_OBJECT (icons->entity_ref));
                        icons->entity_ref = NULL ;
                }

                g_free (icons);

                icons = NULL;
                PRIVATE (a_this)->type_icons = NULL;
        }
}

/**
 *Loads the type icons used to represent different types 
 *of XML nodes in some components of mlview.
 *@param a_this the current instance of MlViewAppContext.
 *@return a MlViewTypeIcons struct which contains the icons.
 */
static struct MlViewTypeIcons *
mlview_app_context_load_type_icons (MlViewAppContext *a_this)
{
        gchar *path = NULL;
        struct MlViewTypeIcons *type_icons = NULL;

        g_return_val_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this), NULL);

        type_icons = g_try_malloc (sizeof (struct MlViewTypeIcons));

        if (!type_icons) {
                mlview_utils_trace_debug ("malloc failed, system may be out of memory");

                return NULL;
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          ELEMENT_ICON, TRUE, NULL);

        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (ELEMENT_ICON) ;
        } else {
                type_icons->element = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          OPEN_ELEMENT_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (OPEN_ELEMENT_ICON) ;
        } else {
                type_icons->open_element = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          TEXT_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (TEXT_ICON) ;
        } else {
                type_icons->text = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          ROOT_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (ROOT_ICON) ;
        } else {
                type_icons->root = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          OPEN_ROOT_ICON, TRUE, NULL) ;
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (OPEN_ROOT_ICON) ;
        } else {
                type_icons->open_root = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          COMMENT_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (COMMENT_ICON) ;
        } else {
                type_icons->comment = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }
  
        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          ENTITY_REF_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (ENTITY_REF_ICON) ;
        } else {
                type_icons->entity_ref = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          PI_ICON, TRUE, NULL);
        if (!path) {
                mlview_utils_trace_debug ("lookup of icon file failed:") ;
                mlview_utils_trace_debug (PI_ICON) ;
        } else {
                type_icons->pi = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path);
        }

        return type_icons;
}

/**
 *Decrements the reference count of the structure which
 *countains the type icons used to represent different types
 *of XML nodes used in some components of mlview.
 *@param a_this the current instance of MlViewAppContext.
 */
void
mlview_app_context_type_icons_unref (MlViewAppContext *a_this)
{
        g_return_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this));
        g_return_if_fail (PRIVATE (a_this));

        g_return_if_fail (PRIVATE (a_this)->type_icons_refcnt);

        PRIVATE (a_this)->type_icons_refcnt--;
        
        if (!PRIVATE (a_this)->type_icons_refcnt) {
                mlview_app_context_unload_type_icons (a_this);
        }
}

/**
 *Returns the type icons used to represent different types
 *of XML nodes in some components of mlview.
 *@param a_this the current instance of MlViewAppContext.
 *@return a MlViewTypeIcons struct which contains the icons.
 */
struct MlViewTypeIcons *
mlview_app_context_type_icons_ref (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this), NULL);

        if (PRIVATE (a_this)->type_icons_refcnt) {
                g_return_val_if_fail (PRIVATE (a_this)->type_icons, NULL);

                PRIVATE (a_this)->type_icons_refcnt++;
        } else {
                g_return_val_if_fail (!PRIVATE (a_this)->type_icons, NULL);
                
                PRIVATE (a_this)->type_icons = mlview_app_context_load_type_icons (a_this);

                g_return_val_if_fail (PRIVATE (a_this)->type_icons, NULL);

                PRIVATE (a_this)->type_icons_refcnt++;
        }

        return PRIVATE (a_this)->type_icons;
}

/**
 *Getter of the xml catalog associated to the current instance of
 *MlViewAppContext.
 *
 *@param the current instance of MlViewAppContext .
 *@return the instance of xmlCatalog associated to the current instance of
 *MlViewAppContext.
 */
xmlCatalog *
mlview_app_context_get_xml_catalog (MlViewAppContext *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_APP_CONTEXT
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return (PRIVATE (a_this)->xml_catalog);
}


/**
 *Presents a dialog to the user asking for the name
 *of an internal subset
 *@param a_name out parameter. The name entered by the user, or
 *NULL if she entered nothing.
 *@return TRUE if the user entered a name, FALSE otherwise.
 */
gboolean
mlview_app_context_ask_internal_subset_node_name (gchar **a_name)
{
        GtkWidget *dialog;
        GtkWidget *hbox;
        GtkWidget *image;
        GtkWidget *label;
        GtkWidget *entry;
        gint res;
        gboolean ret = FALSE;

        g_return_val_if_fail (a_name, FALSE);

        *a_name = NULL;

        dialog = gtk_dialog_new_with_buttons 
                (_("Name of the internal subset node"),
                 NULL, GTK_DIALOG_MODAL,
                 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
                 GTK_STOCK_OK, GTK_RESPONSE_OK,
                 NULL);

        gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
        gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

        hbox = gtk_hbox_new (FALSE, 6);
        image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION,
                                          GTK_ICON_SIZE_DIALOG);
        gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
        label = gtk_label_new (_("Internal subset node name:"));
        entry = gtk_entry_new ();
        gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);

        gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                            hbox, FALSE, FALSE, 0);

        gtk_widget_show_all (dialog);

        res = gtk_dialog_run (GTK_DIALOG (dialog));
        switch (res) {
        case GTK_RESPONSE_OK:
                *a_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
                ret = TRUE;
                break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
		break;
        default:
                g_assert_not_reached ();
	}

        gtk_widget_destroy (dialog);
        return ret;
}
