/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gtksourceview/gtksourceview.h> 
#include <gtksourceview/gtksourcelanguage.h> 
#include <gtksourceview/gtksourcelanguagesmanager.h>
#include <gtksourceview/gtksourcebuffer.h>
#include "mlview-source-view.h"
#include "mlview-xml-document.h"

#define PRIVATE(obj) ((obj)->priv)

struct _MlViewSourceViewPriv {

	/*a pointer to the document object model*/
	MlViewXMLDocument *xml_doc ;

	/*a pointer to the application context hold by this object.*/
	MlViewAppContext *app_context ;

	/* 
	 * the gtksourceview we use to actually implement 
	 * this source editing view
	 */
	GtkSourceView *native_sv ;

	/* The language manager, used by native_sv */
	GtkSourceLanguagesManager *languages_manager ;

	GtkUIManager *ui_manager ;

	GtkActionGroup *action_group ;

	guint popup_edit_menu_merge_id ;

	guint edit_menu_merge_id ;

	/* <gtksourceviewparameters> */
	gboolean show_line_numbers ;
	guint tabs_width ;
        gboolean set_autoindent ;
	gboolean set_show_margin ;
	guint margin ;
	/*</gtksourceviewparameters>*/

	/*is set if the doc has been modified, but not this view*/
	gboolean document_changed ;

	/*is set if this view has been modified but not the doc (yet)*/
	gboolean view_changed ;

	gboolean dispose_has_run ;
} ;

static MlViewViewAdapterClass *gv_parent_class = NULL ;

static void dispose (GObject *a_this) ;

static void finalize (GObject *a_this) ;

static void mlview_source_view_iview_init (MlViewIView *a_iface) ;

static enum MlViewStatus connect_to_doc (MlViewIView *a_this,
					 MlViewXMLDocument *a_doc) ;

static enum MlViewStatus disconnect_from_doc (MlViewIView *a_this,
					         MlViewXMLDocument *a_doc) ;

static enum MlViewStatus get_must_rebuild_upon_document_reload 
					(MlViewIView *a_this,
		                         gboolean *a_result) ;

static void document_changed_cb (MlViewXMLDocument *a_this,
		                 MlViewSourceView *a_user_data) ;

static enum MlViewStatus serialize_and_load_doc (MlViewSourceView *a_this) ;

static void is_swapped_out_cb (MlViewIView *a_this, gpointer a_data) ;

static void is_swapped_in_cb (MlViewIView *a_this, gpointer a_data) ;

static GtkSourceLanguagesManager * get_languages_manager (MlViewSourceView *a_this) ;

static enum MlViewStatus set_language (MlViewSourceView *a_this,
		                       GtkSourceLanguage *a_language) ;

static enum MlViewStatus set_language_from_mime_type (MlViewSourceView *a_this,
		                                      const gchar *a_mime_type) ;

static enum MlViewStatus set_default_language (MlViewSourceView *a_this) ;

static enum MlViewStatus get_source_buffer (MlViewSourceView *a_this, 
		                            GtkSourceBuffer **a_source_buffer) ;

/* not used yet
static enum MlViewStatus get_native_source_view (MlViewSourceView *a_this,
                                                 GtkSourceView **a_source_view) ;
*/

static enum MlViewStatus save_text_buffer_into_xml_doc (MlViewSourceView *a_this) ;

static enum MlViewStatus get_document (MlViewIView *a_this,
				       MlViewXMLDocument **a_doc) ;

static void changed_cb (GtkTextBuffer *a_text_buffer, 
	                gpointer a_data) ;

static void going_to_save_cb (MlViewXMLDocument *a_doc,
		              MlViewSourceView *a_view) ;

GtkTextBuffer *get_text_buffer (MlViewSourceView *a_this) ;

/*
static enum MlViewStatus get_last_dangling_opened_tag (MlViewSourceView *a_this,
		                                       GtkTextIter *a_iter,
						       gchar **a_tag_name) ;
*/
static gboolean button_press_event_cb (GtkWidget *a_widget,
				       GdkEventButton *a_event,
				       gpointer a_user_data) ;

static enum MlViewStatus handle_contextual_menu_request (MlViewSourceView *a_this,
							 GtkWidget *a_src_widget,
							 GdkEvent *a_event)  ;

enum MlViewStatus get_contextual_menu (MlViewSourceView *a_this,
				       GtkWidget **a_menu_ptr) ;

static enum MlViewStatus build_contextual_menu (MlViewSourceView *a_this) ;


static enum MlViewStatus build_edit_menu_body (MlViewSourceView *a_this,
					       const char *a_menu_root_path) ;

static const gchar * build_edit_menu_root_path (MlViewSourceView *a_this,
					        gboolean a_popup) ;

static enum MlViewStatus mlview_source_view_undo (MlViewIView *a_this) ;

static enum MlViewStatus mlview_source_view_redo (MlViewIView *a_this) ;

static enum MlViewStatus mlview_source_view_can_undo (MlViewIView *a_this,
                                                      gboolean *a_can_undo) ;

static enum MlViewStatus mlview_source_view_can_redo (MlViewIView *a_this,
                                                      gboolean *a_can_redo) ;

static void close_all_tag_action_cb (GtkAction *a_action,
				     gpointer user_data) ;

static void close_tag_action_cb (GtkAction *a_action,
				 gpointer user_data) ;

static gboolean widget_realized_cb (GtkWidget *a_widget,
                                    gpointer a_user_data) ; 

static void undo_state_changed_cb (GtkSourceBuffer *source_buffer,
                                   gboolean an_arg,
				   gpointer a_source_view) ;

static GtkActionEntry gv_edit_menu_actions [] = {
	{
		"CloseTagAction", NULL, N_("Close last tag"),
		"<control>t", N_("Closes the last opened tag, if any"),
		G_CALLBACK (close_tag_action_cb)
	},

	{
		"CloseAllTagsAction", NULL, N_("Close all tags"),
		"<control>T", N_("Closes all the relevant opened tags, if any"),
		G_CALLBACK (close_all_tag_action_cb)
	}
} ;

/*************************************************
 * Private methods and signal callbacks
 ************************************************/

GtkTextBuffer *
get_text_buffer (MlViewSourceView *a_this) 
{
	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this) && PRIVATE (a_this)
			      && PRIVATE (a_this)->native_sv,
			      NULL) ;

	g_return_val_if_fail (GTK_IS_SOURCE_VIEW (PRIVATE (a_this)->native_sv),
			      NULL) ;

	return gtk_text_view_get_buffer 
		(GTK_TEXT_VIEW (PRIVATE (a_this)->native_sv)) ;
}

static gboolean
contextual_menu_requested_cb (MlViewAppContext *a_ctxt,
		              GtkWidget *a_source_widget,
			      GdkEvent *a_event,
			      gpointer a_user_data)
{
	
	MlViewSourceView *source_view = NULL ;

	g_return_val_if_fail (a_ctxt && MLVIEW_IS_APP_CONTEXT (a_ctxt)
			      && a_source_widget && a_user_data,
			      FALSE) ;
	source_view = MLVIEW_SOURCE_VIEW (a_user_data) ;
	handle_contextual_menu_request 
		(source_view, a_source_widget, a_event) ;
	return FALSE ;

}

static enum MlViewStatus 
handle_contextual_menu_request (MlViewSourceView *a_this,
		                GtkWidget *a_source_widget,
				GdkEvent *a_event) 
{
	GtkWidget *menu = NULL ;
	GdkEventButton *event_button = NULL ;
	MlViewAppContext *ctxt = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	/*
	 * make sure the widget that was the source of the request is
	 * the native source view widget
	 */
	if (a_source_widget != GTK_WIDGET (PRIVATE (a_this)->native_sv)) {
		return MLVIEW_ERROR ;
	}

	/*make sure the event is really a button press*/
	if (a_event->type != GDK_BUTTON_PRESS) {
		return MLVIEW_ERROR ;
	}
	event_button = (GdkEventButton*) a_event ;
	/*TODO: finish this ! */
	get_contextual_menu (a_this, &menu) ;
	if (!menu) {
		mlview_utils_trace_debug ("menu construction failed !") ;
		return MLVIEW_ERROR ;
	}
	
	ctxt = mlview_source_view_get_application_context (a_this) ;
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL,
	                ctxt, ((GdkEventButton*)a_event)->button,
			((GdkEventButton*)a_event)->time) ;
	return MLVIEW_OK ;
}

enum MlViewStatus
get_contextual_menu (MlViewSourceView *a_this,
		     GtkWidget **a_menu_ptr) 
{
	GtkUIManager *ui_manager = NULL ;
	GtkWidget *tmp_widget = NULL, *menu = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	ui_manager = mlview_source_view_get_ui_manager (a_this) ;
	g_return_val_if_fail (ui_manager, MLVIEW_BAD_PARAM_ERROR) ;
	tmp_widget = gtk_ui_manager_get_widget 
	(ui_manager,
	 "/SourceViewPopupEditMenu/CloseTagMenuitem") ;
	if (!tmp_widget) {
		build_contextual_menu (a_this) ;
	}
	menu = gtk_ui_manager_get_widget 
	(ui_manager, "/SourceViewPopupEditMenu") ;
	g_return_val_if_fail (menu, MLVIEW_ERROR) ;
	gtk_widget_show_all (menu) ;
	*a_menu_ptr = menu ;
	if (!*a_menu_ptr) { 
		return MLVIEW_ERROR ;
	}
	return MLVIEW_OK ;
}

static enum MlViewStatus
build_contextual_menu (MlViewSourceView *a_this)
{
	gchar * menu_root_path = NULL ;
	enum MlViewStatus status = MLVIEW_OK ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	menu_root_path = (gchar*)build_edit_menu_root_path (a_this, TRUE) ; 
	g_return_val_if_fail (menu_root_path, MLVIEW_ERROR) ;
	status = build_edit_menu_body (a_this, menu_root_path) ;

	return status;
}

static const gchar *
build_edit_menu_root_path (MlViewSourceView *a_this,
		           gboolean a_popup)
{
	gchar *menu_root_path = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this),
			      NULL) ;

	if (a_popup == TRUE) {
		menu_root_path = (gchar*)"/SourceViewPopupEditMenu";
	} else {
		menu_root_path = (gchar*)"MainMenubar/EditMenu";
	}
	return menu_root_path ;

}

/**
 * Really build the body of the contextual menu
 * proposed by this view
 * @param a_this the current instance of #MlViewSourceView
 * @param a_menu_root_path the path of the menu element
 * that contains this contextual menu 
 * @return MLVIEW_OK upon succesful completion, an error code
 * otherwise
 */
static enum MlViewStatus
build_edit_menu_body (MlViewSourceView *a_this,
		      const char *a_menu_root_path)
{
	guint *merge_id = NULL ;
	/*gchar *parent_menu_path = NULL ;*/
	GtkUIManager *ui_manager = NULL ;

	/*TODO: finish this !*/
	ui_manager = mlview_source_view_get_ui_manager (a_this) ;
	g_return_val_if_fail (ui_manager, MLVIEW_ERROR) ;

	if (!strcmp (a_menu_root_path, "/MainMenubar/EditMenu")) {
		if (!PRIVATE (a_this)->edit_menu_merge_id) {
			PRIVATE (a_this)->edit_menu_merge_id =
			gtk_ui_manager_new_merge_id  (ui_manager);
		}
		merge_id = &PRIVATE (a_this)->edit_menu_merge_id ;
	} else if (!strcmp (a_menu_root_path, "/SourceViewPopupEditMenu")){
		if (!PRIVATE (a_this)->popup_edit_menu_merge_id) {
			PRIVATE (a_this)->popup_edit_menu_merge_id =
			gtk_ui_manager_new_merge_id (ui_manager) ;
		}
		merge_id = &PRIVATE (a_this)->popup_edit_menu_merge_id ;
	} else {
		mlview_utils_trace_debug ("Unknow menu root path") ;
		mlview_utils_trace_debug (a_menu_root_path) ;
		return MLVIEW_ERROR ;
	}
	/*now the real meat of the contextual menu creation goes here !*/

	gtk_ui_manager_add_ui (ui_manager,
	                       *merge_id,
			       a_menu_root_path,
			       "CloseTagMenuitem",
			       "CloseTagAction",
			       GTK_UI_MANAGER_AUTO,
			       FALSE) ;

	gtk_ui_manager_add_ui (ui_manager,
	                       *merge_id,
			       a_menu_root_path,
			       "CloseAllTagsMenuitem",
			       "CloseAllTagsAction",
			       GTK_UI_MANAGER_AUTO,
			       FALSE) ;

	gtk_ui_manager_add_ui (ui_manager,
	                       *merge_id,
			       a_menu_root_path,
			       "SourceViewEditMenuSeparator1",
			       NULL,
			       GTK_UI_MANAGER_SEPARATOR,
			       FALSE) ;

	gtk_ui_manager_ensure_update (ui_manager) ;

	return MLVIEW_OK ;
}


static enum MlViewStatus 
mlview_source_view_undo (MlViewIView *a_this)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
	                      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	status = get_source_buffer (MLVIEW_SOURCE_VIEW (a_this), 
	                            &source_buffer) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;

	gtk_source_buffer_undo (source_buffer) ;

	return status ;
}

static enum MlViewStatus 
mlview_source_view_redo (MlViewIView *a_this)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (MLVIEW_IS_IVIEW (a_this)
	                      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;


	status = get_source_buffer (MLVIEW_SOURCE_VIEW (a_this), &source_buffer) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;

	gtk_source_buffer_redo (source_buffer) ;

	return status ;
}

static enum MlViewStatus 
mlview_source_view_can_undo (MlViewIView *a_this, gboolean *a_can_undo)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
	                      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	status = get_source_buffer (MLVIEW_SOURCE_VIEW (a_this), &source_buffer) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;
	*a_can_undo = gtk_source_buffer_can_undo (source_buffer) ;
	return status ;
}

static enum MlViewStatus 
mlview_source_view_can_redo (MlViewIView *a_this, gboolean *a_can_redo)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
	                      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	status = get_source_buffer (MLVIEW_SOURCE_VIEW (a_this), &source_buffer) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;
	*a_can_redo = gtk_source_buffer_can_redo (source_buffer) ;
	return status ;
}

static 
gboolean widget_realized_cb (GtkWidget *a_widget,
			     gpointer a_user_data)
{
	MlViewSourceView *view = NULL ;

	g_return_val_if_fail (a_user_data 
	                      && MLVIEW_IS_SOURCE_VIEW (a_user_data),
			      FALSE) ;
	view = MLVIEW_SOURCE_VIEW (a_user_data) ;
	g_return_val_if_fail (view, FALSE) ;
	if (!GTK_WIDGET_NO_WINDOW (a_widget)) {
		gtk_widget_add_events (a_widget, GDK_BUTTON3_MOTION_MASK) ;
		g_signal_connect (G_OBJECT (a_widget),
		                  "button-press-event",
				  G_CALLBACK (button_press_event_cb),
				  view) ;
	} else {
		mlview_utils_trace_debug ("Aaargh !!") ;
	}
	return FALSE ;

}

static gboolean
button_press_event_cb (GtkWidget *a_widget,
		       GdkEventButton *a_event,
		       gpointer a_user_data) 
{
	MlViewSourceView *source_view = NULL ;
	MlViewAppContext *app_context = NULL ;

	g_return_val_if_fail (a_widget && GTK_IS_WIDGET (a_widget),
			      FALSE) ;
	g_return_val_if_fail (a_event && a_user_data, FALSE) ;
	source_view = MLVIEW_SOURCE_VIEW (a_user_data) ;
	g_return_val_if_fail (source_view, FALSE) ;
	app_context = mlview_source_view_get_application_context (source_view) ;
	g_return_val_if_fail (app_context, FALSE) ;

	switch (a_event->type) {
		case GDK_BUTTON_PRESS:
			if (a_event->button == 3) {
				/*
				 * user pressed the right mouse button.
				 * Let's trigger the contextual menu mecanism.
				 */
				mlview_app_context_notify_contextual_menu_request 
					(app_context, 
					 GTK_WIDGET (PRIVATE 
						 (source_view)->native_sv),
					 (GdkEvent*)a_event) ;
				return TRUE ;
			        	
			}
		default:
			break ;
	}
	return FALSE ;
}

static void 
close_all_tag_action_cb (GtkAction *a_action,
			 gpointer user_data) 
{
	MlViewSourceView *thiz = NULL ;
	gboolean tag_has_been_closed = FALSE ;
	enum MlViewStatus status = MLVIEW_OK ;

	thiz = MLVIEW_SOURCE_VIEW (user_data) ;
	g_return_if_fail (thiz) ;

	while (TRUE) {

		status = mlview_source_view_close_currently_opened_tag 
		(thiz, &tag_has_been_closed) ;

		if (status != MLVIEW_OK) {
			mlview_utils_trace_debug 
			("Something bad happened "
		         "during the tag closing algorithm") ;
			return ;
		}
		if (tag_has_been_closed == FALSE)
			break ;
		else
			tag_has_been_closed = FALSE ;
	}
}

static void 
close_tag_action_cb (GtkAction *a_action,
		     gpointer user_data) 
{
	MlViewSourceView *thiz = NULL ;
	gboolean tag_has_been_closed = FALSE ;
	enum MlViewStatus status = MLVIEW_OK ;

	thiz = MLVIEW_SOURCE_VIEW (user_data) ;
	g_return_if_fail (thiz) ;


	/*Try to close the last opened tag*/
	status = mlview_source_view_close_currently_opened_tag 
		(thiz, &tag_has_been_closed) ;

	if (status != MLVIEW_OK) {
		mlview_utils_trace_debug 
		("Something bad happened "
		 "during the tag closing algorithm") ;
		return ;
	}
}

static void 
going_to_save_cb (MlViewXMLDocument *a_doc,
	          MlViewSourceView *a_view)
{
	g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
			  && a_view && MLVIEW_IS_SOURCE_VIEW (a_view)
			  && PRIVATE (a_view)) ;
	if (PRIVATE (a_view)->view_changed == TRUE) {
		save_text_buffer_into_xml_doc (a_view) ;
	}
}

static void 
changed_cb (GtkTextBuffer *a_text_buffer, 
	    gpointer a_data)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (a_text_buffer && GTK_IS_TEXT_BUFFER (a_text_buffer)
			  && a_data && MLVIEW_IS_SOURCE_VIEW (a_data)) ;

	thiz = MLVIEW_SOURCE_VIEW (a_data) ;
	g_return_if_fail (thiz) ;
	PRIVATE (thiz)->view_changed  = TRUE ;
}

static void 
document_changed_cb (MlViewXMLDocument *a_doc,
                     MlViewSourceView *a_thiz)
{
	g_return_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
			  && MLVIEW_IS_XML_DOCUMENT (a_doc)
			  && MLVIEW_IS_SOURCE_VIEW (a_thiz) 
			  && PRIVATE (a_thiz)) ;

	PRIVATE (a_thiz)->document_changed = TRUE ;
}

static void
is_swapped_in_cb (MlViewIView *a_this, gpointer a_data)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			  && MLVIEW_IS_SOURCE_VIEW (a_this)) ;
	thiz = MLVIEW_SOURCE_VIEW (a_this) ;
	g_return_if_fail (thiz && PRIVATE (thiz)) ;

	if (PRIVATE (thiz)->document_changed) {
		serialize_and_load_doc (thiz) ;
	}
}

static void
is_swapped_out_cb (MlViewIView *a_this, gpointer a_data)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			  && MLVIEW_IS_SOURCE_VIEW (a_this)) ;
	thiz = MLVIEW_SOURCE_VIEW (a_this) ;
	g_return_if_fail (thiz) ;
	save_text_buffer_into_xml_doc (thiz) ;
}

static void 
undo_state_changed_cb (GtkSourceBuffer *source_buffer,
		       gboolean an_arg,
		       gpointer a_source_view)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (source_buffer 
			  && GTK_IS_SOURCE_BUFFER (source_buffer)) ;

	thiz = MLVIEW_SOURCE_VIEW (a_source_view) ;
	g_return_if_fail (thiz && PRIVATE (thiz) 
			  && PRIVATE (thiz)->app_context) ;

	mlview_app_context_notify_view_undo_state_changed 
				(PRIVATE (thiz)->app_context) ;
}
/**
 * Private method that returns the languages manager associated to
 * this source view
 * @param a_this the current instance of #MlViewSourceView
 * @return the associated languages manager.
 */
static GtkSourceLanguagesManager * 
get_languages_manager (MlViewSourceView *a_this)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this),
			      NULL) ;

	if (!PRIVATE (a_this)->languages_manager) {
		PRIVATE (a_this)->languages_manager = 
			gtk_source_languages_manager_new () ;
		g_return_val_if_fail (PRIVATE (a_this)->languages_manager, NULL) ;
	}
	return PRIVATE (a_this)->languages_manager ; 
}


static enum MlViewStatus 
set_language (MlViewSourceView *a_this,
              GtkSourceLanguage *a_language)
{
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && a_language 
			      && GTK_IS_SOURCE_LANGUAGE (a_language), 
			      MLVIEW_BAD_PARAM_ERROR) ;
	source_buffer = GTK_SOURCE_BUFFER (gtk_text_view_get_buffer 
		(GTK_TEXT_VIEW (PRIVATE (a_this)->native_sv))) ;
	g_return_val_if_fail (source_buffer, MLVIEW_ERROR) ;

	gtk_source_buffer_set_highlight (source_buffer, TRUE) ;
	gtk_source_buffer_set_language (source_buffer, a_language) ;

	return MLVIEW_ERROR ;
}

static enum MlViewStatus 
set_language_from_mime_type (MlViewSourceView *a_this,
			     const gchar *a_mime_type)
{
	GtkSourceLanguagesManager *lm = NULL ;
	GtkSourceLanguage *language = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && a_mime_type,
			      MLVIEW_BAD_PARAM_ERROR) ;
	lm = get_languages_manager (a_this) ;
	g_return_val_if_fail (lm, MLVIEW_ERROR) ;
	language = gtk_source_languages_manager_get_language_from_mime_type 
		(lm, a_mime_type) ;
	g_return_val_if_fail (language, MLVIEW_ERROR) ;
	set_language (a_this, language) ;
	return MLVIEW_OK ;
}

static enum MlViewStatus 
set_default_language (MlViewSourceView *a_this)
{
	gchar *mime_type = NULL ;
	MlViewXMLDocument *doc = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this), 
			      MLVIEW_BAD_PARAM_ERROR) ;

	get_document (MLVIEW_IVIEW (a_this), &doc) ;
	g_return_val_if_fail (doc, MLVIEW_ERROR) ;
	mime_type = (gchar*) mlview_xml_document_get_mime_type (doc) ;
	g_return_val_if_fail (mime_type, MLVIEW_ERROR) ;
	set_language_from_mime_type (a_this, mime_type) ;
	return MLVIEW_OK ;
}

/************************************
 * implem of the MlViewIView methods.
 ************************************/

static enum MlViewStatus 
get_must_rebuild_upon_document_reload (MlViewIView *a_this,
                                       gboolean *a_result)
{

	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	*a_result = FALSE ;
	return MLVIEW_OK ;
}

static enum MlViewStatus
connect_to_doc (MlViewIView *a_this,
		MlViewXMLDocument *a_doc)
{

	g_signal_connect (G_OBJECT (a_doc),
	                  "document-changed",
			  G_CALLBACK (document_changed_cb),
			  a_this) ;

	g_signal_connect (G_OBJECT (a_doc),
			  "going-to-save",
			  G_CALLBACK (going_to_save_cb),
			  a_this) ;
	return MLVIEW_OK ;
}

static enum MlViewStatus
disconnect_from_doc (MlViewIView *a_this,
		     MlViewXMLDocument *a_doc)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (a_doc),
	                                      G_CALLBACK (document_changed_cb),
					      a_this) ;

	g_signal_handlers_disconnect_by_func (G_OBJECT (a_doc),
			                      G_CALLBACK (going_to_save_cb),
					      a_this) ;
	return MLVIEW_OK ;
}

/* not used yet
static enum MlViewStatus
execute_action_cb (MlViewIView *a_this,
		   MlViewAction *a_action)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			      && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && a_action,
			      MLVIEW_BAD_PARAM_ERROR) ;

	return MLVIEW_OK ;
}



static enum MlViewStatus
update_contextual_menu_cb (MlViewIView *a_this,
		           GtkWidget **a_menu)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;
	return MLVIEW_OK ;
}
*/

/*************************************
 * gobject mandatory methods implems
 ************************************/

static void
mlview_source_view_class_init (MlViewSourceViewClass *a_class)
{
	GObjectClass *gobject_class = NULL ;

	g_return_if_fail (a_class != NULL) ;

	gv_parent_class = g_type_class_peek_parent (a_class) ;
	g_return_if_fail (gv_parent_class) ;
	gobject_class = G_OBJECT_CLASS (a_class) ;	

	gobject_class->dispose = dispose ;
	gobject_class->finalize = finalize ;
}

static void
mlview_source_view_init (MlViewSourceView *a_this) 
{
	g_return_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
		          && MLVIEW_IS_IVIEW (a_this)) ;

	if (PRIVATE (a_this) == NULL) {
		PRIVATE (a_this) = g_try_malloc (sizeof (MlViewSourceViewPriv)) ;
		if (!PRIVATE (a_this)) {
			mlview_utils_trace_debug ("g_try_malloc failed") ;
			return ;
		}
		memset (PRIVATE (a_this), 0, sizeof (MlViewSourceViewPriv)) ;
	}

}

static void
mlview_source_view_iview_init (MlViewIView *a_iface)
{
	a_iface->get_document = get_document ;
	a_iface->connect_to_doc = connect_to_doc ;
	a_iface->disconnect_from_doc = disconnect_from_doc ;
	a_iface->get_must_rebuild_upon_document_reload = 
			get_must_rebuild_upon_document_reload ;
	
	a_iface->undo = mlview_source_view_undo;
	a_iface->redo = mlview_source_view_redo;
	a_iface->can_undo = mlview_source_view_can_undo ;
	a_iface->can_redo = mlview_source_view_can_redo ;
}

static void 
dispose (GObject *a_this)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)) ;
	g_return_if_fail (MLVIEW_IS_IVIEW (a_this)) ;

	thiz = MLVIEW_SOURCE_VIEW (a_this) ;
	g_return_if_fail (thiz && PRIVATE (thiz)) ;

	if (PRIVATE (thiz)->dispose_has_run) {
		return ;
	}

	if (PRIVATE (thiz)->xml_doc)  {
		mlview_iview_disconnect_from_doc (MLVIEW_IVIEW (thiz), 
						  PRIVATE (thiz)->xml_doc) ;
		g_object_unref (PRIVATE (thiz)->xml_doc) ; 
		PRIVATE (thiz)->xml_doc = NULL  ;
	}
	PRIVATE (thiz)->dispose_has_run = TRUE ;
	if (gv_parent_class 
	    && G_OBJECT_CLASS (gv_parent_class)->dispose) {
		 G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
	}
}

static void 
finalize (GObject *a_this)
{
	MlViewSourceView *thiz = NULL ;

	g_return_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)) ;

	thiz = MLVIEW_SOURCE_VIEW (a_this) ;
	g_return_if_fail (thiz) ;

	if (PRIVATE (thiz)) {
		g_free (PRIVATE (thiz)) ;
		PRIVATE (thiz) = NULL ;
	}
        if (gv_parent_class
            && G_OBJECT_CLASS (gv_parent_class)->finalize) {
                G_OBJECT_CLASS (gv_parent_class)->finalize (a_this) ;
        }
}

/*****************************
 * Other misc private methods
 * ***************************/

/**************************************
 * Tag closing related private methods
 *************************************/

/**
 * Gets the name of the rightmost tag that has been opened and not
 * closed.
 * @param a_this the current instance of #MlViewSourceView
 * @param a_iter an iterator that points to where to start parsing from
 * @param a_tag_name out parameter. The name of the rightmost dangling tag.
 * i.e, the rightmost tag that has been opened and that hasn't been closed.
 * @return MLVIEW_OK upon successful completion, an error code otherwise.
 */
static enum MlViewStatus 
get_last_dangling_opened_tag (MlViewSourceView *a_this,
			      GtkTextIter *a_iter,
			      gchar **a_tag_name) 
{
	GtkTextMark *insert_mark = NULL ;
	GtkTextIter cur = {0} , *stag_end = NULL,*tmp_iter = NULL ; 
	GtkTextBuffer *text_buffer = NULL ;
	gboolean found_lt = FALSE, is_an_empty_tag ;
	enum MlViewStatus status = MLVIEW_OK ;
	GString *tag_name = NULL ;
	GList *attrs = NULL ;/*a list of struc NameValuePair for the attributes*/
	GList *closed_tags_stack = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;
	text_buffer = get_text_buffer (a_this) ;
	if (!text_buffer) {
		mlview_utils_trace_debug ("could not get text buffer") ;
		return MLVIEW_ERROR ;
	}
	insert_mark = gtk_text_buffer_get_insert (text_buffer) ;
	g_return_val_if_fail (insert_mark, MLVIEW_ERROR) ;
	gtk_text_buffer_get_iter_at_mark (text_buffer, &cur, 
			                  insert_mark) ;

fetch: /*go find the previous opened tag*/
	
	while (1) {
		gunichar cur_char = 0 ;

		cur_char = gtk_text_iter_get_char (&cur) ;
		if (cur_char == '<') {
			found_lt = TRUE ;
			break ;
		}

		if (!gtk_text_iter_backward_char (&cur)) {
			found_lt = FALSE ;
			break ;
		}
	}
	if (!found_lt) {
		status = MLVIEW_NO_STAG_ERROR ;
		goto cleanup ;
	}
	status = mlview_utils_parse_start_tag2 (&cur, &tag_name, 
			                        &attrs, &tmp_iter,
						&is_an_empty_tag) ;
	/*END OF FETCH*/

	if (status == MLVIEW_OK && is_an_empty_tag == FALSE) {
		gchar *tag = NULL ;
		gchar** ptr = NULL ;
		/* 
		 * the first tag we see when walking backward is an open tag
		 * so we found the last tag start we were looking for. 
		 */ 

		/*
		 * check if this tag doesn't have a matching closing tag ...
		 */
		if (closed_tags_stack) {
			ptr = &tag ;
			mlview_utils_peek_from_stack (closed_tags_stack, 
						      (gpointer*)ptr) ;
		}
		if (tag && !strcmp (tag, tag_name->str)) {
			/*
			 * yes, this stag has a matching closing tag.
			 * pop the closed_tags_staack and go backward fetch
			 * another tag.
			 */
			ptr = &tag ;
			closed_tags_stack = mlview_utils_pop_from_stack 
					(closed_tags_stack, 
					 (gpointer*)ptr) ;
			if (tag) {
				g_free (tag) ;
				tag = NULL ;
			}
			if (!gtk_text_iter_backward_char (&cur)) {
				status = MLVIEW_ERROR ;
				goto cleanup ;
			}
			goto fetch ;
		}

		/*now prepare data for the caller and get out gently.*/
		if (a_iter) {
			a_iter = tmp_iter ;
		}
		tmp_iter = NULL ;
		*a_tag_name = g_strdup (tag_name->str) ;

		status =  MLVIEW_OK ;
		goto cleanup ;
	} else if (status == MLVIEW_OK && is_an_empty_tag == TRUE){
		/*
		 * hmmh, we found an empty tag. we need to go backward fetch
		 * an open empty tag ...
		 */
		if (!gtk_text_iter_backward_char (&cur)) {
			status = MLVIEW_NO_DANGLING_OPEN_TAG_FOUND_ERROR ;
			goto cleanup ;
		}
		goto fetch ;
	} else {
		/*what we found was not an empty tag.*/

		/*is it a closing tag ?*/
		if (tag_name) {
			g_string_free (tag_name, TRUE) ;
			tag_name = NULL ;
		}
		status = mlview_utils_parse_closing_tag2 (&cur, &tag_name) ;
		if (status == MLVIEW_OK) {
			/*yes, it is a closing tag. let's push on a stack*/
			gchar *tag_name_str = NULL ;
			tag_name_str = g_strdup (tag_name->str) ;
			closed_tags_stack = mlview_utils_push_on_stack 
				(closed_tags_stack, tag_name_str) ;
			tag_name = NULL ;
		}

		if (!gtk_text_iter_backward_char (&cur)) {
			status = MLVIEW_ERROR ;
			goto cleanup ;
		}
		goto fetch ;
	}

cleanup:
	if (stag_end) {
		gtk_text_iter_free (stag_end) ;
		stag_end = NULL ;
	}
	if (tmp_iter) {
		gtk_text_iter_free (tmp_iter) ;
		tmp_iter = NULL ;
	}
	if (tag_name) {
		g_string_free (tag_name, TRUE) ;
		tag_name = NULL ;
	}
	return status ;

}

/**
 * Gets the source buffer hold by the source view.
 * @param a_this the current instance of #MlViewSourceView
 * @param a_source_buffer, the result
 * @return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
static enum MlViewStatus
get_source_buffer (MlViewSourceView *a_this, 
		   GtkSourceBuffer **a_source_buffer)
{
	GtkTextBuffer *text_buffer = NULL ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this) && a_source_buffer,
			      MLVIEW_BAD_PARAM_ERROR) ;

	g_return_val_if_fail (*a_source_buffer == NULL, 
			      MLVIEW_BAD_PARAM_ERROR) ;

	g_return_val_if_fail 
		(PRIVATE (a_this)->native_sv 
	         && GTK_IS_SOURCE_VIEW (PRIVATE (a_this)->native_sv),
		 MLVIEW_BAD_PARAM_ERROR) ;

	text_buffer = 
		gtk_text_view_get_buffer 
		   (GTK_TEXT_VIEW (PRIVATE (a_this)->native_sv)) ;
	g_return_val_if_fail (text_buffer, MLVIEW_ERROR) ;
	source_buffer = GTK_SOURCE_BUFFER (text_buffer) ;
	g_return_val_if_fail (source_buffer, MLVIEW_ERROR) ;

	*a_source_buffer = source_buffer ;
	return MLVIEW_OK ;
}

/**
 * NOT USED YET:
 * Gets the native source view (instance of GtkSourceView)
 * used internally by the current instance of #MlViewSourceView
 * @param a_this the current instance of #MlViewSourceView
 * @param a_view out parameter. Where to put the native source view
 * @return MLVIEW_OK upon sucessful completion, an error code otherwise
 *
static enum MlViewStatus
get_native_source_view (MlViewSourceView *a_this,
                        GtkSourceView **a_view)
{
	enum MlViewStatus status =  MLVIEW_OK ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
	                      && PRIVATE (a_this), 
			      MLVIEW_BAD_PARAM_ERROR) ;
	*a_view = PRIVATE (a_this)->native_sv ;
	return status ;
}
*/

/**
 * Serializes the document object model into an in memory buffer, and
 * load that in memory buffer into the source view.
 * @param a_this the current instance of #MlViewSourceView
 * @return MLVIEW_OK upon successful completion, an error code otherwise.
 */
static enum MlViewStatus
serialize_and_load_doc (MlViewSourceView *a_this)
{
	MlViewXMLDocument *doc = NULL ;
	GtkSourceBuffer *source_buffer = NULL ;
        gchar *doc_buffer = NULL ;	
	gint nb_bytes_writen = 0 ;
	enum MlViewStatus status = MLVIEW_OK ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;


	mlview_iview_get_document (MLVIEW_IVIEW (a_this), &doc) ;
	if (!doc) {
		return MLVIEW_ERROR ;
	}
	mlview_xml_document_save_xml_doc2 (doc, &doc_buffer,
	                                   &nb_bytes_writen) ;
	if (nb_bytes_writen <= 0 || ! doc_buffer) {
		g_warning ("(nb_bytes_writen <= 0 || !doc_buffer) failed") ;
		status = MLVIEW_OK ;
		goto cleanup ;
	}
	get_source_buffer (a_this, &source_buffer) ;
	if (!source_buffer) { 
		g_warning ("source_buffer failed") ;
		status = MLVIEW_ERROR ;
		goto cleanup ;
	}
        g_return_val_if_fail (source_buffer, MLVIEW_ERROR) ;

	gtk_source_buffer_begin_not_undoable_action (source_buffer) ;
	gtk_text_buffer_set_text (GTK_TEXT_BUFFER (source_buffer), 
				  doc_buffer, nb_bytes_writen) ;
	gtk_source_buffer_end_not_undoable_action (source_buffer) ;

	PRIVATE (a_this)->document_changed = FALSE ;

cleanup:
	if (doc_buffer) {
		g_free (doc_buffer) ;
		doc_buffer = NULL ;
	}
	return status ; 
}

/**
 * Slam the changes that occured into the view into the document object model.
 * @param a_this the current instance of #MlViewSourceView
 * @return MLVIEW_OK upon successful completion, an error code otherwise.
 */
static enum MlViewStatus
save_text_buffer_into_xml_doc (MlViewSourceView *a_this)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkTextIter start = {0} ;
	GtkTextIter end = {0} ;
	gchar *raw_buffer = NULL ;
	MlViewXMLDocument *doc = NULL ;
	GtkSourceBuffer *source_buffer = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	/* do not use the get_document() function here, otherwise,
	 * it can loop forever because that function calls 
	 * serialize_and_load_doc ().
	 */
	doc = PRIVATE (a_this)->xml_doc ;
	g_return_val_if_fail (doc, MLVIEW_BAD_PARAM_ERROR) ;
	get_source_buffer (a_this, &source_buffer) ;  
	g_return_val_if_fail (source_buffer, MLVIEW_ERROR) ;
	gtk_text_buffer_get_iter_at_offset (GTK_TEXT_BUFFER (source_buffer),
					    &start, 0) ;
	gtk_text_buffer_get_iter_at_offset (GTK_TEXT_BUFFER (source_buffer), 
			                    &end, -1) ;
	raw_buffer = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (source_buffer),
			                       &start, &end, TRUE) ;
	g_return_val_if_fail (raw_buffer, MLVIEW_BAD_PARAM_ERROR) ;
	/*slam the changes that occured at view level into the document*/
	status = mlview_xml_document_reload_from_buffer (doc, raw_buffer, 
			                                 TRUE) ;
	if (status == MLVIEW_OK) {
		/*flag the view as being in sync with the doc*/
		PRIVATE (a_this)->view_changed = FALSE ;
	}

/*out:*/
	if (raw_buffer) {
		g_free (raw_buffer) ;
		raw_buffer = NULL ;
	}
	
	return status ; 
}


/**********************************
 *A couple Methods to overide some 
 *of MlViewIView's methods.
 *Normally, MlViewViewAdapter provides
 *default implementations for MlViewIView
 *pure virtual methods. But here, there
 *are some methods that we'd like to override
 **********************************/

/**
 * Overrides the MlViewIView::get_document() methods.
 * Actually, this methods tries to parse the content of
 * the source view buffer, and update the dom before giving it
 * back. Of course, this should be done only if 
 * the buffer is "dirty". Dirty means the content of the buffer
 * hasn't been parsed to update the dom since the user last modified
 * it.
 * @param a_this the current instance of the (source) view
 * @param a_doc the returned xml document
 */
static enum MlViewStatus
get_document (MlViewIView *a_this,
              MlViewXMLDocument **a_doc)
{
	enum MlViewStatus status = MLVIEW_OK ;
	MlViewSourceView *source_view = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
	                      && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	source_view = MLVIEW_SOURCE_VIEW (a_this) ;
	g_return_val_if_fail (source_view, MLVIEW_ERROR) ;

	if (PRIVATE (source_view)->view_changed == TRUE) {
		status = save_text_buffer_into_xml_doc (source_view) ;
	}
	*a_doc = PRIVATE (source_view)->xml_doc ;
	return MLVIEW_OK ;
}

/*********************
 * public methods
 *********************/

GType 
mlview_source_view_get_type (void)
{

	static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewSourceViewClass),
                        NULL, /*base_init*/
                        NULL, /*base_finalize*/
                        (GClassInitFunc)
                        mlview_source_view_class_init,
                        NULL, /*class_finalize*/
                        NULL, /*class data*/
                        sizeof (MlViewSourceView),
                        0, /*n_prealloc*/
                        (GInstanceInitFunc)
                        mlview_source_view_init
                } ;
                type = g_type_register_static
                        (MLVIEW_TYPE_VIEW_ADAPTER,
                         "MlViewSourceView", &type_info, 0);

		static const GInterfaceInfo iview_info = {
			(GInterfaceInitFunc) mlview_source_view_iview_init,
			NULL,
			NULL
		} ;
		g_type_add_interface_static (type, MLVIEW_TYPE_IVIEW,
		                             &iview_info) ;
        }
        return type;
}

GtkWidget *
mlview_source_view_new (MlViewXMLDocument *a_doc,
		        const gchar *a_name,
			MlViewAppContext *a_app_context)
{
	MlViewSourceView *source_view = NULL ;	

	g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
			      && a_app_context, NULL) ;

	source_view = g_object_new (MLVIEW_TYPE_SOURCE_VIEW, NULL) ;
	g_return_val_if_fail (source_view, NULL) ;/*this never fail*/
	mlview_source_view_construct (source_view, a_doc, a_name, a_app_context) ;
	return GTK_WIDGET (source_view) ;
}

enum MlViewStatus
mlview_source_view_set_default_options (MlViewSourceView *a_this)
{

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this) 
			      && PRIVATE (a_this)->native_sv
			      && GTK_IS_SOURCE_VIEW (PRIVATE (a_this)->native_sv),
			      MLVIEW_BAD_PARAM_ERROR) ;

	PRIVATE (a_this)->show_line_numbers = FALSE;
	gtk_source_view_set_show_line_numbers 
		(PRIVATE (a_this)->native_sv,
	         PRIVATE (a_this)->show_line_numbers) ;

	PRIVATE (a_this)->tabs_width = 4 ;
	gtk_source_view_set_tabs_width 
		(PRIVATE (a_this)->native_sv,
		 PRIVATE (a_this)->tabs_width) ;

	PRIVATE (a_this)->set_autoindent = FALSE ;
	gtk_source_view_set_auto_indent (PRIVATE (a_this)->native_sv,
			                 PRIVATE (a_this)->set_autoindent) ;

	PRIVATE (a_this)->set_show_margin = FALSE ;
	gtk_source_view_set_show_margin (PRIVATE (a_this)->native_sv,
			                 PRIVATE (a_this)->set_show_margin) ;
	PRIVATE (a_this)->margin = 2 ;
	gtk_source_view_set_margin (PRIVATE (a_this)->native_sv,
			            PRIVATE (a_this)->margin) ;

	set_default_language (a_this) ;

	return MLVIEW_OK ;
}

enum MlViewStatus
mlview_source_view_construct (MlViewSourceView *a_this,
		               MlViewXMLDocument *a_doc,
			       const gchar *a_view_name,
			       MlViewAppContext *a_app_context)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkSourceBuffer *source_buffer = NULL ;
	GtkWidget *scrolled_win = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
			      && a_app_context 
			      && MLVIEW_IS_APP_CONTEXT (a_app_context),
			      MLVIEW_BAD_PARAM_ERROR) ;

	/*
	 * Now build the native_sv, feed it with the serialized xml,
	 * connect to the appropriate signals.
	 */
	PRIVATE (a_this)->native_sv = GTK_SOURCE_VIEW (gtk_source_view_new ()) ;
	g_return_val_if_fail (PRIVATE (a_this)->native_sv, MLVIEW_ERROR) ;

	status = mlview_iview_set_document (MLVIEW_IVIEW (a_this),
			                    a_doc) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;
	mlview_xml_document_ref (a_doc) ;
	PRIVATE (a_this)->xml_doc = a_doc ;

	mlview_source_view_set_default_options (a_this) ;
	serialize_and_load_doc (a_this) ;

	PRIVATE (a_this)->app_context = a_app_context ;
	/*
	 * Connect to the other signals of MlViewIView
	 */
	g_signal_connect (G_OBJECT (a_this),
			  "is-swapped-in",
			  G_CALLBACK (is_swapped_in_cb),
			  a_doc) ; 
	g_signal_connect (G_OBJECT (a_this),
			  "is-swapped-out",
			  G_CALLBACK (is_swapped_out_cb),
			  a_doc) ; 
	get_source_buffer (a_this, &source_buffer) ;
	
	if (source_buffer) {
		g_signal_connect (source_buffer,
				  "changed",
				  G_CALLBACK (changed_cb),
				  a_this) ;
		g_signal_connect (source_buffer,
		                  "can-undo",
				  G_CALLBACK (undo_state_changed_cb),
				  a_this) ;
		g_signal_connect (source_buffer,
		                  "can-redo",
				  G_CALLBACK (undo_state_changed_cb),
				  a_this) ;
	} else {
		mlview_utils_trace_debug ("Could not get source buffer") ;
	}

	/*
	 * connect to the MlViewAppContext signals
	 */
	 g_signal_connect (G_OBJECT (a_app_context),
	                   "contextual-menu-requested",
			   G_CALLBACK (contextual_menu_requested_cb),
			   a_this) ;


	scrolled_win = gtk_scrolled_window_new (NULL, NULL) ;
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
			                GTK_POLICY_AUTOMATIC, 
					GTK_POLICY_AUTOMATIC) ;
	
	gtk_box_pack_start (GTK_BOX (a_this), 
			    scrolled_win, TRUE, TRUE, 0) ;
	gtk_container_add (GTK_CONTAINER (scrolled_win),
			   GTK_WIDGET (PRIVATE (a_this)->native_sv)) ;

	mlview_iview_connect_to_doc (MLVIEW_IVIEW (a_this), a_doc) ;

	g_signal_connect (G_OBJECT (PRIVATE (a_this)->native_sv),
	                  "realize",
			  G_CALLBACK (widget_realized_cb),
			  a_this) ;
	gtk_widget_show_all (GTK_WIDGET (PRIVATE (a_this)->native_sv)) ;
	gtk_widget_show (GTK_WIDGET (a_this)) ;
	return MLVIEW_OK ;
}

MlViewAppContext * 
mlview_source_view_get_application_context (MlViewSourceView *a_this)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this), NULL) ;
	return PRIVATE (a_this)->app_context ;
}

/**
 * Gets the instance of GtkUIManager associated to the current instance of
 * #MlViewSourceView. As of today, it's the global one used across all the
 * application. It has just been cached here.
 * @param a_this the current instance of #MlViewSourceView.
 * @return the instance of GtkUIManager associated to the sourceview, or NULL
 * if something bad happened.
 */
GtkUIManager * 
mlview_source_view_get_ui_manager (MlViewSourceView *a_this)
{
	MlViewAppContext *app_context = NULL ;
	GtkActionGroup *action_group = NULL ;
	gchar *file_path = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this)
			      && PRIVATE (a_this),
			      NULL) ;

	if (!PRIVATE (a_this)->ui_manager) {
		/*
		 * get the application wide ui manager from the 
		 * application context.
		 *
		 */
		app_context = PRIVATE (a_this)->app_context ;
		g_return_val_if_fail (app_context, NULL) ;
		PRIVATE (a_this)->ui_manager = mlview_app_context_get_element 
			(app_context, "MlViewUIManager") ; ;
		g_return_val_if_fail (PRIVATE (a_this)->ui_manager, NULL) ;
		/*
		 * check if an action group of name "SourceViewEditMenuAction"
		 * has been inserted into the ui manager. If yes, use that one.
		 * If not, create a new one.
		 */
		mlview_utils_lookup_action_group (PRIVATE (a_this)->ui_manager,
				                  "SourceViewEditMenuActions",
						  &action_group) ;
		if (!action_group) {
			action_group = gtk_action_group_new 
				("SourceViewEditMenuActions") ;
			gtk_action_group_set_translation_domain 
				(action_group, GETTEXT_PACKAGE) ;

			gtk_action_group_add_actions 
			(action_group, 
			 gv_edit_menu_actions,
			 sizeof (gv_edit_menu_actions)/sizeof (GtkActionEntry),
			 a_this) ;

			PRIVATE (a_this)->action_group = action_group ;
			gtk_ui_manager_insert_action_group 
				(PRIVATE (a_this)->ui_manager,
			         action_group, 0);
		}
		file_path = mlview_utils_locate_file ("source-view-edit-menu.xml") ;
		g_return_val_if_fail (file_path, NULL) ;
		gtk_ui_manager_add_ui_from_file (PRIVATE (a_this)->ui_manager,
				              file_path, 0) ;
		if (file_path) {
			g_free (file_path) ;
			file_path = NULL ;
		}
	}
	return PRIVATE (a_this)->ui_manager ;
}

enum MlViewStatus
mlview_source_view_is_there_an_opened_tag (MlViewSourceView *a_this,
		                           gboolean *a_answer) 
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkTextIter iter = {0} ;
	gchar *tag_name = NULL ; 

	g_return_val_if_fail (a_this && MLVIEW_IS_SOURCE_VIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	status = get_last_dangling_opened_tag (a_this, &iter,
			                       &tag_name) ;
	g_return_val_if_fail (status == MLVIEW_OK, status) ;
	if (tag_name) {
		*a_answer = TRUE ;
	} else {
		*a_answer = FALSE ; 	
	}
	if (tag_name) {
		g_free (tag_name) ;
		tag_name = NULL ;
	}
	return MLVIEW_OK ;
}

/**
 * Closes the last opened dangling tag in the opened tag stack.
 * @param a_this the current instance of #MlViewSourceView .
 * @param a_has_been_closed if the pointer is non null, the pointed
 * value is set to TRUE if a dangling tag has effectively been closed,
 * FALSE otherwise.
 * @return MLVIEW_OK upon successful completion, an error code otherwise.
 * Note that if the opened tag stack is empty (no dangling opened tag is 
 * present in the buffer) then the function won't close any tag, obviously.
 * the completion will nevertheless be considered as successful and this
 * function will still return MLVIEW_OK. An error message really means 
 * "something bad happened.".
 */
enum MlViewStatus
mlview_source_view_close_currently_opened_tag (MlViewSourceView *a_this,
                                               gboolean *a_has_been_closed)
{
	enum MlViewStatus status = MLVIEW_OK ;
	GtkTextIter iter = {0} , insert_iter = {0} ;
	GtkTextMark *text_mark = NULL ;
	GtkTextBuffer *text_buffer = NULL, **text_buffer_ptr = NULL;
	gchar *tag_name = NULL, *closing_tag = NULL ;
	gint len = 0 ;

	status = get_last_dangling_opened_tag (a_this, &iter,
			                      &tag_name) ;
	if (status != MLVIEW_OK || !tag_name) {
		if (a_has_been_closed)
			*a_has_been_closed = FALSE ;
		return MLVIEW_OK ;
	}

	/*
	 * get an iter at the current pointer location.
	 */
	text_buffer_ptr = &text_buffer ;
	status = get_source_buffer (a_this, (GtkSourceBuffer**)text_buffer_ptr) ;
	g_return_val_if_fail ((status == MLVIEW_OK) && text_buffer,
			      MLVIEW_ERROR)  ;
	text_mark = gtk_text_buffer_get_insert (text_buffer) ;
	g_return_val_if_fail (text_mark, MLVIEW_ERROR) ;
	gtk_text_buffer_get_iter_at_mark (text_buffer, &insert_iter, text_mark) ;
	/*No need to delete special mark "insert"*/
	text_mark = NULL ;

	/*
	 * build the closing tag 
	 */
	closing_tag = g_strjoin ("", "</", tag_name, ">", NULL) ;
	len = strlen (closing_tag) ;
	gtk_text_buffer_insert (GTK_TEXT_BUFFER (text_buffer),
			        &insert_iter, closing_tag, len) ;
	if (closing_tag) {
		g_free (closing_tag) ;
		closing_tag = NULL ;
	}
	if (a_has_been_closed)
		*a_has_been_closed = TRUE ;
	return MLVIEW_OK ;
}
