// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - file-utils.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <glib.h>

#ifdef HAVE_GIO
#include <gio/gfile.h>
#else
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <glibmm-utils/date.h>
#endif // HAVE_GIO

#include <glibmm/convert.h>
#include "file-utils.hh"

namespace paperbox {

    bool
    open_file_with_xdg (const Glib::ustring& file)
    {
	bool   result;
	gchar* quoted_filename = g_shell_quote (file.c_str());
	gchar* command = g_strconcat ("xdg-open ", quoted_filename, NULL);
        
	g_free (quoted_filename);
	result = g_spawn_command_line_async (command, NULL);
	g_free (command);

	return result;
    }

    Glib::ustring
    get_file_modification_time(const Glib::ustring& uri)
    {
        Glib::ustring res;

#ifdef HAVE_GIO
        GFile* file;
        GFileQueryInfoFlags flags;
        GFileInfo* info;
        GError* error = 0;

        file = g_file_new_for_path (uri.c_str());

        info = g_file_query_info (file,
                                  G_FILE_ATTRIBUTE_TIME_MODIFIED,
                                  flags,
                                  NULL,
                                  &error);

        if (info == NULL) {
            g_printerr ("Error getting time info: %s\n", error->message);
            g_error_free (error);
            return res;
        }

        const GFileAttributeValue *value;

        value = g_file_info_get_attribute (info,
                                           G_FILE_ATTRIBUTE_TIME_MODIFIED);

        struct timeval tv;
        struct tm* ptm;
        char time_string[256];

        tv.tv_sec = g_file_attribute_value_get_uint64 (value);

        ptm = localtime(&tv.tv_sec);
        strftime(time_string, sizeof(time_string), "%c", ptm);

        res = Glib::locale_to_utf8(time_string);
#else
        GnomeVFSFileInfo* info = NULL;
        GnomeVFSResult result;

        info = gnome_vfs_file_info_new();
        result = gnome_vfs_get_file_info (uri.c_str(),
                                          info,
                                          GNOME_VFS_FILE_INFO_DEFAULT);

        if (result == GNOME_VFS_OK)
            Glib::Util::get_modification_date(info->mtime, res);
        else
            g_warning("no file info for %s: %s",
                      uri.c_str(),
                      gnome_vfs_result_to_string (result));

        gnome_vfs_file_info_unref(info);
#endif // HAVE_GIO

        return res;
    }

} // namespace paperbox
