// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - browser.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <algorithm>
#include <iostream>
#include <boost/bind.hpp>
#include <glibmm/fileutils.h>
#include <glibmm/main.h>
#include <glibmm-utils/log-stream-utils.h>
#include "browser.hh"

namespace paperbox {

    using std::auto_ptr;
    using std::list;
    using std::vector;
    using boost::shared_ptr;
    using Glib::ustring;

    ///

    struct doc_mtime_compare_desc
        : public std::binary_function<shared_ptr<Document>,
                                      shared_ptr<Document>,
                                      bool>
    {
        bool
        operator()(const shared_ptr<Document>& lhs,
                   const shared_ptr<Document>& rhs) const
            {
                return (lhs->get_mtime() > rhs->get_mtime());
            }
    };

    ///

    auto_ptr<Browser> Browser::instance_;

    Browser*
    Browser::instance()
    {
        if (! instance_.get())
            instance_.reset(new Browser());

        return instance_.get();
    }

    Browser::Browser()
    {
        tracker_client_.reset(new TrackerPhone()); // let exception propagate

        connect_to_tracker_signals();
        tracker_client_->request_all_documents();
    }

    Browser::~Browser()
    {
    }

    void
    Browser::connect_to_tracker_signals()
    {
        tracker_client_->signal_documents().connect(
            sigc::mem_fun(*this, &Browser::on_documents));

        tracker_client_->signal_tags_added().connect(
            sigc::mem_fun(*this, &Browser::on_tags_added));

        tracker_client_->signal_tags_removed().connect(
            sigc::mem_fun(*this, &Browser::on_tags_removed));
    }

    void
    Browser::on_documents(const std::queue<ustring>& uris)
    {
        uri_queue_ = uris;

        // begin retrieving information
        Glib::signal_idle().connect(
            sigc::mem_fun(*this, &Browser::on_idle_initial_document_retrieval));
    }

    void
    Browser::dump_document_data()
    {
        std::for_each(docs_.begin(),
                      docs_.end(),
                      boost::bind(
                          &Document::print,
                          boost::bind(&doc_map::value_type::second, _1)));
    }

    bool
    Browser::on_idle_initial_document_retrieval()
    {
        if (uri_queue_.empty()) {
            // The queue being empty at the beginning and not after a pop()
            // means that  either there is nothing on the desktop,
            // or trackerd is (re)indexing and being unresponsive.
            LOG_DD("Cancelling document retrieval from tracker - "
                   << "is it (re)indexing? "
                   << "Try running PaperBox again after a minute.");
            return false; // idle function done
        }

        ustring uri = uri_queue_.front();

        // get metadata, emit for new documents
        try {
            shared_ptr<Document> doc;
            tracker_client_->get_document(uri, doc);

            docs_[uri.raw()] = doc;
            signal_new_document_.emit(doc);
        }
        catch (const Glib::FileError& ex) {
            LOG_EXCEPTION(uri << ": " << ex.what());
        }

        uri_queue_.pop();

        if (uri_queue_.empty())
            return false; // done
        else
            return true;
    }

    Browser::SignalNewDocument&
    Browser::signal_new_document()
    {
        return signal_new_document_;
    }

    Browser::SignalTagsChanged&
    Browser::signal_tags_changed()
    {
        return signal_tags_changed_;
    }

    void
    Browser::on_tags_added(const ustring& uri,
                           const vector<ustring>& tags)
    {
        doc_map::iterator doc_iter = docs_.find(uri.raw());
        shared_ptr<Document> doc = doc_iter->second;
        // Ignore if a mime type which not in our index has been tagged
        if (! doc.get()) return;

        vector<ustring>::const_iterator it(tags.begin());
        vector<ustring>::const_iterator end(tags.end());

        for ( ; it != end; ++it) doc->add_tag(*it);

        vector<ustring> tags_removed_none;
        signal_tags_changed_.emit(uri, tags, tags_removed_none);
    }

    // Does the validation work and forwards the add request to TrackerPhone
    void
    Browser::add_tags(const ustring& uri, const vector<ustring>& tags)
    {
        doc_map::iterator doc_iter = docs_.find(uri.raw());

        if (doc_iter == docs_.end() || tags.empty()) return;

        // only include the tags that do not intersect with the existing
        shared_ptr<Document> doc = doc_iter->second;
        vector<ustring> existing_tags = doc->get_tags();
        vector<ustring> tags_copy(tags);

        vector<ustring>::iterator it(existing_tags.begin());
        vector<ustring>::iterator end(existing_tags.end());
        for ( ; it != end; ++it) {
            tags_copy.erase(
                std::remove(tags_copy.begin(), tags_copy.end(), *it),
                tags_copy.end());
        }

        if (tags_copy.empty()) return;

        tracker_client_->add_tags(uri, tags_copy);
    }

    void
    Browser::on_tags_removed(const ustring& uri,
                             const vector<ustring>& tags)
    {
        doc_map::iterator doc_iter = docs_.find(uri.raw());
        shared_ptr<Document> doc = doc_iter->second;
        // Ignore if a mime type which not in our index has been un-tagged
        if (! doc.get()) return;

        vector<ustring>::const_iterator it(tags.begin());
        vector<ustring>::const_iterator end(tags.end());

        for ( ; it != end; ++it) doc->remove_tag(*it);

        vector<ustring> tags_added_none;
        signal_tags_changed_.emit(uri, tags_added_none, tags);
    }

    // Does the validation work and forwards the remove request to TrackerPhone
    void
    Browser::remove_tags(const ustring& uri,
                         const std::vector<ustring>& tags)
    {
        doc_map::iterator doc_iter = docs_.find(uri.raw());

        if (doc_iter == docs_.end() || tags.empty()) return;

        // only include the tags that intersect with the existing
        shared_ptr<Document> doc = doc_iter->second;
        vector<ustring> existing_tags = doc->get_tags();
        vector<ustring> tags_copy(tags);

        vector<ustring>::iterator it(tags_copy.begin());
        vector<ustring>::iterator end(tags_copy.end());
        for ( ; it != end; ++it) {
            if (find(existing_tags.begin(), existing_tags.end(), *it) ==
                existing_tags.end()) {
                // invalid, not found
                tags_copy.erase(
                    std::remove(tags_copy.begin(), tags_copy.end(), *it),
                    tags_copy.end());
            }
        }

        if (tags_copy.empty()) return;

        tracker_client_->remove_tags(uri, tags_copy);
    }

    void
    Browser::rename_tag(const ustring& /*uri*/,
                        const ustring& /*from_tag*/,
                        const ustring& /*to_tag*/)
    {
        //TODO later as a high-level functionality, in UI it should
        // not be strictly bound to one document. It should rather be
        // done in some kind of a manage-tags dialog.
        g_debug("rename_tag not implemented yet");
    }

    void
    Browser::get_all_documents(doc_vector& docs)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it)
            docs.push_back(it->second);
    }

    void
    Browser::get_all_documents(list<shared_ptr<Document> >& docs)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it)
            docs.push_back(it->second);
    }

    void
    Browser::get_recent_documents(doc_vector& docs, int count)
    {
        list<shared_ptr<Document> > ldocs;
        get_all_documents(ldocs);

        // sort by modtime, descending
        ldocs.sort(doc_mtime_compare_desc());

        // copy and return
        list<shared_ptr<Document> >::iterator it(ldocs.begin());
        list<shared_ptr<Document> >::iterator end(ldocs.end());
        for ( int i = 0; (it != end) && (i < count); ++it) {
            docs.push_back(*it);
            ++i;
        }
    }

    void
    Browser::get_untagged_documents(doc_vector& docs)
    {
        list<shared_ptr<Document> > all_docs;
        get_all_documents(all_docs);

        list<shared_ptr<Document> >::iterator it(all_docs.begin());
        list<shared_ptr<Document> >::iterator end(all_docs.end());

        for ( ; it != end; ++it)
            if (! (*it)->get_tags().size())
                docs.push_back(*it);
    }

    void
    Browser::get_documents_for_tag(const ustring& tag,
                                   doc_vector& docs_ret)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it) {
            shared_ptr<Document> doc = it->second;
            if (doc->contains_tag(tag))
                docs_ret.push_back(doc);
        }
    }

    void
    Browser::get_documents_for_tag_bundle(const vector<ustring>& tags,
                                          doc_vector& docs_ret)
    {
        doc_map::iterator it(docs_.begin());
        doc_map::iterator end(docs_.end());

        for ( ; it != end; ++it) {
            shared_ptr<Document> doc = it->second;

            vector<ustring>::const_iterator tit(tags.begin());
            vector<ustring>::const_iterator tend(tags.end());
            for ( ; tit != tend; ++tit) {
                if (doc->contains_tag(*tit)) {
                    docs_ret.push_back(doc);
                    break;
                }
            }
        }
    }

} // namespace paperbox
