/******************************************************************************
 JTEUndoTabShift.cc

	Class to undo indenting a block of text.

	BASE CLASS = JTEUndoBase

	Copyright  1996-97 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JTEUndoTabShift.h>
#include <JTextEditor.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JTEUndoTabShift::JTEUndoTabShift
	(
	JTextEditor*	te,
	const JIndex	startLine
	)
	:
	JTEUndoBase(te)
{
	itsStartLine  = startLine;
	itsEndChar    = te->GetLineStart(startLine);
	itsShiftCount = 0;
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JTEUndoTabShift::~JTEUndoTabShift()
{
}

/******************************************************************************
 Undo (virtual)

 ******************************************************************************/

void
JTEUndoTabShift::Undo()
{
	JTextEditor* te = GetTE();
	te->SetSelection(te->GetLineStart(itsStartLine), itsEndChar);

	if (itsShiftCount > 0)
		{
		te->TabSelectionLeft(itsShiftCount);		// deletes us
		}
	else if (itsShiftCount < 0)
		{
		te->TabSelectionRight(-itsShiftCount);		// deletes us
		}
	else if (itsShiftCount == 0)
		{
		te->SameUndo(this);
		}
}

/******************************************************************************
 HandleShiftLeft

 ******************************************************************************/

void
JTEUndoTabShift::HandleShiftLeft
	(
	const JSize tabCount
	)
{
	assert( IsActive() );

	itsShiftCount -= (JInteger) tabCount;
}

/******************************************************************************
 HandleShiftRight

 ******************************************************************************/

void
JTEUndoTabShift::HandleShiftRight
	(
	const JSize tabCount
	)
{
	assert( IsActive() );

	itsShiftCount += (JInteger) tabCount;
}

/******************************************************************************
 UpdateEndChar

	Call this after SetSelection() to update the end of the affected text.
	(We can't store a line number, because this may change when word
	 wrapping is on.)

 ******************************************************************************/

void
JTEUndoTabShift::UpdateEndChar()
{
	Activate();		// cancel SetSelection()

	JIndex startChar;
	const JBoolean ok = (GetTE())->GetSelection(&startChar, &itsEndChar);
	assert( ok );
}

/******************************************************************************
 Cast to JTEUndoTabShift*

	Not inline because they are virtual

 ******************************************************************************/

JTEUndoTabShift*
JTEUndoTabShift::CastToJTEUndoTabShift()
{
	return this;
}

const JTEUndoTabShift*
JTEUndoTabShift::CastToJTEUndoTabShift()
	const
{
	return this;
}
