/******************************************************************************
 JTableData.cc

	Abstract class for data in some kind of 2D arrangement.  The common
	ground for TableData objects is that the data is accessed by giving
	the row and column indices.

	BASE CLASS = virtual JBroadcaster

	Copyright  1996 John Lindal. All rights reserved.

 ******************************************************************************/

#include <JTableData.h>
#include <JOrderedSetUtil.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JTableData::JTableData()
	:
	JBroadcaster()
{
	itsRowCount = itsColCount = 0;
}

/******************************************************************************
 Copy constructor

 ******************************************************************************/

JTableData::JTableData
	(
	const JTableData& source
	)
	:
	JBroadcaster(source)
{
	itsRowCount = source.itsRowCount;
	itsColCount = source.itsColCount;
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JTableData::~JTableData()
{
}

/******************************************************************************
 Assignment operator

 ******************************************************************************/

const JTableData&
JTableData::operator=
	(
	const JTableData& source
	)
{
	if (this == &source)
		{
		return *this;
		}

	JBroadcaster::operator=(source);

	itsRowCount = source.itsRowCount;
	itsColCount = source.itsColCount;

	return *this;
}

/******************************************************************************
 RowDeleted (protected)

	We use "Deleted" to avoid conflicting with RowRemoved message.

	Not inline because it uses assert

 ******************************************************************************/

void
JTableData::RowDeleted()
{
	assert( itsRowCount > 0 );
	itsRowCount--;
}

/******************************************************************************
 ColDeleted (protected)

	We use "Deleted" to avoid conflicting with ColRemoved message.

	Not inline because it uses assert

 ******************************************************************************/

void
JTableData::ColDeleted()
{
	assert( itsColCount > 0 );
	itsColCount--;
}

/******************************************************************************
 JBroadcaster messages

 ******************************************************************************/

// message types

const JCharacter* JTableData::kRowInserted    = "RowInserted::JTableData";
const JCharacter* JTableData::kRowDuplicated  = "RowDuplicated::JTableData";
const JCharacter* JTableData::kRowRemoved     = "RowRemoved::JTableData";
const JCharacter* JTableData::kAllRowsRemoved = "AllRowsRemoved::JTableData";
const JCharacter* JTableData::kRowMoved       = "RowMoved::JTableData";

const JCharacter* JTableData::kColInserted    = "ColInserted::JTableData";
const JCharacter* JTableData::kColDuplicated  = "ColDuplicated::JTableData";
const JCharacter* JTableData::kColRemoved     = "ColRemoved::JTableData";
const JCharacter* JTableData::kAllColsRemoved = "AllColsRemoved::JTableData";
const JCharacter* JTableData::kColMoved       = "ColMoved::JTableData";

const JCharacter* JTableData::kRectChanged    = "RectChanged::JTableData";

/******************************************************************************
 AdjustIndex

 ******************************************************************************/

void
JTableData::RowInserted::AdjustIndex
	(
	JIndex* index
	)
	const
{
	JAdjustIndexAfterInsert(GetIndex(), 1, index);
}

JBoolean
JTableData::RowRemoved::AdjustIndex
	(
	JIndex* index
	)
	const
{
	return JAdjustIndexAfterRemove(GetIndex(), 1, index);
}

void
JTableData::ColInserted::AdjustIndex
	(
	JIndex* index
	)
	const
{
	JAdjustIndexAfterInsert(GetIndex(), 1, index);
}

JBoolean
JTableData::ColRemoved::AdjustIndex
	(
	JIndex* index
	)
	const
{
	return JAdjustIndexAfterRemove(GetIndex(), 1, index);
}

void
JTableData::RowColDupd::AdjustIndex
	(
	JIndex* index
	)
	const
{
	JAdjustIndexAfterInsert(itsNewIndex, 1, index);
}

void
JTableData::RowColMoved::AdjustIndex
	(
	JIndex* index
	)
	const
{
	JAdjustIndexAfterMove(itsOrigIndex, itsNewIndex, index);
}
