/******************************************************************************
 JUNIXDirInfo.h

	Interface for the JUNIXDirInfo class

	Copyright  1996 by Glenn W. Bach. All rights reserved.
	Copyright  1997-98 by John Lindal. All rights reserved.

 ******************************************************************************/

#ifndef _H_JUNIXDirInfo
#define _H_JUNIXDirInfo

#include <JContainer.h>
#include <JPtrArray.h>
#include <JError.h>
#include <jTime.h>

class JRegex;
class JString;
class JUNIXDirEntry;
class JProgressDisplay;

typedef JOrderedSetT::CompareResult
	(JCompareUNIXEntries)(JUNIXDirEntry * const &, JUNIXDirEntry * const &);

class JUNIXDirInfo : public JContainer
{
public:

	static JBoolean	Create(const JCharacter* dirName, JUNIXDirInfo** obj);
	static JBoolean	OKToCreateUNIXDirInfo(const JCharacter* dirName);

	JUNIXDirInfo(const JUNIXDirInfo& source);

	virtual ~JUNIXDirInfo();

	const JUNIXDirInfo& operator=(const JUNIXDirInfo& source);

	const JString&	GetCWD() const;
	JBoolean		IsWritable() const;

	JError	GoUp();
	JError	GoDown(const JCharacter* dirName);
	void	GoToClosest(const JCharacter* dirName);
	JError	GoTo(const JCharacter* dirName);

	void		ChangeSort(JCompareUNIXEntries* f, const JOrderedSetT::SortOrder order);

	JBoolean	HiddenVisible() const;
	void		ShowHidden(const JBoolean showHidden);

	JBoolean	DirsVisible() const;
	void		ShowDirs(const JBoolean showDirs);

	JBoolean	FilesVisible() const;
	void		ShowFiles(const JBoolean showFiles);

	JBoolean	OthersVisible() const;
	void		ShowOthers(const JBoolean showOthers);

	JBoolean	HasWildcardFilter() const;
	void		SetWildcardFilter(const JCharacter* filterStr);
	void		ClearWildcardFilter();

	JBoolean	HasContentFilter() const;
	JError		SetContentFilter(const JCharacter* regexStr);
	void		ClearContentFilter();

	void		ChangeProgressDisplay(JProgressDisplay* pg);
	void		UseDefaultProgressDisplay();

	JBoolean	Update();			// updates if necessary
	void		ForceUpdate();		// updates regardless

	JSize					GetEntryCount() const;
	const JUNIXDirEntry&	GetEntry(const JIndex index) const;
	JBoolean				FindEntry(const JCharacter* name, JIndex* index) const;
	JBoolean				ClosestMatch(const JCharacter* prefixStr, JIndex* index) const;

	static JBoolean	BuildRegexFromWildcardFilter(const JCharacter* filterStr,
												 JString* regexStr);

protected:

	JUNIXDirInfo(const JCharacter* dirName);

	virtual JBoolean	IsVisible(const JUNIXDirEntry& entry) const;
	JBoolean			MatchesNameFilter(const JUNIXDirEntry& entry) const;
	JBoolean			MatchesContentFilter(const JUNIXDirEntry& entry) const;

private:

	JString*	itsCWD;
	JBoolean	itsIsWritableFlag;
	time_t		itsModTime;
	time_t		itsStatusTime;

	JBoolean	itsShowHiddenFlag;
	JBoolean	itsShowDirsFlag;
	JBoolean	itsShowFilesFlag;
	JBoolean	itsShowOthersFlag;
	JRegex*		itsNameRegex;		// can be NULL
	JRegex*		itsContentRegex;	// can be NULL

	JPtrArray<JUNIXDirEntry>*	itsDirEntries;		// everything, current sorting fn
	JPtrArray<JUNIXDirEntry>*	itsVisEntries;		// visible, current sorting fn
	JPtrArray<JUNIXDirEntry>*	itsAlphaEntries;	// visible, sorted alphabetically

	JProgressDisplay*	itsPG;		// can be NULL

private:

	void	CopyLists(const JUNIXDirInfo& source);

	JError	BuildInfo();
	void	ApplyFilters(const JBoolean update);

	static void	AppendRegex(const JCharacter* origStr, JString* regexStr);

public:

	// JBroadcaster messages

	static const JCharacter* kContentsWillBeUpdated;
	static const JCharacter* kContentsChanged;
	static const JCharacter* kPathChanged;
	static const JCharacter* kPermissionsChanged;

	class ContentsWillBeUpdated : public JBroadcaster::Message
		{
		public:

			ContentsWillBeUpdated()
				:
				JBroadcaster::Message(kContentsWillBeUpdated)
				{ };
		};

	class ContentsChanged : public JBroadcaster::Message
		{
		public:

			ContentsChanged()
				:
				JBroadcaster::Message(kContentsChanged)
				{ };
		};

	class PathChanged : public JBroadcaster::Message
		{
		public:

			PathChanged()
				:
				JBroadcaster::Message(kPathChanged)
			{ };
		};

	class PermissionsChanged : public JBroadcaster::Message
		{
		public:

			PermissionsChanged()
				:
				JBroadcaster::Message(kPermissionsChanged)
			{ };
		};
};


/******************************************************************************
 GetCWD

 ******************************************************************************/

inline const JString&			 
JUNIXDirInfo::GetCWD()
	const
{
	return *itsCWD;
}

/******************************************************************************
 Filters

 ******************************************************************************/

inline JBoolean
JUNIXDirInfo::HiddenVisible()
	const
{
	return itsShowHiddenFlag;
}

inline JBoolean
JUNIXDirInfo::DirsVisible()
	const
{
	return itsShowDirsFlag;
}

inline JBoolean
JUNIXDirInfo::FilesVisible()
	const
{
	return itsShowFilesFlag;
}

inline JBoolean
JUNIXDirInfo::OthersVisible()
	const
{
	return itsShowOthersFlag;
}

inline JBoolean
JUNIXDirInfo::HasWildcardFilter()
	const
{
	return JNegate( itsNameRegex == NULL );
}

inline JBoolean
JUNIXDirInfo::HasContentFilter()
	const
{
	return JNegate( itsContentRegex == NULL );
}

/******************************************************************************
 IsWritable

 ******************************************************************************/

inline JBoolean
JUNIXDirInfo::IsWritable()
	const
{
	return itsIsWritableFlag;
}

/******************************************************************************
 GetEntryCount

 ******************************************************************************/

inline JSize
JUNIXDirInfo::GetEntryCount()
	const
{
	return GetElementCount();
}

/******************************************************************************
 GetEntry

 ******************************************************************************/

inline const JUNIXDirEntry&
JUNIXDirInfo::GetEntry
	(
	const JIndex index
	)
	const
{
	return *(itsVisEntries->NthElement(index));
}

#endif
