/******************************************************************************
 JXKeyModifiers.cc

	Class to represent the states of the modifiers keys that X defines.

	Copyright  1996 John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXKeyModifiers.h>
#include <jAssert.h>

/******************************************************************************
 Constructor

 ******************************************************************************/

JXKeyModifiers::JXKeyModifiers()
{
	Clear();
}

JXKeyModifiers::JXKeyModifiers
	(
	const unsigned int state
	)
{
	SetState(state);
}

/******************************************************************************
 GetState

	Return the state of the specified key.

 ******************************************************************************/

JBoolean
JXKeyModifiers::GetState
	(
	const JIndex i
	)
	const
{
	assert( 0 < i && i <= kXModifierCount );
	return itsState[i-1];
}

/******************************************************************************
 AllOff

 ******************************************************************************/

JBoolean
JXKeyModifiers::AllOff()
	const
{
	for (JIndex i=0; i<kXModifierCount; i++)
		{
		if (itsState[i])
			{
			return kFalse;
			}
		}
	return kTrue;
}

/******************************************************************************
 SetState

	Set the state of the specified key.

 ******************************************************************************/

void
JXKeyModifiers::SetState
	(
	const JIndex	i,
	const JBoolean	pushed
	)
{
	assert( 0 < i && i <= kXModifierCount );
	itsState[i-1] = pushed;
}

/******************************************************************************
 GetState

	Return the state as a bit vector.

 ******************************************************************************/

unsigned int
JXKeyModifiers::GetState()
	const
{
	unsigned int state = 0;
	for (JIndex i=0; i<kXModifierCount; i++)
		{
		if (itsState[i])
			{
			state |= (1L << i);
			}
		}
	return state;
}

/******************************************************************************
 SetState

	Set the state of all the keys.

 ******************************************************************************/

void
JXKeyModifiers::SetState
	(
	const unsigned int state
	)
{
	for (JIndex i=0; i<kXModifierCount; i++)
		{
		itsState[i] = JConvertToBoolean( (state & (1L << i)) != 0 );
		}
}

/******************************************************************************
 Clear

 ******************************************************************************/

void
JXKeyModifiers::Clear()
{
	for (JIndex i=0; i<kXModifierCount; i++)
		{
		itsState[i] = kFalse;
		}
}

/******************************************************************************
 GetState (static)

	Get the state of the given key in the given bit vector.

 ******************************************************************************/

JBoolean
JXKeyModifiers::GetState
	(
	const unsigned int	state,
	const JIndex		i
	)
{
	assert( 0 < i && i <= kXModifierCount );
	return JConvertToBoolean( (state & (1L << (i-1))) != 0 );
}

/******************************************************************************
 SetState (static)

	Set the state of the given key in the given bit vector.

 ******************************************************************************/

unsigned int
JXKeyModifiers::SetState
	(
	const unsigned int	state,
	const JIndex		i,
	const JBoolean		pushed
	)
{
	if (pushed)
		{
		return (state | (1L << (i-1)));
		}
	else
		{
		return (state & ~(1L << (i-1)));
		}
}
