/* $Id: ImageCache.c,v 1.1.1.1 2001/04/09 13:36:42 glgay Exp $ */
/*
 Copyright (C) 1996 Peter Williams

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

#include <Xm/XmP.h>
#include <Xmext/ImageCache.h>
#include <Xmext/XmextXpm.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* 
*  If it matches the name then fine grab it.
*  FIX ME:
*  An optional width, height, and depth should be added
*  for exactness is desired.
*/
struct _XmextImageCacheRec 
{
   char *name;
   Pixmap pixmap; 
};
typedef struct _XmextImageCacheRec XmextImageCacheRec, *XmextImageCache;

static XmextImageCache _image_cache = NULL; 
static int _image_cache_size = 0;

static char * _search_path = NULL;

static char * _search_type[] = { "pixmaps", 
                                 "icons", 
                                 "mini-icons",
                                 "bitmaps" };

static char *_set_pattern = "%%B:"
                            "%s/%%L/%%T/%%N/%%B:"
                            "%s/%%l/%%T/%%N/%%B:"
                            "%s/%%T/%%N/%%B:"
                            "%s/%%L/%%T/%%B:"
                            "%s/%%l/%%T/%%B:"
                            "%s/%%T/%%B:"
                            "%s/%%T/%%B:"
                            "%s/%%B:"
                            "/usr/lib/X11/%%L/%%T/%%N/%%B:"
                            "/usr/lib/X11/%%l/%%T/%%N/%%B:"
                            "/usr/lib/X11/%%T/%%N/%%B:"
                            "/usr/lib/X11/%%L/%%T/%%B:"
                            "/usr/lib/X11/%%l/%%T/%%B:"
                            "/usr/lib/X11/%%T/%%B:"
                            "/usr/include/X11/%%T/%%B";

static char *
_internal_strdup(char *str)
{
   char *tmp;
   if(str == NULL) 
   { 
      return NULL; 
   }
   tmp = (char *) XtMalloc(sizeof(char)*strlen(str)+1);
   return strcpy(tmp,str);
}

static void
__XmextCreateSearchPath()
{
   char *XBMLANGPATH;
   char *XAPPLRESDIR;

   XBMLANGPATH = _internal_strdup((char *)getenv("XBMLANGPATH"));
   XAPPLRESDIR = _internal_strdup((char *)getenv("XAPPLRESDIR"));

   if(XBMLANGPATH)
   {
      _search_path = XBMLANGPATH;

      if(XAPPLRESDIR)
      {
         XtFree(XAPPLRESDIR);
      }
   }
   else 
   {
      char *HOME;
      int HOME_len;

      HOME = (char *)getenv("HOME");
      HOME_len = strlen(HOME);

      if(XAPPLRESDIR)
      {
         int XAPPLRESDIR_len;

         XAPPLRESDIR_len = strlen(XAPPLRESDIR);

         _search_path = (char*)XtMalloc(strlen(_set_pattern)
                                       + XAPPLRESDIR_len * 6
                                       + HOME_len * 2 + 1);

         sprintf(_search_path, _set_pattern,
                 XAPPLRESDIR,XAPPLRESDIR,XAPPLRESDIR,
                 XAPPLRESDIR,XAPPLRESDIR,XAPPLRESDIR,
                 HOME,HOME);

      }
      else 
      {
         _search_path = (char*)XtMalloc(strlen(_set_pattern)
                                        + HOME_len * 8 + 1);

         sprintf(_search_path, _set_pattern,
                 HOME,HOME,HOME,HOME,HOME,HOME,HOME,HOME);
      }
   }
}

/* 
 * Note: the Cache is going to look like a leak 
 * because it doesn't get freed at exit.
 * For grins I will install a routine to do this
 * thru the atexit function.
 */ 
static void
_internal_cache_destructor()
{
   int i;
   for (i=0;i<_image_cache_size; i++)
   {
      XtFree((char *) _image_cache[i].name);
   }
   XtFree((char *) _image_cache);
}

void
XmextAddPixmapToCache(char *name, Pixmap pmap)
{
   if(_image_cache_size <= 0)
   {
      _image_cache = (XmextImageCache) XtMalloc(sizeof(XmextImageCacheRec));
      _image_cache[0].name = _internal_strdup(name);
      _image_cache[0].pixmap = pmap;
      _image_cache_size = 1;
      /* 
       * Setup destructor for cache so it doesn't look like a leak. 
       * This is a bit anal. 8^)
       */
      atexit(_internal_cache_destructor);
   }
   else
   {
      _image_cache = (XmextImageCache) 
                     XtRealloc((char *) _image_cache, 
                              sizeof(XmextImageCacheRec)*(_image_cache_size+1));
      
      _image_cache[_image_cache_size].name = _internal_strdup(name);
      _image_cache[_image_cache_size].pixmap = pmap;
      _image_cache_size++;
   }
}

Pixmap
XmextGetPixmapFromCache(char *name)
{
   int i;

   if(_image_cache == NULL)
   {
      return XmUNSPECIFIED_PIXMAP;
   }

   for(i=0; i<_image_cache_size; i++)
   {
      if(strcmp(_image_cache[i].name,name) == 0)
      {
         return _image_cache[i].pixmap;
      }
   }

   return XmUNSPECIFIED_PIXMAP;
}

static Pixmap
_XmextGetPixmap(Display *dpy, Window w, char *fname)
{
   static Colormap _cmap;

   XmextXpmAttributes xpm_attrib;
   Pixmap pmap;
   Pixmap mask;

   char *pathname_to_pixmap = NULL;

   /* initialize colormap if it hasn't been done */
   if(_cmap == (Colormap) NULL)
   {
      XWindowAttributes w_attrib;
      XGetWindowAttributes(dpy,w,&w_attrib);
      _cmap=w_attrib.colormap;
   }

   pmap = XmextGetPixmapFromCache(fname);
   if (pmap != XmUNSPECIFIED_PIXMAP)
   {
      return pmap;
   }

   if(_search_path == NULL)
   {
      __XmextCreateSearchPath();
   }
   
   /* 
    * Attempt to find pixmap in search_path 
    * if an absolute path was not given. 
    */
   if(fname != NULL && fname[0]=='/')
   {
      pathname_to_pixmap = _internal_strdup(fname);
   }
   else
   {
      SubstitutionRec sub;
      Cardinal i, n;

      sub.match = 'B';
      sub.substitution = fname;
      
      n = XtNumber(_search_type);

      for(i=0; 
          i<n && pathname_to_pixmap==NULL; 
          i++)
      {
         pathname_to_pixmap = XtResolvePathname(dpy,
                                                _search_type[i],
                                                NULL,
                                                NULL,
                                                _search_path,
                                                &sub,
                                                1,
                                                NULL); 
      }
   }
   
   if (pathname_to_pixmap == NULL
       || strlen(pathname_to_pixmap) == 0)
   {
      return XmUNSPECIFIED_PIXMAP;
   }

   xpm_attrib.colormap=_cmap;
   xpm_attrib.closeness=40000;
   xpm_attrib.valuemask=XmextXpmSize | XmextXpmReturnPixels 
                        | XmextXpmColormap | XmextXpmCloseness;

   if(_XmextXpmReadFileToPixmap(dpy,w,pathname_to_pixmap,&pmap,
                                &mask,&xpm_attrib) == XmextXpmSuccess)
   {
      XmextAddPixmapToCache(fname,pmap);
   }
   else
   {
      /* could not find it so lets return it as unspecified */
      pmap = XmUNSPECIFIED_PIXMAP;
   }

   XtFree(pathname_to_pixmap);

   return pmap;
}

Pixmap
XmextGetPixmap(Widget w, char *fname)
{
   Screen *screen;
   Window root;
   Display *dpy;

   /* The widget might not have a window yet so just use the root window */
   screen = XtScreen(w);
   root = RootWindowOfScreen(screen);
   dpy = XtDisplay(w);

   return _XmextGetPixmap(dpy, root, fname);

}

static Boolean
_XmextCvtStringToPixmap(Display *dpy,
                        XrmValue *args,
                        Cardinal *num_args,
                        XrmValue *from,
                        XrmValue *to,
                        XtPointer *converter_data)
{
   static Pixmap _pmap;
   Screen *screen;
   Window root;
   char *name;

   if (*num_args != 1)
   {
      XtErrorMsg("wrongParameters",
                 "cvtStringToPixmap",
                 "XtToolkitError",
                 "String to Pixmap conversion needs screen argument",
                 NULL,
                 NULL);
   }


   /* get arguments */
   screen = *((Screen **) args[0].addr);
   name = (char *)from->addr;

   /* grab root window */
   root = RootWindowOfScreen(screen);

   /* over kill check */
   if (name == NULL
       || strcmp(name,"None") == 0
       || strcmp(name,"XmUNSPECIFIED_PIXMAP") == 0)
   {
      _pmap = XmUNSPECIFIED_PIXMAP;
   }
   else {
      _pmap = _XmextGetPixmap(dpy, root, name);
   }

   if (to->addr == NULL)
   {
      to->addr = (XPointer)&_pmap;
      to->size = sizeof(Pixmap);
   }
   else
   {
      if (to->size >= sizeof(Pixmap))
      {
         *((Pixmap *)to->addr) = _pmap;
         to->size = sizeof(Pixmap);
      }
      else
      {
         XtDisplayStringConversionWarning(dpy, (char*)from->addr, XmRPixmap);
      }
   }

   converter_data = converter_data; /* keeps compiler happy */

   return True;
}

static void
_XmextRegisterConverters()
{
   static XtConvertArgRec args[] = {
      {
         XtBaseOffset,
         (XtPointer)XtOffsetOf(WidgetRec, core.screen),
         sizeof(Screen *)
      }
   };

   XtSetTypeConverter(XmRString,  /* source type */
                      XmRPixmap,  /* target type */
                      _XmextCvtStringToPixmap, /* converter routine */
                      args,       /* args for converter routine */
                      XtNumber(args), /* number of args for converter */
                      XtCacheNone, /* caching instructions */
                      NULL);      /* destructor function */
 
   XtSetTypeConverter(XmRString,  /* source type */
                      XmRXmBackgroundPixmap,  /* target type */
                      _XmextCvtStringToPixmap, /* converter routine */
                      args,       /* args for converter routine */
                      XtNumber(args), /* number of args for converter */
                      XtCacheNone, /* caching instructions */
                      NULL);      /* destructor function */

}

void
XmextRegisterConverters()
{
   _XmextRegisterConverters();
}

