/****************************************************************************/
/*   xrObject.C                                                             */
/****************************************************************************/
/*                                                                          */
/*   Copyright (c) 1992, 1993 Bernhard Strassl                              */
/*       Vienna User Interface Group                                        */
/*       Institute for Applied Computer Science and Information Systems     */
/*       University of Vienna, Austria                                      */
/*                                                                          */
/*   Copyright (c) 1994 Bernhard Strassl                                    */
/*       at-systems development                                             */
/*                                                                          */
/*   See file COPYRIGHT in this directory for details.                      */
/*   For any questions mail to bernhard@atnet.co.at                         */
/*                                                                          */
/****************************************************************************/

// #include <stdlib.h>
// #include <unistd.h>
// #include <sys/wait.h>

#include <stdarg.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#ifdef WIN31
ostrstream debugout;
ostrstream* _dbout = &debugout;
class _saver
{ 	ofstream strm;
public:
	_saver() : strm("debug.out") {}
	~_saver() { strm << debugout.str(); }
};
_saver _dbout_saver;
#endif

ofstream errlog("rciclt.err");

#include <ctype.h>

#include <xmObject.h>

#ifndef WIN31
extern "C"
{
int atoi(const char*);
void exit(int);
}
#endif

#undef cb
#undef display
#undef screen
#undef mainWindow

char _xr_object_tmpstrstrbuf[4096];
#define TMPSTMRBUF _xr_object_tmpstrstrbuf,4096


// Application /////////////////////////////////////////////

XmApp* App;

void (*noX )() = NULL;

XmApp::XmApp(int argc, char** argv)
{
	App = this;
	appArgc = argc;
	appArgv = argv;
	appShell = NULL;

	toplevelShells = new XmObject*[100];
	numShells = 0;

	appContext = 0;
	display = NULL;
	screen = NULL;
	rscDb = NULL;
	unitType = 0;	// unused...
	useWindows = FALSE;
	active = FALSE;

	ostrstream initCmd;
	char* aStr;
	int i = 1;

	while(aStr = getCmdArg(i++))
		initCmd << aStr << " ";
	initCmd.put('\0');

	rciInitApp(initCmd.str());
	initialize();
	run();
}

extern char* _control_get_text_buffer;
extern char* _edit_get_text_buffer;
extern char** _listbox_items_ptr;
extern int* _listbox_indices_ptr;

extern char* lastReplyText;

XmApp::~XmApp()
{
	delete toplevelShells;

	delete _control_get_text_buffer;
	delete _edit_get_text_buffer;

	if(_listbox_items_ptr)
	{	for(int i = 0; _listbox_items_ptr[i]; i++)
			delete _listbox_items_ptr[i];
		delete _listbox_items_ptr;
	}
	delete _listbox_indices_ptr;
	delete lastReplyText;

	rciQuitApp();
	exit(1);
}

void XmApp::run()
{
	if(appShell)
	{
//#ifdef WIN31
		active = TRUE;
//#endif
		appShell->realize();
		active = TRUE;
		rciMainLoop();
		return;
	}
	errlog << "xmApp: Error - no application window created, exiting...\n";
}

void XmApp::quit(bool force)
{
#ifndef __GNUG__
	if(!force)
	{	for(int i = 0; i < numShells; i++)
			if(!((XmUserDialog*)toplevelShells[i])->queryClose())
				return;
	}
#endif
	active = FALSE;

	for(int j = 0; j < numShells; j++)
		delete toplevelShells[j];
	delete this;
}

char* XmApp::getCmdArg(int n)
{
	return(n < appArgc ? appArgv[n] : NULL);
}

XmObject* XmApp::findObject(char*)
{
	return(NULL);
}

bool XmApp::setMainObject(XmObject* obj)
{
	for(int i = 0; i < numShells; i++)
	{	if(toplevelShells[i] == obj)
		{	appShell = obj;
			return(TRUE);
		}
	}
	return(FALSE);
}

Widget XmApp::getMainWindow()
{
	return(appShell ? appShell->handle() : NULL);
}

void XmApp::setScreenUnitType(unsigned char u)
{
	if(!active)
		unitType = u;
	else
		errlog << "xmApp: Warning - cannot change unit type of running application!\n";
}

Widget XmApp::newMainWindow(XmObject* anObj, char* name, int width, int height, xmStyle s)
{
	Widget w = NULL;
	w = anObj->handle();
	if(!appShell)
		w->setRoot();
	if( TRUE /* appShell || !rciRootApp */ )
	{	w->cmd(RCI_CREATE, RCI_TOPLEVEL);
		w->arg() = (int)0;
		w->arg() = (unsigned long)s;
		w->execCreate();
		if(w->isOk() && width && height)
		{	w->cmd(RCI_CONFIG, RCI_SIZE);
			w->arg() = width;
			w->arg() = height;
			w->execCmd();
		}
	}
	if(!appShell)
		appShell = anObj;
	if(numShells < 100)
		toplevelShells[numShells++] = anObj;
	return(w);
}

void XmApp::removeMainWindow(XmObject* obj)
{
	if(!active)
		return;

	for(int i = 0; i < numShells; i++)
	{	if(toplevelShells[i] == obj)
		{	for(int j = i; j < numShells; j++)
				toplevelShells[j] = toplevelShells[j + 1];
			numShells--;
			break;
		}
	}
	if(!numShells)
	{	delete this;
		return;
	}
	if(obj == appShell)
		appShell = toplevelShells[0];
}


main(int argc, char** argv)
{
	new XmApp(argc, argv);
}

// Manager //////////////////////////////////////////////////

// will provide support for a more sufficient application
// architecture, not really implemented yet....

XmManager::~XmManager()
{
	if(objects)
	{	if(objectCount)
			errlog << "xmManager: Warning - deleting manager with " << objectCount << " references!\n";
	}
}

bool XmManager::addObject(XmObject* anObject, objType t)
{
	XmObject** newArray;

	if(t == WindowObj || t == UserDialog || t == DialogWindow)
	{	if(objectCount == maxObjects)
		{	newArray = new XmObject*[objectCount ? objectCount * 2 : 5];
			for(int i = 0; i < objectCount; i++)
				newArray[i] = objects[i];
			delete objects;
			objects = newArray;
		}
		objects[objectCount++] = anObject;
		return(TRUE);
	}
	return(FALSE);
}

bool XmManager::removeObject(XmObject* anObject)
{
	for(int i = 0; i < objectCount; i++)
	{	if(objects[i] == anObject)
		{	--objectCount;
			for(int j = i; j < objectCount; j++)
				objects[j] = objects[j + 1];
			return(TRUE);
		}
	}
	return(FALSE);
}

XmObject* XmManager::findObject(char* n, objType t)
{
	for(int i = 0; i < objectCount; i++)
	{	if(!strcmp(objects[i]->getName(), n) && objects[i]->objectType() == t)
			return(objects[i]);
	}
	return(NULL);
}

void XmManager::changed(char* txt, XmObject* except)
{
	for(int i = 0; i < objectCount; i++)
	{	// errlog << objects[i]->getName() << ", ";
		if(objects[i] != except)
			objects[i]->update(txt);
	}
	if(manager)
		manager->changed(txt);
}


// Object //////////////////////////////////////////////////


void XmObject::init(char* n, XmObject* p)
{
// errlog << "XmObject::init " << this << " \n";
	magic = MAGIC_OBJECT_CONST;
	basePtr = this;
	parent = p; 
	win = (Window )NULL;
	if(!n)
		n = "unnamedObject";
	wid = new XmRciController(this, n, (p ? p->handle() : NULL));
	wname =  strcpy(new char[strlen(n) + 1], n);
	wargcount = 0;
	gc = (GC )NULL;
	inputMask = 0;
	cdptrs = NULL;	// not needed because handled by XmRciCallback objects...
	numcdptrs = 0;
	destroyWidget = FALSE;	// avoids overhead in all cases where a widget tree is destroyed
}

XmObject::~XmObject()
{
	if(magic)
	{	magic = 0L;
		if(destroyWidget)
			XmObject::destroy();	// can't call virtual here...
		delete wname;
	}
	else
		errlog << "xmObject: Warning - deleting invalid object pointer!\n";
}

void genericCallback(Widget w, XtPointer cl_data, XtPointer ca_data)
{
	cb_data* d = (cb_data* )cl_data;

//	if(d->object->valid())			// ensure that object has not been deleted...
	{	void* answer = NULL;

		if(d->client_data == CB_XM_DATA)
			answer = ca_data;
		else if(d->client_data == CB_XM_HANDLE)
			answer = w;
		else if(d->client_data == CB_OBJ_NAME)
			answer = getObjName(w);
		else if(d->client_data > CB_OBJ_NAME)
			answer = d->client_data;

// errlog << "calling: " << d->object << " / " << ((XmObject*)d->object) << " ???\n";

		((d->object)->*(d->callback))(answer);
	}
}

char XmObject::parse(char* str)		// caution: parses string in place!
{
	char c;
	for(int i = 0; c = str[i]; i++)
	{	if(c == '&')
		{	for(int j = i; str[j]; j++)
				str[j] = str[j + 1];
			return(str[i]);
		}
	}
	return(str[0]);
}

bool XmObject::setCbFor(Widget w, char* evtName, XmObject* obj, cbProc cb, bool set, void* data)
{
	cb_data* d;

	if(!wid)
		return(FALSE);

	if(set)
	{	d = new cb_data;
		d->callback = cb;
		d->object = obj->getBase();
		d->client_data = (XtPointer )data;
// errlog << "setCbFor: " << obj << " / " << obj->getBase() << " ???\n";

		w->cmd(RCI_SETCALLBK);
		w->arg() = (int)cbName2Reason(evtName);
		w->arg() = (unsigned long)data;
		if(w->execCmd())
		{	w->addCbk(evtName, d);
			return(TRUE);
		}
	}
	else
	{
		if(w->removeCbk(evtName, d))
		{
			delete d;
		}
	}
	return(FALSE);
}

// A comment to the usage of the XmObject::XXXarg() functions: use them to collect
// a unknown number of Args in several functions, avoid them in one shot operations
// (use a local arg array instead, there may be some pending things in the wargs
// array which should take affect later).

int XmObject::argIndex(String n)
{
	for(int i = 0; i < wargcount; i++)
	{	if(!strcmp(wargs[i].name, n))
			return(i);
	}
	return(-1);
}

void XmObject::setArg(String n, XtArgVal  v)
{
	addArg(n, v);
	if(wid)
	{	//XtSetValues(wid, wargs, wargcount);
		wargcount = 0;
	}
}

void XmObject::changeArg(String n, XtArgVal  v)
{
	int ndx;

	if((ndx = argIndex(n)) != -1)
		wargs[ndx].value = v;			// dont use this func with in-place-args!!
	else
		addArg(n, v);
}

void XmObject::forceArg(String n, XtArgVal  v)
{
	Arg a;

	if(wid)
	{	if(n)
		{	; //XtSetArg(a, n, v);
			//XtSetValues(wid, &a, 1);
		}
		else
		{	//XtSetValues(wid, wargs, wargcount);
			wargcount = 0;
		}
	}
	else
		errlog << "xmObject: Warning - forceArg " << n << " ignored: object not created.\n";
}

void XmObject::syscall(char*)	// callback name, not implemented yet...
{
}

char RPB[500];

char* XmObject::getResourcePath(bool append)
{
	int olen, alen, clen;

	olen = append ? strlen(RPB) : 0;
	alen = parent ? 0 : strlen(App->getName());
	clen = strlen(wname) + alen;

	char* tmp = strcpy(new char[strlen(RPB) + 1], RPB);
	strncpy(&RPB[clen + 1], tmp, olen);
	delete tmp;
	RPB[clen + olen + 1] = '\0';
	RPB[alen] = '*';
	strncpy(&RPB[alen + 1], wname, strlen(wname));

	if(parent)
		return(parent->getResourcePath(TRUE));
	if(alen)
		strncpy(RPB, App->getName(), alen);
	return(RPB);
}

bool XmObject::checkResource(char* aName)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr(TMPSTMRBUF);

	rstr << getResourcePath() << "." << aName; rstr.put('\0');
	//return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
	return(FALSE);
}

bool XmObject::realize()
{
	if(!wid)
	{	errlog << "xmObject: error - cannot realize object (no widget created).\n";
		return(FALSE);
	}
	if(App->isActive())
	{	if(parent && !isObjRealized(parent->handle()))
		{	errlog << "xmObject: error - cannot realize object (parent not realized).\n";
			return(FALSE);
		}
		wid->cmd(RCI_CONFIG, RCI_VISIBLE);
		wid->arg() = 1;
		return(wid->execRealize());
	}
	return(FALSE);
}

bool XmObject::destroy()
{
	if(wid)			// may have been destroyed already in subclass
	{	wid->cmd(RCI_DESTROY);
		if(wid->execCmd())
		{	delete wid;
			wid = NULL;
			if(!parent)
				App->removeMainWindow(this);	// app window...
			return(TRUE);
		}
	}
	return(FALSE);
}

bool XmObject::hide()
{
	if(isVisible())
	{	wid->cmd(RCI_CONFIG, RCI_VISIBLE);
		wid->arg() = 0;
		return(wid->execCmd());
	}
	return(FALSE);
}

bool XmObject::show()
{
	if(!isVisible())
	{	wid->cmd(RCI_CONFIG, RCI_VISIBLE);
		wid->arg() = 1;
		return(wid->execCmd());
	}
	return(FALSE);
}

bool XmObject::isRealized()
{
	return(wid && isObjRealized(wid) ? TRUE : FALSE);
}

bool XmObject::isVisible()
{
	int visible = 0;

	if(wid)
	{	if(!wid->cmd(RCI_GETCONFIG, RCI_VISIBLE)->execValRequest())
			return(FALSE);
		visible = wid->answer(0);
	}
	return(visible ? TRUE : FALSE);
}

bool XmObject::move(int nx, int ny)
{
	if(wid)
	{	wid->cmd(RCI_CONFIG, RCI_LOCATION);
		wid->arg() = nx;
		wid->arg() = ny;
		return(wid->execIfVisible());
	}
	return(FALSE);
}

bool XmObject::resize(int nw, int nh)
{
	if(wid)
	{	wid->cmd(RCI_CONFIG, RCI_SIZE);
		wid->arg() = nw;
		wid->arg() = nh;
		return(wid->execIfVisible());
	}
	return(FALSE);
}

bool XmObject::reframe(int nx, int ny, int nw, int nh)
{
	if(wid)
		return(move(nx, ny) && resize(nw, nh) ? TRUE : FALSE);
	return(FALSE);
}

bool XmObject::getFrame(int& fx, int& fy, int& fw, int& fh)
{
	if(wid)
	{	if(!wid->cmd(RCI_GETCONFIG, RCI_LOCATION)->execValRequest())
			return(FALSE);
		fx = wid->answer(0);
		fy = wid->answer(1);
		if(!wid->cmd(RCI_GETCONFIG, RCI_SIZE)->execValRequest())
			return(FALSE);
		fw = wid->answer(0);
		fh = wid->answer(1);
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::setBackgroundColor(char*)
{
	return(FALSE);
}

bool XmObject::setForegroundColor(char*)
{
	return(FALSE);
}

char* _object_cursor_names[] = {
	"default", "busy", "left-arrow",
	"up-arrow", "cross", "i-beam", NULL
};

#define XC_X_cursor 0
#define XC_arrow 2
#define XC_watch 150
#define XC_center_ptr 22
#define XC_crosshair 34
#define XC_xterm 152

int _object_cursor_constants[] = {
	XC_X_cursor, XC_watch, XC_arrow,
	XC_center_ptr, XC_crosshair, XC_xterm
};

bool XmObject::setCursor(char* n)
{
	for(int i = 0; _object_cursor_names[i]; i++)
	{	if(!strcmp(_object_cursor_names[i], n))
			return(setCursor(i));
	}
	return(FALSE);
}

bool XmObject::setCursor(int n)
{
	return(FALSE);
}

// Menu base class //////////////////////////////////////////////////////////////


XmMenu::XmMenu(char* name, XmObject* par) : XmObject(name, par)
{
	nextItemPos = -1;
	numSubmenues = 0;
}

XmMenu::~XmMenu()
{
}

XmObject* XmMenu::parentOfMenu()	// find the root in a submenue hierarchy
{
	if(parent->objectType() == Menu)
		return(((XmMenu* )parent)->parentOfMenu());
	return(parent);
}

bool XmMenu::checkMenuResource(char* anItem)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr(TMPSTMRBUF);

	rstr << getResourcePath() << "*" << anItem << ".labelString"; rstr.put('\0');
	//return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
	return(FALSE);
}

#define menuResourcePath(x)  x->getPath()

bool XmMenu::addSeparator(char* name)
{
	Widget wp, w;
	
	if(!wid || !(wp = getCurrentLabel()))
		return(FALSE);
	if(!name)
		name = "separator";
	w = new XmRciController(NULL, name, wp);
	w->cmd(RCI_CREATE, RCI_MENUENTRY);
	w->arg() = wp->getId();
	w->arg() = name;
	w->arg() = "---";
	w->arg() = (void*)NULL;
	return(w->execCreate());
}

bool XmMenu::removeSeparator(char* name)
{
	char* n = name;
	Widget w;

	if(!n)
		n = "Separator";
	if(!(w = getObjChild(wid, n)))
		return(FALSE);
	w->cmd(RCI_DESTROY);
	if(w->execCmd())
	{	delete w;
		return(TRUE);
	}
	return(FALSE);
}

void menuCallback(Widget w, XtPointer cl_data, XtPointer p)
{
	cb_data* d = (cb_data* )cl_data;
	char* aStr = NULL;

	if(d->client_data)
		aStr = getObjText(w);
	else
		aStr = getObjName(w);

	((d->object)->*(d->callback))(aStr);
}


#ifdef NO_VA_OBJS
//#ifdef SUN
VA_OBJ_SUBST_IMPL(XmMenu, addItem, addItems)
//#else
//VA_OBJ_SUBST_IMPL
//#endif
#define NO_VARARGS
#endif

#define MAXITEMS 20

bool XmMenu::addItems(Entry entry, ...)
{
	Arg a;
	int argno = 0;
	char* names[MAXITEMS], * aName;
	char* items[MAXITEMS], * anItem;
	XmManager* receivers[MAXITEMS];
	cbProc procs[MAXITEMS];
	void* data[MAXITEMS];
	bool flags1[MAXITEMS];
	bool flags2[MAXITEMS];
	Entry anEntry;
	bool locate = FALSE;

	names[argno] = entry.name;
	items[argno] = entry.text;
	receivers[argno] = entry.receiver;
	procs[argno] = entry.callback;
	data[argno] = entry.client_data;
	flags1[argno] = entry.checkable;
	flags2[argno++] = entry.radio;
#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, entry);
	while(1)
	{	anEntry = va_arg(ap, Entry);
		if(!(names[argno] = anEntry.name))
			break;
		items[argno] = anEntry.text;
		receivers[argno] = anEntry.receiver;
		procs[argno] = anEntry.callback;
		data[argno] = anEntry.client_data;
		flags1[argno] = anEntry.checkable;
		flags2[argno++] = anEntry.radio;
	}
	va_end(ap);
#else
	names[argno] = NULL;
#endif

	for(int i = 0; aName = names[i]; i++)
	{	char tmpbuf[1024];
		ostrstream tmpstrm(tmpbuf, 1024);
		tmpstrm << items[i]; tmpstrm.put('\0');
		anItem = tmpstrm.str();

		if(names[i] == items[i])  // name and text are identical...
			aName = anItem;
		Widget w, wp;
		cb_data* d = new cb_data;

		wp = getCurrentLabel();
		w = new XmRciController(NULL, aName, wp);
		w->cmd(RCI_CREATE, RCI_MENUENTRY);
		w->arg() = wp->getId();
		w->arg() = aName;
		w->arg() = anItem;
		w->arg() = (void*)d; // doesn't make sense...
		w->arg() = (flags1[i] ? (flags2[i] ? 2 : 1) : 0); // entry type...
		if(!w->execCreate())
			return(FALSE);
		wargcount = 0;
		d->callback = procs[i];
		d->object = receivers[i] ? receivers[i] : parentOfMenu()->getBase();
		d->client_data = data[i];
		w->addCbk(ANY_CALLBACK, d); // aName);
	}
	return(TRUE);
}

#ifdef NO_VA_OBJS
#undef NO_VARARGS
#endif

bool XmMenu::removeItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;
#ifndef NO_VARARGS
	char* item = first;
	va_list ap;

	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		if(!w->cmd(RCI_DESTROY)->execCmd())
			return(FALSE);
		delete w;
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
	if(!w->cmd(RCI_DESTROY)->execCmd())
		return(FALSE);
	delete w;
#endif
	return(ret);

}

bool XmMenu::enableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;

#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		w->cmd(RCI_CONFIG, RCI_SENSITIVE);
		w->arg() = 1;
		if(!w->execCmd())
			return(FALSE);
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
	w->cmd(RCI_CONFIG, RCI_SENSITIVE);
	w->arg() = 0;
	if(!w->execCmd())
		return(FALSE);
#endif
	return(ret);
}

bool XmMenu::disableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;

#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
		w->cmd(RCI_CONFIG, RCI_SENSITIVE);
		w->arg() = 0;
		if(!w->execCmd())
			return(FALSE);
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
	w->cmd(RCI_CONFIG, RCI_SENSITIVE);
	w->arg() = 1;
	if(!w->execCmd())
		return(FALSE);
#endif
	return(ret);
}

XmSubMenu* XmMenu::addSubmenu(char* xn) // not implemented yet...
{
	XmSubMenu* aMenu = NULL;
	return(aMenu);
}

XmSubMenu* XmMenu::submenuAt(char* xn)
{
	XmSubMenu* aMenu = NULL;
	return(aMenu);
}

bool XmMenu::removeSubmenu(char* n)
{
	return(FALSE);
}

bool XmMenu::changeItemText(char* itemName, char* xnewText)
{
	Widget w;
	Arg a[2];
	ostrstream tmpstrm(TMPSTMRBUF);
	tmpstrm << xnewText; tmpstrm.put('\0');
	char* newText = tmpstrm.str();

	if(!(w = getObjChild(getCurrentLabel(), itemName)))
		return(FALSE);
	w->cmd(RCI_CONFIG, RCI_LABEL);
	w->arg() = newText;
	return(w->execCmd());
}

bool XmMenu::getItemStatus(char* itemName)
{
	Widget w;

	if(!(w = getObjChild(getCurrentLabel(), itemName)))
		return(FALSE);
	if(!w->cmd(RCI_GETVAL, RCI_STATE)->execCmd())
		return(FALSE);
	return(((int)w->answer(0)) ? TRUE : FALSE);
}

bool XmMenu::setItemStatus(char* itemName, bool newStatus)
{
	Widget w;

	if(!(w = getObjChild(getCurrentLabel(), itemName)))
		return(FALSE);
	w->cmd(RCI_SETVAL, RCI_STATE);
	w->arg() = newStatus;
	return(w->execCmd());
}

// Popup menu /////////////////////////////////////////////////////////////////


XmPopupMenu::XmPopupMenu(char* xname, XmObject* par, bool defaultMenu) : XmMenu(xname, par)
{
}

void XmPopupMenu::makeDefault(bool sr)
{
}

bool XmPopupMenu::setLabel(char* l)
{
	return(FALSE);
}

bool XmPopupMenu::popup()
{
	return(FALSE);
}


// Dropdown menu /////////////////////////////////////////////////////////////////

XmDropdownMenu::XmDropdownMenu(char* name, XmObject* par) : XmMenu(name, par)
{
	numDropdowns = 0;
	nextLabelPos = -1;
	currentLabel = (Widget )NULL;

	wid->cmd(RCI_CREATE, RCI_MENUBAR);
	wid->arg() = par->handle()->getId();
	wid->execCreate();
}

#ifdef NO_VA_OBJS
#ifdef SUN
VA_OBJ_SUBST_IMPL(XmDropdownMenu, addLabel, addLabels)
#endif
//#define NO_VARARGS
#endif

#ifdef SUN
bool XmDropdownMenu::addLabels(Entry first, ...)
{
	return(FALSE);
}
#endif

bool XmDropdownMenu::addLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;

	for(int i = 0; xitem = items[i]; i++)
	{	char tmpbuf[1024];
		ostrstream tmpstrm(tmpbuf, 1024);
		tmpstrm << xitem; tmpstrm.put('\0');
		item = tmpstrm.str();
		char mm = parse(item);
		char* aName = item;	// should be parameter...!!
		Widget w;

		w = new XmRciController(NULL, aName, wid);
		w->cmd(RCI_CREATE, RCI_MENULABEL);
		w->arg() = wid->getId();
		w->arg() = aName;
		w->arg() = item;
		if(!w->execCreate())
			return(FALSE);
		currentLabel = w;
	}
	return(TRUE);
}

bool XmDropdownMenu::removeLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;
	bool ret = TRUE;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;

	for(int i = 0; xitem = items[i]; i++)
	{	Widget w;
		if(!(w = getObjChild(wid, xitem)))
		{	ret = FALSE;
			continue;
		}
		if(!w->cmd(RCI_DESTROY)->execCmd())
			return(FALSE);
		delete w;
	}
	return(ret);
}

bool XmDropdownMenu::setCurrentLabel(char* n)
{
	Widget lastLabel = currentLabel;

	if(currentLabel = wid->getChild(n))
		return(TRUE);	
	errlog << "xmDropdownMenu: Warning - cannot find label " << n << " using: " << /* hex(long(lastLabel)) << */ "\n";
	currentLabel = lastLabel;
	return(FALSE);
}

bool XmDropdownMenu::changeLabelText(char*, char*)
{
	return(FALSE);
}

// subMenu /////////////////////////////////////////////////////////////////////

XmSubMenu::XmSubMenu(char* n, XmMenu* par) : XmMenu(n, par)
{
}

XmSubMenu::~XmSubMenu()
{
}


// Dialog base class //////////////////////////////////////////////////////////

XmDialog::XmDialog(char* n, XmObject* par, xmStyle s) : XmObject(n, par)
{
	basePtr = (XmObject* )((void* )this);
	dlgStyle = (s & XmSdlgWinModal || s & XmSdlgAppModal || s & XmSdlgSysModal || s & XmSpopup) ? s : XmSdlgModeless;

	wid->setToplevel();
}

XmDialog::~XmDialog()
{
}


bool XmDialog::setLabelAndMode(char* label)
{
	wid->cmd(RCI_CONFIG, RCI_LABEL);
	wid->arg() = label;
	return(wid->execCmd());
}

bool XmDialog::run()
{
	if(dlgStyle & XmSdlgModeless)
	{	errlog << "xmDialog: Error - cannot run a modeless dialog.\n";
		return(FALSE);
	}
	completed = returned = FALSE;
	while(!completed)
		rciNextEvent();
	return(returned);
}

void XmDialog::ok(void*)
{
	returned = completed = TRUE;
}

void XmDialog::cancel(void*)
{
	completed = TRUE;
}

void XmDialog::close(void*)
{
	delete this;
}

// Predefined system dialogs //////////////////////////////////////////////////

XmSystemDialog::XmSystemDialog(char* n, XmObject* par, xmStyle s) : XmDialog(n, par, s)
{
	destroyOnDelete();

	// as the current implementation coveres only modal dialogs there are
	// no more special initialization / cleanup functions necessary.
	// everything is done inside the constructors, the modal open functions only free
	// return the global return value assigned by the last opened system dialog
}

int lastReplyCode = 0;
char* lastReplyText = NULL;

XmMsgBox::XmMsgBox(char* text, char* label, xmStyle s, XmObject* rec, cbProc okCb) : XmSystemDialog("msg", NULL, s ? s : XmSmsgOkCancel | XmSdlgAppModal)
{
	wid->cmd(RCI_SYSDLG, RCI_MESSAGEBOX);
	wid->arg() = text;
	wid->arg() = label;
	wid->arg() = (unsigned long)(s ? s : XmSmsgOkCancel | XmSdlgAppModal);
	wid->arg() = int(rec ? rec->handle()->getId() : 0);
	if(wid->execValRequest())
		lastReplyCode = wid->answer(0);
}


XmMsgBox::XmMsgBox(char* n, XmObject* par, xmStyle s) : XmSystemDialog(n, par, (s ? s : XmSmsgOkCancel | XmSdlgAppModal))
{
}

void XmMsgBox::setDefaultButtonProcs(XmObject*, cbProc)
{
}

bool XmMsgBox::showMsg()
{
	delete this;
	return(lastReplyCode ? TRUE : FALSE);
}

bool XmMsgBox::setButtonText(msgButton, char*)
{
	if(!wid)
		return(FALSE);
	return(FALSE);
}


XmPrompter::XmPrompter(char* n, XmObject* par, xmStyle s) : XmMsgBox(n, par, s)
{
}

XmPrompter::XmPrompter(char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmMsgBox("prompter", (XmObject* )NULL, s)
{
	wid->cmd(RCI_SYSDLG, RCI_PROMPTER);
	wid->arg() = prompt;
	wid->arg() = label;
	wid->arg() = def;
	wid->arg() = (unsigned long)(s ? s : XmSmsgOkCancel | XmSdlgAppModal);
	wid->arg() = int(rec ? rec->handle()->getId() : 0);
	if(wid->execValRequest())
	{	lastReplyCode = wid->answer(0);
		char* answer = wid->answer(1);
		delete lastReplyText;
		lastReplyText = _mkstr(answer);
	}
}

char* XmPrompter::prompt()
{
	char* txt = NULL;

	if(lastReplyCode)
		txt = lastReplyText;
	delete this;
	return(txt);
}

bool XmPrompter::setText(char* txt)
{
	return(FALSE);
}

char promptBuf[500];

char* XmPrompter::getText()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
	txt = NULL;
	return(txt);
}

XmListPrompter::XmListPrompter(char** list, char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("listPrompter", NULL, s)
{
}

int XmListPrompter::promptIndex()
{
	int ndx = -1;

	if(run())
		ndx = getIndex();
	delete this;
	return(ndx);
}

XmListPrompter* XmListPrompter::selectText(char*)
{
	return(this);
}

XmListPrompter* XmListPrompter::selectIndex(int)
{
	return(this);
}

int XmListPrompter::getIndex()
{
	return(0);
}

XmFileSelector::XmFileSelector(char* path, char* label, char* deflt, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("fileSelector", NULL, s)
{
	wid->cmd(RCI_SYSDLG, RCI_FILESELECTOR);
	wid->arg() = path;
	wid->arg() = label ? label : "Select File";
	wid->arg() = deflt ? deflt : "";
	wid->arg() = (unsigned long)(s ? s : XmSmsgOkCancel | XmSdlgAppModal);
	wid->arg() = int(rec ? rec->handle()->getId() : 0);
	if(wid->execValRequest())
	{	lastReplyCode = wid->answer(0);
		char* answer = wid->answer(1);
		delete lastReplyText;
		lastReplyText = _mkstr(answer);
	}
}

char* XmFileSelector::promptFile()
{
	char* txt;

	if(!lastReplyCode)
	{	delete this;
		return(NULL);
	}
	txt = lastReplyText;
	delete this;
	return(txt);
}

bool XmFileSelector::setPath(char* txt)
{
	return(FALSE);
}

char* XmFileSelector::getPath()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
	return(getText());
}


// Window classes /////////////////////////////////////////////////////////////


XmDialogWindow::XmDialogWindow(char* n, XmObject* par, xmStyle s) : XmUserDialog(n, par, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::XmDialogWindow(char* n, XmObject* par, XmManager* mgr, xmStyle s) : XmUserDialog(n, par, mgr, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::~XmDialogWindow()
{
//	if(manager)
//		manager->removeObject(this);
	delete dropdown;
	delete popup;
}


bool XmDialogWindow::realize()
{
	return(XmUserDialog::realize());
}

XmControl* XmDialogWindow::findControl(ctrlType t, char* n, bool w)
{
	return(XmDialogPane::findControl(t, n, w));
}

bool XmDialogWindow::add(XmControl* c, XmObject* rec, cbProc cb, void* cd)	
{
	return(XmDialogPane::add(c, rec, cb, cd));
}

void XmWindow::initWindowContents(char* n)
{
	ostrstream pn(TMPSTMRBUF);

	pn << n << "Panes"; pn.put('\0');

	panes = new XmPaneArea(pn.str(), 0, 0, 1, 1);
	add(panes);
	toolbars[0] = toolbars[1] = toolbars[2] = toolbars[3] = NULL;
	statusbar = NULL;
}

void XmWindow::menuSet()
{
	checkLayout();
}

bool XmWindow::realize()
{
	checkLayout();
	return(XmUserDialog::realize());
}

inline int orNdx(xmStyle s)
{
	return(s == XmSleft ? 0 : (s == XmSright ? 1 : (s == XmSbottom ? 3 : 2)));
}


void XmWindow::checkLayout()
{
/*
	if(toolbars[2]) // top
	if(toolbars[3]) // bottom
	if(toolbars[0]) // left
	if(toolbars[1]) // right
*/
}

bool XmWindow::addSubpane(ctrlType ct, char* n, char* l, int y, int h, XmObject* rec, cbProc cb)
{
	XmControl* aControl;

	switch(ct)
	{	case Edit:
		aControl = new XmEdit(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case ListBox:
		aControl = new XmListBox(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case Drawing:
		case GroupBox: case StaticText: case StaticImage: case PushButton:
		case CheckBox: case RadioButton: case ScrollBar: case ComboBox: case Pane: 
		case PaneArea: case ToolBar:
		errlog << "xmWindow: Warning - control type " << ct << " not supported as subpane, ignoring...\n"; 
		return(FALSE);
	}
	return(addSubpane(aControl, l, rec, cb));
}

bool XmWindow::addSubpane(XmControl* c, char* l, XmObject* rec, cbProc cbk)
{
	if(!panes->numSubpanes())
		checkLayout();

	return(panes->addSubpane(c, l, rec, cbk));
}

XmToolBar* XmWindow::addToolbar(xmStyle s)
{
	ostrstream tbName(TMPSTMRBUF);
	int ndx = orNdx(s);
	XmToolBar* aToolbar;

	if(toolbars[ndx])
		return(NULL);
	tbName << wname << "-Toolbar-" << ndx; tbName.put('\0');
	toolbars[ndx] = aToolbar = new XmToolBar(tbName.str(), (s == XmSleft || s == XmSright) ? XmSvertical : XmShorizontal);
	add(aToolbar);
	checkLayout();
	return(aToolbar);
}

bool XmWindow::removeToolbar(XmToolBar* tb)
{
	int ndx = -1;

	for(int i = 0; i < 4; i++)
	{	if(toolbars[i] == tb)
		{	ndx = i;
			break;
		}
	}
	if(ndx == -1)
		return(FALSE);
	toolbars[ndx] = NULL;
	checkLayout();
	remove((char* )tb->getName());
	return(TRUE);
}

XmToolBar* XmWindow::toolbar(xmStyle s)
{
	return(toolbars[orNdx(s)]);
}

XmStatusBar* XmWindow::createStatusBar()
{
	ostrstream stbName(TMPSTMRBUF);

	if(statusbar)
		return(NULL);

	stbName << wname << "-Statusbar"; stbName.put('\0');
	statusbar = new XmStatusBar(stbName.str());
	add(statusbar);
	checkLayout();
	return(statusbar);
}

#ifndef __GNUG__

#define tbWinStyle (XmSdlgModeless | XmSbordered | XmSmoveable | XmScloseable | XmStitled | XmSsysMenu)

XmToolBox::XmToolBox(char* n, XmObject* par, xmStyle s, int r) : XmDialogWindow(n, par, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

XmToolBox::XmToolBox(char* n, XmObject* par, XmManager* m, xmStyle s, int r) : XmDialogWindow(n, par, m, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

#endif

// rci/tip specific stuff //////////////////////////////////////////////////////

void rciShowError(char* str)
{
	errlog << "*** error: " << str << "\n";
}


/////////////////////////////////////////////////////////////////////////

XmRciController::XmRciController(XmObject* anObj, char* n, XmRciController* par)
{
	rciId = rciGetId(RCI_ID_XM);
	name = NULL;
	setName(n);
	rciOk = FALSE;
	rciToplevel = FALSE;
	rciRealized = FALSE;
	rciRoot = FALSE;
	callbacks = new mincoll<XmRciCallback*>;
	obj = anObj;
	parent = NULL;
	setParent(par);
	children = new mincoll<XmRciController*>;
}

XmRciController::~XmRciController()
{
	setParent(NULL);
	delete name;

	int i, casz = callbacks->size(), chsz = children->size();
	mincoll<XmRciController*> childrenCopy;

	for(i = 0; i < casz; i++)
		delete callbacks->at(i);
	for(i = 0; i < chsz; i++) 
		childrenCopy.add(children->at(i));
	for(i = 0; i < chsz; i++) 
		delete childrenCopy.at(i);

	callbacks->reset();
	children->reset();

	delete callbacks;
	delete children;
}

bool XmRciController::addCbk(char* n, void* d, char* wn)
{
	callbacks->add(new XmRciCallback(n, d, wn));
	return(TRUE);
}

bool XmRciController::removeCbk(char* n, void* d, char* wn)
{
	XmRciCallback* c;
	
	for(int i = 0; c = callbacks->at(i); i++)
	{	if(c->equ(n, d, wn))
		{	callbacks->remove(i);
			return(TRUE);
		}
	}
	return(FALSE);
}

void XmRciController::callCallbacks(int reason, char* callData)
{
	XmRciCallback* cb;
	for(int i = 0; cb = callbacks->at(i); i++)
	{	if(cb->matches(reason))
		{	if((void*)callData != (void*)cb->client_data)
				; // errlog << "callCallbacks failure...\n";

			if(!obj) // must be a menu object...
				menuCallback(this, cb->client_data, name);
			else
				genericCallback(this, cb->client_data, NULL);
		}
		if(!isValid())	// object may have been deleted in a callback...!
			break;
	}
}

void XmRciController::addChild(XmRciController* c)
{
	children->add(c);
}

void XmRciController::removeChild(XmRciController* c)
{
	children->remove(c);
}

XmRciController* XmRciController::getChild(char* n)
{
	XmRciController* c;
	
	for(int i = 0; c = children->at(i); i++)
	{	if(	(c->name && !strcmp(n, c->name)) ||
			(c = c->getChild(n)))
		{	return(c);
		}
	}
	return(0);
}

void XmRciController::processArgs(ArgList lst, int n)
{
	if(n)
	{	for(int i = 0; i < n; i++)
		{	if(!strcmp(lst[i].name, "text"))
			{	((XmControl*)obj)->setText((char*)lst[i].value);
				delete (char*)lst[i].value;
				continue;
			}
			// other args should go here...
		}
	}
}
