/* ****************************** -*-c++-*- *******************************/
/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

#ifndef OBJECT_H
#define OBJECT_H

#include <am_inc.h>

#include TYPES__H

// Am_Slot_Key is used to name the slots of objects.
typedef unsigned short Am_Slot_Key;

extern void Am_Print_Key (ostream& os, Am_Slot_Key key);

// Am_Constraint represents a constraint object.  Its type is opaque in this
// context.  The file formula.h declares the standard amulet factory for
// creating constraints.
class Am_Constraint;

// The Am_Constraint_Context is an opaque type used to represent a constraint's
// local state when used in the special Am_Object::Get () method.
class Am_Constraint_Context;

/////////////////////////////////////////////////////////////////////////////
// Main Object code
/////////////////////////////////////////////////////////////////////////////

enum Am_Inherit_Rule { Am_LOCAL, Am_INHERIT, Am_COPY, Am_STATIC };

class Am_Object_Data;

// The Am_Object class.  This is used to represent Amulet objects like
// windows, graphical objects, and interators.
class Am_Object {
  Am_WRAPPER_DECL (Am_Object)
 public:
  Am_Object ()
  { data = NULL; }

  // Creates an instance of the object.  The name of the new instance is
  // given by the parameter.  See the documentation for a description of
  // instantiation.
  Am_Object Create (const char* new_name = NULL);

  // Makes a copy of the object as though an instance was created of the
  // object's prototype.  The slot values, and parts of the source are copied
  // as identically as possible.
  Am_Object Copy ();

  // Test whether object is same as another.
  bool operator== (const Am_Object& test_object) const;
  bool operator!= (const Am_Object& test_object) const;
  
  // Returns whether the prototype parameter is in the prototype chain of the
  // object.  Will return true if the parameter is the object itself.
  bool Is_Instance_Of (Am_Object prototype) const;

  // Answers the question is the object a part of the object given in the
  // owner parameter.  An object is considered to be part of itself.
  bool Is_Part_Of (Am_Object owner) const;

  // Get the value of a slot.  Using the versions with the Constraint_Context
  // allows the slot to be accessed by a constraint.
  const Am_Value& Get (Am_Slot_Key key) const;
  void Get (Am_Slot_Key key, Am_Value& value) const;
  const Am_Value& Get (Am_Constraint_Context& cc, Am_Slot_Key key) const;
  void Get (Am_Constraint_Context& cc, Am_Slot_Key key, Am_Value& value) const;

  // Prints the name of the object to the given output stream.
  void Print_Name (ostream& os) const;

  // Prints all the slots of the object and their values for debugging to cout
  void Text_Inspect() const;
  // Prints one slot of the object and its values for debugging to cout
  void Text_Inspect (Am_Slot_Key key) const;

  // Get and set the object's name in the global name registry
  Am_Object& Set_Name (const char* name);
  const char* Get_Name () const;
  
  // Returns an object's key if the object is a named part of another object.
  // Returns Am_NO_SLOT when the object is not a part or is an unnamed part.
  // Returns Am_NO_INHERIT which object is an unnamed part that doesn't
  // get inherited when owner is instantiated.
  Am_Slot_Key Get_Key () const;

  // Return the object  from which this object was instanced.
  Am_Object Get_Prototype () const;

  // Queries a slot for an object.  This causes an error if the slot does
  // not actually contain an object.  The Constraint_Context version is used
  // when constraints are involved.
  Am_Object Get_Object (Am_Slot_Key key) const;
  Am_Object Get_Object (Am_Constraint_Context& cc, Am_Slot_Key key) const;

  // Returns the object's owner.  Returns NULL if the object does not have an
  // owner.  The Constrain_Context version is for applying a constraint to the
  // owner slot of the object.
  Am_Object Get_Owner () const;
  Am_Object Get_Owner (Am_Constraint_Context& cc) const;

  // Find a named part of an object.  The Constraint_Context version can be
  // used when one wants to apply a constraint to the part slot.
  Am_Object Get_Part (Am_Slot_Key key) const;
  Am_Object Get_Part (Am_Constraint_Context& cc, Am_Slot_Key key) const;

  // Find a named part of the owner of an object.  The Constraint_Context
  // version can be used when one wants to apply a constraint to the part slot.
  Am_Object Get_Sibling (Am_Slot_Key key) const;
  Am_Object Get_Sibling (Am_Constraint_Context& cc, Am_Slot_Key key) const;

  // Returns the type of the value stored in a slot.  If the slot does not
  // exist, Am_NONE is returned.
  Am_Value_Type Get_Slot_Type (Am_Slot_Key key) const;

  // Returns true if slot is inherited from some prototype.
  bool Is_Slot_Inherited (Am_Slot_Key key) const;
  
  // Make a slot independent from all other slots.
  void Make_Unique (Am_Slot_Key key);
  
  // Returns whether value of slot is independent from all other slots.
  bool Is_Unique (Am_Slot_Key key);
  
  // Set the value of a slot.
  Am_Object& Set (Am_Slot_Key key, Am_Wrapper* value);
  Am_Object& Set (Am_Slot_Key key, Am_Ptr value);
  Am_Object& Set (Am_Slot_Key key, int value);
  Am_Object& Set (Am_Slot_Key key, long value);
#if !defined(NEED_BOOL)
  Am_Object& Set (Am_Slot_Key key, bool value);
#endif
  Am_Object& Set (Am_Slot_Key key, float value);
  Am_Object& Set (Am_Slot_Key key, double value);
  Am_Object& Set (Am_Slot_Key key, char value);
  Am_Object& Set (Am_Slot_Key key, const char* value);
  Am_Object& Set (Am_Slot_Key key, const Am_String& value);
  Am_Object& Set (Am_Slot_Key key, Am_Generic_Procedure* value);
  Am_Object& Set (Am_Slot_Key key, Am_Method_Wrapper* value);
  Am_Object& Set (Am_Slot_Key key, const Am_Value& value);

  // Version of set used in formulas to let the constraint handle the
  // setting of the slot.
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  Am_Wrapper* value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, Am_Ptr value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, int value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, long value);
#if !defined(NEED_BOOL)
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, bool value);
#endif
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, float value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, double value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key, char value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  const char* value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  const Am_String& value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  Am_Generic_Procedure* value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  Am_Method_Wrapper* value);
  Am_Object& Set (Am_Constraint_Context& cc, Am_Slot_Key key,
                  const Am_Value& value);

  // Note_Changed allows one to cause a slot to act as though its value has
  // been changed even if the actual value stored has not changed.  This
  // function is especially useful for slot changes due to side effects.
  void Note_Changed (Am_Slot_Key key);
  void Note_Changed (Am_Constraint_Context& cc, Am_Slot_Key key);

  void Note_Unchanged (Am_Constraint_Context& cc, Am_Slot_Key key);
  
  // Apply a constraint to the slot.  The constraint is added to the slot.
  Am_Object& Set (Am_Slot_Key key, Am_Constraint* constraint);

  // Adds a part to an object.  The part either be named by providing a key or
  // unnamed.
  Am_Object& Add_Part (Am_Object new_part, bool inherit = true);
  Am_Object& Add_Part (Am_Slot_Key key, Am_Object new_part);

  // Destroys the slot.
  void Remove_Slot (Am_Slot_Key key);
  
  // Removes a part from an object.  The part can be named either by a key or
  // by the actual part object.
  void Remove_Part (Am_Slot_Key key);
  void Remove_Part (Am_Object part);

  // Make the object no longer be a part.
  void Remove_From_Owner ();

  // Removes all locally defined constraints from a slot.
  void Remove_Constraint (Am_Slot_Key key);

  // Destroy the object and all its parts.
  void Destroy ();


  // Get and Set "Advanced" properties of slots
  Am_Object& Set_Demon_Bits (Am_Slot_Key key, unsigned short bits);
  unsigned short Get_Demon_Bits (Am_Slot_Key key) const;

  Am_Object& Set_Inherit_Rule (Am_Slot_Key key, Am_Inherit_Rule rule);
  Am_Inherit_Rule Get_Inherit_Rule (Am_Slot_Key key) const;
  
  Am_Object& Set_Type_Check (Am_Slot_Key key, unsigned short type);
  unsigned short Get_Type_Check (Am_Slot_Key key) const;

  Am_Object& Set_Read_Only (Am_Slot_Key key, bool read_only);
  bool Get_Read_Only (Am_Slot_Key key) const;

  Am_Object& Set_Single_Constraint_Mode (Am_Slot_Key key, bool mode);
  bool Get_Single_Constraint_Mode (Am_Slot_Key key) const;

};

// The NULL object.
extern Am_Object Am_No_Object;

// Create a Type ID for Am_Object.
const Am_Value_Type Am_OBJECT = Am_WRAPPER | 1;

// This is a helper class used for iterating through an object's instances.
// To initialize the iterator, assign the prototype object to it, or assign
// the object when the iterator gets created.
class Am_Instance_Iterator {
 public:
  Am_Instance_Iterator ();
  Am_Instance_Iterator (Am_Object object);
  
  Am_Instance_Iterator& operator= (Am_Object object);
  
  unsigned short Length ();   // Number of instances in the list.
  void Start ();                  // Begin list at the start.
  void Next ();                   // Move to next element in list.
  bool Last ();                   // Is this the last element?
  Am_Object Get ();              // Get the current element.
 private:
  Am_Object current;
  Am_Object prototype;
};

class Am_Slot_Iterator_Data;

// This is a helper class used for iterating through the slots of an object.
// The iterator is initialized by setting it with the object whose slots are
// to be examined.  The elements of the list are actually slot keys and not
// actual slots.  This iterator will list all keys that can be accessed from
// the object including those defined in prototype objects.
class Am_Slot_Iterator {
 public:
  Am_Slot_Iterator ();
  Am_Slot_Iterator (Am_Object object);
  ~Am_Slot_Iterator ();
  
  Am_Slot_Iterator& operator= (Am_Object object);
  
  unsigned short Length ();       // Number of slots in the list.
  void Start ();                  // Begin list at the start.
  void Next ();                   // Move to next element in list.
  bool Last ();                   // Is this the last element?
  Am_Slot_Key Get ();             // Get the current element.
 private:
  Am_Slot_Iterator_Data* data;
  Am_Object context;
};

// This is a helper class used for iterating through an object's parts.
// The iterator is inialized by assigning the owner object to the iterator.
// The elements of the list will include both named and unnamed parts.
class Am_Part_Iterator {
 public:
  Am_Part_Iterator ();
  Am_Part_Iterator (Am_Object object);
  
  Am_Part_Iterator& operator= (Am_Object object);
  
  unsigned short Length ();       // Number of parts in the list.
  void Start ();                  // Begin list at the start.
  void Next ();                   // Move to next element in list.
  bool Last ();                   // Is this the last element?
  Am_Object Get ();               // Get the current element.
 private:
  Am_Object current;
  Am_Object owner;
};

// Prints out an identifying name for the object to the output stream.
extern ostream& operator<< (ostream& os, const Am_Object& object);

// The root object.  This object is instanced to create other objects.  The
// Am_Root_Object has no slots and no parts.
extern Am_Object Am_Root_Object;

// An empty constraint context.  Using this context with GV will cause GV to
// act just like Get.  Similar for other cc using methods.
extern Am_Constraint_Context* Am_Empty_Constraint_Context;

// Popular method type:
Am_Define_Method_Type (Am_Object_Method, void, (Am_Object))
  

// Macros that automatically append the cc parameter for constraint Gets and
// Sets.
#define GV(s1) Get (cc, s1)
#define GVM(s1, value) Get (cc, s1, value)
#define GV_Object(s1) Get_Object (cc, s1)
#define GV_Owner() Get_Owner (cc)
#define GV_Part(s1) Get_Part (cc, s1)
#define GV_Sibling(s1) Get_Sibling (cc, s1)

#define SV(s1, val) Set (cc, s1, val)

#endif
